# -*- coding: utf-8 -*-
__license__   = 'GPL v3'
__copyright__ = '2016,2017,2018,2019,2020,2021,2022,2023 DaltonST'
__my_version__ = "1.0.27" #qt.core

from qt.core import (Qt, QMimeData, QDialog, QDrag, QLabel,  QFont, QWidget,
                                       QIcon, QGridLayout, QGroupBox, QMargins, QScrollArea,
                                       QSize, QPushButton, QVBoxLayout, QHBoxLayout, QCheckBox,
                                       QLineEdit, QRadioButton, QCompleter, QTextEdit, QTextOption,
                                       QFileDialog, QApplication, QTimer, QModelIndex)

import os,sys
import apsw
import ast
import datetime
from datetime import datetime
from functools import partial
from PIL import Image
from PIL.ExifTags import TAGS
import shutil
import struct
import time
from time import sleep

from calibre import isbytestring
from calibre.constants import DEBUG, iswindows, filesystem_encoding
from calibre.ebooks.metadata.book.base import Metadata
from calibre.gui2 import error_dialog, info_dialog, FileDialog, __init__, gprefs
from calibre.ptempfile import PersistentTemporaryFile
from calibre.utils.html2text import html2text

from polyglot.builtins import as_unicode, iteritems, map, range, unicode_type


from calibre_plugins.media_file_importer.config import prefs

#~ Number of Known Image file extensions:  92
#~ Number of Known Audio file extensions:  290
#~ Number of Known Video file extensions:  143

IMAGE_FILE_FILTER=set(['.ani','.art','.bufr','.cal','.cur','.bm','.bmp','.dcx','.dds','.dwg','.dxf','.eps','.fax','.fif','.fits','.flc','.fli','.flo','.fpx','.ftex','.g3','.gbr','.gd','.gif','.grib','.hdf5','.ico','.icns','.ief','.iefs','.im','.img','.imt','iptc','.jbg','.jfif','.jfif-tbnl',',jcp','.jpe','.jpeg','.jpg','.jps','.jpx','.jut','.mac','mcidas','.mic','.msp','.mcf','.naa','.nap','.naplps','.nif','.niff','.pbm','.pcd','.pct','.pcx','.pgm','.pic','.pict','.pixar','.pm','.png','.pnm','.ppm','.psd','.qif','.qti','.qtif','.ras','.rast','.rf','.rgb','.rp','.sgi','.spider','.svf','.svg','.tga','.tif','.tiff','.turbot','.wal','.wbmp','.webp','.wmf','.x-png','.xbm','.xif','.xpm','.xwd'])

AUDIO_FILE_FILTER=set(['.aac','.aif','.aifc','.aiff','.ape','.atp','.au','.dct','.dss','.dvf','.flac','.fuk','.gsd','.gsm','.it','.jam','.kar','.la','.lam','.lma','.m2a','.m3u','.mid','.midi','.mjf','.mod','.mp2','.mp3','.mpa','.mpg','.mpga','.msv','.my','.oga','.ogg','.oma','.omg','.pcm','.pfunk','.qcp','.ra','.ram','.raw','.rm','.rmi','.rmm','.rmp','.rpm','.s3m','.sid','.snd','.tsi','.tsp','.voc','.vox','.vqe','.vqf','.vql','.wav','.weba','.xm''4mp','.669','.6cm','.8cm','.8med','.8vx','.aa3','.aac','.abc','.ac3','.acd','.act','.ahx','.aif','.aifc','.aiff','.ai','.akp','.al','.alaw','.all','.amr','.am','.am','.ape','.ae','.au','.aud','.aup','.avr','.band','.bap','.bdd','.box','.bwf','.c01','.caf','.cda','.cdr','.cel','.cmf','.copy','.cpr','.ch','.cwp','.d00','.d01','.dcf','.dcm','.dct','.dewf','.df2','.dfc','.dig','.dig','.dl','.dm','.dmf','.df','.dm','.dp','.d','.dtm','.dvf','.DWD','.ear','.efa','.efe','.efk','.efq','.ef','.efv','.emd','.ep','.f2r','.f32','.f3r','.f64','.far','.fff','.flac','.flp','.fm','.fzb','.fzf','.fzv','.g721','.g723','.g726','.gig','.gp5','.gm','.gm','.ic','.iff','.in','.in','.it','.iti','.it','.jam','.k25''k26','.kar','.kin','.kmp','.koz','.koz','.krz','.kc','.kf','.ktp','.l','.lo','.lvp','.m3u','.m4a','.m4b','.m4p','.m4r','.ma1','.mdl','.med','.mid','.midi','.miniuf','.mka','.mmf','.mod','.mp1','.mp2','.mp3','.mpa','.mpc','.mpd','.mpga','.mpu','.mp_','.mv','.mt2','.mte','.mti','.mtm','.mtp','.mt','.mu','.mw','.nap','.nrt','.na','.nf','.nt','.ntn','.nwc','.odm','.ogg','.okt','.omf','.omg','.pac','.pat','.pbf','.pca','.pcm','.phy','.pla','.pl','.pna','.prg','.prg','.pm','.ptm','.pt','.pvc','.qcp','.r','.ra','.ram','.raw','.rb','.rfl','.rmf','.rmi','.rmx','.rng','.rol','.rn','.rti','.rtm','.rt','.3i','.3m','.af','.am','.b','.bi','.bk','.c2','.d','.d','.d2','.d','.dx','.f','.f2','.fk','.fl','.hn','.ib','.id','.mf','.mp','.nd','.nd','.nd','.ng','.ou','.ppack','.prg','.eq','.nd','.tm','.tx','.vx','.w','.wa','.yh','.yw','.yx','.td0','.tfmx','.thx','.toc','.tp','.txw','.u','.ub','.ulaw','.ult','.ulw','.uni','.uf','.uflib','.uw','.uwf','.vag','.val','.vmd','.vmf','.vmf','.voc','.voi','.vox','.vqf','.vrf','.w01','.wav','.wave','.wax','.wfb','.wfd','.wfp','.wma','.wow','.wpk','.wproj','.wrk','.wu','.wut','.wv','.wvc','.wwu','.xf','.xi','.xm','.xmf','.xmi','.xp','.xt','.zvd'])

VIDEO_FILE_FILTER=set(['.3g2','.3gp','.afl','.asf','.asx','.avchd','.avi','.avs','.dif','.dl','.dv','.fli','.flv','.fmf','.gl','.isu','.m1v','.m2v','.mjpg','.mkv','.moov','.mov','.movie','.mp2','.mp3','.mp4','.mpa','.mpe','.mpeg','.mpeg-4','.mpg','.mv','.ogv','.qt','.qtc','.rv','.scm','.swf','.vdo','.viv','.vivo','.vob','.vos','.webm','wmv','.xdr','.xsr','.3g2','.3gp','.3gp2','.3gpp','.3mm','.60d','.aep','.ajp','.amv','.asf','.asx','.avb','.avi','.avs','.bik','.bix','.box','.byu','.camrec','.cvc','.d2v','.dat','.dce','.dif','.dir','.divx','.dmb','.dpg','.dv','.dvr-ms','.dxr','.eye','.fcp','.flc','.fli','.flv','.flx','.gl','.grasp','.gvi','.gvp','.ifo','.imovieproj','.imovieproject','.ivf','.ivs','.izz','.izzy','.lsf','.lsx','.m1v','.m21','.m2v','.m4e','.m4u','.m4v','.mjp','.mkv','.mod','.moov','.mov','.movie','.mp21','.mp4','.mpe','.mpeg','.mpg','.mpv2','.mqv','.msh','.mswmm','.mvb','.mvc','.nsv','.nvc','.ogm','.pds','.piv','.playlist','.pro','.prproj','.prx','.qt','.qtch','.qtz','.rm','.rmvb','.rp','.rts','.rts','.sbk','.scm','.scm','.sfvidcap','.smil','.smv','.spl','.srt','.ssm','.str','.svi','.swf','.swi','.tda3mt','.tivo','.ts','.vdo','.veg','.vf','.vfw','.vid','.viewlet','.viv','.vivo','.vob','.vp6','.vp7','.vro','.w32','.wcp','.wm','.wmd','.wmv','.wmx','.wvx','.yuv'])

OTHER_BINARY_FILE_FILTER=['*','*.*']

#-----------------------------------------------------------------------------------------
# mp3 metadata
#-----------------------------------------------------------------------------------------
import string

MPG_MD_STEREO                = 0
MPG_MD_JOINT_STEREO     = 1
MPG_MD_DUAL_CHANNEL  = 2
MPG_MD_MONO                 = 3

t_bitrate = [
  [
    [0,32,48,56,64,80,96,112,128,144,160,176,192,224,256],
    [0,8,16,24,32,40,48,56,64,80,96,112,128,144,160],
    [0,8,16,24,32,40,48,56,64,80,96,112,128,144,160]
    ],
  [
    [0,32,64,96,128,160,192,224,256,288,320,352,384,416,448],
    [0,32,48,56,64,80,96,112,128,160,192,224,256,320,384],
    [0,32,40,48,56,64,80,96,112,128,160,192,224,256,320]
    ]
  ]

t_sampling_freq = [
  [22050, 24000, 16000],
  [44100, 48000, 32000]
  ]

frequency_tbl = {0:22050,1:24000,2:16000,3:44100,4:48000,5:32000,6:64000}
#-----------------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------


#--------------------------------------------------------------------------------------------------
#--------------------------------------------------------------------------------------------------
class SizePersistedDialog(QDialog):

    initial_extra_size = QSize(300, 300)

    def __init__(self, parent, unique_pref_name):
        QDialog.__init__(self, parent)
        self.unique_pref_name = unique_pref_name
        self.geom = gprefs.get(unique_pref_name, None)

    def resize_dialog(self):

        if self.geom is None:
            self.resize(self.sizeHint()+self.initial_extra_size)
        else:
            self.restoreGeometry(self.geom)

    def save_dialog_geometry(self):
        geom = bytearray(self.saveGeometry())
        gprefs[self.unique_pref_name] = geom
#--------------------------------------------------------------------------------------------------
#--------------------------------------------------------------------------------------------------
class MediaFileImporterDialog(SizePersistedDialog):

    def __init__(self,maingui,guidb,icon,tmp_directory):

        parent = None
        unique_pref_name = 'mediafileimporter:gui_parameters_dialog'
        SizePersistedDialog.__init__(self, parent, unique_pref_name)
        self.maingui = maingui
        self.guidb = guidb
        self.icon = icon
        self.lib_path = self.maingui.library_view.model().db.library_path
        self.tmp_directory = tmp_directory

        for k,v in iteritems(prefs.defaults):
            if not k in prefs:
                prefs[k] = v
        #END FOR
        prefs

        #-----------------------------------------------------
        #-----------------------------------------------------
        #-----------------------------------------------------
        #-----------------------------------------------------

        QWidget.__init__(self)

        self.setAcceptDrops(True)

        self.setWindowTitle('Media File Importer')
        self.setWindowIcon(icon)

        #-----------------------------------------------------
        #-----------------------------------------------------
        self.layout_mfi = QVBoxLayout()
        self.layout_mfi.setAlignment(Qt.AlignCenter)
        self.setLayout(self.layout_mfi)
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.scroll_area_frame = QScrollArea()
        self.scroll_area_frame.setAlignment(Qt.AlignCenter)
        self.scroll_area_frame.setWidgetResizable(True)
        self.scroll_area_frame.ensureVisible(600,600)

        self.layout_mfi.addWidget(self.scroll_area_frame)       # the scroll area is now the child of the parent of self.layout_mfi

        # NOTE: the self.scroll_area_frame.setWidget(self.scroll_widget) is at the end of the init() AFTER all children have been created and assigned to a layout...
        #-----------------------------------------------------
        self.scroll_widget = QWidget()
        self.layout_mfi.addWidget(self.scroll_widget)           # causes automatic reparenting of QWidget to the parent of self.layout_mfi, which is:  self .
        #-----------------------------------------------------
        self.layout_top0 = QVBoxLayout()
        self.layout_top0.setSpacing(0)
        self.layout_top0.setAlignment(Qt.AlignCenter)
        #-----------------------------------------------------
        self.scroll_widget.setLayout(self.layout_top0)        # causes automatic reparenting of any widget later added below to the above parent
        #-----------------------------------------------------

        self.setStyleSheet("QToolTip { color: #ffffff; background-color: #2a82da; border: 1px solid white; }")

        s = "<p style='white-space:wrap'>If you drop a file's full path onto the Drop Target, MFI will create a new book in Calibre using the specified metadata for all created books.<br><br> The Calibre book format that will be created is always '.txt', and will be empty.  You may use it in any manner that you wish."
        self.setToolTip(s)

        font = QFont()
        font.setBold(False)
        font.setPointSize(10)

        #~ ------------------------------------------------------------

        self.top_groupbox = QGroupBox('')
        self.layout_top0.addWidget(self.top_groupbox)
        self.top_groupbox.setMinimumWidth(525)
        #~ self.top_groupbox.setMaximumWidth(525)

        self.layout_top = QVBoxLayout()
        self.layout_top.setAlignment(Qt.AlignCenter)

        self.top_groupbox.setLayout(self.layout_top)

        #-----------------------------------------------------
        #-----------------------------------------------------

        s = "<p style='white-space:wrap'>These values may be used to set the metadata for all of the empty books created as a result of Dropping file paths onto the Drop Target below."
        s = s + "<br><br>Leave a value blank to deactivate a particular metadata option.  However, Author and Title are always required, and will be defaulted if left blank.  Do not leave blank."
        s = s + "<br><br><br>Dynamic variables may be used in lieu of specific text as values for most of the metadata columns."
        s = s + "<br><br>{path} means 'use the full path of the files' as the metadata value."
        s = s + "<br><br><br>{directory_path:n} means 'use the full path of the files, but only down to the nth directory above the file itself in the full path' as the metadata value.  0 means the final directory in which the file itself is contained.  So, {directory_path:0} = (Path - File Name).  Valid values: 0 - 9. "
        s = s + "<br><br>{directory_name:n} means 'use the name of the nth directory above the file itself in the full path' as the metadata value.  0 means the final directory in which the file itself is contained.  Valid values: 0 - 9.  This variable does not yield a valid path.  Example: you have a directory named 'Vacations'."
        s = s + "<br><br>{file_name} means 'use the file name (with extension) of the file itself' as the metadata value."
        s = s + "<br><br>{file_name_no_extension} means 'use the file name with no extension' as the metadata value."
        s = s + "<br><br>{file_extension} means 'use the file extension of the file itself' as the metadata value."
        s = s + "<br><br>Example: '{directory_path:0}/{file_name_no_extension}.xyz' means 'use the file path with no extension, then add a new file extension of .xyz' as the metadata value."
        s = s + "<br><br>Example: '{directory_path:0}/coverart.jpg' means 'use the file named coverart.jpg that is in the same directory as the source file' as the metadata value."
        s = s + "<br><br><br>{image_exif_data} means 'get the EXIF (Exchangeable Image File Format) data from the image (if it exists)' as the metadata value."
        s = s + "<br><br>{image_exif_date_time} means 'get the EXIF DateTime from the image (if it exists)' as the metadata value."
        s = s + "<br><br>{image_exif_date_time_original} means 'get the EXIF DateTimeOriginal from the image (if it exists)' as the metadata value."
        s = s + "<br><br>{image_exif_date_time_digitized} means 'get the EXIF DateTimeDigitized from the image (if it exists)' as the metadata value."
        s = s + "<br><br><br>{today} means 'the current date in the format YYYY-MM-DD' as the metadata value."
        s = s + "<br><br><br>Other Dynamic Variables that are more specialized, such as '{embroidery_file_info}', will be found in the Home Page for MFI."
        s = s + "<br><br><br>Up to four (4) Custom Column Search Names (including the leading '#') may be specified.  Leave the name and value blank to deactivate a Custom Column option.  Note: CC datatypes of 'composite' and 'rating' may not be used."
        s = s + "<br><br>"

        self.top_groupbox.setToolTip(s)

        #-----------------------------------------------------
        #-----------------------------------------------------
        #-----------------------------------------------------
        #-----------------------------------------------------

        self.mode_groupbox = QGroupBox('Mode:')
        self.mode_groupbox.setToolTip("<p style='white-space:wrap'>Specify the type of Media Files that you will be adding.  MFI will enter the selected 'Mode' to tailor its functionality accordingly.\
                                                                                                              <br><br>If you Drag and Drop or Select a mixture of Media Files, only those files with file extensions for the type indicated here will be processed.  All others will be ignored.")
        self.mode_groupbox.setMinimumHeight(75)
        #~ self.mode_groupbox.setMaximumHeight(75)
        self.layout_top.addWidget(self.mode_groupbox)

        self.layout_mode = QVBoxLayout()
        self.layout_mode.setAlignment(Qt.AlignCenter)

        self.mode_groupbox.setLayout(self.layout_mode)

        self.mode_layout = QHBoxLayout()
        self.mode_layout.setAlignment(Qt.AlignCenter)
        self.layout_mode.addLayout(self.mode_layout)

        self.mode_images_radio = QRadioButton('Images')
        self.mode_images_radio.setToolTip("<p style='white-space:wrap'>Image files with the appropriate file extensions will be selected.")
        self.mode_layout.addWidget(self.mode_images_radio)

        self.mode_audio_radio = QRadioButton('Audio')
        self.mode_audio_radio.setToolTip("<p style='white-space:wrap'>Audio files with the appropriate file extensions will be selected.")
        self.mode_layout.addWidget(self.mode_audio_radio)

        self.mode_video_radio = QRadioButton('Video')
        self.mode_video_radio.setToolTip("<p style='white-space:wrap'>Video files with the appropriate file extensions will be selected.")
        self.mode_layout.addWidget(self.mode_video_radio)

        self.mode_binary_radio = QRadioButton('Other Binary')
        self.mode_binary_radio.setToolTip("<p style='white-space:wrap'>All files will be selected, regardless of type of file.")
        self.mode_layout.addWidget(self.mode_binary_radio)

        self.mode = prefs['MODE']
        if self.mode == "image":
            self.mode_images_radio.setChecked(True)
        elif self.mode == "audio":
            self.mode_audio_radio.setChecked(True)
        elif self.mode == "video":
            self.mode_video_radio.setChecked(True)
        elif self.mode == "binary":
            self.mode_binary_radio.setChecked(True)
        else:
            self.mode_images_radio.setChecked(True)
        #-----------------------------------------------------
        #-----------------------------------------------------
        #-----------------------------------------------------
        #-----------------------------------------------------
        font.setBold(True)
        font.setPointSize(12)

        msg = '<font color = "red">...Drag and Drop Prohibited...</font>'
        self.messages_label = QLabel(msg)
        self.messages_label.setAlignment(Qt.AlignCenter)
        self.messages_label.setTextFormat(Qt.RichText)
        self.messages_label.setFont(font)
        self.layout_top.addWidget(self.messages_label)
        #-----------------------------------------------------
        #-----------------------------------------------------
        #-----------------------------------------------------
        #-----------------------------------------------------
        font.setBold(False)
        font.setPointSize(10)

        self.options_groupbox = QGroupBox('Metadata for Media Files:')
        self.options_groupbox.setToolTip("<p style='white-space:wrap'>All Calibre books added by MFI will be assigned the metadata specified in this section.")
        self.options_groupbox.setMinimumHeight(500)
        self.options_groupbox.setMinimumWidth(500)
        #~ self.options_groupbox.setMaximumWidth(500)

        self.layout_top.addWidget(self.options_groupbox)

        self.layout_options = QGridLayout()

        self.layout_options.setVerticalSpacing(1)

        self.options_groupbox.setLayout(self.layout_options)

        r = 1

        self.option_titles_label = QLabel('Title:')
        self.layout_options.addWidget(self.option_titles_label,r,0,r,1)

        self.option_titles_to_add_lineedit = QLineEdit(self)
        self.option_titles_to_add_lineedit.setText(prefs['TITLE'])
        self.option_titles_to_add_lineedit.setToolTip("<p style='white-space:wrap'>If you wish to specify an title for these empty books while they are being added to Calibre, specify it here.")
        self.layout_options.addWidget(self.option_titles_to_add_lineedit,r,1,r,2)

        r = r + 1

        self.option_authors_label = QLabel('Authors:')
        self.layout_options.addWidget(self.option_authors_label,r,0,r,1)

        self.option_authors_to_add_lineedit = QLineEdit(self)
        self.option_authors_to_add_lineedit.setText(prefs['AUTHORS'])
        self.option_authors_to_add_lineedit.setToolTip("<p style='white-space:wrap'>If you wish to specify an Author for these empty books while they are being added to Calibre, specify it here.")
        self.layout_options.addWidget(self.option_authors_to_add_lineedit,r,1,r,2)

        r = r + 1

        self.option_series_label = QLabel('Series:')
        self.layout_options.addWidget(self.option_series_label,r,0,r,1)

        self.option_series_to_add = QLineEdit(self)
        self.option_series_to_add.setText(prefs['SERIES'])
        self.option_series_to_add.setToolTip("<p style='white-space:wrap'>If you wish to specify a Series Name for these empty books while they are being added to Calibre, specify it here.")
        self.layout_options.addWidget(self.option_series_to_add,r,1,r,2)

        r = r + 1

        self.option_publisher_label = QLabel('Publisher:')
        self.layout_options.addWidget(self.option_publisher_label,r,0,r,1)

        self.option_publisher_to_add = QLineEdit(self)
        self.option_publisher_to_add.setText(prefs['PUBLISHER'])
        self.option_publisher_to_add.setToolTip("<p style='white-space:wrap'>If you wish to specify a Publisher for these empty books while they are being added to Calibre, specify it here.")
        self.layout_options.addWidget(self.option_publisher_to_add,r,1,r,2)

        r = r + 1

        self.option_tags_label = QLabel('Tags:')
        self.layout_options.addWidget(self.option_tags_label,r,0,r,1)

        self.option_tags_to_add_lineedit = QLineEdit(self)
        self.option_tags_to_add_lineedit.setText(prefs['TAGS'])
        self.option_tags_to_add_lineedit.setToolTip("<p style='white-space:wrap'>If you wish to add a Tag to these empty books while they are being added to Calibre, specify it here.")
        self.layout_options.addWidget(self.option_tags_to_add_lineedit,r,1,r,2)

        r = r + 1

        self.option_identifier_label = QLabel("Identifier 'zmfi':")
        self.layout_options.addWidget(self.option_identifier_label,r,0,r,1)

        self.option_identifier_to_add_lineedit = QLineEdit(self)
        self.option_identifier_to_add_lineedit.setText(prefs['IDENTIFIER'])
        self.option_identifier_to_add_lineedit.setToolTip("<p style='white-space:wrap'>If you wish to add a 'zmfi' Identifier to these empty books while they are being added to Calibre, specify the value for that identifier here.")
        self.layout_options.addWidget(self.option_identifier_to_add_lineedit,r,1,r,2)

        r = r + 1

        self.option_comments_label = QLabel('Comments:')
        self.layout_options.addWidget(self.option_comments_label,r,0,r,1)

        self.option_comments_to_add_lineedit = QLineEdit(self)
        self.option_comments_to_add_lineedit.setText(prefs['COMMENTS'])
        self.option_comments_to_add_lineedit.setToolTip("<p style='white-space:wrap'>If you wish to add comments to these empty books while they are being added to Calibre, specify the comments here.")
        self.layout_options.addWidget(self.option_comments_to_add_lineedit,r,1,r,2)

        r = r + 1

        self.option_cover_label = QLabel('Cover Path: ')
        self.layout_options.addWidget(self.option_cover_label,r,0,r,1)

        self.option_cover_to_add_lineedit = QLineEdit(self)
        self.option_cover_to_add_lineedit.setText(prefs['COVER_PATH'])
        self.option_cover_to_add_lineedit.setToolTip("<p style='white-space:wrap'>If you wish to add a Cover for these empty books while they are being added to Calibre, specify it here.")
        self.layout_options.addWidget(self.option_cover_to_add_lineedit,r,1,r,2)

        #--------------------------------------------------
        #--------------------------------------------------
        autocompletion_field_list = []
        tmp_list = self.guidb.custom_field_keys()
        for row in tmp_list:
            autocompletion_field_list.append(row)
        #END FOR
        del tmp_list
        autocompletion_field_list.sort()
        #--------------------------------------------------
        #--------------------------------------------------
        self.option_cc1_completer = QCompleter(autocompletion_field_list)
        self.option_cc1_completer.setCompletionMode(QCompleter.CompletionMode.UnfilteredPopupCompletion)
        self.option_cc1_completer.setCaseSensitivity(Qt.CaseInsensitive)
        self.option_cc1_completer.setMaxVisibleItems(50)
        #--------------------------------------------------
        #--------------------------------------------------

        r = r + 1

        self.option_custom_column_1_to_add_lineedit = QLineEdit(self)
        self.option_custom_column_1_to_add_lineedit.setMinimumWidth(150)
        #~ self.option_custom_column_1_to_add_lineedit.setMaximumWidth(150)
        self.option_custom_column_1_to_add_lineedit.setText(prefs['CUSTOM_COLUMN_1'])
        self.option_custom_column_1_to_add_lineedit.setToolTip("<p style='white-space:wrap'>Optional: If you wish to add a Custom Column value to these empty books while they are being added to Calibre, specify it here.")
        self.layout_options.addWidget(self.option_custom_column_1_to_add_lineedit,r,0,r,1)

        self.option_custom_column_1_to_add_lineedit.setCompleter(self.option_cc1_completer)

        self.option_custom_column_1_to_add_value_lineedit = QLineEdit(self)
        self.option_custom_column_1_to_add_value_lineedit.setText(prefs['CUSTOM_COLUMN_1_VALUE'])
        self.option_custom_column_1_to_add_value_lineedit.setToolTip("<p style='white-space:wrap'>Optional: If you wish to add a Custom Column value to these empty books while they are being added to Calibre, specify it here.")
        self.layout_options.addWidget(self.option_custom_column_1_to_add_value_lineedit,r,1,r,2)

        #--------------------------------------------------
        #--------------------------------------------------
        self.option_cc2_completer = QCompleter(autocompletion_field_list)
        self.option_cc2_completer.setCompletionMode(QCompleter.CompletionMode.UnfilteredPopupCompletion)
        self.option_cc2_completer.setCaseSensitivity(Qt.CaseInsensitive)
        self.option_cc2_completer.setMaxVisibleItems(50)
        #--------------------------------------------------
        #--------------------------------------------------

        r = r + 1

        self.option_custom_column_2_to_add_lineedit = QLineEdit(self)
        self.option_custom_column_2_to_add_lineedit.setMinimumWidth(150)
        #~ self.option_custom_column_2_to_add_lineedit.setMaximumWidth(150)
        self.option_custom_column_2_to_add_lineedit.setText(prefs['CUSTOM_COLUMN_2'])
        self.option_custom_column_2_to_add_lineedit.setToolTip("<p style='white-space:wrap'>Optional: If you wish to add a Custom Column value to these empty books while they are being added to Calibre, specify it here.")
        self.layout_options.addWidget(self.option_custom_column_2_to_add_lineedit,r,0,r,1)

        self.option_custom_column_2_to_add_lineedit.setCompleter(self.option_cc2_completer)

        self.option_custom_column_2_to_add_value_lineedit = QLineEdit(self)
        self.option_custom_column_2_to_add_value_lineedit.setText(prefs['CUSTOM_COLUMN_2_VALUE'])
        self.option_custom_column_2_to_add_value_lineedit.setToolTip("<p style='white-space:wrap'>Optional: If you wish to add a Custom Column value to these empty books while they are being added to Calibre, specify it here.")
        self.layout_options.addWidget(self.option_custom_column_2_to_add_value_lineedit,r,1,r,2)


        #--------------------------------------------------
        #--------------------------------------------------
        self.option_cc3_completer = QCompleter(autocompletion_field_list)
        self.option_cc3_completer.setCompletionMode(QCompleter.CompletionMode.UnfilteredPopupCompletion)
        self.option_cc3_completer.setCaseSensitivity(Qt.CaseInsensitive)
        self.option_cc3_completer.setMaxVisibleItems(50)
        #--------------------------------------------------
        #--------------------------------------------------

        r = r + 1

        self.option_custom_column_3_to_add_lineedit = QLineEdit(self)
        self.option_custom_column_3_to_add_lineedit.setMinimumWidth(150)
        #~ self.option_custom_column_3_to_add_lineedit.setMaximumWidth(150)
        self.option_custom_column_3_to_add_lineedit.setText(prefs['CUSTOM_COLUMN_3'])
        self.option_custom_column_3_to_add_lineedit.setToolTip("<p style='white-space:wrap'>Optional: If you wish to add a Custom Column value to these empty books while they are being added to Calibre, specify it here.")
        self.layout_options.addWidget(self.option_custom_column_3_to_add_lineedit,r,0,r,1)

        self.option_custom_column_3_to_add_lineedit.setCompleter(self.option_cc3_completer)

        self.option_custom_column_3_to_add_value_lineedit = QLineEdit(self)
        self.option_custom_column_3_to_add_value_lineedit.setText(prefs['CUSTOM_COLUMN_3_VALUE'])
        self.option_custom_column_3_to_add_value_lineedit.setToolTip("<p style='white-space:wrap'>Optional: If you wish to add a Custom Column value to these empty books while they are being added to Calibre, specify it here.")
        self.layout_options.addWidget(self.option_custom_column_3_to_add_value_lineedit,r,1,r,2)

        #--------------------------------------------------
        #--------------------------------------------------
        #--------------------------------------------------
        #--------------------------------------------------
        self.option_cc4_completer = QCompleter(autocompletion_field_list)
        self.option_cc4_completer.setCompletionMode(QCompleter.CompletionMode.UnfilteredPopupCompletion)
        self.option_cc4_completer.setCaseSensitivity(Qt.CaseInsensitive)
        self.option_cc4_completer.setMaxVisibleItems(50)
        #--------------------------------------------------
        #--------------------------------------------------

        r = r + 1

        self.option_custom_column_4_to_add_lineedit = QLineEdit(self)
        self.option_custom_column_4_to_add_lineedit.setMinimumWidth(150)
        #~ self.option_custom_column_4_to_add_lineedit.setMaximumWidth(150)
        self.option_custom_column_4_to_add_lineedit.setText(prefs['CUSTOM_COLUMN_4'])
        self.option_custom_column_4_to_add_lineedit.setToolTip("<p style='white-space:wrap'>Optional: If you wish to add a Custom Column value to these empty books while they are being added to Calibre, specify it here.")
        self.layout_options.addWidget(self.option_custom_column_4_to_add_lineedit,r,0,r,1)

        self.option_custom_column_4_to_add_lineedit.setCompleter(self.option_cc4_completer)

        self.option_custom_column_4_to_add_value_lineedit = QLineEdit(self)
        self.option_custom_column_4_to_add_value_lineedit.setText(prefs['CUSTOM_COLUMN_4_VALUE'])
        self.option_custom_column_4_to_add_value_lineedit.setToolTip("<p style='white-space:wrap'>Optional: If you wish to add a Custom Column value to these empty books while they are being added to Calibre, specify it here.")
        self.layout_options.addWidget(self.option_custom_column_4_to_add_value_lineedit,r,1,r,2)

        #--------------------------------------------------
        #--------------------------------------------------

        r = r + 1

        self.copy_file_to_calibre_folder_checkbox = QCheckBox("Copy File to Calibre?")
        self.copy_file_to_calibre_folder_checkbox.setToolTip("<p style='white-space:wrap'>Do you want to copy the original source file into the same directory as the new Calibre book being created?")
        self.layout_options.addWidget(self.copy_file_to_calibre_folder_checkbox,r,0,r,0)

        if prefs['COPY_ORIGINAL_FILE_TOO'] == unicode_type("True"):
            self.copy_file_to_calibre_folder_checkbox.setChecked(True)

        self.keep_original_path_identifier_checkbox = QCheckBox("Keep 'Original Path' Identifier?")
        self.keep_original_path_identifier_checkbox.setToolTip("<p style='white-space:wrap'>MFI temporarily creates an Identifier for each book with its original source file path.  When MFI has finished updating each book's metadata, that temporary identifier, 'zmfi_origpath', is deleted.  If you want to keep that Identifier, then check this checkbox.")
        self.layout_options.addWidget(self.keep_original_path_identifier_checkbox,r,1,r,1)

        if prefs['KEEP_ORIGINAL_PATH_IDENTIFIER'] == unicode_type("True"):
            self.keep_original_path_identifier_checkbox.setChecked(True)

        r = r + 1

        self.copy_same_filenames_to_same_folder_checkbox = QCheckBox("Copy Same 'Paths Excluding File Extensions' to Same Calibre Directory?")
        self.copy_same_filenames_to_same_folder_checkbox.setToolTip("<p style='white-space:wrap'>Do you want all media files that have the identical paths except for the final file extension to be treated as a single collection and copied to the same Calibre subdirectory?  Example:  you have the identical photos in several formats (.jpg, .png, .bmp), so you want them to be in the same 'book' in Calibre.")
        self.layout_options.addWidget(self.copy_same_filenames_to_same_folder_checkbox,r,0,r,0)

        if prefs['COPY_SAME_BASE_FILENAMES_TO_SAME_CALIBRE_DIRECTORY'] == unicode_type("True"):
            self.copy_same_filenames_to_same_folder_checkbox.setChecked(True)

        r = r + 1

        font.setPointSize(6)

        self.file_extension_priority_sequence_lineedit = QLineEdit(self)
        self.file_extension_priority_sequence_lineedit.setText(prefs['FILE_EXTENSION_PRIORITY_SEQUENCE'])
        self.file_extension_priority_sequence_lineedit.setAlignment(Qt.AlignLeft)
        self.file_extension_priority_sequence_lineedit.setToolTip("<p style='white-space:wrap'>Active only with the checkbox just above has been 'checked'.  From left-to-right, <b>optionally</b> specify the priority sequence of the <b>most important</b> (not all) file extensions for processing different file extensions for files otherwise having the identical name and path.\
                                                                                                    <br><br>The format is:  'Mode=.xyz;'  where 'Mode' would be 'Image', 'Audio', 'Video' and 'Binary'.  The '.xyz' would be the file extension, including the dot.  A semi-colon (';') must follow the file extension.  Do not use single or double quotes, or any other punctuation marks.   \
                                                                                                    <br><br>Example:  you have embroidery design files that are for the identical design.  However, you want .pcs files to be imported first since they have Color Data, whereas the .hus and .vip files do not.  Otherwise, the Color Data would not appear in the comments.\
                                                                                                    <br><br>Example:  Image=.jpg;Image=.png;Audio=.mp3;Audio=.acc;Video=.mkv;Video=.mp4;Video=.avi;Binary=.pcs;Binary=.hus     ")
        self.layout_options.addWidget(self.file_extension_priority_sequence_lineedit,r,0,r,0)

        font.setPointSize(10)

        r = r + 1

        self.push_button_save_validate = QPushButton(" ", self)
        self.push_button_save_validate.setText("Validate, Save && Lock Options")
        self.push_button_save_validate.clicked.connect(self.validate_and_save_settings)
        self.push_button_save_validate.setDefault(True)
        self.push_button_save_validate.setFont(font)
        #~ self.push_button_save_validate.setMinimumWidth(200)
        #~ self.push_button_save_validate.setMaximumWidth(200)
        self.push_button_save_validate.setToolTip("<p style='white-space:wrap'>Validate, save, and then lock the current values used above.  If you wish to change any options after they are validated and locked, you must restart MFI.")
        self.layout_options.addWidget(self.push_button_save_validate,r,0,r,0)

        #-----------------------------------------------------
        #-----------------------------------------------------

        self.bottom_layout = QHBoxLayout()
        self.bottom_layout.setAlignment(Qt.AlignCenter)
        self.layout_top.addLayout(self.bottom_layout)

        self.push_button_select_files_by_directory = QPushButton(" ", self)
        self.push_button_select_files_by_directory.setText("Select Media Files")
        self.push_button_select_files_by_directory.clicked.connect(self.select_files_by_directory)
        self.push_button_select_files_by_directory.setFont(font)
        self.push_button_select_files_by_directory.setMinimumWidth(300)
        #~ self.push_button_select_files_by_directory.setMaximumWidth(300)
        self.push_button_select_files_by_directory.setToolTip("<p style='white-space:wrap'>Select Files using a File Selection dialog.")
        self.bottom_layout.addWidget(self.push_button_select_files_by_directory)

        self.push_button_select_files_by_directory.hide()

        self.push_button_exit = QPushButton(" ", self)
        self.push_button_exit.setText("Exit")
        self.push_button_exit.clicked.connect(self.mfi_exit)
        self.push_button_exit.setFont(font)
        self.push_button_exit.setMinimumWidth(75)
        self.push_button_exit.setMaximumWidth(75)
        self.push_button_exit.setToolTip("<p style='white-space:wrap'>Exit")
        self.bottom_layout.addWidget(self.push_button_exit)

        #-----------------------------------------------------
        #-----------------------------------------------------
        self.options_groupbox.resize(self.sizeHint())
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.scroll_widget.resize(self.sizeHint())
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.scroll_area_frame.setWidget(self.scroll_widget)    # now that all widgets have been created and assigned to a layout...
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.scroll_area_frame.resize(self.sizeHint())
        #-----------------------------------------------------
        #-----------------------------------------------------

        self.resize(self.sizeHint())

        self.resize_dialog()

        self.parameter_execution_set = set()

        self.build_custom_column_dicts()

        self.okay_to_drop_files = False

        self.mfi_books_just_added_list = []

        self.show()
    #----------------------------------------------------
    #----------------------------------------------------
    #----------------------------------------------------
    def mfi_exit(self):
        if len(self.mfi_books_just_added_list) > 0:
           self.refresh_cache_and_library_view()
        self.hide()
        msg = ('MFI has exited normally.')
        self.maingui.status_bar.showMessage(msg)
        if DEBUG: print(msg)
        self.maingui.library_view.model().start_metadata_backup()
        self.close()
    #----------------------------------------------------
    #----------------------------------------------------
    def refresh_cache_and_library_view(self):
        QApplication.instance().processEvents()
        self.maingui.library_view.model().refresh()   #refreshes cache from db
        db = self.maingui.current_db.new_api
        db.reload_from_db(clear_caches=False)
        db.refresh_format_cache()  #otherwise, only empty book TXT format will show prior to next restart...
        frozen = db.all_book_ids()
        books = list(frozen)
        self.maingui.iactions['Edit Metadata'].refresh_books_after_metadata_edit(books)  #refreshes books in library view from cache
        self.maingui.tags_view.recount()
        QApplication.instance().processEvents()

    #----------------------------------------------------
    #----------------------------------------------------
    def dragEnterEvent(self, e):

        if not self.okay_to_drop_files:
            return

        if e.mimeData().hasText():
            e.accept()
        elif e.mimeData().hasUrls():
            e.reject()
        elif e.mimeData().hasHtml():
            e.reject()
        elif e.mimeData().hasImage():
            e.reject()
        else:
            formats = e.mimeData.formats()
            if DEBUG: print("unknown: formats are:  ", as_unicode(formats))
            e.reject()
    #----------------------------------------------------
    def dropEvent(self, e):

        if not self.okay_to_drop_files:
            return

        self.messages_label.setText("Drop Received; Processing...")
        self.repaint()

        file_dict = {}

        if e.mimeData().hasText():
            #~ if DEBUG: print("e.mimeData().hasText():")
            file_dict['text/plain'] = e.mimeData().text()

        elif e.mimeData().hasUrls():           # Future Use...
            if DEBUG: print("e.mimeData().hasUrls():")
            file_dict['text/uri-list'] = e.mimeData().urls()

        self.messages_label.setText("")
        self.repaint()

        if len(file_dict) > 0:
            self.create_books_control(file_dict)
            self.mfi_exit()  # user must click the MFI icon if they want to import more items...

        del file_dict
    #----------------------------------------------------
    def create_books_control(self,file_dict,dragndrop=True):

        self.okay_to_drop_files = False
        self.push_button_select_files_by_directory.hide()
        self.push_button_exit.hide()

        self.messages_label.setText("Creating new Calibre books...wait...")
        self.repaint()

        mfi_list = []

        if dragndrop:
            #~ for mimetype,getterdata in file_dict.iteritems():
            for mimetype,getterdata in iteritems(file_dict):
                #~ if DEBUG: print(mimetype,getterdata)
                file_path_list = self.extract_data_control(mimetype,getterdata)
                file_path_list.sort()
                mfi_list.append(file_path_list)
            #END FOR
        else:
            file_path_list = []
            #~ for path,v in file_dict.iteritems():
            for path,v in iteritems(file_dict):
                file_path_list.append(path)
            #END FOR
            file_path_list.sort()
            mfi_list.append(file_path_list)

        msg = ('Calibre is adding media file books')
        self.maingui.status_bar.showMessage(msg)

        msg = '<font color = "green">...Adding Media File Books...</font>'
        self.messages_label.setText(msg)
        self.repaint()

        self.source_file_list = []

        for file_path_list in mfi_list:
            for paths in file_path_list:
                if isinstance(paths,list):
                    for path in paths:
                        self.source_file_list.append(path)
                else:
                    self.source_file_list.append(paths)
        #END FOR

        del mfi_list

        if prefs['COPY_SAME_BASE_FILENAMES_TO_SAME_CALIBRE_DIRECTORY'] == unicode_type("True"):
            self.sort_source_files_by_priority()
            self.source_file_dict_in_use = True
            self.source_file_dict = {}
            self.source_file_path_dict = {}
            # enable identical paths excluding the file extension to be treated as a collection
            for path in self.source_file_list:
                file_name,file_extension = os.path.splitext(path) # C:/Users/xxxx/Pictures/uuuuuu/14752     .jpg   <<== C:/Users/xxxx/Pictures/uuuuuu/14752.jpg
                self.source_file_path_dict[path] = file_name,file_extension
                if DEBUG: print("file_name,file_extension", "    ", file_name, file_extension)
                if file_name in self.source_file_dict:
                    tmp_list = self.source_file_dict[file_name]
                    tmp_list = self.convert_string_to_list(tmp_list)
                    if not tmp_list:
                        if DEBUG: print("tmp_list data error; not a list for: ", path)
                        continue
                    tmp_list.append(file_extension)
                    tmp_list.sort()
                    self.source_file_dict[file_name] = as_unicode(tmp_list)
                    if DEBUG:
                        for row in tmp_list:
                            print("row in tmp_list: ", row)
                    del tmp_list
                else:
                    tmp_list = []
                    tmp_list.append(file_extension)
                    self.source_file_dict[file_name] = as_unicode(tmp_list)
                    if DEBUG: print("tmp_list: ", as_unicode(tmp_list))
                    del tmp_list
            #END FOR
        else:
            self.source_file_dict_in_use = False

        self.empty_txt_file_dir = self.tmp_directory  # passed from ui.py
        if isbytestring(self.empty_txt_file_dir):
            self.empty_txt_file_dir = self.empty_txt_file_dir.decode(filesystem_encoding)
        self.empty_txt_file_dir = self.empty_txt_file_dir.replace(os.sep,'/')

        if DEBUG: print("self.empty_txt_file_dir before adding books",self.empty_txt_file_dir)

        self.temporary_file_list = []
        self.mfi_books_just_added_list = []

        self.run_calibre_add_books()

        msg = ('MFI is now updating the Calibre metadata of the new books')
        self.maingui.status_bar.showMessage(msg)
        if DEBUG: print(msg)

        msg = '<font color = "green">...Updating Metadata of the New Books...</font>'
        self.messages_label.setText(msg)
        self.repaint()

        sleep(0)

        self.update_metadata_of_new_books()

        if prefs['COPY_ORIGINAL_FILE_TOO'] == unicode_type("True"):
            msg = ('MFI is copying the original media files to Calibre')
            self.maingui.status_bar.showMessage(msg)
            if DEBUG: print(msg)
            msg = '<font color = "green">...Copying Original Files to Calibre...</font>'
            self.messages_label.setText(msg)
            self.repaint()
            self.copy_original_files_too()

        msg = ('MFI has finished processing the media files')
        self.maingui.status_bar.showMessage(msg)
        if DEBUG: print(msg)

        msg = '<font color = "green">...MFI is finishing...</font>'
        self.messages_label.setText(msg)
        self.repaint()

        try:
            for path in self.temporary_file_list:
                if path is not None:
                    if os.path.isfile(path):
                        os.remove(path)
                        #~ if DEBUG: print("Removing self.temporary_file_list: ", path)
            #END FOR
        except Exception as e:
            if DEBUG: print("Exception in removing self.temporary_file_list: ", as_unicode(e))

        msg = ('MFI has removed its temporary files')
        self.maingui.status_bar.showMessage(msg)
        if DEBUG: print(msg)
    #~ #----------------------------------------------------
    def sort_source_files_by_priority(self):

        priority_sequence = prefs['FILE_EXTENSION_PRIORITY_SEQUENCE']  #  Image=.jpg;Image=.png;Audio=.mp3;Audio=.acc;Video=.mkv;Video=.mp4;Video=.avi;Binary=.pcs;Binary=.hus
        priority_sequence = priority_sequence.strip()
        if len(priority_sequence) == 0:
            return

        priority_dict = {}   # file extension = priority integer
        if self.mode_is_image:
            priority_sequence = priority_sequence.lower()
            priority_sequence = priority_sequence.replace("image",";image")  # in case the user forgot it...
            s_split = priority_sequence.split(";")
            i = 1000
            for row in s_split:
                s = row.strip()
                if  s > " ":
                    if s.count("image") > 0:
                        s = s.replace("image=","")
                        s = s.strip()
                        s = s.lower()
                        priority_dict[s] = i
                        s = s.upper()
                        priority_dict[s] = i
                        s = s.title()
                        priority_dict[s] = i
                        i = i - 1
            #END FOR
        elif self.mode_is_audio:
            priority_sequence = priority_sequence.lower()
            priority_sequence = priority_sequence.replace("audio",";audio")  # in case the user forgot it...
            s_split = priority_sequence.split(";")
            i = 1000
            for row in s_split:
                s = row.strip()
                if  s > " ":
                    if s.count("audio") > 0:
                        s = s.replace("audio=","")
                        s = s.strip()
                        s = s.lower()
                        priority_dict[s] = i
                        s = s.upper()
                        priority_dict[s] = i
                        s = s.title()
                        priority_dict[s] = i
                        i = i - 1
            #END FOR
        elif self.mode_is_video:
            priority_sequence = priority_sequence.lower()
            priority_sequence = priority_sequence.replace("video",";video")  # in case the user forgot it...
            s_split = priority_sequence.split(";")
            i = 1000
            for row in s_split:
                s = row.strip()
                if  s > " ":
                    if s.count("video") > 0:
                        s = s.replace("video=","")
                        s = s.strip()
                        s = s.lower()
                        priority_dict[s] = i
                        s = s.upper()
                        priority_dict[s] = i
                        s = s.title()
                        priority_dict[s] = i
                        i = i - 1
            #END FOR
        elif self.mode_is_binary:
            priority_sequence = priority_sequence.lower()
            priority_sequence = priority_sequence.replace("binary",";binary")  # in case the user forgot it...
            s_split = priority_sequence.split(";")
            i = 1000
            for row in s_split:
                s = row.strip()
                if  s > " ":
                    if s.count("binary") > 0:
                        s = s.replace("binary=","")
                        s = s.strip()
                        s = as_unicode(s)
                        s = s.lower()
                        priority_dict[s] = i
                        s = s.upper()
                        priority_dict[s] = i
                        s = s.title()
                        priority_dict[s] = i
                        i = i - 1
            #END FOR
        else:
            return

        self.source_file_list.sort()
        sorted_list = []
        for path in self.source_file_list:
            fname,fext = os.path.splitext(path)
            fext = fext.lower()
            fext = as_unicode(fext)
            if fext in priority_dict:
                priority = priority_dict[fext]
            else:
                priority = 0
            r = priority,path
            sorted_list.append(r)
        #END FOR
        sorted_list = sorted(sorted_list,key=lambda x: x[0], reverse=True)
        del self.source_file_list
        self.source_file_list = []
        for row in sorted_list:
            priority,path = row
            self.source_file_list.append(path)
            if DEBUG: print("priority, path: ", as_unicode(priority),path)
    #----------------------------------------------------
    def convert_string_to_list(self,value):
        value = as_unicode(value)
        newvalue = ast.literal_eval(value)
        if isinstance(newvalue,list):
            return newvalue
        else:
            return None
    #-----------------------------------------------------------------------------------------
    def extract_data_control(self,mimetype,getterdata):

        d_list = []

        if mimetype == "text/plain":
            file_path_list = self.extract_file_paths_text(getterdata)
        elif mimetype == "text/uri-list":
            file_path_list = self.extract_file_paths_urls(getterdata)
        else:
            return d_list

        d_list.append(file_path_list)

        del file_path_list

        return d_list
    #-----------------------------------------------------------------------------------------
    def extract_file_paths_text(self,data):
        file_path_list = []
        if data.count("file:") > 0:     # standard MIME drag-and-drop format  for files:    file:///s:/.....
            file_list = data.split("file:///")
            if len(file_list) > 0:
                for path in file_list:
                    path = path.strip()
                    if path > " ":
                        file_path_list.append(path)
                        #~ if DEBUG: print("raw path: ", path)
                #END FOR
        else:
            if DEBUG: print("no 'file:' in data...",as_unicode(data))  # do nothing, but can inspect the data format if desired...

        del data

        return file_path_list
    #-----------------------------------------------------------------------------------------
    def extract_file_paths_urls(self,data):
        file_path_list = []

        if DEBUG: print("extract_file_paths_urls...to be developed...")

        if isinstance(data,list):
            for url in data:
                if DEBUG: print(url)
            #END FOR

        return  file_path_list
    #-----------------------------------------------------------------------------------------
    def run_calibre_add_books(self):

        self.batch_timestamp = time.asctime( time.localtime(time.time()) )
        self.batch_timestamp = as_unicode(self.batch_timestamp.strip())
        self.batch_timestamp = self.batch_timestamp.replace(" ","_")
        self.batch_timestamp = as_unicode(self.batch_timestamp)

        exception_error_message = ""

        sleep(0)

        try:
            n_total_added = 0
            collection_item_processed_set = set()
            for source_file in self.source_file_list:
                if DEBUG: print("source_file being processed: ", source_file)  #should be sorted by priority...
                if DEBUG: print("self.source_file_dict_in_use", self.source_file_dict_in_use)
                if self.source_file_dict_in_use:
                    if source_file in self.source_file_path_dict:     #~ self.source_file_path_dict[path] = file_name,file_extension
                        file_name,file_extension = self.source_file_path_dict[source_file]
                        if file_name in collection_item_processed_set:
                            continue  # create only 1 book per 'collection' of media files, not 1 per 'each' media file
                        else:
                            collection_item_processed_set.add(file_name)
                msg = '<font color = "black">[FILE]</font>'
                s = "....." + source_file[-45: ]
                msg = msg.replace("[FILE]",s)
                self.messages_label.setText(msg)
                self.repaint()
                sleep(0)
                #~ --------------------------------------------
                title = prefs['TITLE']
                authors = prefs['AUTHORS']
                series = prefs['SERIES']
                tags = prefs['TAGS']
                identifier = prefs['IDENTIFIER']
                cover = prefs['COVER_PATH']
                publisher = prefs['PUBLISHER']
                comments = prefs['COMMENTS']
                #~ --------------------------------------------
                title = self.expand_dynamic_variables(title,source_file)
                authors = self.expand_dynamic_variables(authors,source_file)
                series = self.expand_dynamic_variables(series,source_file)
                tags = self.expand_dynamic_variables(tags,source_file)
                identifier = self.expand_dynamic_variables(identifier,source_file)
                cover = self.expand_dynamic_variables(cover,source_file)
                publisher = self.expand_dynamic_variables(publisher,source_file)
                comments = self.expand_dynamic_variables(comments,source_file)
                #~ --------------------------------------------

                if title > " ":
                    title = as_unicode(' --title ') + '"' + title + '"'
                else:
                    title = as_unicode(' --title ') + as_unicode('"[TTT] "')
                    title = title.replace("[TTT]",path.strip())

                if authors > " ":
                    authors = ' --authors ' + '"'+ authors + '"'
                else:
                    authors = ' --authors MFI '

                if series > " ":
                    series = ' --series ' + '"'+ series + '"'
                else:
                    series = " "

                if tags > " ":
                    #~ if DEBUG: print(tags)
                    tags = tags.replace(";",",")
                    tags = tags.replace("'","")
                    tags = tags.replace('"','')
                    tags = tags.strip()
                    tags = ' --tags ' + '"'+ tags + '"  '
                else:
                    tags = " "

                if identifier > " ":
                    identifier = '  --identifier ' + '"zmfi$'+ identifier + '" ' + '  --identifier ' + '"zmfi_batch$' + self.batch_timestamp +  '"  '
                else:
                    identifier = '  --identifier ' + '"zmfi_batch$' + self.batch_timestamp +  '"  '

                if prefs['COPY_ORIGINAL_FILE_TOO'] == unicode_type("True") or prefs['KEEP_ORIGINAL_PATH_IDENTIFIER']  == unicode_type("True"):
                    if isbytestring(source_file):
                        source_file = source_file.decode(filesystem_encoding)
                    source_file = source_file.replace(os.sep, '/')
                    identifier = identifier + '  --identifier ' + '"zmfi_origpath$' + source_file  +  '"   '

                if cover > " ":
                    cover = ' --cover ' + '"'+ cover + '"'
                else:
                    cover = " "

                if publisher > " ":
                    publisher = ' --publisher ' + '"'+ publisher + '"'
                else:
                    publisher = " "

                if comments > " ":
                    comments = ' --comments ' + '"'+ comments + '"'
                else:
                    comments = " "

                #~ --------------------------------------------
                #~ s_add_param =  authors + as_unicode('|||') + title + str ('|||') + tags + as_unicode('|||') + series + as_unicode('|||') + identifier + as_unicode('|||')+ cover + as_unicode('|||') + empty_file_path
                s_add_param =  authors + as_unicode('|||') + title + as_unicode('|||') + tags + as_unicode('|||') + series + as_unicode('|||') + identifier + as_unicode('|||') + cover + as_unicode('|||')  + publisher + as_unicode('|||')  + comments + as_unicode('|||')   # + empty_file_path appended here
                if DEBUG: print("[0]: ", s_add_param)
                prefix = 'temp_mfi_'
                suffix = ".txt"
                empty_file_path = self.create_empty_txt_file(suffix,prefix,self.empty_txt_file_dir)
                #~ if DEBUG: print("[1]: ", "empty_file_path: ", empty_file_path)
                if isbytestring(empty_file_path):
                    empty_file_path = empty_file_path.decode(filesystem_encoding)
                empty_file_path = empty_file_path.replace(os.sep, '/')
                s_add_param = s_add_param + '  "' + empty_file_path + '"    '        # double quotes
                #~ if DEBUG: print("[2]: ", s_add_param)
                n_total_added = n_total_added + 1
                msg = "Adding Empty Book: " + as_unicode(n_total_added)
                self.maingui.status_bar.showMessage(msg)
                sleep(0)
                #~ ----------------------------------------
                try:
                    is_valid = self.mfi_cli_add_book(s_add_param)    # New for Calibre 3.0.0 ........................................................................
                    if not is_valid:
                        if DEBUG: print("NOT is_valid = mfi_cli_add_book(s_add_param) :", s_add_param)
                except Exception as e:
                    x = "[run_calibre_add_books]ADD TERMINATED PREMATURELY.  "
                    x = x + "ERROR: " +  "  >>>>" + as_unicode(e)
                    if DEBUG: print(x)
                    exception_error_message = exception_error_message + x + "<br><br>"
                #~ ----------------------------------------
            #END FOR

            self.maingui.iactions['Edit Metadata'].refresh_books_after_metadata_edit(self.mfi_books_just_added_list)

        except Exception as e:
            n_total_added = 0
            x = "Calibre:  'ADD' TERMINATED PREMATURELY.  "
            x = x + "<br><br>ERROR: " +  "  >>>>" + as_unicode(e)
            if DEBUG: print(x)
            exception_error_message = exception_error_message + x + "<br><br>"
            try:
                del s_add_param
            except:
                pass
        # -----------------------------------

        self.show()

        msg = 'MFI: Calibre has added ' + as_unicode(n_total_added) + ' new empty book(s)'
        if DEBUG: print(msg)
        self.maingui.status_bar.showMessage(msg)

        if exception_error_message != "":
            msg = exception_error_message
            return error_dialog(self.maingui, _('MFI'),_(msg), show=True)
    #-----------------------------------------------------------------------------------------
    def create_empty_txt_file(self,suffix='.txt',prefix="MFI",dir=None):
        suffix = ".txt"
        dir = self.empty_txt_file_dir
        fileobj = PersistentTemporaryFile(suffix=suffix, prefix=prefix, dir=dir, mode='w+b')
        path = fileobj.__getattr__('name')
        path = path.replace(os.sep,'/')
        fileobj.close()
        self.temporary_file_list.append(path)
        del fileobj
        return path
    #-----------------------------------------------------------------------------------------
    def mfi_cli_add_book(self,param):
        #~ s_add_param =  authors + as_unicode('|||') + title + str ('|||') + tags + as_unicode('|||') + series + as_unicode('|||') + identifier + as_unicode('|||') + cover + as_unicode('|||')  + publisher + as_unicode('|||')  + comments + as_unicode('|||')   # + empty_file_path appended here

        param = param.strip()
        args = param.split("|||")
        args_list = []
        for row in args:
            s = row.strip()
            args_list.append(s)
            #~ if DEBUG: print("args_list: ", s)
        #END FOR

        authors = args_list[0]
        title = args_list[1]
        tags = args_list[2]
        series = args_list[3]
        identifier = args_list[4]
        cover = args_list[5]
        publisher = args_list[6]
        comments = args_list[7]
        empty_file_path = args_list[8]

        path = empty_file_path
        path = path.replace('"',"")  # no double quotes
        file_name,file_extension = os.path.splitext(path)
        if file_extension.startswith("."):
            file_extension = file_extension[1: ]
        format_map = {file_extension:path}
        if DEBUG: print("format_map: ", as_unicode(format_map))

        mi = Metadata(_('Unknown'))

        authors = authors.strip()
        if authors > " ":
            authors = authors.replace('"','').strip()
            authors = authors.replace("--authors","").strip()
            authors_list = []
            authors_list.append(authors)
            mi.authors = authors_list
            del authors_list
            if DEBUG: print("authors: ", authors)

        title = title.strip()
        if title > " ":
            title = title.replace('"','').strip()
            title = title.replace("'","'").strip()
            title = title.replace("--title","").strip()
            mi.title = title
            if DEBUG: print("title: ", title)

        tags = tags.strip()
        if tags > " ":   # tag1,tag2
            tags = tags.replace('"','').strip()
            tags = tags.replace("--tags","").strip()
            tags_list = tags.split(",")
            mi.tags = tags_list
            del tags_list
            if DEBUG: print("tags: ", tags)

        series = series.strip()
        if series > " ":
            series = series.replace('"','').strip()
            series = series.replace("--series","").strip()
            mi.series = series
            if DEBUG: print("series: ", series)

        identifier = identifier.strip()
        if identifier > " ":   #  ' --identifier ' + '"zmfi:'+ identifier + '" ' + '  --identifier ' + '"zmfi_batch:' + self.batch_timestamp +  '" ' + '  --identifier ' + '"zmfi_origpath:' + source_file  +  '" '
            identifiers_dict = {}
            identifier_list = identifier.split("--identifier")
            for row in identifier_list:
                row = row.strip()
                if ":" in row:
                    row = row.replace('"','')
                    row_list = row.split("$")   # cannot split on a colon : because a windows path may be the value of the identifier type...
                    k = row_list[0].strip()
                    v = row_list[1].strip()
                    if k.startswith("z"):
                        if v > " ":
                            identifiers_dict[k] = v
                            if DEBUG: print(k,":",v)
            #END FOR
            del identifier_list
            mi.identifiers = identifiers_dict
            del identifiers_dict

        cover = cover.strip()
        if cover > " ":
            cover = cover.replace("--cover","").strip()
            cover = cover.replace('"',"")  # no double quotes
            mi.cover = cover
            if DEBUG: print("cover: ", cover)

        publisher = publisher.strip()
        if publisher > " ":
            publisher = publisher.replace("--publisher","").strip()
            publisher = publisher.replace('"',"")  # no double quotes
            mi.publisher = publisher
            if DEBUG: print("publisher: ", publisher)

        comments = comments.strip()
        if comments > " ":
            comments = comments.replace("--comments","").strip()
            if comments.startswith('"'):
                comments = comments[1: ]
            if comments.endswith('"'):
                comments = comments[0:-1]
            mi.comments = comments
            if DEBUG: print("comments: ", comments)

        books = []
        book = mi,format_map     # mi = Metadata(_('Unknown'))
        books.append(book)

        db = self.maingui.current_db.new_api   # Cache object

        try:
            ids, duplicates = db.add_books(books, add_duplicates=True, apply_import_tags=False, preserve_uuid=False, run_hooks=True, dbapi=None)
            if DEBUG: print("ids, duplicates = db.add_books>>>>>>>>>> ids:", as_unicode(ids), "type: ", type(ids), "  duplicates: ", as_unicode(duplicates))
            book = int(ids[0])
            if DEBUG: print("book:", as_unicode(book))
            self.mfi_books_just_added_list.append(book)
        except Exception as e:
            if DEBUG: print("=============>>> Exception in db.add_books: ", as_unicode(e) )
            return False

        del args
        del args_list

        mi = Metadata(_('Unknown'))
        id_map = {}
        mi.author_sort = "MFI Temporary Author Sort: " + as_unicode(book)
        id_map[book] = mi
        payload = []
        edit_metadata_action = self.maingui.iactions['Edit Metadata']
        edit_metadata_action.apply_metadata_changes(id_map, callback=None, merge_tags=True, merge_comments=True, icon=None)

        del id_map
        del mi

        tmp = []
        tmp.append(book)
        self.maingui.iactions['Edit Metadata'].refresh_books_after_metadata_edit(tmp)

        self.maingui.library_view.model().refresh_ids(ids)
        #~ self.send_listener_message_to_refreshdb()     # Calibre Version 5.7 removed RC

        return True
    #-----------------------------------------------------------------------------------------
    def build_custom_column_dicts(self):

        self.custom_column_id_dict = {}
        self.custom_column_datatype_dict = {}
        self.custom_column_datatype_by_id_dict = {}
        self.custom_column_is_multiple_dict = {}
        self.custom_column_normalized_dict = {}
        self.custom_column_display_dict = {}

        my_db,my_cursor,is_valid = self.apsw_connect_to_current_guidb()
        if not is_valid:
            error_dialog(self.maingui, _('MFI'),_('Database Connection Error.  Restart Calibre.'), show=True)
            return

        mysql = "SELECT id,label,datatype,is_multiple,normalized,display FROM custom_columns  "
        my_cursor.execute(mysql)
        tmp_rows = my_cursor.fetchall()
        my_db.close()
        if not tmp_rows:
            pass
        else:
            if len(tmp_rows) == 0:
                pass
            else:
                for row in tmp_rows:
                    id,label,datatype,is_multiple,normalized,display = row
                    label = as_unicode(label)
                    self.custom_column_id_dict[label] = as_unicode(id)                  # no leading '#'
                    self.custom_column_datatype_dict[label] = as_unicode(datatype)
                    self.custom_column_datatype_by_id_dict[as_unicode(id)] = as_unicode(datatype)
                    self.custom_column_is_multiple_dict[label] = is_multiple
                    self.custom_column_normalized_dict[label] = normalized
                    self.custom_column_display_dict[label] = display
                #END FOR
                del tmp_rows
   #-----------------------------------------------------------------------------------------
    def apsw_connect_to_current_guidb(self):

        my_db = self.maingui.library_view.model().db
        path = my_db.library_path
        if isbytestring(path):
            path = path.decode(filesystem_encoding)
        path = path.replace(os.sep, '/')
        self.lib_path = path
        path = os.path.join(path, 'metadata.db')
        path = path.replace(os.sep, '/')
        try:
            my_db = apsw.Connection(path)
        except Exception as e:
            if DEBUG: print(as_unicode(e))
            path = path.encode("ascii", "ignore")
            msg = as_unicode("MFI cannot use the path that you selected:" + as_unicode(path) + " - " + as_unicode(e) )
            self.maingui.status_bar.showMessage(msg)
            return None,None,False
        my_cursor = my_db.cursor()
        mysql_pragma = "PRAGMA main.busy_timeout = 10000;"      #PRAGMA busy_timeout = milliseconds;
        my_cursor.execute(mysql_pragma)

        return my_db,my_cursor,True
    #-----------------------------------------------------------------------------------------
    def validate_and_save_settings(self):

        #~ --------------------------------------------
        self.save_dialog_geometry()
        #~ --------------------------------------------

        if self.mode_images_radio.isChecked():
            self.mode = "image"
            self.mode_is_image = True
            self.mode_is_audio = False
            self.mode_is_video = False
            self.mode_is_binary =  False
        elif self.mode_audio_radio.isChecked():
            self.mode = "audio"
            self.mode_is_image = False
            self.mode_is_audio = True
            self.mode_is_video = False
            self.mode_is_binary =  False
        elif self.mode_video_radio.isChecked():
            self.mode = "video"
            self.mode_is_image = False
            self.mode_is_audio = False
            self.mode_is_video = True
            self.mode_is_binary =  False
        elif self.mode_binary_radio.isChecked():
            self.mode = "binary"
            self.mode_is_image = False
            self.mode_is_audio = False
            self.mode_is_video = False
            self.mode_is_binary =  True
        else:
            self.mode_is_image = False
            self.mode_is_audio = False
            self.mode_is_video = False
            self.mode_is_binary =  False
            self.mode = ""
            return

        #~ --------------------------------------------

        prefs['MODE'] = self.mode.strip()
        prefs['TITLE'] = self.option_titles_to_add_lineedit.text().strip()
        prefs['AUTHORS'] = self.option_authors_to_add_lineedit.text().strip()
        prefs['SERIES'] = self.option_series_to_add.text().strip()
        prefs['TAGS'] = self.option_tags_to_add_lineedit.text().strip()
        prefs['PUBLISHER'] = self.option_publisher_to_add.text().strip()
        prefs['IDENTIFIER'] =  self.option_identifier_to_add_lineedit.text().strip()
        prefs['COMMENTS'] =  self.option_comments_to_add_lineedit.text().strip()
        prefs['COVER_PATH'] =  self.option_cover_to_add_lineedit.text().strip()
        prefs['CUSTOM_COLUMN_1'] =  self.option_custom_column_1_to_add_lineedit.text().strip()
        prefs['CUSTOM_COLUMN_1_VALUE'] =  self.option_custom_column_1_to_add_value_lineedit.text().strip()
        prefs['CUSTOM_COLUMN_2'] =  self.option_custom_column_2_to_add_lineedit.text().strip()
        prefs['CUSTOM_COLUMN_2_VALUE'] =  self.option_custom_column_2_to_add_value_lineedit.text().strip()
        prefs['CUSTOM_COLUMN_3'] =  self.option_custom_column_3_to_add_lineedit.text().strip()
        prefs['CUSTOM_COLUMN_3_VALUE'] =  self.option_custom_column_3_to_add_value_lineedit.text().strip()
        prefs['CUSTOM_COLUMN_4'] =  self.option_custom_column_4_to_add_lineedit.text().strip()
        prefs['CUSTOM_COLUMN_4_VALUE'] =  self.option_custom_column_4_to_add_value_lineedit.text().strip()

        if self.copy_file_to_calibre_folder_checkbox.isChecked():
            prefs['COPY_ORIGINAL_FILE_TOO'] = unicode_type("True")
        else:
            prefs['COPY_ORIGINAL_FILE_TOO'] = unicode_type("False")

        if self.keep_original_path_identifier_checkbox.isChecked():
            prefs['KEEP_ORIGINAL_PATH_IDENTIFIER'] = unicode_type("True")
        else:
            prefs['KEEP_ORIGINAL_PATH_IDENTIFIER'] = unicode_type("False")

        if self.copy_same_filenames_to_same_folder_checkbox.isChecked():
            prefs['COPY_SAME_BASE_FILENAMES_TO_SAME_CALIBRE_DIRECTORY']  = unicode_type("True")
            #~ requires this too:
            prefs['COPY_ORIGINAL_FILE_TOO'] = unicode_type("True")
            self.copy_file_to_calibre_folder_checkbox.setChecked(True)
        else:
            prefs['COPY_SAME_BASE_FILENAMES_TO_SAME_CALIBRE_DIRECTORY']  = unicode_type("False")

        prefs['FILE_EXTENSION_PRIORITY_SEQUENCE'] = self.file_extension_priority_sequence_lineedit.text().strip()

        prefs

        #~ --------------------------------------------
        #~ --------------------------------------------
        is_valid = True
        #~ ----------------
        #~ validate Author & Title
        #~ ----------------
        title = prefs['TITLE']
        authors = prefs['AUTHORS']
        if title > " " and authors > " ":
            pass
        else:
            is_valid = False
            error = "Both Title and Authors are required by the calibre to add a new book."
            self.hide()
            error_dialog(self.maingui, _('Validation of Metadata:'),_(error), show=True)
            self.show()
        #~ ----------------
        #~ validate {path}
        #~ ----------------
        self.cover_file_path = None
        if self.mode_is_image:
            path = prefs['COVER_PATH']
            if  "/" in path or "\\" in path:
                is_valid,path = self.check_file_for_existence(path)
                if not is_valid:
                    is_valid = False
                    error = "Mode is 'Images': Invalid 'Cover Path' [1]: " + path
                    self.hide()
                    error_dialog(self.maingui, _('Validation of Metadata:'),_(error), show=True)
                    self.show()
                else:
                    prefs['COVER_PATH'] = path
                    self.option_cover_to_add_lineedit.setText(path)
            else:
                if path == "{path}":
                    pass
                else:
                    if path > " ":
                        is_valid = False
                        error = "Invalid 'Cover Path' [2]: " + path
                        self.hide()
                        error_dialog(self.maingui, _('Validation of Metadata:'),_(error), show=True)
                        self.show()
            #~ --------------------------------------------
            #~ --------------------------------------------
        elif self.mode_is_audio:
            if "{path}" in prefs['COVER_PATH']:
                is_valid = False
                error = "Invalid 'Cover Path' for Mode: " + self.mode
                self.hide()
                error_dialog(self.maingui, _('Validation of Metadata:'),_(error), show=True)
                self.show()
            #~ for k,v in prefs.iteritems():
            for k,v in iteritems(prefs):
                v = as_unicode(v)
                if "{image_exif" in v:
                    is_valid = False
                    error = "{image_exif_data} is Invalid for Mode: " + self.mode
                    self.hide()
                    error_dialog(self.maingui, _('Validation of Metadata:'),_(error), show=True)
                    self.show()
                    break
            #~ --------------------------------------------
            #~ --------------------------------------------
        elif self.mode_is_video:
            if "{path}" in prefs['COVER_PATH']:
                is_valid = False
                error = "Invalid 'Cover Path' for Mode: " + self.mode
                self.hide()
                error_dialog(self.maingui, _('Validation of Metadata:'),_(error), show=True)
                self.show()
            #~ for k,v in prefs.iteritems():
            for k,v in iteritems(prefs):
                v = as_unicode(v)
                if "{image_exif_" in v:
                    is_valid = False
                    error = "{image_exif_data} is Invalid for Mode: " + self.mode
                    self.hide()
                    error_dialog(self.maingui, _('Validation of Metadata:'),_(error), show=True)
                    self.show()
                    break
            #END FOR

            #~ --------------------------------------------
            #~ --------------------------------------------
        elif self.mode_is_binary:
            pass  # anything goes; it is up to the user to get the file extension priorities consistent with what they are trying to do with {path} etc.

        #~ for k,v in prefs.iteritems():
        for k,v in iteritems(prefs):
            v = as_unicode(v)
            if "{embroidery_file_info}" in v:
                if not self.mode_is_binary:
                    is_valid = False
                    error = "{embroidery_file_info} is Invalid for Mode: " + self.mode
                    self.hide()
                    error_dialog(self.maingui, _('Validation of Metadata:'),_(error), show=True)
                    self.show()
                    break
        #END FOR

        tags = prefs['TAGS']
        if "{today}" in tags:
            is_valid = False
            error = "{today} is Invalid for use in Tags: " + tags
            self.hide()
            error_dialog(self.maingui, _('Validation of Metadata:'),_(error), show=True)
            self.show()
        if "{image_exif_" in tags:
            is_valid = False
            error = "{image_exif_data} is Invalid for use in Tags: " + tags
            self.hide()
            error_dialog(self.maingui, _('Validation of Metadata:'),_(error), show=True)
            self.show()


        #~ --------------------------------------------
        #~ --------------------------------------------

        cc1 = prefs['CUSTOM_COLUMN_1']
        cc1v = prefs['CUSTOM_COLUMN_1_VALUE']
        cc2 = prefs['CUSTOM_COLUMN_2']
        cc2v = prefs['CUSTOM_COLUMN_2_VALUE']
        cc3 = prefs['CUSTOM_COLUMN_3']
        cc3v = prefs['CUSTOM_COLUMN_3_VALUE']
        cc4 = prefs['CUSTOM_COLUMN_4']
        cc4v = prefs['CUSTOM_COLUMN_4_VALUE']

        if cc1.startswith("#"):
            cc1 = cc1[1: ]
        if cc2.startswith("#"):
            cc2 = cc2[1: ]
        if cc3.startswith("#"):
            cc3 = cc3[1: ]
        if cc4.startswith("#"):
            cc4 = cc4[1: ]

        if cc1 > " ":
            if cc1 in self.custom_column_id_dict:
                datatype = self.custom_column_datatype_dict[cc1]
                if datatype == "rating":
                    is_valid = False
                    error = "Invalid Custom Column 1 - datatype is 'rating': " + cc1
                    self.hide()
                    error_dialog(self.maingui, _('Validation of Metadata:'),_(error), show=True)
                    self.show()
                elif datatype == "composite":
                    is_valid = False
                    error = "Invalid Custom Column 1 - datatype is 'composite': " + cc1
                    self.hide()
                    error_dialog(self.maingui, _('Validation of Metadata:'),_(error), show=True)
                    self.show()
                else:
                    if not cc1v > " ":
                        is_valid = False
                        error = "Missing Custom Column 1 Value for: " + cc1
                        self.hide()
                        error_dialog(self.maingui, _('Validation of Metadata:'),_(error), show=True)
                        self.show()
                    else:
                        if datatype == 'integer':
                            try:
                                n = int(cc1v)
                            except:
                                is_valid = False
                                error = "Invalid Custom Column 1 Value: not an integer: " + cc1v
                                self.hide()
                                error_dialog(self.maingui, _('Validation of Metadata:'),_(error), show=True)
                                self.show()
                        elif datatype == 'float':
                            try:
                                n = float(cc1v)
                            except:
                                is_valid = False
                                error = "Invalid Custom Column 1 Value: not a floating point number: " + cc1v
                                self.hide()
                                error_dialog(self.maingui, _('Validation of Metadata:'),_(error), show=True)
                                self.show()
                        elif datatype == "enumeration":
                            display = self.custom_column_display_dict[cc1]
                            if not as_unicode(cc1v) in as_unicode(display):
                                is_valid = False
                                error = "Invalid Custom Column 1 Value: Not a datatype = 'enumeration' valid value: " + cc1v
                                self.hide()
                                error_dialog(self.maingui, _('Validation of Metadata:'),_(error), show=True)
                                self.show()
            else:
                is_valid = False
                error = "Invalid Custom Column 1:" + cc1
                self.hide()
                error_dialog(self.maingui, _('Validation of Metadata:'),_(error), show=True)
                self.show()

        if cc2 > " ":
            if cc2 in self.custom_column_id_dict:
                datatype = self.custom_column_datatype_dict[cc2]
                if datatype == "rating":
                    is_valid = False
                    error = "Invalid Custom Column 2 - datatype is 'rating': " + cc2
                    self.hide()
                    error_dialog(self.maingui, _('Validation of Metadata:'),_(error), show=True)
                    self.show()
                elif datatype == "composite":
                    is_valid = False
                    error = "Invalid Custom Column 2 - datatype is 'composite': " + cc2
                    self.hide()
                    error_dialog(self.maingui, _('Validation of Metadata:'),_(error), show=True)
                    self.show()
                else:
                    if not cc2v > " ":
                        is_valid = False
                        error = "Missing Custom Column 2 Value for: " + cc2
                        self.hide()
                        error_dialog(self.maingui, _('Validation of Metadata:'),_(error), show=True)
                        self.show()
                    else:
                        if datatype == 'integer':
                            try:
                                n = int(cc2v)
                            except:
                                is_valid = False
                                error = "Invalid Custom Column 2 Value: not an integer: " + cc2v
                                self.hide()
                                error_dialog(self.maingui, _('Validation of Metadata:'),_(error), show=True)
                                self.show()
                        elif datatype == 'float':
                            try:
                                n = float(cc2v)
                            except:
                                is_valid = False
                                error = "Invalid Custom Column 2 Value: not a floating point number: " + cc2v
                                self.hide()
                                error_dialog(self.maingui, _('Validation of Metadata:'),_(error), show=True)
                                self.show()
                        elif datatype == "enumeration":
                            display = self.custom_column_display_dict[cc2]
                            if not as_unicode(cc2v) in as_unicode(display):
                                is_valid = False
                                error = "Invalid Custom Column 2 Value: Not a datatype = 'enumeration' valid value: " + cc2v
                                self.hide()
                                error_dialog(self.maingui, _('Validation of Metadata:'),_(error), show=True)
                                self.show()
            else:
                is_valid = False
                error = "Invalid Custom Column 2:" + cc2
                self.hide()
                error_dialog(self.maingui, _('Validation of Metadata:'),_(error), show=True)
                self.show()

        if cc3 > " ":
            if cc3 in self.custom_column_id_dict:
                datatype = self.custom_column_datatype_dict[cc3]
                if datatype == "rating":
                    is_valid = False
                    error = "Invalid Custom Column 3 - datatype is 'rating': " + cc3
                    self.hide()
                    error_dialog(self.maingui, _('Validation of Metadata:'),_(error), show=True)
                    self.show()
                elif datatype == "composite":
                    is_valid = False
                    error = "Invalid Custom Column 3 - datatype is 'composite': " + cc3
                    self.hide()
                    error_dialog(self.maingui, _('Validation of Metadata:'),_(error), show=True)
                    self.show()
                else:
                    if not cc3v > " ":
                        is_valid = False
                        error = "Missing Custom Column 3 Value for: " + cc3
                        self.hide()
                        error_dialog(self.maingui, _('Validation of Metadata:'),_(error), show=True)
                        self.show()
                    else:
                        if datatype == 'integer':
                            try:
                                n = int(cc3v)
                            except:
                                is_valid = False
                                error = "Invalid Custom Column 3 Value: not an integer: " + cc3v
                                self.hide()
                                error_dialog(self.maingui, _('Validation of Metadata:'),_(error), show=True)
                                self.show()
                        elif datatype == 'float':
                            try:
                                n = float(cc3v)
                            except:
                                is_valid = False
                                error = "Invalid Custom Column 3 Value: not a floating point number: " + cc3v
                                self.hide()
                                error_dialog(self.maingui, _('Validation of Metadata:'),_(error), show=True)
                                self.show()
                        elif datatype == "enumeration":
                            display = self.custom_column_display_dict[cc3]
                            if not as_unicode(cc3v) in as_unicode(display):
                                is_valid = False
                                error = "Invalid Custom Column 3 Value: Not a datatype = 'enumeration' valid value: " + cc3v
                                self.hide()
                                error_dialog(self.maingui, _('Validation of Metadata:'),_(error), show=True)
                                self.show()
            else:
                is_valid = False
                error = "Invalid Custom Column 3:" + cc3
                self.hide()
                error_dialog(self.maingui, _('Validation of Metadata:'),_(error), show=True)
                self.show()

        if cc4 > " ":
            if cc4 in self.custom_column_id_dict:
                datatype = self.custom_column_datatype_dict[cc4]
                if datatype == "rating":
                    is_valid = False
                    error = "Invalid Custom Column 4 - datatype is 'rating': " + cc4
                    self.hide()
                    error_dialog(self.maingui, _('Validation of Metadata:'),_(error), show=True)
                    self.show()
                elif datatype == "composite":
                    is_valid = False
                    error = "Invalid Custom Column 4 - datatype is 'composite': " + cc4
                    self.hide()
                    error_dialog(self.maingui, _('Validation of Metadata:'),_(error), show=True)
                    self.show()
                else:
                    if not cc4v > " ":
                        is_valid = False
                        error = "Missing Custom Column 4 Value for: " + cc4
                        self.hide()
                        error_dialog(self.maingui, _('Validation of Metadata:'),_(error), show=True)
                        self.show()
                    else:
                        if datatype == 'integer':
                            try:
                                n = int(cc4v)
                            except:
                                is_valid = False
                                error = "Invalid Custom Column 4 Value: not an integer: " + cc4v
                                self.hide()
                                error_dialog(self.maingui, _('Validation of Metadata:'),_(error), show=True)
                                self.show()
                        elif datatype == 'float':
                            try:
                                n = float(cc4v)
                            except:
                                is_valid = False
                                error = "Invalid Custom Column 4 Value: not a floating point number: " + cc4v
                                self.hide()
                                error_dialog(self.maingui, _('Validation of Metadata:'),_(error), show=True)
                                self.show()
                        elif datatype == "enumeration":
                            display = self.custom_column_display_dict[cc4]
                            if not as_unicode(cc4v) in as_unicode(display):
                                is_valid = False
                                error = "Invalid Custom Column 4 Value: Not a datatype = 'enumeration' valid value: " + cc4v
                                self.hide()
                                error_dialog(self.maingui, _('Validation of Metadata:'),_(error), show=True)
                                self.show()
            else:
                is_valid = False
                error = "Invalid Custom Column 4:" + cc4
                self.hide()
                error_dialog(self.maingui, _('Validation of Metadata:'),_(error), show=True)
                self.show()

        #~ --------------------------------------------
        #~ --------------------------------------------
        if is_valid:
            self.set_lineedits_to_read_only(True)
        else:
            self.okay_to_drop_files = False
            msg = '<font color = "red">...Drag and Drop Prohibited...</font>'
            self.messages_label.setText(msg)
            self.repaint()
    #-----------------------------------------------------------------------------------------
    def set_lineedits_to_read_only(self,readonly=False):
        if readonly:
            self.okay_to_drop_files = True
            msg = '<font color = "green">...Drag and Drop Allowed...</font>'
            self.messages_label.setText(msg)
            self.mode_groupbox.hide()
            self.push_button_save_validate.hide()
            self.option_titles_to_add_lineedit.setReadOnly(True)
            self.option_authors_to_add_lineedit.setReadOnly(True)
            self.option_series_to_add.setReadOnly(True)
            self.option_publisher_to_add.setReadOnly(True)
            self.option_tags_to_add_lineedit.setReadOnly(True)
            self.option_identifier_to_add_lineedit.setReadOnly(True)
            self.option_comments_to_add_lineedit.setReadOnly(True)
            self.option_cover_to_add_lineedit.setReadOnly(True)
            self.option_custom_column_1_to_add_lineedit.setReadOnly(True)
            self.option_custom_column_1_to_add_value_lineedit.setReadOnly(True)
            self.option_custom_column_2_to_add_lineedit.setReadOnly(True)
            self.option_custom_column_2_to_add_value_lineedit.setReadOnly(True)
            self.option_custom_column_3_to_add_lineedit.setReadOnly(True)
            self.option_custom_column_3_to_add_value_lineedit.setReadOnly(True)
            self.option_custom_column_4_to_add_lineedit.setReadOnly(True)
            self.option_custom_column_4_to_add_value_lineedit.setReadOnly(True)
            self.copy_file_to_calibre_folder_checkbox.setDisabled(True)
            self.keep_original_path_identifier_checkbox.setDisabled(True)
            self.copy_same_filenames_to_same_folder_checkbox.setDisabled(True)
            self.file_extension_priority_sequence_lineedit.setReadOnly(True)
            self.push_button_select_files_by_directory.show()
            self.push_button_exit.show()
    #-----------------------------------------------------------------------------------------
    def check_file_for_existence(self,file):
        is_valid = False
        file = file.strip()
        if file.startswith('"'):
            file = file[1: ]
        if file.endswith('"'):
            file = file[0:-1]
        if isbytestring(file):
            file = file.decode(filesystem_encoding)
        file = file.replace(os.sep,'/')
        if os.path.isfile(file):
            is_valid = True
        #~ if DEBUG: print("check_file_for_existence: ", is_valid, file)
        return is_valid,file
    #-----------------------------------------------------------------------------------------
    def validate_directory_n(self):

        is_valid = True

        title = prefs['TITLE']
        is_valid = check_directory_for_existence(title)
        if not is_valid: return is_valid,title
        authors = prefs['AUTHORS']
        is_valid = check_directory_for_existence(authors)
        if not is_valid: return is_valid,authors
        series = prefs['SERIES']
        is_valid = check_directory_for_existence(series)
        if not is_valid: return is_valid,series
        tags = prefs['TAGS']
        is_valid = check_directory_for_existence(tags)
        if not is_valid: return is_valid,tags
        identifier = prefs['IDENTIFIER']
        is_valid = check_directory_for_existence(identifier)
        if not is_valid: return is_valid,identifier
        comments = prefs['COMMENTS']
        is_valid = check_directory_for_existence(comments)
        if not is_valid: return is_valid,comments
        cc1v = prefs['CUSTOM_COLUMN_1_VALUE']
        is_valid = check_directory_for_existence(cc1v)
        if not is_valid: return is_valid,cc1v
        cc2v = prefs['CUSTOM_COLUMN_2_VALUE']
        is_valid = check_directory_for_existence(cc2v)
        if not is_valid: return is_valid,cc2v
        cc3v = prefs['CUSTOM_COLUMN_3_VALUE']
        is_valid = check_directory_for_existence(cc3v)
        if not is_valid: return is_valid,cc3v
        cc4v = prefs['CUSTOM_COLUMN_4_VALUE']
        is_valid = check_directory_for_existence(cc4v)
        if not is_valid: return is_valid,cc4v

        return is_valid,None
    #-----------------------------------------------------------------------------------------
    def check_directory_for_existence(self,dir):
        if  "{directory:" in dir:
            dir = dir.strip()
            if len(dir) !=  13:          # {directory:0} thru {directory:9} are the only valid values...
                return False
            return True
        else:
            dir = dir.strip()
            if isbytestring(dir):
                dir = dir.decode(filesystem_encoding)
            dir = dir.replace(os.sep,'/')
            is_valid = self.validate_directory_path(dir)
            return is_valid
    #-----------------------------------------------------------------------------------------
    def validate_directory_path(self,dir):
        is_valid = False
        if os.path.isdir(dir):
            is_valid = True
        return is_valid
    #-----------------------------------------------------------------------------------------
    def extract_directory_path_n(self,value,filepath):

        if not "{directory_path:" in value:
            return True,None

        s_split = value.split("{directory_path:")
        if not len(s_split) > 0:
            return True,None

        s = ""
        for row in s_split:
            #~ print("row after 1st split: ", row)
            s = row.strip()
            if "}" in s:
                break
        #END FOR

        s_split = s.split("}")
        if not len(s_split) > 0:
            return True,None

        n = None
        for row in s_split:
            #~ print("row after 2nd split: ", row)
            s = row.strip()
            if s > " ":
                if s.isdigit():
                    n = row.strip()
                    break
        #END FOR

        try:
            n_level = int(n)     # 0 thru 9 are the only valid values...
        except:
            if DEBUG: print("extract_directory_path_n  error >> n: ", as_unicode(n))
            return False,None

        if n_level >= 0 and n_level <= 9:
            dir =  os.path.dirname(filepath)
            if n_level == 1:
                dir = os.path.dirname(dir)
                if n_level == 2:
                    dir = os.path.dirname(dir)
                    if n_level == 3:
                        dir = os.path.dirname(dir)
                        if n_level == 4:
                            dir = os.path.dirname(dir)
                            if n_level == 5:
                                dir = os.path.dirname(dir)
                                if n_level == 6:
                                    dir = os.path.dirname(dir)
                                    if n_level == 7:
                                        dir = os.path.dirname(dir)
                                        if n_level == 8:
                                            dir = os.path.dirname(dir)
                                            if n_level == 9:
                                                dir = os.path.dirname(dir)
        if isbytestring(dir):
            dir = dir.decode(filesystem_encoding)
        dir = dir.replace(os.sep,'/')

        return True,dir
    #------------------------------------------------------------------------------------------
    def extract_directory_name_n(self,value,filepath):

        if not "{directory_name:" in value:
            return True,None

        s_split = value.split("{directory_name:")
        if not len(s_split) > 0:
            return True,None

        valuex = value.replace("{directory_name:","{directory_path:")
        is_valid,dir = self.extract_directory_path_n(valuex,filepath)
        if is_valid:
            if dir:
                dir = os.path.basename(os.path.normpath(dir))
                if dir:
                    dir = dir.strip()
                    if isbytestring(dir):
                        dir = dir.decode(filesystem_encoding)
                    dir = dir.replace(os.sep,'/')
                    return True,dir
                else:
                    return False,value
        else:
            return False,value
    #-----------------------------------------------------------------------------------------
    def update_metadata_of_new_books(self):

        self.mfi_books_just_added_list = list(set(self.mfi_books_just_added_list))
        self.mfi_books_just_added_list.sort()

        #-----------------------------
        my_db,my_cursor,is_valid = self.apsw_connect_to_current_guidb()
        if not is_valid:
            if DEBUG: print("Error: could not connect to current guidb.")
            return
        #-----------------------------

        if DEBUG: print("self.update_custom_columns_control(my_db, my_cursor)")

        is_valid = self.update_custom_columns_control(my_db, my_cursor)

        if is_valid:
            self.batch_timestamp = unicode_type(self.batch_timestamp)
            mysql = "DELETE FROM identifiers WHERE type = 'zmfi_batch' and val = ? "
            my_cursor.execute("begin")
            my_cursor.execute(mysql,([self.batch_timestamp]))
            my_cursor.execute("commit")

        my_db.close()

        if is_valid:
            self.refresh_cache_and_library_view()

    #-----------------------------------------------------------------------------------------
    def expand_dynamic_variables(self,value,path):

        if not path:
            return value

        if not "/" in path:
            return value

        if not "{" in value:
            return value

        if path.endswith("/"):
            path = path[0:-1]

        path = path.replace("/",os.sep)   # so Windows users can see 'real' paths in their metadata...

        if "{path}" in value:
            value = value.replace("{path}",path)

        if "{directory_path:" in value:
            if value.startswith("/") or value.startswith("\\"):
                value = value[1: ]
            is_valid,dir = self.extract_directory_path_n(value,path)
            if not is_valid:
                dir = "...error..."
                if DEBUG: print("Error in: extract_directory_path_n for path: ", path)
            value = value.replace("{directory_path:0}",dir)
            value = value.replace("{directory_path:1}",dir)
            value = value.replace("{directory_path:2}",dir)
            value = value.replace("{directory_path:3}",dir)
            value = value.replace("{directory_path:4}",dir)
            value = value.replace("{directory_path:5}",dir)
            value = value.replace("{directory_path:6}",dir)
            value = value.replace("{directory_path:7}",dir)
            value = value.replace("{directory_path:8}",dir)
            value = value.replace("{directory_path:9}",dir)

        if "{directory_name:" in value:
            if value.startswith("/") or value.startswith("\\"):
                value = value[1: ]
            is_valid,dir = self.extract_directory_name_n(value,path)
            if (not is_valid) or (not dir):
                dir = "...error..."
            value = value.replace("{directory_name:0}",dir)
            value = value.replace("{directory_name:1}",dir)
            value = value.replace("{directory_name:2}",dir)
            value = value.replace("{directory_name:3}",dir)
            value = value.replace("{directory_name:4}",dir)
            value = value.replace("{directory_name:5}",dir)
            value = value.replace("{directory_name:6}",dir)
            value = value.replace("{directory_name:7}",dir)
            value = value.replace("{directory_name:8}",dir)
            value = value.replace("{directory_name:9}",dir)

        if "{file_name}" in value:
            fname = os.path.basename(path)
            value = value.replace("{file_name}",fname)

        if "{file_extension}" in value:
            base_name = os.path.basename(path)
            filebasename,file_extension = os.path.splitext(base_name)
            value = value.replace("{file_extension}",file_extension)

        if "{file_name_no_extension}" in value:
            base_name = os.path.basename(path)
            filebasename,file_extension = os.path.splitext(base_name)
            value = value.replace("{file_name_no_extension}",filebasename)

        if "{image_exif_" in value:
            exif_data,dt,dto,dtd = self.get_image_exif(path)
            value = value.replace("{image_exif_data}",exif_data)
            value = value.replace("{image_exif_date_time}",dt)
            value = value.replace("{image_exif_date_time_original}",dto)
            value = value.replace("{image_exif_date_time_digitized}",dtd)
            del exif_data
            del dt
            del dto
            del dtd

        if "{today}" in value:
            todaydate = datetime.now()
            todaydate = as_unicode(todaydate)
            todaydate = todaydate[0:10]
            value = as_unicode(todaydate).strip()

        #~ Support for {mp3_info} in MFI has been removed after Python 3 conversion

        if "{embroidery_file_info}" in value:
            base_name = os.path.basename(path)
            filebasename,file_extension = os.path.splitext(base_name)
            file_extension = file_extension.lower()
            if file_extension == '.pcs':
                pcs_info = self.pfaff_pcs_metadata_get(path,file_extension)
                value = value.replace("{embroidery_file_info}",pcs_info)
            elif file_extension == '.hus' or file_extension == '.vip':
                hus_vip_info = self.hus_vip_metadata_get(path,file_extension)
                value = value.replace("{embroidery_file_info}",hus_vip_info)


        return value
     #-----------------------------------------------------------------------------------------
    def get_image_exif(self,path):
        try:
            data = ""
            ret = {}
            i = Image.open(path)
            info = i._getexif()
            if not isinstance(info,dict):
                return data,data,data,data
            for tag, value in info.items():
                decoded = TAGS.get(tag, tag)
                ret[decoded] = value
            #END FOR
            dt = ""
            dto = ""
            dtd = ""
            #~ for k,v in ret.iteritems():
            for k,v in iteritems(ret):
                data = data + as_unicode(k) + ":" + as_unicode(v) + "; "
                if k == "DateTime":
                    dt = as_unicode(v)
                if k == "DateTimeOriginal":
                    dto = as_unicode(v)
                if k == "DateTimeDigitized":
                    dtd = as_unicode(v)
            #END FOR
            if data.endswith(";"):
                data = data[0:-1]
            try:
                del i
                del info
                del decoded
                del ret
            except:
                pass
        except Exception as e:
            if DEBUG: print("get_image_exif", e)
            data = ""
            dt = ""
            dto = ""
            dtd = ""

        return data,dt,dto,dtd
    #-----------------------------------------------------------------------------------------
    def update_custom_columns_control(self,my_db, my_cursor):

        self.selected_books_list = self.mfi_books_just_added_list

        cc_to_update_list = []

        cc1 = prefs['CUSTOM_COLUMN_1']
        cc1v = prefs['CUSTOM_COLUMN_1_VALUE']

        if cc1 > " " and cc1v > " ":
            if cc1.startswith("#"):
                cc1 = cc1[1: ]
            r = cc1,cc1v
            cc_to_update_list.append(r)

        cc2 = prefs['CUSTOM_COLUMN_2']
        cc2v = prefs['CUSTOM_COLUMN_2_VALUE']

        if cc2 > " " and cc2v > " ":
            if cc2.startswith("#"):
                cc2 = cc2[1: ]
            r = cc2,cc2v
            cc_to_update_list.append(r)

        cc3 = prefs['CUSTOM_COLUMN_3']
        cc3v = prefs['CUSTOM_COLUMN_3_VALUE']

        if cc3 > " " and cc3v > " ":
            if cc3.startswith("#"):
                cc3 = cc3[1: ]
            r = cc3,cc3v
            cc_to_update_list.append(r)

        cc4 = prefs['CUSTOM_COLUMN_4']
        cc4v = prefs['CUSTOM_COLUMN_4_VALUE']

        if cc4 > " " and cc4v > " ":
            if cc4.startswith("#"):
                cc4 = cc4[1: ]
            r = cc4,cc4v
            cc_to_update_list.append(r)

        if len(cc_to_update_list) == 0:
            return True

        self.conversion_error_list = []
        del self.conversion_error_list
        self.conversion_error_list = []

        my_db,my_cursor,is_valid = self.apsw_connect_to_current_guidb()
        if not is_valid:
            error_dialog(self.maingui, _('MFI'),_('Database Connection Error.  Restart Calibre.'), show=True)
            return False

        for r in cc_to_update_list:
            cc,val = r
            cc = cc.replace("#","")
            tdt = self.custom_column_datatype_dict[cc]
            tocolid = self.custom_column_id_dict[cc]
            toismultiple = self.custom_column_is_multiple_dict[cc]
            toisnormalized = self.custom_column_normalized_dict[cc]
            my_cursor.execute("begin")
            for book in self.selected_books_list:
                path = self.get_book_original_file_path(my_db,my_cursor,book)
                valnew = self.expand_dynamic_variables(val,path)
                self.update_target_custom_columns_from_mfi(my_db,my_cursor,book,valnew,tocolid,tdt,toisnormalized,toismultiple)
            #END FOR
            my_cursor.execute("commit")
        #END FOR
        my_db.close()

        if len(self.conversion_error_list) > 0:
            if len(self.conversion_error_list) > 25:
                self.conversion_error_list = self.conversion_error_list[0:25]
            msg = ""
            for error in self.conversion_error_list:
                msg = msg + error + "\n"
            #END FOR
            error_dialog(self.maingui, _('MFI Data Conversion Errors'),_(msg), show=True)

        return True  # to allow deletion of the zmfi_batch identifiers for this timestamp
    #-----------------------------------------------------------------------------------------
    def convert_textual_to_simple_text(self,value):
        value = html2text(value)
        value = value.strip()
        return value,None
    #-----------------------------------------------------------------------------------------
    def convert_textual_to_integer(self,value):
        newint = None
        error = None
        value = html2text(value)
        value = value.replace(" ","")
        value = value.strip()
        try:
            value = as_unicode(value)
            newint = int(float(value))    # may have a decimal point...
        except:
            error = "MFI value of " + as_unicode(value) + " could NOT be converted to an integer..."
            if DEBUG: print(error)
        return newint,error
    #-----------------------------------------------------------------------------------------
    def convert_textual_to_float(self,value):
        newfloat = None
        error = None
        value = html2text(value)
        value = value.replace(" ","")
        value = value.strip()
        try:
            newfloat = float(value)
        except:
            error = "MFI value of " + as_unicode(value) + " could NOT be converted to a float..."
            if DEBUG: print(error)
        return newfloat,error
    #-----------------------------------------------------------------------------------------
    def convert_textual_to_datetime(self,value):
        newdatetime = None
        value = html2text(value)
        value = value.replace(" ","")
        value = value[0:19]
        value = value.strip()
        try:
            newdatetime = datetime.strptime(value, "%a %b %d %H:%M:%S")           #  2016-06-19 20:18:33
        except:
            value = value[0:10]
            newdatetime = value + " 12:00:00.000000+00:00"           # 2016-06-19 12:00:00.000000+00:00
        return newdatetime,None
    #-----------------------------------------------------------------------------------------
    def convert_textual_to_boolean(self,value):
        newboolean = None
        error = None
        value = html2text(value)
        value = value.replace(" ","")
        value = value.strip()
        value = value.lower()
        if as_unicode(value) == as_unicode("yes") or value == "yes" or as_unicode(value) == as_unicode(1) or as_unicode(value) == as_unicode("true") or value == "true":
            newboolean = 1
        elif as_unicode(value) == as_unicode("no") or value == "no" or as_unicode(value) == as_unicode(0) or as_unicode(value) == as_unicode("false") or value == "false":
            newboolean = 0
        else:
            error = "MFI value of " + as_unicode(value) + " could NOT be converted to a boolean..."
            if DEBUG: print(error)
        return newboolean,error
    #-----------------------------------------------------------------------------------------
    def update_target_custom_columns_from_mfi(self,my_db,my_cursor,book,value,tocolid,tdt,toisnormalized,toismultiple):
        book = int(book)
        tocolid = as_unicode(tocolid)
        error = None
        #~ if DEBUG: print("book: ", as_unicode(book), " value: ", as_unicode(value), " tdt: ", tdt)
        if tdt == "comments":
            pass
        elif tdt == "text":
            value,error = self.convert_textual_to_simple_text(value)
        elif tdt == "series":
            value,error = self.convert_textual_to_simple_text(value)
        elif tdt == "enumeration":
            value,error = self.convert_textual_to_simple_text(value)
        elif tdt == "int":
            value,error = self.convert_textual_to_integer(value)
        elif tdt == "float":
            value,error = self.convert_textual_to_float(value)
        elif tdt == "datetime":
            value,error = self.convert_textual_to_datetime(value)
        elif tdt == "bool":
            value,error = self.convert_textual_to_boolean(value)
        elif tdt == "rating":
            value,error = self.convert_textual_to_integer(value)
            #~ if DEBUG: print("rating: ", as_unicode(value), as_unicode(error))
            if not error:
                if value < 0:
                    value = 0
                if value in [0,2,4,6,8,10]:
                    pass
                else:
                    if value > 0 and value < 11:
                        value = 2 * value
                        if value > 10:
                            value = 10
        else:
            if DEBUG: print("update_target_custom_columns_from_mfi --- unsupported to-custom column datatype conversion: " + tdt)
            return

        if not error:
            pass
        else:
            self.conversion_error_list.append(error)
            return

        if toisnormalized == 1:
            if toismultiple == 1:
                value_list = self.format_taglike_data(value)
                value = None
            else:
                value_list = None
            self.update_normalized_custom_column(my_db,my_cursor,book,tocolid,value,value_list)
        else:
            if toisnormalized == 0:
                self.update_non_normalized_custom_column(my_db,my_cursor,book,tocolid,value)
            else:
                return
    #-----------------------------------------------------------------------------------------
    def update_normalized_custom_column(self,my_db,my_cursor,book,tocolid,value,value_list):

        if not value and not value_list:
            return
        if value:
            if not value > " ":
                value = None
        if value_list:
            if len(value_list) == 0:
                value_list == None

        if value:
            self.update_basic_table(my_db,my_cursor,book,tocolid,value)
            self.update_book_link_table(my_db,my_cursor,book,tocolid,value)

        if value_list:
            for value in value_list:
                self.update_basic_table(my_db,my_cursor,book,tocolid,value)
                self.update_book_link_table(my_db,my_cursor,book,tocolid,value)
    #-----------------------------------------------------------------------------------------
    def update_basic_table(self,my_db,my_cursor,book,tocolid,value):
        mysql = "INSERT OR IGNORE INTO custom_column_[N] (id,value) VALUES (null,?) "
        mysql = mysql.replace("[N]",as_unicode(tocolid))
        my_cursor.execute(mysql,([value]))
        my_cursor.execute("commit")
        my_cursor.execute("begin")
    #-----------------------------------------------------------------------------------------
    def update_book_link_table(self,my_db,my_cursor,book,tocolid,value):
        datatype = self.custom_column_datatype_by_id_dict[as_unicode(tocolid)]
        if datatype == 'series':
            mysql = "INSERT OR REPLACE INTO books_custom_column_[N]_link (id,book,value,extra) VALUES (null,?,(SELECT id FROM custom_column_[N] WHERE value = ?),0 )  "
        else:
            mysql = "INSERT OR REPLACE INTO books_custom_column_[N]_link (id,book,value) VALUES (null,?,(SELECT id FROM custom_column_[N] WHERE value = ?) )  "
        mysql = mysql.replace("[N]",as_unicode(tocolid))
        my_cursor.execute(mysql,(book,value))
    #-----------------------------------------------------------------------------------------
    def update_non_normalized_custom_column(self,my_db,my_cursor,book,tocolid,value):
        if not value:
            return
        if not as_unicode(value) > " ":
            return
        mysql = "INSERT OR REPLACE INTO custom_column_[N] (id,book,value) VALUES (null,?,?) "
        mysql = mysql.replace("[N]",as_unicode(tocolid))
        my_cursor.execute(mysql,(book,value))
    #-----------------------------------------------------------------------------------------
    def format_taglike_data(self,value):

        try:
            del value_list
        except:
            pass

        value_list = []

        value = value.replace(";",",")
        value = value.strip()
        if not "," in value:
            if value > " ":
                value_list.append(value)
        else:
            s_split = value.split(",")
            n = len(s_split)
            if n > 0:
                for row in s_split:
                    if row:
                        if row > " ":
                            row = row.strip()
                            value_list.append(row)
                #END FOR
        return value_list
    #-----------------------------------------------------------------------------------------
    def get_book_original_file_path(self,my_db,my_cursor,book):
        path = None
        mysql = "SELECT val FROM identifiers WHERE book = ? AND type = 'zmfi_origpath' "
        tmp_rows=list(my_cursor.execute(mysql,([book])) )
        for row in tmp_rows:
            for col in row:
                path = col
                break
            #END FOR
            break
        #END FOR
        return path
    #-----------------------------------------------------------------------------------------
    def validate(self):
        return True
    #-----------------------------------------------------------------------------------------
    def copy_original_files_too(self):

        self.origpath_dirpath_dict = {}
        self.book_origpath_dict = {}
        self.book_origfilename_dict = {}
        self.book_original_format_dict = {}  # upper case, lower case, mixed case...
        self.book_format_dict = {}
        self.book_dirpath_dict = {}
        self.book_filename_dict = {}

        #-----------------------------
        my_db,my_cursor,is_valid = self.apsw_connect_to_current_guidb()
        if not is_valid:
            if DEBUG: print("Error: could not connect to current guidb...")
            return
        #-----------------------------1
        mysql = "SELECT books.id,books.path,identifiers.val FROM books,identifiers WHERE books.id = ? AND identifiers.book = books.id AND identifiers.type = 'zmfi_origpath'   "
        for book in self.mfi_books_just_added_list:
            my_cursor.execute(mysql,([book]))
            tmp_rows = my_cursor.fetchall()
            if not tmp_rows:
                continue
            for row in tmp_rows:
                book,dirpath,origpath = row
                dirpath = os.path.join(self.lib_path,dirpath)
                dirpath = dirpath.replace(os.sep, '/')
                if isbytestring(dirpath):
                    dirpath = dirpath.decode(filesystem_encoding)
                origpath = origpath.replace(os.sep, '/')
                if isbytestring(origpath):
                    origpath = origpath.decode(filesystem_encoding)
                self.origpath_dirpath_dict[origpath] = dirpath
                self.book_origpath_dict[book] = origpath
                self.book_dirpath_dict[book] = dirpath
                fname = os.path.basename(dirpath)
                self.book_filename_dict[book] = fname
            #END FOR
            del tmp_rows
        #END FOR

        self.extrafile_to_dirpath_dict = {}

        #~ for origpath,dirpath in self.origpath_dirpath_dict.iteritems():
        for origpath,dirpath in iteritems(self.origpath_dirpath_dict):
            if os.path.isfile(origpath):
                shutil.copy(origpath,dirpath)
                if DEBUG: print("origpath: ", origpath, " was copied to dirpath (directory only): ", dirpath)
                #~ self.source_file_dict_in_use = False
                if self.source_file_dict_in_use:
                    if origpath in self.source_file_path_dict:     #~ self.source_file_path_dict[path] = file_name,file_extension
                        file_name,file_extension = self.source_file_path_dict[origpath]
                        if file_name in self.source_file_dict:
                            ext_list = self.source_file_dict[file_name]
                            ext_list = self.convert_string_to_list(ext_list)
                            for ext in ext_list:
                                extrafile = file_name  + ext
                                extrafile = extrafile.replace(os.sep, '/')
                                if isbytestring(extrafile):
                                    extrafile = extrafile.decode(filesystem_encoding)
                                if DEBUG: print("shutil.copy --- extrafile to directory dirpath: ",extrafile,"   ", dirpath)
                                if os.path.isfile(extrafile):
                                    shutil.copy(extrafile,dirpath)
                            #END FOR
                        else:
                            continue
                    else:
                        continue
                else:
                    continue
            else:
                continue
        #END FOR

        sleep(0)

        # now add the original media file into metadata.db table 'data' as a 'format' so that the books can be copied properly among other libraries.
        # also, the Check Library function will show that all is well, and there are no extraneous files in the library.

        self.data_txt_book_name_dict = {}

        #~ STEP 1 of 3  Get just-added books from table data...
        mysql = "SELECT book,name FROM data WHERE book = ? AND format = 'TXT'  "
        for book in self.mfi_books_just_added_list:
            book = int(book)
            my_cursor.execute(mysql,([book]))
            tmp_rows = my_cursor.fetchall()
            if not tmp_rows:
                continue
            for row in tmp_rows:
                book,name = row
                self.data_txt_book_name_dict[book] = name
            #END FOR
            del tmp_rows
        #END FOR

        sleep(0)

        #~ STEP 2 of 3    Calibre does various things automatically as soon as table data is changed...so table data must be updated in a single step...
        mysql = "INSERT OR IGNORE INTO data (id,book,format,uncompressed_size,name) VALUES (null,?,?,0,?) "
        for book in self.mfi_books_just_added_list:
            book = int(book)
            if book in self.data_txt_book_name_dict:
                if DEBUG: print("book: ", as_unicode(book))
                name = self.data_txt_book_name_dict[book]
                if DEBUG: print("name: ", name)    #    name:  14730 - xxxxxx  as in name:  14730 - xxxxxx.bmp
                origpath = self.book_origpath_dict[book]
                if DEBUG: print("origpath: ", origpath)
                base_name = os.path.basename(origpath)
                if DEBUG: print("base_name: ", base_name)
                filebasename,file_extension = os.path.splitext(base_name)
                if DEBUG: print("filebasename,file_extension: ", filebasename, "      ", file_extension)
                self.book_origfilename_dict[book] = filebasename
                self.book_original_format_dict[book] = file_extension
                file_extension = file_extension.upper()
                file_extension = file_extension.replace(".","")
                self.book_format_dict[book] = file_extension
                my_cursor.execute("begin")
                my_cursor.execute(mysql,(book,file_extension,name))
                my_cursor.execute("commit")
        #END FOR

        sleep(0)

        #~ STEP 3 of 3         # this will rename the original file to the calibre-determined name for it just added to table data...
        for book in self.mfi_books_just_added_list:
            book = int(book)
            dirpath = self.book_dirpath_dict[book]                            # this is the directory in Calibre, not the full path of any file...
            format = self.book_original_format_dict[book]                # no dot...
            temppath = self.book_origfilename_dict[book]
            #~ # so now need to get the 'name' of the .txt file from table data, and rename the original media file to the exact .txt file name but with the media file's extension...
            mysql = "SELECT book,name FROM data WHERE book = ? AND format = 'TXT' AND name NOT NULL"
            my_cursor.execute(mysql,([book]))
            tmp_rows = my_cursor.fetchall()
            if not tmp_rows:
                continue
            for row in tmp_rows:
                bookx,name = row
                if bookx != book:
                    continue
                format = format.lower()
                fullpath = os.path.join(dirpath,name)
                if isbytestring(fullpath):
                    fullpath = fullpath.decode(filesystem_encoding)
                newpath = fullpath.replace(os.sep, '/')
                newpath = newpath + format
                temppath2 = os.path.join(dirpath,temppath)
                temppath2 = temppath2 + format
                if isbytestring(temppath2):
                    temppath2 = temppath2.decode(filesystem_encoding)
                temppath2 = temppath2.replace(os.sep, '/')
                try:
                    if os.path.isfile(temppath2):
                        if temppath2 != newpath:
                            shutil.copy(temppath2,newpath)
                        if os.path.isfile(newpath):
                            os.remove(temppath2)
                            if self.source_file_dict_in_use:
                                origpath = self.book_origpath_dict[book]
                                origgeneric,origext = os.path.splitext(origpath)
                                origext = as_unicode(origext)
                                if origgeneric in self.source_file_dict:
                                    name = self.data_txt_book_name_dict[book]                 #    name:  14730      as in:  14730.bmp
                                    if DEBUG: print("simple name: ", name)
                                    tmp_list = self.source_file_dict[origgeneric]
                                    tmp_list = self.convert_string_to_list(tmp_list)
                                    for ext in tmp_list:
                                        if as_unicode(ext) != as_unicode(origext):
                                            ext = as_unicode(ext)
                                            old = temppath + ext    # old is the original file name when first copied from the source directory, and has not ever been renamed to use calibre's specified name...
                                            #~ if DEBUG: print("old before joined with dirpath: ", old)
                                            #~ if DEBUG: print("dirpath: ", dirpath)
                                            old = os.path.join(dirpath,old)
                                            if isbytestring(old):
                                                old = old.decode(filesystem_encoding)
                                            old = old.replace(os.sep, '/')
                                            #~ if DEBUG: print("old after joined with dirpath: ", old)
                                            new = name + ext
                                            new = os.path.join(dirpath,new)
                                            if isbytestring(new):
                                                new = new.decode(filesystem_encoding)
                                            new = new.replace(os.sep, '/')
                                            #~ if DEBUG: print("new: ", new)
                                            if old != new:
                                                if os.path.isfile(old):
                                                    shutil.copy(old,new)
                                                    #~ if DEBUG: print("new was copied: ", new)
                                                    if os.path.isfile(new):
                                                        os.remove(old)
                                                        #~ if DEBUG: print("old was removed: ", old)
                                                        newformat = ext.upper()
                                                        newformat = newformat.replace(".","").strip()
                                                        self.add_extra_file_to_table_data(my_db,my_cursor,book,newformat,name)
                                                        sleep(0)
                                    #END FOR
                                else:
                                    if DEBUG: print("Error: ------- origgeneric not in self.source_file_dict: ", origgeneric)
                                    if DEBUG:
                                        #~ for k,v in self.source_file_dict.iteritems():
                                        for k,v in iteritems(self.source_file_dict):
                                            print("k,v in self.source_file_dict.iteritems(): ", k, ">>>>>", v)
                                        #END FOR
                                        my_db.close()
                                        return
                        else:
                            if DEBUG: print("Error: ------- newpath is not a valid file: ", newpath)
                    else:
                        if DEBUG: print("Error: ------- temppath is not a valid file: ", temppath)

                except Exception as e:
                    if DEBUG: print("os.rename oldpath,newpath,exception: ", temppath, newpath, " >>>>> ", e)
            #END FOR
            del tmp_rows
        #END FOR

        sleep(0)

        q = prefs['KEEP_ORIGINAL_PATH_IDENTIFIER']
        if q == unicode_type("False"):
            mysql = "DELETE FROM identifiers WHERE book = ? and type = 'zmfi_origpath' "
            my_cursor.execute("begin")
            for book in self.mfi_books_just_added_list:
                my_cursor.execute(mysql,([book]))
            #END FOR
            my_cursor.execute("commit")
        else:
            if iswindows:
                mysql = "UPDATE identifiers SET val = replace(val,'/','\\') WHERE type =  'zmfi_origpath' AND val LIKE '%/%' "
                my_cursor.execute("begin")
                my_cursor.execute(mysql)
                my_cursor.execute("commit")

        my_db.close()
    #-----------------------------------------------------------------------------------------
    def add_extra_file_to_table_data(self,my_db,my_cursor,book,format,name):
        my_cursor.execute("begin")
        mysql = "INSERT OR IGNORE INTO data (id,book,format,uncompressed_size,name) VALUES (null,?,?,0,?) "
        my_cursor.execute(mysql,(book,format,name))
        my_cursor.execute("commit")
        if DEBUG: print("Added to metadata.db table 'data': ", as_unicode(book), format, name)
    #-----------------------------------------------------------------------------------------
    def select_files_by_directory(self):

        if not self.okay_to_drop_files:
            return

        file_list = self.choose_files_to_add()
        if not file_list:
            return
        if DEBUG:
            for file in file_list:
                print("select_files_by_directory: ",file)
            #END FOR

        file_list = list(set(file_list))
        file_list.sort()

        self.messages_label.setText("Files Selected; Processing...")
        self.repaint()

        file_dict = {}

        for file in file_list:
            if isbytestring(file):
                file = file.decode(filesystem_encoding)
            file = file.replace(os.sep, '/')
            if self.mode_is_image:
                fbn,fext = os.path.splitext(os.path.basename(file))
                fext = fext.lower()
                if fext in IMAGE_FILE_FILTER:
                    file_dict[file] = None
                else:
                    continue
            elif self.mode_is_audio:
                fbn,fext = os.path.splitext(os.path.basename(file))
                fext = fext.lower()
                if fext in AUDIO_FILE_FILTER:
                    file_dict[file] = None
                else:
                    continue
            elif self.mode_is_video:
                fbn,fext = os.path.splitext(os.path.basename(file))
                fext = fext.lower()
                if fext in VIDEO_FILE_FILTER:
                    file_dict[file] = None
                else:
                    continue
            else:
                # anything goes here...can mix and match audio,video,image, and even ebooks...e.g. .pdf with .mp3 and .avi all in one book...
                fbn,fext = os.path.splitext(os.path.basename(file))
                fext = fext.lower()
                file_dict[file] = None
        #END FOR

        del file_list

        if len(file_dict) > 0:
            self.create_books_control(file_dict,False)
            self.mfi_exit()  # user must click the MFI icon if they want to import more items...
        else:
            error = "No files compatible with the 'mode' were selected.  Nothing done."
            error_dialog(self.maingui, _('No Compatible Files Specified:'),_(error), show=True)
    #-----------------------------------------------------------------------------------------
    def choose_files_to_add(self):
        name = "choose_mfi_files_to_add"
        title = "Choose Files to Add"
        all_files = True
        select_only_single_file = False
        filters = []
        window = None   # parent = None
        mode = QFileDialog.FileMode.ExistingFile if select_only_single_file else QFileDialog.FileMode.ExistingFiles
        fd = FileDialog(title=title, name=name, filters=filters, parent=window, add_all_files_filter=all_files, mode=mode)
        fd.setParent(None)
        if fd.accepted:
            return fd.get_files()
        return None
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    # MP3 metadata is no longer supported in MFI as of Calibre upgrade from Python 2 to Python 3.
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    # .hus & .vip embroidery file metadata
    #-----------------------------------------------------------------------------------------
    #~ http://www.123digitizing.com/embroidery-digitizing-file-formats.php
    #~ .hus - The stitch-based file format used by Husqvarna/Viking embroidery home sewing machines.
    #~ .vip - The stitch-based file format used by  Husqvarna Viking/Pfaff embroidery home sewing machines.
    #-----------------------------------------------------------------------------------------
    #~ 00000000: [5b af] [c8] [00] [9f 1c 00 00] - [01 00 00 00] [ea 01] [e5 01]   ........ ........
             #~ 0x00 	2 bytes 	HUS: Always [0x5B 0xAF]. VIP: Always 0x5D 0xFC
             #~ 0x02 	1 byte   	HUS: Must be [0xC8.] VIP: Must be 0x90
             #~ 0x03 	1 byte   	HUS: Always [0x00]. VIP: Always 0x01
             #~ 0x04 	[4 bytes] 	NumberOfStitches - The total number of stitches defined in this file.
             #~ 0x08 	[4 bytes] 	NumberOfColors - The total number of colors used in this pattern.
            #~ 0x0C 	[2 bytes] 	+X hoop-size offset as a positive number (same as +X from DST)
            #~ 0x0E 	[2 bytes] 	+Y hoop-size offset as a positive number (same as -Y from DST)
    #~ 00000010: [16 fe] [1b fe] 2c 00 00 00 - d9 03 00 00 ea 18 00 00   ........ ........
          #~ 0x10 	2 bytes 	-X hoop-size offset as a negative number (same as -X from DST)
          #~ 0x12 	2 bytes 	-Y hoop-size offset as a negative number (same as +Y from DST)
    #~ 00000020: [43 6f 6f 6c 20 44 6f 74] - [00 00] 00 00 1c a0 33 68   Cool.Dot ......3h
            #~ 0x20 	[8 bytes] 	String identifier
            #~ 0x28 	[2 bytes] 	*Unknown
    #-----------------------------------------------------------------------------------------
    def hus_vip_metadata_get(self,filename,file_extension):
        value = file_extension + " metadata not found"
        if file_extension != ".hus":
            if file_extension != ".vip":
                return value
        try:
            fp = open(filename,'rb')
            data = fp.read(4)     # read 1 * 4 bytes     # identifier of file type
            byteorder = sys.byteorder
            if byteorder == "big":
                cs = struct.calcsize('>1i')
                if len(data) != cs:
                    if DEBUG: print("big: len(data) != cs:",as_unicode(cs))
                    return value
            elif byteorder == "little":
                cs = struct.calcsize('<1i')
                if len(data) != cs:
                    if DEBUG: print("little: len(data) != cs:",as_unicode(cs))
                    return value
            else:
                return value

            data = fp.read(8)   # read 2 * 4 standard bytes  for an integer      stitches and colors

            if byteorder == "big":
                t = struct.unpack('>2i',data)
                if DEBUG: print("big tuple t: ",as_unicode(t))
            else:
                t = struct.unpack('<2i',data)
                if DEBUG: print("little tuple t: ",as_unicode(t))

            NumberOfStitches = 0
            NumberOfColors = 0

            for i,v in enumerate(t):
                if DEBUG: print("i,v", as_unicode(i),as_unicode(v))
                if i == 0:
                    NumberOfStitches = as_unicode(v)
                elif i == 1:
                    NumberOfColors = as_unicode(v)
            #END FOR

            fp.close()

            value = "Number of Stitches: " + NumberOfStitches + ";  Number of Colors: " + NumberOfColors

        except Exception as e:
            if DEBUG: print("Exception in hus_vip_metadata_get(self,filename,file_extension): ", e)

        return value

    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    # .PCS (Pfaff home) Design format embroidery file metadata
    #-----------------------------------------------------------------------------------------
    #~ http://www.achatina.de/sewing/main/TECHNICL.HTM
        #~ File structure :
        #~ Byte (hex)   	Value (hex) 	Explanation
        #~ 0000 	           32 	lead-in character for PCS-file
        #~ 0001 	           02 or 03 	hoop size. 02 = small , 03 = large
        #~ 0002-0003 	10 00 	No. of colors (always 16)
        #~ 0004-0043 	NN NN NN NN 	16 x 4 Byte color definition
        #~ 0044-0045 	NN NN 	No. of stitches in file , LSB first, max 65536 stitches, does NOT include color changes
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def pfaff_pcs_metadata_get(self,filename,file_extension):
        value = file_extension + " metadata not found"
        if file_extension != ".pcs":
            return value
        try:
            fp = open(filename,'rb')
            data = fp.read(1)     # read 1 * 1 bytes     # identifier of file type
            data = fp.read(1)     # read 1 * 1 bytes     # hoop size
            byteorder = sys.byteorder
            if byteorder == "big":
                cs = struct.calcsize('>1B')       #  standard size = 1 byte for B                         https://docs.python.org/2/library/struct.html#format-characters
                if len(data) != cs:
                    if DEBUG: print("big: len(data) != cs:",as_unicode(cs))
                    return value
            elif byteorder == "little":
                cs = struct.calcsize('<1B')
                if len(data) != cs:
                    if DEBUG: print("little: len(data) != cs:",as_unicode(cs))
                    return value
            else:
                return value

            if byteorder == "big":
                t = struct.unpack('>1B',data)
                if DEBUG: print("HoopSize big tuple t: ",as_unicode(t))
            else:
                t = struct.unpack('<1B',data)
                if DEBUG: print("HoopSize little tuple t: ",as_unicode(t))

            HoopSize = "unknown"
            NumberOfStitches = 0
            NumberOfColors = 0

            for i,v in enumerate(t):
                if DEBUG: print("i,v", as_unicode(i),as_unicode(v))
                HoopSize = as_unicode(v).replace("0","")
                if HoopSize == "2":
                    HoopSize = "small"
                elif  HoopSize == "3":
                    HoopSize = "large"
                break
            #END FOR

            if DEBUG: print("HoopSize: ", HoopSize)

            data = fp.read(2)   # read 2 * 1 bytes       number of colors

            if byteorder == "big":
                t = struct.unpack('>2B',data)
                if DEBUG: print("NumberOfColors big tuple t: ",as_unicode(t))
            else:
                t = struct.unpack('<2B',data)
                if DEBUG: print("NumberOfColors little tuple t: ",as_unicode(t))

            for i,v in enumerate(t):
                if DEBUG: print("NumberOfColors i,v", as_unicode(i),as_unicode(v))
                NumberOfColors = as_unicode(v)
                break
            #END FOR

            if DEBUG: print("NumberOfColors: ", NumberOfColors)

            data = fp.read(64)   # read 16 * 4 bytes       16 x 4 Byte color definition

            if byteorder == "big":
                t = struct.unpack('>16i',data)      # i, not B;  standard size = 4 bytes for i
                if DEBUG: print("ColorDefinitions big tuple t: ",as_unicode(t))
            else:
                t = struct.unpack('<16i',data)
                if DEBUG: print("ColorDefinitions little tuple t: ",as_unicode(t))   # t:  (11846883, 11846883, 16777215, 13026246, 11240271, 8098551, 7038969, 8430793, 10862286, 2976152, 4342082, 12441240, 2206257, 9744053, 10862286, 2976152)

            ColorDefinitions = '<p style="font-size:20px"<b>'
            for i,v in enumerate(t):
                if DEBUG: print("ColorDefinitions: i,v", as_unicode(i),as_unicode(v))
                v = hex(v)   #  0xb4c4e3  = bluish grey
                v = as_unicode(v)
                v = v[2: ]    #   b4c4e3  = bluish grey
                if not len(v) == 6:
                    continue
                s = '<a style="color:#[V]">Color: [V];  </a>'
                s = s.replace("[V]",v)
                ColorDefinitions = ColorDefinitions + as_unicode(s) + " "
            #END FOR

            ColorDefinitions = ColorDefinitions + "</b></p>"

            if DEBUG: print("ColorDefinitions: ", ColorDefinitions)

            value = "Number of Colors: " + NumberOfColors + "<br>Hoop Size: " + HoopSize + "<br> ColorDefinitions: " + ColorDefinitions

            fp.close()

            del fp

        except Exception as e:
            if DEBUG: print("Exception in def pfaff_pcs_metadata_get(self,filename,file_extension): ", as_unicode(e))

        if DEBUG: print(value)

        return value
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
#END OF mfi_dialog.py