﻿#!/usr/bin/env python
#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai

from __future__ import (unicode_literals, division, absolute_import, print_function)

import base64
import dateutil.parser
import dateutil.tz
import json
import random
import re
import time

from calibre.utils.date import parse_only_date

from calibre_plugins.overdrive_link.book import LibraryBook
from calibre_plugins.overdrive_link.formats import (FORMAT_ADOBE_EPUB, FORMAT_ADOBE_PDF, FORMAT_MP3)
from calibre_plugins.overdrive_link.library import SearchableLibrary
from calibre_plugins.overdrive_link.net import open_url
from calibre_plugins.overdrive_link.author_prep import normalize_author
from calibre_plugins.overdrive_link.title_prep import normalize_title
from calibre_plugins.overdrive_link.parseweb import (LibraryError, beautiful_soup)
from calibre_plugins.overdrive_link.json import js_value

from .python_transition import (IS_PYTHON2)
if IS_PYTHON2:
    from .python_transition import (http, repr, str, urllib)
else:
    import http.cookiejar
    import urllib.parse


__license__ = 'GPL v3'
__copyright__ = '2012-2022, John Howell <jhowell@acm.org>'


CLOUD_LIBRARY_FORMAT_NAMES = {
    'EPUB': FORMAT_ADOBE_EPUB,
    'EPUB3': FORMAT_ADOBE_EPUB,
    'PDF': FORMAT_ADOBE_PDF,
    'MP3': FORMAT_MP3,
    }


class CloudLibrary(SearchableLibrary):
    id = '3m'                   # Original name was 3M Cloud Library
    name = 'CloudLibrary'

    ebook_formats_supported = {FORMAT_ADOBE_EPUB, FORMAT_ADOBE_PDF}
    audiobook_formats_supported = {FORMAT_MP3}

    formats_supported = ebook_formats_supported | audiobook_formats_supported

    supports_recommendation = True
    sign_in_affects_get_current_availability = True     # can determine wait for books on hold

    @staticmethod
    def validate_library_id(library_id, migrate=True, config=None):
        if (':' in library_id) or ('/' in library_id):
            purl = urllib.parse.urlparse(library_id)
            m = re.match(r'^/library/([^/]+)/', purl.path)
            if not m:
                raise ValueError('cloudLibrary library id cannot be extracted from url: "%s"' % library_id)
            library_id = m.group(1)

        if not re.match(r'^([0-9a-zA-Z_]+)$', library_id):
            raise ValueError('cloudLibrary library id must be alphanumeric: "%s"' % library_id)

        return library_id

    @staticmethod
    def validate_book_id(book_id, library_id):
        if not re.match(r'^([0-9a-z]+)$', book_id):
            raise ValueError('cloudLibrary book id must be alphanumeric: "%s"' % book_id)

        return book_id

    @staticmethod
    def book_url(library_id, book_id):
        return 'http://ebook.yourcloudlibrary.com/library/%s/Featured/ItemDetail/%s' % (library_id, book_id)

    def __init__(self):
        self.cookiejar = http.cookiejar.CookieJar()

        self.recommendation_allowed = False
        self.search_audiobooks_allowed = False

        self.holds_checked = False
        self.holds = {}
        self.newrelic_id = self.agent_id = self.account_id = self.trust_key = self.license_key = self.application_id = None

    def sign_in(self, use_credentials):
        self.signin_required = True     # prevent access after failure to initialize
        self.cookiejar.clear()

        self.library_url = 'https://ebook.yourcloudlibrary.com/library/%s/Featured' % self.library_id

        response = open_url(self.log, self.library_url, cookiejar=self.cookiejar)

        if "The requested resource was not found" in response.data_string:
            raise Exception('Library id "%s" is unknown at cloudLibrary' % self.library_id)

        # Parse the html results for analysis
        soup = beautiful_soup(response.data_string)

        scripts = soup.findAll('script', attrs={'type': "text/javascript"})
        for script in scripts:
            if 'mmmWebPatron.UserSession=' in str(script):
                library_info = js_value(self.log, str(script), 'mmmWebPatron.UserSession=')["Library"]
                break
        else:
            raise Exception('Missing mmmWebPatron.UserSession')

        if library_info["Is3MCatalogVisible"]:
            self.log.info('Search for recommendable books is supported by %s' % self.name)
            self.recommendation_allowed = True

        if library_info["CanSearchForAudioBooks"]:
            self.log.info('Search for audiobooks is supported by %s' % self.name)
            self.search_audiobooks_allowed = True

        for script in scripts:
            if '.loader_config=' in str(script):
                loader_config = js_value(self.log, str(script), '.loader_config=', fix_keys=True)
                break
        else:
            raise Exception('Missing loader_config')

        self.newrelic_id = loader_config['xpid']
        self.agent_id = loader_config['agentID']
        self.account_id = loader_config['accountID']
        self.trust_key = loader_config['trustKey']      # value same as accountID
        self.license_key = loader_config['licenseKey']
        self.application_id = loader_config['applicationID']

        if self.card_number and use_credentials:
            self.signin_required = True

            url = 'https://ebook.yourcloudlibrary.com/uisvc/%s/Patron/LoginPatron' % self.library_id

            data = {}
            data["UserId"] = self.card_number
            data["Password"] = self.card_pin

            self.log.info('Post %s' % url)

            result = self.open_cloud_library_url(
                    url, data=json.dumps(data, separators=(',', ':')), log_request=False)

            if not result.get('Success', False):
                msg = (result.get('ErrorMessage', '') or result.get('Message', '') or
                       ' '.join(result.get('Messages', [])) or result.get('FailureReason', '') or
                       'Reason unknown')

                raise LibraryError('Sign in failed: %s' % msg)

            self.log.info('Sign in successful')
            self.signed_in = True
        else:
            self.signin_required = False

    def find_books(self, books, search_author, search_title, keyword_search):
        '''
        Search cloudLibrary for books that match an author/title (or subsets thereof) and
        return the found matches with cloudLibrary identifiers.
        books = Set of Books to be updated
        '''

        find_recommendable = self.config.check_recommendable and self.recommendation_allowed
        search_ebook = len(self.config.search_formats & self.ebook_formats_supported) > 0
        search_audiobook = len(self.config.search_formats & self.audiobook_formats_supported) > 0

        if search_ebook and search_audiobook:
            if self.search_audiobooks_allowed:
                media = 'all'
                desired_formats = self.formats_supported
            else:
                media = 'ebook'
                desired_formats = self.ebook_formats_supported
        elif search_ebook:
            media = 'ebook'
            desired_formats = self.ebook_formats_supported
        elif search_audiobook and self.search_audiobooks_allowed:
            media = 'audio'
            desired_formats = self.audiobook_formats_supported
        else:
            return False    # no desired formats available for search

        RESULTS_PER_PAGE = 20

        page_num = 1
        has_more = True
        search_id = None
        results_processed = 0

        while (has_more):
            url = 'https://ebook.yourcloudlibrary.com/uisvc/%s/Search/%s?media=%s&src=%s' % (
                self.library_id, 'CatalogSearch' if page_num == 1 else 'Next', media, '3m' if find_recommendable else 'lib')

            data = {}

            if page_num == 1:
                data["SearchString"] = ' '.join([search_author, search_title]).strip()
                data["SortBy"] = ''     # title, author, publication_date, creationDate, rating
            else:
                data["SearchId"] = search_id

            data["from"] = (page_num - 1) * RESULTS_PER_PAGE
            data["count"] = RESULTS_PER_PAGE

            result = self.open_cloud_library_url(url, data=json.dumps(data, separators=(',', ':')))

            search_id = result.get('SearchId', '')
            has_more = result.get('HasMore', False)
            num_results = len(result.get('Items', []))

            if has_more and num_results != RESULTS_PER_PAGE:
                self.log.error('Expected %d results, but received %d' % (RESULTS_PER_PAGE, num_results))

            for item in result.get('Items', []):
                document_id = item['Id']

                title = normalize_title(item.get('Title', ''))
                subtitle = item.get('SubTitle', '')
                if subtitle:
                    title = '%s: %s' % (title, subtitle)

                authors = [normalize_author(a, unreverse=True) for a in item.get('Authors', '').split(';')]

                publisher = item.get('Publisher', '')

                if 'PublicationDate' in item:
                    pubdate = dateutil.parser.parse(str(item['PublicationDate'])).replace(tzinfo=dateutil.tz.tzutc())  # ISO 8601 format
                elif 'PublicationYear' in item:
                    pubdate = parse_only_date(str(item['PublicationYear']), assume_utc=True)
                else:
                    pubdate = None

                isbn = item.get('ISBN', '')

                format = item.get('Format', '')
                if format in CLOUD_LIBRARY_FORMAT_NAMES:
                    formats = set([CLOUD_LIBRARY_FORMAT_NAMES[format]])
                else:
                    self.log.error('Unexpected format: %s' % format)
                    continue

                allowed_action = item.get('AllowedPatronAction', 'Borrow')      # always 'None' if not signed in
                available = recommendable = False
                #self.log.info('AllowedPatronAction=%s, signed_in=%s' % (allowed_action, self.signed_in))

                if allowed_action in ['Borrow', 'PlaceHold', 'RemoveHold', 'Return'] or (allowed_action == 'None' and not self.signed_in):
                    available = True
                elif allowed_action in ['Suggest', 'RemoveSuggest']:
                    recommendable = True
                elif allowed_action not in ['None', 'None_BorrowLimit']:
                    self.log.error('Unknown AllowedPatronAction (%s) for %s' % (allowed_action, document_id))
                    continue

                lbook = LibraryBook(
                        authors=authors, title=title,
                        publisher=publisher, pubdate=pubdate, isbn=isbn, formats=formats,
                        available=available, recommendable=recommendable,
                        lib=self, book_id=document_id, search_author=search_author,
                        allow_get_from_cache=False)  # prevent old cached results from being used

                if formats.isdisjoint(desired_formats):
                    self.log.info('Ignoring wrong format: %s' % repr(lbook))
                elif not (available or find_recommendable):
                    self.log.info('Ignoring unavailable book: %s' % repr(lbook))
                else:
                    self.log.info('Found: %s' % repr(lbook))
                    books.add(lbook)

                results_processed += 1

            page_num += 1

        return False

    def get_current_book_availability(self, book_id):
        self.check_current_holds()
        available_copies = 0
        availability_date = None
        number_waiting_overall = 0
        is_on_hold = book_id in self.holds

        if is_on_hold:
            self.log.info('Using availability date from %s holds list for %s' % (self.name, book_id))
            availability_date = self.holds[book_id]

        else:
            # Can get info for book using:
            # https://ebook.yourcloudlibrary.com/uisvc/(library-id)/Item/GetItem?id=d6vez9
            # Accept: application/json, text/plain, */*

            item = self.open_cloud_library_url(
                        'https://ebook.yourcloudlibrary.com/uisvc/%s/Item/GetItem?id=%s&media=all&src=lib' % (self.library_id, book_id),
                        return_errors=True)

            error = item.get('error')
            if error:
                if error in ["The requested catalog item was not found", "The digital item is either not valid or no longer in the catalog"]:
                    self.log.info('Book not available: %s' % error)
                    return False

                raise LibraryError(error)

            available_copies = 0
            availability_date = None
            number_waiting_overall = 0
            allowed_action = item.get('AllowedPatronAction', '')

            if allowed_action in ['Borrow', 'Return']:
                available_copies = 1

            elif allowed_action in ['PlaceHold', 'RemoveHold']:
                if allowed_action == 'RemoveHold':
                    self.log.info('Book is on hold for current user')

                availability_date = item.get("GuaranteedAvilabilityDate", None)
                if availability_date:
                    availability_date = dateutil.parser.parse(availability_date).replace(tzinfo=dateutil.tz.tzutc())
                else:
                    number_waiting_overall = 1      # wait not provided, guess

            elif allowed_action in ['None', 'None_BorrowLimit', 'Suggest', 'RemoveSuggest']:
                self.log.info('Book not available: AllowedPatronAction=%s' % allowed_action)
                return False

            else:
                self.log.warn('Unknown AllowedPatronAction (%s)' % allowed_action)

        # estimate availability
        wait_weeks = self.calculate_wait_weeks(
                library_copies=1, available_copies=available_copies,
                number_waiting_overall=number_waiting_overall, availability_date=availability_date)

        return (wait_weeks, is_on_hold, book_id)

    def check_current_holds(self):
        if self.holds_checked or not self.signed_in:
            return

        self.holds_checked = True

        try:
            result = self.open_cloud_library_url('https://ebook.yourcloudlibrary.com/uisvc/%s/Patron' % self.library_id)

            book_count = result.get('BookCount', {})

            for book_id, date in book_count.get('HoldEndDates', {}).items():
                self.log.info('Found hold at %s: book_id=%s available_date=%s' % (self.name, book_id, date))
                self.holds[book_id] = dateutil.parser.parse(date).replace(tzinfo=dateutil.tz.tzutc())  # ISO 8601 format

        except Exception as e:
            self.log.exception('', e)

    def open_cloud_library_url(self, url, **kwargs):
        return_errors = kwargs.pop('return_errors', False)

        ty = "Browser"
        id = "".join(random.choice("0123456789abcdef") for _ in range(16))
        tr = "".join(random.choice("0123456789abcdef") for _ in range(32))
        ti = int(time.time() * 1000)

        newrelic = '{"v":[0,1],"d":{"ty":"%s","ac":"%s","ap":"%s","id":"%s","tr":"%s","ti":%s}}' % (ty, self.account_id, self.agent_id, id, tr, ti)

        kwargs['cookiejar'] = self.cookiejar
        kwargs['content_type'] = "application/json;charset=UTF-8"
        kwargs['referer'] = self.library_url
        kwargs['addheaders'] = [
            ("Accept", "application/json, text/plain, */*"),
            ("X-NewRelic-ID", self.newrelic_id),
            ("newrelic", base64.b64encode(newrelic.encode('ascii')).decode('ascii')),
            ("tracestate", "%s@nr=0-1-%s-%s-%s----%s" % (self.account_id, self.trust_key, self.agent_id, id, ti)),
            ("traceparent", "00-%s-%s-01" % (tr, id)),
            ]
        kwargs['retry_on_internal_error'] = False
        kwargs['expect_errors'] = [401, 500]

        response = open_url(self.log, url, **kwargs)

        if response.is_httperror_exception:
            result = json.loads(response.response_data)    # some API errors returned this way

            msg = (result.get('ErrorMessage', '') or result.get('Message', '') or
                   ' '.join(result.get('Messages', [])) or result.get('FailureReason', '') or
                   'Reason unknown')

            if return_errors:
                return {'error': msg}

            raise LibraryError('HTTP error %d: %s' % (response.code, msg))

        return json.loads(response.data_string)
