
#!/Python3/python
# -*- coding: utf-8 -*-

from __future__ import unicode_literals, division, absolute_import, print_function

#********************************************************************************#
#                                                                                #
# MIT Licence(OSI)                                                               #
# Copyright (c) 2017 Bill Thompson                                               #
#                                                                                #
# Permission is hereby granted, free of charge, to any person obtaining a copy   # 
# of this software and associated documentation files (the "Software"), to deal  # 
# in the Software without restriction, including without limitation the rights   #
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell      #
# copies of the Software, and to permit persons to whom the Software is          #
# furnished to do so, subject to the following conditions:                       # 
#                                                                                #
# The above copyright notice and this permission notice shall be included in all #
# copies or substantial portions of the Software.                                #
#                                                                                # 
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR     # 
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,       #
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE    #
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER         # 
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,  # 
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE  # 
# SOFTWARE.                                                                      #
#                                                                                #  
#********************************************************************************#
import os, os.path, sys, codecs, shutil, inspect, time
import options
#from tempfile import mkdtemp                 
from updater import updateCheck
import tkinter as tk
import tkinter.messagebox as mbox

try:
    from sigil_bs4 import BeautifulSoup
except:
    from bs4 import BeautifulSoup
    
iswindows = sys.platform.startswith('win')
isosx = sys.platform.startswith('darwin')
islinux = sys.platform.startswith('linux')    

SITE_URL = "https://www.mobileread.com/forums/showpost.php?p=4015437&postcount=1"
PLUGIN_PATH = os.path.dirname(os.path.abspath(inspect.getfile(inspect.currentframe())))
options.PLUGIN_PATH = PLUGIN_PATH


metadata_list    =  [
                      ["<dc:identifier", "UUID", "ERROR"],
                      ["<dc:title", "Book title", "ERROR"],
                      ["<dc:language", "Language", "ERROR"],
                      ["<dc:publisher", "Publisher name","ERROR"],
                      ["opf:role=\"aut\"", "Author name", "ERROR"],
                      ["opf:event=\"creation\"", "Creation date", "ERROR"],
                      ["opf:event=\"publication\"", "Publication date", "WARNING"],
                      ["opf:event=\"modification\"", "Modification date", "WARNING"],
                      ["<dc:subject", "Book subject", "WARNING"],
                      ["<dc:rights", "Copyright","WARNING"]
                    ]
                     
guide_list       =  [
                      ["toc", "TOC", "ERROR"],
                      ["cover", "Book cover", "WARNING"],
                      ["text", "Begin read", "WARNING"]
                    ]

third_party_list =  [
                      "Jutoh",
                      "Calibre",
                      "Sigil",
                      "Scrivener",
                      "Epubor",
                      "Atlantis"
                    ]

def checkOPFData(bk):
    """ Check the data in the OPF <metadata> and <guide> sections
    """
    results = []
    E_results = []
    W_results = []
    
    results.extend(checkMissingMetadata(bk))
    results.extend(checkDuplicateMetadataEntries(bk))
    results.extend(checkThirdPartyMetadataEntries(bk))
    results.extend(checkGuideEntries(bk))
    
    print('>>> Show ERRORS and WARNINGS only:')
    for e in results:
        print(e)
    
    return(results)

def checkMissingMetadata(bk): 
    print('\nIn checkMissingMetadata()...\n')
    prefs = bk.getPrefs()
    filename = 'content.opf'
    all_results = []
    meta_lines = []
    opf = bk.get_opf()
    
    # get the data from the OPF metadata section
    mdata = bk.getmetadataxml()
    meta_lines = mdata.splitlines(True) 
    data = []
    
    for data in metadata_list:
        name = data[1]
        severity = data[2]
        if data[0] not in " ".join(meta_lines):
            msg = severity + ' [METADATA]: ' + name + ' is missing.'
            bk.add_result(severity.lower(), filename, 0, msg)
            msg2 = severity + ' [METADATA]:--> ' + name + ' is missing.'
            all_results.append('  ' + msg2)    
        else:
            if prefs['show_errors_and_warnings_only'] == False:
                msg = 'INFO [METADATA]: ' + name + ' is present.' 
                linum = getLineNumber2(opf, data[0])
                bk.add_result('info', filename, linum, msg)

    return(all_results)    

def checkGuideEntries(bk):
    print('\nIn checkGuideEntries()...\n')
    
    prefs = bk.getPrefs()
    filename = 'content.opf'
    opf = bk.get_opf()
    new_list = []
    items_str = ''
    all_results = []
    data = []
    
    for (typer, title, href) in bk.getguide():
        new_list.append(typer)

    guide_items_str = " ".join(new_list)
    for data in guide_list:
        metatag = data[0]
        name = data[1]
        severity = data[2]
        if metatag not in guide_items_str:
            msg = severity + ' [GUIDE]: ' + name + ' guide reference is missing.'
            bk.add_result(severity.lower(), filename, 0, msg)
            msg2 = severity + ' [GUIDE]:--> ' + name + ' is missing.'
            all_results.append('  ' + msg2)
        else:
            if prefs['show_errors_and_warnings_only'] == False:
                msg = 'INFO [GUIDE]: ' + name + ' guide reference is present.' 
                linum = getLineNumber2(opf, metatag)
                bk.add_result('info', filename, linum, msg)

    time.sleep(0.5)
    return(all_results)      
    
def checkThirdPartyMetadataEntries(bk):
    """ Removes all the useless promtional 
        third party entries from the 
        <metadata> section of the OPF file.
    """
    print('\nIn checkThirdPartyMetadataEntries()...')
    
    all_results = []
    filename = 'content.opf'
    prefs = bk.getPrefs()
    data = bk.getmetadataxml()
    lines = data.splitlines(True)
    new_lines = []
    linenum = []
    
    for line in lines:
        for text in prefs['third_party_list']:
            if text.lower() in line.lower() and \
                '<dc:identifier' not in line.lower():
                linenum = getLineNumber(bk.get_opf(), line.strip())
                for num in linenum:
                    xml = BeautifulSoup(line, 'xml')
                    if '<dc:contributor' in line and (xml.get_text() != None or xml.get_text() != ''):
                        msg = 'WARNING [METADATA]: Third party promotion data can be removed: { \'\'' + xml.get_text() + '\'\' in line }' 
                        bk.add_result('warning', filename, num, msg)
                        msg2 = '  WARNING [METADATA] Line ' + num + ':--> Third party promotion data can be removed: { "' + xml.get_text() + '" in line }'
                        all_results.append(msg2)
                        break
                    else:
                        if 'meta' in line and 'name=' in line:
                            msg = 'WARNING [METADATA]: Third party promotion data can be removed:  { \'\'' + text + '\'\' in line }' 
                            bk.add_result('warning', filename, num, msg)
                            msg2 = '  WARNING [METADATA] Line ' + num + ':--> Third party promotion data can be removed: { "' + text + '" in line }'
                            all_results.append(msg2)
                            break       
                            
    return(all_results)    
    
def checkDuplicateMetadataEntries(bk):
    print('In checkDuplicateMetadataEntries()...\n')
    
    prefs = bk.getPrefs()
    filename = 'content.opf'
    id_list = []
    id_results = []
    all_results = []
    id_results.append('\nCheck for duplicate <metadata> entries:\n') 
    
    lines = []
    data = bk.getmetadataxml()
    lines = data.splitlines(True)
    
    id = []
    count = 0
    print('Check for duplicate <metadata> entries:')
    for id in metadata_list:
        for line in lines:
            if id[0] in line:
                id_list = getLineNumber(bk.get_opf(), id[0]) 
                count += 1
                
        if count > 1 and id[0] == '<dc:identifier':
            msg = 'ERROR [METADATA]: Duplicate UUID entry found.'
            for linenum in id_list:
                bk.add_result('error', filename, linenum, msg)
                msg2 = '  ERROR [METADATA] Line ' + linenum + ':--> Duplicate UUID entry found.'
                all_results.append(msg2)

        elif count > 1 and id[0] == '<dc:title':
            msg = 'ERROR [METADATA]: Duplicate book title entry found.' 
            for linenum in id_list:
                bk.add_result('error', filename, linenum, msg)
                msg2 = '  ERROR [METADATA] Line ' + linenum + ':--> Duplicate book title entry found.'
                all_results.append(msg2)
            
        elif count > 1 and id[0] == 'opf:role="aut"':
            msg = 'ERROR [METADATA]: Duplicate author name entry found.'       
            for linenum in id_list:
                bk.add_result('error', filename, linenum, msg)
                msg2 = '  ERROR [METADATA] Line ' + linenum + ':--> Duplicate author name entry found.'
                all_results.append(msg2)
                
        elif count > 1 and id[0] == '<dc:language':
            msg = 'ERROR [METADATA]: Duplicate language entry found.' 
            for linenum in id_list:
                bk.add_result('error', filename, linenum, msg)
                msg2 = '  ERROR [METADATA] Line ' + linenum + ':--> Duplicate language entry found.'
                all_results.append(msg2)
                
        elif count > 1 and id[0] == '<dc:subject':
            msg = 'ERROR [METADATA]: Duplicate subject entry found.' 
            for linenum in id_list:
                bk.add_result('error', filename, linenum, msg)
                msg2 = '  ERROR [METADATA] Line ' + linenum + ':--> Duplicate subject entry found.'
                all_results.append(msg2)
                
        elif count > 1 and id[0] == '<dc:publisher':
            msg = 'ERROR [METADATA]: Duplicate publisher entry found.'     
            for linenum in id_list:
                bk.add_result('error', filename, linenum, msg)
                msg2 = '  ERROR [METADATA] Line ' + linenum + ':--> Duplicate publisher entry found.'
                all_results.append(msg2)
                
        elif count > 1 and id[0] == 'opf:event="creation"':
            msg = 'ERROR [METADATA]: Duplicate creation date entry found.' 
            for linenum in id_list:
                bk.add_result('error', filename, linenum, msg)
                msg2 = '  ERROR [METADATA] Line ' + linenum + ':--> Duplicate creation date entry found.'
                all_results.append(msg2)
                
        elif count > 1 and id[0] == 'opf:event="publication"':
            msg = 'ERROR [METADATA]: Duplicate publication date entry found.' 
            for linenum in id_list:
                bk.add_result('error', filename, linenum, msg)
                msg2 = '  ERROR [METADATA] Line ' + linenum + ':--> Duplicate publication date entry found.'
                all_results.append(msg2)
                
        elif count > 1 and id[0] == 'opf:event="modification"':
            msg = 'ERROR [METADATA]: Duplicate modification date entry found.'
            for linenum in id_list:
                bk.add_result('error', filename, linenum, msg)
                msg2 = '  ERROR [METADATA] Line ' + linenum + ':--> Duplicate modification date entry found.'
                all_results.append(msg2)
                
        elif count > 1 and id[0] == '<dc:rights':
            msg = 'ERROR [METADATA]: Duplicate rights entry found.'    
            for linenum in id_list:
                bk.add_result('error', filename, linenum, msg)
                msg2 = '  ERROR [METADATA] Line ' + linenum + ':--> Duplicate copyright entry found.'
                all_results.append(msg2)
                
        count = 0    
        
    return(all_results)        
            

def getLineNumber(opf, text):
    """ Gets the line numbers and returns a list
    """
    print('\n >>> In getLineNumber()...')
    opf_data = opf.splitlines()
    linenum = []
    
    # assign a line number to the line
    for index, line in enumerate(opf_data):
        if text in line:
            linum = index + 1
            linenum.append(str(linum))

    if len(linenum) == 0:
        linenum.append('0')
        
    return(linenum)
    
def getLineNumber2(opf, text):
    """ Gets the line number and returns a a number as a string
    """
    print('\n >>> In getLineNumber2()...')
    opf_data = opf.splitlines()
    linenum = ''
    
    # assign a line number to the line
    for index, line in enumerate(opf_data):
        if text in line:
            linum = index + 1
            linenum = str(linum)
            
    if linenum == '' or linenum == None:
        linenum = '0'
        
    return(linenum)  
    
def saveLogFile(bk, error_list):
    """ Saves all the results to a log file.
    """
    
    error_list.sort()
    prefs = bk.getPrefs()
    
    outfile = os.path.join(prefs['logfile_path'], prefs['logfile_name'])
    outfp = open(outfile, 'wt', encoding='utf-8')
    data = '\n\n<<<<<<<<<<<<<   VerifyOPFData Log File   >>>>>>>>>>>>>>\n'
    data +='<<<<<<<<<<<<< (Errors and Warnings only) >>>>>>>>>>>>>>\n\n'
    outfp.write(data)
    if len(error_list) != 0: 
        for line in error_list:
            outfp.write(line + '\n')
    else:
        outfp.write('NO ERRORTS OR WARNINGS !!\n')
        
    outfp.close()
    msg = 'The LOG FILE has been saved to: \n\n' + outfile
    show_msgbox('Log File Location', msg, msgtype='info')
        
    return(0)    


def show_msgbox(title, msg, msgtype='info'):
    """ For general information, warnings and errors
    """
    localRoot = tk.Tk()
    localRoot.withdraw()
    localRoot.option_add('*font', 'Helvetica -12')
    localRoot.quit()
    if msgtype == 'info':
        return(mbox.showinfo(title, msg))
    elif msgtype == 'warning':
        return(mbox.showwarning(title, msg))
    elif msgtype == 'error':
        return(mbox.showerror(title, msg))
        
     
def is_connected():
    try:
        sock = socket.create_connection(('8.8.8.8', 53), 1)
        sock.close()
        return True
    except:
        pass

    return(False)      
    
    
def run(bk):
    print('Python version: ', sys.version, '\n')
    print('Running VerifyOPFData plugin...')
    error_list = []
    
    if is_connected: 
        #check for new plugin versions
        latest_version, installed_version = updateCheck(SITE_URL, PLUGIN_PATH)
        if latest_version and latest_version != installed_version:
            options.NEW_PLUGIN_VERSION = True
            options.MSG_NEW_VERSION_AVAILABLE = "A new plugin version is now available from MR - v" + latest_version 
    
    fnames = []
    for id, href in bk.text_iter():
        fnames.append(os.path.basename(href))
            
    # default epub error    
    if len(fnames) == 1 and fnames[0] == 'Section0001.xhtml':
        msg = 'Default epub contains no data. Please try again.'
        print('\n>>> File Type Error: ' + msg + '\n\nAbort plugin...')
        show_msgbox('Error', msg, msgtype='error')
        return(0)    
        
    # check the file type -- html or epub
    if len(fnames) == 1:
        msg = 'Epub contains insufficient data or is the wrong file type.\n\nPlease try again.'
        print('\n>>> File Type Error: ' + msg + '\n\nAbort plugin...')
        show_msgbox('File Type Error', msg, msgtype='error')
        return(0)
        
    prefs = bk.getPrefs()
    if 'third_party_list' not in prefs:
        prefs['third_party_list'] = third_party_list
    if 'save_logfile' not in prefs:
        prefs['save_logfile'] = False
    if 'logfile_name' not in prefs:
        prefs['logfile_name'] = 'OPF_LOG.txt'    
    if 'logfile_path' not in prefs:
        prefs['logfile_path'] = os.path.join(os.path.expanduser('~'), 'Desktop')
    if 'show_errors_and_warnings_only' not in prefs:
        prefs['show_errors_and_warnings_only'] = True     
       
    # check and process the OPF file
    error_list = checkOPFData(bk)
        
    # Notify the user if there are no errors or warnings found
    if 'ERROR' not in " ".join(error_list) and \
        'WARNING' not in " ".join(error_list):   
        msg = 'No ERRORS or WARNINGS !!'
        show_msgbox('Error Report', msg, msgtype='info') 
    
    # if True then save all errors/warnings to a log file
    if prefs['save_logfile'] == True:    
        saveLogFile(bk, error_list) 

    # inform user if new plugin version is available
    if options.NEW_PLUGIN_VERSION == True:
        msg = options.MSG_NEW_VERSION_AVAILABLE
        show_msgbox('CheckInternalLinks', msg, msgtype='info') 
    
    bk.savePrefs(prefs)    
    print('\n-- Completed SUCCESSFULLY...')
    return(0)                
    
def main():
    print('I reached main when I should not have\n')
    return(-1)

if __name__ == "__main__":
    sys.exit(main())                         