# -*- coding: utf-8 -*-
__license__   = 'GPL v3'
__copyright__ = '2017,2018,2019,2020,2021,2022,2023 DaltonST'
__my_version__ = "1.0.7"  #Qt.core

from qt.core import (Qt, QDialog, QComboBox, QObject,
                                        QLabel, QWidget, QPushButton, QTabWidget,
                                        QVBoxLayout, QScrollArea, QLayout, QHBoxLayout,
                                        QSize, QSizePolicy, QIcon, QGroupBox, QDialogButtonBox, QToolTip, QFont)
import os,sys
import collections
from copy import deepcopy
from functools import partial

from calibre.constants import DEBUG
from calibre.gui2 import gprefs
from calibre.gui2 import error_dialog, question_dialog

from polyglot.builtins import as_unicode, iteritems, map

from calibre_plugins.library_splitter.config import prefs

#-----------------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
class SizePersistedDialog(QDialog):
    initial_extra_size = QSize(10, 10)

    def __init__(self, parent, unique_pref_name):
        QDialog.__init__(self, parent)
        self.unique_pref_name = unique_pref_name
        self.geom = gprefs.get(unique_pref_name, None)

    def resize_dialog(self):
        if self.geom is None:
            self.resize(self.sizeHint()+self.initial_extra_size)
        else:
            self.restoreGeometry(self.geom)

    def save_dialog_geometry(self):
        geom = bytearray(self.saveGeometry())
        gprefs[self.unique_pref_name] = geom
#-----------------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
class LibrarySplitterDialog(SizePersistedDialog):
    #-----------------------------------------------------------------------------------------
    def __init__(self,gui,icon,guidb,exit_all,copytolibraryaction_object):
        parent = gui
        unique_pref_name = 'library_splitter:gui_parameters_dialog'
        SizePersistedDialog.__init__(self, parent, unique_pref_name)
        #-----------------------------------------------------
        self.gui = gui
        #-----------------------------------------------------
        self.guidb = guidb
        #-----------------------------------------------------
        self.icon = icon
        #-----------------------------------------------------
        self.exit_all = exit_all
        #-----------------------------------------------------
        self.copytolibraryaction_object = copytolibraryaction_object
        #-----------------------------------------------------
        self.myparentprefs = collections.OrderedDict([])
        prefsdefaults = deepcopy(prefs.defaults)
        tmp_list = []
        for k,v in iteritems(prefs):
            tmp_list.append(k)
        #END FOR
        for k,v in iteritems(prefsdefaults):
            tmp_list.append(k)
        #END FOR
        tmp_set = set(tmp_list)
        tmp_list = list(tmp_set)  #no duplicates
        del tmp_set
        tmp_list.sort()
        for k in tmp_list:
            self.myparentprefs[k] = " "  # ordered by key
        #END FOR
        del tmp_list
        for k,v in iteritems(prefs):
            self.myparentprefs[k] = v
        #END FOR
        for k,v in iteritems(prefsdefaults):
            if not k in prefs:
                prefs[k] = v
            else:
                if not prefs[k] > " ":
                    prefs[k] = v
            if not k in self.myparentprefs:
                self.myparentprefs[k] = v
            else:
                if not self.myparentprefs[k] > " ":
                    self.myparentprefs[k] = v
        #END FOR
        for k,v in iteritems(self.myparentprefs):
            prefs[k] = v
        #END FOR
        prefs  #prefs now synched

        #-----------------------------------------------------
        #-----------------------------------------------------
        self.init_tooltips_for_parent()
        self.setToolTip(self.parent_tooltip)
        #-----------------------------------------------------
        #-----------------------------------------------------
        # Tab 0: LibrarySplitterTab
        #-----------------------------------------------------
        #-----------------------------------------------------
        from calibre_plugins.library_splitter.library_splitter_dialog import LibrarySplitterTab
        self.LibrarySplitterTab = LibrarySplitterTab(self.gui,self.guidb,self.icon,self.myparentprefs,self.save_dialog_geometry,self.copytolibraryaction_object)
        #-----------------------------------------------------
        #-----------------------------------------------------
        #-----------------------------------------------------
        #    Parent             LibrarySplitterDialog
        #-----------------------------------------------------
        font = QFont()
        font.setBold(False)
        font.setPointSize(10)

        tablabel_font = QFont()
        tablabel_font.setBold(False)
        tablabel_font.setPointSize(10)

        #-----------------------------------------------------
        self.setWindowTitle('Library Splitter')
        self.setWindowIcon(icon)
        #-----------------------------------------------------
        self.layout_frame = QVBoxLayout()
        self.layout_frame.setAlignment(Qt.AlignLeft)
        self.setLayout(self.layout_frame)
        #-----------------------------------------------------
        #-----------------------------------------------------
        #-----------------------------------------------------
        n_width = 600

        self.LStabWidget = QTabWidget()
        self.LStabWidget.setMinimumWidth(n_width)
        self.LStabWidget.setFont(tablabel_font)

        self.LStabWidget.addTab(self.LibrarySplitterTab,"Split Current Library")
        self.LibrarySplitterTab.setToolTip("<p style='white-space:wrap'>Copy or Move some or all of your books from the Current Library to one or more Target Libraries.  Each book's Target Library (either via its OS Path or just its Library Name) is specified in the 'Target Library Names' Custom Column that you have created and then populated as desired.")
        self.LibrarySplitterTab.setMinimumWidth(n_width)

           #-----------------------------------------------------
        self.layout_frame.addWidget(self.LStabWidget)
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.button_box = QDialogButtonBox()
        self.button_box.setOrientation(Qt.Horizontal)
        self.button_box.setCenterButtons(True)

        self.layout_frame.addWidget(self.button_box)
        #-----------------------------------------------------
        self.push_button_exit_only = QPushButton("Exit")
        self.push_button_exit_only.clicked.connect(self.exit_all)
        self.push_button_exit_only.setDefault(False)
        self.push_button_exit_only.setFont(font)
        self.push_button_exit_only.setToolTip("<p style='white-space:wrap'>Exit without saving.")

        self.button_box.addButton(self.push_button_exit_only,QDialogButtonBox.AcceptRole)
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.resize_dialog()
        #-----------------------------------------------------
        self.LStabWidget.setCurrentIndex(0)
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------

    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    # OTHER
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def init_tooltips_for_parent(self):
        self.setStyleSheet("QToolTip { color: #ffffff; background-color: #2a82da; border: 1px solid white; }")
        self.parent_tooltip = "<p style='white-space:wrap'>" + "<p style='white-space:wrap'>Copy or Move some or all of your books from the Current Library to one or more Target Libraries.  Each book's Target Library (either via its OS Path or just its Library Name) is specified in the 'Target Library Names' Custom Column that you have created and then populated as desired."
    #-----------------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
class LibrarySplitterTab(QWidget):
    def __init__(self,mygui,myguidb,icon,mymainprefs,mysavedialoggeometry,copytolibraryaction_object):
        super(LibrarySplitterTab, self).__init__()
        #-----------------------------------------------------
        self.gui = mygui
        #-----------------------------------------------------
        self.guidb = myguidb
        #-----------------------------------------------------
        self.icon = icon
        #-----------------------------------------------------
        self.mytabprefs = mymainprefs
        #-----------------------------------------------------
        self.save_dialog_geometry = mysavedialoggeometry
        #-----------------------------------------------------
        self.copytolibraryaction_object = copytolibraryaction_object
        #-----------------------------------------------------
        font = QFont()
        font.setBold(False)
        font.setPointSize(10)
        #-----------------------------------------------------
        self.layout_top = QVBoxLayout()
        self.layout_top.setSpacing(0)
        self.layout_top.setAlignment(Qt.AlignLeft)
        self.setLayout(self.layout_top)
        #-----------------------------------------------------
        self.scroll_area_frame = QScrollArea()
        self.scroll_area_frame.setAlignment(Qt.AlignLeft)
        self.scroll_area_frame.setWidgetResizable(True)
        self.scroll_area_frame.ensureVisible(600,600)

        self.layout_top.addWidget(self.scroll_area_frame)       # the scroll area is now the child of the parent of self.layout_top

        # NOTE: the self.scroll_area_frame.setWidget(self.scroll_widget) is at the end of the init() AFTER all children have been created and assigned to a layout...

        #-----------------------------------------------------
        self.scroll_widget = QWidget()
        self.layout_top.addWidget(self.scroll_widget)           # causes automatic reparenting of QWidget to the parent of self.layout_top, which is:  self .
        #-----------------------------------------------------
        self.layout_frame = QVBoxLayout()
        self.layout_frame.setSpacing(0)
        self.layout_frame.setAlignment(Qt.AlignLeft)

        self.scroll_widget.setLayout(self.layout_frame)        # causes automatic reparenting of any widget later added to self.layout_frame to the parent of self.layout_frame, which is:  QWidget .

        #-----------------------------------------------------
        self.ls_groupbox = QGroupBox('')
        #~ self.ls_groupbox.setMaximumWidth(400)
        self.ls_groupbox.setToolTip("<p style='white-space:wrap'>The settings that control 'Library Splitter'.")
        self.layout_frame.addWidget(self.ls_groupbox)

        self.ls_layout = QVBoxLayout()
        self.ls_layout.setSpacing(0)
        self.ls_layout.setAlignment(Qt.AlignCenter)
        self.ls_groupbox.setLayout(self.ls_layout)
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.spacing0 = QLabel()
        self.layout_frame.addWidget(self.spacing0)
        self.spacing0.setMaximumHeight(20)
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.combo_layout = QHBoxLayout()
        self.combo_layout.setAlignment(Qt.AlignCenter)
        self.ls_layout.addLayout(self.combo_layout)

        self.name_label = QLabel()
        self.name_label.setText("'Target Library Names' Custom Column:   ")
        self.name_label.setToolTip("<p style='white-space:wrap'>This is the Custom Column of type 'comments' and of the 'plain text' or 'short text' variety that will have the Target Library (either via its OS Path or just its Library Name) for each book to be copied or moved from the Current Library.")
        self.name_label.setFont(font)
        self.combo_layout.addWidget(self.name_label)

        self.custom_column_target_library_combobox = QComboBox()
        self.custom_column_target_library_combobox.setEditable(False)
        self.custom_column_target_library_combobox.setFont(font)
        self.custom_column_target_library_combobox.setMaximumWidth(200)
        self.custom_column_target_library_combobox.setToolTip("<p style='white-space:wrap'>This is the Custom Column of type 'comments' and of the 'plain text' or 'short text' variety that will have the Target Library (either via its OS Path or just its Library Name) for each book to be copied or moved from the Current Library.")
        self.combo_layout.addWidget(self.custom_column_target_library_combobox)

        tmp_list = self.guidb.custom_field_keys(include_composites=False)
        custom_field_keys = []
        for label in tmp_list:
            cc_metadata_dict = self.guidb.custom_field_metadata(label)
            for k,v in iteritems(cc_metadata_dict):
                if k == label:
                    cc_datatype = v['datatype']
                    if cc_datatype == "comments":
                        cc_display_dict = v['display']
                        if 'interpret_as' in cc_display_dict:
                            if cc_display_dict['interpret_as'] == "short-text" or cc_display_dict['interpret_as'] == "long-text":
                                custom_field_keys.append(label)
                                break
            #END FOR
        #END FOR

        del tmp_list

        try:
            for label in custom_field_keys:
                self.custom_column_target_library_combobox.addItem(label)
            #END FOR
        except Exception as e:
            if DEBUG: print(as_unicode(e))

        cc = prefs['LS_TARGET_LIBRARY_TEXT_CUSTOM_COLUMN']
        if cc in custom_field_keys:
            i = self.custom_column_target_library_combobox.findText(cc,Qt.MatchExactly)
            self.custom_column_target_library_combobox.setCurrentIndex(i)
        #-----------------------------------------------------
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.button_box = QDialogButtonBox()
        self.button_box.setOrientation(Qt.Horizontal)
        self.button_box.setCenterButtons(True)

        self.layout_frame.addWidget(self.button_box)
        #-----------------------------------------------------
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.push_button_save_only = QPushButton("Save Settings")
        self.push_button_save_only.clicked.connect(self.save_settings)
        self.push_button_save_only.setDefault(True)
        self.push_button_save_only.setFont(font)
        self.push_button_save_only.setToolTip("<p style='white-space:wrap'>Save all user settings.")
        self.button_box.addButton(self.push_button_save_only,QDialogButtonBox.AcceptRole)
        #-----------------------------------------------------
        self.push_button_copy_only = QPushButton("Execute: Copy [Selected Books]")
        self.push_button_copy_only.clicked.connect(self.execute_copy)
        self.push_button_copy_only.setDefault(False)
        self.push_button_copy_only.setFont(font)
        self.push_button_copy_only.setToolTip("<p style='white-space:wrap'>'Copy' only.  Copy selected books to the appropriate target library.")
        self.button_box.addButton(self.push_button_copy_only,QDialogButtonBox.AcceptRole)
        #-----------------------------------------------------
        self.push_button_copy_then_delete = QPushButton("Execute: Move [Selected Books]")
        self.push_button_copy_then_delete.clicked.connect(self.execute_move)
        self.push_button_copy_then_delete.setDefault(False)
        self.push_button_copy_then_delete.setFont(font)
        self.push_button_copy_then_delete.setToolTip("<p style='white-space:wrap'>'Move' (Copy then Delete).  Copy selected books to the appropriate target library, and then delete the selected books from the current library.")
        self.button_box.addButton(self.push_button_copy_then_delete,QDialogButtonBox.AcceptRole)
        #-----------------------------------------------------
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.scroll_widget.resize(self.sizeHint())
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.scroll_area_frame.setWidget(self.scroll_widget)    # now that all widgets have been created and assigned to a layout...
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.scroll_area_frame.resize(self.sizeHint())
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.resize(self.sizeHint())

        if len(custom_field_keys) == 0:
            self.invalid_target_custom_column = True
            error_dialog(self.gui, _('Library Splitter'),_(("No Custom Columns found that are of the correct type and variety.  Refer to the ToolTips for more information.")), show=True)
        else:
            self.invalid_target_custom_column = False

        del custom_field_keys
    #-----------------------------------------------------------------------------------------
    def validate(self):
        return True
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def exit_only(self):
        self.save_dialog_geometry()
        self.exit_all
    #-----------------------------------------------------------------------------------------
    def save_settings(self):
        self.save_dialog_geometry()
        #---------------------------------------
        self.mytabprefs["LS_TARGET_LIBRARY_TEXT_CUSTOM_COLUMN"] = self.custom_column_target_library_combobox.currentText()
        #---------------------------------------
        for k,v in iteritems(self.mytabprefs):
            v = as_unicode(v)
            v = v.strip()
            prefs[k] = v
        #END FOR
        prefs

        cc = self.custom_column_target_library_combobox.currentText()
        if not cc.startswith("#"):
            self.invalid_target_custom_column = True
        else:
            self.invalid_target_custom_column = False
    #-----------------------------------------------------------------------------------------
    def execute_copy(self):

        self.save_settings()

        if self.invalid_target_custom_column:
            error_dialog(self.gui, _('Library Splitter'),_(("No Custom Columns found that are of the correct type and variety.  Refer to the ToolTips for more information.")), show=True)
            return

        if question_dialog(self.gui, "Library Splitter", "Copy the selected books, including their Custom Column metadata, to the specified Target Library?  Are you sure?"):
            pass
        else:
            return

        self.execution_type_move = False

        self.execution_control()
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def execute_move(self):

        self.save_settings()

        if self.invalid_target_custom_column:
            error_dialog(self.gui, _('Library Splitter'),_(("No Custom Columns found that are of the correct type and variety.  Refer to the ToolTips for more information.")), show=True)
            return

        if question_dialog(self.gui, "Library Splitter", "Move the selected books, including their Custom Column metadata, to the specified Target Library?  Are you sure?"):
            pass
        else:
            return

        self.execution_type_move = True

        self.execution_control()
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def execution_control(self):

        self.selected_books_list = self.get_selected_books()

        n_books = len(self.selected_books_list)
        if n_books == 0:
             return error_dialog(self.gui, _('Library Splitter'),_('No Books Were Selected.'), show=True)

        self.selected_books_list.sort()

        self.current_library = os.path.normcase(os.path.abspath(self.guidb.library_path))
        self.current_library = self.current_library.replace(os.sep, '/')
        self.current_library = self.current_library.strip()

        self.locations_list = []   #library name only
        tmp_list = []
        self.target_library_full_path_dict = {}  #[full path] = library name    library name is Not unique
        self.target_library_full_path_reverse_dict = {}  #[library name] = full path

        stats = gprefs.get('library_usage_stats', {})
        for k,v in iteritems(stats):
            p = os.path.normcase(os.path.abspath(k))
            p = p.replace(os.sep, '/')
            p = p.strip()
            if not self.current_library == p:  #  target library names will be invalid if they are not in the list created here, implicitly disallowing copying to the self.current_library library
                row = v,p
                tmp_list.append(row)
                self.target_library_full_path_dict[p] = v
        #END FOR

        tmp_list.sort(reverse=True)  # sort descending by number of times library used by Calibre GUI

        for row in tmp_list:
            v,k = row
            k = os.path.normcase(os.path.abspath(k))
            k = k.replace(os.sep, '/')
            k = k.strip()
            if not k in self.locations_list:
                self.locations_list.append(k)
            head,tail = os.path.split(k)
            tail = tail.lower().strip()
            if not tail in self.target_library_full_path_reverse_dict:  # library name is not unique, so use only most-used per stats (sorted descending)
                self.target_library_full_path_reverse_dict[tail] = k
        #END FOR

        del stats
        del tmp_list

        self.calibre_copy_to_action_control()
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def calibre_copy_to_action_control(self):

        target_library_name_cc = prefs['LS_TARGET_LIBRARY_TEXT_CUSTOM_COLUMN']
        target_library_name_cc = as_unicode(target_library_name_cc)

        custom_columns_metadata_dict = self.gui.current_db.field_metadata.custom_field_metadata()

        book_target_library_path_dict = {}
        target_library_path_set = set()

        for book in self.selected_books_list:
                book_mi_object = self.guidb.get_metadata(book, index_is_id=True, get_cover=False, get_user_categories=True, cover_as_data=False)    # a :class:`Metadata` object.
                target_library_path = book_mi_object.get(target_library_name_cc)
                #~ if DEBUG: print("book for book_mi_object", as_unicode(book), "   original target_library_path: ", as_unicode(target_library_path))
                if target_library_path:
                    target_library_path = target_library_path.replace('"','')
                    target_library_path = target_library_path.replace(os.sep, '/')
                    if "/" in target_library_path:
                        target_library_path = os.path.normcase(os.path.abspath(target_library_path))
                        target_library_path = target_library_path.replace(os.sep, '/')
                    else:
                        target_library_path = target_library_path.lower()
                    target_library_path = target_library_path.strip()
                    if self.current_library == target_library_path:  # cannot copy to itself
                        continue
                    if target_library_path in self.target_library_full_path_dict:   # user specified full paths in the target library name custom column
                        book_target_library_path_dict[book] = target_library_path  # full path
                        target_library_path_set.add(target_library_path)
                        if DEBUG: print("user specified full qpath:  good selected book: ", as_unicode(book), " cc target_library_path", target_library_path)
                    else:  # user specified simple library name in the target library name custom column, and not the full path
                        found = False
                        for libname,path in iteritems(self.target_library_full_path_reverse_dict):  #self.target_library_full_path_reverse_dict[tail] = path
                            if libname == target_library_path:
                                book_target_library_path_dict[book] = path  # full path
                                target_library_path = path
                                target_library_path_set.add(target_library_path)
                                found = True
                                if DEBUG: print("user specified simple library name: good selected book: ", as_unicode(book), " cc target_library_path", target_library_path)
                                break
                        #END FOR
                else:
                    pass
        #END FOR

        current_books_list = []
        for possiblepath in self.locations_list:
            del current_books_list
            current_books_list = []
            if not possiblepath in target_library_path_set:
                continue
            for book,targetpath in iteritems(book_target_library_path_dict):
                if possiblepath == targetpath:
                    book = int(book)
                    current_books_list.append(book)  #final check that book has a known calibre library as its target...
            #END FOR
            if len(current_books_list) > 0:
                current_books_set = set(current_books_list)
                self.gui.library_view.clearSelection()
                self.gui.library_view.select_rows(current_books_set, using_ids=True, change_current=False, scroll=False)
                self.copytolibraryaction_object.copy_to_library(possiblepath, self.execution_type_move)
        #END FOR

        del book_target_library_path_dict
        del target_library_path_set
        del self.locations_list
        del self.selected_books_list
        del custom_columns_metadata_dict
        del self.target_library_full_path_dict
        del self.target_library_full_path_reverse_dict
        del self.current_library
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def get_selected_books(self):

        try:
            del self.selected_books_list
        except:
            pass

        self.selected_books_list = []

        book_ids_list = []

        book_ids_list = list(map(partial(self.convert_id_to_book), self.gui.library_view.get_selected_ids()))  #https://stackoverflow.com/questions/50671360/map-in-python-3-vs-python-2
        if not isinstance(book_ids_list,list):
            if DEBUG: print("error: polyglot map failed to create a valid selected book_ids_list")
            error_dialog(self.gui, _('Library Splitter'),_(("Python 2 to Python 3 compatibility error for polyglot 'map' in 'def get_selected_books'.  Inform plug-in author.")), show=True)
            return self.selected_books_list

        self.number_selected_books = len(book_ids_list)
        if self.number_selected_books == 0:
            del book_ids_list
            return self.selected_books_list

        for item in book_ids_list:
            s = item['calibre_id']
            s = int(s)
            self.selected_books_list.append(s)
        #END FOR

        del book_ids_list

        return self.selected_books_list
    #-----------------------------------------------------------------------------------------
    def convert_id_to_book(self, idval):
        book = {}
        book['calibre_id'] = idval
        return book
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
#END of library_splitter_dialog.py