# -*- coding: utf-8 -*-
__license__   = 'GPL v3'
__copyright__ = '2015,2016,2017,2018,2019,2020,2021,2022,2023 DaltonST'
__my_version__ = "2.0.45"  # QTableView Sorting; Improve DEBUG for enumeration

from qt.core import (Qt, QDialog, QTableWidgetItem, QIcon, QByteArray, QSize)

from calibre.gui2 import question_dialog, error_dialog, info_dialog, gprefs
from calibre.utils.icu import sort_key
from calibre_plugins.consolidate_all_library_metadata.tag_rules_list_editor_ui_2 import Ui_TagRulesListEditor2

from polyglot.builtins import as_unicode, iteritems, map, range, unicode_type

class NameTableWidgetItem(QTableWidgetItem):

    def __init__(self, txt):
        QTableWidgetItem.__init__(self, txt)
        self.initial_value = txt
        self.current_value = txt
        self.previous_value = txt

    def data(self, role):
        if role == Qt.DisplayRole:
            return self.current_value
        elif role == Qt.EditRole:
            return self.current_value
        else:
            return QTableWidgetItem.data(self, role)

    def setData(self, role, data):
        if role == Qt.EditRole:
            self.current_value = self.initial_value
        QTableWidgetItem.setData(self, role, data)

    def text(self):
        return self.initial_value

    def initial_text(self):
        return self.initial_value

    def previous_text(self):
        return self.initial_value

    def setText(self, txt):
        QTableWidgetItem.setText(self.initial_value)

    def __ge__(self, other):
        return sort_key(as_unicode(self.text())) >= sort_key(as_unicode(other.text()))

    def __lt__(self, other):
        return sort_key(as_unicode(self.text())) < sort_key(as_unicode(other.text()))


class SubjectTableWidgetItem(QTableWidgetItem):

    def __init__(self, txt):
        QTableWidgetItem.__init__(self, txt)
        self.initial_value = txt
        self.current_value = txt
        self.previous_value = txt

    def data(self, role):
        if role == Qt.DisplayRole:
            return self.current_value
        elif role == Qt.EditRole:
            return self.current_value
        else:
            return QTableWidgetItem.data(self, role)

    def setData(self, role, data):
        if role == Qt.EditRole:
            self.previous_value = self.current_value
            self.current_value = data
        QTableWidgetItem.setData(self, role, data)

    def text(self):
        return self.current_value

    def initial_text(self):
        return self.initial_value

    def previous_text(self):
        return self.previous_value

    def setText(self, txt):
        self.current_value = txt
        QTableWidgetItem.setText(txt)

    def __ge__(self, other):
        return sort_key(as_unicode(self.text())) >= sort_key(as_unicode(other.text()))

    def __lt__(self, other):
        return sort_key(as_unicode(self.text())) < sort_key(as_unicode(other.text()))

class RegexTableWidgetItem(QTableWidgetItem):

    def __init__(self, txt):
        QTableWidgetItem.__init__(self, txt)
        self.initial_value = txt
        self.current_value = txt
        self.previous_value = txt

    def data(self, role):
        if role == Qt.DisplayRole:
            return self.current_value
        elif role == Qt.EditRole:
            return self.current_value
        else:
            return QTableWidgetItem.data(self, role)

    def setData(self, role, data):
        if role == Qt.EditRole:
            self.previous_value = self.current_value
            self.current_value = data
        QTableWidgetItem.setData(self, role, data)

    def text(self):
        return self.current_value

    def initial_text(self):
        return self.initial_value

    def previous_text(self):
        return self.previous_value

    def setText(self, txt):
        self.current_value = txt
        QTableWidgetItem.setText(txt)

    def __ge__(self, other):
        return sort_key(as_unicode(self.text())) >= sort_key(as_unicode(other.text()))

    def __lt__(self, other):
        return sort_key(as_unicode(self.text())) < sort_key(as_unicode(other.text()))


class TagRulesListEditor2(QDialog, Ui_TagRulesListEditor2):


    def __init__(self, window, cat_name, tag_to_match, data, sorter, _actually_insert_tag_rules_rows_2, guidb):
        QDialog.__init__(self, window)
        Ui_TagRulesListEditor2.__init__(self)
        self.setupUi(self)

        self._actually_insert_tag_rules_rows_2 = _actually_insert_tag_rules_rows_2
        self.guidb = guidb

        # Put the category name into the title bar
        t = self.windowTitle()
        self.setWindowTitle(t + ' (' + cat_name + ')')
        # Remove help icon on title bar
        icon = self.windowIcon()
        self.setWindowFlags(self.windowFlags()&(~Qt.WindowContextHelpButtonHint))
        self.setWindowIcon(icon)

        # Get saved geometry info
        try:
            self.table_column_widths = gprefs.get('calm_tag_rules_list_editor_2_table_widths', None)
        except:
            pass

        # initialization
        self.to_add_to_tag_rules_table = []
        self.all_tags = {}
        self.subjects = {}
        self.regexes = {}

        for row in data:
            s1, s2 = row
            self.all_tags[s1] = s1
            #~ self.subjects[s1] = s2        #  no BISAC subject codes in CALM
            self.subjects[s1] = ''
            self.regexes[s1] = ''

        # Set up the column headings
        self.down_arrow_icon = QIcon(I('arrow-down.png'))
        self.up_arrow_icon = QIcon(I('arrow-up.png'))
        self.blank_icon = QIcon(I('blank.png'))

        self.table.setColumnCount(3)

        self.name_col = QTableWidgetItem(_('Target Tags Missing Tag Rules [table: _tag_rules ]'))
        self.table.setHorizontalHeaderItem(0, self.name_col)
        self.name_col.setIcon(self.up_arrow_icon)

        self.subject_col = QTableWidgetItem(_('Newtag'))
        self.table.setHorizontalHeaderItem(1, self.subject_col)
        self.subject_col.setIcon(self.blank_icon)

        self.regex_col = QTableWidgetItem(_('Make /REGEX/? [1 = True, Blank or 0 = False]'))
        self.table.setHorizontalHeaderItem(2, self.regex_col)
        self.regex_col.setIcon(self.blank_icon)

        # Capture clicks on the horizontal header to sort the table columns
        hh = self.table.horizontalHeader()
        hh.setSectionsClickable(True)
        hh.sectionClicked.connect(self.header_clicked)
        hh.sectionResized.connect(self.table_column_resized)
        self.name_order = 0

        # Add the tags gotten in ui.py
        select_item = None
        self.table.setRowCount(len(self.all_tags))
        for row,tag in enumerate(sorted(self.all_tags.keys(), key=sorter)):
            item = NameTableWidgetItem(tag)
            item.setFlags(item.flags() | Qt.ItemIsSelectable | Qt.ItemIsEditable)
            self.table.setItem(row, 0, item)
            subject = self.subjects[tag]
            item = SubjectTableWidgetItem(subject)
            item.setFlags(item.flags() | Qt.ItemIsSelectable | Qt.ItemIsEditable)
            self.table.setItem(row, 1, item)
            regex = self.regexes[tag]
            item = RegexTableWidgetItem(regex)
            item.setFlags(item.flags() | Qt.ItemIsSelectable | Qt.ItemIsEditable)
            self.table.setItem(row, 2, item)


        # Scroll to the selected item if there is one
        if select_item is not None:
            self.table.setCurrentItem(select_item)

        self.delete_button.clicked.connect(self.add_tags_to_insert_list)
        self.buttonBox.accepted.connect(self.accepted)

        try:
            geom = gprefs.get('calm_tag_rules_list_editor_2_dialog_geometry', None)
            if geom is not None:
                self.restoreGeometry(QByteArray(geom))
            else:
                self.resize(self.sizeHint()+QSize(150, 100))
        except:
            pass

    def table_column_resized(self, col, old, new):
        self.table_column_widths = []
        for c in range(0, self.table.columnCount()):
            self.table_column_widths.append(self.table.columnWidth(c))

    def resizeEvent(self, *args):
        QDialog.resizeEvent(self, *args)
        if self.table_column_widths is not None:
            for c,w in enumerate(self.table_column_widths):
                self.table.setColumnWidth(c, w)
        else:
            # the vertical scroll bar might not be rendered, so might not yet
            # have a width. Assume 25. Not a problem because user-changed column
            # widths will be remembered
            w = self.table.width() - 25 - self.table.verticalHeader().width()
            w /= self.table.columnCount()
            for c in range(0, self.table.columnCount()):
                self.table.setColumnWidth(c, w)

    def save_geometry(self):
        gprefs['calm_tag_rules_list_editor_2_table_widths'] = self.table_column_widths
        gprefs['calm_tag_rules_list_editor_2_dialog_geometry'] = bytearray(self.saveGeometry())

    def accepted(self):
        self.save_geometry()
        self.add_tags_to_table_tag_rules()

    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def add_tags_to_table_tag_rules(self):
        self._actually_insert_tag_rules_rows_2(self.guidb,self.to_add_to_tag_rules_table)
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------

    def add_tags_to_insert_list(self):

        mappings = self.table.selectedItems()
        if not mappings:
            error_dialog(self, _('No items selected'),
                         _('You must select at least one item from the list.')).exec_()
            return
        ct = ', '.join([as_unicode(item.text()) for item in mappings])
        if not question_dialog(self, _('Are you sure?'),
            '<p>'+_('Are you sure you want to Easy-Add the following Mappings to the CALM Tag Rules Table for Future Automatic Renaming?')+'<br><br>'+ct):
            return

        i = 0
        for item in mappings:
            if i == 0:
                tag = item.text()     #each column in each row is a separate row in this list.  so, Tag is row 1, NEWTAG for that Tag is row 2, REGEX for that Tag is row 3, next Tag is row 4...
                i = i + 1
            else:
                if i == 1:
                    subject = item.text()
                    i = i + 1
                else:
                    regex = item.text()
                    s  = '|||'.join([(as_unicode(tag)), (as_unicode(subject)), (as_unicode(regex))])
                    self.to_add_to_tag_rules_table.append(s)
                    i = 0
        #END FOR

    def header_clicked(self, idx):
        self.table.setSortingEnabled(True)
        self.do_sort_by_name()

    def do_sort_by_name(self):
        if self.name_order == 0:
            self.name_order = 1
            so = Qt.DescendingOrder
        else:
            self.name_order = 0
            so = Qt.AscendingOrder
        #~ self.name_order = 1 if self.name_order == 0 else 0  #Qt6 change
        self.table.sortByColumn(0, so)
        self.name_col.setIcon(self.down_arrow_icon if self.name_order else self.up_arrow_icon)
    #--------------------------------------------------------------------
    #--------------------------------------------------------------------
    #END
