# -*- coding: utf-8 -*-
__license__   = 'GPL v3'
__copyright__ = '2019,2020,2021,2022,2023 DaltonST'
__my_version__ = "1.0.19"  # Qt.core

from qt.core import Qt, QMenu, QIcon, QAction, QSize, QUrl, QInputDialog, QFont,QFileDialog,QColorDialog,QColor

import os,sys,apsw,ast,zipfile
from functools import partial

from calibre import isbytestring
from calibre.constants import filesystem_encoding, DEBUG, config_dir
from calibre.gui2 import error_dialog, question_dialog, info_dialog
from calibre.gui2.actions import InterfaceAction
from calibre.gui2.preferences.coloring import RulesModel
EXIM_VERSION = RulesModel.EXIM_VERSION
del RulesModel

from calibre.utils.serialize import json_dumps, json_loads   #cannot use simple json.dumps, json.loads...

from polyglot.builtins import as_unicode, range, unicode_type

from calibre_plugins.entities_manager.common_utils import set_plugin_icon_resources, get_icon, create_menu_action_unique

PLUGIN_ICONS = ['images/entities_manager.png','images/add_link.png','images/add_note.png','images/change.png',
'images/clear_input.png','images/close.png','images/delete.png','images/information.png','images/invert.png','images/matrix.png','images/ok.png',
'images/refresh.png','images/reload.png','images/create.png','images/select_matrix_item.png','images/open.png',
'images/association.png','images/delete_association.png','images/delete_all_associations.png','images/link.png','images/note.png',
'images/copy_link.png','images/copy_note.png','images/entity.png','images/element.png', 'images/list_entities.png', 'images/utilities.png',
'images/add.png','images/user_type.png','images/colors.png','images/icons.png','images/binoculars.png']

ASSOCIATION_ENTITY_TYPE = "~Association"

MODE_ADD = "add"
MODE_REMOVE = "remove"

#--------------------------------------------------------------------------------------------
class ActionEntitiesManager(InterfaceAction):

    name = 'Entities Manager'
    action_spec = ('EM','images/entities_manager.png', 'Associate Calibre Entities to Links, Notes, and other information independent of any particular book.', None)
    action_type = 'global'
    accepts_drops = False
    auto_repeat = False
    priority = 9
    popup_type = 1

    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def genesis(self):
        icon_resources = self.load_resources(PLUGIN_ICONS)
        set_plugin_icon_resources(self.name, icon_resources)
        self.build_menu()
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def build_menu(self):
        self.menu = QMenu(self.gui)
        self.menu.setTearOffEnabled(True)
        self.menu.setWindowTitle('Entities Manager')
        self.menu.clear()
        #------------------------------------------
        self.menu.addSeparator()
        unique_name = "Manage Entities, Links, Notes && Associations"
        create_menu_action_unique(self, self.menu, unique_name, 'images/entities_manager.png',
                              triggered=partial(self.manage_entities),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.menu.addSeparator()
        unique_name = "    "
        create_menu_action_unique(self, self.menu, unique_name, '',
                              triggered=partial(self.dummy),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.menu.addSeparator()
        unique_name = "List Entities, Links, Notes && Associations"
        create_menu_action_unique(self, self.menu, unique_name, 'images/list_entities.png',
                              triggered=partial(self. list_all_entities_all_info),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.menu.addSeparator()
        unique_name = "Display Links, Notes && Associations for Entity at Cursor"
        create_menu_action_unique(self, self.menu, unique_name, 'images/entity.png',
                              triggered=partial(self.display_entity_info),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.menu.addSeparator()
        unique_name = "   "
        create_menu_action_unique(self, self.menu, unique_name, '',
                              triggered=partial(self.dummy),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.menu.addSeparator()
        unique_name = "Quick Execute Link for Entity at Cursor"
        create_menu_action_unique(self, self.menu, unique_name, 'images/link.png',
                              triggered=partial(self.quick_execute_link),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.menu.addSeparator()
        unique_name = "Quick Display Note for Entity at Cursor"
        create_menu_action_unique(self, self.menu, unique_name, 'images/note.png',
                              triggered=partial(self.quick_display_note),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.menu.addSeparator()
        unique_name = "  "
        create_menu_action_unique(self, self.menu, unique_name, '',
                              triggered=partial(self.dummy),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.menu.addSeparator()
        unique_name = "Quick Create Entity from Cursor Type && Value"
        create_menu_action_unique(self, self.menu, unique_name, 'images/create.png',
                              triggered=partial(self.quick_create_entity),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.menu.addSeparator()
        unique_name = "Quick Add Link to Entity at Cursor"
        create_menu_action_unique(self, self.menu, unique_name, 'images/add_link.png',
                              triggered=partial(self.quick_add_link),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.menu.addSeparator()
        unique_name = "Quick Add Note to Entity at Cursor"
        create_menu_action_unique(self, self.menu, unique_name, 'images/add_note.png',
                              triggered=partial(self.quick_add_note),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.menu.addSeparator()
        unique_name = " "
        create_menu_action_unique(self, self.menu, unique_name, '',
                              triggered=partial(self.dummy),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.menu.addSeparator()
        #------------------------------------------
        self.user_types_menu = QMenu('[Menu] User Entity Types')

        self.user_types_action = self.menu.addMenu(self.user_types_menu)
        self.user_types_menu.setIcon(get_icon('images/user_type.png'))

        self.user_types_menu.setTearOffEnabled(True)
        self.user_types_menu.setWindowTitle('EM User Entity Types')
        self.user_types_menu.clear()

        self.user_types_menu.addSeparator()
        unique_name = "Add User 'Entity Type'"
        create_menu_action_unique(self, self.user_types_menu, unique_name, 'images/add.png',
                              triggered=partial(self.user_type_add),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.user_types_menu.addSeparator()
        unique_name = "Change User 'Entity Type'"
        create_menu_action_unique(self, self.user_types_menu, unique_name, 'images/change.png',
                              triggered=partial(self.user_type_change),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.user_types_menu.addSeparator()
        unique_name = "Delete User 'Entity Type'"
        create_menu_action_unique(self, self.user_types_menu, unique_name, 'images/delete.png',
                              triggered=partial(self.user_type_delete),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.user_types_menu.addSeparator()

        t = "<p style='white-space:wrap'>Maintain 'User Entity Types'.  They will <i>automatically</i> be prefixed with a '~' symbol to differentiate them from Standard and Custom Columns.  "
        t = t + "<br><br>Reserved UET '~Association' may not be changed or deleted.  '~Association' is used by EM to create Association-Type Entities having Values of the Association."
        self.user_types_menu.setToolTip(t)
        #------------------------------------------
        self.menu.addMenu(self.user_types_menu)
        self.menu.addSeparator()
        unique_name = " "
        create_menu_action_unique(self, self.menu, unique_name, '',
                              triggered=partial(self.dummy),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.menu.addSeparator()
        #------------------------------------------
        self.icons_menu = QMenu('[Menu] Manage Entity Column Icon Rules')

        self.icons_action = self.menu.addMenu(self.icons_menu)
        self.icons_menu.setIcon(get_icon('images/icons.png'))

        self.icons_menu.setTearOffEnabled(True)
        self.icons_menu.setWindowTitle("EM Manage Entity Column Icon Rules")
        self.icons_menu.clear()

        unique_name = "Select Default Reserved Icon for 'Column Icon Rules' for EM"
        create_menu_action_unique(self, self.icons_menu, unique_name, 'images/icons.png',
                              triggered=partial(self.select_default_reserved_icon_column_icon_rules_em),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.icons_menu.addSeparator()
        unique_name = "Select Default Reserved Icon for 'Column Icon Rules' by Entity Type"
        create_menu_action_unique(self, self.icons_menu, unique_name, 'images/icons.png',
                              triggered=partial(self.select_default_reserved_icon_column_icon_rules_entity_type),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.icons_menu.addSeparator()
        unique_name = "Select Default Reserved Icon for 'Column Icon Rules' by Entity (Type + Value)"
        create_menu_action_unique(self, self.icons_menu, unique_name, 'images/icons.png',
                              triggered=partial(self.select_default_reserved_icon_column_icon_rules_entity),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.icons_menu.addSeparator()
        unique_name = "                         "
        create_menu_action_unique(self, self.icons_menu, unique_name, '',
                              triggered=partial(self.dummy),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.icons_menu.addSeparator()
        unique_name = "Activate 'Column Icon Rules' by Entity Type"
        create_menu_action_unique(self, self.icons_menu, unique_name, 'images/icons.png',
                              triggered=partial(self.activate_column_icon_rules_entity_type),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.icons_menu.addSeparator()
        unique_name = "Specify 'Icon Only if Link or Note' by Entity Type"
        create_menu_action_unique(self, self.icons_menu, unique_name, 'images/icons.png',
                              triggered=partial(self.restrict_link_note_column_icon_rules_entity_type),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.icons_menu.addSeparator()
        unique_name = "                         "
        create_menu_action_unique(self, self.icons_menu, unique_name, '',
                              triggered=partial(self.dummy),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.icons_menu.addSeparator()
        unique_name = "                         "
        create_menu_action_unique(self, self.icons_menu, unique_name, '',
                              triggered=partial(self.dummy),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.icons_menu.addSeparator()
        unique_name = "Add/Refresh Column 'Icon with Text' Rules for Entities"
        create_menu_action_unique(self, self.icons_menu, unique_name, 'images/icons.png',
                              triggered=partial(self.add_column_icon_with_text_rules),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.icons_menu.addSeparator()
        unique_name = "                         "
        create_menu_action_unique(self, self.icons_menu, unique_name, '',
                              triggered=partial(self.dummy),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.icons_menu.addSeparator()
        unique_name = "Remove Column 'Icon with Text' Rules for Entities"
        create_menu_action_unique(self, self.icons_menu, unique_name, 'images/icons.png',
                              triggered=partial(self.remove_column_icon_with_text_rules),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.icons_menu.addSeparator()
        unique_name = "                         "
        create_menu_action_unique(self, self.icons_menu, unique_name, '',
                              triggered=partial(self.dummy),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.menu.addMenu(self.icons_menu)
        self.icons_menu.addSeparator()
        unique_name = "Activate/Deactivate Icon Rules Automatic Backup"
        create_menu_action_unique(self, self.icons_menu, unique_name, 'images/icons.png',
                              triggered=partial(self.activate_deactivate_rules_backup_icons),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.icons_menu.addSeparator()

        t = "<p style='white-space:wrap'>Refer to Preferences > Look & Feel > Column Icons.  Icon rules of type 'icon with text' will automatically be created for all existing Entities. "
        t = t + "<br><br>Before making any changes to any rules, all existing 'icon with text' rules (user and automatically created) will be backed up to an Export .rules file.   The path of each file \
        is prefixed with the current Library's name, and includes the date and time in the file name.  Periodically, you may wish to purge files that you no longer need to retain."
        t = t + "<br><br>The icon used is named 'entity_reserved' and is located in the Calibre configuration directory 'calibre/config/cc_icons/.  Do not use the icon for manual rules, \
        or your manual rules will be automatically deleted the next time that these menu options execute."
        t = t + '<br><br>Entity Values with a double-quote (") cannot have icon rules created for them.' + "  Use single-quotes (') instead of double-quotes in Entity Values if icons are desired."
        t = t + "<br><br>If you wish to view <i>only</i> user-created icon rules, and not EM-created icon rules, in Look & Feel, simply first 'Remove' all of the Entity icon rules.  When \
        finished, 'Add' them back.  "
        t = t + "<br><br>It is advisable to periodically use Calibre's Library Maintenance > Check Library to automatically vacuum/defragment/compress the Calibre database.  This is \
        true whether you use icons or not; it is a 'best practice' for Calibre."
        t = t + "<br><br>Important: If you add new Custom Columns to this Library, \
        you must click 'Manage Entities' at least once so it can automatically peform necessary housekeeping to activate the new Custom Column for use in Entity Column Icons. "
        self.icons_menu.setToolTip(t)
        #------------------------------------------
        self.menu.addSeparator()
        self.color_menu = QMenu('[Menu] Manage Entity Column Color Rules')

        self.color_action = self.menu.addMenu(self.color_menu)
        self.color_menu.setIcon(get_icon('images/colors.png'))

        self.color_menu.setTearOffEnabled(True)
        self.color_menu.setWindowTitle('EM Manage Entity Column Color Rules')
        self.color_menu.clear()

        unique_name = "Add New Unique Color to EM Reserved Color List"
        create_menu_action_unique(self, self.color_menu, unique_name, 'images/colors.png',
                              triggered=partial(self.add_new_unique_color_to_reserved_color_list),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.color_menu.addSeparator()
        unique_name = "Select Default Unique Color for 'Column Color Rules' for EM"
        create_menu_action_unique(self, self.color_menu, unique_name, 'images/colors.png',
                              triggered=partial(self.select_default_unique_color_column_color_rules_em),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.color_menu.addSeparator()
        unique_name = "Select Default Unique Color for 'Column Color Rules' by Entity Type"
        create_menu_action_unique(self, self.color_menu, unique_name, 'images/colors.png',
                              triggered=partial(self.select_default_unique_color_column_color_rules_entity_type),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.color_menu.addSeparator()
        unique_name = "Select Default Unique Color for 'Column Color Rules' by Entity (Type + Value)"
        create_menu_action_unique(self, self.color_menu, unique_name, 'images/colors.png',
                              triggered=partial(self.select_default_unique_color_column_color_rules_entity),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.color_menu.addSeparator()
        unique_name = "                         "
        create_menu_action_unique(self, self.color_menu, unique_name, '',
                              triggered=partial(self.dummy),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.color_menu.addSeparator()
        unique_name = "Activate 'Column Color Rules' by Entity Type"
        create_menu_action_unique(self, self.color_menu, unique_name, 'images/colors.png',
                              triggered=partial(self.activate_column_color_rules_entity_type),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.color_menu.addSeparator()
        unique_name = "Specify 'Color Only if Link or Note' by Entity Type"
        create_menu_action_unique(self, self.color_menu, unique_name, 'images/colors.png',
                              triggered=partial(self.restrict_link_note_column_color_rules_entity_type),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.color_menu.addSeparator()
        unique_name = "                         "
        create_menu_action_unique(self, self.color_menu, unique_name, '',
                              triggered=partial(self.dummy),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.color_menu.addSeparator()
        unique_name = "                         "
        create_menu_action_unique(self, self.color_menu, unique_name, '',
                              triggered=partial(self.dummy),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.color_menu.addSeparator()
        unique_name = "Add/Refresh Column Color Rules for Entities"
        create_menu_action_unique(self, self.color_menu, unique_name, 'images/colors.png',
                              triggered=partial(self.add_column_color_rules),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.color_menu.addSeparator()
        unique_name = "                         "
        create_menu_action_unique(self, self.color_menu, unique_name, '',
                              triggered=partial(self.dummy),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.color_menu.addSeparator()
        unique_name = "Remove Column Color Rules for Entities"
        create_menu_action_unique(self, self.color_menu, unique_name, 'images/colors.png',
                              triggered=partial(self.remove_column_color_rules),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.color_menu.addSeparator()
        unique_name = "                         "
        create_menu_action_unique(self, self.color_menu, unique_name, '',
                              triggered=partial(self.dummy),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.menu.addMenu(self.color_menu)
        self.color_menu.addSeparator()
        unique_name = "Activate/Deactivate Color Rules Automatic Backup"
        create_menu_action_unique(self, self.color_menu, unique_name, 'images/colors.png',
                              triggered=partial(self.activate_deactivate_rules_backup_colors),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.color_menu.addSeparator()

        t = "<p style='white-space:wrap'>Refer to Preferences > Look & Feel > Column Colors.  Color rules will automatically be created for all existing Entities if their Types have been so activated here. "
        t = t + "<br><br>Before making any changes to any rules, all existing color rules (user and automatically created) will be backed up to an Export .rules file.   The path of each file \
        is prefixed with the current Library's name, and includes the date and time in the file name.  Periodically, you may wish to purge files that you no longer need to retain."
        t = t + "<br><br>The HTML color codes used are set by you.  Do not use the identical HTML color codes for manual rules, \
        or your manual rules will be automatically deleted the next time that these menu options execute.  Refer to Color Code customization."
        t = t + '<br><br>Entity Values with a double-quote (") cannot have color rules created for them.' + "  Use single-quotes (') instead of double-quotes in Entity Values if colors are desired."
        t = t + "<br><br>If you wish to view <i>only</i> user-created color rules, and not EM-created color rules, in Look & Feel, simply first 'Remove' all of the Entity color rules.  When \
        finished reviewing the manual rules, 'Add' the EM rules back.  "
        t = t + "<br><br>It is advisable to periodically use Calibre's Library Maintenance > Check Library to automatically vacuum/defragment/compress the Calibre database.  This is \
        true whether you use icons or not; it is a 'best practice' for Calibre."
        t = t + "<br><br>Important: If you add new Custom Columns to this Library, \
        you must click 'Manage Entities' at least once so it can automatically peform necessary housekeeping to activate any new Custom Columns for use in Entity Column Colors. "
        self.color_menu.setToolTip(t)
        #------------------------------------------
        self.menu.addSeparator()
        unique_name = " "
        create_menu_action_unique(self, self.menu, unique_name, '',
                              triggered=partial(self.dummy),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.menu.addSeparator()
        #------------------------------------------
        unique_name = "Find All Books Having Entities"
        create_menu_action_unique(self, self.menu, unique_name, 'images/binoculars.png',
                              triggered=partial(self.find_all_books_having_entities),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.menu.addSeparator()
        unique_name = "Display List of Entities Types && Values"
        create_menu_action_unique(self, self.menu, unique_name, 'images/binoculars.png',
                              triggered=partial(self.show_entities_combobox_dialog),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        #------------------------------------------
        self.menu.addSeparator()
        unique_name = " "
        create_menu_action_unique(self, self.menu, unique_name, '',
                              triggered=partial(self.dummy),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.menu.addSeparator()
        #------------------------------------------
        self.utilities_menu = QMenu('[Menu] Utilities')

        self.utilities_action = self.menu.addMenu(self.utilities_menu)
        self.utilities_menu.setIcon(get_icon('images/utilities.png'))

        self.utilities_menu.setTearOffEnabled(True)
        self.utilities_menu.setWindowTitle('EM Utilities')
        self.utilities_menu.clear()

        self.utilities_menu.addSeparator()
        unique_name = "Import Calibre's 'Author Links' for Author-Type Entities using Link Name of 'Standard'"
        create_menu_action_unique(self, self.utilities_menu, unique_name, 'images/create.png',
                              triggered=partial(self.import_author_links),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.utilities_menu.addSeparator()
        unique_name = "                         "
        create_menu_action_unique(self, self.utilities_menu, unique_name, '',
                              triggered=partial(self.dummy),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.utilities_menu.addSeparator()
        unique_name = "                         "
        create_menu_action_unique(self, self.utilities_menu, unique_name, '',
                              triggered=partial(self.dummy),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        self.utilities_menu.addSeparator()
        unique_name = "Purge the 'Entities Manager' Database [Current Library Only]"
        create_menu_action_unique(self, self.utilities_menu, unique_name, 'images/delete.png',
                              triggered=partial(self.uninstall_entities_manager),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        #------------------------------------------
        self.menu.addMenu(self.utilities_menu)
        #------------------------------------------
        self.qaction.setMenu(self.menu)
        self.qaction.setIcon(get_icon(PLUGIN_ICONS[0]))
        self.qaction.triggered.connect(self.manage_entities)
        self.gui.keyboard.finalize()
        #------------------------------------------
        t = "<p style='white-space:wrap'>'Manage Entities' allows Links and Notes to be created for any 'Entity', and to 'Associate' those Entities with other Entities without reference to any particular book."
        t = t + "<br><br>'Quick (Pick)' means that the Entity is selected from the current Library View cursor column and row."
        t = t + "<br><br>'User Entity Types' are pseudo-Columns that you may create in lieu of using any 'real' Entity Types found in Calibre as 'real' Columns, Standard or Custom. \
        They will automatically be prefixed with a '~' symbol to differentiate them from Standard and Custom Columns."
        self.menu.setToolTip(t)
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def initialization_complete(self):
        self.guidb = self.gui.library_view.model().db
        self.library_path = self.guidb.library_path
        from calibre.gui2.ui import get_gui
        self.maingui = get_gui()
        del get_gui
        self.open_url = None
        self.icon = get_icon(PLUGIN_ICONS[0])
        self.font = QFont()
        self.font.setBold(False)
        self.font.setPointSize(12)
        self.my_cursor = None
        self.my_db = None
        self.do_special_conversions()
        self.copy_icon_to_icon_dir("entity_reserved.png")
        self.copy_icon_to_icon_dir("entity_reserved_blue.png")
        self.copy_icon_to_icon_dir("entity_reserved_brain.png")
        self.copy_icon_to_icon_dir("entity_reserved_red.png")
        self.copy_icon_to_icon_dir("entity_reserved_red_white_blue.png")
        self.copy_icon_to_icon_dir("entity_reserved_entity.png")
        self.copy_icon_to_icon_dir("entity_reserved_link.png")
        self.copy_icon_to_icon_dir("entity_reserved_link_note.png")
        self.copy_icon_to_icon_dir("entity_reserved_mandatory.png")
        self.copy_icon_to_icon_dir("entity_reserved_note.png")
        self.copy_icon_to_icon_dir("entity_reserved_square_blue.png")
        self.copy_icon_to_icon_dir("entity_reserved_star.png")
        self.copy_icon_to_icon_dir("entity_reserved_think.png")
        self.copy_icon_to_icon_dir("entity_reserved_transparent.png")
        self.copy_icon_to_icon_dir("entity_reserved_world.png")
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def library_changed(self,guidb):
        self.guidb = self.gui.library_view.model().db
        self.library_path = self.guidb.library_path
        try:
            self.entities_manager_dialog.close()
        except:
            pass
        try:
            self.entity_browser_dialog.close()
        except:
            pass
        self.my_cursor = None
        self.my_db = None
        self.do_special_conversions()
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #~ MAIN MENU SELECTIONS
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def manage_entities(self):
        self.guidb = self.gui.library_view.model().db
        try:
            self.entities_manager_dialog.close()
        except:
            pass
        user_types_list = self.get_user_types()
        from calibre_plugins.entities_manager.entities_manager_dialog import EntitiesManagerDialog
        self.entities_manager_dialog = EntitiesManagerDialog(self.gui,self.qaction.icon(),self.guidb,self.plugin_path,self.list_all_entities_all_info,user_types_list)
        self.entities_manager_dialog.show()
        self.entities_manager_dialog.setAttribute(Qt.WA_DeleteOnClose)
        del EntitiesManagerDialog
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def list_all_entities_all_info(self):

        self.entity_id_list = []
        self.entity_id_dict = {}
        self.entity_id_link_dict = {}
        self.entity_id_note_dict = {}
        self.entity_id_associations_dict = {}
        #~ -----------------------------
        entities_list = self.get_all_entity_id_types_values(close_apsw=False)
        #~ -----------------------------
        type_set = set()
        for row in entities_list:
            entity_id,type,value = row
            type_set.add(type)
        #END FOR
        type_list = list(type_set)
        del type_set
        type_list.sort()
        type_list.insert(0,"All Types")
        title = "List All Entities, Links, Notes, and Associations"
        label = "Select Which Entity Type?"
        selected_type,ok = QInputDialog.getItem(None, title, label, type_list,0,False)
        if not ok:
            return
        #~ -----------------------------
        for row in entities_list:
            entity_id,type,value = row
            r = type,value,entity_id
            if selected_type == "All Types" or selected_type == type:
                self.entity_id_list.append(r)
                self.entity_id_dict[entity_id] = type,value
        #END FOR
        del entities_list
        #~ -----------------------------
        self.entity_id_list.sort()
        #~ -----------------------------
        for row in self.entity_id_list:
            type,value,entity_id = row
            link_list = self.get_entity_links(entity_id,connect=False)
            self.entity_id_link_dict[entity_id] = link_list
            note_list = self.get_entity_notes(entity_id=entity_id,connect=False)
            self.entity_id_note_dict[entity_id] = note_list
            associations_list = self.get_entity_associations(entity_id,connect=False)
            self.entity_id_associations_dict[entity_id] = associations_list
        #END FOR
        del link_list
        del note_list
        del associations_list
        #~ -----------------------------
        info = ""
        for row in self.entity_id_list:
            type,value,entity_id = row
            entity_info = self.format_all_entities_all_info(entity_id,type,value)
            info = info + entity_info + "________________________________________________________________________<br>"
        #END FOR
        del row
        del entity_info
        #~ -----------------------------
        try:
            self.all_entities_viewer_dialog.close()
        except:
            pass
        #~ -----------------------------
        title = "All Entities' Information"   #not html
        from calibre_plugins.entities_manager.entities_viewer_dialog import EntitiesViewerDialog
        self.all_entities_viewer_dialog = EntitiesViewerDialog(self.qaction.icon(),title,info,self.font)
        self.all_entities_viewer_dialog.show()
        self.all_entities_viewer_dialog.setAttribute(Qt.WA_DeleteOnClose)
        del EntitiesViewerDialog
        #~ -----------------------------
        del self.entity_id_list
        del self.entity_id_dict
        del self.entity_id_link_dict
        del self.entity_id_note_dict
        del self.entity_id_associations_dict
   #------------------------------------------------------------------------------------------
    def format_all_entities_all_info(self,entity_id,type,value):
        if entity_id in self.entity_id_link_dict:
            link_list = self.entity_id_link_dict[entity_id]
        else:
            link_list = []
        if entity_id in self.entity_id_note_dict:
            note_list = self.entity_id_note_dict[entity_id]
        else:
            note_list = []
        if entity_id in self.entity_id_associations_dict:
            associations_list = self.entity_id_associations_dict[entity_id]
        else:
            associations_list = []
        #~ -----------------------------
        entity_id_type_value_list = []
        for row in associations_list:
            entity_id_1,entity_id_2,unifying_id = row
            if entity_id_1 != entity_id:
                t,v = self.get_entity_type_value(entity_id_1)
                r = t,v,"1",entity_id_1
                entity_id_type_value_list.append(r)
            if entity_id_2 != entity_id:
                t,v = self.get_entity_type_value(entity_id_2)
                r = t,v,"2",entity_id_2
                entity_id_type_value_list.append(r)
            if unifying_id != entity_id and unifying_id != entity_id_1 and unifying_id != entity_id_2:
                t,v = self.get_entity_type_value(unifying_id)
                r = t,v,"3",unifying_id
                entity_id_type_value_list.append(r)
            del r
            del row
        #END FOR
        #~ -----------------------------
        tab = "&nbsp;&nbsp;&nbsp;&nbsp;"
        entity_separator = "&nbsp;&nbsp;&nbsp;"
        #~ -----------------------------
        info = "<b>Entity:</b>" + tab + type + entity_separator + unicode_type(value) + "<br>"
        #~ -----------------------------
        if len(link_list) > 0:
            s = ""
            for row in link_list:
                link_id,name,link = row
                s = s + "<b>Link:</b>" + tab  + name + tab + link + "<br>"
            #END FOR
            info = info + s
        else:
            s = "<b>Link:</b>" + "<br>"
            info = info + s
        #~ -----------------------------
        if len(note_list) > 0:
            s = ""
            for row in note_list:
                note_id,name,note = row
                s = s + "<b>Note:</b>" + tab + name + "<br>" + note + "<br>"
            #END FOR
            s = s + "<br>"
            info = info + s
        else:
            s = "<b>Note:</b>" + "<br>"
            info = info + s
        #~ -----------------------------
        if len(entity_id_type_value_list) > 0:
            entity_id_type_value_list.sort()
            s = ""
            for row in entity_id_type_value_list:
                t,v,num,id = row
                s = s + "<b>Association:</b>" + tab + t + entity_separator + v + "<br>"
            #END FOR
            s = s + "<br>"
            info = info + s
        else:
            s = "<b>Association:</b>" + "<br>"
            info = info + s
        #~ --------------------------------------------------------------------------------------
        return info #single entity
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def display_entity_info(self):
        entity_id,type,value = self.quick_pick_entity()
        if entity_id is None:
            return
        #~ -----------------------------
        link_list = self.get_entity_links(entity_id=entity_id,connect=False)
        note_list = self.get_entity_notes(entity_id=entity_id,connect=False)
        associations_list = self.get_entity_associations(entity_id,connect=False)

        entity_id_type_value_list = []
        for row in associations_list:
            entity_id_1,entity_id_2,unifying_id = row
            if entity_id_1 != entity_id:
                t,v = self.get_entity_type_value(entity_id_1)
                r = t,v,"1",entity_id_1
                entity_id_type_value_list.append(r)
            if entity_id_2 != entity_id:
                t,v = self.get_entity_type_value(entity_id_2)
                r = t,v,"2",entity_id_2
                entity_id_type_value_list.append(r)
            if unifying_id != entity_id:
                t,v = self.get_entity_type_value(unifying_id)
                r = t,v,"3",unifying_id
                entity_id_type_value_list.append(r)
            del r
            del row
        #END FOR
        #~ -----------------------------
        self.apsw_close()
        #~ -----------------------------
        tab = "&nbsp;&nbsp;&nbsp;&nbsp;"
        entity_separator = "&nbsp;&&nbsp;"
        #~ -----------------------------
        info = "<b>Entity:</b>" + tab + type + entity_separator + value + "<br><br>"
        #~ -----------------------------
        if len(link_list) > 0:
            s = ""
            for row in link_list:
                link_id,name,link = row
                s = s + "<b>Link:</b>" + tab  + name + tab + link + "<br><br>"
            #END FOR
            info = info + s
        else:
            s = "<b>Link:</b>" + "<br><br>"
            info = info + s
        #~ -----------------------------
        if len(note_list) > 0:
            s = ""
            for row in note_list:
                note_id,name,note = row
                s = s + "<b>Note:</b>" + tab + name + "<br>" + note + "<br>"
            #END FOR
            s = s + "<br>"
            info = info + s
        else:
            s = "<b>Note:</b>" + "<br><br>"
            info = info + s
        #~ -----------------------------
        if len(entity_id_type_value_list) > 0:
            entity_id_type_value_list.sort()
            s = ""
            for row in entity_id_type_value_list:
                t,v,num,id = row
                s = s + "<b>Association:</b>" + tab + t + entity_separator + v + "<br>"
            #END FOR
            s = s + "<br>"
            info = info + s
        else:
            s = "<b>Association:</b>" + "<br><br>"
            info = info + s
        #~ --------------------------------------------------------------------------------------
        try:
            self.entity_viewer_dialog.close()
        except:
            pass
        #~ --------------------------------------------------------------------------------------
        title = "Entity:  '" + type + " & " + value + "'"   #not html

        from calibre_plugins.entities_manager.entity_viewer_dialog import EntityViewerDialog
        self.entity_viewer_dialog = EntityViewerDialog(self.qaction.icon(),title,info,self.font)
        self.entity_viewer_dialog.show()
        self.entity_viewer_dialog.setAttribute(Qt.WA_DeleteOnClose)
        del EntityViewerDialog
        del link_list
        del note_list
        del associations_list
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def get_all_entity_id_types_values(self,close_apsw=True):
        entities_list = []
        #~ -----------------------------
        is_valid = self.apsw_connect_to_library()
        if not is_valid:
            if DEBUG: print("Entities Manager: Fatal Error - apsw_connect_to_library failed.  Try Again.")
            return
        #~ -----------------------------
        mysql = "SELECT entity_id,type,value FROM _entities ORDER BY type ASC, value ASC "
        self.my_cursor.execute(mysql)
        tmp_rows = self.my_cursor.fetchall()
        if close_apsw:
            self.apsw_close()
        if not tmp_rows:
            tmp_rows = []
        if len(tmp_rows) == 0:
            return entities_list
        for row in tmp_rows:
            entities_list.append(row)
        #END FOR
        del tmp_rows
        return entities_list
   #------------------------------------------------------------------------------------------
    def get_entity_type_value(self,entity_id):
        mysql = "SELECT type,value FROM _entities WHERE entity_id = ?"
        self.my_cursor.execute(mysql,([entity_id]))
        tmp_rows = self.my_cursor.fetchall()
        if not tmp_rows:
            return None
        if len(tmp_rows) == 0:
            return None
        for row in tmp_rows:
            type,value = row
        #END FOR
        del tmp_rows
        return type,value
   #------------------------------------------------------------------------------------------
    def get_entity_id(self,type,value):

        entity_id = None

        mysql = "SELECT entity_id FROM _entities WHERE type = ? AND value = ?"
        self.my_cursor.execute(mysql,(type,value))
        tmp_rows = self.my_cursor.fetchall()
        if not tmp_rows:
            return None
        if len(tmp_rows) == 0:
            return None
        for row in tmp_rows:
            for col in row:
                entity_id = col
                break
        #END FOR
        return entity_id
   #------------------------------------------------------------------------------------------
    def get_bcol(self):
        bcol = None
        current_col = self.maingui.library_view.currentIndex().column()
        bcol = self.maingui.library_view.column_map[current_col]
        return bcol
    #-----------------------------------------------------------------------------------------
    def get_entity_links(self,entity_id=None,connect=True):
        link_list = []

        if connect:
            is_valid = self.apsw_connect_to_library()
            if not is_valid:
                if DEBUG: print("Entities Manager: Fatal Error - apsw_connect_to_library failed.  Try Again.")
                return

        if entity_id is not None:
            mysql = "SELECT link_id,name,link FROM _entities_links WHERE entity_id = ? ORDER BY name ASC"
            self.my_cursor.execute(mysql,([entity_id]))
        else:
            mysql = "SELECT entity_id FROM _entities_links GROUP BY entity_id"  #for answering "does an entity_id have any links?"  in colors and icons.
            self.my_cursor.execute(mysql)

        tmp_rows = self.my_cursor.fetchall()

        if connect:
            self.apsw_close()

        if not tmp_rows:
            return link_list
        if len(tmp_rows) == 0:
            return link_list
        for row in tmp_rows:
            if entity_id is None:
                for col in row:
                    link_id = col
                    link_list.append(link_id)
            else:
                link_list.append(row)
        #END FOR
        del tmp_rows
        return link_list
    #-----------------------------------------------------------------------------------------
    def get_entity_notes(self,entity_id=None,connect=True):
        note_list = []

        if connect:
            is_valid = self.apsw_connect_to_library()
            if not is_valid:
                if DEBUG: print("Entities Manager: Fatal Error - apsw_connect_to_library failed.  Try Again.")
                return

        if entity_id is not None:
            mysql = "SELECT note_id,name,note FROM _entities_notes WHERE entity_id = ? ORDER BY name ASC"
            self.my_cursor.execute(mysql,([entity_id]))
        else:
            mysql = "SELECT entity_id FROM _entities_notes GROUP BY entity_id"  #for answering "does an entity_id have any notes?"  in colors and icons.
            self.my_cursor.execute(mysql)

        tmp_rows = self.my_cursor.fetchall()

        if connect:
            self.apsw_close()

        if not tmp_rows:
            return note_list
        if len(tmp_rows) == 0:
            return note_list
        for row in tmp_rows:
            if entity_id is None:
                for col in row:
                    note_id = col
                    note_list.append(note_id)
            else:
                note_list.append(row)
        #END FOR
        del tmp_rows
        return note_list
    #-----------------------------------------------------------------------------------------
    def get_entity_associations(self,entity_id,connect=True):
        associations_list = []

        if connect:
            is_valid = self.apsw_connect_to_library()
            if not is_valid:
                if DEBUG: print("Entities Manager: Fatal Error - apsw_connect_to_library failed.  Try Again.")
                return

        mysql = "SELECT entity_id_1,entity_id_2,unifying_id FROM _entities_associations WHERE entity_id_1 = ? OR entity_id_2 = ? OR unifying_id = ? "
        self.my_cursor.execute(mysql,(entity_id,entity_id,entity_id))
        tmp_rows = self.my_cursor.fetchall()
        if connect:
            self.apsw_close()
        if not tmp_rows:
            return associations_list
        if len(tmp_rows) == 0:
            return associations_list
        for row in tmp_rows:
            associations_list.append(row)
        #END FOR
        del tmp_rows
        return associations_list
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def quick_pick_entity(self,close_apsw=False):

        entity_id = None
        type = None
        value = None

        book_id = self.maingui.library_view.current_book
        if book_id is None:
            return entity_id,type,value

        bcol = self.get_bcol()      #selected library_view column
        if not bcol:
            error_dialog(self.gui, _('EM'),_('No Column Was Selected. Your cursor must be within a cell underneath the desired Column.'), show=True)
            return entity_id,type,value

        mi = self.guidb.new_api.get_metadata(book_id)
        fm = None

        if bcol.startswith("#"):
            type = bcol
            fm = mi.metadata_for_field(bcol)  # field metadata dict
            datatype = fm['datatype']
            if  datatype == "int" or datatype == "float" or datatype == "bool" or datatype == "ratings" or datatype == "composite":
                del mi
                del fm
                return entity_id,type,value #unsupported custom column
        else:
            if bcol == "authors":
                type = "Author"
            elif bcol == "publisher":
                type = "Publisher"
            elif bcol == "series":
                type = "Series"
            elif bcol == "tags":
                type = "Tags"
            elif bcol == "title":
                type = "Title"
            else:
                return entity_id,type,value  #unsupported standard column

        value = mi.get(bcol, None)
        if isinstance(value,list):
            if len(value) > 0:
                if len(value) > 1:
                    title = "Value?"
                    label = "Multiple Values for Selected Column: Choose One (1)"
                    value,ok = QInputDialog.getItem(None, title, label, value,0,False)
                    if not ok:
                        return entity_id,type,value
                else:
                    value = value[0]
            else:
                value = None

        del mi
        del fm

        msg = "Quick Pick Entity Type: " + bcol + "  Value: " + as_unicode(value)
        self.maingui.status_bar.show_message(_(msg), 2000)

        is_valid = self.apsw_connect_to_library()
        if not is_valid:
            if DEBUG: print("Entities Manager: Fatal Error - apsw_connect_to_library failed.  Try Again.")
            return None,None,None

        entity_id = self.get_entity_id(type,value)

        if close_apsw:
            self.apsw_close()

        return entity_id,type,value
    #-----------------------------------------------------------------------------------------
    def quick_execute_link(self):

        entity_id,type,value = self.quick_pick_entity(close_apsw=False)
        if entity_id is None:
            return

        link_list = self.get_entity_links(entity_id,connect=False)

        self.apsw_close()

        if not isinstance(link_list,list):
            return

        if len(link_list) == 0:
            return

        items = []

        for row in link_list:
            link_id,name,link = row
            items.append(name)
        #END FOR

        if len(link_list) > 1:
            self.quick_execute_multiple_links(entity_id,type,value,link_list)
            return
        else:
            selected_name = items[0]
            del items

        for row in link_list:
            link_id,name,link = row
            if name == selected_name:
                selected_link = link
                break
        #END FOR

        link = selected_link.strip()

        link = link.replace("\\","/")
        if link.startswith('"') and link.endswith('"'):  #Windows copy-path artifacts
            link = link[1:-1]

        if not link > " ":
            return

        if self.open_url is None:
            from calibre.gui2 import open_url
            self.open_url = open_url
            del open_url

        try:
            self.open_url(QUrl(link, QUrl.ParsingMode.TolerantMode))
            del link
        except Exception as e:
            msg = "Open URL Error for Link:  " + as_unicode(e)
            if DEBUG: print(msg)
            error_dialog(None, _('EM'),_(msg), show=True)
    #-----------------------------------------------------------------------------------------
    def quick_execute_multiple_links(self,entity_id,type,value,link_list):
        from calibre_plugins.entities_manager.link_viewer_dialog import LinkViewerDialog
        self.link_viewer_dialog = LinkViewerDialog(self.qaction.icon(),self.font,entity_id,type,value,link_list)
        self.link_viewer_dialog.show()
        self.link_viewer_dialog.setAttribute(Qt.WA_DeleteOnClose)
        del LinkViewerDialog
    #-----------------------------------------------------------------------------------------
    def quick_display_note(self):

        entity_id,type,value = self.quick_pick_entity(close_apsw=False)
        if entity_id is None:
            return

        note_list = self.get_entity_notes(entity_id,connect=False)

        self.apsw_close()

        if not isinstance(note_list,list):
            return

        if len(note_list) == 0:
            return

        items = []

        for row in note_list:
            note_id,name,note = row
            items.append(name)
        #END FOR

        if len(note_list) > 1:
            title = "Notes"
            label = "Choose Note for Multiple Notes"
            selected_name,ok = QInputDialog.getItem(None, title, label, items,0,False)
            if not ok:
                del note_list
                del items
                return
        else:
            selected_name = items[0]
            del items

        for row in note_list:
            note_id,name,note = row
            if name == selected_name:
                selected_note = note
                break
        #END FOR

        note = selected_note.strip()

        try:
            self.note_viewer_dialog.close()
        except:
            pass

        title = "Note '" + selected_name + "' for Entity  '" + type + " - " + value + "'"

        from calibre_plugins.entities_manager.note_viewer_dialog import NoteViewerDialog
        self.note_viewer_dialog = NoteViewerDialog(self.qaction.icon(),title,note,self.font)
        self.note_viewer_dialog.show()
        self.note_viewer_dialog.setAttribute(Qt.WA_DeleteOnClose)
        del NoteViewerDialog
    #-----------------------------------------------------------------------------------------
    def quick_create_entity(self):
        entity_id,type,value = self.quick_pick_entity()
        if entity_id is not None:  #already exists
            return
        if type is None or value is None:
            return

        mysql = "INSERT OR IGNORE INTO _entities(entity_id,type,value) VALUES (null,?,?)"
        self.my_cursor.execute("begin")
        self.my_cursor.execute(mysql,(type,value))
        self.my_cursor.execute("commit")
        self.apsw_close()

        msg = "Entity created for:  " + type + " - " + value
        self.gui.status_bar.show_message(_(msg), 10000)
    #-----------------------------------------------------------------------------------------
    def quick_add_link(self):
        entity_id,type,value = self.quick_pick_entity()
        if entity_id is None:
            return

        title = "Quick Add Link"
        label = "Specify the Link Name for the new Link for the current Entity"
        name,ok = QInputDialog.getText(None, title, label)
        if not ok:
            return

        name = name.title().strip()
        if not name > " ":
            return

        label = "Specify the Link Value for the new Link for the current Entity"
        link,ok = QInputDialog.getText(None, title, label)
        if not ok:
            return

        name = name.title().strip()
        link = link.strip()

        if (not name > " ") or (not link > " "):
            return

        mysql = "INSERT OR IGNORE INTO _entities_links (link_id,entity_id,name,link) VALUES (null,?,?,?)   "
        self.my_cursor.execute("begin")
        self.my_cursor.execute(mysql,(entity_id,name,link))
        self.my_cursor.execute("commit")

        self.apsw_close()
    #-----------------------------------------------------------------------------------------
    def quick_add_note(self):
        entity_id,type,value = self.quick_pick_entity()
        if entity_id is None:
            return

        title = "Quick Add Note"
        label = "Specify the Name for the new Note for the current Entity"
        name,ok = QInputDialog.getText(None, title, label)
        if not ok:
            return

        name = name.title().strip()
        if not name > " ":
            return

        label = "Specify the Note Value for the new Note to the current Entity"
        note,ok = QInputDialog.getMultiLineText(None, title, label)
        if not ok:
            return

        name = name.title().strip()
        note = note.strip()

        if (not name > " ") or (not note > " "):
            return

        mysql = "INSERT OR IGNORE INTO _entities_notes (note_id,entity_id,name,note) VALUES (null,?,?,?)   "
        self.my_cursor.execute("begin")
        self.my_cursor.execute(mysql,(entity_id,name,note))
        self.my_cursor.execute("commit")

        self.apsw_close()
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #~ USER-DEFINED ENTITY TYPES
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def user_type_add(self):

        title = "Add User Entity Type"
        label = "Specify the Pseudo-Column's ~Type (no spaces)"
        type,ok = QInputDialog.getText(None, title, label)
        if not ok:
            return

        type = type.replace(" ","")

        label = "Specify the Pseudo-Column's 'Label' (name/title)"
        label,ok = QInputDialog.getText(None, title, label)
        if not ok:
            return

        type = type.replace("#","").strip()
        type = type.replace("~","").strip()

        label = label.strip()

        if not type > " " or not label > " ":
            return

        s =  type[0:1].upper()  #Allow Camel-Casing e.g. ~FutureBooks by not title casing
        type = type[1: ]
        type = '~' + s + type
        label = label.title()

        is_valid = self.apsw_connect_to_library()
        if not is_valid:
            if DEBUG: print("Entities Manager: Fatal Error - apsw_connect_to_library failed.  Try Again.")

        try:
            mysql = "INSERT OR IGNORE INTO _entities_user_types (type_id,type,label) VALUES (null,?,?)   "
            self.my_cursor.execute("begin")
            self.my_cursor.execute(mysql,(type,label))
            self.my_cursor.execute("commit")
        except Exception as e:
            if DEBUG: print("Add User Entity Type Failure: ", as_unicode(e))

        self.apsw_close()
    #-----------------------------------------------------------------------------------------
    def user_type_change(self):

        is_valid = self.apsw_connect_to_library()
        if not is_valid:
            if DEBUG: print("Entities Manager: Fatal Error - apsw_connect_to_library failed.  Try Again.")
            return

        user_types_list = self.get_user_types()
        types_list = []
        for row in user_types_list:
            type,label = row
            if type == ASSOCIATION_ENTITY_TYPE:  #EM Reserved Entity Type...
                continue
            r = type + " >>> " + label
            types_list.append(r)
        #END FOR
        del user_types_list
        types_list.sort()

        title = "Change the Label/Title of a User Defined Entity Type"
        label = "Select Entity Type to Change"
        selected_type,ok = QInputDialog.getItem(None, title, label, types_list,0,False)
        del types_list
        if not ok:
            self.apsw_close()
            return

        s_split = selected_type.split(" >>> ")

        if not isinstance(s_split,list):
            self.apsw_close()
            return
        if not len(s_split) > 0:
            self.apsw_close()
            return

        selected_type = s_split[0].strip()
        name = s_split[1].strip()

        label = "Specify the New Label/Title for " + selected_type + " >>> " + name
        name,ok = QInputDialog.getText(None, title, label)
        if not ok:
            self.apsw_close()
            return

        name = name.strip()

        mysql = "UPDATE _entities_user_types SET label = ? WHERE type = ? "
        self.my_cursor.execute("begin")
        self.my_cursor.execute(mysql,(name,selected_type))
        self.my_cursor.execute("commit")

        self.apsw_close()
    #-----------------------------------------------------------------------------------------
    def user_type_delete(self):

        is_valid = self.apsw_connect_to_library()
        if not is_valid:
            if DEBUG: print("Entities Manager: Fatal Error - apsw_connect_to_library failed.  Try Again.")

        user_types_list = self.get_user_types()

        types_list = []
        for row in user_types_list:
            type,label = row
            if type == ASSOCIATION_ENTITY_TYPE:  #EM Reserved Entity Type...
                continue
            r = type + " >>> " + label
            types_list.append(r)
        #END FOR
        del user_types_list
        types_list.sort()

        title = "Delete a User Defined Entity Type"
        label = "Select Entity Type to Delete"
        selected_type,ok = QInputDialog.getItem(None, title, label, types_list,0,False)
        del types_list
        if not ok:
            self.apsw_close()
            return

        s_split = selected_type.split(" >>> ")

        if not isinstance(s_split,list):
            self.apsw_close()
            return
        if not len(s_split) > 0:
            self.apsw_close()
            return

        type = s_split[0].strip()
        label = s_split[1].strip()

        msg = "Delete User Defined Entity Type " + type + ", including all Entities for that Entity Type plus their Associations, Links and Notes?  Are you sure?  This cannot be undone."
        if question_dialog(self.gui, 'EM',msg):
            pass
        else:
            self.apsw_close()
            msg = "Deletion of User Defined Entity Type has been cancelled.  Nothing done."
            return info_dialog(self.gui, 'EM Utility',msg).show()

        self.my_cursor.execute("begin")
        mysql = "DELETE FROM _entities WHERE type = ? "
        self.my_cursor.execute(mysql,([type]))
        mysql = "DELETE FROM _entities_user_types WHERE type = ? "
        self.my_cursor.execute(mysql,([type]))
        mysql = "DELETE FROM _entities_type_table_mapping WHERE type = ? "
        self.my_cursor.execute(mysql,([type]))
        self.my_cursor.execute("commit")

        self.my_cursor.execute("begin")
        mysql = "DELETE FROM _entities_links WHERE entity_id NOT IN (SELECT entity_id FROM _entities)  "
        self.my_cursor.execute(mysql)
        mysql = "DELETE FROM _entities_notes WHERE entity_id NOT IN (SELECT entity_id FROM _entities)  "
        self.my_cursor.execute(mysql)
        mysql = "DELETE FROM _entities_associations WHERE ((entity_id_1 NOT IN (SELECT entity_id FROM _entities)) \
                                                                                      OR (entity_id_2 NOT IN (SELECT entity_id FROM _entities)) \
                                                                                      OR (unifying_id NOT IN (SELECT entity_id FROM _entities)))"
        self.my_cursor.execute(mysql)
        self.my_cursor.execute("commit")

        self.apsw_close()

        msg = "User Defined Entity Type " + type + " has been deleted, including all its related Entities, Link, Notes and Associations."
        info_dialog(self.gui, 'EM Utility',msg).show()
    #-----------------------------------------------------------------------------------------
    def get_user_types(self):

        if self.my_cursor is None or self.my_db is None:
            close_db = True
        else:
            close_db = False

        if close_db:
            is_valid = self.apsw_connect_to_library()
            if not is_valid:
                if DEBUG: print("Entities Manager: Fatal Error - apsw_connect_to_library failed.  Try Again.")
                return user_types_list

        user_types_list = []

        try:
            mysql = "SELECT type,label FROM _entities_user_types ORDER BY type ASC"
            self.my_cursor.execute(mysql)
            tmp_rows = self.my_cursor.fetchall()
        except:
            tmp_rows = []  #very first use of plugin before tables created...
        if close_db:
            self.apsw_close()
        if not tmp_rows:
            return user_types_list
        if len(tmp_rows) == 0:
            return user_types_list
        for row in tmp_rows:
            user_types_list.append(row)
        #END FOR
        del tmp_rows

        return user_types_list
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #~ UTILITIES
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def import_author_links(self):
        if question_dialog(self.gui, 'EM',"Import all current Calibre 'Author Links', creating/enhancing Entities with a Type of 'Author' with a Value of the Author's Name.  Are you sure?"):
            pass
        else:
            return

        is_valid = self.apsw_connect_to_library()
        if not is_valid:
            if DEBUG: print("Entities Manager: Fatal Error - apsw_connect_to_library failed.  Try Again.")

        mysql = "SELECT name,link FROM authors WHERE link IS NOT NULL AND link > ' ' ORDER BY name ASC"
        self.my_cursor.execute(mysql)
        tmp_rows = self.my_cursor.fetchall()
        if not tmp_rows:
            tmp_rows = []
        self.my_cursor.execute("begin")
        for row in tmp_rows:
            value,link = row
            type = "Author"
            mysql = "INSERT OR IGNORE INTO _entities(entity_id,type,value) VALUES (null,?,?)"
            self.my_cursor.execute(mysql,(type,value))
        #END FOR
        total_added = 0
        self.my_cursor.execute("commit")
        self.my_cursor.execute("begin")
        for row in tmp_rows:
            type = "Author"
            value,link = row
            if value > " " and link > " ":
                name = "Standard"
                entity_id = self.get_entity_id(type,value)
                if not isinstance(entity_id,int):
                    continue
                link_list = self.get_entity_links(entity_id,connect=False)
                n = None
                for r in link_list:
                    id,n,lnk = r
                    if name == n:
                        break
                #END FOR
                if name == n:
                    continue
                total_added = total_added + 1
                mysql = "INSERT OR IGNORE INTO _entities_links (link_id,entity_id,name,link) VALUES (null,?,?,?) "
                self.my_cursor.execute(mysql,(entity_id,name,link))
        #END FOR
        self.my_cursor.execute("commit")
        del tmp_rows
        self.apsw_close()
        msg = "Standard Calibre Author Links have been imported into Entities Manager if they did not already exist with the Link Name of 'Standard'.<br><br>Total Imported: " + as_unicode(total_added)
        return info_dialog(self.gui, 'EM Utility',msg).show()
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def uninstall_entities_manager(self):
        if question_dialog(self.gui, 'EM','Delete All EM Data and Uninstall All EM Tables and Column Rules?  Are you sure?'):
            pass
        else:
            return

        self.remove_column_icon_with_text_rules()
        self.remove_column_color_rules()

        is_valid = self.apsw_connect_to_library()
        if not is_valid:
            if DEBUG: print("Entities Manager: Fatal Error - apsw_connect_to_library failed.  Try Again.")
            return

        self.my_cursor.execute("begin")
        mysql = "DROP TABLE IF EXISTS _entities_manager_preferences  "
        self.my_cursor.execute(mysql)
        mysql = "DROP TABLE IF EXISTS _entities_preferences  "
        self.my_cursor.execute(mysql)
        mysql = "DROP TABLE IF EXISTS _entities_type_preferences  "
        self.my_cursor.execute(mysql)
        mysql = "DROP TABLE IF EXISTS _entities_user_types  "
        self.my_cursor.execute(mysql)
        mysql = "DROP TABLE IF EXISTS _entities_type_table_mapping  "
        self.my_cursor.execute(mysql)
        mysql = "DROP TABLE IF EXISTS _entities_notes  "
        self.my_cursor.execute(mysql)
        mysql = "DROP TABLE IF EXISTS _entities_links   "
        self.my_cursor.execute(mysql)
        mysql = "DROP TABLE IF EXISTS _entities_associations   "
        self.my_cursor.execute(mysql)
        mysql = "DROP TABLE IF EXISTS _entities  "
        self.my_cursor.execute(mysql)
        self.my_cursor.execute("commit")
        self.apsw_close()

        return info_dialog(self.gui, 'EM Utility', 'Entities Manager tables and data have been deleted from this Library (only).\
        <br><br>Tables will be created automatically if you use any further EM functions in this Library.\
        <br><br>This Calibre Library should now be "checked" to vacuum/defragment/compress it.').show()
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def dummy(self):
        return
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #~ SPECIAL CONVERSION
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def do_special_conversions(self):
        self.convert_user_defined_type_symbol()
    #-----------------------------------------------------------------------------------------
    def convert_user_defined_type_symbol(self):
        #~  @ --> ~        @ used through Version 1.0.3
        is_valid = self.apsw_connect_to_library()
        if not is_valid:
            if DEBUG: print("Entities Manager: Fatal Error - apsw_connect_to_library failed.  Try Again.")
            return

        try:
            self.my_cursor.execute("begin")
            mysql = "UPDATE _entities SET type = replace(type,'@','~'),value = replace(value,'@','~') "
            self.my_cursor.execute(mysql)
            mysql = "UPDATE _entities_type_table_mapping SET type = replace(type,'@','~') "
            self.my_cursor.execute(mysql)
            mysql = "UPDATE _entities_user_types SET type = replace(type,'@','~') "
            self.my_cursor.execute(mysql)
            self.my_cursor.execute("commit")
        except:
            pass #no EM tables for this library...

        self.apsw_close()
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #~ COLUMN ICON RULES
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def add_column_icon_with_text_rules(self):
        self.maintain_column_icon_with_text_rules(mode=MODE_ADD)
    #-----------------------------------------------------------------------------------------
    def remove_column_icon_with_text_rules(self):
        self.maintain_column_icon_with_text_rules(mode=MODE_REMOVE)
    #-----------------------------------------------------------------------------------------
    def maintain_column_icon_with_text_rules(self,mode=MODE_ADD):
        is_valid = self.apsw_connect_to_library()
        if not is_valid:
            return error_dialog(self.gui, _('EM'),_('Entities Manager could not connect to the database.  Please try again.'), show=True)
        self.entity_reserved_icon_filename_prefix = self.get_em_preference('column_icon_rules_reserved_icon_filename_prefix')
        self.entity_reserved_icon_filename_default = self.get_em_preference('column_icon_rules_reserved_icon_filename_default')
        self.entity_type_preference_add_column_icon_rule_dict = self.get_entity_type_preference_dict('add_column_icon_rule')
        self.entity_type_preference_column_icon_filename_dict = self.get_entity_type_preference_dict('column_icon_filename')
        self.entity_type_preference_icon_if_link_note_dict = self.get_entity_type_preference_dict('icon_if_link_note')
        self.entity_id_preference_dict = self.get_entity_id_preference_dict('column_icon_filename')
        self.entity_id_link_list = self.get_entity_links(entity_id=None,connect=False)
        self.entity_id_link_set = set(self.entity_id_link_list)
        self.entity_id_note_list = self.get_entity_notes(entity_id=None,connect=False)
        self.entity_id_note_set = set(self.entity_id_note_list)
        self.apsw_close()

        is_valid = self.copy_icon_to_icon_dir(self.entity_reserved_icon_filename_default)
        if not is_valid:
            return error_dialog(self.gui,'EM','New Entity Column Icon Rules Were Not Created.  Entity Icon Error.', show=True)
        is_valid = self.get_entities_data(connect=True)
        if not is_valid:
            return error_dialog(self.gui,'EM','New Entity Column Icon Rules Were Not Created.  Existing Entities Could Not Be Retrieved Error.', show=True)
        is_valid = self.add_new_entity_column_icon_rules(kind="icon",prefname = 'column_icon_rules',rule_type="icon with text",png=self.entity_reserved_icon_filename_default,mode=mode)
        #~ self.apsw_close()
        if not is_valid:
            return error_dialog(self.gui,'EM','New Entity Column Icon Rules Were Not Created.  Rule Creation Error.', show=True)
        else:
            return info_dialog(self.gui, 'EM Maintain Entity Column Icon Rules',self.info_msg, show=True)
    #-----------------------------------------------------------------------------------------
    def add_new_entity_column_icon_rules(self,kind=None,prefname=None,rule_type=None,png=None,mode=None):
        if kind is None or prefname is None or rule_type is None or png is None or mode is None:
            return False
        #-------------------------------
        #-------------------------------
        self.rule_kind = kind
        self.prefname = prefname
        self.rule_type = rule_type
        self.png = png
        self.reserved_icon_prefix =  as_unicode(self.entity_reserved_icon_filename_prefix) #any icon starting with "entity_reserved" is an EM-only icon.

        self.db = self.gui.library_view.model().db
        self.fm = self.db.field_metadata
        #---------------------------------
        #---------------------------------
        from calibre.gui2.preferences.coloring import ConditionEditor, RuleEditor, RulesModel, EditRules
        from calibre.library.coloring import Rule
        #---------------------------------
        edit_rules_widget = EditRules()  #has the export button and the add rule button...
        edit_rules_widget.initialize(self.fm, self.db.prefs, None, self.prefname)   #it gets all of the existing prefs rules     mi=None
        edit_rules_widget.show()
        edit_rules_widget.hide()
        #~ edit_rules_widget.pref_name = 'column_icon_rules'

        #---------------------------------
        is_valid = self.export_all_rules_to_backup_dir(edit_rules_widget,rule_type="icon")
        if not is_valid:
            edit_rules_widget.close()
            return False
        #---------------------------------

        n1 = len(edit_rules_widget.model.rules)
        total_original_rules = as_unicode(n1)
        #~ if DEBUG: print("number of original rows in edit_rules_widget.model.rules prior to EM rule deletion: ", as_unicode(n1))
        total_deleted = 0
        for g in range(0,n1):
            n2 = len(edit_rules_widget.model.rules)
            if n2 == 0:
                break
            for row_number in range(0,n2):
                try:
                    rule = edit_rules_widget.model.rules[row_number]
                    x,y,robject = rule
                    r = robject.template  #what you see in an export file
                    if self.reserved_icon_prefix in as_unicode(r):  #any icon starting with "entity_reserved" is an EM-only icon.
                        index = edit_rules_widget.model.index(row_number)
                        edit_rules_widget.model.remove_rule(index)
                        total_deleted = total_deleted + 1
                except:
                    continue  #removing rows causes row number to become invalid
                if len(edit_rules_widget.model.rules) == 0:
                    break
            #END FOR
        #END FOR
        #~ if DEBUG: print("-----------number of icon rules deleted: ", as_unicode(total_deleted))
        n = len(edit_rules_widget.model.rules)
        #~ if DEBUG: print("number of original rows in edit_rules_widget.model.rules AFTER REMOVALS of all prior EM rules: ", as_unicode(n))
        #---------------------------------
        is_valid = self.apsw_connect_to_library()
        if not is_valid:
            return False
        #---------------------------------
        rule_editor_dialog = RuleEditor(self.fm, self.prefname)    #'column_icon_rules'
        rule_editor_dialog.show()
        rule_editor_dialog.hide()
        rule_editor_dialog.rule_kind = self.rule_kind
        #---------------------------------
        rule_editor_dialog.populate_icon_filenames()
        #~ rule_editor_dialog.rule_icon_files.append(self.png)
        #~ rule_editor_dialog.filename_box.setCurrentIndex(rule_editor_dialog.filename_box.findText(self.png))
        #---------------------------------
        rule_editor_dialog.conditions = []
        rule_editor_dialog.add_blank_condition()
        #---------------------------------
        total_entities = as_unicode(len(self.real_entities_list))
        total_entities_excluded = 0
        duplicates_checking_set = set()
        total_added = 0
        total_entities_skipped = 0  #due to user customization
        icon_to_use = None

        if mode != MODE_REMOVE:

            for row in self.real_entities_list:
                entity_id,type,value = row

                if type in self.entity_type_preference_add_column_icon_rule_dict:
                    active = self.entity_type_preference_add_column_icon_rule_dict[type]
                    if active == "0":
                        total_entities_skipped = total_entities_skipped + 1
                        continue

                if type in self.entity_type_preference_icon_if_link_note_dict:
                    active = self.entity_type_preference_icon_if_link_note_dict[type]
                    if active == "0":  #does not matter to the user if exists or not
                        pass
                    else:
                        if (entity_id in self.entity_id_link_set) or (entity_id in self.entity_id_note_set):
                            pass
                        else:
                            total_entities_skipped = total_entities_skipped + 1
                            continue
                else:
                    total_entities_skipped = total_entities_skipped + 1
                    continue

                if not value > " ":
                    total_entities_excluded = total_entities_excluded + 1
                    continue

                if '"' in value:
                    total_entities_excluded = total_entities_excluded + 1
                    continue  # double-quote " in value causes future template parsing errors not visible to the user unless in debug...

                if type in self.entity_column_dict:
                    current_column,is_multiple = self.entity_column_dict[type]
                    current_column = current_column.lower()
                    if type.startswith("#"):
                        current_column = type
                    #~ if DEBUG: print("current_column: ", type)
                else:
                    if DEBUG: print("Fatal Error: type NOT in self.entity_column_dict: ",type)
                    return False

                if is_multiple:
                    action = "has"
                else:
                    action = "is"
                c = (current_column,action,value)      # ('tags', 'has', value)
                if c in duplicates_checking_set:
                    #~ if DEBUG: print("new duplicate rule being skipped: ", as_unicode(c))
                    continue
                else:
                    duplicates_checking_set.add(c)
                ruleobject = Rule(edit_rules_widget.model.fm)  # SIGNATURE = '# BasicColorRule():'
                ruleobject.add_condition(current_column, action, value)  # ('tags', 'has', value)

                icon_to_use = self.png

                if type in self.entity_type_preference_column_icon_filename_dict:
                    icon_to_use = self.entity_type_preference_column_icon_filename_dict[type]
                    #~ if DEBUG: print("entity type uses: ", icon_to_use, "  for entity_id: ", entity_id)

                if entity_id in self.entity_id_preference_dict:
                    icon_to_use = self.entity_id_preference_dict[entity_id]
                    #~ if DEBUG: print("entity id uses: ", entity_id, icon_to_use)

                if not icon_to_use in rule_editor_dialog.rule_icon_files:
                    rule_editor_dialog.rule_icon_files.append(icon_to_use)

                i = rule_editor_dialog.filename_box.findText(icon_to_use)
                if i == -1:
                    icon_to_use = self.png  #user deleted the icon file after customization to use it...
                    #~ if DEBUG: print(".findText not found; switching to default: ", icon_to_use)

                rule_editor_dialog.filename_box.setCurrentIndex(rule_editor_dialog.filename_box.findText(icon_to_use))

                ruleobject.color = icon_to_use  #yes, color is the icon filename(s) for icon kinds...

                #~ if DEBUG: print("currently used icon_to_use: ", icon_to_use)

                rule_editor_dialog.column_box.setCurrentText(current_column)
                idx = rule_editor_dialog.kind_box.findText(self.rule_type)
                rule_editor_dialog.kind_box.setCurrentIndex(idx)
                edit_rules_widget.pref_name = self.prefname
                edit_rules_widget.model.pref_name = self.prefname
                rule_index = edit_rules_widget.model.add_rule(rule_editor_dialog.rule_kind, current_column, ruleobject)
                edit_rules_widget.changed.emit()
                rule_editor_dialog.apply_rule(self.rule_type, current_column, ruleobject)
                rule_editor_dialog.accept()
                del ruleobject
                total_added = total_added + 1
            #END FOR
            #~ if DEBUG: print("Number of EM column icon rules maintained: ", as_unicode(total_added))
        else:
            pass
        #---------------------------------
        #---------------------------------
        if len(edit_rules_widget.model.rules) > 0 or total_deleted > 0:
            edit_rules_widget.model.commit(self.db.prefs)
        #---------------------------------
        m = self.maingui.library_view.model()   #some columns are not refreshing their icons after just above completes...this completely fixes that.
        m.beginResetModel(), m.endResetModel()
        self.maingui.library_view.refresh_grid()
        #---------------------------------
        total_rules = as_unicode(len(edit_rules_widget.model.rules))
        #~ if DEBUG: print("Grand Total Rules now in Preferences > Look & Feel > Column Icon Rules: ", total_rules )
        #~ if DEBUG: print("total_entities_skipped: ", as_unicode(total_entities_skipped))
        #---------------------------------
        edit_rules_widget.close()
        rule_editor_dialog.close()
        #---------------------------------
        if mode == MODE_ADD:
            self.info_msg = self.info_msg + "<br><br>MODE:  'Add/Refresh' "
            self.info_msg = self.info_msg + "<br><br>Total Entities in EM: " + total_entities
            self.info_msg = self.info_msg + "<br><br>Total Entities in EM excluded due to unusable Values: " + as_unicode(total_entities_excluded)
            total_eligible = int(total_entities) - int(total_entities_excluded)
            self.info_msg = self.info_msg + "<br><br>Total Eligible Entities in EM: " + as_unicode(total_eligible)
            self.info_msg = self.info_msg + "<br><br>Total Eligible Entities in EM excluded per User Customization: " + as_unicode(total_entities_skipped)
            self.info_msg = self.info_msg + "<br><br>Final Eligible Entities in EM: " +  as_unicode(int(total_entities) - int(total_entities_excluded) - total_entities_skipped)
            self.info_msg = self.info_msg + "<br><br>Total Original Rules <b>Before</b> Entities Updates: " + total_original_rules
            self.info_msg = self.info_msg + "<br><br>Total Old EM Rules Deleted: " + as_unicode(total_deleted)
            self.info_msg = self.info_msg + "<br><br>Total New EM Rules Added: " + as_unicode(total_added)
            total_manual_rules = int(total_rules) - int(total_entities) + int(total_entities_excluded) + total_entities_skipped
            self.info_msg = self.info_msg + "<br><br>Total Manually-Created Rules: " + as_unicode(total_manual_rules)
            self.info_msg = self.info_msg + "<br><br>Total Rules <b>After</b> Entities Updates: " + total_rules
            self.info_msg = self.info_msg + "<br>"
        elif mode == MODE_REMOVE:
            self.info_msg = self.info_msg + "<br><br>MODE:  'Remove' "
            self.info_msg = self.info_msg + "<br><br>Total Original Rules <b>Before</b> Entities Updates: " + total_original_rules
            self.info_msg = self.info_msg + "<br><br>Total Manually-Created Rules: " + total_rules
            self.info_msg = self.info_msg + "<br><br>Total Rules <b>After</b> Entities Updates: " + total_rules
            self.info_msg = self.info_msg + "<br>"

        if DEBUG: print(self.info_msg)

        del edit_rules_widget
        del rule_editor_dialog
        del self.db
        del self.fm
        del self.real_entities_list
        del self.png
        del icon_to_use
        del self.prefname
        del self.rule_type
        del total_original_rules
        del total_rules
        total_eligible = None
        del total_eligible
        del total_entities_skipped
        total_manual_rules = None
        del total_manual_rules
        del total_entities_excluded
        del total_entities
        del self.entity_type_preference_add_column_icon_rule_dict
        del self.entity_type_preference_column_icon_filename_dict
        del self.entity_type_preference_icon_if_link_note_dict
        del self.entity_id_preference_dict
        del self.entity_id_link_list
        del self.entity_id_link_set
        del self.entity_id_note_list
        del self.entity_id_note_set

        return True
    #-----------------------------------------------------------------------------------------
    def copy_icon_to_icon_dir(self,icon_file=None):
        if icon_file is None:
            return False
        icon_dir = os.path.join(config_dir, 'cc_icons')
        if not os.path.isdir(icon_dir):
            os.mkdir(icon_dir)
        if not os.path.isdir(icon_dir):
            return False
        del icon_dir
        zipfile_path = self.plugin_path
        zip_dir_name = "cc_icons"
        zip_icon_path = zip_dir_name + "/" + icon_file
        zfile = zipfile.ZipFile(zipfile_path)
        try:
            zfile.extract(zip_icon_path, config_dir)
            del zfile
            return True
        except:
            del zfile
            return False
   #-----------------------------------------------------------------------------------------
    def select_default_reserved_icon_column_icon_rules_em(self):
        #changes the default icon from entity_reserved.png to entity_reservedxxxxxxxxx.png
        is_valid = self.apsw_connect_to_library()
        if not is_valid:
            if DEBUG: print("Entities Manager: Fatal Error - apsw_connect_to_library failed.  Try Again.")
            return

        reserved_prefix = self.get_em_preference('column_icon_rules_reserved_icon_filename_prefix')
        current_default = self.get_em_preference('column_icon_rules_reserved_icon_filename_default')

        icon_filename = self.get_user_selected_reserved_icon_filename()
        if icon_filename is None:
            self.apsw_close()
            return info_dialog(self.gui, _('EM'),_('Nothing Done.'), show=True)

        head,icon_filename = os.path.split(icon_filename)
        icon_filename = icon_filename.strip()

        if (not icon_filename.startswith(reserved_prefix)) or (" " in icon_filename) or (not icon_filename.endswith(".png")):
            self.apsw_close()
            msg = "New Default Icon Filename for EM is Invalid. Filename does not start with:  '" + reserved_prefix  + "' or ends with '.png' or contains a space.   Nothing Done."
            return error_dialog(self.gui, _('EM'),_(msg), show=True)

        is_valid = self.check_icon_filename_for_existance(icon_filename)
        if not is_valid:
            msg = "New Default Icon Filename for EM Does Not Exist in the '.../calibre/cc_icons/'    directory.  Nothing Done."
            return error_dialog(self.gui, _('EM'),_(msg), show=True)

        is_valid = self.change_em_preference('column_icon_rules_reserved_icon_filename_default',icon_filename)
        self.apsw_close()
        if not is_valid:
            return error_dialog(self.gui, _('EM'),_('Selection of a Default EM Reserved Icons Failed.  Nothing Done. [1]'), show=True)

        msg = "New Current EM Default Icon Filename: " +  icon_filename
        msg = msg + "<br><br>This icon will be used for each Entity Type for which you have not specified another icon in Preferences elsewhere in this menu."
        msg = msg + "<br><br>This icon will be used for each Entity (Type + Value) for which you have not specified another icon in Preferences elsewhere in this menu."
        msg = msg + "<br><br>EM will automatically 'remove' <b>any</b> Column Icon Rule with <b>any</b> icon filename starting with '" + reserved_prefix + "' whenever it updates Icon Rules."
        info_dialog(self.gui, 'EM Icon Maintenance',msg).show()
   #-----------------------------------------------------------------------------------------
    def select_default_reserved_icon_column_icon_rules_entity_type(self):

        is_valid = self.apsw_connect_to_library()
        if not is_valid:
            if DEBUG: print("Entities Manager: Fatal Error - apsw_connect_to_library failed.  Try Again.")
            return

        reserved_prefix = self.get_em_preference('column_icon_rules_reserved_icon_filename_prefix')

        try:
            mysql = "SELECT type,'dummy' FROM _entities_type_table_mapping WHERE type NOT LIKE '~%' ORDER BY type "
            self.my_cursor.execute(mysql)
            tmp_rows = self.my_cursor.fetchall()
            type_list = []
            for row in tmp_rows:
                type,dummy = row
                type_list.append(type)
            #END FOR
            del tmp_rows
        except:
            self.apsw_close()
            return error_dialog(self.gui, _('EM'),_('Entities Manager has not yet been initialized.  Select the main menu item.'), show=True)

        mysql = "SELECT type,value FROM _entities_type_preferences WHERE preference = 'column_icon_filename'"
        self.my_cursor.execute(mysql)
        tmp_rows = self.my_cursor.fetchall()
        type_icon_preferences_dict = {}
        n_width = 0
        for row in tmp_rows:
            type,value = row
            type_icon_preferences_dict[type] = value
            if len(type) > n_width:
                n_width = len(type)
        #END FOR
        del tmp_rows

        n_width = n_width + 1
        spacer = "                                                                                                "

        items_list = []

        for type in type_list:
            if type in type_icon_preferences_dict:
                value = type_icon_preferences_dict[type]
            else:  #should not ever happen, but...
                value = self.get_em_preference('column_icon_rules_reserved_icon_filename_default')
            if not value.startswith(reserved_prefix):
                continue
            n = n_width - len(type)
            s = spacer[0:n]
            r = type + s + "Icon: " + value
            items_list.append(r)
        #END FOR

        title = "Select Entity Type to Edit its Default Icon for Column Icon Rules"
        label = "Select Entity Type to Edit its Default Icon for Column Icon Rules"
        selected_row,ok = QInputDialog.getItem(None, title, label, items_list,0,False)
        if not ok:
            self.apsw_close()
            return info_dialog(self.gui, _('EM'),_('Nothing Done.'), show=True)

        s_split = selected_row.split("Icon:")
        type = s_split[0].strip()
        value = s_split[1].strip()

        icon_filename = self.get_user_selected_reserved_icon_filename()
        if icon_filename is None:
            self.apsw_close()
            return info_dialog(self.gui, _('EM'),_('Nothing Done.'), show=True)

        head,icon_filename = os.path.split(icon_filename)
        icon_filename = icon_filename.strip()

        if (not icon_filename.startswith(reserved_prefix)) or (" " in icon_filename) or (not icon_filename.endswith(".png")):
            self.apsw_close()
            msg = "New Default Icon Filename for Entity Type is Invalid. Filename does not start with:  '" + reserved_prefix  + "' or ends with '.png' or contains a space.   Nothing Done."
            return error_dialog(self.gui, _('EM'),_(msg), show=True)

        is_valid = self.check_icon_filename_for_existance(icon_filename)
        if not is_valid:
            msg = "New Default Icon Filename for Entity Type Does Not Exist in the '.../calibre/cc_icons/'    directory.  Nothing Done."
            return error_dialog(self.gui, _('EM'),_(msg), show=True)

        preference = 'column_icon_filename'

        mysql = "INSERT OR REPLACE INTO  _entities_type_preferences (type,preference,value) VALUES (?,?,?) "
        self.my_cursor.execute("begin")
        self.my_cursor.execute(mysql,(type,preference,icon_filename))
        self.my_cursor.execute("commit")

        msg = "Entity Type: " + type + " has been assigned a Default Icon Filename of: " + icon_filename
        info_dialog(self.gui, 'EM Entity Type Default Icon Update',msg).show()

        self.apsw_close()
   #-----------------------------------------------------------------------------------------
    def select_default_reserved_icon_column_icon_rules_entity(self):

        is_valid = self.apsw_connect_to_library()
        if not is_valid:
            if DEBUG: print("Entities Manager: Fatal Error - apsw_connect_to_library failed.  Try Again.")
            return

        reserved_prefix = self.get_em_preference('column_icon_rules_reserved_icon_filename_prefix')

        try:
            mysql = "SELECT type,'dummy' FROM _entities_type_table_mapping WHERE type NOT LIKE '~%' ORDER BY type "
            self.my_cursor.execute(mysql)
            tmp_rows = self.my_cursor.fetchall()
            type_list = []
            for row in tmp_rows:
                type,dummy = row
                type_list.append(type)
            #END FOR
            del tmp_rows
        except:
            self.apsw_close()
            return error_dialog(self.gui, _('EM'),_('Entities Manager has not yet been initialized.  Select the main menu item.'), show=True)

        title = "Select Entity Type & Value to Assign its Default Icon"
        label = "First, Select the Entity's Type"
        type,ok = QInputDialog.getItem(None, title, label,type_list,0,False)
        if not ok:
            self.apsw_close()
            return info_dialog(self.gui, _('EM'),_('Nothing Done.'), show=True)

        self.get_entities_data(connect=False)
        value_list = []
        for row in self.real_entities_list:
            entity_id,entity_type,entity_value = row
            if entity_type == type:
                value_list.append(entity_value)
        #END FOR
        value_list.sort()

        title = "Select Entity Type & Value to Assign its Default Icon"
        label = "Next, Select the Entity Type's Value"
        value,ok = QInputDialog.getItem(None, title, label,value_list,0,False)
        if not ok:
            self.apsw_close()
            return info_dialog(self.gui, _('EM'),_('Nothing Done.'), show=True)

        icon_filename = self.get_user_selected_reserved_icon_filename()
        if icon_filename is None:
            self.apsw_close()
            return info_dialog(self.gui, _('EM'),_('Nothing Done.'), show=True)

        head,icon_filename = os.path.split(icon_filename)
        icon_filename = icon_filename.strip()

        if (not icon_filename.startswith(reserved_prefix)) or (" " in icon_filename) or (not icon_filename.endswith(".png")):
            self.apsw_close()
            msg = "Selected Entity Reserved Icon Default Filename is Invalid. Filename does not start with:  '" + reserved_prefix  + "'   Nothing Done. [0]"
            return error_dialog(self.gui, _('EM'),_(msg), show=True)

        is_valid = self.check_icon_filename_for_existance(icon_filename)
        if not is_valid:
            msg = "New Default Icon Filename for Entity Does Not Exist in the '.../calibre/cc_icons/'    directory.  Nothing Done."
            return error_dialog(self.gui, _('EM'),_(msg), show=True)

        entity_id = self.get_entity_id(type,value)
        if entity_id is None:
            is_valid = self.create_entity(type,value)
            if not is_valid:
                self.apsw_close()
                return error_dialog(self.gui, _('EM'),_('New Entity could not be created.  Try again.'), show=True)
            else:
                was_created = True
        else:
            was_created = False

        preference = 'column_icon_filename'

        mysql = "INSERT OR REPLACE INTO _entities_preferences (entity_id,preference,value) VALUES (?,?,?) "
        self.my_cursor.execute("begin")
        self.my_cursor.execute(mysql,(entity_id,preference,icon_filename))
        self.my_cursor.execute("commit")

        msg = "Entity " + type + " " + value + " has been assigned an icon default of: " + icon_filename
        if was_created:
            msg = msg + "<br><br>Entity did not already exist, so it was created before assigning it this default icon filename."
        info_dialog(self.gui, 'EM Entity Default Icons Assignment',msg).show()

        self.apsw_close()
   #-----------------------------------------------------------------------------------------
    def activate_column_icon_rules_entity_type(self):

        is_valid = self.apsw_connect_to_library()
        if not is_valid:
            if DEBUG: print("Entities Manager: Fatal Error - apsw_connect_to_library failed.  Try Again.")
            return

        try:
            mysql = "SELECT type,'dummy' FROM _entities_type_table_mapping WHERE type NOT LIKE '~%' ORDER BY type "
            self.my_cursor.execute(mysql)
            tmp_rows = self.my_cursor.fetchall()
            type_list = []
            for row in tmp_rows:
                type,dummy = row
                type_list.append(type)
            #END FOR
            del tmp_rows
        except:
            self.apsw_close()
            return error_dialog(self.gui, _('EM'),_('Entities Manager has not yet been initialized.  Select the main menu item.'), show=True)

        mysql = "SELECT type,value FROM _entities_type_preferences WHERE preference = 'add_column_icon_rule'  "
        self.my_cursor.execute(mysql)
        tmp_rows = self.my_cursor.fetchall()
        type_icon_preferences_dict = {}
        for row in tmp_rows:
            type,value = row
            type_icon_preferences_dict[type] = value
        #END FOR
        del tmp_rows

        items_list = []

        for type in type_list:
            if type in type_icon_preferences_dict:
                value = type_icon_preferences_dict[type]
            else:  #should not ever happen, but...
                value = "0"
            if value == "1":
                value_name = "True"
            else:
                value_name = "False"
            r = type + "      Active?    " + value_name
            items_list.append(r)
        #END FOR

        title = "Select Entity Type to Edit its Icons Rule Activation"
        label = ""
        selected_row,ok = QInputDialog.getItem(None, title, label, items_list,0,False)
        if not ok:
            self.apsw_close()
            return info_dialog(self.gui, _('EM'),_('Nothing Done.'), show=True)

        s_split = selected_row.split("Active?")
        type = s_split[0].strip()

        answer_list = []
        answer_list.append("True")
        answer_list.append("False")

        title = "Specify 'Column Icons Rules' Status for Chosen Entity Type"
        label = "Specify 'Column Icons Rules' Status for Chosen Entity Type.  Active is True.  Inactive is False.                                                      "
        answer,ok = QInputDialog.getItem(None, title, label, answer_list,0,False)
        if not ok:
            self.apsw_close()
            return info_dialog(self.gui, _('EM'),_('Nothing Done.'), show=True)

        if answer == "True":
            value = "1"
        else:
            value = "0"

        preference = 'add_column_icon_rule'

        mysql = "INSERT OR REPLACE INTO  _entities_type_preferences (type,preference,value) VALUES (?,?,?) "
        self.my_cursor.execute("begin")
        self.my_cursor.execute(mysql,(type,preference,value))
        self.my_cursor.execute("commit")

        msg = "Entity Type: " + type + " is Active:  " + answer
        info_dialog(self.gui, 'EM Entity Type Column Icons Rule Activate',msg).show()

        self.apsw_close()
   #-----------------------------------------------------------------------------------------
    def restrict_link_note_column_icon_rules_entity_type(self):

        is_valid = self.apsw_connect_to_library()
        if not is_valid:
            if DEBUG: print("Entities Manager: Fatal Error - apsw_connect_to_library failed.  Try Again.")
            return

        try:
            mysql = "SELECT type,'dummy' FROM _entities_type_table_mapping WHERE type NOT LIKE '~%' ORDER BY type "
            self.my_cursor.execute(mysql)
            tmp_rows = self.my_cursor.fetchall()
            type_list = []
            for row in tmp_rows:
                type,dummy = row
                type_list.append(type)
            #END FOR
            del tmp_rows
        except:
            self.apsw_close()
            return error_dialog(self.gui, _('EM'),_('Entities Manager has not yet been initialized.  Select the main menu item.'), show=True)

        mysql = "SELECT type,value FROM _entities_type_preferences WHERE preference = 'icon_if_link_note'  "
        self.my_cursor.execute(mysql)
        tmp_rows = self.my_cursor.fetchall()
        type_icon_preferences_dict = {}
        for row in tmp_rows:
            type,value = row
            type_icon_preferences_dict[type] = value
        #END FOR
        del tmp_rows

        items_list = []

        for type in type_list:
            if type in type_icon_preferences_dict:
                value = type_icon_preferences_dict[type]
            else:  #should not ever happen, but...
                value = "0"
            if value == "1":
                value_name = "True"
            else:
                value_name = "False"
            r = type + "      Active?    " + value_name
            items_list.append(r)
        #END FOR

        title = "Select Entity Type to Edit its Icons 'Only-if-Link-or-Note' Activation"
        label = "Select Entity Type to Edit its Icons 'Only-if-Link-or-Note' Activation"
        selected_row,ok = QInputDialog.getItem(None, title, label, items_list,0,False)
        if not ok:
            self.apsw_close()
            return info_dialog(self.gui, _('EM'),_('Nothing Done.'), show=True)

        s_split = selected_row.split("Active?")
        type = s_split[0].strip()

        answer_list = []
        answer_list.append("True")
        answer_list.append("False")

        title = "Specify 'Only-if-Link-or-Note' Status for Chosen Entity Type"
        label = "Active is True.  Inactive is False.                                                      "
        answer,ok = QInputDialog.getItem(None, title, label, answer_list,0,False)
        if not ok:
            self.apsw_close()
            return info_dialog(self.gui, _('EM'),_('Nothing Done.'), show=True)

        if answer == "True":
            value = "1"
        else:
            value = "0"

        preference = 'icon_if_link_note'

        mysql = "INSERT OR REPLACE INTO  _entities_type_preferences (type,preference,value) VALUES (?,?,?) "
        self.my_cursor.execute("begin")
        self.my_cursor.execute(mysql,(type,preference,value))
        self.my_cursor.execute("commit")

        msg = "Entity Type: " + type + " is Active:  " + answer
        info_dialog(self.gui, "EM Entity Type 'Only-if-Link-or-Note' Activate",msg).show()

        self.apsw_close()
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #~ COMMON
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def create_entity(self,type=None,value=None):
        if type is None or value is None:
            return False

        mysql = "INSERT OR IGNORE INTO _entities(entity_id,type,value) VALUES (null,?,?)"
        self.my_cursor.execute("begin")
        self.my_cursor.execute(mysql,(type,value))
        self.my_cursor.execute("commit")
        self.apsw_close()

        msg = "Entity created for:  " + type + " - " + value
        self.gui.status_bar.show_message(_(msg), 10000)
        return True
    #-----------------------------------------------------------------------------------------
    def get_table_custom_columns(self):

        self.custom_column_label_dict = {}
        self.custom_column_label_list = []

        try:
            mysql = "SELECT id,label,name,datatype,display,is_multiple,normalized FROM custom_columns"
            self.my_cursor.execute(mysql)
            tmp_rows = self.my_cursor.fetchall()
            if tmp_rows is None:
                tmp_rows = []
            for row in tmp_rows:
                id,label,name,datatype,display,is_multiple,normalized = row
                if datatype != "composite" and datatype != "ratings" and datatype != "int" and datatype != "float" and datatype != "bool" and datatype != "datetime":
                    if datatype == "comments":
                        if not '"interpret_as": "short-text"' in display:
                            continue
                    label = "#" + label
                    id = as_unicode(id)
                    r = id,label,name,datatype,display,is_multiple,normalized
                    self.custom_column_label_dict[label] = r
                    self.custom_column_label_list.append(label)
            #END FOR
            self.custom_column_label_list.sort()
            del tmp_rows
        except Exception as e:
            if DEBUG: print("Error in get_table_custom_columns: ", as_unicode(e))
    #-----------------------------------------------------------------------------------------
    def get_entities_data(self,connect=True):

        if connect:
            is_valid = self.apsw_connect_to_library()
            if not is_valid:
                if DEBUG: print("Entities Manager: Fatal Error - apsw_connect_to_library failed.  Try Again.")
                return False

        self.get_table_custom_columns()

        self.entity_column_dict = {}

        for type in self.custom_column_label_list:
            id,label,name,datatype,display,is_multiple,normalized = self.custom_column_label_dict[type]
            self.entity_column_dict[type] = name,is_multiple
        #END FOR

        self.entity_column_dict['Author'] = 'Authors',1
        self.entity_column_dict['Authors'] = 'Authors',1
        self.entity_column_dict['Publisher'] = 'Publisher',0
        self.entity_column_dict['Series'] = 'Series',0
        self.entity_column_dict['Tags'] = 'Tags',1
        self.entity_column_dict['Title'] = 'Title',0

        try:
            mysql = "SELECT entity_id,type,value FROM _entities WHERE type NOT LIKE '~%' "
            self.my_cursor.execute(mysql)
            tmp_rows = self.my_cursor.fetchall()
            if connect:
                self.apsw_close()
        except:  #no em tables exist yet...
            if connect:
                self.apsw_close()
            return False

        if not tmp_rows:
            return False
        if len(tmp_rows) == 0:
            return False

        self.real_entities_list = []
        for row in tmp_rows:
            #~ if DEBUG: print(as_unicode(row))
            self.real_entities_list.append(row)
        #END FOR
        del tmp_rows
        #~ if DEBUG: print("Number of Entities for which to add column icon/color rules: ", as_unicode(len(self.real_entities_list)))
        return True
    #-----------------------------------------------------------------------------------------
    def export_all_rules_to_backup_dir(self,edit_rules_widget,rule_type=None):

        is_valid = self.apsw_connect_to_library()
        if not is_valid:
            if DEBUG: print("Entities Manager: Fatal Error - apsw_connect_to_library failed.  Try Again.")
            return

        if rule_type == "icon":
            type = "column_icon_rules"
            active = self.get_em_preference("automatic_icon_rules_backup_is_active")
            if active == "0":
                self.apsw_close()
                return False
        elif rule_type == "color":
            type = "column_color_rules"
            active = self.get_em_preference("automatic_color_rules_backup_is_active")
            if active == "0":
                self.apsw_close()
                return False
        else:
            self.apsw_close()
            return False

        self.apsw_close()

        file_dir = os.path.join(config_dir,"plugins")
        file_dir = file_dir.replace("\\","/")
        file_dir = os.path.join(file_dir,"em_rules_backup")
        file_dir = file_dir.replace("\\","/")
        if not os.path.isdir(file_dir):
            os.mkdir(file_dir)
        if not os.path.isdir(file_dir):
            return False

        from datetime import datetime
        now = unicode_type(datetime.now())
        now = now[0:19].strip()
        now = now.replace("-","_")
        now = now.replace(":","_")
        now = now.replace(".","_")
        now = now.replace(" ","_")

        head,tail = os.path.split(self.library_path)

        fname = tail + "_" + rule_type + "_" + now + ".rules"

        file_path = os.path.join(file_dir,fname)
        file_path = file_path.replace("\\","/")

        if DEBUG: print("export file: ", file_path)

        rules = {
                'version': edit_rules_widget.model.EXIM_VERSION,
                'type': type,
                'rules': edit_rules_widget.model.rules_as_list(for_export=True)
                    }

        with lopen(file_path, 'wb') as f:
            f.write(json_dumps(rules, indent=2))

        type = type.title()

        self.info_msg = "Original Column " + type + " Rules have been backed up to file: " + file_path

        return True
    #-----------------------------------------------------------------------------------------
    def get_user_selected_reserved_icon_filename(self):
        file_dir = os.path.join(config_dir,"cc_icons")
        file_dir = file_dir.replace("\\","/")
        if not os.path.isdir(file_dir):
            return None
        import_tuple = QFileDialog.getOpenFileName(None,"",file_dir,("Entity Reserved Icon Files(entity_reserved*.png)") )
        if not import_tuple:
            return None
        path, dummy = import_tuple
        if not path:
            return None
        if isbytestring(path):
            path = path.decode(filesystem_encoding)
        path = path.replace(os.sep, '/')
        if os.path.isfile(path):
            return path
        else:
            return None
    #-----------------------------------------------------------------------------------------
    def check_icon_filename_for_existance(self,icon_filename):
        file_dir = os.path.join(config_dir,"cc_icons")
        file_dir = file_dir.replace("\\","/")
        if not os.path.isdir(file_dir):
            return False
        path = os.path.join(file_dir,icon_filename)
        path = path.replace("\\","/")
        #~ if DEBUG: print("check icon filename path: ", path)
        if not os.path.isfile(path):
            return False
        else:
            return True
    #-----------------------------------------------------------------------------------------
    def apsw_connect_to_library(self):

        try:
            self.my_db.close()
        except:
            pass

        if isbytestring(self.library_path):
            self.library_path = self.library_path.decode(filesystem_encoding)
        self.library_path = self.library_path.replace(os.sep, '/')

        path = os.path.join(self.library_path, 'metadata.db')
        path = path.replace(os.sep, '/')

        if isbytestring(path):
            path = path.decode(filesystem_encoding)

        if path.endswith("/"):
            path = path[0:-1]

        self.library_metadatadb_path = path

        try:
            self.my_db = apsw.Connection(path)
            is_valid = True
        except Exception as e:
            if DEBUG: print("path to metadata.db is: ", path)
            if DEBUG: print("error: ", as_unicode(e))
            is_valid = False
            return is_valid

        self.my_cursor = self.my_db.cursor()

        mysql = "PRAGMA main.busy_timeout = 25000;"      #milliseconds;
        self.my_cursor.execute(mysql)

        return is_valid
   #-----------------------------------------------------------------------------------------
    def apsw_close(self):
        self.my_db.close()
        self.my_cursor = None
        self.my_db = None
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #~ COLUMN COLOR RULES
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def add_column_color_rules(self):
        self.maintain_column_color_rules(mode=MODE_ADD)
    #-----------------------------------------------------------------------------------------
    def remove_column_color_rules(self):
        self.maintain_column_color_rules(mode=MODE_REMOVE)
    #-----------------------------------------------------------------------------------------
    def maintain_column_color_rules(self,mode=MODE_ADD):
        is_valid = self.apsw_connect_to_library()
        if not is_valid:
            return error_dialog(self.gui, _('EM'),_('Entities Manager could not connect to the database.  Please try again.'), show=True)
        self.entity_reserved_color_list = self.get_em_preference('unique_column_color_rules_reserved_color_list')
        self.entity_reserved_color_default = self.get_em_preference('unique_column_color_rules_reserved_color_default')
        self.entity_type_preference_add_column_color_rule_dict = self.get_entity_type_preference_dict('add_column_color_rule')
        self.entity_type_preference_column_color_code_dict = self.get_entity_type_preference_dict('column_color_code')
        self.entity_type_preference_color_if_link_note_dict = self.get_entity_type_preference_dict('color_if_link_note')
        self.entity_id_preference_dict = self.get_entity_id_preference_dict('column_color_code')
        self.entity_id_link_list = self.get_entity_links(entity_id=None,connect=False)
        self.entity_id_link_set = set(self.entity_id_link_list)
        self.entity_id_note_list = self.get_entity_notes(entity_id=None,connect=False)
        self.entity_id_note_set = set(self.entity_id_note_list)
        self.apsw_close()

        #~ if DEBUG: print("link set: ",  as_unicode(self.entity_id_link_set))
        #~ if DEBUG: print("note set", as_unicode(self.entity_id_note_set))

        is_valid = self.get_entities_data(connect=True)
        if not is_valid:
            return error_dialog(self.gui,'EM','New Entity Column Color Rules Were Not Created.  Existing Entities Could Not Be Retrieved Error. [1]', show=True)
        is_valid = self.add_new_entity_column_color_rules(kind="color",prefname = 'column_color_rules',rule_type="color",color=self.entity_reserved_color_default,mode=mode)
        #~ self.apsw_close()
        if not is_valid:
            return error_dialog(self.gui,'EM','New Entity Column Color Rules Were Not Created.  Rule Creation Error. [2]', show=True)
        else:
            return info_dialog(self.gui, 'EM Maintain Entity Column Color Rules',self.info_msg, show=True)
    #-----------------------------------------------------------------------------------------
    def add_new_entity_column_color_rules(self,kind=None,prefname=None,rule_type=None,color=None,mode=None):
        if kind is None or prefname is None or rule_type is None or color is None or mode is None:
            if DEBUG: print("ERROR:  if kind is None or prefname is None or rule_type is None or color is None or mode is None:")
            return False
        #-------------------------------
        #-------------------------------
        self.rule_kind = kind
        self.prefname = prefname
        self.rule_type = rule_type
        self.color = color

        self.db = self.gui.library_view.model().db
        self.fm = self.db.field_metadata
        #-------------------------------
        from calibre.gui2.preferences.coloring import ConditionEditor, RuleEditor, RulesModel, EditRules
        from calibre.library.coloring import Rule
        #-------------------------------
        edit_rules_widget = EditRules()  #has the export button and the add rule button...
        edit_rules_widget.initialize(self.fm, self.db.prefs, None, 'column_color_rules')   #creates the model that gets all of the prefs rules
        edit_rules_widget.show()
        edit_rules_widget.hide()
        #~ edit_rules_widget.pref_name = 'column_color_rules'

        #-------------------------------
        is_valid = self.export_all_rules_to_backup_dir(edit_rules_widget,rule_type="color")
        if not is_valid:
            edit_rules_widget.close()
            return False
        #-------------------------------

        n1 = len(edit_rules_widget.model.rules)
        total_original_rules = as_unicode(n1)
        #~ if DEBUG: print("number of original rows in edit_rules_widget.model.rules: ", as_unicode(n1))
        total_deleted = 0
        for g in range(0,n1):
            n2 = len(edit_rules_widget.model.rules)
            if n2 == 0:
                break
            for row_number in range(0,n2):
                try:
                    rule = edit_rules_widget.model.rules[row_number]
                    x,y,robject = rule
                    r = robject.template  #what you see in an export file
                    delete = self.check_color_code_for_deletion(r)
                    if delete:
                        index = edit_rules_widget.model.index(row_number)
                        edit_rules_widget.model.remove_rule(index)
                        total_deleted = total_deleted + 1
                        #~ if DEBUG: print("row_number deleted: ", as_unicode(row_number))
                    del rule
                    del r
                    del sr
                except:
                    continue  #removing rows...
                if len(edit_rules_widget.model.rules) == 0:
                    break
            #END FOR
        #END FOR
        #~ if DEBUG: print("-----------number of color rules deleted: ", as_unicode(total_deleted))
        n = len(edit_rules_widget.model.rules)
        #~ if DEBUG: print("number of original rows in edit_rules_widget.model.rules AFTER REMOVALS of all prior EM column color rules: ", as_unicode(n))
        #-------------------------------
        is_valid = self.apsw_connect_to_library()
        if not is_valid:
            if DEBUG: print("NOT is_valid = self.apsw_connect_to_library()")
            return False
        #-------------------------------
        rule_editor_dialog = RuleEditor(self.fm, self.prefname)   #'column_color_rules'
        rule_editor_dialog.show()
        rule_editor_dialog.hide()
        rule_editor_dialog.rule_kind = self.rule_kind
        #-------------------------------
        rule_editor_dialog.conditions = []
        rule_editor_dialog.add_blank_condition()
        #-------------------------------
        total_entities = as_unicode(len(self.real_entities_list))
        total_entities_excluded = 0
        duplicates_checking_set = set()
        total_added = 0
        total_entities_skipped = 0  #due to user customization
        color_to_use = None

        if mode != MODE_REMOVE:

            for row in self.real_entities_list:
                entity_id,type,value = row
                #~ if DEBUG: print(as_unicode(row))

                if type in self.entity_type_preference_add_column_color_rule_dict:
                    active = self.entity_type_preference_add_column_color_rule_dict[type]
                    if active == "0":
                        total_entities_skipped = total_entities_skipped + 1
                        continue

                if type in self.entity_type_preference_color_if_link_note_dict:
                    active = self.entity_type_preference_color_if_link_note_dict[type]
                    if active == "0":  #does not matter to the user if exists or not
                        pass
                    else:
                        if (entity_id in self.entity_id_link_set) or (entity_id in self.entity_id_note_set):
                            pass
                        else:
                            total_entities_skipped = total_entities_skipped + 1
                            continue
                else:
                    total_entities_skipped = total_entities_skipped + 1
                    continue

                if not value > " ":
                    total_entities_excluded = total_entities_excluded + 1
                    continue

                if '"' in value:
                    total_entities_excluded = total_entities_excluded + 1
                    continue  # double-quote " in value causes future template parsing errors not visible to the user unless in debug...

                if type in self.entity_column_dict:
                    current_column,is_multiple = self.entity_column_dict[type]
                    current_column = current_column.lower()
                    if type.startswith("#"):
                        current_column = type
                    #~ if DEBUG: print("current_column: ", type)
                else:
                    if DEBUG: print("........................Fatal Error: type NOT in self.entity_column_dict: ",type)
                    return False

                if is_multiple:
                    action = "has"
                else:
                    action = "is"
                c = (current_column,action,value)      # ('tags', 'has', value)
                if c in duplicates_checking_set:
                    #~ if DEBUG: print("new duplicate rule being skipped: ", as_unicode(c))
                    continue
                else:
                    duplicates_checking_set.add(c)
                ruleobject = Rule(edit_rules_widget.model.fm)  # SIGNATURE = '# BasicColorRule():'
                ruleobject.add_condition(current_column, action, value)  # ('tags', 'has', value)

                color_to_use = self.entity_reserved_color_default
                if type in self.entity_type_preference_column_color_code_dict:
                    color_to_use = self.entity_type_preference_column_color_code_dict[type]
                if entity_id in self.entity_id_preference_dict:
                    color_to_use = self.entity_id_preference_dict[entity_id]

                ruleobject.color = color_to_use

                rule_editor_dialog.column_box.setCurrentText(current_column)
                edit_rules_widget.pref_name = self.prefname
                edit_rules_widget.model.pref_name = self.prefname
                rule_index = edit_rules_widget.model.add_rule(rule_editor_dialog.rule_kind, current_column, ruleobject)
                edit_rules_widget.changed.emit()
                rule_editor_dialog.apply_rule(self.rule_type, current_column, ruleobject)
                rule_editor_dialog.accept()
                del ruleobject
                total_added = total_added + 1
            #END FOR
            #~ if DEBUG: print("Number of EM column color rules maintained: ", as_unicode(n_added))
        else:
            pass
        #-------------------------------
        #-------------------------------
        if len(edit_rules_widget.model.rules) > 0 or total_deleted > 0:
            edit_rules_widget.model.commit(self.db.prefs)
        #-------------------------------
        total_rules = as_unicode(len(edit_rules_widget.model.rules))
        #~ if DEBUG: print("Grand Total Rules now in Preferences > Look & Feel > Column Color Rules: ", total_rules )
        #~ if DEBUG: print("total_entities_skipped: ", as_unicode(total_entities_skipped))
        #-------------------------------
        edit_rules_widget.close()
        rule_editor_dialog.close()
        #-------------------------------
        #-------------------------------
        m = self.maingui.library_view.model()   #some columns are not refreshing their colors after just above completes...this completely fixes that.
        m.beginResetModel(), m.endResetModel()
        self.maingui.library_view.refresh_grid()
        #-------------------------------
        if mode == MODE_ADD:
            self.info_msg = self.info_msg + "<br><br>MODE:  'Add/Refresh' "
            self.info_msg = self.info_msg + "<br><br>Total Entities in EM: " + total_entities
            self.info_msg = self.info_msg + "<br><br>Total Entities in EM excluded due to unusable Values: " + as_unicode(total_entities_excluded)
            total_eligible = int(total_entities) - int(total_entities_excluded)
            self.info_msg = self.info_msg + "<br><br>Total Eligible Entities in EM: " + as_unicode(total_eligible)
            self.info_msg = self.info_msg + "<br><br>Total Eligible Entities in EM excluded per User Customization: " + as_unicode(total_entities_skipped)
            self.info_msg = self.info_msg + "<br><br>Final Eligible Entities in EM: " +  as_unicode(int(total_entities) - int(total_entities_excluded) - total_entities_skipped)
            self.info_msg = self.info_msg + "<br><br>Total Original Rules <b>Before</b> Entities Updates: " + total_original_rules
            self.info_msg = self.info_msg + "<br><br>Total Old EM Rules Deleted: " + as_unicode(total_deleted)
            self.info_msg = self.info_msg + "<br><br>Total New EM Rules Added: " + as_unicode(total_added)
            total_manual_rules = int(total_rules) - int(total_entities) + int(total_entities_excluded) + total_entities_skipped
            self.info_msg = self.info_msg + "<br><br>Total Manually-Created Rules: " + as_unicode(total_manual_rules)
            self.info_msg = self.info_msg + "<br><br>Total Rules <b>After</b> Entities Updates: " + total_rules
            self.info_msg = self.info_msg + "<br>"
        elif mode == MODE_REMOVE:
            self.info_msg = self.info_msg + "<br><br>MODE:  'Remove' "
            self.info_msg = self.info_msg + "<br><br>Total Original Rules <b>Before</b> Entities Updates: " + total_original_rules
            self.info_msg = self.info_msg + "<br><br>Total Manually-Created Rules: " + total_rules
            self.info_msg = self.info_msg + "<br><br>Total Rules <b>After</b> Entities Updates: " + total_rules
            self.info_msg = self.info_msg + "<br>"

        del edit_rules_widget
        del rule_editor_dialog
        del self.db
        del self.fm
        del self.real_entities_list
        del self.color
        del color_to_use
        del self.prefname
        del self.rule_type
        del total_original_rules
        del total_rules
        total_eligible = None
        del total_eligible
        del total_entities_skipped
        total_manual_rules = None
        del total_manual_rules
        del total_entities_excluded
        del total_entities
        del self.entity_type_preference_add_column_color_rule_dict
        del self.entity_type_preference_column_color_code_dict
        del self.entity_type_preference_color_if_link_note_dict
        del self.entity_id_preference_dict
        del self.entity_id_link_list
        del self.entity_id_link_set
        del self.entity_id_note_list
        del self.entity_id_note_set

        return True
   #-----------------------------------------------------------------------------------------
    def check_color_code_for_deletion(self,template):
        delete = False
        for color in self.entity_reserved_color_list:
            if (color in template) or (as_unicode(color) in as_unicode(template)) :
                delete = True
                break
        #END FOR
        return delete
   #-----------------------------------------------------------------------------------------
   #-----------------------------------------------------------------------------------------
    def add_new_unique_color_to_reserved_color_list(self):

        is_valid = self.apsw_connect_to_library()
        if not is_valid:
            if DEBUG: print("Entities Manager: Fatal Error - apsw_connect_to_library failed.  Try Again.")
            return

        #~ ------------------------
        #automatically validate and repair the list (on general principles only)
        current_list = self.get_em_preference('unique_column_color_rules_reserved_color_list')
        tmp = []
        for color in current_list:
            if (color is not None) and (color != ""):
                if (not color.startswith("#")) or (not len(color) == 7) or (" " in color):
                    continue
            else:
                continue
            if color.startswith("#"):
                tmp.append(color.lower())
        #END FOR
        del current_list
        current_list = tmp
        s = as_unicode(current_list)
        is_valid = self.change_em_preference('unique_column_color_rules_reserved_color_list',current_list)
        #~ ------------------------

        col = QColorDialog.getColor(Qt.white,None)
        if col.isValid():
            new_color = unicode_type(col.name())
        else:
            new_color = None

        if new_color is None:
            new_color = "ERROR"
        if new_color == "":
            new_color = "ERROR"
        new_color = new_color.strip()
        if (not new_color.startswith("#")) or (not len(new_color) == 7) or (" " in new_color):
            self.apsw_close()
            return error_dialog(self.gui, _('EM'),_('Invalid HTML 7-Character Color.  Nothing Done.'), show=True)

        msg = "<b>Permanently</b> add color " + '<span style="color:' + new_color + ';">' + new_color + "</span> to the EM Reserved Colors List?<br><br>This cannot be undone.<br><br>Are you sure?"
        if question_dialog(self.gui, 'EM',msg):
            pass
        else:
            self.apsw_close()
            return info_dialog(self.gui, _('EM'),_('Nothing Done.'), show=True)


        current_list = self.get_em_preference('unique_column_color_rules_reserved_color_list')

        if not new_color in current_list:
            current_list.append(new_color)

        s = as_unicode(current_list)
        is_valid = self.change_em_preference('unique_column_color_rules_reserved_color_list',current_list)
        if not is_valid:
            self.apsw_close()
            return error_dialog(self.gui, _('EM'),_('Addition of a Color to the EM Reserved Color List failed.  Nothing Done.'), show=True)
        current_list = self.get_em_preference('unique_column_color_rules_reserved_color_list')

        msg = "New Available Color: " +  '<span style="color:' + new_color + ';">' + new_color + "</span><br>"
        msg = msg + "All available permanent (not removable) Color Codes: "
        for color in current_list:
            if color is not None:
                msg = msg + "<br>" + color
        #END FOR
        msg = msg + "<br>EM will automatically 'remove' <b>any</b> Column Color rule with <b>any</b> of the above colors whenever it updates Color Rules."
        return info_dialog(self.gui, 'EM Unique Color Maintenance',msg).show()
   #-----------------------------------------------------------------------------------------
    def select_default_unique_color_column_color_rules_em(self):
        is_valid = self.apsw_connect_to_library()
        if not is_valid:
            if DEBUG: print("Entities Manager: Fatal Error - apsw_connect_to_library failed.  Try Again.")
            return

        current_default = self.get_em_preference('unique_column_color_rules_default_reserved_color')
        current_list = self.get_em_preference('unique_column_color_rules_reserved_color_list')

        if not current_default in current_list:
            current_list.append(current_default)
            s = as_unicode(current_list)
            is_valid = self.change_em_preference('unique_column_color_rules_reserved_color_list',s)
            if not is_valid:
                self.apsw_close()
                return error_dialog(self.gui, _('EM'),_('Selection of a Default EM Reserved Color Failed.  Nothing Done. [0]'), show=True)

        msg = "All available permanent (not removable) Color Codes from which to choose:<br>"
        for color in current_list:
            if color is not None:
                msg = msg + '<span style="color:' + color + ';">' + color + "</span><br>"
        #END FOR
        info_dialog(self.gui, 'EM Unique Color Maintenance',msg).show()

        title = "Select New EM Default Color for EM 'Column Color Rules'"
        label = title
        selected_color,ok = QInputDialog.getItem(None, title, label, current_list,0,False)
        if not ok:
            self.apsw_close()
            return info_dialog(self.gui, _('EM'),_('Nothing Done.'), show=True)

        is_valid = self.change_em_preference('unique_column_color_rules_default_reserved_color',selected_color)
        self.apsw_close()
        if not is_valid:
            return error_dialog(self.gui, _('EM'),_('Selection of a Default EM Reserved Color Failed.  Nothing Done. [1]'), show=True)

        msg = "New Current Default Color: " +  '<span style="color:' + selected_color + ';">' + selected_color + "</span><br><br>"
        msg = msg + "This color will be used for each Entity Type for which you have not previously specified another Unique Color in Preferences.  "
        msg = msg + "This color will be used for each Entity (Type + Value) for which you have not previously specified another Unique Color in Preferences.  "
        msg = msg + "All available permanent (not removable) Color Codes:<br>"
        for color in current_list:
            if color is not None:
                msg = msg + '<span style="color:' + color + ';">' + color + "</span><br>"
        #END FOR
        msg = msg + "<br><br>EM will automatically 'remove' <b>any</b> Column Color rule with <b>any</b> of the above colors whenever it updates Color Rules."
        info_dialog(self.gui, 'EM Unique Color Maintenance',msg).show()
   #-----------------------------------------------------------------------------------------
    def select_default_unique_color_column_color_rules_entity_type(self):

        is_valid = self.apsw_connect_to_library()
        if not is_valid:
            if DEBUG: print("Entities Manager: Fatal Error - apsw_connect_to_library failed.  Try Again.")
            return

        try:
            mysql = "SELECT type,'dummy' FROM _entities_type_table_mapping WHERE type NOT LIKE '~%' ORDER BY type "
            self.my_cursor.execute(mysql)
            tmp_rows = self.my_cursor.fetchall()
            type_list = []
            for row in tmp_rows:
                type,dummy = row
                type_list.append(type)
            #END FOR
            del tmp_rows
        except:
            self.apsw_close()
            return error_dialog(self.gui, _('EM'),_('Entities Manager has not yet been initialized.  Select the main menu item.'), show=True)

        mysql = "SELECT type,value FROM _entities_type_preferences WHERE preference = 'column_color_code'"
        self.my_cursor.execute(mysql)
        tmp_rows = self.my_cursor.fetchall()
        type_color_preferences_dict = {}
        for row in tmp_rows:
            type,value = row
            type_color_preferences_dict[type] = value
        #END FOR
        del tmp_rows

        items_list = []

        for type in type_list:
            if type in type_color_preferences_dict:
                value = type_color_preferences_dict[type]
            else:  #should not ever happen, but...
                value = self.get_em_preference('unique_column_color_rules_default_reserved_color')
            r = type + " Color Code: " + value
            items_list.append(r)
        #END FOR

        title = "Select Entity Type to Edit its Default Color"
        label = ""
        selected_row,ok = QInputDialog.getItem(None, title, label, items_list,0,False)
        if not ok:
            self.apsw_close()
            return info_dialog(self.gui, _('EM'),_('Nothing Done.'), show=True)

        s_split = selected_row.split("Color Code:")
        type = s_split[0].strip()
        value = s_split[1].strip()

        current_default = self.get_em_preference('unique_column_color_rules_default_reserved_color')
        current_list = self.get_em_preference('unique_column_color_rules_reserved_color_list')

        current_list.sort()

        msg = "All available permanent (not removable) Color Codes from which to choose:<br>"
        for color in current_list:
            if color is not None:
                msg = msg + '<span style="color:' + color + ';">' + color + "</span><br>"
        #END FOR
        info_dialog(self.gui, 'EM Unique Color Maintenance',msg).show()

        title = "Select Default Color for Chosen Entity Type"
        label = "Chosen Entity Type: " + type
        value,ok = QInputDialog.getItem(None, title, label, current_list,0,False)
        if not ok:
            self.apsw_close()
            return info_dialog(self.gui, _('EM'),_('Nothing Done.'), show=True)

        preference = 'column_color_code'

        mysql = "INSERT OR REPLACE INTO  _entities_type_preferences (type,preference,value) VALUES (?,?,?) "
        self.my_cursor.execute("begin")
        self.my_cursor.execute(mysql,(type,preference,value.strip()))
        self.my_cursor.execute("commit")

        msg = "Entity Type: " + type + " has been assigned a color code default of: " + '<span style="color:' + value + ';">' + value + "</span>"
        info_dialog(self.gui, 'EM Entity Type Default Color Update',msg).show()

        self.apsw_close()
   #-----------------------------------------------------------------------------------------
    def select_default_unique_color_column_color_rules_entity(self):

        is_valid = self.apsw_connect_to_library()
        if not is_valid:
            if DEBUG: print("Entities Manager: Fatal Error - apsw_connect_to_library failed.  Try Again.")
            return

        try:
            mysql = "SELECT type,'dummy' FROM _entities_type_table_mapping WHERE type NOT LIKE '~%' ORDER BY type "
            self.my_cursor.execute(mysql)
            tmp_rows = self.my_cursor.fetchall()
            type_list = []
            for row in tmp_rows:
                type,dummy = row
                type_list.append(type)
            #END FOR
            del tmp_rows
        except:
            self.apsw_close()
            return error_dialog(self.gui, _('EM'),_('Entities Manager has not yet been initialized.  Select the main menu item.'), show=True)

        title = "Select Entity Type & Value to Assign its Default Color"
        label = "First, Select the Entity's Type"
        type,ok = QInputDialog.getItem(None, title, label,type_list,0,False)
        if not ok:
            self.apsw_close()
            return info_dialog(self.gui, _('EM'),_('Nothing Done.'), show=True)

        self.get_entities_data(connect=False)
        value_list = []
        for row in self.real_entities_list:
            entity_id,entity_type,entity_value = row
            if entity_type == type:
                value_list.append(entity_value)
        #END FOR
        value_list.sort()

        title = "Select Entity Type & Value to Assign its Default Color"
        label = "Next, Select the Entity Type's Value"
        value,ok = QInputDialog.getItem(None, title, label,value_list,0,False)
        if not ok:
            self.apsw_close()
            return info_dialog(self.gui, _('EM'),_('Nothing Done.'), show=True)

        current_list = self.get_em_preference('unique_column_color_rules_reserved_color_list')

        msg = "All available permanent (not removable) Color Codes from which to choose:<br>"
        for color in current_list:
            if color is not None:
                msg = msg + '<span style="color:' + color + ';">' + color + "</span><br>"
        #END FOR
        info_dialog(self.gui, 'EM Unique Color Maintenance',msg).show()

        title = "Select Default Color for Chosen Entity"
        label = "Chosen Entity: " + type + " " + value
        color,ok = QInputDialog.getItem(None, title, label, current_list,0,False)
        if not ok:
            self.apsw_close()
            return info_dialog(self.gui, _('EM'),_('Nothing Done.'), show=True)

        entity_id = self.get_entity_id(type,value)
        if entity_id is None:
            is_valid = self.create_entity(type,value)
            if not is_valid:
                self.apsw_close()
                return error_dialog(self.gui, _('EM'),_('New Entity could not be created.  Try again.'), show=True)
            else:
                was_created = True
        else:
            was_created = False

        preference = 'column_color_code'

        mysql = "INSERT OR REPLACE INTO _entities_preferences (entity_id,preference,value) VALUES (?,?,?) "
        self.my_cursor.execute("begin")
        self.my_cursor.execute(mysql,(entity_id,preference,color))
        self.my_cursor.execute("commit")

        msg = "Entity " + type + " " + value + " has been assigned a color code default of: " + '<span style="color:' + color + ';">' + color + "</span>"
        if was_created:
            msg = msg + "<br><br>Entity did not already exist, so it was created before assigning it this default color code."
        info_dialog(self.gui, 'EM Entity Default Color Assignment',msg).show()

        self.apsw_close()
   #-----------------------------------------------------------------------------------------
    def activate_column_color_rules_entity_type(self):

        is_valid = self.apsw_connect_to_library()
        if not is_valid:
            if DEBUG: print("Entities Manager: Fatal Error - apsw_connect_to_library failed.  Try Again.")
            return

        try:
            mysql = "SELECT type,'dummy' FROM _entities_type_table_mapping WHERE type NOT LIKE '~%' ORDER BY type "
            self.my_cursor.execute(mysql)
            tmp_rows = self.my_cursor.fetchall()
            type_list = []
            for row in tmp_rows:
                type,dummy = row
                type_list.append(type)
            #END FOR
            del tmp_rows
        except:
            self.apsw_close()
            return error_dialog(self.gui, _('EM'),_('Entities Manager has not yet been initialized.  Select the main menu item.'), show=True)

        mysql = "SELECT type,value FROM _entities_type_preferences WHERE preference = 'add_column_color_rule'  "
        self.my_cursor.execute(mysql)
        tmp_rows = self.my_cursor.fetchall()
        type_color_preferences_dict = {}
        for row in tmp_rows:
            type,value = row
            type_color_preferences_dict[type] = value
        #END FOR
        del tmp_rows

        items_list = []

        for type in type_list:
            if type in type_color_preferences_dict:
                value = type_color_preferences_dict[type]
            else:  #should not ever happen, but...
                value = "0"
            if value == "1":
                value_name = "True"
            else:
                value_name = "False"
            r = type + "      Active?    " + value_name
            items_list.append(r)
        #END FOR

        title = "Select Entity Type to Edit its Color Rule Activation"
        label = ""
        selected_row,ok = QInputDialog.getItem(None, title, label, items_list,0,False)
        if not ok:
            self.apsw_close()
            return info_dialog(self.gui, _('EM'),_('Nothing Done.'), show=True)

        s_split = selected_row.split("Active?")
        type = s_split[0].strip()

        answer_list = []
        answer_list.append("True")
        answer_list.append("False")

        title = "Specify 'Column Color Rules' Status for Chosen Entity Type"
        label = "Active is True.  Inactive is False.                                                      "
        answer,ok = QInputDialog.getItem(None, title, label, answer_list,0,False)
        if not ok:
            self.apsw_close()
            return info_dialog(self.gui, _('EM'),_('Nothing Done.'), show=True)

        if answer == "True":
            value = "1"
        else:
            value = "0"

        preference = 'add_column_color_rule'

        mysql = "INSERT OR REPLACE INTO  _entities_type_preferences (type,preference,value) VALUES (?,?,?) "
        self.my_cursor.execute("begin")
        self.my_cursor.execute(mysql,(type,preference,value))
        self.my_cursor.execute("commit")

        msg = "Entity Type: " + type + " is Active:  " + answer
        info_dialog(self.gui, 'EM Entity Type Column Color Rule Activate',msg).show()

        self.apsw_close()
   #-----------------------------------------------------------------------------------------
    def restrict_link_note_column_color_rules_entity_type(self):

        is_valid = self.apsw_connect_to_library()
        if not is_valid:
            if DEBUG: print("Entities Manager: Fatal Error - apsw_connect_to_library failed.  Try Again.")
            return

        try:
            mysql = "SELECT type,'dummy' FROM _entities_type_table_mapping WHERE type NOT LIKE '~%' ORDER BY type "
            self.my_cursor.execute(mysql)
            tmp_rows = self.my_cursor.fetchall()
            type_list = []
            for row in tmp_rows:
                type,dummy = row
                type_list.append(type)
            #END FOR
            del tmp_rows
        except:
            self.apsw_close()
            return error_dialog(self.gui, _('EM'),_('Entities Manager has not yet been initialized.  Select the main menu item.'), show=True)

        mysql = "SELECT type,value FROM _entities_type_preferences WHERE preference = 'color_if_link_note'  "
        self.my_cursor.execute(mysql)
        tmp_rows = self.my_cursor.fetchall()
        type_color_preferences_dict = {}
        for row in tmp_rows:
            type,value = row
            type_color_preferences_dict[type] = value
        #END FOR
        del tmp_rows

        items_list = []

        for type in type_list:
            if type in type_color_preferences_dict:
                value = type_color_preferences_dict[type]
            else:  #should not ever happen, but...
                value = "0"
            if value == "1":
                value_name = "True"
            else:
                value_name = "False"
            r = type + "      Active?    " + value_name
            items_list.append(r)
        #END FOR

        title = "Select Entity Type to Edit its Color 'Only-if-Link-or-Note' Activation"
        label = ""
        selected_row,ok = QInputDialog.getItem(None, title, label, items_list,0,False)
        if not ok:
            self.apsw_close()
            return info_dialog(self.gui, _('EM'),_('Nothing Done.'), show=True)

        s_split = selected_row.split("Active?")
        type = s_split[0].strip()

        answer_list = []
        answer_list.append("True")
        answer_list.append("False")

        title = "Specify 'Only-if-Link-or-Note' Status for Chosen Entity Type"
        label = "Active is True.  Inactive is False.                                                      "
        answer,ok = QInputDialog.getItem(None, title, label, answer_list,0,False)
        if not ok:
            self.apsw_close()
            return info_dialog(self.gui, _('EM'),_('Nothing Done.'), show=True)

        if answer == "True":
            value = "1"
        else:
            value = "0"

        preference = 'color_if_link_note'

        mysql = "INSERT OR REPLACE INTO  _entities_type_preferences (type,preference,value) VALUES (?,?,?) "
        self.my_cursor.execute("begin")
        self.my_cursor.execute(mysql,(type,preference,value))
        self.my_cursor.execute("commit")

        msg = "Entity Type: " + type + " is Active:  " + answer
        info_dialog(self.gui, "EM Entity Type 'Only-if-Link-or-Note' Activate",msg).show()

        self.apsw_close()
   #-----------------------------------------------------------------------------------------
    def get_em_preference(self,preference):
        if "_list" in preference:
            is_list = True
        else:
            is_list = False

        value = None

        mysql = "SELECT value,'dummy'  FROM  _entities_manager_preferences WHERE preference = ? "
        self.my_cursor.execute(mysql,([preference]))
        tmp_rows = self.my_cursor.fetchall()
        for row in tmp_rows:
            value,dummy = row
            break
        #END FOR
        del tmp_rows

        if is_list:
            if value is None:
                value = []
            else:
                value = ast.literal_eval(as_unicode(value))
                value.sort()

        return value
   #-----------------------------------------------------------------------------------------
    def change_em_preference(self,preference,value):
        preference = preference.lower()
        if isinstance(value,list):
            value = as_unicode(value)
        try:
            mysql = "INSERT OR REPLACE INTO _entities_manager_preferences (preference,value) VALUES (?,?)"
            self.my_cursor.execute("begin")
            self.my_cursor.execute(mysql,(preference,value))
            self.my_cursor.execute("commit")
            return True
        except Exception as e:
            if DEBUG: print("change_em_preference exception: ", as_unicode(e))
            return False
   #-----------------------------------------------------------------------------------------
    def activate_deactivate_rules_backup_icons(self):
        self.activate_deactivate_rules_backup("icon")
   #-----------------------------------------------------------------------------------------
    def activate_deactivate_rules_backup_colors(self):
        self.activate_deactivate_rules_backup("color")
   #-----------------------------------------------------------------------------------------
    def activate_deactivate_rules_backup(self,kind):
        if kind == "icon":
            preference = "automatic_icon_rules_backup_is_active"
            title = "Activate or Deactivate Icon Rules Automatic Backup"
        else:
            preference = "automatic_color_rules_backup_is_active"
            title = "Activate or Deactivate EM "
            title = "Activate or Deactivate Color Rules Automatic Backup"

        answer_list = []
        answer_list.append("True")
        answer_list.append("False")


        label = "Active is True.  Inactive is False.                                                      "
        answer,ok = QInputDialog.getItem(None, title, label, answer_list,0,False)
        if not ok:
            return info_dialog(self.gui, _('EM'),_('Nothing Done.'), show=True)

        if answer == "True":
            value = "1"
        else:
            value = "0"

        is_valid = self.apsw_connect_to_library()
        if not is_valid:
            if DEBUG: print("Entities Manager: Fatal Error - apsw_connect_to_library failed.  Try Again.")
            return
        self.change_em_preference(preference,value)
        self.apsw_close()
        msg = title + ":  Now Active?  " + answer
        return info_dialog(self.gui, "EM Automatic Rules Backup Activation/Deactivation",msg).show()
   #-----------------------------------------------------------------------------------------
    def get_entity_type_preference_dict(self,preference):
        preference_dict = {}
        mysql = "SELECT type,value  FROM  _entities_type_preferences WHERE preference = ? "
        self.my_cursor.execute(mysql,([preference]))
        tmp_rows = self.my_cursor.fetchall()
        for row in tmp_rows:
            type,value = row
            preference_dict[type] = value
        #END FOR
        del tmp_rows
        return preference_dict
   #-----------------------------------------------------------------------------------------
    def get_entity_id_preference_dict(self,preference):
        entity_id_preference_dict = {}
        mysql = "SELECT entity_id,value  FROM  _entities_preferences WHERE preference = ? "
        self.my_cursor.execute(mysql,([preference]))
        tmp_rows = self.my_cursor.fetchall()
        for row in tmp_rows:
            entity_id,value = row  #entity_id is text, not int
            entity_id_preference_dict[entity_id] = value
        #END FOR
        del tmp_rows
        return entity_id_preference_dict
   #-----------------------------------------------------------------------------------------
   #-----------------------------------------------------------------------------------------
   #~ SEARCH & SELECT
   #-----------------------------------------------------------------------------------------
   #-----------------------------------------------------------------------------------------
    def find_all_books_having_entities(self):
        is_valid = self.apsw_connect_to_library()
        if not is_valid:
            if DEBUG: print("Entities Manager: Fatal Error - apsw_connect_to_library failed.  Try Again.")
            return

        self.get_table_custom_columns()

        mysql = "SELECT type,value, (SELECT type_table FROM _entities_type_table_mapping \
                                                                                   WHERE _entities_type_table_mapping.type = _entities.type) AS type_table \
                        FROM _entities \
                        WHERE type IN (SELECT type FROM _entities_type_table_mapping \
                                                                     WHERE type NOT LIKE '~%')"

        self.my_cursor.execute(mysql)
        tmp_rows = self.my_cursor.fetchall()
        if not tmp_rows:
            self.apsw_close()
            return
        if len(tmp_rows) == 0:
            self.apsw_close()
            return
        type_table_dict = {}
        type_value_list = []
        type_list = []
        for row in tmp_rows:
            type,value,type_table = row
            column,is_multiple,normalized = self.get_table_tech_data(type)
            if not column is None:
                type_table_dict[type] = type_table,column,is_multiple,normalized
                r = type,value
                type_value_list.append(r)
                if not type in type_list:
                    type_list.append(type)
        #END FOR
        del tmp_rows
        type_list.sort()
        type_value_list.sort()

        tmp_rows = []
        book_list = []
        pick_list = []

        for type in type_list:
            type_table,column,is_multiple,normalized = type_table_dict[type]
            for row in type_value_list:
                entity_type,entity_value = row
                if entity_type != type:
                    continue
                if normalized == 0:
                    if type.startswith("#"):
                        mysql = "SELECT book FROM [TABLE] WHERE value = ?"
                        mysql = mysql.replace("[TABLE]",type_table)
                    elif type == "Title":
                        mysql = "SELECT id FROM books WHERE title = ?"
                    else:
                        continue
                else:
                    if type.startswith("#"):
                        #~           SELECT book FROM books_custom_column_16_link WHERE value IN (SELECT id FROM custom_column_16 WHERE custom_column_16.value = ?)
                        mysql = "SELECT book FROM [LINKTABLE] WHERE value IN (SELECT id FROM [TABLE] WHERE [TABLE].value = ?)"
                        link_table = "books_" + type_table + "_link"
                        mysql = mysql.replace("[LINKTABLE]",link_table)
                        mysql = mysql.replace("[TABLE]",type_table)
                    else:
                        mysql = "SELECT book FROM [LINKTABLE] WHERE [COLUMN] IN (SELECT id FROM [TABLE] WHERE [TABLE].name = ?)"
                        link_table = "books_" + type_table + "_link"
                        mysql = mysql.replace("[LINKTABLE]",link_table)
                        mysql = mysql.replace("[TABLE]",type_table)
                        mysql = mysql.replace("[COLUMN]",column)

                try:
                    self.my_cursor.execute(mysql,([entity_value]))
                    tmp_rows = self.my_cursor.fetchall()
                except Exception as e:
                    if DEBUG: print("mysql exception: ",mysql, " >>> ", as_unicode(e))
                    continue
                if not tmp_rows:
                    continue
                if len(tmp_rows) == 0:
                    continue
                for r in tmp_rows:
                    for col in r:
                        if isinstance(col,int):
                            book_list.append(col)
                            pick = entity_type,entity_value
                            pick_list.append(pick)
            #END FOR
        #END FOR

        self.apsw_close()

        del tmp_rows
        del type_list
        del type_value_list
        del type_table_dict

        book_list = list(set(book_list))
        book_list.sort()
        if DEBUG: print("number of books: ", as_unicode(len(book_list)))

        found_dict = {}
        s_true = 'true'
        for book in book_list:
            found_dict[book] = s_true
        #END FOR

        marked_ids = dict.fromkeys(found_dict, s_true)
        self.maingui.current_db.set_marked_ids(marked_ids)
        self.maingui.search.clear()
        self.maingui.search.set_search_string('marked:true')

        pick_list = list(set(pick_list))
        pick_list.sort()

        from calibre_plugins.entities_manager.books_found_dialog import BooksFoundDialog
        self.books_found_dialog = BooksFoundDialog(self.maingui,self.icon,self.font,pick_list)
        self.books_found_dialog.show()
        del BooksFoundDialog
   #-----------------------------------------------------------------------------------------
    def get_table_tech_data(self,type):
        column = None
        is_multiple = None
        normalized = None
        if not type.startswith("#"):
            if type == "Author":
                column = "author"
                normalized = 1
                is_multiple = 1
            elif type == "Publisher":
                column = "publisher"
                normalized = 1
                is_multiple = 0
            elif type == "Series":
                column = "series"
                normalized = 1
                is_multiple = 0
            elif type == "Tags":
                column = "tag"
                normalized = 1
                is_multiple = 1
            elif type == "Title":
                column = "title"
                normalized = 0
                is_multiple = 0
            else:
                pass
        else:
            column = "value"
            if type in self.custom_column_label_dict:
                r = self.custom_column_label_dict[type]
                id,label,name,datatype,display,is_multiple,normalized = r
            else:
                pass
        return column,is_multiple,normalized
   #-----------------------------------------------------------------------------------------
    def get_table_custom_columns(self):

        self.custom_column_label_dict = {}
        self.custom_column_label_list = []

        try:
            mysql = "SELECT id,label,name,datatype,display,is_multiple,normalized FROM custom_columns"
            self.my_cursor.execute(mysql)
            tmp_rows = self.my_cursor.fetchall()
            if tmp_rows is None:
                tmp_rows = []
            for row in tmp_rows:
                id,label,name,datatype,display,is_multiple,normalized = row
                if datatype != "composite" and datatype != "ratings" and datatype != "int" and datatype != "float" and datatype != "bool" and datatype != "datetime":
                    if datatype == "comments":
                        if not '"interpret_as": "short-text"' in display:
                            continue
                    label = "#" + label
                    id = as_unicode(id)
                    r = id,label,name,datatype,display,is_multiple,normalized
                    self.custom_column_label_dict[label] = r
                    self.custom_column_label_list.append(label)
            #END FOR
            self.custom_column_label_list.sort()
            del tmp_rows
        except Exception as e:
            if DEBUG: print("Error in get_table_custom_columns: ", as_unicode(e))
    #-----------------------------------------------------------------------------------------
    def show_entities_combobox_dialog(self):
        entities_list = self.get_all_entity_id_types_values(close_apsw=True)
        from calibre_plugins.entities_manager.entities_combobox_dialog import EntitiesComboboxDialog
        self.entities_combobox_dialog = EntitiesComboboxDialog(self.maingui,self.icon,self.font,entities_list)
        self.entities_combobox_dialog.show()
        del EntitiesComboboxDialog
    #-----------------------------------------------------------------------------------------
#-------------------------------------------------------------------------------------------------------------------------------------
#-------------------------------------------------------------------------------------------------------------------------------------
#-------------------------------------------------------------------------------------------------------------------------------------
#-------------------------------------------------------------------------------------------------------------------------------------
#-------------------------------------------------------------------------------------------------------------------------------------
#END of ui.py