#!/bin/sh
#set -ex

PARAMS=$#

MYNAME=${0/*\//}

if [ $# != 0 ]; then
    if [ "$1" == "--help" ]; then
        echo "Usage : $MYNAME --help"
        echo "        $MYNAME FILE_TYPE"
        exit
    fi
    FILE_TYPE=$1    
fi

if [ "$FILE_TYPE" == "" ]; then
    echo "Usage : $MYNAME FILE_TYPE"
    echo "FILE_TYPE is a string that identifies the file type."
    echo "eg. Intel 80386, ARM, MIPS"
    exit	
fi

## Test the C compiler ##
echo "1. compiling"

cat > hello.c <<_eof_
#include <stdio.h>
int main() { printf("Hello, world!\n"); return 0; }
_eof_

gcc -static hello.c -o hello-static
gcc hello.c -o hello

## Test the C++ compiler. ##

cat > hello.cc <<_eof_
#include <iostream>
int main() { std::cout << "Hello, c++!\n"; return 0; }
_eof_

g++ -static hello.cc -o hellocc-static
g++ hello.cc -o hellocc


## Test file type ##
echo "2. checking file type"

TYPE=`file hello`
TEST1=`awk -v MNE="$FILE_TYPE" -v TP="$TYPE" -- 'BEGIN {print match(TP,MNE)}'`
TYPE=`file hello-static`
TEST2=`awk -v MNE="$FILE_TYPE" -v TP="$TYPE" -- 'BEGIN {print match(TP,MNE)}'`
TYPE=`file hellocc`
TEST3=`awk -v MNE="$FILE_TYPE" -v TP="$TYPE" -- 'BEGIN {print match(TP,MNE)}'`
TYPE=`file hellocc-static`
TEST4=`awk -v MNE="$FILE_TYPE" -v TP="$TYPE" -- 'BEGIN {print match(TP,MNE)}'`
if [ "$TEST1" == 0 ] || [ "$TEST2" == 0 ] || [ "$TEST3" == 0 ] || 
   [ "$TEST4" == 0 ]; then
    echo "    file types        fail"
else
    echo "    file types        ok"
fi


## Test strip ##
echo "3. stripping binaries"

TEST1=`strip hello 2>&1`
TEST2=`strip hello-static 2>&1`
TEST3=`strip hellocc 2>&1`
TEST4=`strip hellocc-static 2>&1`
if [ "$TEST1" ] || [ "$TEST2" ] || [ "$TEST3" ] || [ "$TEST4" ]; then
    echo "    strip             fail"
    echo "$TEST1 $TEST2 $TEST3 $TEST4"
else
    echo "    strip             ok"
fi
	   
## Test library dependencies ##
echo "4. testing library dependencies"

TEST=`ldd ./hello`
TEST=`awk -v TEST="$TEST" 'BEGIN {print match(TEST,"not found")}'`
if [ "$TEST" == 0 ]; then
    echo "    hello             ok"
else
    echo "    hello             fail"
    echo `ldd ./hello`
fi
TEST=`ldd ./hello-static`
TEST=`awk -v TEST="$TEST" 'BEGIN {print match(TEST,"not a dynamic executable")}'`
if [ "$TEST" != 0 ]; then
    echo "    hello-static      ok"
else
    echo "    hello-static      fail"
    echo `ldd ./hello-static`
fi
TEST=`ldd ./hellocc`
TEST=`awk -v TEST="$TEST" 'BEGIN {print match(TEST,"not found")}'`
if [ "$TEST" == 0 ]; then
    echo "    hello c++         ok"
else
    echo "    hello c++         fail"
    echo `ldd ./hellocc`
fi
TEST=`ldd ./hellocc-static`
TEST=`awk -v TEST="$TEST" 'BEGIN {print match(TEST,"not a dynamic executable")}'`
if [ "$TEST" != 0 ]; then
    echo "    hello-static c++  ok"
else
    echo "    hello-static c++  fail"
    echo `ldd ./hellocc-static`
fi
	

## Test running programs ##
echo "5. running binaries"

TEST=`./hello`
if [ "$TEST" = "Hello, world!" ]; then
    echo "    hello             ok"
else
    echo "    hello             fail"
fi
TEST=`./hello-static`
if [ "$TEST" = "Hello, world!" ]; then
    echo "    hello-static      ok"
else
    echo "    hello-static      fail"
fi
TEST=`./hellocc`
if [ "$TEST" = "Hello, c++!" ]; then
    echo "    hello c++         ok"
else
    echo "    hello c++         fail"
fi
TEST=`./hellocc-static`
if [ "$TEST" = "Hello, c++!" ]; then
    echo "    hello-static c++  ok"
else
    echo "    hello-static c++  fail"
fi

	

