#!/usr/bin/env python
from calibre.utils.cleantext import clean_ascii_chars
from calibre_plugins.Adlibris.dictor import dictor

from calibre.ebooks.metadata.book.base import Metadata
from lxml.html import fromstring
from threading import Thread

from calibre_plugins.Adlibris import Adlibris

__license__ = "GPL v3"
__copyright__ = "2022, J-H based on the work by Pr.BarnArt and Grant Drake"
__docformat__ = "restructuredtext en"

import socket
import re
import datetime
import json


class Worker(Thread):  # Get  details

    """
    Get book details from Adlibris.com book page in a separate thread
    related to Adlibris
    """

    name = "Worker"
    description = "Get book details from adlibris.com book page in a separate thread"
    author = "J-H"
    version = (0, 2, 0)
    minimum_calibre_version = (0, 6, 0)

    def __init__(
        self, match_i, result_queue, browser, log, relevance, plugin, timeout=20
    ):
        Thread.__init__(self)
        self.daemon = True
        self.url = match_i
        self.result_queue = result_queue
        self.log, self.timeout = log, timeout
        self.relevance, self.plugin = relevance, plugin
        self.browser = browser.clone_browser()
        self.cover_url = self.isbn = None

    def run(self):
        try:
            self.get_details()
        except:
            self.log.exception(f"get_details failed for url: {self.url}")

    def get_details(self):
        try:
            raw = (
                self.browser.open_novisit(
                    self.url, timeout=self.timeout).read().strip()
            )
        except Exception as e:
            if callable(getattr(e, "getcode", None)) and e.getcode() == 404:
                self.log.error(f"URL malformed: {self.url}")
                return
            attr = getattr(e, "args", [None])
            attr = attr if attr else [None]
            if isinstance(attr[0], socket.timeout):
                msg = "Adlibris.com timed out. Try again later."
                self.log.error(msg)
            else:
                msg = f"Failed to make details query: {self.url}"
                self.log.exception(msg)
            return
        raw = raw.decode("utf-8", errors="replace")

        if "<title>404 - " in raw:
            self.log.error(f"URL malformed: {self.url}")
            return
        try:
            root = fromstring(clean_ascii_chars(raw))
        except:
            msg = f"Failed to parse Adlibris.com details page: {self.url}"
            self.log.exception(msg)
            return
        self.parse_details(root)

    def parse_details(self, root):
        # Parse all book details
        bookData = root.xpath(
            '//script[contains(text(),"window.pageData")]/text()')
        bookInfo = str(bookData[0]).strip()
        bookInfo = re.sub("window.pageData =", "", bookInfo)
        if bookInfo[-1] == ";":
            bookInfo = bookInfo[:-1]  # remove the trailing ';'
        data = json.loads(bookInfo)

        self.isbn = dictor(data, "ProductVariants.0.Ean")
        title = dictor(data, "ProductVariants.0.Title")
        authors = dictor(data, "ProductVariants.0.Authors")
        if authors is None or "Unknown" in authors:
            editors = dictor(
                data, "ProductVariants.0.ProductInfo.Editors.Values", search="Value"
            )
            if editors:
                authors = editors

        if not title or not authors:
            self.log.error(f"Could not find title/authors for {self.url}")
            return

        mi = Metadata(title, authors)

        # Description
        mi.comments = dictor(data, "ProductVariants.0.Description")
        # Cover
        self.cover_url = dictor(
            data, "ProductVariants.0.Images.ProductImages.0.Url")
        mi.has_cover = bool(self.cover_url)

        # Series
        series = dictor(
            data, "ProductVariants.0.ProductInfo.Series.Values.0.Value")
        series_index = dictor(
            data, "ProductVariants.0.ProductInfo.Series.Values.0.Unit")

        if series is not None:
            mi.series = series
            if series_index is not None:
                series_index = re.search("(\d+)", series_index)
                if series_index is not None and float(series_index.group(1)) < 3000:
                    mi.series_index = series_index.group(1)

        mi.publisher = dictor(
            data, "ProductVariants.0.ProductInfo.Publisher.Values.0.Value")

        # Publishing Date
        date_text = dictor(
            data, "ProductVariants.0.ProductInfo.Published.Values.0.Value"
        )
        if date_text:
            try:
                year = int(date_text[:4])
                month = int(date_text[5:7])
                day = int(date_text[8:10])
                from calibre.utils.date import utc_tz

                mi.pubdate = datetime.datetime(year, month, day, tzinfo=utc_tz)
            except:
                self.log.exception(
                    f"Error parsing published date for url: {self.url}")

        # Book's Language
        mi.language = dictor(data, "ProductVariants.0.GroupedLanguageEn")

        mi.source_relevance = self.relevance

        if self.isbn:
            mi.set_identifier(Adlibris.ID_NAME, self.isbn)
            if self.cover_url:
                self.plugin.cache_identifier_to_cover_url(
                    self.isbn, self.cover_url)

        self.plugin.clean_downloaded_metadata(mi)
        self.result_queue.put(mi)
