#!/usr/bin/env python

__license__   = 'GPL v3'
__copyright__ = 'Bob McBobberson'
__docformat__ = 'restructuredtext en'

from PyQt5.Qt import QDialog, QVBoxLayout, QPushButton, QMessageBox, QLabel, QComboBox, QCheckBox, QProgressBar, QTextEdit, QSizePolicy

from PyQt5.QtWidgets import * 
from PyQt5 import QtCore, QtGui 
from PyQt5.QtGui import * 
from PyQt5.QtCore import * 
import sys 
import time

from calibre_plugins.CalibreCV.config import prefs

import urllib, json, re, logging, threading
if sys.version_info[0] >= 3:
    import urllib.request
    from urllib.request import urlopen
else:
    from urllib import urlopen

class CalibreCVDialog(QDialog):
    def __init__(self, gui, icon, do_user_config):
        QDialog.__init__(self, gui)
        self.gui = gui
        self.do_user_config = do_user_config

        # The current database shown in the GUI
        # db is an instance of the class LibraryDatabase from db/legacy.py
        # This class has many, many methods that allow you to do a lot of
        # things. For most purposes you should use db.new_api, which has
        # a much nicer interface from db/cache.py
        self.db = gui.current_db

        self.l = QVBoxLayout()
        self.setLayout(self.l)

        self.setWindowTitle('CalibreCV')
        self.setWindowIcon(icon)

        self.label = QLabel('CalibreCV Behavior:')
        self.l.addWidget(self.label)

        self.cccombobox = QComboBox(self)
        self.cccombobox.addItem("Using Title - Drop down and Hover for Details")
        self.cccombobox.setItemData(0, 'This expects the title of a book to have a specific formatted string containing volume ID and issue#. e.g.:\nDesired Title --- v1234 n0124\nWhere --- is required and the number after v matches Comic Vine\'s volume ID. The number after n is the issue number.', QtCore.Qt.ToolTipRole)
        self.cccombobox.addItem('Using Series - Drop down and Hover for Details')
        self.cccombobox.setItemData(1, 'This expects the series name to match the Comic Vine volume ID and the series number equals the issue number.', QtCore.Qt.ToolTipRole)
        self.cccombobox.addItem('Using IDs - Drop down and Hover for Details')
        self.cccombobox.setItemData(2, 'This expects the custom columns with lookup comicvineissueid and comicvinevolumeid. These must match Comic Vine\'s IDs for the issue and the volume.', QtCore.Qt.ToolTipRole)
        self.cccombobox.setCurrentIndex(prefs['cc_mode'])
        self.cccombobox.currentIndexChanged.connect(self.set_cccombobox)
        self.l.addWidget(self.cccombobox)
 
        self.settitlefromcomicvine = QCheckBox("Update title from Comic Vine?", self)
#        self.settitlefromcomicvine.setCheckState(prefs['name_from_cv'])
        self.settitlefromcomicvine.setChecked(prefs['name_from_cv'])
        self.settitlefromcomicvine.stateChanged.connect(self.set_titlefromcomicvine)
        self.l.addWidget(self.settitlefromcomicvine)

        self.update_metadata_button = QPushButton('Update metadata from Comic Vine', self)
        self.update_metadata_button.clicked.connect(self.spawn_thread)
        self.l.addWidget(self.update_metadata_button)

        self.conf_button = QPushButton('Configure this plugin', self)
        self.conf_button.clicked.connect(self.config)
        self.l.addWidget(self.conf_button)

        self.progress = QProgressBar(self)
        self.l.addWidget(self.progress)
        
        self.outputlog = QTextEdit(self)
        self.outputlog.setDisabled(True)
        self.l.addWidget(self.outputlog)

        self.resize(self.sizeHint())
        self.resize(1280,720)
        
    def set_cccombobox(self):
        prefs['cc_mode'] = self.cccombobox.currentIndex()

    def set_titlefromcomicvine(self):
        prefs['name_from_cv'] = self.settitlefromcomicvine.checkState()
        
    def spawn_thread(self):
        self.update_metadata_button.setEnabled(False)
        threading.Thread(target=self.update_metadata).start()        
        
    def update_metadata(self):
        '''
        Set the metadata in the files in the selected book's record to
        match the current metadata in the database.
        '''
        from calibre.ebooks.metadata.meta import set_metadata
        from calibre.gui2 import error_dialog, info_dialog

        # Get currently selected books
        rows = self.gui.library_view.selectionModel().selectedRows()
        if not rows or len(rows) == 0:
            return error_dialog(self.gui, 'Cannot update metadata', 'No books selected', show=True)

        # Map the rows to book ids
        ids = list(map(self.gui.library_view.model().id, rows))
        db = self.db.new_api
        
        # Set the current book number
        current_book = 0
        
        for book_id in ids:

            # TODO:
            # Add button to force issue number to -1
            #

            # Set the delay between calls to the ComicVine API
            cvAPIDelay = float(prefs['cv_api_delay']) / 1000
            print("cvAPIDelay")
            print(cvAPIDelay)

            # Get the current metadata for this book from the db
            mi = db.get_metadata(book_id, get_cover=True, cover_as_data=True)

            # calibrecv using... Title=0, Series=1, IDs=2
            calibrecv_using = 0
            try:
                if prefs['cc_mode'] == 0:
                    self.outputlog.insertPlainText(str(mi.get('title')) + '...\n')
                    calibrecv_using = 0
                elif prefs['cc_mode'] == 1:
                    self.outputlog.insertPlainText(str(mi.get('series')) + ' [' + str(mi.get('series_index')) + ']...\n')
                    calibrecv_using = 1
                elif prefs['cc_mode'] == 2:
                    self.outputlog.insertPlainText('VolumeID: ' + str(mi.get('#comicvinevolumeid')) + ' IssueID: ' + str(mi.get('#comicvineissueid')) + '...\n')
                    calibrecv_using = 2
                else:
                    raise
            except:
                self.outputlog.insertPlainText('Error: Invalid option in self.cccombobox\n')

            # Typecast the issue variables
            ishname = str('')
            ishvolume = str('')
            ishnumber = str('')
            ishsite = str('')
            ishauthors = []
            ishcreators = []
            ishcharacters = []

            # If calibrecv using...
            try:
                if calibrecv_using == 0:
                    title = mi.get('title')
                    ishvolume = title[title.find("--- v") + 5:title.find(" n")]
                    ishnumber = title[title.find(" n") + 2:]

            except:
                self.outputlog.insertPlainText('Error. Unable to match data in Title field to known values. Please validate.\n')
    
            try:    
                if calibrecv_using == 1:
                    ishvolume = str(mi.get('series'))
                    ishnumber = str(mi.get('series_index'))
            except:
                self.outputlog.insertPlainText('Unable to match data in Series and Number fields to known values. Please validate.\n')
            
            try:
                if calibrecv_using == 2:
                    ishvolume = str(mi.get('#comicvinevolumeid'))
                    ishnumber = str(mi.get('#comicvineissueid'))
            except:
                self.outputlog.insertPlainText('Unable to match data in #comicvineissueid and #comicvinevolumeid fields to known values. Please validate.\n')

            try:
                if (not ishnumber) or (not ishvolume):
                    raise
            except:
                self.outputlog.insertPlainText('Issue and/or volume numbers are blank. Please validate.\n')

            # If the issue number is a float with a trailing '.0', strip it.
            if (ishnumber[-2:] == '.0'):
                ishnumber = ishnumber[0:-2]
                
            # If the issue number has any leading zeros, strip 'em.
            ishnumber = ishnumber.lstrip('0')

            # Grab the series information from Comic Vine
            response = ''
            try:
                if sys.version_info[0] >= 3:
                    theurl = 'https://comicvine.gamespot.com/api/volumes/?api_key=' + str(prefs['cv_api_key']) + '&format=json&filter=id:' + ishvolume
                    req = urllib.request.Request(theurl, data=None, headers={'User-Agent': 'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_9_3) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/35.0.1916.47 Safari/537.36'})
                    response = urllib.request.urlopen(req).read()
                    print(cvAPIDelay)
                    time.sleep(cvAPIDelay)

                else:
                    url = 'https://comicvine.gamespot.com/api/volumes/?api_key=' + str(prefs['cv_api_key']) + '&format=json&filter=id:' + ishvolume
                    response = urllib.urlopen(url)
                    print(cvAPIDelay)
                    time.sleep(cvAPIDelay)

            except:
               self.outputlog.insertPlainText('Unable to retrieve series information from Comic Vine API. Please validate issue and volume numbers.\n')

            # Update series information
            try:
                data = ''
                if sys.version_info[0] >= 3:
                    data = json.loads(response)
                else:
                    data = json.loads(response.read())

                for series in data['results']:
                    mi.set("series", series['name'] + ' (' + series['start_year'] + ')')
                    mi.set("publisher", series['publisher']['name'])
                    mi.set("#comicvinevolumeid", ishvolume)
            except:
                self.outputlog.insertPlainText('Unable to update series data from json data.\n')

            # Grab the issue information from Comic Vine
            try:
                if calibrecv_using == 0:
                    theurl = 'https://comicvine.gamespot.com/api/issues/?api_key=' + str(prefs['cv_api_key']) + '&filter=volume:' + ishvolume + ',issue_number:' + ishnumber + '&format=json'
                elif calibrecv_using == 1:
                    theurl = 'https://comicvine.gamespot.com/api/issues/?api_key=' + str(prefs['cv_api_key']) + '&filter=volume:' + ishvolume + ',issue_number:' + ishnumber + '&format=json'
                elif calibrecv_using == 2:
                    theurl = 'https://comicvine.gamespot.com/api/issues/?api_key=' + str(prefs['cv_api_key']) + '&filter=volume:' + ishvolume + ',id:' + ishnumber + '&format=json'
                else:
                    raise

                if sys.version_info[0] >= 3:
                    req = urllib.request.Request(theurl, data=None, headers={'User-Agent': 'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_9_3) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/35.0.1916.47 Safari/537.36'})
                    response = urllib.request.urlopen(req).read()
                    print(cvAPIDelay)
                    time.sleep(cvAPIDelay)
                else:
                    response = urllib.urlopen(theurl)
                    print(cvAPIDelay)
                    time.sleep(cvAPIDelay)

            except:
                self.outputlog.insertPlainText('Unable to retrieve issue information from Comic Vine API. Please validate issue and volume numbers.\n')

            # Update issue information
            try:
                if sys.version_info[0] >= 3:
                    data = json.loads(response)
                else:
                    data = json.loads(response.read())

                for issues in data['results']:
                    mi.set("pubdate", issues['cover_date'])
                    mi.set("#comicvineissueid", issues['id'])
                    mi.set("series_index", issues['issue_number'])
                    ishsite = issues['api_detail_url']
                    
                    if (self.settitlefromcomicvine.checkState() == 0) and (ishname):
                        mi.set("title", ishname)
                    else:
                        mi.set("title", issues['volume']['name'])

            except:
                self.outputlog.insertPlainText('Unable to update issue data from json data.\n')

            # Grab detailed issue information from Comic Vine
            try:
                theurl = ishsite + '?api_key=' + prefs['cv_api_key'] + '&format=json'

                if sys.version_info[0] >= 3:
                    req = urllib.request.Request(theurl, data=None, headers={'User-Agent': 'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_9_3) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/35.0.1916.47 Safari/537.36'})
                    response = urllib.request.urlopen(req).read()
                    print(cvAPIDelay)
                    time.sleep(cvAPIDelay)
                else:
                    response = urllib.urlopen(theurl)
                    print(cvAPIDelay)
                    time.sleep(cvAPIDelay)

            except:
                self.outputlog.insertPlainText('Unable to retrieve detailed issue information from Comic Vine API. Please validate issue and volume numbers.\n')

            # Update detailed issue information
            try:
                if sys.version_info[0] >= 3:
                    data = json.loads(response)
                else:
                    data = json.loads(response.read())

                for person in data['results']['person_credits']:
                    person_role = person['role']
                    ishcreators.append(person['name'])

                    if (person_role.find('writer') >= 0):
                        ishauthors.append(person['name'])
                
                for character in data['results']['character_credits']:
                    ishcharacters.append(character['name'])

                if not ishauthors:
                    ishauthors.append('Unknown')
                
                mi.set('authors', ishauthors)
                mi.set('#creators', ishcreators)
                mi.set('#characters', ishcharacters)
                mi.set('#physicalcopy', 0)

            except:
                self.outputlog.insertPlainText('Unable to update detailed issue data from json data.\n')

            # Commit the changes
            changed_ids = db.set_metadata(book_id, mi, force_changes=True, allow_case_change=False)

            # Update metadata to file
            fmts = db.formats(book_id)
            if not fmts:
                continue
            for fmt in fmts:
                fmt = fmt.lower()
                # Get a python file object for the format. This will be either
                # an in memory file or a temporary on disk file
                ffile = db.format(book_id, fmt, as_file=True)
                ffile.seek(0)
                # Set metadata in the format
                set_metadata(ffile, mi, fmt)
                ffile.seek(0)
                # Now replace the file in the calibre library with the updated
                # file. We dont use add_format_with_hooks as the hooks were
                # already run when the file was first added to calibre.
                db.add_format(book_id, fmt, ffile, run_hooks=False)

            # Update the progress bar
            current_book = current_book + 1
            self.progress.setValue(int((current_book * 100) / len(ids)))

        self.update_metadata_button.setEnabled(True)

    def config(self):
        self.do_user_config(parent=self)
        self.label.setText('CalibreCV Behavior:')
