#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2011, Kovid Goyal <kovid@kovidgoyal.net>'
__docformat__ = 'restructuredtext en'

import os, shutil, urllib

# calibre Python 3 compatibility.
import six
from six import text_type as unicode
try:
    from urllib.request import urlretrieve
except ImportError:
    from urllib import urlretrieve

try:
    from PyQt5.Qt import Qt, QVBoxLayout, QHBoxLayout, QLabel, QPushButton, \
                     QLineEdit, QFormLayout, QComboBox, QDialog, \
                     QGridLayout, QGroupBox, QRadioButton, QDialogButtonBox
    from PyQt5 import QtCore
except ImportError:
    from PyQt4.Qt import Qt, QVBoxLayout, QHBoxLayout, QLabel, QPushButton, \
                     QLineEdit, QFormLayout, QComboBox, QDialog, \
                     QGridLayout, QGroupBox, QRadioButton, QDialogButtonBox
    from PyQt4 import QtCore

from calibre.gui2 import error_dialog, question_dialog, choose_files

import calibre_plugins.search_the_internet.config as cfg

try:
    _fromUtf8 = QtCore.QString.fromUtf8
except AttributeError:
    _fromUtf8 = lambda s: s

try:
    load_translations()
except NameError:
    print("Search The Internet::dialog.py - exception when loading translations")
    pass # load_translations() added in calibre 1.9

class PickTestBookDialog(QDialog):

    def __init__(self, parent=None):
        QDialog.__init__(self, parent)
        self.setWindowTitle(_('Select test data'))
        layout = QVBoxLayout(self)
        self.setLayout(layout)

        c = cfg.plugin_prefs[cfg.STORE_TEST_NAME]
        selected_idx = c[cfg.TEST_LAST_BOOK_KEY]
        self.data_items = c[cfg.TEST_VALUES_KEY]

        combo_layout = QHBoxLayout()
        lbl_choose = QLabel(_('&Select test book:'), self)
        lbl_choose.setMinimumSize(100, 0)
        combo_layout.addWidget(lbl_choose, 0, Qt.AlignLeft)
        self._book_combo = TestDataComboBox(self, self.data_items)
        self._book_combo.currentIndexChanged.connect(self.combo_index_changed)
        lbl_choose.setBuddy(self._book_combo)
        self._book_combo.setMinimumSize(200, 0)
        combo_layout.addWidget(self._book_combo, 1, Qt.AlignLeft)
        layout.addLayout(combo_layout)

        group_box = QGroupBox(self)
        f = QFormLayout()
        self._title_edit = QLineEdit('')
        f.addRow(QLabel(_('Title:')), self._title_edit)
        self._author_edit = QLineEdit('')
        f.addRow(QLabel(_('Author:')), self._author_edit)
        self._publisher_edit = QLineEdit('')
        f.addRow(QLabel(_('Publisher:')), self._publisher_edit)
        self._isbn_edit = QLineEdit('')
        f.addRow(QLabel('ISBN:'), self._isbn_edit)
        group_box.setLayout(f)
        layout.addWidget(group_box)

        button_box = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        button_box.accepted.connect(self.ok_clicked)
        button_box.rejected.connect(self.reject)
        layout.addWidget(button_box)
        self.resize(self.sizeHint())
        self._book_combo.setCurrentIndex(selected_idx)
        # Force the display of the currently selected item in case index changed event not fired
        self.combo_index_changed()

    def ok_clicked(self):
        # Persist the test data and selected index into the JSON file
        test_data = {}
        test_data[cfg.TEST_LAST_BOOK_KEY] = self._book_combo.currentIndex()
        data_row = self.data_items[self._book_combo.currentIndex()]
        data_row['author'] = unicode(self._author_edit.text()).strip()
        data_row['title'] = unicode(self._title_edit.text()).strip()
        data_row['publisher'] = unicode(self._publisher_edit.text()).strip()
        data_row['isbn'] = unicode(self._isbn_edit.text()).strip()
        test_data[cfg.TEST_VALUES_KEY] = self.data_items
        cfg.plugin_prefs[cfg.STORE_TEST_NAME] = test_data
        self.accept()

    def combo_index_changed(self):
        # Update the dialog contents with metadata for the selected item
        selected_idx = self._book_combo.currentIndex()
        data_item = self.data_items[selected_idx]
        self._author_edit.setText(_fromUtf8(data_item['author']))
        self._title_edit.setText(_fromUtf8(data_item['title']))
        self._publisher_edit.setText(_fromUtf8(data_item['publisher']))
        self._isbn_edit.setText(_fromUtf8(data_item['isbn']))


class TestDataComboBox(QComboBox):

    def __init__(self, parent, data_items):
        QComboBox.__init__(self, parent)
        self.populate_combo(data_items)

    def populate_combo(self, data_items):
        self.clear()
        for i, data in enumerate(data_items):
            self.insertItem(i, data['display'])


class PickImageDialog(QDialog): # {{{

    def __init__(self, parent=None, resources_dir='', image_names=[]):
        QDialog.__init__(self, parent)
        self.resources_dir = resources_dir
        self.image_names = image_names
        self.setWindowTitle(_('Add New Image'))
        v = QVBoxLayout(self)

        group_box = QGroupBox(_('&Select image source'), self)
        v.addWidget(group_box)
        grid = QGridLayout()
        self._radio_web = QRadioButton(_('From &web domain favicon'), self)
        self._radio_web.setChecked(True)
        self._web_domain_edit = QLineEdit(self)
        self._radio_web.setFocusProxy(self._web_domain_edit)
        grid.addWidget(self._radio_web, 0, 0)
        grid.addWidget(self._web_domain_edit, 0, 1)
        grid.addWidget(QLabel('e.g. www.amazon.com'), 0, 2)
        self._radio_file = QRadioButton(_('From .png &file'), self)
        self._input_file_edit = QLineEdit(self)
        self._input_file_edit.setMinimumSize(200, 0)
        self._radio_file.setFocusProxy(self._input_file_edit)
        pick_button = QPushButton('...', self)
        pick_button.setMaximumSize(24, 20)
        pick_button.clicked.connect(self.pick_file_to_import)
        grid.addWidget(self._radio_file, 1, 0)
        grid.addWidget(self._input_file_edit, 1, 1)
        grid.addWidget(pick_button, 1, 2)
        group_box.setLayout(grid)

        save_layout = QHBoxLayout()
        lbl_filename = QLabel(_('&Save as filename:'), self)
        lbl_filename.setMinimumSize(155, 0)
        self._save_as_edit = QLineEdit('', self)
        self._save_as_edit.setMinimumSize(200, 0)
        lbl_filename.setBuddy(self._save_as_edit)
        lbl_ext = QLabel('.png', self)
        save_layout.addWidget(lbl_filename, 0, Qt.AlignLeft)
        save_layout.addWidget(self._save_as_edit, 0, Qt.AlignLeft)
        save_layout.addWidget(lbl_ext, 1, Qt.AlignLeft)
        v.addLayout(save_layout)

        button_box = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        button_box.accepted.connect(self.ok_clicked)
        button_box.rejected.connect(self.reject)
        v.addWidget(button_box)
        self.resize(self.sizeHint())
        self._web_domain_edit.setFocus()
        self.new_image_name = None

    @property
    def image_name(self):
        return self.new_image_name

    def pick_file_to_import(self):
        images = choose_files(None, 'menu icon dialog', _('Select a .png file for the menu icon'),
                             filters=[('PNG Image Files', ['png'])], all_files=False, select_only_single_file=True)
        if not images:
            return
        f = images[0]
        if not f.lower().endswith('.png'):
            return error_dialog(self, _('Cannot select image'),
                    _('Source image must be a .png file.'), show=True)
        self._input_file_edit.setText(f)
        self._save_as_edit.setText(os.path.splitext(os.path.basename(f))[0])

    def ok_clicked(self):
        # Validate all the inputs
        save_name = unicode(self._save_as_edit.text()).strip()
        if not save_name:
            return error_dialog(self, _('Cannot import image'),
                    _('You must specify a filename to save as.'), show=True)
        self.new_image_name = os.path.splitext(save_name)[0] + '.png'
        if save_name.find('\\') > -1 or save_name.find('/') > -1:
            return error_dialog(self, _('Cannot import image'),
                    _('The save as filename should consist of a filename only.'), show=True)
        if not os.path.exists(self.resources_dir):
            os.makedirs(self.resources_dir)
        dest_path = os.path.join(self.resources_dir, self.new_image_name)
        if save_name in self.image_names or os.path.exists(dest_path):
            if not question_dialog(self, _('Are you sure?'), '<p>'+
                    _('An image with this name already exists - overwrite it?'),
                    show_copy_button=False):
                return

        if self._radio_web.isChecked():
            domain = unicode(self._web_domain_edit.text()).strip()
            if not domain:
                return error_dialog(self, _('Cannot import image'),
                        _('You must specify a web domain url'), show=True)
            url = 'http://www.google.com/s2/favicons?domain=' + domain
            urlretrieve(url, dest_path)
            return self.accept()
        else:
            source_file_path = unicode(self._input_file_edit.text()).strip()
            if not source_file_path:
                return error_dialog(self, _('Cannot import image'),
                        _('You must specify a source file.'), show=True)
            if not source_file_path.lower().endswith('.png'):
                return error_dialog(self, _('Cannot import image'),
                        _('Source image must be a .png file.'), show=True)
            if not os.path.exists(source_file_path):
                return error_dialog(self, _('Cannot import image'),
                        _('Source image does not exist!'), show=True)
            shutil.copyfile(source_file_path, dest_path)
            return self.accept()
