#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2011, Grant Drake <grant.drake@gmail.com>'
__docformat__ = 'restructuredtext en'

from collections import OrderedDict
from PyQt5.Qt import QWidget, QGridLayout, QLabel, QPushButton, QGroupBox, QVBoxLayout
from calibre.utils.config import JSONConfig

from calibre_plugins.clean_comments.common_utils import (KeyValueComboBox, KeyboardConfigDialog, ImageTitleLayout, debug_print,                                                         get_library_uuid)

import copy, os

try:
    debug_print("Clean Comments::config.py - loading translations")
    load_translations()
except NameError:
    debug_print("Clean Comments::config.py - exception when loading translations")
    pass # load_translations() added in calibre 1.9                                 

PREFS_NAMESPACE = 'CleanCommentsPlugin'
PREFS_KEY_SETTINGS = 'settings'

STORE_NAME = 'Options'
KEY_APPLY_MARKDOWN  = 'applyMarkdon'
KEY_KEEP_LINK_URL = 'keepLinkUrl'

SHOW_MARKDOWN = OrderedDict([('keep', _('Apply markup: italic, bold, blockquote')),
                        ('none', _('No markers'))])

SHOW_URL = OrderedDict([('keep', _('Keep URL  --  [Link text](Link URL)')),
                        ('none', _('Delete URL  --  Link text'))])

DEFAULT_LIBRARY_VALUES = {
    KEY_APPLY_MARKDOWN: 'none',
    KEY_KEEP_LINK_URL: 'none',
}

def get_library_config(db):
    library_id = get_library_uuid(db)
    library_config = None

    if library_config is None:
        library_config = db.prefs.get_namespaced(PREFS_NAMESPACE, PREFS_KEY_SETTINGS,
                                             copy.deepcopy(DEFAULT_LIBRARY_VALUES))
    return library_config

def set_library_config(db, library_config):
    db.prefs.set_namespaced(PREFS_NAMESPACE, PREFS_KEY_SETTINGS, library_config)

class ConfigWidget(QWidget):

    def __init__(self, plugin_action):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        layout = QVBoxLayout(self)
        self.setLayout(layout)

        library_config = get_library_config(self.plugin_action.gui.current_db)
        
        title_layout = ImageTitleLayout(self, 'images/clean_comments.png', _('Clean Comments Options'))
        layout.addLayout(title_layout)
        
        # --- Markdown options ---
        markdown_group_box = QGroupBox(_('Markdown Options:'), self)
        layout.addWidget(markdown_group_box)
        markdown_group_box_layout = QGridLayout()
        markdown_group_box.setLayout(markdown_group_box_layout)
        
        markdown_group_box_layout.addWidget(QLabel(_("By default the plugin clean html and markdown. You can adjust the output below.")), 0, 1)

        markdown_group_box_layout.addWidget(QLabel(_("Apply markdown:"), self), 1, 1)
        post_show1 = library_config.get(KEY_APPLY_MARKDOWN, DEFAULT_LIBRARY_VALUES[KEY_APPLY_MARKDOWN])
        self.showCombo1 = KeyValueComboBox(self, SHOW_MARKDOWN, post_show1)
        markdown_group_box_layout.addWidget(self.showCombo1, 2, 1)

        markdown_group_box_layout.addWidget(QLabel(_('Link URL:'), self), 3, 1)
        post_show3 = library_config.get(KEY_KEEP_LINK_URL, DEFAULT_LIBRARY_VALUES[KEY_KEEP_LINK_URL])
        self.showCombo2 = KeyValueComboBox(self, SHOW_URL, post_show3)
        markdown_group_box_layout.addWidget(self.showCombo2, 4, 1)

        keyboard_shortcuts_button = QPushButton(_('Keyboard shortcuts...'), self)
        keyboard_shortcuts_button.setToolTip(_(
                    'Edit the keyboard shortcuts associated with this plugin'))
        keyboard_shortcuts_button.clicked.connect(self.edit_shortcuts)
        layout.addWidget(keyboard_shortcuts_button)
        layout.addStretch(1)

    def save_settings(self):
        db = self.plugin_action.gui.current_db
        library_config = get_library_config(db)

        library_config[KEY_APPLY_MARKDOWN] = self.showCombo1.selected_key()
        library_config[KEY_KEEP_LINK_URL] = self.showCombo2.selected_key()

        set_library_config(db, library_config)

    def edit_shortcuts(self):
        self.save_settings ()
        self.plugin_action.build_menus ()
        d = KeyboardConfigDialog(self.plugin_action.gui, self.plugin_action.action_spec[0])
        if d.exec_() == d.Accepted:
            self.plugin_action.gui.keyboard.finalize()
