import java.io.File;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.PrintWriter;
import java.io.InputStreamReader;
import java.net.URL;
import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserFactory;
import org.xmlpull.v1.XmlPullParserException;

/**
 * This class provides a RSS feed formatter that produces an iLiad friendly
 * HTML and manifest.xml for a RSS feed. The RSS can either be supplied as
 * a URL or as a local file.
 *
 * This class requires the XPP3 package: http://www.extreme.indiana.edu/xgws/xsoap/xpp/
 *
 * Usage: java TheDailyiLiadian http://newsrss.bbc.co.uk/rss/newsonline_uk_edition/front_page/rss.xml bbc
 * 
 * @author Scott Turner (scotty1024@mac.com)
 * @version $Revision: 1.4 $
 * @Date October 21, 2006
 */
public class TheDailyiLiadian {

    private static int itemCount = 0;
    private static File destination = null;
    private static StringBuffer index = new StringBuffer(16384);
    private static XmlPullParserFactory factory = null;

    static File imageDirectory = null;

    public static void main(String[] args)
	throws Exception
    {
	String rssURL = args[0];
	destination = new File(args[1]);
	if (destination.exists()) {
	    destination.delete();
	}
	destination.mkdirs();

	// Create images directory
	imageDirectory = new File(destination, "images");
	imageDirectory.mkdir();

	factory = XmlPullParserFactory.newInstance(System.getProperty(XmlPullParserFactory.PROPERTY_NAME), null);
	factory.setFeature(XmlPullParser.FEATURE_PROCESS_NAMESPACES, true);
	    
	XmlPullParser xpp = factory.newPullParser();

	if (rssURL.toLowerCase().startsWith("http://")) {
	    URL url = new URL(rssURL);
	    xpp.setInput ( new InputStreamReader(url.openStream()));
	} else {
	    xpp.setInput ( new FileReader(new File(rssURL)));
	}

	String name = null;

	index.append("<html>\n");
	index.append("<head>\n");
	index.append("<title>The Daily iLidian</title>\n");
	index.append("</head>\n");
	index.append("<body>\n");

	for (int eventType = xpp.getEventType(); eventType != XmlPullParser.END_DOCUMENT; eventType = xpp.next()) {
	    if (eventType == XmlPullParser.START_TAG) {
		name = xpp.getName();
		if ("rss".equals(name)) {
		    String rssVersion = xpp.getAttributeValue(null, "version"); 
		    System.out.println("RSS Version: " + rssVersion);
		    if ("0.91".equals(rssVersion)) {
			parseRSS(xpp, rssVersion);
		    } else if ("2.0".equals(rssVersion)) {
			parseRSS(xpp, rssVersion);
		    } else {
			System.err.println("Unsupported feed version");
			System.exit(1);
		    }
		} else {
		    System.out.println("Unknown Start Tag: " + name);
		}
	    } else if (eventType == XmlPullParser.END_TAG) {
		name = xpp.getName();
		if ("rss".equals(name)) {
		    index.append("</body>\n");
		    index.append("</html>\n");
		    File indexFile = new File(destination, "index.html");
		    PrintWriter out = new PrintWriter( new FileOutputStream( indexFile));
		    out.print(index);
		    out.close();
		    return;
		}
	    }
	}
    }

    private static void parseRSS(XmlPullParser xpp, String aRSSVersion)
	throws Exception
    {
	for (int eventType = xpp.next(); eventType != XmlPullParser.END_DOCUMENT; eventType = xpp.next()) {
	    if (eventType == XmlPullParser.START_TAG) {
		String name = xpp.getName();
		if ("channel".equals(name)) {
		    parseRSSChannel(xpp);
		    System.out.println("Channel contained " + itemCount + " items.");
		    return;
		} else {
		    System.out.println("Unknown Start Tag: " + name);
		}
	    }
	}
    }	

    private static void parseRSSChannel(XmlPullParser xpp)
	throws Exception
    {
	StringBuffer text = null;
	int holderForStartAndLength[] = new int[2];

	String title = null;
	String link = null;
	String description = null;
	String language = null;
	String date = null;

	boolean notDumpedHeader = true;

	for (int eventType = xpp.getEventType(); eventType != XmlPullParser.END_DOCUMENT; eventType = xpp.next()) {
	    if (eventType == XmlPullParser.START_TAG) {
		String name = xpp.getName();
		if ("item".equals(name)) {
		    if (notDumpedHeader) {
			index.append("Channel Title: ").append(title).append("<br/>\n");
			index.append("Link: <a href=\"").append(link).append("\">").append(link).append("</a><br/>\n");
			index.append("Description: ").append(description).append("<br/>\n");
			notDumpedHeader = false;
		    }
		    parseRSSItem(xpp);
		    itemCount++;
		}
		if (text != null) {
		    text.setLength(0);
		}
	    } else if ((eventType == XmlPullParser.TEXT) && !xpp.isWhitespace()) {
		char ch[] = xpp.getTextCharacters(holderForStartAndLength);
		if (holderForStartAndLength[1] != 0) {
		    if (text == null) {
			text = new StringBuffer(128);
		    }
		    text.append(ch, holderForStartAndLength[0], holderForStartAndLength[1]);
		}
	    } else if (eventType == XmlPullParser.END_TAG) {
		String name = xpp.getName();

		if ("channel".equals(name)) {
		    File indexFile = new File(destination, "manifest.xml");
		    PrintWriter out = new PrintWriter(new FileOutputStream( indexFile));
		    out.println("<?xml version=\"1.0\" encoding=\"utf-8\"?>");
		    out.println("<package><metadata><dc-metadata><Title>The Daily iLiadian</Title><Description>" + description + "</Description><Date>" + date + "</Date><Format/><Identifier/><Language>" + language + "</Language><Type/></dc-metadata><y-metadata><startpage>index.html</startpage><version>000</version></y-metadata></metadata></package>");
		    out.close();

		    return;
		} else if ("title".equals(name)) {
		    if (text.length() > 0) {
			title = text.toString();
		    }
		} else if ("link".equals(name)) {
		    if (text.length() > 0) {
			link = text.toString();
		    }
		} else if ("description".equals(name)) {
		    if (text.length() > 0) {
			description = text.toString();
		    }
		} else if ("date".equals(name) ||
			   "pubDate".equals(name)) {
		    if (text.length() > 0) {
			date = text.toString();
		    }
		} else if ("language".equals(name)) {
		    if (text.length() > 0) {
			language = text.toString();
		    }
		} else if ("".equals(name)) {
		    if (text.length() > 0) {
			language = text.toString();
		    }

		    /*
Unknown End Tag: copyright
Unknown End Tag: webMaster
Unknown End Tag: pubDate
Unknown End Tag: lastBuildDate
Unknown End Tag: category
Unknown End Tag: generator
Unknown End Tag: docs
		    */
		} else {
		    System.out.println("Unknown End Tag: " + name);
		}
	    }
	}
    }

    private static void parseRSSItem(XmlPullParser xpp)
	throws Exception
    {
	StringBuffer text = null;
	int holderForStartAndLength[] = new int[2];

	String title = null;
	String link = null;
	String description = null;
	String creator = null;
	String date = null;
	String guid = null;
	String author = null;
	String category = null;

	for (int eventType = xpp.getEventType(); eventType != XmlPullParser.END_DOCUMENT; eventType = xpp.next()) {
	    if (eventType == XmlPullParser.START_TAG) {
		String name = xpp.getName();
		if (text != null) {
		    text.setLength(0);
		}
	    } else if ((eventType == XmlPullParser.TEXT) && !xpp.isWhitespace()) {
		char ch[] = xpp.getTextCharacters(holderForStartAndLength);
		if (holderForStartAndLength[1] != 0) {
		    if (text == null) {
			text = new StringBuffer(128);
		    }
		    text.append(ch, holderForStartAndLength[0], holderForStartAndLength[1]);
		}
	    } else if (eventType == XmlPullParser.END_TAG) {
		String name = xpp.getName();
		if ("item".equals(name)) {
		    index.append("<br/>\n");
		    index.append("<hr/>\n");
		    index.append("<br/>\n");
		    index.append("Link: <a href=\"").append(link).append("\">").append(link).append("</a><br/>\n");
		    index.append("Description: ").append(description).append("<br/>\n");
		    if (creator != null) {
			index.append("Creator: ").append(creator).append("<br/>\n");
		    }
		    if (date != null) {
			index.append("Date: ").append(date).append("<br/>\n");
		    }
		    return;
		} else if ("title".equals(name)) {
		    if (text.length() > 0) {
			title = text.toString();
		    }
		} else if ("link".equals(name)) {
		    if (text.length() > 0) {
			link = text.toString();
		    }
		} else if ("description".equals(name)) {
		    if (text.length() > 0) {
			description = text.toString();
		    }
		} else if ("creator".equals(name)) {
		    if (text.length() > 0) {
			creator = text.toString();
		    }
		} else if ("date".equals(name) ||
			   "pubDate".equals(name)) {
		    if (text.length() > 0) {
			date = text.toString();
		    }
		} else if ("guid".equals(name)) {
		    if (text.length() > 0) {
			guid = text.toString();
		    }
		} else if ("author".equals(name)) {
		    if (text.length() > 0) {
			author = text.toString();
		    }
		} else if ("category".equals(name)) {
		    if (text.length() > 0) {
			category = text.toString();
		    }

		} else {
		    System.out.println("Unknown End Tag: " + name);
		}
	    }
	}
    }
}
