﻿using System;
using System.Collections.Generic;
using System.Text;

using System.IO;

namespace ConvertIMPGUI
{
    class CIMPFile
    {
        #region Class encoded_string_t
        class encoded_string_t
        {
            public int offset;
            public int length;
        }
        #endregion

        #region Class bitStream
        class bitStream
        {
            public int bitCount = 0;
            public int bitBuffer = 0;
        }
        #endregion

        private IMPHeader _header = new IMPHeader();
        private IMPBookProperties _bookProperties = new IMPBookProperties();
        private IMPTableContents _tableOfContents = new IMPTableContents();
        private IMPRESFiles _resFiles = new IMPRESFiles();
        private List<SWIndex> _swIndexes = new List<SWIndex>();
        private string IMPResourceFile = "";

        private int EOF = -1;
        private string IMPFilePath = "";

        public CIMPFile() { }

        public CIMPFile(string impFile)
        {
            IMPFilePath = impFile;
        }

        #region Properties

        #region Header
        public IMPHeader Header
        {
            get { return _header; }
            set { _header = value; }
        }
        #endregion

        #region BookProperties
        public IMPBookProperties BookProperties
        {
            get { return _bookProperties; }
            set { _bookProperties = value; }
        }
        #endregion

        #region TableOfContents
        public IMPTableContents TableOfContents
        {
            get { return _tableOfContents; }
            set { _tableOfContents = value; }
        }
        #endregion

        #region RESFiles
        public IMPRESFiles RESFiles
        {
            get { return _resFiles; }
            set { _resFiles = value; }
        }
        #endregion

        #region SWIndexes
        public List<SWIndex> SWIndexes
        {
            get { return _swIndexes; }
            set { _swIndexes = value; }
        }
        #endregion

        #region Content
        public string Content
        {
            get { return IMPResourceFile; }
        }
        #endregion

        #endregion

        #region Private Methods used int Reversing IMP File

        #region ReadInt
        private int ReadInt(ref BinaryReader br, int count)
        {
            int nResult = 0;
            byte[] buffer = br.ReadBytes(count);

            for (int i = 0; i < count; i++)
                nResult = (nResult << 8) + buffer[i];

            return nResult;
        }
        #endregion

        #region ReadString
        private string ReadString(ref BinaryReader br)
        {
            string strResult = "";

            char ch = ' ';

            while (ch != '\0')
            {
                strResult += ch;
                ch = br.ReadChar();
            }

            return strResult.Trim();
        }
        #endregion

        #region GetInt
        private int GetInt(byte[] buffer, int start, int count)
        {
            int nResult = 0;

            for (int i = start; i < start + count; i++)
                nResult = (nResult << 8) + buffer[i];

            return nResult;
        }
        #endregion

        #region GetString
        private string GetString(byte[] buffer, int start, int count)
        {
            string strResult = "";

            for (int i = start; i < start + count; i++)
                strResult += "" + (char)buffer[i];

            return strResult.Trim();
        }
        #endregion

        #region ReadOffset
        private int ReadOffset(byte[] buffer)
        {
            return GetInt(buffer, 10, 4);
        }
        #endregion

        #region ReadSW
        private void ReadSW(int offsetToIndex, byte[] buffer)
        {
            int offset = offsetToIndex;

            while (true)
            {
                SWIndex swIndex = new SWIndex();
                swIndex.sequenceNumber = GetInt(buffer, offset, 1);
                offset += 1;
                swIndex.lengthOfIndex = GetInt(buffer, offset, 4);
                offset += 4;
                swIndex.offsetToStartOfIndex = GetInt(buffer, offset, 4);
                offset += 4;
                int empty = GetInt(buffer, offset, 4);
                offset += 5;
                swIndex.fileType = GetString(buffer, offset, 4);
                offset += 4;

                swIndex.sequenceNumbers = new int[swIndex.lengthOfIndex];

                int sOffset = swIndex.offsetToStartOfIndex;
                for (int i = 0; i < swIndex.lengthOfIndex; i++)
                {
                    swIndex.sequenceNumbers[i] = GetInt(buffer, sOffset, 2);
                    sOffset += 2;
                }

                _swIndexes.Add(swIndex);
                if (offset >= buffer.Length)
                    break;
            }
        }
        #endregion

        #region ReadPNG
        static void ReadPNG(string fileName, int offsetToIndex, byte[] buffer)
        {
            BinaryWriter bw = new BinaryWriter(File.Create(@"C:\Temp\eBook\" + fileName + ".PNG"));

            bw.Write(buffer, 32, offsetToIndex - 32);

            bw.Close();
        }
        #endregion

        #region Wrap
        private int Wrap(int value, int limit)
        {
            return (((value) < (limit)) ? (value) : ((value) - (limit)));
        }
        #endregion

        #region BitFileGetBitsInt
        private int BitFileGetBitsInt(ref bitStream stream, byte[] buffer, ref int bufferOffset, ref int bits, int count)
        {
            return BitFileGetBitsLE(ref stream, buffer, ref bufferOffset, ref bits, count);
        }
        #endregion

        #region BitFileGetBitsLE
        private int BitFileGetBitsLE(ref bitStream stream, byte[] buffer, ref int bufferOffset, ref int bits, int count)
        {
            UInt16 bytes = 0;
            byte shifts = 0;
            int offset = 0;
            int remaining = count;
            int returnValue = 0;

            while (remaining >= 8)
            {
                returnValue = BitFileGetChar(ref stream, buffer, ref bufferOffset);

                bytes = (byte)returnValue;
                remaining -= 8;
                offset++;
            }

            if (remaining != 0)
            {
                shifts = (byte)(8 - remaining);

                while (remaining > 0)
                {
                    returnValue = BitFileGetBit(ref stream, buffer, ref bufferOffset);

                    if (returnValue == EOF)
                        return EOF;

                    bytes <<= 1;
                    bytes |= (byte)(returnValue & 0x01);
                    remaining--;
                }
            }

            byte byte2 = 0;
            if (bytes >= 0xFF)
            {
                bits = bytes;
            }
            else
            {
                shifts = (byte)bytes;
                bytes = byte2;
                byte2 = shifts;
                string s = string.Format("0x{0:X}{1:X}", bytes, byte2);
                bits = Convert.ToUInt16(s, 16);
            }

            return count;
        }
        #endregion

        #region BitFileGetBit
        private int BitFileGetBit(ref bitStream stream, byte[] buffer, ref int offset)
        {
            if (offset == buffer.Length)
                return EOF;

            int returnValue = 0;

            if (stream.bitCount == 0)
            {
                returnValue = (int)buffer[offset++];

                stream.bitCount = 8;
                stream.bitBuffer = returnValue;
            }

            stream.bitCount--;

            returnValue = (stream.bitBuffer >> stream.bitCount);
            return (returnValue & 0x01);
        }
        #endregion

        #region BitFileGetChar
        private int BitFileGetChar(ref bitStream stream, byte[] buffer, ref int offset)
        {
            if (offset == buffer.Length)
                return EOF;

            int returnValue = (int)buffer[offset++];

            if (stream.bitCount == 0)
            {
                return returnValue;
            }

            byte tmp = (byte)((byte)returnValue >> stream.bitCount);
            tmp |= (byte)((stream.bitBuffer) << (8 - (stream.bitCount)));
            stream.bitBuffer = returnValue;

            returnValue = tmp;

            return returnValue;
        }
        #endregion

        #region ReadDATA_FRK
        private string ReadDATA_FRK(string fileName, int offsetToIndex, byte[] buffer)
        {
            int offset = 0;

            int OFFSET_BITS = 14;
            int LENGTH_BITS = 3;
            int WINDOW_SIZE = (1 << OFFSET_BITS);
            byte c = 0;
            int i = 0, nextChar = 1;
            encoded_string_t code = new encoded_string_t();
            int uncoded = 1;
            int MAX_UNCODED = 2;
            int MAX_CODED = ((1 << LENGTH_BITS) + MAX_UNCODED);
            string strFile = "";
            bitStream fIn = new bitStream();

            byte[] slidingWindow = new byte[WINDOW_SIZE];
            byte[] uncodedLookahead = new byte[WINDOW_SIZE];

            for (int x = 0; x < WINDOW_SIZE; x++)
                slidingWindow[x] = (byte)' ';

            for (int x = 0; x < WINDOW_SIZE; x++)
                uncodedLookahead[x] = (byte)' ';

            while (true)
            {
                if (offset == buffer.Length)
                    break;

                c = (byte)BitFileGetBit(ref fIn, buffer, ref offset);

                if (c == uncoded)
                {
                    if (offset == buffer.Length)
                        break;

                    c = (byte)BitFileGetChar(ref fIn, buffer, ref offset);

                    if (c == 0x0e || c == 0x0f || c == 0x15 || c == 0x16 || c == 0xA5)
                    {
                        strFile += "" + (char)0x0D;
                        strFile += "" + (char)0x0A;
                        strFile += "" + (char)0x7C;

                        if (c == 0x0F)
                        {
                            strFile += "" + (char)0x5E;
                            strFile += "" + (char)0x7C;
                        }
                    }
                    else
                    {
                        if (c == 0x14 || c == 0x0A || c == 0x0B)
                        {
                            strFile += "" + (char)0x0D;
                            strFile += "" + (char)0x0A;

                            if (c == 0x14)
                            {
                                strFile += "___";
                                strFile += "___";
                                strFile += "___";
                                strFile += "___";
                            }
                            else
                            {
                                if (c == 0x0B)
                                {
                                    strFile += "" + (char)0x0D;
                                    strFile += "" + (char)0x0A;
                                }
                            }
                        }
                        else
                        {
                            if (c == 0x13)
                            {
                                strFile += "_";
                            }
                            else if (c == 0xD5)
                            {
                                strFile += "'";
                            }
                            else
                            {
                                strFile += "" + (char)c;
                            }
                        }
                    }

                    if (c == 0x0D)
                    {
                        strFile += "" + (char)0x0A;
                    }

                    slidingWindow[nextChar] = c;
                    nextChar = Wrap(((int)nextChar + 1), WINDOW_SIZE);
                }
                else
                {
                    code.offset = 0;
                    code.length = 0;

                    int coffset = 0;
                    int clength = 0;

                    if (BitFileGetBitsInt(ref fIn, buffer, ref offset, ref coffset, OFFSET_BITS) == EOF)
                        break;

                    if (BitFileGetBitsInt(ref fIn, buffer, ref offset, ref clength, LENGTH_BITS) == EOF)
                        break;

                    if (coffset == 0 && clength == 0)
                        break;

                    code.offset = coffset;
                    code.length = clength;
                    code.length += MAX_UNCODED + 1;

                    for (i = 0; i < code.length; i++)
                    {
                        c = slidingWindow[Wrap((code.offset + i), WINDOW_SIZE)];
                        if (code.offset + code.length > nextChar)
                        {
                            if (i >= nextChar - code.offset)
                            {
                                c = uncodedLookahead[i - nextChar + code.offset];
                            }
                        }
                        if (c == 0x0e || c == 0x0f || c == 0x15 || c == 0x16 || c == 0xA5)
                        {
                            strFile += "" + (char)0x0D;
                            strFile += "" + (char)0x0A;
                            strFile += "" + (char)0x7C;
                            if (c == 0x0f)
                            {
                                strFile += "" + (char)0x5E;
                                strFile += "" + (char)0x7C;
                            }
                        }
                        else
                        {
                            if (c == 0x14 || c == 0x0A || c == 0x0B)
                            {
                                strFile += "" + (char)0x0D;
                                strFile += "" + (char)0x0A;
                                if (c == 0x14)
                                {
                                    strFile += "___";
                                    strFile += "___";
                                    strFile += "___";
                                    strFile += "___";
                                }
                                else
                                {
                                    if (c == 0x0B)
                                    {
                                        strFile += "" + (char)0x0D;
                                        strFile += "" + (char)0x0A;
                                    }
                                }
                            }
                            else
                            {
                                if (c == 0x13)
                                {
                                    strFile += "_";
                                }
                                else
                                {
                                    strFile += "" + (char)c;
                                }
                            }
                        }
                        if (c == 0x0d)
                        {
                            strFile += "" + (char)0x0A;
                        }
                        uncodedLookahead[i] = c;
                    }

                    /* write out decoded string to sliding window */
                    for (i = 0; i < code.length; i++)
                    {
                        slidingWindow[(nextChar + i) % WINDOW_SIZE] =
                            uncodedLookahead[i];
                    }

                    nextChar = Wrap((nextChar + code.length), WINDOW_SIZE);
                }
            }

            return strFile;
        }
        #endregion

        #endregion

        #region ReverseIMP
        public void ReverseIMP()
        {
            byte[] buffer = null;

            BinaryReader br = new BinaryReader(File.OpenRead(IMPFilePath));

            _header = new IMPHeader();

            #region Read Header
            buffer = new byte[48];

            _header.version = ReadInt(ref br, 2); // IMP Version

            buffer = br.ReadBytes(8);
            _header.constant = ASCIIEncoding.ASCII.GetString(buffer).Trim(); // CONSTANT

            buffer = br.ReadBytes(8); // UNKNOWN

            _header.fileCount = ReadInt(ref br, 2); // Files in RES Directory

            _header.dictionaryLength = ReadInt(ref br, 2); // Dictionary Length (RES File Length)

            _header.bytesRemaining = ReadInt(ref br, 2); // Bytes Remaining

            buffer = br.ReadBytes(4); // UNKNOWN

            buffer = br.ReadBytes(4); // UNKNOWN

            _header.compression = (ReadInt(ref br, 4) == 1); // Is compressed ?

            _header.encryption = (ReadInt(ref br, 4) == 1); // Is encrypted ?

            buffer = br.ReadBytes(4);
            _header.zoomState = ReadInt(ref br, 4); // IMP Book Zoom State

            buffer = br.ReadBytes(4); // UNKNOWN
            #endregion

            _bookProperties = new IMPBookProperties();

            #region Read Book Properties
            _bookProperties.ID = ReadString(ref br);
            _bookProperties.Bookshelf_Category = ReadString(ref br);
            _bookProperties.Subcategory = ReadString(ref br);
            _bookProperties.Title = ReadString(ref br);
            _bookProperties.LastName = ReadString(ref br);
            _bookProperties.MiddleName = ReadString(ref br);
            _bookProperties.FirstName = ReadString(ref br);
            _bookProperties.ResFile = ASCIIEncoding.ASCII.GetString(br.ReadBytes(_header.dictionaryLength));
            #endregion

            _tableOfContents = new IMPTableContents();

            #region Read Table of Contents
            for (int i = 0; i < _header.fileCount; i++)
            {
                IMPTableContentsItem item = new IMPTableContentsItem();

                buffer = new byte[4];
                buffer = br.ReadBytes(4);
                item.fileName = ASCIIEncoding.ASCII.GetString(buffer).Trim();

                if (item.fileName == "")
                    item.fileName = "DATA.FRK";

                buffer = br.ReadBytes(4);

                item.fileSize = ReadInt(ref br, 4); // (int)(buffer[0] | buffer[1] | buffer[2] | buffer[3]);

                buffer = br.ReadBytes(4);
                item.fileType = ASCIIEncoding.ASCII.GetString(buffer).Trim();

                buffer = br.ReadBytes(4);

                System.Diagnostics.Debug.WriteLine(item.ToString());

                _tableOfContents.table.Add(item);
            }
            #endregion

            _resFiles = new IMPRESFiles();

            #region Read Files
            for (int i = 0; i < _header.fileCount; i++)
            {
                IMPRESFile file = new IMPRESFile();
                buffer = new byte[4];

                buffer = br.ReadBytes(4);
                file.fileName = ASCIIEncoding.ASCII.GetString(buffer).Trim();

                if (file.fileName == "")
                    file.fileName = "DATA.FRK";

                buffer = br.ReadBytes(4);

                file.fileSize = ReadInt(ref br, 4);

                buffer = br.ReadBytes(4);
                file.fileType = ASCIIEncoding.ASCII.GetString(buffer).Trim();

                if (file.fileType == "")
                    file.fileType = "DATA.FRK";

                buffer = br.ReadBytes(4);

                file.buffer = br.ReadBytes(file.fileSize);

                file.currentPosition = (int)br.BaseStream.Position;

                _resFiles.list.Add(file);
            }
            #endregion

            br.Close();

            foreach (IMPRESFile file in _resFiles.list)
            {
                int indexOffset = ReadOffset(file.buffer);

                switch (file.fileType)
                {
                    case "!!sw":
                        ReadSW(indexOffset, file.buffer);
                        break;
                }
            }
        }
        #endregion

        #region ReverseResourceFile
        public void ReverseResourceFile()
        {
            try
            {
                foreach (IMPRESFile file in _resFiles.list)
                {
                    int indexOffset = ReadOffset(file.buffer);

                    switch (file.fileType)
                    {
                        case "!!sw":
                            //ReadSW(indexOffset, file.buffer);
                            break;
                        case "PNG":
                            break;
                        case "DATA.FRK":
                            IMPResourceFile = ReadDATA_FRK(file.fileName, indexOffset, file.buffer);
                            break;
                        case "!!cm":
                            //ReadCM(indexOffset, file.buffer);
                            break;
                    }
                }
            }
            catch (Exception exc)
            {
                System.Diagnostics.Debug.WriteLine(exc.ToString());
            }
        }
        #endregion

        public MemoryStream GetImage(ref IMPRESFile file)
        {
            int indexOffset = ReadOffset(file.buffer);
            MemoryStream ms = new MemoryStream();
            ms.Write(file.buffer, 32, indexOffset - 32);
            ms.Seek(0, SeekOrigin.Begin);
            return ms;
        }
    }
}
