#!/usr/bin/env python3
# Based on gftool's list-panose
# Copyright 2016 The Fontbakery Authors
# Copyright 2017 The Google Font Tools Authors
# And gftool's nametable-from-filename
# Copyright 2013,2016 The Font Bakery Authors
# Copyright 2017 The Google Font Tools Authors
# As well as fontforge's SFDefaultOS2Info() @ https://github.com/fontforge/fontforge/blob/master/fontforge/tottf.c
# Copyright (C) 2000-2012 by George Williams
#
# c.f., also https://github.com/rsms/inter/blob/master/misc/tools/fontinfo.py
#
# For the Panose keys & values, c.f., https://rsms.me/fonttools-docs/_modules/fontTools/ttLib/tables/O_S_2f_2.html
# & https://github.com/fontforge/fontforge/blob/f28b9e6b54acbae1cb8435d4ad6adf59fc8588db/contrib/fonttools/showttf.c#L745
#
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Id: fix-panose.py 17121 2020-04-20 23:57:05Z NiLuJe $
#

import argparse
from enum import Enum, unique
from fontTools import ttLib
from gftools.constants import (PLATFORM_ID__WINDOWS, NAMEID_FULL_FONT_NAME)

# Dumb down the bWeight logic to make Kobos happy...
FOR_KOBO = False

# Panose constants
@unique
class PanoseID(Enum):
	bFamilyType      = 0	# Family
	bSerifStyle      = 1	# Serif Type
	bWeight          = 2	# Weight
	bProportion      = 3	# Proportion
	bContrast        = 4	# Contrast
	bStrokeVariation = 5	# Stroke Variation
	bArmStyle        = 6	# Arm Style
	bLetterForm      = 7	# Letterform
	bMidline         = 8	# Midline
	bXHeight         = 9	# X-Height

PanoseConstants = {
	PanoseID.bFamilyType      : [ "Any", "No Fit", "Text & Display", "Script", "Decorative", "Pictoral" ],
	PanoseID.bSerifStyle      : [ "Any", "No Fit", "Cove", "Obtuse Cove", "Square Cove", "Obtuse Square Cove", "Square", "Thin", "Bone", "Exaggerated", "Triangle", "Normal Sans", "Obtuse Sans", "Perp Sans", "Flared", "Rounded" ],
	PanoseID.bWeight          : [ "Any", "No Fit", "Very Light", "Light", "Thin", "Book", "Medium", "Demi", "Bold", "Heavy", "Black", "Nord" ],
	PanoseID.bProportion      : [ "Any", "No Fit", "Old Style", "Modern", "Even Width", "Expanded", "Condensed", "Very Expanded", "Very Condensed", "Monospaced" ],
	PanoseID.bContrast        : [ "Any", "No Fit", "None", "Very Low", "Low", "Medium Low", "Medium", "Medium High", "High", "Very High" ],
	PanoseID.bStrokeVariation : [ "Any", "No Fit", "No Variation", "Gradual/Diagonal", "Gradual/Transitional", "Gradual/Vertical", "Gradual/Horizontal", "Rapid/Vertical", "Rapid/Horizontal", "Instant/Vertical", "Instant/Horizontal" ],
	PanoseID.bArmStyle        : [ "Any", "No Fit", "Straight Arms/Horizontal", "Straight Arms/Wedge", "Straight Arms/Vertical", "Straight Arms/Single Serif", "Straight Arms/Double Serif", "Non-Straight Arms/Horizontal", "Non-Straight Arms/Wedge", "Non-Straight Arms/Vertical", "Non-Straight Arms/Single Serif", "Non-Straight Arms/Double Serif" ],
	PanoseID.bLetterForm      : [ "Any", "No Fit", "Normal/Contact", "Normal/Weighted", "Normal/Boxed", "Normal/Flattened", "Normal/Rounded", "Normal/Off-Center", "Normal/Square", "Oblique/Contact", "Oblique/Weighted", "Oblique/Boxed", "Oblique/Flattened", "Oblique/Rounded", "Oblique/Off-Center", "Oblique/Square" ],
	PanoseID.bMidline         : [ "Any", "No Fit", "Standard/Trimmed", "Standard/Pointed", "Standard/Serifed", "High/Trimmed", "High/Pointed", "High/Serifed", "Constant/Trimmed", "Constant/Pointed", "Constant/Serifed", "Low/Trimmed", "Low/Pointed", "Low/Serifed" ],
	PanoseID.bXHeight         : [ "Any", "No Fit", "Constant/Small", "Constant/Standard", "Constant/Large", "Ducking/Small", "Ducking/Standard", "Ducking/Large" ],
}

parser = argparse.ArgumentParser(description='Automatically set Panose of the fonts, like FontForge\'s Default checkbox')
parser.add_argument('font', nargs="+")

# Helper function to set a specific Panose attribute to a specific value
def set_panose(panose, key, val):
	print("Setting {:>2} => {:<16} to {} => {}".format(key.value, key.name, val, PanoseConstants[key][val]))
	setattr(panose, key.name, val)

def main():
	options = parser.parse_args()

	for i, font in enumerate(options.font):
		ttfont = ttLib.TTFont(font)

		# We'll need the full fontname...
		fontname = None
		for field in ttfont['name'].names:
			if field.nameID == NAMEID_FULL_FONT_NAME and field.platformID == PLATFORM_ID__WINDOWS:
				fontname = field.toUnicode()
		# Abort early if that failed...
		if fontname is None:
			raise TypeError("Failed to compute full font name (try gftools nametable-from-filename)!")
		print("Processing font: {}".format(fontname))
		# Make it lowercase to make our comparisons easier
		fontname = fontname.lower()

		# Start by setting every field to "Any"
		for k in list(ttfont['OS/2'].panose.__dict__.keys()):
			val = getattr(ttfont['OS/2'].panose, k)
			key = PanoseID[k]
			print("Clearing {} => {:<16} (was {} => {})".format(key.value, key.name, val, PanoseConstants[key][val]))
			setattr(ttfont['OS/2'].panose, k, 0)

		# FF SFDefaultOS2Simple
		set_panose(ttfont['OS/2'].panose, PanoseID['bFamilyType'], 2)
		set_panose(ttfont['OS/2'].panose, PanoseID['bWeight'], 5)
		set_panose(ttfont['OS/2'].panose, PanoseID['bProportion'], 3)

		# FF SFDefaultOS2Info
		# pfmfamily = 0x40
		if "script" in fontname:
			set_panose(ttfont['OS/2'].panose, PanoseID['bFamilyType'], 3)

		# FF OS2WeightCheck
		weight = ttfont['OS/2'].usWeightClass
		# NOTE: We optionally limit ourselves to Bold & Book to make the Kobo renderers happy...
		if FOR_KOBO:
			# Demi Bold & up => Bold, otherwise => Book
			if weight >= 600:
				set_panose(ttfont['OS/2'].panose, PanoseID['bWeight'], 8)
			else:
				set_panose(ttfont['OS/2'].panose, PanoseID['bWeight'], 5)
		else:
			if weight >= 950:
				set_panose(ttfont['OS/2'].panose, PanoseID['bWeight'], 11)
			elif weight >= 900:
				set_panose(ttfont['OS/2'].panose, PanoseID['bWeight'], 10)
			elif weight >= 800:
				set_panose(ttfont['OS/2'].panose, PanoseID['bWeight'], 9)
			elif weight >= 700:
				set_panose(ttfont['OS/2'].panose, PanoseID['bWeight'], 8)
			elif weight >= 600:
				set_panose(ttfont['OS/2'].panose, PanoseID['bWeight'], 7)
			elif weight >= 500:
				set_panose(ttfont['OS/2'].panose, PanoseID['bWeight'], 6)
			elif weight >= 400:
				set_panose(ttfont['OS/2'].panose, PanoseID['bWeight'], 5)
			elif weight >= 300:
				set_panose(ttfont['OS/2'].panose, PanoseID['bWeight'], 4)
			elif weight >= 200:
				set_panose(ttfont['OS/2'].panose, PanoseID['bWeight'], 3)
			elif weight >= 100:
				set_panose(ttfont['OS/2'].panose, PanoseID['bWeight'], 2)
			else:
				set_panose(ttfont['OS/2'].panose, PanoseID['bWeight'], 0)

		# FF SFDefaultOS2Info
		if "ultra" in fontname and "condensed" in fontname:
			set_panose(ttfont['OS/2'].panose, PanoseID['bProportion'], 8)
		elif "extra" in fontname and "condensed" in fontname:
			set_panose(ttfont['OS/2'].panose, PanoseID['bProportion'], 8)
		elif "semi" in fontname and "condensed" in fontname:
			set_panose(ttfont['OS/2'].panose, PanoseID['bProportion'], 6)
		elif "condensed" in fontname or "narrow" in fontname:
			set_panose(ttfont['OS/2'].panose, PanoseID['bProportion'], 6)
		elif "ultra" in fontname and "expanded" in fontname:
			set_panose(ttfont['OS/2'].panose, PanoseID['bProportion'], 7)
		elif "extra" in fontname and "expanded" in fontname:
			set_panose(ttfont['OS/2'].panose, PanoseID['bProportion'], 7)
		elif "semi" in fontname and "expanded" in fontname:
			set_panose(ttfont['OS/2'].panose, PanoseID['bProportion'], 5)
		elif "expanded" in fontname:
			set_panose(ttfont['OS/2'].panose, PanoseID['bProportion'], 5)

		# pfmfamily = 0x30 (PitchFamily)
		# NOTE: Assuming fix-isfixedpitch was run first, and/or font is sane
		if ttfont['post'].isFixedPitch == 1:
			set_panose(ttfont['OS/2'].panose, PanoseID['bProportion'], 9)

		new_font = font + ".fix"
		print("Saving font to {}".format(new_font))
		ttfont.save(new_font)

if __name__ == '__main__':
	main()
