import sys, time, os
from threading import Event

from calibre.utils.ipc.server import Server
from calibre.utils.ipc.job import ParallelJob
from calibre.utils.logging import Log

#import from this plugin
from calibre_plugins.tts_to_mp3_plugin.utils import create_single_wav, create_single_mp3
from calibre_extensions.winsapi import ISpVoice


def get_job_details(job):
    '''
    Convert the job result into a set of parameters including a detail message
    summarising the success of the overall jobs action.
    '''
    good_files, bad_files = job.result
    if not hasattr(job, 'html_details'):
        job.html_details = job.details
    det_msg = []
    for pad_track, name, errmsg, book_label in sorted(bad_files):
        msg = '%s %s (%s)'% (pad_track, name, errmsg)
        det_msg.append(msg)
    det_msg = '\n'.join(det_msg)
    return good_files, bad_files, det_msg


# Functions to perform epub mod/show action using worker jobs
def do_book_action_worker(files_to_proc, bad_files, cpus, notification=lambda x,y:x):
    '''
    Master job, to launch child jobs to create an MP3
        for each selected spine file in a book.
    This is run as a worker job in the background to keep the UI more
    responsive and get around the memory leak issues as it will launch
    a child job for each spine file as a worker process
    '''
    server = Server(pool_size=cpus)

    # Queue all the jobs
    for dmp3meta, djobmeta in files_to_proc:
        mp3_dir = dmp3meta.get('mp3_dir')
        book_id = dmp3meta.get('book_id')
        pad_track = dmp3meta.get('pad_track')
        jobdesc = '%d_%s' % (book_id, pad_track)
        
        book_label = djobmeta.get('book_label')
        container_root = djobmeta.get('container_root')
        dest_dir = djobmeta.get('dest_dir')
        
        job = ParallelJob(
            'arbitrary', 
            jobdesc, 
            done=None, 
            args=('calibre_plugins.tts_to_mp3_plugin.jobs', 
                'do_bookfile_single_worker',
                (dmp3meta, djobmeta))
            )
        job._dmp3meta = dmp3meta
        #job._djobmeta = djobmeta
        server.add_job(job)

    # This server is an arbitrary_n job, so there is a notifier available.
    # Set the % complete to a small number to avoid the 'unavailable' indicator
    notification(0.01, 'Creating MP3s...')

    # dequeue the job results as they arrive, saving the results
    total = len(files_to_proc)
    count = 0
    good_files = []
    
    while True:
        job = server.changed_jobs_queue.get()
        # A job can 'change' when it is not finished, 
        # e.g if it produces a notification. Ignore these.
        job.update()
        if not job.is_finished:
            continue
            
        # A job really finished. Get the information.
        result = job.result
        dmp3meta = job._dmp3meta
        
        book_id = dmp3meta.get('book_id')
        pad_track = dmp3meta.get('pad_track')
        name = dmp3meta.get('name')
        wordcount = dmp3meta.get('wordcount')
        
        count = count + 1
        
        # Add this job's output to the current log
        print('\nLogfile for %s / %s / %s' % (book_id, pad_track, name))
        resok, restext = result
        print(restext)
        #print(job.details) # too much output from lame.exe
        if resok:
            good_files.append((pad_track, name, restext, wordcount, book_label, mp3_dir, container_root, dest_dir))
        else:
            bad_files.append((pad_track, name, restext, book_label))

        notification(float(count)/total, 'MP3s created={0}'.format(len(good_files)))
        
        if count >= total:
            # All done!
            break

    server.close()
    # return the 2 lists as the job result
    return good_files, bad_files


def do_bookfile_single_worker(dmp3meta, djobmeta):
    # Child job, create MP3 for this file, when run as worker job
    log = Log()
    abort = Event()
    return process_file(log, abort, dmp3meta, djobmeta)


def process_file(log, abort, dmp3meta, djobmeta):
    # Actually perform the work
    if abort.is_set():
        errmsg = '*** Job cancelled ***'
        log.error(errmsg)
        return False, errmsg
    
    res = ''
    cycle_log = []
    voice_name = djobmeta.get('voice_name')
    voice_rate = djobmeta.get('voice_rate')
    voice_id = djobmeta.get('voice_id')
    lame_path = djobmeta.get('lame_path')
    
    wav_file_name = dmp3meta.get('wav_file_name')
    mp3_file_name = dmp3meta.get('mp3_file_name')
        
    try:
        spVoice = ISpVoice()
        spVoice.set_current_voice(voice_id)
        spVoice.set_current_rate(voice_rate)
                
        create_single_wav(dmp3meta, spVoice, cycle_log.append)
        create_single_mp3(dmp3meta, lame_path, cycle_log.append)
        
        res = '\n'.join(cycle_log)
        log.info(res)
    except ValueError as e:
        log.error('MP3 failed %s' % mp3_file_name)
        log.exception('ERROR: %s' % e)
    except:
        log.error('MP3 failed %s' % mp3_file_name)
        log.exception('ERROR: %s' % sys.exc_info()[:2])
        
    if not os.path.exists(mp3_file_name):
        errmsg = "*** MP3 not created: %s" % mp3_file_name
        log.error(errmsg)
        return False, errmsg
        
    return True, res
