#!/usr/bin/env python
# -*- coding: utf-8 -*-
# vim:ts=4:sw=4:softtabstop=4:smarttab:expandtab

"""
Copyright (c) 2017 John Crew
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
for commercial and non-commercial purposes are permitted provided that the
following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
this list of conditions and the following disclaimer in the documentation and/or
other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
"""

"""
HOW TO USE THIS PLUGIN
1. In the main module insert the following:
    from UpdateChecker import CheckForUpdates, SetUpdateTimes

2. In the def run(bk) section insert the following code:
  root = tk.Tk()
  prefs = bk.getPrefs()
  
  #Specify the url for the web page that contains the plugin eg:
  url = "https://www.mobileread.com/forums/showthread.php?t=285771"
  #and then insert:
  CheckForUpdates(root, prefs, url)
  ...
  ...more code
  ...
  # Save prefs to back to json
  bk.savePrefs(prefs)

IMPORTANT: CheckForUpdates(url, prefs, root) must be called somewhere between
prefs = bk.getPrefs() and bk.savePrefs(prefs) in the main module

3.  Enusre that the dialog calling this program is not modal ie avoid code
such as:
    self.top.grab_set() 				#Make dialog box modal

4. Optional: Include similar code to that below to allow the user to set
reminder times:
        tk.Button(self.dlgframe, text="Set time before checking for updates", command=lambda: SetUpdateTimes(self.prefs, self.top)).grid(column=xxx, row=xxx)

The time in hours between checks for an update is governed by the time (in
hours) stored in prefs['CheckInHours']. The default is 24 hours.

The time in minutes between reminders that an update exists is governed by the
time (in minutes) stored in prefs['RemindInMinutes']. The default is 60 mins.
"""

import tkinter as tk                #Essential for custom dialog box using tk. commands
import tkinter.ttk as ttk           #Essential for ttk. commands
from tkinter import *               #Essential for root = Tk()

from sigil_bs4 import BeautifulSoup
import re
import urllib.request as urlRead
import os, inspect     #needed to determine the plugin path
from datetime import datetime, timedelta

from GenUtils import 	centerWindow	#Needed to centerWindow

import webbrowser

#The following keys will be accepted for the entry boxes that requires a time to
#be entered. Other key presses will cause the computer to beep
Safekeys=('Up', 'Down', 'Left',  'Home', 'End', 'Right', 'Control_R',
'Control_L', 'Alt_R', 'Alt_L', 'BackSpace', 'Tab', 'Escape', 'Return', 'Delete',
 'Win_L', 'F4', 'Shift_L', 'Shift_R')

def Close(win):
    #Used to close the window for SetUpdateTimes and ShowUpdateDlg
    win.destroy()

def SetUpdateTimes(prefs, root):
    #This function allows the user to set times between updates
    
    def SaveUpdateTimes(prefs, top1):
        #This function stores the times between checking for update and
        #reminders in the preferences dictionary
        if int(HoursToWait.get()) <1:
            HoursToWait.set(1) #Min time to delay before checking for update
        if int(HoursToWait.get()) > 168: #168 hrs = 1 week
            HoursToWait.set(168)  # Max time delay before checking for update is 1 week
        prefs['CheckInHours'] = int(HoursToWait.get())
        
        if int(MinutesToRemind.get()) <5:
            MinutesToRemind.set(5) #Minimum mins to delay before displaying reminder that an update is available
        if int(MinutesToRemind.get()) > 1440: #1440 hrs = 1 day
            MinutesToRemind.set(1440)  # Max  mins to delay before displaying reminder that an update is available
        prefs['RemindInMinutes'] = int(MinutesToRemind.get())
        
        Close(top1)
        return
    
    def CheckKey(event):
        #This function will allow the entry boxes for entering times to accept
        #numeric keys and the list of keys in Safekeys; other keys will cause
        #the computer to beep
        if (event.char not in ('0', '1', '2', '3', '4', '5', '6', '7', '8', '9')) and (event.keysym not in Safekeys):
            txtHours.bell() #Chime if an invalid key is pressed
            return 'break' 
     
    prefs.defaults['CheckInHours'] = 24
    prefs.defaults['RemindInMinutes'] = 60
            
    top1 = tk.Toplevel(root)
    top1.title("Set update timing")
    top1.grab_set()  # Make dialog box modal
    DlgFrame = ttk.Frame(top1, padding="15 15 12 12")  # replace root with top
    DlgFrame.grid(column=0, row=0, sticky=(N, W, E, S))
    DlgFrame.columnconfigure(0, weight=1)
    DlgFrame.rowconfigure(0, weight=1)
    
    tk.Label(DlgFrame, text="Number of hours before checking for a new update = ").grid(column=0, row=5, sticky=E)

    HoursToWait = StringVar()
    HoursToWait.set(int(prefs['CheckInHours'])) #Initialise time in hours - Get this later from the prefs file

    txtHours = Entry(DlgFrame, textvariable=HoursToWait, width = 5)
    txtHours.grid(column=1, row=5, sticky = W)     #Error may occur if this is attached to previous line            
    txtHours.bind('<KeyPress>', CheckKey)
    
    tk.Label(DlgFrame, text="Number of minutes before reminding that there is a new update available = ").grid(column=0, row=10, sticky=E)

    MinutesToRemind = StringVar()
    MinutesToRemind.set(int(prefs['RemindInMinutes'])) #Initialise time in mins

    txtMins = Entry(DlgFrame, textvariable=MinutesToRemind, width = 5)
    txtMins.grid(column=1, row=10, sticky = W)     #Error may occur if this is attached to previous line            
    txtMins.bind('<KeyPress>', CheckKey)
    
    tk.Button(DlgFrame, text="Close", command=lambda: SaveUpdateTimes(prefs, top1), width=30).grid(column=2, row=20)
    
    centerWindow(top1)

def OpenWebPage(top, url):
    webbrowser.open_new_tab(url)
    Close(top)

def ShowUpdateDlg(root, prefs, url):
    top = tk.Toplevel(root)
    top.title("Update available")
    top.grab_set()  # Make dialog box modal
    DlgFrame = ttk.Frame(top, padding="15 15 12 12") 
    DlgFrame.grid(column=0, row=0, sticky=(N, W, E, S))
    DlgFrame.columnconfigure(0, weight=1)
    DlgFrame.rowconfigure(0, weight=1)

    tk.Label(DlgFrame, text="An update for this plugin is available", fg = "blue", bg = "yellow",font=(None, 15)).grid(column=0, row=0, sticky=W)
    
    tk.Label(DlgFrame, text="").grid(column=0, row=10, sticky=W) #Blank row
    
    btnOpenPage = tk.Button(DlgFrame, text="Go to web page", command=lambda: OpenWebPage(top, url), width=30)
    btnOpenPage.grid(column=0, row=20)
    btnRemindLater = tk.Button(DlgFrame, text="Remind me later", command=lambda: Close(top), width=30)
    btnRemindLater.grid(column=2, row=20)
        
    centerWindow(top)
    top.attributes("-topmost", True)  # Force this window to be on top of all others
                
def checkVersion(root, prefs, url):
    PublishedVersion=None
    installedVersion = None

    try:        
        #Get the path for the installed plugin
        pluginPath = os.path.dirname(os.path.abspath(inspect.getfile(inspect.currentframe())))
        pluginFile = os.path.join(pluginPath, "plugin.xml")
    except:
        print("Cannot find the XML file for the plugin")
        return (False)

    try:
         #Open the XML plugin file
        f=open(pluginFile, 'r', encoding="utf-8")
    except IOError:
        print("Error opening the plugin file")
        return (False)

    #Use the XML file to obtain the version number and compile the search expression
    xmlData=f.read()                                        #read the XML file
    v=re.search(r"<version>([^<]*)</version>", xmlData)     #and search for the version number
    installedVersion=v.group(1)                             #store the version number
    n=re.search(r"<name>([^<]*)</name>", xmlData)           #Search for the name of the plugin
    searchPattern = n.group(1) + "_v(.*?).zip"              #and use it to compile a search expression
    f.close                                                 #and close the XML file

    try:
        #read in the webpage where the zip file is located
        r = urlRead.urlopen(url).read()
    except:
        print("Could not access ", url)
        return (False)

    #Get the version number of the published plugin from the webpage
    soup = BeautifulSoup(r, 'html.parser')
    soup.prettify()
    urltext=soup.get_text() #Get the text on the webpage
    
    #Use the search expression to get the version number of the published plugin
    m=re.search(searchPattern, urltext) #Search webpage for the plugin
    if m == None:        #if the plugin is not found on the webpage
        print("Match not found") #print error message and return
        return (False)
    PublishedVersion=m.group(1)   #otherwise get the version number from the web

    print("Published version ", PublishedVersion)           #print version numbers of the published and
    print("Installed version ", installedVersion)           #installed plugins
    
    #Compare the plugins and display a message
    if installedVersion == PublishedVersion:  
        print("You have the latest version of this plugin")
        return (True)
    else:
        print("An update for this plugin is available at ",url)
        prefs = ShowUpdateDlg(root, prefs, url)
        return (False)

def CheckForUpdates(root, prefs, url):
    CurrentTime = datetime.now()
    # Set defaults if json pref file does not exist
    prefs.defaults['CheckInHours'] = 24 #Check every 24 hours for an update
    prefs.defaults['RemindInMinutes'] = 60
    prefs.defaults['NextCheckTime'] = str(CurrentTime)
    NextCheckdt = datetime.strptime(prefs['NextCheckTime'], "%Y-%m-%d %H:%M:%S.%f")

    if datetime.now() > NextCheckdt:
        print('Checking for updates...')
        IsLatestInstalled= checkVersion(root, prefs, url)
        if IsLatestInstalled == False: #Update is available
            #Set the next check to be much shorter than the CheckInHours value
            #in case user does not update the program this time
            NextCheckdt = CurrentTime + timedelta(minutes=prefs['RemindInMinutes'])
        else:
            print("You have the latest version of this plugin")
            NextCheckdt = CurrentTime + timedelta(hours=prefs['CheckInHours'])
            
    prefs['NextCheckTime'] = str(NextCheckdt)
    #print("Will check for updates/remind update is available at: ", prefs['NextCheckTime'])
    print("Will check for updates/remind update is available at: ", prefs['NextCheckTime'])
