#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
This module processes hyphenated words in an ePub book.
First, it reads allowed hyphenated words from a UTF text file and stores them in
the list, keepHyphens

Then it examines the ePub book for hyphenated words. For each pair of hyphenated
words that is found, the plugin determines whether the pair are already present
in keepHyphens. If not, then the pair of hyphenated words are added to the list
checkList.
After examing the ePub file, the plugin appends each pair of hyphenated words in
checklist to the UTF text file.
"""

"""
Copyright (c) 2019 John Crew

All rights reserved.

Redistribution and use in source and binary forms, with or without modification, for commercial and non-commercial purposes are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.

3. Neither the name of the copyright holder nor contributors may be used to endorse or promote products derived from this software.

4. The source code, and any derived source code, must be made available to users of this software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
"""

from tkinter import Tk, ttk, IntVar, messagebox, Frame, N, E, S, W, NS, EW, Scrollbar, Label, Button, SUNKEN, DISABLED, filedialog, Text, Checkbutton, Toplevel

#from tkinter import Tk, ttk, StringVar, IntVar, messagebox, Frame, GROOVE, N, E, S, W, NS, EW,Canvas, Scrollbar, Label, Button, SUNKEN, DISABLED, filedialog, Text, Checkbutton, Toplevel

import re

from GenUtils import centerWindow

class dlgHyphenation:
	dlgTop=""

	def __init__(self, prefs):
		self.keepHyphensList = []	#List that will contain allowed hyphenated words
		self.checkList = []	#List to hold hyphenated words that are not found in the list keepHyphens that contains acceptable hyphenated words
		self.cbStatesList = [] #List to hold state of checkbox marks
		self.HyphenFileName = prefs['HyphenatedWordFileName'] #Initialise file name

	def OpenHyphenExceptionFile(self):
		"""
		This function populates a list called keepHyphens from a list of hyphenated words in a text file.
		The text file is located at F:\\Programming\\Python\\keepHyphens.txt.
		It is called by an external module?
		"""

		self.keepHyphensList = []	#Initalise list that will contain allowed hyphenated words

		try:
			f=open(self.HyphenFileName, 'r', encoding="utf-8")

		except IOError:
			showerror(title='WARNING', message='Cannot read the file with allowed hyphenated words', detail='Please ensure you have a file containing allowed hyphenated words if you want to retain hyphenation for selected words', icon='"warning"')
			return(False)

		else:
			if f != None:
				for line in f:
					self.keepHyphensList.append(line.rstrip('\n'))
				f.close()
				return(True)
			else:
				return(False)

	def UpdateHyphenFile(self):
		try:
			f=open(self.HyphenFileName, 'a', encoding='utf-8')
		except IOError:
			showerror(title='WARNING', message='Cannot write file with allowed hyphenated words', detail='Your changes cannot be saved', icon='"warning"')
			print("Cannot open file with allowed hyphenated words")
		else:
			for i, item in enumerate(self.checkList): #Enumerate is better than for i in range....
				if self.cbStatesList[i].get() ==1: #If this item is ticked...
					f.write("\n")
					f.write(item)	#Save it to the file of allowed hyphenated words
			f.close()
		self.Close()

	def setStatesOn(self):
		for i, item in enumerate(self.checkList):
			self.cbStatesList[i].set('1')

	def setStatesOff(self):
		global checkList, vars
		for i, item in enumerate(self.checkList):
			self.cbStatesList[i].set('0')

	def FindHyphens(self, html):
		#Search the eBook for hyphenated words; if hyphenated words are not in the list keepHyphens
		#then add them to the temporary list, checkList, for the user to check

		found1 = re.search(r"(?s)(\w+)[ ]?-[ ]?(\w+)(?![^<>]*>)(?!.*<body[^>]*>)", html)
		if found1 is not None:
			for m in re.finditer(r"(?s)(\w+)[ ]?-[ ]?(\w+)(?![^<>]*>)(?!.*<body[^>]*>)", html):
				hyphenatedWord = m.group(0)
				hyphenatedWord = hyphenatedWord.lower()
				if hyphenatedWord not in self.keepHyphensList and hyphenatedWord not in self.checkList:
					self.checkList.append(hyphenatedWord)

	def ShowCheckBoxes(self, parent):
		"""
		This procedure will display the dialog box for selecting hyphenated
		words to be retained. It iterates through checkList and displays each
		hyphenated word in this list with a checkbox
		"""
		self.top = Toplevel(parent)
		dlgHyphenation.dlgTop=self.top 		#Needed for calling code to destroy the dialog window

		self.top.grab_set() 				#Make dialog box modal
		self.top.title("Hyphenated words in this ePub")

		mainframe = ttk.Frame(self.top, padding="15 15 12 12", borderwidth=5, relief = SUNKEN, height=400, width=800)
		mainframe.grid(column=0, row=0, sticky=(N, W, E, S)) #Associate a grid with the frame to hold widgets
		mainframe.columnconfigure(0, weight=1, minsize=400)
		mainframe.rowconfigure(0, weight=1, minsize=100)

		Label(mainframe, text="First column, first row").grid(column=0, row=0, sticky=W)
		#ttk.Label(mainframe, text="First column, first row").grid(column=0, row=0, sticky=W)

		btnframe = ttk.Frame(self.top, padding="15 15 12 12", borderwidth=5, relief = SUNKEN, height=400, width=800)
		btnframe.grid(column=0, row=1, sticky=(N, W, E, S))

		Button(btnframe, text='Save selection', command = self.UpdateHyphenFile).grid(column=0, row=1, sticky=W)
		Button(btnframe, text='Tick all', command=self.setStatesOn).grid(column=2, row=1, sticky=W)
		Button(btnframe, text='Untick all', command=self.setStatesOff).grid(column=3, row=1, sticky=W)
		Button(btnframe, text='Close', command=self.Close).grid(column=4, row=1, sticky=W)

		defaultbg = self.top.cget('bg')
		vsb = Scrollbar(mainframe, orient="vertical")
		text = Text(mainframe, width=40, height=20, yscrollcommand=vsb.set, bg=defaultbg)
		vsb.config(command=text.yview)
		vsb.grid(column=1, row=0, sticky=NS)
		text.grid(column=0, row=0, sticky=EW)

		self.cbStatesList = []	#List of checkbox states
		for i, item in enumerate(self.checkList): #More efficent than using range
			var = IntVar()
			cb = Checkbutton(mainframe, text=item,  variable=var)
			text.window_create("end", window=cb)
			text.insert("end", "\n") # to force one checkbox per line
			self.cbStatesList.append(var)
		# # # print (i+1) #Number of items in list
		text.config(state=DISABLED) #Don't allow user to write to the text box that contains the checkboxes

		centerWindow(self.top)

	def OKPressed(self, event):
		self.Close()

	def EscapePressed(self, event):
		self.Cancel()

	def Cancel(self):
		"""
		Called when the user clicks the "Cancel" button
		"""
		#Put tidying up code here
		self.top.destroy()

	def Close(self):
		"""
		Called when the user clicks the "Close" button
		"""
		self.top.destroy()


def main():
	print ("This module should not be run as a stand-alone module")
	return -1

if __name__ == "__main__":
	sys.exit(main())