#!/usr/bin/env python
# vim:fileencoding=utf-8:ts=4:sw=4:sta:et:sts=4:ai
# -*- coding: utf-8 -*-
from __future__ import absolute_import
from __future__ import division
from __future__ import print_function
from __future__ import unicode_literals

from math import ceil
import webbrowser

from PyQt5 import QtCore
from PyQt5 import QtWidgets as QtGui
from PyQt5.Qt import QColor
from PyQt5.Qt import QColorDialog
from PyQt5.Qt import QGridLayout
from PyQt5.Qt import QGroupBox
from PyQt5.Qt import QHBoxLayout
from PyQt5.Qt import QIcon
from PyQt5.Qt import QImage
from PyQt5.Qt import QLabel
from PyQt5.Qt import QPainter
from PyQt5.Qt import QPixmap
from PyQt5.Qt import QPushButton
from PyQt5.Qt import QRect
from PyQt5.Qt import QScrollArea
from PyQt5.Qt import QSpinBox
from PyQt5.Qt import QTimer
from PyQt5.Qt import QVBoxLayout
from PyQt5.Qt import Qt
from PyQt5.QtCore import QFile
from PyQt5.QtCore import QIODevice
from PyQt5.QtCore import QSize
from PyQt5.QtCore import QThread

from calibre.gui2 import pixmap_to_data
from calibre_plugins.prettify_cover.common_utils import get_icon
from calibre_plugins.prettify_cover.common_utils import SizePersistedDialog
from calibre_plugins.prettify_cover.scriptfu import get_gimp_bin
from calibre_plugins.prettify_cover.scriptfu import run_gimp
from calibre_plugins.prettify_cover.scriptfu import find_plugin_ng
from calibre_plugins.prettify_cover import __version__

__license__ = "GPL v3"
__copyright__ = "Copyright 2015-2021"
__author__ = "Michael Dinkelaker"
__email__ = "michael.dinkelaker@gmail.com"
__docformat__ = "restructuredtext en"

load_translations()

class CoverOptionsDialog(SizePersistedDialog):
    signal = QtCore.pyqtSignal()

    # button states
    state_enable = 'ENABLE'
    state_disable = 'DISABLE'
    state_connect = 'CONNECT'
    state_disconnect = 'DISCONNECT'
    state_flat = 'FLAT'


    def __init__(self, parent, images_dir, cover, w, h):
        title = _('Prettify Cover: Select a book-cover')
        self.dlg_title = title.split(':')[0] + " :"+title.split(':')[1]
        self.cover_tooltip = _("Ctrl + left-click for fullsize preview")
        self.selection = -1
        self.ia = parent
        self.gui = parent.gui
        self.cover = cover
        self.original_cover = cover
        self.thumbsize = 225
        self.w = w
        self.h = h

        self.oversized = False
        self.current_bg_color = -1
        self.is_updating = False

        # icon sizes
        self.icons_alignment_size_o = 34
        self.icons_alignment_size = 34

        # check for GIMP
        self.gimp_opt1 = "10"
        self.gimp_opt2 = "2"
        self.has_gimp = False        
        self.gimp_location = get_gimp_bin()        
        if self.gimp_location:
            self.has_gimp = True
            # check for NG plugin
            self.has_gimp_plugin_ng = find_plugin_ng()

        # DEBUG
        # self.has_gimp = False

        r = float(self.h / self.thumbsize)
        self.nwidth = ceil(self.w / r)
        self.nheight = ceil(self.h / r)
        self.images_dir = images_dir
        self.current_align = self.current_align_o = self.ia.align_default
        self.map_btns = self.map_covers = self.map_align = \
            self.map_align_o = self.map_icons = self.btn_o_left = \
            self.btn_oh_center = self.btn_o_right = \
            self.btn_top = self.btn_v_center = self.btn_bottom = \
            self.btn_o_top = self.btn_ov_center = self.btn_o_bottom = \
            self.btn_left = self.btn_h_center = self.btn_right = \
            self.btn_top = self.btn_v_center = self.btn_bottom = \
            self.btn_o_left = self.btn_oh_center = self.btn_o_right = \
            self.cpkr = self.xpos = self.ypos = self.btn_o_left2 = \
            self.btn_o_right2 = None
            # v 1.5.3 not ready
            #self.btn_o_up = self.btn_o_down = None

        self.white_gbox = self.black_gbox = self.cust_col_gbox = \
            self.cont_aw_gbox = self.cut_out_gbox = self.combo = None

        self.map_disenable = {}
        self.alignment_buttons_o = []
        self.alignment_buttons = []

        SizePersistedDialog.__init__(self,
                                     self.gui,
                                     'Prettify Cover plugin: cover ' +
                                     'options dialog')
        self.mapping_init = False
        self.build_ui(self.nwidth, self.nheight)

    def refresh_ui(self):
        QtGui.qApp.processEvents()

    def showEvent(self, event):
        super(CoverOptionsDialog, self).showEvent(event)
        self.signal.emit()
        self.generate_thumbs()
        self.init_colorpicker()
        self.add_alignment_buttons()
        self.init_mappings()
        self.init_alignment_buttons_gfx()
        self.change_both_states(self.state_connect)
        self.add_cover_tooltip()
        self.refresh_ui()
        self.xpos = self.geometry().x()
        self.ypos = self.geometry().y()

    def closeEvent(self, event):
        super(CoverOptionsDialog, self).closeEvent(event)
        self.signal.emit()

    def __exit__(self):     # , *args):
        self.terminate()

    def add_cover_tooltip(self):
        #  adds toottip info, how to fullsize preview the cover
        for cover in self.map_covers:
            cover.setToolTip(self.cover_tooltip)

        # no tooltip for content-aware cover button
        if self.has_gimp:
            self.btn_cover_content_aware.setToolTip("")

    def init_colorpicker(self):
        self.cpkr = QColorDialog()
        self.cpkr.setWindowFlags(Qt.Widget)
        self.cpkr.setOptions(QColorDialog.DontUseNativeDialog)
        for i in range(1):
            self.cpkr.setCustomColor(i, QColor(self.ia.color_palette[i]))
        # open colorpicker btn
        self.set_color_icon(self.ia.most_used_color, self.btn_colpicker)

    def init_alignment_buttons_gfx(self):
        self.toggle_alignment_btn(self.map_align, self.current_align,
                                  self.icons_alignment_size)
        self.toggle_alignment_btn(self.map_align_o, self.current_align_o,
                                  self.icons_alignment_size_o)

    def submitclose(self):
        sender = self.sender()

        if sender == self.btn_cover_white:
            self.selection = 0
        elif sender == self.btn_cover_black:
            self.selection = 1
        elif sender == self.btn_cover_custom:
            self.selection = 2
        elif self.has_gimp and sender == self.btn_cover_content_aware:
            self.selection = 3
        elif sender == self.btn_cover_cutout:
            self.selection = 4
            self.oversized = True

        #  hook for ctrl+click to preview
        if QtGui.qApp.mouseButtons() & Qt.RightButton or \
                QtGui.qApp.keyboardModifiers() & Qt.ControlModifier:
            self.fullsize_preview(self.selection)

        else:
            self.accept()
            self.hide()
            self.gui = None
            self.close()
            return self.selection

    def fullsize_preview(self, vid):
        oversize = False
        align = self.current_align
        if vid == 0:
            color = self.ia.col_white
        elif vid == 1:
            color = self.ia.col_black
        elif vid == 2:
            color = self.ia.most_used_color
        elif vid == 4:
            color = self.ia.col_black
            align = self.current_align_o
            oversize = True

        if vid == 3:
            filename = self.images_dir + "/tmp_cover.png"
            qi = QImage()
            qi.load(filename)
            pxmp = QPixmap.fromImage(qi)
        else:
            pxmp = self.ia.prettify_cover_for_book(self.cover, self.w, self.h,
                                                   color, True, 0, 0, align,
                                                   self, oversize)

        if self.has_gimp:
            gimp2_btn_state = self.btn_gimp2.isEnabled()
            self.btn_gimp2.setEnabled(False)

        self.change_both_states(self.state_disable)
        self.change_both_states(self.state_disconnect)
        d = FullSizePreviewDlg(self, pxmp)
        d.exec_()
        self.change_both_states(self.state_enable)
        self.change_both_states(self.state_connect)

        if self.has_gimp:
            self.btn_gimp2.setEnabled(gimp2_btn_state)

    def add_alignment_buttons(self):
        if self.current_align == self.ia.align_h_center:
            # horizontal / _o vertical
            self.current_align = self.ia.align_h_center
            self.current_align_o = self.ia.align_v_center
            self.btn_left = QPushButton("")
            self.btn_h_center = QPushButton("")
            self.btn_right = QPushButton("")
            self.btn_o_top = QPushButton("")
            self.btn_ov_center = QPushButton("")
            self.btn_o_bottom = QPushButton("")
            #self.btn_o_up = QPushButton("")
            #self.btn_o_down = QPushButton("")

            self.alignment_buttons = [
                self.btn_left,
                self.btn_h_center,
                self.btn_right
            ]
            self.alignment_buttons_o = [
                self.btn_o_top,
                self.btn_ov_center,
                self.btn_o_bottom,
                #self.btn_o_up,
                #self.btn_o_down
            ]
        else:
            # vertical / _o horizontal
            self.current_align = self.ia.align_v_center
            self.current_align_o = self.ia.align_h_center
            self.btn_top = QPushButton("")
            self.btn_v_center = QPushButton("")
            self.btn_bottom = QPushButton("")
            self.btn_o_left = QPushButton("")
            self.btn_oh_center = QPushButton("")
            self.btn_o_right = QPushButton("")
            self.btn_o_left2 = QPushButton("")          
            self.btn_o_right2 = QPushButton("")

            self.alignment_buttons = [
                self.btn_top,
                self.btn_v_center,
                self.btn_bottom
            ]
            self.alignment_buttons_o = [
                self.btn_o_left,
                self.btn_oh_center,
                self.btn_o_right,
                self.btn_o_left2,                
                self.btn_o_right2
            ]

        for abtn in self.alignment_buttons:
            self.a_layout.addWidget(abtn)

        for abtno in self.alignment_buttons_o:
            self.o_layout.addWidget(abtno)

    def remove_alignment_buttons(self):
        for abtn in self.alignment_buttons:
            self.a_layout.removeWidget(abtn)
        for abtno in self.alignment_buttons_o:
            self.o_layout.removeWidget(abtno)

    def udpate_custom_color_thumb(self, color):
        tmp = self.ia.prettify_cover_for_book(self.cover, self.nwidth,
                                              self.nheight, color, True, 0, 0,
                                              self.current_align, self)
        self.update_btn_pxmp(self.btn_cover_custom, tmp)

    def generate_thumbs(self, alignment_change=False, only_oversized=False):
        if not only_oversized:
            self.clear_thumb(self.btn_cover_white)
            self.clear_thumb(self.btn_cover_black)
            self.clear_thumb(self.btn_cover_custom)

            self.change_covers_state(self.state_disable)

            tmp = self.ia.prettify_cover_for_book(self.cover, self.nwidth,
                                                  self.nheight,
                                                  self.ia.col_white, True, 0,
                                                  0, self.current_align, self)
            self.update_btn_pxmp(self.btn_cover_white, tmp)
            tmp = self.ia.prettify_cover_for_book(self.cover, self.nwidth,
                                                  self.nheight,
                                                  self.ia.col_black,
                                                  True, 0, 0,
                                                  self.current_align, self)
            self.update_btn_pxmp(self.btn_cover_black, tmp)
            self.udpate_custom_color_thumb(self.ia.most_used_color)
            self.change_covers_state(self.state_enable)

        if not alignment_change or only_oversized:
            self.clear_thumb(self.btn_cover_cutout)
            tmp = self.ia.prettify_cover_for_book(self.cover, self.nwidth,
                                                  self.nheight,
                                                  self.ia.col_black, True, 0,
                                                  0, self.current_align_o,
                                                  self, True)
            self.update_btn_pxmp(self.btn_cover_cutout, tmp)

    def generate_content_aware_thumb(self):
        # 0.9.2: render to full cover size and scale img down to thumbnail size
        self.change_both_states(self.state_disable)
        self.clear_thumb(self.btn_cover_content_aware)
        self.btn_gimp2.setEnabled(False)
        self.gimp_opt1 = str(self.swidth.value())

        d = CustomProgressBar(self.ia)
        d.start_heal(self.cover, self.w, self.h,
                     self.ia.col_aware, True, self.gimp_opt1,
                     self.gimp_opt2, self.current_align)
        if d.exec_():
            if self.ia.action_gimp_error is True:
                self.wilbert_eek_gimp_error()
            else:
                # enable button click
                d.new_cover = d.new_cover.scaled(self.nwidth, self.nheight,
                                                 Qt.KeepAspectRatio,
                                                 Qt.SmoothTransformation)
                self.btn_cover_content_aware.clicked.connect(self.submitclose)
                self.update_btn_pxmp(self.btn_cover_content_aware, d.new_cover,
                                     not self.ia.action_gimp_error)
                self.btn_gimp2.setEnabled(True)
        else:
            self.wilbert_eek_gimp_error()
            self.btn_gimp2.setEnabled(False)

        self.change_both_states(self.state_enable)
        #  add tooltip for cover now
        self.btn_cover_content_aware.setToolTip(self.cover_tooltip)

    def wilbert_eek_gimp_error(self):
        try:
            self.btn_cover_content_aware.disconnect()
        except Exception:
            pass
        px = QPixmap.fromImage(self.ia.get_gimp_failed_thumb(self.nwidth,
                                                             self.nheight))
        self.update_btn_pxmp(self.btn_cover_content_aware, px,
                             not self.ia.action_gimp_error)
        self.ia.show_gimp_error(self.gui)

    def update_btn_pxmp(self, btn, pxmp, enable=True):
        if enable:
            ico = QIcon(pxmp)
            btn.setIcon(ico)

    # change normal push buttons and alignment buttons
    def change_buttons_state(self, state):
        self.change_map_state(self.map_btns, state)
        self.change_map_state(self.map_align, state)
        self.change_map_state(self.map_align_o, state)

        # just for dis/enabling
        if state == self.state_enable or state == self.state_disable:
            self.change_map_state(self.map_disenable, state)

    def set_alignment_button_state(self):
        self.change_map_state(self.map_align, self.state_flat)
        self.change_map_state(self.map_align_o, self.state_flat)

    # change cover buttons
    def change_covers_state(self, state):
        self.change_map_state(self.map_covers, state)

    def change_map_state(self, bmap, state):
        if self.mapping_init:
            for widget in bmap:
                if state == self.state_connect:
                    widget.clicked.connect(bmap[widget])
                elif state == self.state_disconnect:
                    widget.disconnect()
                elif state == self.state_enable:
                    widget.setEnabled(True)
                elif state == self.state_disable:
                    widget.setEnabled(False)
                elif state == self.state_flat:
                    widget.setFlat(True)
                else:
                    raise Exception("Error: dialogs.change_map_state(). " +
                                    "unknown state %s for widget " % (state))

    def change_both_states(self, state):
        self.change_covers_state(state)
        self.change_buttons_state(state)

    def init_mappings(self):
        # map_icns index numbers are matching with the
        # self.ia.alignment_ codes
        self.map_icons = ["images/ico_left.png",
                          "images/ico_center_horizontal.png",
                          "images/ico_right.png", "images/ico_top.png",
                          "images/ico_center_vertical.png", "images/ico_bottom.png",
                          "images/ico_left2.png", "images/ico_right2.png",
                          "images/ico_up.png", "images/ico_down.png"]

        self.map_btns = {
                self.btn_colpicker: self.show_colorpicker,
                self.btn_help: self.show_help_dlg
        }

        self.map_covers = {
                self.btn_cover_white: self.submitclose,
                self.btn_cover_black: self.submitclose,
                self.btn_cover_custom: self.submitclose,
                self.btn_cover_cutout: self.submitclose
        }

        self.map_disenable = {
                self.white_gbox: None,
                self.black_gbox: None,
                self.cust_col_gbox: None,
                self.cut_out_gbox: None,
                self.align_gbox: None,
                self.colpicker_gbox: None
        }

        # 0.9.8: add GIMP related mappings to dicts if needed
        if self.has_gimp:
            self.map_btns[self.btn_gimp] = self.run_gimp
            self.map_btns[self.btn_preview] = \
                self.generate_content_aware_thumb

            self.map_disenable[self.gimp_gbox] = \
                self.map_disenable[self.combo] = \
                self.map_disenable[self.cont_aw_gbox] = \
                self.map_disenable[self.label_ca_width] = \
                self.map_disenable[self.label_ca_order] = \
                self.map_disenable[self.swidth] = \
                None

            #  plugins: ng
            if self.has_gimp_plugin_ng:
                self.map_btns[self.btn_ng] = self.run_plugin_ng

        if self.current_align == self.ia.align_h_center:
            # horizontal alignment
            self.map_align = {
                self.btn_left: self.change_alignment,
                self.btn_h_center: self.change_alignment,
                self.btn_right: self.change_alignment
            }

            self.map_align_o = {
                self.btn_o_top: self.change_alignment_o,
                self.btn_ov_center: self.change_alignment_o,
                self.btn_o_bottom: self.change_alignment_o,
                # v 1.5.3 not ready
                #self.btn_o_up: self.change_alignment_o,
                #self.btn_o_down: self.change_alignment_o
            }
            self.btn_o_left = self.btn_oh_center = self.btn_o_right = \
                self.btn_top = self.btn_v_center = self.btn_bottom = None
        else:
            # vertical alignment
            self.map_align = {
                self.btn_top: self.change_alignment,
                self.btn_v_center: self.change_alignment,
                self.btn_bottom: self.change_alignment
            }

            # horizontal alignment
            self.map_align_o = {
                self.btn_o_left: self.change_alignment_o,
                self.btn_oh_center: self.change_alignment_o,
                self.btn_o_right: self.change_alignment_o,
                self.btn_o_left2: self.change_alignment_o,
                self.btn_o_right2: self.change_alignment_o
            }
            self.btn_o_top = self.btn_ov_center = self.btn_o_bottom = \
                self.btn_left = self.btn_h_center = self.btn_right = None
        self.mapping_init = True

    def toggle_alignment_btn(self, bmap, align, s):
        # alignment is matching the icon index number
        for widget in bmap:  # no specific order
            # findout which button/ico pos it is            
            # black button
            pos = self.get_icon_position(widget)
            fname = self.map_icons[pos]
            print("FNAME: ", fname)
            if pos == align:
                # white (highlight) button
                fname = fname.replace('ico', 'wico')
            widget.setIcon(get_icon(fname))
            self.set_icon_size(widget, QRect(0, 0, s, s), s, s)

    def get_icon_position(self, w):
        # todo: use a dict   
        print("got: ",w)
        if w == self.btn_h_center or w == self.btn_oh_center:
            return 1
        elif w == self.btn_v_center or w == self.btn_ov_center:
            return 4        
        elif w == self.btn_top or w == self.btn_o_top:
            return 3
        elif w == self.btn_left or w == self.btn_o_left:
            return 0        
        elif w == self.btn_bottom or w == self.btn_o_bottom:
            return 5
        elif w == self.btn_right or w == self.btn_o_right:
            return 2
        #elif w == self.btn_o_up:
        #    return 8
        #elif w == self.btn_o_down:
        #    return 9
        elif w == self.btn_o_left2:
            return 6
        elif w == self.btn_o_right2:
            return 7
        else:
            raise ValueError('Error with the toggle icon mapping. ' +
                             '(dialogs.get_icon_position)')
        return 0

    def change_alignment(self):
        sender = self.sender()
        self.current_align = self.get_alignment_from_sender(sender)
        self.toggle_alignment_btn(self.map_align, self.current_align,
                                  self.icons_alignment_size)
        self.generate_thumbs(True)

    def change_alignment_o(self):
        sender = self.sender()
        self.current_align_o = self.get_alignment_from_sender(sender)
        self.toggle_alignment_btn(self.map_align_o, self.current_align_o,
                                  self.icons_alignment_size_o)
        self.generate_thumbs(True, True)

    def get_alignment_from_sender(self, sender):
        if sender == self.btn_left or sender == self.btn_o_left:
            result_align = self.ia.align_left
        elif sender == self.btn_right or sender == self.btn_o_right:
            result_align = self.ia.align_right
        elif sender == self.btn_top or sender == self.btn_o_top:
            result_align = self.ia.align_top
        elif sender == self.btn_bottom or sender == self.btn_o_bottom:
            result_align = self.ia.align_bottom
        elif sender == self.btn_v_center or sender == self.btn_ov_center:
            result_align = self.ia.align_v_center
        elif sender == self.btn_h_center or sender == self.btn_oh_center:
            result_align = self.ia.align_h_center
        else:
            raise ValueError('Error: getting alignment from sender. ' +
                             '(dialogs.get_alignment_from_sender)')

        return result_align

    def clear_thumb(self, btn):
        pixmap = QPixmap(self.nwidth, self.nheight)
        pixmap.fill(QColor(self.ia.col_silver))
        ico = QIcon(pixmap)
        btn.setIcon(ico)

    def add_gbox(self, label, button):
        groupbox = QGroupBox(label)
        groupbox.setFlat(True)
        layout = QVBoxLayout()
        groupbox.setLayout(layout)
        layout.addWidget(button)
        return groupbox

    def build_ui(self, bwidth, bheight):
        spacing = 12
        self.block_updates = True
        self.setWindowTitle(self.dlg_title)
        self.setWindowIcon(get_icon('images/prettify_cover.png'))

        rect = QRect(0, 0, bwidth, bheight)
        pixmap = QPixmap(bwidth, bheight)
        pixmap.fill(QColor(self.ia.col_silver))
        ico = QIcon(pixmap)

        """ 0.9.8
        don't bother showing any GIMP related stuff
        unless GIMP is installed.
        """
        # Core UI
        bsmin = (self.icons_alignment_size - 8)
        bsmax = self.icons_alignment_size
        hlayout = QHBoxLayout()
        hlayout.setAlignment(Qt.AlignHCenter)
        self.main_layout = QGridLayout(self)
        self.main_layout.setAlignment(Qt.AlignHCenter)
        self.btn_cover_white = QPushButton(ico, "")
        self.btn_cover_black = QPushButton(ico, "")
        self.btn_cover_custom = QPushButton(ico, "")
        self.btn_cover_cutout = QPushButton(ico, "")
        self.btn_colpicker = QPushButton(ico, "")
        self.btn_help = QPushButton(_('Help'))
        self.set_icon_size(self.btn_cover_white, rect, bwidth, bheight)
        self.set_icon_size(self.btn_cover_black, rect, bwidth, bheight)
        self.set_icon_size(self.btn_cover_custom, rect, bwidth, bheight)
        self.set_icon_size(self.btn_cover_cutout, rect, bwidth, bheight)
        self.set_icon_size(self.btn_colpicker, QRect(0, 0, bsmin, bsmin),
                           bsmax, bsmax)
        self.setLayout(self.main_layout)

        if not self.has_gimp:
            # Without GIMP
            pos_black_gbox = [3, 1]
            pos_btn_help = [4, 1]
            pos_cust_col_gbox = [1, 1]
            pos_cust_col_btn = [2, 1]
            pos_align_gbox = [2, 0]
        else:
            # With GIMP
            self.btn_cover_content_aware = QPushButton(ico, "")
            self.btn_gimp = QPushButton(ico, "")
            #  0.9.7 edit content-aware image
            self.btn_gimp2 = QPushButton(ico, "")
            self.set_icon_size(self.btn_cover_content_aware, rect,
                               bwidth, bheight)
            self.set_icon_size(self.btn_gimp, QRect(0, 0, bsmin, bsmin),
                               bsmax, bsmax)
            self.set_icon_size(self.btn_gimp2, QRect(0, 0, bsmin, bsmin),
                               bsmax, bsmax)

            self.btn_gimp2.clicked.connect(self.run_gimp)
            self.btn_gimp2.setEnabled(False)
            self.btn_gimp2.setIcon(get_icon('images/gimp.png'))

            self.btn_gimp.setIcon(get_icon('images/gimp.png'))
            self.btn_gimp.setEnabled(True)

            self.gimp_gbox = QGroupBox(_('Edit with GIMP'))
            self.gimp_gbox.adjustSize()
            self.gimp_gbox.setFlat(True)
            self.gimp_gbox.setLayout(hlayout)

            if self.has_gimp_plugin_ng:
                # 0.9.8 add nationanal geo. plugin btn
                self.btn_ng = QPushButton(ico, "")
                self.btn_ng.setCheckable(True)
                self.btn_ng.setIcon(get_icon('images/ico_ng.png'))
                self.set_icon_size(self.btn_ng, QRect(0, 0, bsmin, bsmin),
                                   bsmax, bsmax)
                self.btn_ng.setEnabled(True)
                self.btn_ng.setToolTip(_("Apply National Geographic filter"
                                         " plug-in"))
                #self.btn_ng.toggle()
            # content-aware options
            self.label_ca_order = QLabel(_("Filling order"))
            self.label_ca_width = QLabel(_("Aspect width (px)"))

            self.swidth = QSpinBox()
            self.swidth.setRange(1, 200)
            self.swidth.setValue(10)

            self.btn_preview = QPushButton()
            self.btn_preview.setObjectName(_("preview"))
            self.btn_preview.setText(_("Generate preview"))

            self.combo = QtGui.QComboBox(self)
            self.combo.addItem(_("Random"))
            self.combo.addItem(_("Inwards"))
            self.combo.addItem(_("Outwards"))
            self.combo.setCurrentIndex(2)
            self.combo.activated[str].connect(self.on_activated)

            self.cont_aw_gbox = QGroupBox(_('Content-aware bg.'))
            self.cont_aw_gbox.adjustSize()
            self.cont_aw_gbox.setFlat(True)
            options_layout = QVBoxLayout()
            options_layout.setAlignment(Qt.AlignCenter)
            options_layout.addWidget(self.label_ca_width)
            options_layout.addWidget(self.swidth)
            options_layout.addSpacing(spacing)
            options_layout.addWidget(self.label_ca_order)
            options_layout.addWidget(self.combo)
            options_layout.addSpacing(spacing)

            ca_img_layout = QVBoxLayout()
            ca_img_layout.addWidget(self.btn_cover_content_aware)

            ca_layout = QHBoxLayout()
            ca_layout.addLayout(options_layout)
            ca_layout.addLayout(ca_img_layout)
            self.cont_aw_gbox.setLayout(ca_layout)
            hlayout.addWidget(self.btn_gimp)

            if self.has_gimp_plugin_ng:
                hlayout.addWidget(self.btn_ng)

            pos_black_gbox = [1, 1]
            pos_btn_help = [4, 1]
            pos_cust_col_gbox = [1, 2]
            pos_cust_col_btn = [2, 2]
            pos_align_gbox = [2, 1]

        # Add items to groupboxes
        self.white_gbox = self.add_gbox(_('White bg.'),
                                        self.btn_cover_white)
        self.main_layout.addWidget(self.white_gbox, 1, 0)
        self.black_gbox = self.add_gbox(_('Black bg.'),
                                        self.btn_cover_black)
        self.main_layout.addWidget(self.black_gbox, pos_black_gbox[0],
                                   pos_black_gbox[1])
        self.cust_col_gbox = self.add_gbox(_('Custom color bg.'),
                                           self.btn_cover_custom)
        self.main_layout.addWidget(self.cust_col_gbox, pos_cust_col_gbox[0],
                                   pos_cust_col_gbox[1])
        self.align_gbox = QGroupBox(_('Image alignment'))
        self.align_gbox.setFlat(True)
        self.cut_out_gbox = self.add_gbox(_('Cut-out'),
                                          self.btn_cover_cutout)

        # Add items to mainlayout
        self.main_layout.addWidget(self.align_gbox, pos_align_gbox[0],
                                   pos_align_gbox[1])
        self.a_layout = QHBoxLayout()
        self.align_gbox.setLayout(self.a_layout)

        hlayout = QHBoxLayout()
        hlayout.setAlignment(Qt.AlignHCenter)
        self.colpicker_gbox = QGroupBox(_('Change color'))
        self.colpicker_gbox.setFlat(True)
        self.colpicker_gbox.setLayout(hlayout)
        hlayout.addWidget(self.btn_colpicker)
        self.main_layout.addWidget(self.colpicker_gbox, pos_cust_col_btn[0],
                                   pos_cust_col_btn[1])

        #  cut-out cover and alignment buttons. pos is fixed!
        self.main_layout.addWidget(self.cut_out_gbox, 3, 0)
        self.o_layout = QHBoxLayout()
        self.main_layout.addLayout(self.o_layout, 4, 0)

        # help button
        self.main_layout.addWidget(self.btn_help, pos_btn_help[0],
                                   pos_btn_help[1])

        self.btn_help.setMaximumHeight(bsmax)

        if self.has_gimp:
            # add gimp items
            tmp_h_layout = QHBoxLayout()
            tmp_h_layout.addWidget(self.btn_preview)
            tmp_h_layout.addWidget(self.btn_gimp2)
            self.main_layout.addLayout(tmp_h_layout, 4, 2)
            # self.main_layout.addWidget(self.btn_preview, 4, 2)
            self.main_layout.addWidget(self.cont_aw_gbox, 3, 1, 1, 2)
            self.main_layout.addWidget(self.gimp_gbox, 2, 0)
            self.btn_preview.setMaximumHeight(bsmax)

            gaps = 2
        else:
            gaps = 3

        sh = self.sizeHint()
        if sh.height() > sh.width():
            diff = sh.height() - sh.width()
            diff /= gaps

        # make a rectangle by adding horizontal spacing between columns
        self.main_layout.setHorizontalSpacing(int(diff))
        self.resize(QSize(sh.height(), sh.height()))
        self.setSizeGripEnabled(False)
        self.refresh_ui()

    def on_activated(self):  # , text):
        self.gimp_opt2 = str(self.combo.currentIndex())

    def set_icon_size(self, btn, rect, w, h):
        btn.setIconSize(rect.size())
        btn.setMinimumSize(w, h)
        btn.setMaximumSize(w, h)

    def show_help_dlg(self):
        webbrowser.open('https://www.mobileread.com/forums/showthread.php?t=267427')

    def set_color_icon(self, color, btn):
        pxmp = QPixmap(48, 48)        
        pxmp.fill(QColor(color))
        self.update_btn_pxmp(btn, pxmp)

    def show_colorpicker(self):
        self.setWindowTitle(_('Prettify Cover: Select a color'))
        sh = self.sizeHint()

        """
        store centered position of this dialog,
        prior to opening the larger color dialog part
        """
        self.change_buttons_state(self.state_disable)
        self.change_covers_state(self.state_disconnect)
        self.cust_col_gbox.setEnabled(True)

        #  reenable groupbox2 for preview and colorpicking
        self.main_layout.addWidget(self.cpkr, 1, 3, 4, 1)
        self.main_layout.setAlignment(self.cpkr, Qt.AlignTop)
        # removes flickering dlg on colorpicking
        self.main_layout.setColumnMinimumWidth(3, (
            self.cpkr.sizeHint().width() + 12))
        self.cpkr.currentColorChanged.connect(self.on_color_changed)

        if self.cpkr.exec_():
            qcol = self.cpkr.selectedColor()
            if qcol.isValid():
                self.ia.most_used_color = self.cpkr.selectedColor().name()
                self.set_color_icon(self.ia.most_used_color,
                                    self.btn_colpicker)

        self.main_layout.removeWidget(self.cpkr)
        self.main_layout.setColumnMinimumWidth(3, 0)
        self.udpate_custom_color_thumb(self.ia.most_used_color)

        """
        store centered position of this dialog,
        prior to opening the larger color dialog parts
        """
        self.refresh_ui()
        self.resize(QSize(sh.height(), sh.height()))
        self.change_buttons_state(self.state_enable)
        self.change_covers_state(self.state_connect)
        self.setWindowTitle(self.dlg_title)
        self.move(self.xpos, self.ypos)

    def on_color_changed(self, color):
        if self.is_updating:
            return

        self.is_updating = True
        if color.isValid() and color.name() != self.current_bg_color:
            self.current_bg_color = color.name()
            self.udpate_custom_color_thumb(self.current_bg_color)
        self.is_updating = False

    def run_plugin_ng(self):
        fn = "tmp_cover.png"
        qi = QImage()
        if self.btn_ng.isChecked():
            self.btn_ng.setIcon(get_icon('images/wico_ng.png'))
            tfile = self.ia.images_dir + "/" + fn
            qi.loadFromData(self.cover)
            qi.save(tfile, "PNG")

            d = CustomProgressBar(self.ia)
            res = d.start_ng(tfile)            
            if d.exec_():
                qi.load(tfile)
                self.cover = pixmap_to_data(qi)
                self.generate_thumbs()
                # self.update_btn_pxmp(self.btn_cover_content_aware,
                #                     QPixmap.fromImage(qi),
                #                     not self.ia.action_gimp_error)
        else:
            self.btn_ng.setIcon(get_icon('images/ico_ng.png'))
            self.cover = self.original_cover
            self.generate_thumbs()

    def run_gimp(self):
        qi = QImage()
        fn = "tmp_cover.png"
        filename = self.ia.images_dir + "/" + fn
        is_content_aware = True
        gimp2_btn_state = self.btn_gimp2.isEnabled()
        self.btn_gimp2.setEnabled(False)

        if self.sender() == self.btn_gimp:
            filename = self.ia.images_dir + "/" + fn
            is_content_aware = False

            tfile = QFile(filename)
            if not tfile.open(QIODevice.WriteOnly):
                return self.ia.info_dialog(self.gui,
                                           _('Error: cannot start GIMP'),
                                           _('saving ') + fn + _(' failed!'),
                                           show=True, show_copy_button=False)
            qi.loadFromData(self.cover)
            qi.save(tfile, "PNG")

        self.setWindowTitle(_('GIMP: Overwrite ') + fn +
                            _(' and exit Gimp to continue'))
        self.change_both_states(self.state_disable)
        self.change_both_states(self.state_disconnect)
        self.refresh_ui()

        #  start a new instance of gimp, to avoid chaos.
        message = _("When done editing.\nFile > Overwrite ") + fn + \
            _("\nFile > Exit\nDiscard Changes")
        if run_gimp(self.gimp_location + " -n -b '((gimp-message \"" +
                    message + " \"))' " + filename):
            qi.load(filename)
            self.setWindowTitle(self.dlg_title)

            if not is_content_aware:
                #  0.9.7:  fix alignment btns
                self.cover = pixmap_to_data(qi)
                old_alignment = self.current_align
                old_alignment_o = self.current_align_o
                self.current_align = self.current_align_o = \
                    self.ia.align_default
                self.generate_thumbs()

                #  cover format (portrait/landscape) did not change
                if (self.current_align > 2 and old_alignment > 2) \
                        or (self.current_align < 3 and old_alignment < 3):
                    # restore old alignment state
                    self.current_align = old_alignment
                    self.current_align_o = old_alignment_o
                    self.generate_thumbs()

                else:
                    """
                    cover format changed, thus alignment
                    buttons need to be changed.
                    """
                    self.remove_alignment_buttons()
                    self.add_alignment_buttons()
                    self.init_mappings()
                    self.init_alignment_buttons_gfx()

            else:
                #  0.9.7  content-aware gimp session
                self.update_btn_pxmp(self.btn_cover_content_aware,
                                     QPixmap.fromImage(qi),
                                     not self.ia.action_gimp_error)

            self.btn_gimp2.setEnabled(gimp2_btn_state)
            self.change_both_states(self.state_enable)
            self.change_both_states(self.state_connect)
        return


class FullSizePreviewDlg(SizePersistedDialog):
    signal = QtCore.pyqtSignal()
    dlg_title = _('Prettify Cover: Cover Preview')

    def __init__(self, parent, pixmap):
        self.ia = parent
        self.gui = parent.gui
        #  set dialog size to image width + vertical scrollbars
        SizePersistedDialog.__init__(self, self.gui, 'Prettify Cover preview')
        self.setWindowIcon(get_icon('images/prettify_cover.png'))
        self.setWindowTitle(_('Fullsize Preview'))
        m_width = int(pixmap.width() + 40)
        self.setMinimumWidth(m_width)
        self.setMaximumWidth(m_width)

        layout = QVBoxLayout(self)
        sa = QScrollArea()
        sa.setHorizontalScrollBarPolicy(Qt.ScrollBarAsNeeded)
        sa.setVerticalScrollBarPolicy(Qt.ScrollBarAsNeeded)

        label = QLabel()
        label.setPixmap(pixmap)
        sa.setWidget(label)
        layout.addWidget(sa)
        self.setLayout(layout)

    def showEvent(self, event):
        super(FullSizePreviewDlg, self).showEvent(event)
        self.showMaximized()


class CustomProgressBar(SizePersistedDialog):
    signal = QtCore.pyqtSignal()

    def __init__(self, parent):
        SizePersistedDialog.__init__(self, parent.gui, 'CustomProgressBar')
        layout = QVBoxLayout(self)
        self.ia = parent
        self.gui = parent.gui
        self.setWindowTitle(_('Please wait'))
        self.setWindowIcon(get_icon('images/prettify_cover.png'))
        self.timeout = 120
        self.pg_bar = QtGui.QProgressBar(self)
        self.pg_bar.setRange(0, 1)
        self.button = QPushButton(_('Abort'))
        self.button.clicked.connect(self.handle_button)
        self.job = ""
        lbl = QLabel(_('generating cover ...'))
        layout.addWidget(lbl)
        layout.addWidget(self.pg_bar)
        layout.addWidget(self.button)
        self.setLayout(layout)
        self.new_cover = None
        self.adjustSize()
        self.resize(230, -1)
        QtGui.qApp.processEvents()

    def start_heal(self, cover, width, height, color, b, gimp_opt1="10",
                   gimp_opt2="2", align=0, oversize=False):

        self.job = "heal"
        self.my_long_task = HealTaskThread(self.ia, cover, width, height,
                                       color, b, gimp_opt1, gimp_opt2, align,
                                       oversize)
        self.my_long_task.taskFinished.connect(self.on_finished)
        self.on_start()

    def start_ng(self, filename):
        self.job = "ng"
        self.timeout = 240
        self.my_long_task = NgTaskThread(self.ia, filename)
        self.my_long_task.taskFinished.connect(self.on_finished)
        self.on_start()

    def closeEvent(self, event):
        self.handle_button()
        super(CustomProgressBar, self).closeEvent(event)
        self.signal.emit()

    def on_start(self):
        self.pg_bar.setRange(0, 0)
        self.timer = QTimer()
        self.timer.setSingleShot(True)
        self.timer.timeout.connect(self.on_timeout)
        self.timer.start(self.timeout * 1000)

        self.my_long_task.start()

    def on_finished(self):
        self.timer.stop()
        # Stop the pulsation
        self.new_cover = self.my_long_task.new_cover
        self.pg_bar.setRange(0, 1)
        self.accept()
        return self.new_cover

    def on_timeout(self):
        if self.job == "heal":
            self.ia.action_gimp_error = self.ia.action_gimp_hang_up = True
            qi = self.ia.get_gimp_failed_thumb(self.my_long_task.width,
                                               self.my_long_task.height)
            self.my_long_task.new_cover = QPixmap.fromImage(qi)

        self.my_long_task.terminate()
        self.on_finished()

    def set_icon_size(self, btn, rect, w, h):
        btn.setIconSize(rect.size())
        btn.setMinimumSize(w, h)
        btn.setMaximumSize(w, h)

    def handle_button(self):
        self.timer.stop()
        self.on_timeout()
        self.on_finished()


class NgTaskThread(QThread):
    taskFinished = QtCore.pyqtSignal()

    def __init__(self, parent, filename):
        QThread.__init__(self)
        self.ia = parent
        self.new_cover = None
        self.filename = filename

    def run(self):
        self.ia.action_gimp_error = self.ia.action_gimp_hang_up = False
        #  this goes action
        self.new_cover = \
            self.ia.ng_cover(self.filename)
        self.taskFinished.emit()


class HealTaskThread(QThread):
    taskFinished = QtCore.pyqtSignal()

    def __init__(self, parent, cover, width, height, color, b, gimp_opt1,
                 gimp_opt2, align, oversize):
        QThread.__init__(self)
        self.ia = parent
        self.cover = cover
        self.width = width
        self.height = height
        self.color = color
        self.oversize = oversize
        self.gimp_opt1 = gimp_opt1
        self.gimp_opt2 = gimp_opt2
        self.b = b
        self.current_align = align
        self.new_cover = None

    def run(self):
        self.ia.action_gimp_error = self.ia.action_gimp_hang_up = False
        self.new_cover = \
            self.ia.prettify_cover_for_book(self.cover, self.width,
                                            self.height, self.color, self.b,
                                            self.gimp_opt1, self.gimp_opt2,
                                            self.current_align, None,
                                            self.oversize)
        self.taskFinished.emit()
