#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
# -*- coding: utf-8 -*-
from __future__ import absolute_import
from __future__ import division
from __future__ import print_function
from __future__ import unicode_literals

import os
import sys

from PyQt5 import Qt as QtGui
from PyQt5.Qt import QAbstractItemView
from PyQt5.Qt import QHBoxLayout
from PyQt5.Qt import QIcon
from PyQt5.Qt import QLabel
from PyQt5.Qt import QPushButton
from PyQt5.Qt import QTableWidget
from PyQt5.Qt import QTableWidgetItem
from PyQt5.Qt import QVBoxLayout
from PyQt5.Qt import QWidget

from calibre.gui2 import info_dialog
from calibre.gui2 import question_dialog
from calibre.utils.config import JSONConfig
from calibre.utils.config import config_dir
from calibre_plugins.prettify_cover.common_utils import KeyboardConfigDialog
from calibre_plugins.prettify_cover.common_utils import ReadOnlyTableWidgetItem
from calibre_plugins.prettify_cover import __version__


__license__ = 'GPL v3'
__copyright__ = "Copyright 2015-2021"
__author__ = "Michael Dinkelaker"
__email__ = "michael.dinkelaker@gmail.com"
__docformat__ = 'restructuredtext en'

load_translations()

STORE_NAME = 'Options'
KEY_SIZES = 'sizes'
KEY_WIDTH = 'width'
KEY_HEIGHT = 'height'
KEY_DEFAULT = 'default'

DEFAULT_STORE_VALUES = {
    KEY_SIZES: [{KEY_WIDTH: 758, KEY_HEIGHT: 1024, KEY_DEFAULT: True}]
}

# This is where all preferences for this plugin will be stored
plugin_prefs = JSONConfig('plugins/Prettify Cover')

# Set defaults
plugin_prefs.defaults[STORE_NAME] = DEFAULT_STORE_VALUES


def get_images_dir():
    return os.path.join(config_dir, 'resources/images/prettify_cover')


class SizesTableWidget(QTableWidget):

    def __init__(self, parent):
        QTableWidget.__init__(self, parent)
        self.ia = parent

    def populate_table(self, data_items):
        self.clear()
        self.setAlternatingRowColors(True)
        self.setRowCount(len(data_items))
        header_labels = [_('Width'), _('Height'), _('Default')]
        self.setColumnCount(len(header_labels))
        self.setHorizontalHeaderLabels(header_labels)
        self.verticalHeader().setDefaultSectionSize(24)
        self.horizontalHeader().setStretchLastSection(True)

        for row, data in enumerate(data_items):
            self.populate_table_row(row, data)

        for col in range(0, 2):
            self.resizeColumnToContents(col)
        self.setSortingEnabled(False)
        self.setMinimumSize(200, 50)
        self.setSelectionBehavior(QAbstractItemView.SelectRows)
        if len(data_items) > 0:
            self.selectRow(0)

    def populate_table_row(self, row, data_item):
        self.setItem(row, 0, QTableWidgetItem(str(data_item[KEY_WIDTH])))
        self.setItem(row, 1, QTableWidgetItem(str(data_item[KEY_HEIGHT])))
        if data_item[KEY_DEFAULT]:
            self.setItem(row, 2, ReadOnlyTableWidgetItem('Default'))
        else:
            self.setItem(row, 2, ReadOnlyTableWidgetItem(''))

    def get_data(self):
        data_items = []
        for row in range(self.rowCount()):
            data_item = {}
            # python 3/2
            if sys.version_info.major > 2:
                data_item[KEY_WIDTH] = int((self.item(row, 0).text()).strip())
                data_item[KEY_HEIGHT] = int((self.item(row, 1).text()).strip())
                data_item[KEY_DEFAULT] = \
                    (self.item(row, 2).text()).strip() == 'Default'
            else:
                data_item[KEY_WIDTH] = \
                    int(unicode(self.item(row, 0).text()).strip())
                data_item[KEY_HEIGHT] = \
                    int(unicode(self.item(row, 1).text()).strip())
                data_item[KEY_DEFAULT] = \
                    (unicode(self.item(row, 2).text()).strip() == 'Default')



            if data_item[KEY_WIDTH] > 0 and data_item[KEY_HEIGHT] > 0:
                data_items.append(data_item)
        if len(data_items) == 0:
            data_items[0][KEY_DEFAULT] = True
        return data_items

    def create_blank_row_data(self):
        data_item = {}
        data_item[KEY_WIDTH] = 0
        data_item[KEY_HEIGHT] = 0
        data_item[KEY_DEFAULT] = False
        return data_item

    def select_and_scroll_to_row(self, row):
        self.selectRow(row)
        self.scrollToItem(self.currentItem())

    def add_row(self):
        self.setFocus()
        # We will insert a blank row below the currently selected row
        row = self.currentRow() + 1
        self.insertRow(row)
        self.populate_table_row(row, self.create_blank_row_data())
        self.select_and_scroll_to_row(row)

    def delete_rows(self):
        self.setFocus()
        rows = self.selectionModel().selectedRows()

        if len(rows) == 0:
            return
        if self.rowCount() == 1 or self.rowCount() == len(rows):
            title = _('Cannot delete')
            message = _('You must have at least one menu item')
            info_dialog(self.ia, title, message, show_copy_button=False)
            return

        message = _('<p>Are you sure you want to delete this menu item?')
        if len(rows) > 1:
            message = _('<p>Are you sure you want to delete the '
                        'selected %d menu items?') % len(rows)
        if not question_dialog(self, _('Are you sure?'), message,
                               show_copy_button=False):
            return
        first_sel_row = self.currentRow()
        for selrow in reversed(rows):
            self.removeRow(selrow.row())
        if first_sel_row < self.rowCount():
            self.select_and_scroll_to_row(first_sel_row)
        elif self.rowCount() > 0:
            self.select_and_scroll_to_row(first_sel_row - 1)

    def set_as_default(self):
        row = self.currentRow()
        if str(self.item(row, 2).text()).strip() == 'Default':
            return
        self.item(row, 2).setText('Default')
        for update_row in range(0, self.rowCount()):
            if update_row != row:
                self.item(update_row, 2).setText('')


class ConfigWidget(QWidget):

    def __init__(self, plugin_action):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        layout = QVBoxLayout(self)
        self.setLayout(layout)

        heading_label = QLabel(_("Set E-Reader format/s:"), self)
        layout.addWidget(heading_label)

        """
        Add a horizontal layout containing
        the table and the buttons next to it
        """
        table_layout = QHBoxLayout()
        layout.addLayout(table_layout)

        self._table = SizesTableWidget(self)
        heading_label.setBuddy(self._table)
        table_layout.addWidget(self._table)

        # Add a vertical layout containing the the buttons to add/remove.
        button_layout = QtGui.QVBoxLayout()
        table_layout.addLayout(button_layout)

        add_button = QtGui.QToolButton(self)
        add_button.setToolTip(_('Add menu item row'))
        add_button.setIcon(QIcon(I('plus.png')))
        button_layout.addWidget(add_button)
        spacer_item = QtGui.QSpacerItem(20, 40, QtGui.QSizePolicy.Minimum,
                                         QtGui.QSizePolicy.Expanding)
        button_layout.addItem(spacer_item)

        delete_button = QtGui.QToolButton(self)
        delete_button.setToolTip(_('Delete menu item row'))
        delete_button.setIcon(QIcon(I('minus.png')))
        button_layout.addWidget(delete_button)

        add_button.clicked.connect(self._table.add_row)
        delete_button.clicked.connect(self._table.delete_rows)

        other_layout = QHBoxLayout()
        layout.addLayout(other_layout)
        set_default_button = QPushButton(_('Set as Toolbar Button Default'),
                                         self)
        set_default_button.setToolTip(_('Set this size as the default on '
                                        'the menu button'))
        set_default_button.clicked.connect(self._table.set_as_default)
        other_layout.addWidget(set_default_button)
        keyboard_shortcuts_button = QPushButton(_('Keyboard shortcuts...'),
                                                self)
        keyboard_shortcuts_button.setToolTip(_('Edit the keyboard shortcuts '
                                               'associated with this plugin'))
        keyboard_shortcuts_button.clicked.connect(self.edit_shortcuts)
        other_layout.addWidget(keyboard_shortcuts_button)
        other_layout.insertStretch(-1)

        c = plugin_prefs[STORE_NAME]
        self._table.populate_table(c[KEY_SIZES])

    def save_settings(self):
        new_prefs = {}
        new_prefs[KEY_SIZES] = self._table.get_data()
        plugin_prefs[STORE_NAME] = new_prefs

    def edit_shortcuts(self):
        self.save_settings()
        """
        Force the menus to be rebuilt immediately,
        so we have all our actions registered
        """
        self.plugin_action.rebuild_menus()
        d = KeyboardConfigDialog(self.plugin_action.gui,
                                 self.plugin_action.action_spec[0])
        if d.exec_() == d.Accepted:
            self.plugin_action.gui.keyboard.finalize()
