#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
# -*- coding: utf-8 -*-
from __future__ import absolute_import
from __future__ import division
from __future__ import print_function
from __future__ import unicode_literals

from PyQt5 import QtCore
from PyQt5.Qt import QColor
from PyQt5.Qt import QImage
from PyQt5.Qt import QMenu
from PyQt5.Qt import QPainter
from PyQt5.Qt import QPixmap
from PyQt5.Qt import QRect
from PyQt5.Qt import QSize
from PyQt5.Qt import QToolButton
from PyQt5.QtCore import QFile
from PyQt5.QtCore import QIODevice
from PyQt5.QtCore import Qt

from calibre.gui2 import error_dialog
from calibre.gui2 import info_dialog
from calibre.gui2 import pixmap_to_data
from calibre.gui2.actions import InterfaceAction
from calibre_plugins.prettify_cover.common_utils import create_menu_action_unique
from calibre_plugins.prettify_cover.common_utils import get_icon
from calibre_plugins.prettify_cover.common_utils import set_plugin_icon_resources
from calibre_plugins.prettify_cover.dialogs import CoverOptionsDialog
from calibre_plugins.prettify_cover.dialogs import CustomProgressBar
from calibre_plugins.prettify_cover.scriptfu import heal_cover
from calibre_plugins.prettify_cover.scriptfu import gimp_ng_cover
from calibre_plugins.prettify_cover import __version__
import calibre_plugins.prettify_cover.config as cfg


from functools import partial
from math import ceil, floor
import operator
import os
import sys


__license__ = "GPL v3"
__copyright__ = "Copyright 2015-2021"
__author__ = "Michael Dinkelaker"
__email__ = "michael.dinkelaker@gmail.com"
__docformat__ = "restructuredtext en"

load_translations()
                
PLUGIN_ICONS = ['images/prettify_cover.png', 'images/ico_top.png', 'images/ico_up.png',
                'images/ico_bottom.png', 'images/ico_center_horizontal.png', 'images/ico_down.png',
                'images/ico_center_vertical.png', 'images/ico_left.png',
                'images/ico_right.png', 'images/wico_top.png',
                'images/wico_bottom.png', 'images/wico_center_horizontal.png',
                'images/wico_center_vertical.png', 'images/wico_left.png',
                'images/wico_right.png', 'images/wilber-eeek.png',
                'images/gimp.png', 'images/ico_ng.png', 'images/wico_ng.png',
                'images/wico_left2.png', 'images/wico_right2.png','images/ico_left2.png',
                'images/ico_right2.png',]


class PrettifyCoverAction(InterfaceAction):
    name = 'Prettify Cover'
    action_spec = ('Prettify Cover', None, None, None)
    popup_type = QToolButton.MenuButtonPopup
    action_type = 'current'
    cover = ''
    col_white = '#ffffff'
    col_black = '#000000'
    col_most = '-1'
    col_aware = '-2'
    col_silver = '#C0C0C0'
    most_used_color = -1
    color_palette = [65535] * 16

    action_gimp_error = False
    action_gimp_hang_up = False
    align_default = -1

    align_left = 0
    align_h_center = 1
    align_right = 2
    align_top = 3
    align_v_center = 4
    align_bottom = 5
    align_left2 = 6
    align_right2 = 7
    align_up = 8
    align_down = 9

    def genesis(self):
        self.menu = QMenu(self.gui)
        self.old_actions_unique_map = {}
        self.default_size_data = None

        """
        Read the plugin icons and store for potential
        sharing with the config widget
        """
        icon_resources = self.load_resources(PLUGIN_ICONS)
        set_plugin_icon_resources(self.name, icon_resources)
        self.rebuild_menus()

        # Assign our menu to this action and an icon
        self.qaction.setMenu(self.menu)
        self.qaction.setIcon(get_icon(PLUGIN_ICONS[0]))
        self.qaction.triggered.connect(self.prettify_covers)
        self.menu.aboutToShow.connect(self.about_to_show_menu)

    def about_to_show_menu(self):
        self.rebuild_menus()

    def rebuild_menus(self):
        c = cfg.plugin_prefs[cfg.STORE_NAME]
        size_data_items = c[cfg.KEY_SIZES]
        m = self.menu
        m.clear()
        self.actions_unique_map = {}

        for size_data in size_data_items:
            menu_text = _('Size: %d(w) x %d(h)') % (size_data[cfg.KEY_WIDTH],
                                                    size_data[cfg.KEY_HEIGHT])
            is_default = bool(size_data[cfg.KEY_DEFAULT])
            ac = create_menu_action_unique(self, m, menu_text,
                                           is_checked=is_default,
                                           triggered=partial(
                                               self.prettify_covers,
                                               size_data[cfg.KEY_WIDTH],
                                               size_data[cfg.KEY_HEIGHT]))

            self.actions_unique_map[ac.calibre_shortcut_unique_name] = \
                ac.calibre_shortcut_unique_name

            if is_default:
                self.default_size_data = size_data
        m.addSeparator()
        create_menu_action_unique(self, m, _('&Customize plugin') + '...',
                                  'config.png', shortcut=False,
                                  triggered=self.show_configuration)
        """
        Before we finalize, make sure we delete any
        actions for menus that are no longer displayed
        """
        # python 3/2
        if sys.version_info.major > 2:
            for menu_id, unique_name in self.old_actions_unique_map.items():
                if menu_id not in self.actions_unique_map:
                    self.gui.keyboard.unregister_shortcut(unique_name)
        else:
            for menu_id, unique_name in self.old_actions_unique_map.iteritems():
                if menu_id not in self.actions_unique_map:
                    self.gui.keyboard.unregister_shortcut(unique_name)

        self.old_actions_unique_map = self.actions_unique_map
        self.gui.keyboard.finalize()

    def prettify_covers(self, width=None, height=None):
        self.images_dir = cfg.get_images_dir()
        if not os.path.exists(self.images_dir):
            os.makedirs(self.images_dir)

        if not width:
            if self.default_size_data:
                width = self.default_size_data[cfg.KEY_WIDTH]
                height = self.default_size_data[cfg.KEY_HEIGHT]
        if not width or not height:
            return

        rows = self.gui.library_view.selectionModel().selectedRows()
        if not rows or len(rows) == 0:
            return
        current_idx = self.gui.library_view.currentIndex()
        db = self.gui.library_view.model().db
        ids = set(self.gui.library_view.get_selected_ids())
        resized_ids = []

        for book_id in ids:
            if db.has_cover(book_id):
                cover = db.cover(book_id, index_is_id=True)

                # lets check that cover size first
                cover_px = QPixmap()
                cover_px.loadFromData(cover)

                if cover_px.width() != width or cover_px.height() != height:
                    self.most_used_color = -1  # calc. most used color
                    d = CoverOptionsDialog(self, self.images_dir, cover,
                                           width, height)
                    if d.exec_():
                        cover = d.cover
                        sel = d.selection
                        align = d.current_align
                        o_align = d.current_align_o
                        opt1 = d.gimp_opt1
                        opt2 = d.gimp_opt2
                        oversize = d.oversized

                        color = ''
                        if sel == 0:
                            color = self.col_white
                        elif sel == 1:
                            color = self.col_black
                        elif sel == 2:
                            color = self.most_used_color
                        elif sel == 4:
                            color = self.col_black
                            align = o_align
                        elif sel < -2 or sel > 3:
                            return info_dialog(self.gui,
                                               _('No covers resized'),
                                               _('wtf? error exception'),
                                               show=True,
                                               show_copy_button=False)

                        # v0.9.2 no need to render the image again
                        if sel == 3:
                            filename = self.images_dir + "/tmp_cover.png"
                            qi = QImage()
                            qi.load(filename)
                            updated_cover = pixmap_to_data(qi)
                        else:
                            d = CustomProgressBar(self)
                            d.start_heal(cover, width, height, color, False,
                                         opt1, opt2, align, oversize)
                            if d.exec_():
                                if self.action_gimp_error or \
                                        self.action_gimp_hang_up:
                                    return
                                updated_cover = d.new_cover
                            else:
                                return error_dialog(self.gui,
                                                    _('Cover generation'),
                                                    _('something did not '
                                                      'work as expected'),
                                                    show=True,
                                                    show_copy_button=False)

                        db.set_cover(book_id, updated_cover)
                        resized_ids.append(book_id)
                    else:
                        # user eXit
                        return

        if len(resized_ids) == 0:
            return info_dialog(self.gui, _('No covers prettified'),
                               _('Either book(s) don\'t have covers or '
                                 'cover(s) have the desired size, already'),
                               show=True, show_copy_button=False)

        self.gui.library_view.model().refresh_ids(resized_ids)
        self.gui.library_view.model().current_changed(current_idx, current_idx)
        if self.gui.cover_flow:
            self.gui.cover_flow.dataChanged()

    def ng_cover(self, fn):
        return gimp_ng_cover(fn)

    def prettify_cover_for_book(self, cover, width, height, color, as_pixmap,
                                gimp_opt1="10", gimp_opt2="2", align=0,
                                parent=None, oversize=False):

        cover_pixmap = QPixmap()
        cover_pixmap.loadFromData(cover)

        width_ratio = height_ratio = aspect_ratio = 0
        """
        0 = scale bigger , 1 = scale smaller
        """
        scale = x = y = 0

        """
        find scale direction and ratio
        this solves floating point problem: from __future__ import division
        """
        if cover_pixmap.width() >= width and cover_pixmap.height() >= height:
            width_ratio = float(cover_pixmap.width() / width)
            height_ratio = float(cover_pixmap.height() / height)
            scale = 1
        else:
            width_ratio = float(width / cover_pixmap.width())
            height_ratio = float(height / cover_pixmap.height())
            scale = 0

        """
        cut-out (oversize) image
        is the method to scale up the smallest side of the cover to the
        desired width or height and crop the overlap centered
        """
        new_width = new_height = 0
        if oversize is False:
            # find ratio
            if scale == 1:
                if width_ratio >= height_ratio:
                    aspect_ratio = width_ratio
                    nalign = self.align_v_center
                else:
                    aspect_ratio = height_ratio
                    nalign = self.align_h_center
                new_width = ceil(cover_pixmap.width() / aspect_ratio)
                new_height = ceil(cover_pixmap.height() / aspect_ratio)
            else:
                if width_ratio <= height_ratio:
                    aspect_ratio = width_ratio
                    nalign = self.align_v_center
                else:
                    aspect_ratio = height_ratio
                    nalign = self.align_h_center
                new_width = ceil(cover_pixmap.width() * aspect_ratio)
                new_height = ceil(cover_pixmap.height() * aspect_ratio)

            cover_pixmap = cover_pixmap.scaled(new_width, new_height,
                                               Qt.IgnoreAspectRatio,
                                               Qt.SmoothTransformation)
            new_width = cover_pixmap.width()
            new_height = cover_pixmap.height()
            x = y = 0

            # set alignment (left/top are x/y 0)
            if align == self.align_default:
                align = nalign

            if parent is not None:
                parent.current_align = align

            if align == self.align_right:
                x = width - new_width

            elif align == self.align_bottom:
                y = height - new_height

            elif align == self.align_h_center:
                x = width - new_width
                if x > 0:
                    x /= 2

            elif align == self.align_v_center:
                y = height - new_height
                if (y > 0):
                    y /= 2

            final_pixmap = QPixmap(width, height)
            if color == self.col_aware:
                final_pixmap.fill(Qt.transparent)  # content-aware bg
            else:
                if str(color) == self.col_most:
                    self.most_used_color = color = \
                        self.find_most_used_color(cover_pixmap)
                final_pixmap.fill(QColor(color))

            qi = QImage(final_pixmap)
            target_rect = QRect(floor(x), floor(y), new_width, new_height)
            p = QPainter(qi)
            p.setRenderHints(QPainter.Antialiasing |
                             QPainter.SmoothPixmapTransform)
            p.drawPixmap(target_rect, cover_pixmap)
            p.end()

            if (color == self.col_aware):
                filename = self.images_dir + "/tmp_cover.png"
                file = QFile(filename)
                if not file.open(QIODevice.WriteOnly):
                    return info_dialog(self.gui,
                                       _('Error: content-aware cover',
                                         'saving thumbnail failed!'),
                                       show=True, show_copy_button=False)
                else:
                    qi.save(file, "PNG")
                    if heal_cover(filename, gimp_opt1, gimp_opt2):
                        qi.load(filename)
                    else:
                        qi = QImage(final_pixmap)
        # end alignment
        else:
            # oversize: find ratio
            nalign = 0
            if scale == 1:
                if width_ratio <= height_ratio:
                    aspect_ratio = width_ratio
                    nalign = self.align_v_center
                else:
                    aspect_ratio = height_ratio
                    nalign = self.align_h_center
                new_width = ceil(cover_pixmap.width() / aspect_ratio)
                new_height = ceil(cover_pixmap.height() / aspect_ratio)
            else:
                if width_ratio <= height_ratio:
                    aspect_ratio = height_ratio
                    nalign = self.align_h_center
                else:
                    aspect_ratio = width_ratio
                    nalign = self.align_v_center
                new_width = ceil(cover_pixmap.width() * aspect_ratio)
                new_height = ceil(cover_pixmap.height() * aspect_ratio)

            cover_pixmap = cover_pixmap.scaled(new_width, new_height,
                                               Qt.IgnoreAspectRatio,
                                               Qt.SmoothTransformation)
            new_width = cover_pixmap.width()
            new_height = cover_pixmap.height()
            x = y = 0

            # set alignment (left/top are x/y 0)
            if align == self.align_default:
                align = nalign
            if parent is not None:
                parent.current_align_o = align

            if align == self.align_right:
                x = new_width - width

            elif align == self.align_bottom:
                y = new_height - height

            elif align == self.align_h_center:
                x = new_width - width
                if x > 0:
                    x /= 2

            elif align == self.align_v_center:
                y = new_height - height
                if (y > 0):
                    y /= 2

            final_pixmap = QPixmap(width, height)
            qi = QImage(final_pixmap)
            target_rect = QRect(0, 0, width, height)
            source_rect = QRect(floor(x), floor(y), width, height)
            p = QPainter(qi)
            p.setRenderHints(QPainter.Antialiasing |
                             QPainter.SmoothPixmapTransform)
            p.drawPixmap(target_rect, cover_pixmap, source_rect)
            p.end()

        if as_pixmap is True:
            return QPixmap.fromImage(qi)
        else:
            return pixmap_to_data(qi)

    def get_gimp_failed_thumb(self, w, h):
        isize = 96
        self.action_gimp_error = True  # show errormsg
        final_pixmap = QPixmap(w, h)
        final_pixmap.fill(QtCore.Qt.transparent)
        x = (final_pixmap.width() - isize) / 2
        y = (final_pixmap.height() - isize) / 2
        target_rect = QRect(x, y, isize, isize)
        icon = get_icon('images/wilber-eeek.png')
        qi = QImage(final_pixmap)
        p = QPainter(qi)
        p.setRenderHints(QPainter.Antialiasing |
                         QPainter.SmoothPixmapTransform)
        p.drawPixmap(target_rect,
                     icon.pixmap(icon.actualSize(QSize(isize, isize))))
        p.end()
        return qi

    def show_gimp_error(self, gui):
        if self.action_gimp_hang_up is True:
            message = _('Either Gimp process timed out (try again!), or '
                        'user abortion.')
        else:
            message = _('Gimp and/or resynthesizer plug-in is not installed')

        return error_dialog(gui, _('Content-aware fill'),
                            message, show=True, show_copy_button=False)

    def show_configuration(self):
        self.interface_action_base_plugin.do_user_config(self.gui)

    def find_most_used_color(self, pixmap):
        image = QImage(pixmap.toImage())
        col = count = greatest = 0
        rgbcount = {}

        width = pixmap.width()
        height = pixmap.height()

        for i in range(0, width):
            for j in range(0, height):
                col = QColor(image.pixel(i, j)).name()

                if rgbcount.__contains__(col):
                    rgbcount[col] += 1
                else:
                    rgbcount[col] = 1
                if rgbcount[col] > count:
                    greatest = col
                    count = rgbcount[col]

        sorted_x = sorted(rgbcount.items(), key=operator.itemgetter(1),
                          reverse=True)
        i = 0
        for item in sorted_x:
            self.color_palette[i] = item[0]

            i += 1
            if i == 16:
                break
        return str(greatest)
