#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
# -*- coding: utf-8 -*-
from __future__ import absolute_import
from __future__ import unicode_literals
from calibre.gui2 import sanitize_env_vars
from calibre_plugins.prettify_cover import __version__

import os
import sys
from sys import platform as _platform
import subprocess

# win registry
if sys.version_info.major > 2:
# python3
    try:        
        import winreg        
    except ImportError:        
        pass
else:
# python2
    try:
        from _winreg import *
    except ImportError:
        pass

__license__ = "GPL v3"
__copyright__ = "Copyright 2015-2021"
__author__ = "Michael Dinkelaker"
__email__ = "michael.dinkelaker@gmail.com"
__docformat__ = "restructuredtext en"

# possible gimp/gimp-console VERSION. Will be build on start
VERSIONS = ["2.10", "2.9", "2.8", "2.7", "2.6"]
CONSOLE_BIN_WIN = []
GIMP_BIN_WIN = []
CONSOLE_BIN_LINUX = ["/usr/bin/gimp-console"]
GIMP_BIN_LINUX = ["/usr/bin/gimp"]
GIMP_PATH_WIN = ""

# MAIN VARS
GIMP_BIN = ""
CONSOLE_BIN = ""
# SET GIMP PATH ON START FOR WINDOWS 
if _platform == "win32":
    try:
        # winreg.KEY_WOW64_64KEY was missing
        with winreg.OpenKey(winreg.HKEY_LOCAL_MACHINE, r'SOFTWARE\Microsoft\Windows\CurrentVersion\\Uninstall\GIMP-2_is1', 0, winreg.KEY_READ | winreg.KEY_WOW64_64KEY) as key:        
            key_value = winreg.QueryValueEx(key, 'InstallLocation')            
            GIMP_PATH_WIN = key_value[0]
    except WindowsError:
        # didnt find the desired key
        print("prettify cover: registry parse error")
        pass    

def is_exe(fpath):
    return os.path.isfile(fpath) and os.access(fpath, os.X_OK)

def find_binary(binaries):
    prg = ""
    for binary in binaries:
        if is_exe(binary):
            prg = binary
            break
    return prg

# BUILD BINARY LISTS FROM VERSION STRING
for v in VERSIONS:
    CONSOLE_BIN_WIN.append(GIMP_PATH_WIN+"\\bin\\gimp-console-"+v+".exe")
    GIMP_BIN_WIN.append(GIMP_PATH_WIN+"\\bin\\gimp-"+v+".exe")
    CONSOLE_BIN_LINUX.append("/usr/bin/gimp-console-"+v)

if _platform == "win32":
    GIMP_BIN = find_binary(GIMP_BIN_WIN)
    CONSOLE_BIN = find_binary(CONSOLE_BIN_WIN)
elif _platform == "linux" or _platform == "linux2":       
    GIMP_BIN = find_binary(GIMP_BIN_LINUX)
    CONSOLE_BIN = find_binary(CONSOLE_BIN_LINUX)

def get_gimp_bin():
    return GIMP_BIN

def get_console_bin():
    return CONSOLE_BIN

def heal_cover(filename, opt0="10", opt1="2"):
    args = "--stack-trace-mode=never -d -f -s -n -i -b '(let* ((run-mode 1) "
    args += '(samplingRadiusParam ' + opt0 + ') (orderParam ' + opt1 + ') ' \
        '(filename "' + filename + '")'
    args += """(image (car (gimp-file-load run-mode filename filename) ))
               (drawable (car (gimp-image-get-active-layer image)  )))
                (gimp-image-undo-disable image)
          (python-fu-heal-transparency run-mode image drawable
          samplingRadiusParam orderParam)
          (gimp-file-save run-mode image drawable filename "content-aware")
          (gimp-quit 0)
         )'"""
    return gimp_fu_it(args, True)


def gimp_ng_cover(filename):
    # (elsamuko-national-geographic run-mode image drawable
    # value value value value value toggle option)
    # http://registry.gimp.org/node/9592

    # args = "--stack-trace-mode=never -n -i -b "
    args = "-n -i -b "
    args += "\"(elsamuko-national-geographic-batch "

    # v0.9.12: special treatment for windows console
    if _platform == "win32":
        filename = filename.replace('/', '\\')
        filename = filename.replace('\\', '\\\\')

    args += '\\"' + filename + '\\"'
    args += ' 60 1 60 25 0.4 1 0)" -b "(gimp-quit 0)"'
    return gimp_fu_it(args)


def gimp_fu_it(args, win_replace=False):
    cmd = "1"
    prg = "1"
    result = False

    # linux
    if _platform == "linux" or _platform == "linux2":               
        if prg != "":
            cmd = CONSOLE_BIN + " " + args
            try:
                with sanitize_env_vars():
                    os.system(cmd)
                    result = True
            except Exception:
                result = False
        else:
            result = False

    # windows
    elif _platform == "win32":        
        if GIMP_PATH_WIN != "" and CONSOLE_BIN != "":            
            startupinfo = subprocess.STARTUPINFO()
            args = args.replace("\n", " ")
            cond = False
            while not cond:
                new = args.replace("  ", " ")
                cond = (args == new)
                args = new

            if win_replace:
                args = args.replace('\\', '/')
                args = args.replace('"', '\\"')
                args = args.replace("'", '"')
            startupinfo.dwFlags |= subprocess.STARTF_USESHOWWINDOW
            startupinfo.wShowWindow = subprocess.SW_HIDE
            cmd = CONSOLE_BIN + " " + args
            try:
                with sanitize_env_vars():                    
                    subprocess.call(cmd, startupinfo=startupinfo)
                    result = True
            except Exception:
                result = False
        else:
            result = False

    # TODO: MAC. elif _platform == "darwin":
    return result

def run_gimp(cmdline):
    result = False
    if _platform == "linux" or _platform == "linux2":
        try:
            with sanitize_env_vars():
                os.system(cmdline)
                result = True
        except Exception:
            result = False

    elif _platform == "win32":
        """
        gimp-win, or cmd doesn't like backslashes
        gimp-win, or cmd doesn't like double quotes within double quotes
        gimp-win, or cmd doesn't like single quotes
        """
        cmdline = cmdline.replace('\\', '/')
        cmdline = cmdline.replace('"', '\\"')
        cmdline = cmdline.replace("'", '"')

        #  fix encoding for python2
        if sys.version_info.major < 3:
            import locale
            os_encoding = locale.getpreferredencoding()
            cmdline = cmdline.encode(os_encoding)

        try:
            with sanitize_env_vars():
                devnull = open(os.devnull, 'w')
                subprocess.call(cmdline, stdout=devnull, stderr=subprocess.STDOUT)
                result = True
        except Exception:
            result = False
    return result

def find_plugin_ng():
    """ Lets check for gimp plugins

    - National Geographic
    Book covers optimized with the N.G. plugins
    look better on 16-color black & white e-reader.
    For now it's just this one.

    :returns: True/False
    """
    path = ""
    ng_script = "elsamuko-national-geographic.scm"
    if _platform == "linux" or _platform == "linux2":
        path = "/usr/share/gimp/2.0/scripts/"
        path2 = "~/.gimp-2.10/scripts/"
    elif _platform == "win32":                 
        path = GIMP_PATH_WIN + "\\share\\gimp\\2.0\\scripts\\"
        path2 = os.getenv('APPDATA')+"\\GIMP\\2.10\\scripts\\"
    path += ng_script
    path2 += ng_script
    if len(path) > 0 and len(path2) >0:
        return os.path.exists(path) or os.path.exists(path2)
    return False
