#!/Python3/python
# -*- coding: utf-8 -*-

from __future__ import unicode_literals, division, absolute_import, print_function

import os, os.path, sys, shutil, inspect, re, time
import options
from PIL import Image
from decimal import *  
from urllib.parse import unquote
from random import sample
from xml.sax.saxutils import escape
import uuid

import tkinter as tk
import tkinter.messagebox as mbox

try:
    from sigil_bs4 import BeautifulSoup, Comment, Tag
except:
    from bs4 import BeautifulSoup, Comment, Tag
    
   
__all__=["processAllTasks", "copyTextFiles2Dir", "copyNCXFile2Dir", "writeNCXFile2Epub", "writeFiles2Epub", "repairHTMLIDs", "repairNCXIDs", "checkOPFIDs", "show_msgbox", "cleanExit", "checkandRepairID", "checkOPFManifestID", "checkOPFSpineID", "prettifyXHTMLFile", "getLineNumberOPF", "getLineNumberNCX", "getLineNumberHTML", "getLineNumberSpine", "addDocType", "removeBlankLines", "replaceBookID", "updateNCXBookID", "updateOPFBookID", "checkBookID", "getLineNumberNCXBookID", "getLineNumberOPFBookID", "reformatNCXFile", "prettifyNCXFile", "repairGuideRefs", "svgAttributes2CamelCase"]


def processAllTasks(bk, wdir):
    print('\n -- Processing automatic tasks...')
    options.RESULTS = []
    
    # copy xhtmml text files to the work dir 
    t_ids, t_fnames = copyTextFiles2Dir(bk, wdir)  
    
    print('\n>>> Cover file...' + t_fnames[0])
    removeCalibreCoverID(wdir, t_fnames[0])
    
    if isValidEpub(bk, wdir, t_fnames) == False:
        print(' >>> Abort plugin...')
        options.SYS_EXIT = True
        return(0) 
    
    # get the epub toc.ncx file
    copyNCXFile2Dir(bk, wdir)
        
    # check and repair all html ids
    for file in t_fnames:
        file = os.path.join(wdir, file)
        removeBlankLines(wdir, file)
        repairHTMLIDs(wdir, file)
        svgAttributes2CamelCase(wdir, file)
   
    # check and repair ncx ids
    repairNCXIDs(bk, wdir, bk.href_to_basename(bk.id_to_href(bk.gettocid())))
 
    # removes ids from guide hrefs
    repairGuideRefs(bk, wdir) 
 
    # check opf ids    
    checkOPFIDs(bk) 
    
    
    # write work dir files back to the epub
    writeFiles2Epub(bk, wdir, t_ids, t_fnames)
    writeNCXFile2Epub(bk, wdir)
    return(0)    
    
def copyTextFiles2Dir(bk, wdir):
    t_ids = list()
    t_hrefs = list()
    t_fnames = list()
    
    for (id, href) in bk.text_iter():
        t_ids.append(id)
        t_hrefs.append(href)
        t_fnames.append(os.path.basename(href))
    
    # copy all xhtml files to the working dir    
    file = str()
    t_fnames_r = list()
    t_ids_r = list()
    i = 0      
    for id in t_ids:
        file = os.path.join(wdir, t_fnames[i])
        print(' -- Copy to work dir...' + t_fnames[i])
        with open(file, 'wt', encoding='utf-8') as outfp:
            try:
                data = bk.readfile(id)          
            except:
                msg = "Critical Error: Cannot copy epub file - " + os.path.basename(file) + " - to working dir.\nFile id(" + id + ") not recognized."  
                show_msgbox('Critical Error', msg, msgtype='error')
                raise Exception(msg)        
                options.SYS_EXIT = True
                return(0) 
                
            #html = BeautifulSoup(data, 'html.parser')
            t_fnames_r.append(t_fnames[i])
            t_ids_r.append(id)
            outfp.writelines(data)    
            i = i + 1
                
    return(t_ids_r, t_fnames_r)      
    
def copyNCXFile2Dir(bk, wdir):    
    ncx_id = bk.gettocid()
    ncx_href = bk.id_to_href(ncx_id)
    ncx_file = bk.href_to_basename(ncx_href)
    #ncx_file = 'toc.ncx'
    
    #print("ncx id is...." + ncx_id)
    
    file = os.path.join(wdir, ncx_file)
    print(' -- Copy NCX file to work dir...' + file)
    with open(file, 'wt', encoding='utf-8') as outfp: 
        try:
            data = bk.readfile(ncx_id)
            #soup = BeautifulSoup(data, 'xml')
            outfp.writelines(data)
        except:
            msg = "Critical Error: Cannot copy html file to the working dir.\nInvalid file id(" + ncx_id + ") not recognized."    
            show_msgbox('Critical Error', msg, msgtype='error')
            raise Exception(msg)      
            options.SYS_EXIT = True
            return(0)    
                        
    return(ncx_id)
    
def writeNCXFile2Epub(bk, wdir):
    ncx_id = bk.gettocid()
    file = os.path.join(wdir, bk.href_to_basename(bk.id_to_href(ncx_id)))
    print(' -- Write NCX file to Epub...' + file)
    with open(file, 'rt', encoding='utf-8') as fp:
        data = fp.read()
        try:        
            bk.writefile(ncx_id, data)
        except:
            msg = "Critical Error: Cannot write the " + bk.href_to_basename(bk.id_to_href(ncx_id)) + " file back to epub.\nFile id('" + ncx_id + "') not recognized"    
            show_msgbox('Critical Error', msg, msgtype='error')
            raise Exception(msg)        
            options.SYS_EXIT = True
            return(0)
            
    return(0)        
    
def writeFiles2Epub(bk, wdir, ids, fnames):
    print(' >>> In writeFiles2Epub()...' )       
    for file in fnames:    
       print(file)    

    count = len(fnames)
    if count == 0: 
        return(0)
    
    i = 0
    print(' ')  
    for file in fnames:
        print(' -- Write xhtml files to epub...' + fnames[i])
        file = os.path.join(wdir, file)
        with open(file, 'rt', encoding='utf-8') as fp:
            data = fp.read()  
            try:            
                bk.writefile(ids[i], data)
                i = i + 1
            except:
                msg = "Critical Error: Cannot write " + os.path.basename(file) + " back to epub.\nFile id(" + ids[i] + ") or file name not recognized."       
                show_msgbox('Critical Error', msg, msgtype='error')
                raise Exception(msg)     
                options.SYS_EXIT = True
                return(0)
                            
    return(0)      
       
def repairHTMLIDs(wdir, file):
    outfile = os.path.join(wdir, 'html_repair.html')
    outfp = open(outfile, 'wt', encoding='utf-8')
    html = open(file, 'rt', encoding='utf-8').read()
    
    soup = BeautifulSoup(html, 'html.parser')
    
    # convert 'name' to 'id' attribute
    for tag in soup.body.find_all():
        if tag.has_attr('name'):      
            tag['id'] = tag['name']
            del tag['name']
    
    # check and repair xhtml file id attributes
    for a_tag in soup.body.find_all(id=True):
        if a_tag.has_attr('id'):
            old_id = a_tag['id']
            reason, new_id, report_only = checkandRepairID(a_tag['id'])
            a_tag['id'] = new_id  
            if old_id != new_id and report_only == False:
                options.RESULTS.append((os.path.basename(file), 'line ' + getLineNumberHTML(file, old_id), '\n' + reason +'"' + old_id  + '"\nReplaced with: "' + new_id + '"\n'))                
            elif report_only == True:
                options.RESULTS.append((os.path.basename(file), 'line ' + getLineNumberHTML(file, old_id), '\n' + reason + old_id  + '\nNot Fixed\n'))      
        else:
            if 'calibre:cover' in str(a_tag):
                a_tag.decompose()
            
    # check and repair hrefs(with bookmarks)
    for href_tag in soup.body.find_all():
        if href_tag.has_attr('href') and '#' in href_tag['href']:
            if 'http:' not in href_tag['href'] and 'https:' not in href_tag['href']:
                old_href = href_tag['href']
                link, id = href_tag['href'].split('#')
                reason, new_id, report_only = checkandRepairID(id)
                new_id = link + '#' + new_id   
                href_tag['href'] = new_id
                if old_href != new_id and report_only == False:
                    options.RESULTS.append((os.path.basename(file), 'line ' + getLineNumberHTML(file, old_href), '\n' + reason + '"' + unquote(old_href, encoding='utf-8') + '"\nReplaced with: "' + unquote(href_tag['href'], encoding='utf-8') + '"\n'))                
                elif report_only == True:
                    options.RESULTS.append((os.path.basename(file), 'line ' + getLineNumberHTML(file, old_href), '\n' + reason + '"' + unquote(old_href, encoding='utf-8') + '"\nNot Fixed\n'))     
    
    outfp.writelines(str(soup))  
    outfp.close()    
    os.remove(file)
    os.rename(outfile, file)
    
def repairNCXIDs(bk, wdir, file):
    print('\n >>> In repairNCXIDs...')
    prettifyNCXFile(bk, wdir)
    
    file = os.path.join(wdir, file)
    outfile = os.path.join(wdir, 'xml_repair.html')
    outfp = open(outfile, 'wt', encoding='utf-8')
    html = open(file, 'rt', encoding='utf-8').read()

    soup = BeautifulSoup(html, 'xml')
    
    # check/repair the ncx book id
    ncx_id = bk.gettocid()
    ncx_href = bk.id_to_href(ncx_id)
    ncx_file = bk.href_to_basename(ncx_href)    
    for meta in soup.find_all('meta'):
        if meta['name'] == 'dtb:uid':
            uid = meta['content']
            if list(uid)[0].isdigit() or not uid.startswith('urn:uuid:'):
                reason = checkBookID(uid)
                meta['content'] = replaceBookID()
                updateOPFBookID(bk, meta['content'])
                options.RESULTS.append((ncx_file +'(Book ID)', 'line ' + getLineNumberNCXBookID(wdir, file, uid), '\n' + reason + '"' + uid + '"\nReplaced with: "' + meta['content'] + '\n'))                
            
    # check/repair navPoint ids
    for nav in soup.find_all('navPoint'):
        if nav.has_attr('id'):
            old_id = nav['id']
            reason, new_id, report_only = checkandRepairID(nav['id']) 
            nav['id'] = new_id
            if old_id != new_id and report_only == False:
                options.RESULTS.append((ncx_file, 'line ' + getLineNumberNCX(wdir, file, old_id), '\n' + reason + '"' + old_id  + '"\nReplaced with: "' + new_id + '"\n'))
            elif report_only == True:
                options.RESULTS.append((ncx_file, 'line ' + getLineNumberNCX(wdir, file, old_id), '\n' + reason + '"' + old_id + '"\nNot Fixed\n'))

    # remove id refs from content hrefs
    for content in soup.find_all('content'):
        href = content['src']
        if '#' in href:
            href = href.split('#')[0]
            content['src'] = href       
            
    outfp.writelines(str(soup.prettify()))
    outfp.close()
    os.remove(file)
    os.rename(outfile, file)    
    return(0)
    
def checkOPFIDs(bk):
    print('\n >>> Check opf ids...')
    opf_data = bk.get_opf()
  
    metadata = bk.getmetadataxml()
    
    # check the ebook string uid
    soup = BeautifulSoup(metadata, 'xml')
    for meta in soup.find_all(id=True, limit=1):
        uid = meta.get_text()
        if list(uid)[0].isdigit() or not uid.startswith('urn:uuid:'):
            reason = checkBookID(uid)
            meta.string = replaceBookID()
            updateNCXBookID(bk, meta.string)
            options.RESULTS.append(('content.opf(Book ID)', 'line ' + getLineNumberOPFBookID(opf_data, uid), '\n' + reason + '"' + uid + '"\nReplaced with: "' + meta.string + '\n'))
    
    # check the id attr in the opf book id line 
    for meta in soup.find_all(id=True, limit=1):
        aid = meta['id']
        reason, new_aid, report_only = checkandRepairID(meta['id']) 
        meta['id'] = new_aid
        if new_aid != aid and report_only == False:
            options.RESULTS.append(('content.opf', 'line ' + getLineNumberOPF(opf_data, aid), '\n' + reason + '"' + aid  + '"\nReplaced with: "' + new_aid + '"\n'))
        elif report_only == True:
            options.RESULTS.append(('content.opf', 'line ' + getLineNumberOPF(opf_data, aid), '\n' + reason + '"' + aid + '"\nNot Fixed\n'))   
       
    bk.setmetadataxml(str(soup.prettify()))          
       
    # check all manifest ids(not fixed) 
    soup = BeautifulSoup(opf_data, 'xml') 
    for item in soup.find_all('item'):
        if item.has_attr('media-type'):
            old_id = item['id']
            reason, new_mid, report_only = checkOPFManifestID(old_id)
            if new_mid != old_id and report_only == False:
                options.RESULTS.append(('content.opf(manifest)', 'line ' + getLineNumberOPF(opf_data, old_id), '\n' + reason + '"' + old_id  + '"\nNot Fixed\n'))
            elif report_only == True:
                options.RESULTS.append(('content.opf(manifest)', 'line ' + getLineNumberOPF(opf_data, old_id), '\n' + reason + '"' + old_id  + '"\nNot Fixed\n'))     
   
    """ 
    # check the manifest toc id(not fixed)
    toc_id = bk.gettocid()
    reason, new_toc_id, report_only = checkandRepairID(toc_id)
    if new_toc_id != toc_id and report_only == False:
        options.RESULTS.append(('content.opf(toc)', 'line ' + getLineNumberOPF(opf_data, old_id), '\n' + reason + '"' + old_id  + '"\nNot Fixed\n'))
    elif report_only == True:
        options.RESULTS.append(('content.opf(toc)', 'line ' + getLineNumberOPF(opf_data, old_id), '\n' + reason + '"' + old_id  + '"\nNot Fixed\n'))    
    """        
            
    # check spine ids(not fixed)        
    s_data = bk.getspine() 
    for data in s_data:
        old_sid = data[0]
        reason, new_sid, report_only = checkOPFSpineID(old_sid)
        if new_sid != old_sid and report_only == False:
            options.RESULTS.append(('content.opf(spine)', 'line ' + getLineNumberSpine(opf_data, old_sid), '\n' + reason + '"' + old_sid  + '"\nNot Fixed\n'))
        elif report_only == True:
            options.RESULTS.append(('content.opf(spine)', 'line ' + getLineNumberSpine(opf_data, old_sid), '\n' + reason + '"' + old_sid  + '"\nNot Fixed\n'))         
            
    return(0)    
    
def show_msgbox(title, msg, msgtype='info'):
    """ For general information, warnings and errors
    """
    localRoot = tk.Tk()
    localRoot.withdraw()
    localRoot.option_add('*font', 'Helvetica -12')
    localRoot.quit()
    if msgtype == 'info':
        return(mbox.showinfo(title, msg))
    elif msgtype == 'warning':
        return(mbox.showwarning(title, msg))
    elif msgtype == 'error':
        return(mbox.showerror(title, msg))
    
def cleanExit(wdir):
    shutil.rmtree(wdir, ignore_errors=True)
    return(0)    
    
def checkandRepairID(id):
    reason = ''
    report_it = False
    new_id = ''
    
    id = id.strip()
    print('\n>>> Check ID...' + id)
    
    # replace id spaces with underscore
    if ' ' in id or '%20' in id:
        id = escape(id)
        id = id.replace(' ', '_')
        id = id.replace('%20', '_')
        reason= 'Illegal spaces: '
    
    char_list  = list(id)
    # ensure first char is always an alpha char
    if char_list[0].isdigit():
        reason = 'Illegal first char digit: '
        new_id = "".join(char_list)
        id = 'x' + new_id    
    
    # check that the id contains legal non-alphanumeric chars
    new_list = list(id)
    for indx, ch in enumerate(new_list):
        
        print('>>> In non-alphanum ch iter...id ' + ch)
        if not ch.isalnum(): 
            if ch != '-' and ch != '_' and ch != '.':
                new_list[indx] = 'x'
                
        if indx == len(new_list) - 1:        
            id = "".join(new_list)
            reason = 'Illegal non-alphanumeric char: '
            break
       
    return(reason, id, report_it)
    
def checkOPFManifestID(id):
    reason = ''
    report_it = False
    new_id = ''
    
    id = id.strip()
    print('\n>>> Check ID...' + id)
    
    char_list  = list(id)
    first_char = char_list[0]
    # ensure first char is always an alpha char
    if first_char.isdigit():
        reason = 'Illegal first char digit: '
        new_id = 'x' + id
        report_it = True        
        #return(reason, new_id, report_it)
    
    # check that the id contains legal non-alphanumeric chars
    for ch in char_list:
        if not ch.isalnum():
            if ch != '-' and ch != '_' and ch != '.':
                reason = 'Illegal non-alphanumeric char: '
                report_it = True       
                return(reason, id, report_it)
       
    return(reason, id, report_it)    
    
def checkOPFSpineID(id):
    reason = ''
    report_it = False
    new_id = ''
    
    id = id.strip()
    
    char_list  = list(id)
    first_char = char_list[0]
    # ensure first char is always an alpha char
    if first_char.isdigit():
        reason = 'Illegal first char digit: '
        new_id = 'x' + id      
        return(reason, new_id, report_it)
    
    # check that the id contains legal non-alphanumeric chars
    for ch in char_list:
        if not ch.isalnum():
            if ch != '-' and ch != '_' and ch != '.':
                reason = 'Illegal non-alphanumeric char: '
                report_it = True       
                return(reason, id, report_it)
       
    return(reason, id, report_it)        
    
def prettifyXHTMLFile(wdir, file):
    # reformat and prettify the XHTML file
    outfile= os.path.join(wdir, 'final_one.css')
    infp = open(file, 'rt', encoding='utf-8')
    outfp = open(outfile, 'wt', encoding='utf-8')
     
    for line in infp:
            
        if line.strip() == '<style>':
            line = '<style type="text/css">'
            outfp.write(line)
            continue            
            
        if line.strip().startswith('<body'):
            line = '<body class="globals">\n'
    
        if '<p></p>' in line.strip() or \
            '<p> </p>' in line.strip():
            continue
         
        if line.strip().startswith('<a') or \
            line.strip().startswith('<span'):
            continue
        
        if '<font>' in line:
            line = line.replace('<font>', '').replace('</font>', '')
            
        line = line.replace(r'&nbsp;', ' ')
        line = line.replace(r'&#160;', ' ')        
        line = line.replace(r'&amp;#160;', ' ')
        line = line.replace(r'&amp;#nbsp;',r'') 
        line = line.replace(r"&#146;", "’")         
        line = line.replace(r"&amp;#146;", "’")      
        line = line.replace(r'&amp;#9;', '')
        line = line.replace("<!--?xml version='1.0' encoding='utf-8'?-->", "")
        
        if line.strip().startswith('<?xml') or \
            line.strip().startswith('<!DOCTYPE') or \
            line.strip().startswith('<html') or \
            line.strip().startswith('<head>') or \
            line.strip().startswith('<meta')or \
            line.strip().startswith('<title>') or \
            line.strip().startswith('<link') or \
            line.strip().startswith('</head>') or \
            line.strip().startswith('<body'):
            line = line.strip()
            if not line:
                continue
            if line.startswith('<meta') or \
                line.startswith('<title>') or \
                line.startswith('<link'):
                line = '  ' + line      
            if line.startswith('<body'):
                line = '\n' + line
            if line.startswith('</body>'):
                outfp.write('\n' + line.rstrip() + '\n')
            else:
                outfp.write(line.rstrip() + '\n')    
        else:
            line = line.strip() 
            if not line:
                continue
            if line.startswith('<p'):
                line = '  ' + line            
            outfp.write('\n' + line + '\n')    
            
    infp.close()
    outfp.close()
    os.remove(file)
    os.rename(outfile, file)
    return(0)                      
    
def getLineNumberOPF(opf, text):
    print('\n >>> In getLineNumberOPF...')
    opf_data = opf.splitlines()
    linenum = ''
    
    # assign a line no to the manifest id line
    for index, line in enumerate(opf_data):
        if text in line and '<itemref' not in line:
            linum = index + 1
            linenum = str(linum)
            return(linenum)    
    
def getLineNumberNCX(wdir, file, text):
    html = open(file, 'rt', encoding='utf-8').read()    
    lines = html.splitlines()
    
    linenum = ''
    # assign a line no to the id line
    for index, line in enumerate(lines):
        if text in line:
            linum = index + 1
            linenum = str(linum)
            return(linenum)  
    
def getLineNumberHTML(file, text):
    print('\n >>> In getLineNumberHTML...')
    output = os.path.join(options.WDIR, 'tester.html')
    outfp = open(output, 'wt', encoding='utf-8')
    with open(file, 'rt', encoding='utf-8') as infp:
        for line in infp:
            #if '<!DOCTYPE' in line:
            #    line = ''    
            if line.strip() == '':
                line = ''
            outfp.write(line) 
    outfp.close()
    os.remove(file)
    os.rename(output, file)    
    
    html = open(file, 'rt', encoding='utf-8').read()     
    lines = html.splitlines()
    linenum = ''
    
    # assign a line no to the id line
    for index, line in enumerate(lines):
        if text in line:
            linum = index + 2
            linenum = str(linum)
            return(linenum)  
            
def getLineNumberSpine(opf, text):
    print('\n >>> In getLineNumberSpine...')
    opf_data = opf.splitlines()
    linenum = ''
    
    # assign a line no to the id line
    for index, line in enumerate(opf_data):
        if text in line and '<itemref' in line:
            linum = index + 1
            linenum = str(linum)
    return(linenum)    
    
def addDocType(wdir, file):
    output = os.path.join(options.WDIR, 'tester.html')
    outfp = open(output, 'wt', encoding='utf-8')
    with open(file, 'rt', encoding='utf-8') as infp:
        doctype = '<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.1//EN"\n  "http://www.w3.org/TR/xhtml11/DTD/xhtml11.dtd">\n' 
        for line in infp:     
            if line.strip() == '<?xml version="1.0" encoding="utf-8"?>':
                line = '<?xml version="1.0" encoding="utf-8"?>\n' + doctype
                outfp.write(line)
            else:
                outfp.write(line)
                
    outfp.close()
    os.remove(file)
    os.rename(output, file)   
    
    output = os.path.join(options.WDIR, 'tester2.html')
    outfp = open(output, 'wt', encoding='utf-8')
    with open(file, 'rt', encoding='utf-8') as infp:
        for line in infp:
            if '<!DOCTYPE' in line:
                line = line.replace('\n\n', '\n')
            outfp.write(line)            
    
    outfp.close()
    os.remove(file)
    os.rename(output, file)   
    
def removeBlankLines(wdir, file):
    output = os.path.join(options.WDIR, 'blanklines.html')
    outfp = open(output, 'wt', encoding='utf-8')
    with open(file, 'rt', encoding='utf-8') as infp:
        for line in infp:
            if line.strip() == '':
                line = ''
            outfp.write(line)    
    
    outfp.close()
    os.remove(file)
    os.rename(output, file)     

def replaceBookID():   
    # create a new book id 
    uid = uuid.uuid4()
    uid_str = 'urn:uuid:' + str(uid)
    print('\n >>> book uuid string generated...' + uid_str)
    return(uid_str)
   
def updateNCXBookID(bk, uuid):
    output = os.path.join(options.WDIR, 'update_ncx.html')
    file = os.path.join(options.WDIR, bk.href_to_basename(bk.id_to_href(bk.gettocid())))
    outfp = open(output, 'wt', encoding='utf-8')
    html = open(file, 'rt', encoding='utf-8').read()
    
    soup = BeautifulSoup(html, 'xml')
    
    for meta in soup.find_all('meta'):
        if meta['name'] == 'dtb:uid':
            meta['content'] = uuid
                
    outfp.writelines(str(soup))
    outfp.close()
    os.remove(file)
    os.rename(output, file)
    return(0)    
    
def updateOPFBookID(bk, uuid):
    metadata = bk.getmetadataxml()
    soup = BeautifulSoup(metadata, 'xml')
    for meta in soup.find_all(id=True, limit=1):
        meta.string = uuid
    bk.setmetadataxml(str(soup)) 
    return(0)        
                
                
def checkBookID(uid):
    reason = ''

    if list(uid)[0].isdigit():
        reason = 'Illegal first char digit: ' 
        return(reason)       
 
    if not uid.startswith('urn:uuid:'):          
        reason = 'Bad format: '
        return(reason)
        
    return(reason)    
    
def getLineNumberNCXBookID(wdir, file, text):
    print('\n >>> In getLineNumberNCXBookID...')
    html = open(file, 'rt', encoding='utf-8').read()
    lines = html.splitlines()
    
    lin = []
    for liner in lines:
        if liner.strip() == '':
            continue
        lin.append(liner)
    lines = lin 
    
    linenum = ''
    # assign a line no to the book id line
    for index, line in enumerate(lines):
        if text in line and '<meta' in line and 'name' in line: 
            linum = index + 1
            linenum = str(linum)
            return(linenum)  
            
def getLineNumberOPFBookID(opf, text):
    print('\n >>> In getLineNumberOPFBookID...')
    opf_data = opf.splitlines()
    linenum = ''
    
    # assign a line no to the book id line
    for index, line in enumerate(opf_data):
        if text in line and 'dc:identifier' in line:
            linum = index + 1
            linenum = str(linum)
            return(linenum)                
            
def reformatNCXFile(wdir):
    # removes the line number problem
    file = os.path.join(wdir, bk.href_to_basename(bk.id_to_href(gettocid())))
    output = os.path.join(wdir, 'ncx_reformat.html')
    outfp = open(output, 'wt', encoding='utf-8')
    with open(file, 'rt', encoding='utf-8') as infp:
        for line in infp:
            if 'http:' in line and 'ncx-2005-1.dtd' in line and 'version' in line and 'xmlns' in line:
                line = line.replace('<ncx', '\n<ncx')
                outfp.write(line)
            else:
                outfp.write(line)            
    outfp.close()
    os.remove(file)
    os.rename(output, file)
    return(0)
    
def prettifyNCXFile(bk, wdir):
    file = os.path.join(wdir, bk.href_to_basename(bk.id_to_href(bk.gettocid())))
    outfile= os.path.join(wdir, 'prettyNCX.css')
    outfp = open(outfile, 'wt', encoding='utf-8')
    html = open(file, 'rt', encoding='utf-8').read()
    
    soup = BeautifulSoup(html,'xml')
    
    outfp.writelines(str(soup.prettify()))
    outfp.close()
    os.remove(file)
    os.rename(outfile, file)
    return(0)
    
def isValidEpub(bk, wdir, t_fnames):
    # check for epub file not loaded
    if len(t_fnames) == 1 and t_fnames[0] == 'Section0001.xhtml':
            msg = 'Epub contains no data. Ensure that you load your epub into Sigil before you run this plugin. Please try again.'
            show_msgbox('Critical Error', msg, msgtype='error')
            print('\n >>> Error: ' + msg)
            return(False)    
    
    # check for invalid imported html file 
    if len(t_fnames) == 1 and ('.html' in t_fnames[0] or '.htm' in t_fnames[0]):
        msg='Only valid epubs should be used with this plugin. Please try again.'
        show_msgbox('Crirical Error', msg, msgtype='error')
        print('\n >>> Error: ' + msg)
        return(False)    
        
    return(True)    
    
def repairGuideRefs(bk, wdir):
    """ Removes all unnecessary ids from the guide hrefs """
    
    new_guide = []
    for (type, title, href, id) in bk.guide_iter():
        if '#' in href:
            new_href = href.split('#')[0]
            new_guide.append((type, title, new_href))
            bk.setguide(new_guide)
        else:
            new_guide.append((type, title, href))
            bk.setguide(new_guide)        
            
    return(0)        
    
def removeCalibreCoverID(wdir, file):
    
    print('\n>>> In removeCalibreCoverID()...')
    file = os.path.join(wdir, file)
    output = os.path.join(wdir, 'remove_meta_id.html')
    outfp = open(output, 'wt', encoding='utf-8')
    html = open(file, 'rt', encoding='utf-8').read() 

    soup = BeautifulSoup(html, 'html.parser')

    for tag in soup.find_all('meta'):
        if tag != None and tag.has_attr('name') and tag['name'] == 'calibre:cover':
            tag.decompose()
          
    outfp.writelines(str(soup))      
    outfp.close()
    os.remove(file)
    os.rename(output, file)
    return(0)    
    
def svgAttributes2CamelCase(wdir, file):
    
    file = os.path.join(wdir, file)
    file = os.path.join(wdir, os.path.basename(file))       
    output = os.path.join(wdir, 'reformat.html')
    outfp = open(output, 'wt', encoding='utf-8')
    html = open(file, 'rt', encoding='utf-8').read()

    soup = BeautifulSoup(html, 'html.parser')
    
    for svg in soup.find_all('svg'):
    
        # reformat the par to camel case
        if svg.has_attr('preserveaspectratio'):
            par = svg['preserveaspectratio']
            del svg['preserveaspectratio']
            svg['preserveAspectRatio'] = par
        
        # reformat the vb to camel case
        if svg.has_attr('viewbox'):
            vb = svg['viewbox']
            del svg['viewbox']
            svg['viewBox'] = vb
               
    outfp.writelines(str(soup.prettyprint_xhtml(indent_level=0, eventual_encoding="utf-8", formatter="minimal", indent_chars="  ")))
    outfp.close()
    os.remove(file)
    os.rename(output, file)        
    return(0)        
                