#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
This plugin will import a text file into an ePub and format it
Based on v0.1.0.5

When the plugin runs it displays a dialog. This includes an option to convert
angular (<>) brackets to html code.
When the user clicks the button 'Get text file' the method ProcessTextFile()
runs. This displays a file dialog box from which the user can select one or more
text files in a folder,
When the user clicks the close button, the text files are added to the html
file.

"""
import tkinter.ttk as ttk  # Essential for ttk. commands
from tkinter import *  # Essential for root = Tk()
from tkinter import messagebox as tkMessageBox        #Needed for messagebox
from tkinter import filedialog  #Needed for the file dialog box
from sigil_bs4 import BeautifulSoup #Needed for detecting the encoding of the text file
import tkinter as tk
import locale
import ntpath
from GenUtils import centerWindow
from UpdateChecker import CheckForUpdates, SetUpdateTimes  #Needed for checking whether the latest version of this plugin is running    

class cImportTextFile():
    dlgTop="" #This is set to self.top below 
    def __init__(self, parent, bk, prefs):
        self.top = tk.Toplevel(parent)
        self.top.title("Text importer")
        cImportTextFile.dlgTop=self.top    #used to ensure this dialog window is destroyed before main section continues
        self.bk=bk
        self.prefs = prefs
        self.is_checked = IntVar()
        
        self.is_checked.set(self.prefs['ConvertAngularBrackets'])
        
        self.dlgframe = ttk.Frame(self.top, padding="15 15 12 12")
        self.dlgframe.grid(column=0, row=0, sticky=(N, W, E, S))
        self.dlgframe.columnconfigure(1, weight=1)
        self.dlgframe.rowconfigure(1, weight=1)

        self.check = ttk.Checkbutton(self.dlgframe, text="Convert angular brackets to html code", variable=self.is_checked)
        self.check.grid(column=0, row=0, sticky=W)
        ttk.Label(self.dlgframe, text="").grid(column=0, row=1, sticky=(E, W))    #Blank row

        tk.Button(self.dlgframe, text="Set time before checking for updates", command=lambda: SetUpdateTimes(self.prefs, self.top)).grid(column=0, row=2)

        ttk.Label(self.dlgframe, text="").grid(column=0, row=3, sticky=(E, W))    #Blank row
        tk.Button(self.dlgframe, text='Get text file', command = self.ProcessTextFile).grid(column=0, row=4, sticky=W, pady=4)
        tk.Button(self.dlgframe, text="Close", command=self.Close, width = 15).grid(column=1, row=4, sticky=(E, W))
        centerWindow(self.top)

    def ProcessTextFile(self):
        """
        This method runs when the button marked 'Get text file' is clicked
        """
        #Request name(s) of the files to open
        FILEOPENOPTIONS = dict(title='Choose a text file to import', initialfile = '', filetypes=[('Text files', ('.txt')), ('All files', ('*.*'))])
        fileList = filedialog.askopenfilenames(**FILEOPENOPTIONS)
        
        for i, filename in enumerate(fileList):
        
            #Get the encoding of the text file
            with open(filename, "rb") as binary_file:
                data = binary_file.read()
                soup = BeautifulSoup(data) 
            
            #Read the file
            with open(filename, 'rt', encoding=soup.original_encoding) as f:
                bodyText = f.read() 
            
            #Replace angular brackets if required
            if self.is_checked.get() == True:
                print("Changing brackets")
                bodyText = bodyText.replace('<', '&lt;')
                bodyText = bodyText.replace('>', '&gt;')
                
            #Replace ambersand sign with html equivalent    
            bodyText = bodyText.replace('&', '&amp;;')
            
            #Replace newlines with paragraph tags
            bodyText = bodyText.replace('\n', '</p>\n\n<p>').replace('\r', '')
            bodyText= bodyText.replace('<p></p>', '<p>&nbsp;</p>')
            
            #Now write the xhtml file
            xml  = '<?xml version="1.0" encoding="UTF-8" standalone="no" ?>\n'
            xml += '<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.1//EN" "http://www.w3.org/TR/xhtml11/DTD/xhtml11.dtd">\n'
            xml += '<html xmlns="http://www.w3.org/1999/xhtml">\n'
            xml += '<head>\n'
            xml += '  <title></title>\n' 
            xml += '</head>\n'
            xml += '<body>\n<p>'
            xml += bodyText
            xml += '</p>'
            xml += '\n</body>\n'
            xml += '</html>\n'
        
            #Set the name of the new xhtml section in the ePub to that of the filename
            Filename=filename
            head, fName = ntpath.split(filename)
            ChapterName = fName[:fName.index(".")] #Remove extension ;fHandle - may need to remove leading .../
        
            #Check whether this file already exists in the ePub
            for (id, href) in self.bk.text_iter():
                if id==ChapterName or href=='Text/'+ChapterName+ '.xhtml':    #If the section already exists
                    reply=tkMessageBox.askquestion("WARNING", "Do you want to delete the current page named "+ ChapterName+".xhtml?")
                    if reply=="yes":        #and it is not wanted
                        bk.deletefile(id)   #then delete it
                    else:                   #otherwise do not import the text file
                        print("Present xhtml page has been retained.")
                        return
                        
            #Add text file to ePub in a new xhtml section
            uid  = ChapterName
            basename = uid +'.xhtml'
            mime = 'application/xhtml+xml'
            self.bk.addfile(uid, basename, xml, mime)

    def Close(self):
        """
        Called when the user clicks the "Close" button
        It destroys the dialog window
        """
        self.prefs['ConvertAngularBrackets'] = self.is_checked.get()
        self.top.destroy()
        

def run(bk):
    root = tk.Tk()
    root.withdraw() #Hide 
    
    #Check that version of Sigil is at least 0.9.1
    print('Python Laucher Version:', bk.launcher_version())
    if bk.launcher_version() < 20151024:
        print("You need to use Sigil 0.9.1 or greater for this plugin")
        print('\n\nPlease click OK to close the Plugin Runner window.')
        return 0        #....and return if Sigil is not greater than 0.9.1
    
    prefs = bk.getPrefs()
    prefs.defaults['ConvertAngularBrackets'] = True
    
    #TODO insert data below when published
    url = "https://www.mobileread.com/forums/showthread.php?t=285771"
    #Prefs use a dictionary, so its is effectively passed by reference
    CheckForUpdates(root, prefs, url)
    textImporterDialog = cImportTextFile(root, bk, prefs) #Initialise instance of class
    root.wait_window(cImportTextFile.dlgTop) #Wait until cImportTextFile closes 
    root.destroy()

    bk.savePrefs(prefs)

    print('\n\nPlease click OK to close the Plugin Runner window.')
    
    root.mainloop()
    
    return 0        #0 - means success, anything else means failure

def main():
    print ("This module should not be run as a stand-alone module")
    return -1
    
if __name__ == "__main__":
    sys.exit(main())

