#!/usr/bin/env python
# -*- coding: utf-8 -*-



__license__   = 'GPL v3'
__copyright__ = '2019, Jürgen Habelt <juergen@habelt-jena.de>'
__docformat__ = 'restructuredtext en'

globals()['load_translations']()


from threading import Thread
from PyQt5.Qt import Qt, QAbstractTableModel, QIcon, QMovie, QItemDelegate, QRadioButton, \
     QLabel, pyqtSignal, QBuffer, QByteArray

try:
    from calibre_plugins.toc_view_generator.logging_factory import LoggingFactory
    from calibre_plugins.toc_view_generator.json_wrapper import JsonConfigWrapper 
    from calibre_plugins.toc_view_generator.invoker import Invoker

except ImportError:
    from logging_factory import LoggingFactory
    from json_wrapper import JsonConfigWrapper 
    from invoker import Invoker
    

class RequirementsModel(QAbstractTableModel):
    '''
    The model for the Requirements View
    '''
    
    def __init__(self, parent = None):
        '''
        Initialization
        '''
        super(RequirementsModel, self).__init__(parent)
        
        self.sample_data = [
            [_('user field configured'), 2]
        ]

        self.icon = globals()['get_icons']('images/icon.png')

        self.icons = []
        icon_paths = ['images/failed.png', 'images/ok.png', 'images/icon.png']
        for path in icon_paths: 
            self.icons.append(globals()['get_icons'](path)) 

        self.animation = Movie('images/question-mark-1.gif')
        self.animation.frameChanged2['QIcon'].connect(self.frameChanged)
        self.animation.start()
    
    
    def updateConfigurationState(self):
        '''
        This is invoked from a client to update the UI with the state
        '''
        self.update_thread = Thread(target = self._updateConfigurationState)
        self.update_thread.start()
    
    
    def _updateConfigurationState(self):
        '''
        Thread function to update the state
        '''
        invoker = Invoker(JsonConfigWrapper.getInstance())
        for idx, func in enumerate([
                invoker.customContentFieldExists
            ]):
            self[idx] = 1 if func() else 0
    
    
    def __setitem__(self, idx, value):
        '''
        Used to set the item state (0 - failed, 1 - ok)
        (initialized 2 - pending)
        '''
        self.sample_data[idx][1] = value
        self.updateState()
        
        
    def rowCount(self, index):
        '''
        The count of rows
        '''
        return len(self.sample_data)
        
        
    def columnCount(self, index):
        '''
        The count of columns
        '''
        return 2
    
    
    def data(self, index, role):
        '''
        Provides data for the View
        '''
        try:
            idx = self.sample_data[index.row()][1]
            txt = self.sample_data[index.row()][0]
            icon = self.icons[idx]
            
            if role == Qt.DisplayRole:
                if index.column() == 0:
                    return txt
             
            elif role == Qt.DecorationRole:
                if index.column() == 1:
                    return icon 

            elif role == Qt.SizeHintRole:
                if index.column() == 1:
                    if icon:
                        return icon.size()
                
            else:
                pass
            
        except:
            pass
        
        return None


    def headerData(self, section, orientation, role):
        '''
        Defines the headers
        '''
        if role == Qt.DisplayRole:
            if orientation == Qt.Horizontal:
                if section == 0:
                    return _('Requirement')
                if section == 1:
                    return _('Status')
                
            if orientation == Qt.Vertical:
                return str(section + 1)
        
        return None


    def flags(self, index):
        '''
        Defines flags whether an item is editable ...
        '''
        if index.row() == 4 and index.column() == 0:
            return Qt.ItemIsEnabled | Qt.ItemIsSelectable | Qt.ItemIsEditable
        
        return Qt.ItemIsEnabled
    
    
    def frameChanged(self, icon):
        '''
        Slot delivers new Icon from movie
        '''
        self.icons[2] = icon
        self.updateState()
        
        
    def updateState(self):
        '''
        Invoked to update the visual appearance
        '''
        topLeft = self.index(0, 1)
        bottomRight = self.index(len(self.sample_data) - 1, 1)
        self.dataChanged.emit(topLeft, bottomRight)
    

class RequirementsDelegate(QItemDelegate):
    '''
    Delegate used to provide custom behavior
    This class is left here for demonstration purposes!
    '''
    def __init__(self, parent = None):
        '''
        Initialization
        '''
        super(RequirementsDelegate, self).__init__(parent)
        
        
    def paint(self, painter, option, index):
        '''
        Custom paint
        Demonstrates how to render a Radio Button into a cell
        '''
        if index.row() == 4 and index.column() == 0:
            painter.save()
            xoff = option.rect.x() + 5
            yoff = option.rect.y() + 5
            painter.translate(xoff, yoff)
            radio = QRadioButton('Radio Button')
            radio.render(painter)
            painter.restore()
            return
                        
        QItemDelegate.paint(self, painter, option, index)


    def createEditor(self, parent, option, index):
        '''
        Custom createEditor
        '''
        if index.row() == 4 and index.column() == 0:
            label = QLabel('TEST', parent)
            return label
        
        return QItemDelegate.createEditor(self, parent, option, index)        
        
        
class Movie(QMovie):    
    '''
    Helps in using and playing an animated gif
    '''
    frameChanged2 = pyqtSignal(['QPixmap'])
    frameChanged2 = pyqtSignal(['QImage'])
    frameChanged2 = pyqtSignal(['QIcon'])
    
    def __init__(self, resource_path):
        '''
        Initialization, defines a byte stream to play
        '''
        super(Movie, self).__init__()
        self.setCacheMode(QMovie.CacheAll)

        self.bytes = globals()['get_resources'](resource_path)
        self.byteArray = QByteArray.fromRawData(self.bytes)
        self.buffer = QBuffer()
        self.buffer.setData(self.byteArray)
        self.buffer.seek(0)
        
        self.setDevice(self.buffer)
        self.logData()
        super(Movie, self).frameChanged.connect(self.handleFrame)
        super(Movie, self).error.connect(self.handleError)
        
        
    def handleFrame(self, no):
        '''
        Called every time a new frame is available 
        '''
        self.frameChanged2['QIcon'].emit(QIcon(self.currentPixmap()))
        
        
    def handleError(self, error):
        '''
        Handles error signals
        '''
        LoggingFactory().getLogger(self).error('Movie error: %s' % str(error))
    
       
    def logData(self):
        '''
        Logs some data of the Movie
        ''' 
        LoggingFactory().getLogger(self).debug('Bytes length: %d' % len(self.bytes))
        LoggingFactory().getLogger(self).debug('Bytes Array: %d' % self.byteArray.length())
        LoggingFactory().getLogger(self).debug('Buffer: %d' % self.buffer.size())
        