#!/usr/bin/env python
# -*- coding: utf-8 -*-




__license__   = 'GPL v3'
__copyright__ = '2019, Jürgen Habelt <juergen@habelt-jena.de>'
__docformat__ = 'restructuredtext en'


import re
import logging, logging.config

try:
    from calibre_plugins.toc_view_generator.json_wrapper import JsonConfigWrapper
    
except ImportError:
    from json_wrapper import JsonConfigWrapper


class LoggingInstance(object):
    '''
    We divide Logging Factory and Logging instance
    '''
    
    @classmethod
    def type(cls, obj):
        '''
        Returns the somehow modified class name of an object
        '''
        tp = str(type(obj))
        dottedName = re.match(r".*?'(.*?)'>", tp).group(1) 
        return dottedName.split('.')[-1]


    def __init__(self):
        '''
        Constructor
        '''
        if 'loggers' in self.__dict__:
            return
        
        self.loggers = {}
        
        configWrapper = JsonConfigWrapper.getInstance() 
        log_path = configWrapper['log_path'] 
        log_level = configWrapper['log_level']

        self._log_level = logging.getLevelName(log_level)         
        self._format = '%(asctime)s - %(name)s - %(levelname)s : %(message)s.'
        self._formatter = CustomFormatter(self._format)
        self._handler = logging.handlers.TimedRotatingFileHandler(filename = log_path, encoding = 'utf-8')
        self._handler.setFormatter(self._formatter)
        
        configWrapper.itemReceived.connect(self.jsonNotification)
        configWrapper.itemSent.connect(self.jsonNotification)


    def getLogger(self, name):
        '''
        Instantiates a logger with given name and additional console handler
        @param name: Can be name (string) or object
        '''
        if type(name) != type(' '):
            name = LoggingInstance.type(name)
        
        if name not in self.loggers:
            logger = logging.getLogger(name)
            self.configure(logger)
            self.loggers[name] = logger
        
        return self.loggers[name]


    def updateLevel(self):
        '''
        Updates the level of existing and upcoming loggers after configuration
        This provides for immediate update
        '''
        configWrapper = JsonConfigWrapper.getInstance() 
        log_level = configWrapper['log_level']
        self._log_level = logging.getLevelName(log_level)
        for _, logger in list(self.loggers.items()):
            logger.setLevel(self._log_level)

    
    def shutdown(self):
        '''
        Shutdown the logging system
        '''
        pass
    
        
    def configure(self, logger):
        '''
        Manually configures the given logger
        '''
        logger.setLevel(self._log_level)
        logger.addHandler(self._handler)


    def jsonNotification(self, msg):
        '''
        Handles Json notifications to decouple this service
        '''
        self.getLogger('JsonConfigWrapper').debug(msg)
        

class CustomFormatter(logging.Formatter):
    '''
    This Custom Formatter depends on the Logger name
    '''
    def __init__(self, fmt):
        '''
        Initialization
        '''
        super(CustomFormatter, self).__init__(fmt)
        
        self.raw_formatter = logging.Formatter('%(message)s')
        
    
    def format(self, record):
        '''
        Performs formatting depending on Logger name
        '''
        if record.name == 'raw':
            return self.raw_formatter.format(record)
        
        return super(CustomFormatter, self).format(record)
