#!/usr/bin/env python
# -*- coding: utf-8 -*-




__license__   = 'GPL v3'
__copyright__ = '2019, Jürgen Habelt <juergen@habelt-jena.de>'
__docformat__ = 'restructuredtext en'

globals()['load_translations']()


import os, platform
from os.path import expanduser
from PyQt5.Qt import pyqtSignal, QObject
from calibre.utils.config import JSONConfig


class JsonConfigWrapper(QObject):
    '''
    A Wrapper of the JSONConfig class, also performs defaults initialization
    It can:
    - Iterate through the settings
    - Create a duplicate
    '''
    itemReceived = pyqtSignal([str])
    itemSent = pyqtSignal([str])

    _instance = None
    
    @classmethod
    def getInstance(cls):
        '''
        This returns the main instance
        Duplicates can be returned by using dup
        '''
        if not cls._instance:
            cls._instance = cls('plugins/toc_view_generator.json')
        return cls._instance
    
    
    def __init__(self, path = None):
        '''
        Initialization
        '''
        super(JsonConfigWrapper, self).__init__()
        
        self.keys = [
            'max_level', 
            'use_strip', 'strip', 
            'omit', 
            'skip', 
            'continuous',
            'header_markup',
            'target_field',
            'user_field',
            'insertion_mode',
            'log_path', 
        ]
        self.extra_keys = ['gui']
        self.extra = {}

        if not path:
            self.prefs = {}                     # for duplicates we use a dictionary
            return
        
        self.path = path
        self.prefs = JSONConfig(path)

        self.prefs.defaults['max_level'] = 2
        self.prefs.defaults['use_strip'] = False
        self.prefs.defaults['strip'] = '^ *\d*\.?\d* *'
        self.prefs.defaults['omit'] = False
        self.prefs.defaults['skip'] = False
        self.prefs.defaults['continuous'] = False
        self.prefs.defaults['header_markup'] = 'h3' 
        self.prefs.defaults['target_field'] = '#content' 
        self.prefs.defaults['user_field'] = '#content'
        self.prefs.defaults['insertion_mode'] = 0
        self.prefs.defaults['log_path'] = self.get_log_path('log', 'calibre.log')
        self.prefs.defaults['force_version_change'] = True
        self.prefs.defaults['version'] = (1, 0, 0)
        self.prefs.defaults['log_level'] = 'INFO'

    
    def __getitem__(self, key):
        '''
        Access to the embedded configuration
        '''
        if key in self.extra_keys:
            if key in list(self.extra.keys()):
                value = self.extra[key]
            else:
                value = None
            self.itemSent.emit("Extra requested: %s => %s" % (key, str(value)))
            return value
        
        else:
            return self.prefs[key]
    
    
    def __setitem__(self, key, value):
        '''
        Access to the embedded configuration
        '''
        if key in self.extra_keys:
            self.extra[key] = value
            self.itemReceived.emit("Extra written: %s => %s" % (key, str(value)))
        else:
            self.prefs[key] = value        
        
        
    def __iter__(self):
        '''
        Iterates through the settings
        '''
        for key in self.keys:
            yield key, self[key]

    
    def __len__(self):
        '''
        After a hint in web
        '''
        return len(self.keys)
        
        
    def dup(self):
        '''
        Provides a duplicate
        '''
        duplicate = JsonConfigWrapper()
        for key, value in self:
            duplicate[key] = value 
         
        return duplicate


    def version_changed(self, version, update = False):
        '''
        Performs a one time check, whether the version has changed
        After this it updates the version
        '''
        changed = list(version) != self['version'] or self['force_version_change']
        
        if update:
            self['version'] = version
        
        return changed
    
    
    def get_log_path(self, relative = None, file = None):
        '''
        Determines an os dependent special directory
        - windows: documents
        - linux: home/documents
        and joins it with relative folder, creating the folder, if it does not exist
        file name is also appended, if present
        '''
        if platform.system() == 'Windows':
            import ctypes.wintypes
            CSIDL_PERSONAL = 5                          # My Documents
            SHGFP_TYPE_CURRENT = 0                      # Get current, not default value
            
            buf= ctypes.create_unicode_buffer(ctypes.wintypes.MAX_PATH)
            ctypes.windll.shell32.SHGetFolderPathW(None, CSIDL_PERSONAL, None, SHGFP_TYPE_CURRENT, buf)
            
            path = buf.value
        else:
            path = expanduser(_('~/Documents'))
            
        if relative:
            path = os.path.join(path, relative)
            if not os.path.exists(path):
                os.makedirs(path)
            
        if file:
            path = os.path.join(path, file)
            
        return path
