#!/usr/bin/env python
# -*- coding: utf-8 -*-

import subprocess
from subprocess import CalledProcessError


__license__   = 'GPL v3'
__copyright__ = '2019, Jürgen Habelt <juergen@habelt-jena.de>'
__docformat__ = 'restructuredtext en'


import platform

try:
    from calibre_plugins.toc_view_generator.logging_factory import LoggingFactory
    from calibre_plugins.toc_view_generator.ruby_port.html_generator import HtmlGenerator

except ImportError:
    from logging_factory import LoggingFactory
    from ruby_port.html_generator import HtmlGenerator
    

class Invoker(object):
    '''
    Responsible for invoking the Ruby script
    '''
    
    def __init__(self, settings):
        '''
        Initialization
        @param settings: this is the duplicate of the settings changed by configuration dialog
        '''
        self.settings = settings
        self.gui = self.settings['gui']


    def invoke3(self, book, result):
        '''
        Invokes the ported Ruby code to extract book marks
        @param book: the path of the e-book
        @param result: the path of the result file with book marks
        @return: 0 for success, -1 otherwise
        '''
        generator = HtmlGenerator(self)
        return generator.generate(book, result)


    def customContentFieldExists(self):
        '''
        Checks whether the custom #content field exists
        '''
        if not self.gui:
            return False
        
        db = self.gui.current_db                                        # query the #content custom field
        return self.settings['user_field'] in db.custom_field_keys()
        

    def commonInvoke(self, args, debug_output = True):
        '''
        Used to invoke a system command and get the exit code and the output back
        '''
        output = ''                                                     # output defaults to empty string
        ecode = 0
        try:
            output = subprocess.check_output(args, universal_newlines = True)
            
        except CalledProcessError as ex:
            LoggingFactory().getLogger(self).exception('Exception in Common Invoke: ')
            ecode = ex.returncode
        
        except Exception as ex:
            LoggingFactory().getLogger(self).exception('Exception in Common Invoke: ')
            ecode = -1
        
        if debug_output:
            LoggingFactory().getLogger(self).debug('Common Invoke Result: (%d, %s)' % (ecode, output))
        
        return (ecode, output)
    
    
    def systemShellCommandExtension(self):
        '''
        Gem maybe other than .cmd in Linux
        '''
        if platform.system() == 'Windows':
            return '.cmd'
        else:
            return ''


    def systemNullDevice(self):
        '''
        To suppress command output independently of system
        '''
        if platform.system() == 'Windows':
            return 'nul'
        else:
            return '/dev/null'
        