#!/usr/bin/env python
# -*- coding: utf-8 -*-



__license__   = 'GPL v3'
__copyright__ = '2019, Jürgen Habelt <juergen@habelt-jena.de>'
__docformat__ = 'restructuredtext en'

globals()['load_translations']()

from PyQt5.Qt import QDialog, QPushButton

try:
    from calibre_plugins.toc_view_generator.json_wrapper import JsonConfigWrapper 
    from calibre_plugins.toc_view_generator.gui.TempConfig_ui import Ui_TempConfig
    from calibre_plugins.toc_view_generator.invoker import Invoker

except ImportError:
    from json_wrapper import JsonConfigWrapper 
    from gui.TempConfig_ui import Ui_TempConfig
    from invoker import Invoker

# This is where all preferences for this plug-in will be stored
prefs = JsonConfigWrapper.getInstance()


class TempConfigMixin(object):
    '''
    Mixin of the code for the Temp Config Dialog
    '''
    
    def __init__(self):
        '''
        Initialization
        '''
        # setup the User Interface
        self.setupUi(self)
        self.target_field_radio_buttons = [ self.commentsRadioButton, self.userRadioButton ]
        self.insertion_options_radio_buttons = [ self.replaceRadioButton, self.appendRadioButton, self.prependRadioButton ]
        
        # we extract all translatable text from the generated UI code
        self.set_text()
        
        # load the settings from configuration file
        self.load_settings()
    
    
    def set_text(self):
        '''
        Contains all translatable text
        '''
        self.contentGenerationGroupBox.setTitle(_("View Generation"))
        self.levelLabel.setText(_("Max Level:"))
        self.levelSpinBox.setToolTip(_("The maximum level of Headings to be displayed in #content"))
        self.useStripCheckBox.setText(_("Strip:"))
        self.useStripCheckBox.setToolTip(_("A flag indicating whether to strip some characters at the beginning of Headings"))
        self.stripLineEdit.setToolTip(_("A regular expression defining the characters to be stripped"))
        self.omitCheckBox.setText(_("Omit:"))
        self.omitCheckBox.setToolTip(_("A flag indicating whether to omit the list item numbers of second level Headings"))
        self.continuousCheckBox.setText(_("Continuous:"))
        self.continuousCheckBox.setToolTip(_("A flag indicating whether to count second level Headings continuously"))

        self.headerMarkupLabel.setText(_("Header Markup:"))
        self.headerMarkupComboBox.setToolTip(_("Use this combo to define the first level\'s HTML markup"))
        self.targetFieldGroupBox.setTitle(_("Target Field Selection"))
        self.commentsRadioButton.setText(_("Comments Field"))
        self.commentsRadioButton.setToolTip(_("Choose to select Comments field as target"))
        self.userRadioButton.setText(_("User Field"))
        self.userRadioButton.setToolTip(_("Choose to select User Defined field as target"))
        self.userFieldLineEdit.setToolTip(_("Use this box to enter the name of a user defined field"))
        self.insertionOptionsGroupBox.setTitle(_("Insertion Options"))
        self.replaceRadioButton.setText(_("Replace"))
        self.replaceRadioButton.setToolTip(_("Used to select the replacement of the field content"))
        self.appendRadioButton.setText(_("Append"))
        self.appendRadioButton.setToolTip(_("Used to select an insertion at the end of the field\'s existing content"))
        self.prependRadioButton.setText(_("Prepend"))
        self.prependRadioButton.setToolTip(_("Used to select an insertion at the begin of the field\'s existing content"))
        
                
    def load_settings(self):
        '''
        Initializes controls with settings
        '''
        self.levelSpinBox.setValue(self.settings['max_level'])
        self.useStripCheckBox.setChecked(self.settings['use_strip'])
        self.stripLineEdit.setText(self.settings['strip'])
        self.omitCheckBox.setChecked(self.settings['omit'])
        self.continuousCheckBox.setChecked(self.settings['continuous'])
        
        self.headerMarkupComboBox.setCurrentText(self.settings['header_markup'])
        
        user_field = self.settings['user_field']
        target_field = self.settings['target_field']
        self.userFieldLineEdit.setText(user_field)
        self.commentsRadioButton.setChecked(target_field == 'comments')
        self.userRadioButton.setChecked(target_field == user_field)
        
        self.insertion_options_radio_buttons[self.settings['insertion_mode']].setChecked(True)
        
        
    def save_settings(self):
        '''
        Saves the settings stored in the controls
        '''
        self.settings['max_level'] = self.levelSpinBox.value()
        self.settings['use_strip'] = self.useStripCheckBox.isChecked()
        self.settings['strip'] = self.stripLineEdit.text()
        self.settings['omit'] = self.omitCheckBox.isChecked()
        self.settings['continuous'] = self.continuousCheckBox.isChecked()
        
        self.settings['header_markup'] = self.headerMarkupComboBox.currentText()
        
        user_field = self.userFieldLineEdit.text()
        target_field = 'comments' if self.commentsRadioButton.isChecked() else user_field
        self.settings['user_field'] = user_field
        self.settings['target_field'] = target_field
        
        for idx, btn in enumerate(self.insertion_options_radio_buttons):
            if btn.isChecked():
                self.settings['insertion_mode'] = idx        


class TempConfigDialog(QDialog, Ui_TempConfig, TempConfigMixin):
    '''
    For temporary configuration during action
    '''
    
    def __init__(self, gui):
        '''
        Initialization, constructs the widget.
        '''
        QDialog.__init__(self, gui)
        self.settings = prefs.dup()

        TempConfigMixin.__init__(self)        
        
        self.setWindowTitle(_('Per Call Settings'))
        self.userFieldLineEdit.setReadOnly(True)
        self.check_user_field()

        self.proceedPushButton = QPushButton(_('Proceed'), self)
        self.proceedPushButton.clicked.connect(self.proceed)
        self.proceedPushButton.setToolTip(_('Proceed with the Toc View Generator action'))
        mainLayout = self.gridLayout_3
        mainLayout.addWidget(self.proceedPushButton, mainLayout.rowCount(), 0)
        
        
    def proceed(self):
        '''
        Continuous the action        
        '''
        self.save_settings()
        self.accept()
       
        
    def check_user_field(self):
        '''
        Checks if the user field is configured properly. If not it resets the selected target 
        field to 'comments'
        '''
        invoker = Invoker(prefs)
        if not invoker.customContentFieldExists():
            self.commentsRadioButton.setChecked(True)
            self.userRadioButton.setEnabled(False)
