#!/usr/bin/env python
# -*- coding: utf-8 -*-



__license__   = 'GPL v3'
__copyright__ = '2019, Jürgen Habelt <juergen@habelt-jena.de>'
__docformat__ = 'restructuredtext en'

globals()['load_translations']()

from PyQt5.Qt import QFileDialog, QWidget, QMessageBox

try:
    from calibre_plugins.toc_view_generator.logging_factory import LoggingFactory
    from calibre_plugins.toc_view_generator.json_wrapper import JsonConfigWrapper 
    from calibre_plugins.toc_view_generator.gui.Config_ui import Ui_Config
    from calibre_plugins.toc_view_generator.gui.TempConfig import TempConfigMixin
    from calibre_plugins.toc_view_generator.gui.RequirementsView import RequirementsView

except ImportError:
    from logging_factory import LoggingFactory
    from json_wrapper import JsonConfigWrapper 
    from gui.Config_ui import Ui_Config
    from gui.TempConfig import TempConfigMixin
    from gui.RequirementsView import RequirementsView

# This is where all preferences for this plug-in will be stored
prefs = JsonConfigWrapper.getInstance()


class ConfigWidget(QWidget, Ui_Config, TempConfigMixin):
    '''
    The configuration widget for the plug-in
    '''
    
    def __init__(self):
        '''
        Initialization
        '''
        QWidget.__init__(self)
        self.settings = prefs

        TempConfigMixin.__init__(self)        
        
        # connects button signals to slots (event handling routines)
        self.connect()
        
        
    def select_log(self):
        '''
        Displays a file dialog to select the Plug-in log file
        '''
        res = QFileDialog.getSaveFileName(self, _('Select Plug-in Log File'), prefs.get_log_path())[0]
        if res:
            self.logFileLineEdit.setText(res)


    def about(self):
        '''
        Get the about text from a file inside the plug-in zip file
        '''
        # text = globals()['get_resources']('about.txt')
        text = _("The Toc View Generator Plug-in\n"
                 "======================\n"
                 "\n"
                 "Created by Jürgen Habelt\n"
                 "\n"
                 "Requires Calibre >= 4.0.0\n"
                 "\n"
                 "This plugin generates view content using the 'Table of Content' of pdf or epub "
                 "files and writing either to an user defined field or the Comments field.")
        
        QMessageBox.about(self, _('About the Toc View Generator Plug-in'), text.decode('utf-8'))


    def requirements(self):
        '''
        Shows the Requirements status of the plug-in
        '''
        try:
            self.requirements = RequirementsView() 
            self.requirements.show()            
            self.requirements.model.updateConfigurationState()
            
        except Exception:
            LoggingFactory().getLogger(self).exception("Exception in Requirements Dialog")
        
        
    def set_text(self):
        '''
        Contains all translatable text
        '''
        self.setWindowTitle(_("Form"))

        TempConfigMixin.set_text(self)

        self.functionalTab.setAccessibleName(_("Functional"))
        self.tabWidget.setTabText(self.tabWidget.indexOf(self.functionalTab), _("Functional"))
        self.tabWidget.setTabToolTip(self.tabWidget.indexOf(self.functionalTab), _("Functional settings and About information"))
        self.basisTab.setAccessibleName(_("Basis"))
        self.tabWidget.setTabText(self.tabWidget.indexOf(self.basisTab), _("Basis"))
        self.tabWidget.setTabToolTip(self.tabWidget.indexOf(self.basisTab), _("Basis settings"))

        self.logFileLabel.setText(_("Log File:"))
        self.logFilePushButton.setText(_("Select ..."))
        self.logFilePushButton.setToolTip(_("Show a file dialog to select the plug-in log file"))
        self.logFileLineEdit.setToolTip(_("Defines the path of the plug-in log file"))
        self.logLevelLabel.setText(_("Log Level:"))
        self.logLevelComboBox.setToolTip(_("Defines the log level of the plug-in log file"))

        self.checkPushButton.setText(_("Check ..."))
        self.checkPushButton.setToolTip(_("Displays a Dialog with requirements status"))
        self.aboutPushButton.setText(_("About ..."))
        self.aboutPushButton.setToolTip(_("Displays an About box for the plug-in"))
        self.linkLabel.setToolTip(_("Invokes a page on the Internet with application hints"))
        self.linkLabel.setText(_("<a href=\"https://juergen.habelt-jena.de/en/programming-projects/define-content/application?start=1\">Application Hints</a>"))
        
        
    def load_settings(self):
        '''
        Initializes controls with settings
        '''
        TempConfigMixin.load_settings(self)

        self.logFileLineEdit.setText(self.settings['log_path'])
        self.logLevelComboBox.setCurrentText(self.settings['log_level'])
        
        
    def save_settings(self):
        '''
        Saves the settings stored in the controls
        '''
        TempConfigMixin.save_settings(self)

        self.settings['log_path'] = self.logFileLineEdit.text()
        self.settings['log_level'] = self.logLevelComboBox.currentText()
        
        
    def connect(self):
        '''
        Connects the event handling slots to buttons
        '''
        self.logFilePushButton.clicked.connect(self.select_log)
        self.checkPushButton.clicked.connect(self.requirements)
        self.aboutPushButton.clicked.connect(self.about)
