#!/usr/bin/env python
# -*- coding: utf-8 -*-

from PyQt5.QtWidgets import QMessageBox

__license__   = 'GPL v3'
__copyright__ = '2019, Jürgen Habelt <juergen@habelt-jena.de>'
__docformat__ = 'restructuredtext en'

globals()['load_translations']()


import os

from PyQt5.Qt import pyqtSignal 

from calibre.utils.config import config_dir
from calibre.gui2 import error_dialog

# The class that all interface action plug-ins must inherit from
from calibre.gui2.actions import InterfaceAction


try:
    from calibre_plugins.toc_view_generator.gui.TempConfig import TempConfigDialog
    from calibre_plugins.toc_view_generator.invoker import Invoker
    from calibre_plugins.toc_view_generator.logging_factory import LoggingFactory
    from calibre_plugins.toc_view_generator.json_wrapper import JsonConfigWrapper 
    from calibre_plugins.toc_view_generator.plugin_worker import PluginWorker 

except ImportError:
    from gui.TempConfig import TempConfigDialog
    from invoker import Invoker
    from logging_factory import LoggingFactory
    from json_wrapper import JsonConfigWrapper 
    from plugin_worker import PluginWorker 


class TocViewGeneratorAction(InterfaceAction):
    '''
    This class performs the work of the plug-in
    '''

    name = 'Toc View Generator'

    # Declare the main action associated with this plug-in
    # The keyboard shortcut can be None if you don't want to use a keyboard shortcut.
    action_spec = (_('Toc View Generator'), None, _('Run the Toc View Generator Action'), None)
    
    action_succeeded = pyqtSignal()
    action_failed = pyqtSignal()

    def genesis(self):
        '''
        Initial setup
        The qaction is automatically created from the action_spec defined above
        '''
        self.result_path = os.path.join(config_dir, 'plugins', 'TocViewGenerator', 'result', 'result.html')
        icon = globals()['get_icons']('images/icon.png')
        self.qaction.setIcon(icon)
        self.qaction.triggered.connect(self.do_action)
        
        self.action_succeeded.connect(self.on_success)
        self.action_failed.connect(self.on_error)
        
        JsonConfigWrapper.getInstance()['gui'] = self.gui
        
        LoggingFactory().getLogger(self).debug('Genesis invoked')        # when invoked?
    
    
    def do_action(self):
        '''
        Performs the plug-in action
        '''
        try:
            self.settings = self.show_dialog()
            if not self.settings:
                return
            
            book_path = self.get_book_path()
            if book_path:
                plugin_worker = PluginWorker(self.gui)
                invoker = Invoker(self.settings)
                
                plugin_worker.start_work(
                    lambda: invoker.invoke3(book_path, self.result_path), 
                    lambda succeeded: self.action_succeeded.emit() if succeeded else self.action_failed.emit() 
                    )

        except Exception:
            LoggingFactory().getLogger(self).exception('Exception during command execution.')
    
    
    def on_success(self):
        '''
        On success the book mark information is stored
        '''
        try:
            bookmarks = open(self.result_path).read()#.decode('utf-8')
            bookmarks = bookmarks.replace('&amp;#', '&#')
            self.set_content(bookmarks)
            LoggingFactory().getLogger(self).info('Target field successfully written')
            QMessageBox.information(self.gui, _('Toc View Generator'), _('Target field successfully written'))
            
        except:
            LoggingFactory().getLogger(self).exception("Cannot write 'Table of Content' information")
            error_dialog(
                self.gui, 
                _('Toc write failed'), 
                _("Cannot write 'Table of Content' information"), 
                show = True)


    def on_error(self):
        '''
        On error an Error Dialog is displayed
        '''
        LoggingFactory().getLogger(self).warning('Cannot extract toc information')
        error_dialog(
            self.gui, 
            _('Toc extraction failed'), 
            _("Cannot extract 'Table of Content' information"), 
            show = True)
    
    
    def apply_settings(self):
        '''
        Prepare the plug-in with new settings from the configuration dialog
        '''
        LoggingFactory().updateLevel()
        QMessageBox.information(self.gui, _('Toc View Generator'), _('Preferences applied'))


    def get_book_path(self):
        '''
        Get the book path of the first selected e-book
        '''
        rows = self.gui.library_view.selectionModel().selectedRows()
        if not rows or len(rows) == 0:
            error_dialog(
                self.gui, 
                _('No books selected'), 
                _('Cannot add to target field as no books are selected'), 
                show = True)
            LoggingFactory().getLogger(self).warning('Cannot add to target field as no books are selected')
            return None

        book_id = self.gui.library_view.model().id(rows[0])          # get the book id
        db = self.gui.current_db.new_api
        formats = db.formats(book_id)
        if 'EPUB' in formats:
            path = db.format_abspath(book_id, 'EPUB')
        elif 'PDF' in formats:
            path = db.format_abspath(book_id, 'PDF')
        else:  
            error_dialog(
                self.gui, 
                _('No required format'), 
                _('Cannot add to target field as no pdf or epub formats are present: %s' % str(formats)), 
                show = True)
            LoggingFactory().getLogger(self).warning('Cannot add to target field as no pdf or epub formats are present: %s' % str(formats))
            return None
        
        return path


    def set_content(self, data):
        '''
        Sets the #content metadata of the first selected book
        '''
        sm = self.gui.library_view.selectionModel()                     # determine selected book
        row = sm.selectedRows()[0]
        book_id = self.gui.library_view.model().id(row)
        
        db = self.gui.current_db.new_api                                # update its #content custom field
        md = db.get_metadata(book_id)

        old_data = md.__getattribute__(self.settings['target_field'])
        mode = self.settings['insertion_mode']
        if mode == 0 or not old_data or old_data.isspace():             # 'replace'
            pass
        elif mode == 1:                                                 # 'append'
            data = '<div>' + old_data + data + '</div>'
        else:                                                           # 'prepend'
            data = '<div>' + data + old_data + '</div>'
        md.__setattr__(self.settings['target_field'], data)
            
        db.set_metadata(book_id, md)

        model = self.gui.library_view.model()                   # refresh changed book including the BookInfo etc.
        model.refresh_ids([book_id], row.row())

    
    def show_dialog(self):
        '''
        This dialog is for temporary configuration
        '''
        dlg = TempConfigDialog(self.gui)
        if dlg.exec_():
            return dlg.settings
        
        return None
