#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2011, Grant Drake <grant.drake@gmail.com>'
__docformat__ = 'restructuredtext en'

import traceback
#SS-add START
import struct, sys
#SS-add END

from calibre.gui2 import error_dialog

from calibre_plugins.quality_check.check_base import BaseCheck
from calibre_plugins.quality_check.common_utils import get_title_authors_text
from calibre_plugins.quality_check.mobi6 import MinimalMobiReader

#SS-add START
# Python code taken from kinde_unpack plug-in

PY2 = sys.version_info[0] == 2
PY3 = sys.version_info[0] == 3

if PY3:
    # Also Note: if decode a bytestring using 'latin-1' (or any other full range 0-255 encoding)
    # in place of ascii you will get a byte value to half-word or integer value
    # one-to-one mapping (in the 0 - 255 range)

    def bchr(s):
        return bytes([s])

    def bstr(s):
        if isinstance(s, str):
            return bytes(s, 'latin-1')
        else:
            return bytes(s)

    def bord(s):
        return s

    def bchar(s):
        return bytes([s])

else:
    def bchr(s):
        return chr(s)

    def bstr(s):
        return str(s)

    def bord(s):
        return ord(s)

    def bchar(s):
        return s
      
class SectionizerLight:
    """ Stolen from Mobi_Unpack and slightly modified. """
    def __init__(self, filename):
        self.data = open(filename, 'rb').read()
        if self.data[:3] == b'TPZ':
            self.ident = 'TPZ'
        else:
            self.palmheader = self.data[:78]
            self.ident = self.palmheader[0x3C:0x3C+8]
        try:
            self.num_sections, = struct.unpack_from(b'>H', self.palmheader, 76)
        except:
            return
        self.filelength = len(self.data)
        try:
            sectionsdata = struct.unpack_from(bstr('>%dL' % (self.num_sections*2)), self.data, 78) + (self.filelength, 0)
            self.sectionoffsets = sectionsdata[::2]
        except:
            print(traceback.error)
            pass

    def loadSection(self, section):
        before, after = self.sectionoffsets[section:section+2]
        return self.data[before:after]
 

class MobiHeaderLight:
    """ Stolen from Mobi_Unpack and slightly modified. """
    def __init__(self, sect, sectNumber):
        self.sect = sect
        self.start = sectNumber
        self.header = self.sect.loadSection(self.start)
        self.records, = struct.unpack_from(b'>H', self.header, 0x8)
        self.length, self.type, self.codepage, self.unique_id, self.version = struct.unpack(b'>LLLLL', self.header[20:40])
        self.mlstart = self.sect.loadSection(self.start+1)[0:4]
        self.crypto_type, = struct.unpack_from(b'>H', self.header, 0xC)

    def isEncrypted(self):
        return self.crypto_type != 0

    def isPrintReplica(self):
        return self.mlstart[0:4] == b'%MOP'

    # Standalone KF8 file
    def isKF8(self):
        return self.start != 0 or self.version == 8

    def isJointFile(self):
        # Check for joint MOBI/KF8
        for i in range(len(self.sect.sectionoffsets)-1):
            before, after = self.sect.sectionoffsets[i:i+2]
            if (after - before) == 8:
                data = self.sect.loadSection(i)
                if data == b'BOUNDARY' and self.version != 8:
                    return True
                    break
        return False
     

#SS-add END

class MobiCheck(BaseCheck):
    '''
    All checks related to working with MOBI formats.
    '''
    MOBI_FORMATS = ['MOBI', 'AZW', 'AZW3']
    #SS-add START
    MOBI_ORIGINAL_FORMAT = ['MOBI']
    #SS-add FINISH
    

    def __init__(self, gui):
        BaseCheck.__init__(self, gui, 'formats:=mobi or formats:=azw or formats:=azw3')

    def perform_check(self, menu_key):
        #SS-add START
        if menu_key == 'check_mobi7_type':
            self.check_mobi_type(menu_key)
        elif menu_key == 'check_kf8_type':
            self.check_mobi_type(menu_key)
        elif menu_key == 'check_dual_mobi_type':
            self.check_mobi_type(menu_key)
        elif menu_key == 'check_mobi_missing_ebok':
        #SS-add FINISH
            self.check_mobi_missing_ebok()
        elif menu_key == 'check_mobi_missing_asin':
            self.check_mobi_missing_asin()
        elif menu_key == 'check_mobi_share_disabled':
            self.check_mobi_share_disabled()
        elif menu_key == 'check_mobi_clipping_limit':
            self.check_mobi_clipping_limit()
        else:
            return error_dialog(self.gui, _('Quality Check failed'),
                                _('Unknown menu key for %s of \'%s\'')%('MobiCheck', menu_key),
                                show=True, show_copy_button=False)
    #SS-add START
    def check_mobi_type(self, menu_key):

        def evaluate_book(book_id, db):
            try:
                show_book = False
                
                for fmt in self.MOBI_ORIGINAL_FORMAT:
                    if not db.has_format(book_id, fmt, index_is_id=True):
                        continue
                    path_to_book = db.format_abspath(book_id, fmt, index_is_id=True)
                    if not path_to_book:
                        self.log.error('ERROR: %s format is missing: %s'%(fmt, get_title_authors_text(db, book_id)))
                        continue
                    # Check MOBI type
                    self.infile = path_to_book
                    self.sect = SectionizerLight(self.infile)
                    if (self.sect.ident != b'BOOKMOBI' and self.sect.ident != b'TEXtREAd') or self.sect.ident == 'TPZ':
                        raise Exception(_('Unrecognized Kindle/MOBI file format!'))
                    mhl = MobiHeaderLight(self.sect, 0)
                    self.version = mhl.version
                    self.isEncrypted = mhl.isEncrypted()
                    if self.sect.ident == b'TEXtREAd':
                        self.isPrintReplica = False
                        self.isComboFile = False
                        self.isKF8 = False
                    else: 
                        self.isPrintReplica = mhl.isPrintReplica()
                        self.isComboFile = mhl.isJointFile()
                        self.isKF8 = mhl.isKF8()

#                    self.ePubVersion = cfg.plugin_prefs['Epub_Version']
#                    self.useHDImages = cfg.plugin_prefs['Use_HD_Images']      
                    
                    if (menu_key == 'check_mobi7_type'):
                        if (not(self.isComboFile) and not(self.isKF8)):
                            self.log('This ebook is a MOBI7 type (OLD in Calibre) <b>%s</b>'% get_title_authors_text(db, book_id))
                            show_book = True
                    elif (menu_key == 'check_kf8_type'):
                        if (self.isKF8):
                            self.log('This ebook is a KF8 type (NEW in Calibre) <b>%s</b>'% get_title_authors_text(db, book_id))
                            show_book = True
                    elif (menu_key == 'check_dual_mobi_type'):
                        if (self.isComboFile):
                            self.log('This ebook is a dual MOBI type (BOTH in Calibre) <b>%s</b>'% get_title_authors_text(db, book_id))
                            show_book = True

                return show_book
            except:
                self.log.error('ERROR parsing book: ', path_to_book)
                self.log(traceback.format_exc())
                return False
        
        self.check_all_files(evaluate_book,
                     no_match_msg='All searched MOBI books type',
                     marked_text=menu_key,
                     status_msg_type='MOBI type')
    #SS-add FINISH

    def check_mobi_missing_ebok(self):

        def evaluate_book(book_id, db):
            try:
                show_book = False
                for fmt in self.MOBI_FORMATS:
                    if not db.has_format(book_id, fmt, index_is_id=True):
                        continue
                    path_to_book = db.format_abspath(book_id, fmt, index_is_id=True)
                    if not path_to_book:
                        self.log.error('ERROR: %s format is missing: %s'%(fmt, get_title_authors_text(db, book_id)))
                        continue
                    with MinimalMobiReader(path_to_book, self.log) as mmr:
                        if mmr.book_header:
                            exth = mmr.book_header.exth
                            if exth:
                                if exth.cdetype == 'EBOK':
                                    # This book is valid
                                    continue
                                else:
                                    self.log('Missing EBOK tag: <b>%s</b>'% get_title_authors_text(db, book_id))
                                    self.log('\tcdetype:', exth.cdetype)
                            else:
                                self.log('Missing EBOK tag: <b>%s</b>'% get_title_authors_text(db, book_id))
                                self.log.error('\tNo EXTH header found in this %s file')
                        else:
                            self.log('Missing EBOK tag: <b>%s</b>'% get_title_authors_text(db, book_id))
                            self.log.error('\tNo valid book header found in this MOBI file')
                        show_book = True

                return show_book
            except:
                self.log.error('ERROR parsing book: ', path_to_book)
                self.log(traceback.format_exc())
                return False

        self.check_all_files(evaluate_book,
                             no_match_msg='All searched MOBI books have EBOK cdetag values',
                             marked_text='mobi_missing_ebok_cdetag',
                             status_msg_type='MOBI books missing EBOK cdetag')


    def check_mobi_missing_asin(self):

        def evaluate_book(book_id, db):
            try:
                show_book = False
                for fmt in self.MOBI_FORMATS:
                    if not db.has_format(book_id, fmt, index_is_id=True):
                        continue
                    path_to_book = db.format_abspath(book_id, fmt, index_is_id=True)
                    if not path_to_book:
                        self.log.error('ERROR: %s format is missing: %s'%(fmt, get_title_authors_text(db, book_id)))
                        continue
                    with MinimalMobiReader(path_to_book, self.log) as mmr:
                        if mmr.book_header:
                            exth = mmr.book_header.exth
                            if exth:
                                if exth.asin:
                                    # This is valid
                                    continue
                                else:
                                    self.log('Missing ASIN: <b>%s</b>'% get_title_authors_text(db, book_id))
                            else:
                                self.log('Missing ASIN: <b>%s</b>'% get_title_authors_text(db, book_id))
                                self.log.error('\tNo EXTH header found in this %s file'%fmt)
                        else:
                            self.log('Missing ASIN: <b>%s</b>'% get_title_authors_text(db, book_id))
                            self.log.error('\tNo valid book header found in this %s file'%fmt)
                        show_book = True

                return show_book
            except:
                self.log.error('ERROR parsing book: ', path_to_book)
                self.log(traceback.format_exc())
                return False

        self.check_all_files(evaluate_book,
                             no_match_msg='All searched MOBI books have ASIN values',
                             marked_text='mobi_missing_asin',
                             status_msg_type='MOBI books missing ASIN')


    def check_mobi_share_disabled(self):

        def evaluate_book(book_id, db):
            try:
                show_book = False
                for fmt in ('MOBI',): #Yes, this is strange but I don't want to rewrite the loop
                    if not db.has_format(book_id, fmt, index_is_id=True):
                        continue
                    path_to_book = db.format_abspath(book_id, 'MOBI', index_is_id=True)
                    if not path_to_book:
                        self.log.error('ERROR: %s format is missing: %s'%(fmt, get_title_authors_text(db, book_id)))
                        continue
                    with MinimalMobiReader(path_to_book, self.log) as mmr:
                        if mmr.book_header:
                            exth = mmr.book_header.exth
                            if exth:
                                if len(exth.asin) and exth.asin2 == exth.asin:
                                    # This is valid for sharing so move on
                                    continue
                                if len(exth.asin) and len(exth.asin2):
                                    self.log('Different ASINs at EXTH 113/504: <b>%s</b>'% get_title_authors_text(db, book_id))
                                elif len(exth.asin) == 0:
                                    self.log('Missing ASIN at EXTH 113: <b>%s</b>'% get_title_authors_text(db, book_id))
                                else:
                                    self.log('Missing ASIN at EXTH 504: <b>%s</b>'% get_title_authors_text(db, book_id))
                            else:
                                self.log('Missing ASIN: <b>%s</b>'% get_title_authors_text(db, book_id))
                                self.log.error('\tNo EXTH header found in this %s file'%fmt)
                        else:
                            self.log('Missing ASIN: <b>%s</b>'% get_title_authors_text(db, book_id))
                            self.log.error('\tNo valid book header found in this %s file'%fmt)
                        show_book = True

                return show_book
            except:
                self.log.error('ERROR parsing book: ', path_to_book)
                self.log(traceback.format_exc())
                return False

        self.check_all_files(evaluate_book,
                             no_match_msg='All searched MOBI books have Twitter/Facebook sharing enabled',
                             marked_text='mobi_share_disabled',
                             status_msg_type='MOBI books unable to share on Twitter/Facebook')


    def check_mobi_clipping_limit(self):

        def evaluate_book(book_id, db):
            try:
                show_book = False
                for fmt in self.MOBI_FORMATS:
                    if not db.has_format(book_id, fmt, index_is_id=True):
                        continue
                    path_to_book = db.format_abspath(book_id, fmt, index_is_id=True)
                    if not path_to_book:
                        self.log.error('ERROR: %s format is missing: %s'%(fmt, get_title_authors_text(db, book_id)))
                        continue
                    with MinimalMobiReader(path_to_book, self.log) as mmr:
                        if mmr.book_header:
                            exth = mmr.book_header.exth
                            if exth:
                                if exth.clipping_limit and exth.clipping_limit < 100:
                                    self.log('Clipping limit of %d%% in: <b>%s</b>'%(exth.clipping_limit, get_title_authors_text(db, book_id)))
                                    show_book = True
                return show_book
            except:
                self.log.error('ERROR parsing book: ', path_to_book)
                self.log(traceback.format_exc())
                return False

        self.check_all_files(evaluate_book,
                             no_match_msg='All searched MOBI books have no clipping limits',
                             marked_text='mobi_clipping_limit',
                             status_msg_type='MOBI books for clipping limits')

