#!/usr/bin/env python
# -*- coding: utf-8 -*-

from __future__ import absolute_import, division, print_function, unicode_literals

__license__   = 'GPL v3'
__copyright__ = u'2019, Jürgen Habelt <juergen@habelt-jena.de>'
__docformat__ = 'restructuredtext en'

globals()['load_translations']()


import sys, os, re
from calibre.utils.config import config_dir

# The class that all Interface Action plug-in wrappers must inherit from
from calibre.customize import InterfaceActionBase

try:
    from calibre_plugins.toc_view_generator.json_wrapper import JsonConfigWrapper
    from calibre_plugins.toc_view_generator.logging_factory import LoggingFactory

except ImportError:
    from json_wrapper import JsonConfigWrapper
    from logging import LoggingFactory
        

class TocViewGeneratorPlugin(InterfaceActionBase):
    '''
    This class is a simple wrapper that provides information about the actual
    plug-in class. The actual interface plug-in class is called InterfacePlugin
    and is defined in the action.py file, as specified in the actual_plugin field
    below.

    The reason for having two classes is that it allows the command line
    calibre utilities to run without needing to load the GUI libraries.
    '''
    name                = "TOC View Generator"
    description         = _(u"TOC View Generator extracts table of content information from e-books and writes it as HTML either to an user defined field or the 'comments' field")
    supported_platforms = ['windows', 'linux', 'osx']
    author              = u"Jürgen Habelt"
    version             = (2, 1, 2)
    minimum_calibre_version = (4, 0, 0)


    #: This field defines the GUI plug-in class that contains all the code that actually does something.
    actual_plugin       = 'calibre_plugins.toc_view_generator.action:TocViewGeneratorAction'

    def initialize(self):
        '''
        Initialize the Plug-in
        '''
        LoggingFactory().getLogger(self).info('Initializing Plug-in: TOC View Generator')
        if JsonConfigWrapper.getInstance().version_changed(version = self.version):
            try:                
                LoggingFactory().getLogger(self).debug('One time initialization after installation')
                self.prepare_scripts()
                JsonConfigWrapper.getInstance().version_changed(version = self.version, update = True)
                LoggingFactory().getLogger(self).info('One time initialization succeeded')

            except Exception:
                LoggingFactory().getLogger(self).exception('Exception during one time initialization')

                

    def prepare_scripts(self):
        '''
        Prepares Ruby scripts for execution
        '''
        self.pluginsdir = os.path.join(config_dir, u"plugins")
        if not os.path.exists(self.pluginsdir):
            os.makedirs(self.pluginsdir)
        
        self.maindir = os.path.join(self.pluginsdir, u"TocViewGenerator")
        if not os.path.exists(self.maindir):
            os.mkdir(self.maindir)
        
        dirs = ['result']
        for entry in dirs:
            path = os.path.join(self.maindir, entry)
            if not os.path.exists(path):
                os.mkdir(path)


    def is_customizable(self):
        '''
        This method must return True to enable customization via
        Preferences->Plugins
        '''
        return True

    
    def can_be_disabled(self):
        '''
        Must return True to enable deactivation
        '''
        return True
    
    
    def config_widget(self):
        '''
        Implement this method and :meth:`save_settings` in your plug-in to
        use a custom configuration dialog.

        This method, if implemented, must return a QWidget. The widget can have
        an optional method validate() that takes no arguments and is called
        immediately after the user clicks OK. Changes are applied if and only
        if the method returns True.

        If for some reason you cannot perform the configuration at this time,
        return a tuple of two strings (message, details), these will be
        displayed as a warning dialog to the user and the process will be
        aborted.

        The base class implementation of this method raises NotImplementedError
        so by default no user configuration is possible.
        '''
        
        # It is important to put this import statement here rather than at the
        # top of the module as importing the config class will also cause the
        # GUI libraries to be loaded, which we do not want when using calibre
        # from the command line
        try:
            from calibre_plugins.toc_view_generator.gui.Config import ConfigWidget
            
        except ImportError:
            from gui.Config import ConfigWidget
            
        return ConfigWidget()

    
    def save_settings(self, config_widget):
        '''
        Save the settings specified by the user with config_widget.
        @param config_widget: The widget returned by :meth:`config_widget`.
        '''
        config_widget.save_settings()

        # Apply the changes
        if self.actual_plugin_:
            self.actual_plugin_.apply_settings()
