# -*- coding: utf-8 -*-
from __future__ import unicode_literals, division, absolute_import, print_function
__license__   = 'GPL v3'
__copyright__ = '2014,2015,2016,2017,2018,2019,2020 DaltonST <DaltonShiTzu@outlook.com>'
__my_version__ = "3.6.106"    # Technical changes after Python 3.8 testing with Calibre 4.99.4

from PyQt5.Qt import (Qt, QDialog, QTableWidgetItem, QIcon, QByteArray, QSize)

from calibre.constants import DEBUG
from calibre.gui2 import question_dialog, error_dialog, info_dialog, gprefs
from calibre.utils.icu import sort_key
from calibre_plugins.quarantine_and_scrub.tag_rules_list_editor_ui import Ui_TagRulesListEditor

class NameTableWidgetItem(QTableWidgetItem):

    def __init__(self, txt):
        QTableWidgetItem.__init__(self, txt)
        self.initial_value = txt
        self.current_value = txt
        self.previous_value = txt

    def data(self, role):
        if role == Qt.DisplayRole:
            return self.current_value
        elif role == Qt.EditRole:
            return self.current_value
        else:
            return QTableWidgetItem.data(self, role)

    def setData(self, role, data):
        if role == Qt.EditRole:
            self.current_value = self.initial_value
        QTableWidgetItem.setData(self, role, data)

    def text(self):
        return self.initial_value

    def initial_text(self):
        return self.initial_value

    def previous_text(self):
        return self.initial_value

    def setText(self, txt):
        QTableWidgetItem.setText(self.initial_value)

    def __ge__(self, other):
        return sort_key(unicode(self.text())) >= sort_key(unicode(other.text()))

    def __lt__(self, other):
        return sort_key(unicode(self.text())) < sort_key(unicode(other.text()))

class RegexTableWidgetItem(QTableWidgetItem):

    def __init__(self, txt):
        QTableWidgetItem.__init__(self, txt)
        self.initial_value = txt
        self.current_value = txt
        self.previous_value = txt

    def data(self, role):
        if role == Qt.DisplayRole:
            return self.current_value
        elif role == Qt.EditRole:
            return self.current_value
        else:
            return QTableWidgetItem.data(self, role)

    def setData(self, role, data):
        if role == Qt.EditRole:
            self.previous_value = self.current_value
            self.current_value = data
        QTableWidgetItem.setData(self, role, data)

    def text(self):
        return self.current_value

    def initial_text(self):
        return self.initial_value

    def previous_text(self):
        return self.previous_value

    def setText(self, txt):
        self.current_value = txt
        QTableWidgetItem.setText(txt)

    def __ge__(self, other):
        return sort_key(unicode(self.text())) >= sort_key(unicode(other.text()))

    def __lt__(self, other):
        return sort_key(unicode(self.text())) < sort_key(unicode(other.text()))


class TagRulesListEditor(QDialog, Ui_TagRulesListEditor):


    def __init__(self, window, cat_name, tag_to_match, data, sorter, _actually_insert_tag_rules_rows, guidb):
        QDialog.__init__(self, window)
        Ui_TagRulesListEditor.__init__(self)
        self.setupUi(self)

        self._actually_insert_tag_rules_rows = _actually_insert_tag_rules_rows
        self.guidb = guidb

        # Put the category name into the title bar
        t = self.windowTitle()
        self.setWindowTitle(t + ' (' + cat_name + ')')
        # Remove help icon on title bar
        icon = self.windowIcon()
        self.setWindowFlags(self.windowFlags()&(~Qt.WindowContextHelpButtonHint))
        self.setWindowIcon(icon)

        # Get saved geometry info
        try:
            self.table_column_widths = gprefs.get('tag_rules_list_editor_table_widths', None)
        except:
            pass

        # initialization
        self.to_add_to_tag_rules_table = []
        self.all_tags = {}
        self.regexes = {}

        for row in data:
            if DEBUG: print("row is: ", bytes(row))
            for col in row:
                s1 = col
                self.all_tags[s1] = s1
                self.regexes[s1] = ''

        # Set up the column headings
        self.down_arrow_icon = QIcon(I('arrow-down.png'))
        self.up_arrow_icon = QIcon(I('arrow-up.png'))
        self.blank_icon = QIcon(I('blank.png'))

        self.table.setColumnCount(2)

        self.name_col = QTableWidgetItem(_('Real Tags Unused In Q+S Tag Rules Table [Select To Purge Via New Rule]'))
        self.table.setHorizontalHeaderItem(0, self.name_col)
        self.name_col.setIcon(self.up_arrow_icon)

        self.regex_col = QTableWidgetItem(_('Make /REGEX/? [1 = True, Blank or 0 = False]'))
        self.table.setHorizontalHeaderItem(1, self.regex_col)
        self.regex_col.setIcon(self.blank_icon)

        # Capture clicks on the horizontal header to sort the table columns
        hh = self.table.horizontalHeader()
        hh.setSectionsClickable(True)
        hh.sectionClicked.connect(self.header_clicked)
        hh.sectionResized.connect(self.table_column_resized)
        self.name_order = 0

        # Add the tags gotten in ui.py
        select_item = None
        self.table.setRowCount(len(self.all_tags))
        for row,tag in enumerate(sorted(self.all_tags.keys(), key=sorter)):
            item = NameTableWidgetItem(tag)
            item.setFlags(item.flags() | Qt.ItemIsSelectable | Qt.ItemIsEditable)
            self.table.setItem(row, 0, item)
            regex = self.regexes[tag]
            item = RegexTableWidgetItem(regex)
            item.setFlags(item.flags() | Qt.ItemIsSelectable | Qt.ItemIsEditable)
            self.table.setItem(row, 1, item)


        # Scroll to the selected item if there is one
        if select_item is not None:
            self.table.setCurrentItem(select_item)

        self.delete_button.clicked.connect(self.add_tags_to_insert_list)
        self.buttonBox.accepted.connect(self.accepted)

        try:
            geom = gprefs.get('tag_rules_list_editor_dialog_geometry', None)
            if geom is not None:
                self.restoreGeometry(QByteArray(geom))
            else:
                self.resize(self.sizeHint()+QSize(150, 100))
        except:
            pass

    def table_column_resized(self, col, old, new):
        self.table_column_widths = []
        for c in range(0, self.table.columnCount()):
            self.table_column_widths.append(self.table.columnWidth(c))

    def resizeEvent(self, *args):
        QDialog.resizeEvent(self, *args)
        if self.table_column_widths is not None:
            for c,w in enumerate(self.table_column_widths):
                self.table.setColumnWidth(c, w)
        else:
            # the vertical scroll bar might not be rendered, so might not yet
            # have a width. Assume 25. Not a problem because user-changed column
            # widths will be remembered
            w = self.table.width() - 25 - self.table.verticalHeader().width()
            w /= self.table.columnCount()
            for c in range(0, self.table.columnCount()):
                self.table.setColumnWidth(c, w)

    def save_geometry(self):
        gprefs['tag_rules_list_editor_table_widths'] = self.table_column_widths
        gprefs['tag_rules_list_editor_dialog_geometry'] = bytearray(self.saveGeometry())

    def accepted(self):
        self.save_geometry()
        self.add_tags_to_table_tag_rules()

    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def add_tags_to_table_tag_rules(self):
        self._actually_insert_tag_rules_rows(self.to_add_to_tag_rules_table)
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------

    def add_tags_to_insert_list(self):

        mappings = self.table.selectedItems()
        if not mappings:
            error_dialog(self, _('No items selected'),
                         _('You must select at least one item from the list.')).exec_()
            return
        ct = ', '.join([unicode(item.text()) for item in mappings])
        if not question_dialog(self, _('Are you sure?'),
            '<p>'+_('Are you sure you want to Easy-Add the following items to the Tag Rules Table for Future Automatic Purging?\
                            <br><br>ISBNs and BISAC Subject Codes Should Not Be Deleted [See User Instructions].')+'<br><br>'+ct):
            return

        i = 0
        for item in mappings:
            if i == 0:
                tag = item.text()     #each column in each row is a separate row in this list.  so, Tag is row 1, REGEX for that Tag is row 2, next Tag is row 3...
                i = i + 1
            else:
                regex = item.text()
                s  = '|||'.join([(unicode(tag)), (unicode(regex))])
                self.to_add_to_tag_rules_table.append(s)
                i = 0
        #END FOR


    def header_clicked(self, idx):
        self.do_sort_by_name()

    def do_sort_by_name(self):
        self.name_order = 1 if self.name_order == 0 else 0
        self.table.sortByColumn(0, self.name_order)
        self.name_col.setIcon(self.down_arrow_icon if self.name_order
                                                    else self.up_arrow_icon)




    #--------------------------------------------------------------------
    #END
