# -*- coding: utf-8 -*-
from __future__ import absolute_import, division, print_function, unicode_literals
__license__   = 'GPL v3'
__copyright__ = '2015,2016,2017,2018,2019,2020  DaltonST <DaltonShiTzu@outlook.com>'
__my_version__ = "1.0.87"   # Miscellaneous technical changes to improve performance.

from PyQt5.Qt import (Qt, QMenu, QDialog, QIcon, QFont, QAction, QSize, QDialogButtonBox, QPushButton, QProgressDialog,
                                        QVBoxLayout, QLabel, QTableWidget, QTableWidgetItem, QApplication, QInputDialog)

import os, sys
import apsw
import ast
from difflib import SequenceMatcher
from functools import partial
import re
import string
import subprocess
import time
from time import sleep
import unicodedata
import zipfile

from calibre import isbytestring
from calibre.constants import filesystem_encoding, DEBUG
from calibre.ebooks.metadata.book.base import Metadata
from calibre.gui2 import question_dialog, error_dialog, gprefs, Dispatcher
from calibre.gui2.actions import InterfaceAction
from calibre.utils.config import JSONConfig
from calibre.utils.serialize import json_loads

from polyglot.builtins import as_bytes, as_unicode, is_py3, iteritems, map, range, unicode_type
if is_py3:
    unichr = chr

from calibre_plugins.multi_column_search.config import prefs
from calibre_plugins.multi_column_search.common_utils import set_plugin_icon_resources, get_icon, create_menu_action_unique
from calibre_plugins.multi_column_search.jobs import start_mcs_build_index, start_mcs_trim_index
from calibre_plugins.multi_column_search.mcs_search_dialog import MCSDialog

from .unidecode import unidecode
from .fuzzywuzzy import fuzz, string_processing, utils
from .metaphone.metaphone import DoubleMetaphone, doublemetaphone as MCSDoubleMetaphone


PLUGIN_ICONS = ['images/mcsicon.png', 'images/mcssmallicon.png', 'images/readinstructionsicon.png', 'images/wrench-hammer.png']

COMPARE_AS_IS = unicode_type("Compare: 'as-is'")
COMPARE_UPPER_CASE = unicode_type("Compare: Upper Case")
COMPARE_NO_SPACES_NO_PUNCTUATION = unicode_type("Compare: No Spaces; No Punctuation")
COMPARE_UPPER_CASE_NO_SPACES_NO_PUNCTUATION = unicode_type("Compare: Upper Case; No Spaces; No Punctuation")
COMPARE_DECOMPOSED_NORMALIZED_ALPHABET = unicode_type('Compare as: Decomposed & Normalized Alphabet')

FUZZYWUZZY_NONE = unicode_type("Fuzzy Equality: None")
FUZZYWUZZY_TOKEN_SORT_RATIO = unicode_type("Fuzzy Equality: Token Sort Ratio")
FUZZYWUZZY_TOKEN_SET_RATIO = unicode_type("Fuzzy Equality: Token Set Ratio")
FUZZYWUZZY_SIMPLE_RATIO_UPPER_CASE_NO_PUNCTUATION = unicode_type("Fuzzy Equality: Simple Ratio, No Punctuation, Upper Case")
FUZZYWUZZY_SIMPLE_RATIO_DECOMPOSED_NORMALIZED_ALPHABET = unicode_type("Fuzzy Equality: Simple Ratio, Decomposed & Normalized Alphabet")
FUZZY_DOUBLEMETAPHONE_SIMPLE = unicode_type("Fuzzy Equality: DoubleMetaphone Sounds-Like")

SHOW_MCS_PROGRESS_BAR_MINIMUM_BOOKS = 150

SEARCHTAB = 'SearchTab'
REGEXTAB = 'RegexTab'
FINALFILTERTAB = 'FinalFilterTab'
SPECIALQUERIESTAB = 'SpecialQueriesTab'
SIMILARITYTAB = 'SimilarityTab'
RAWSQLTAB = 'RawSQLTab'
VIRTUALCOLUMNTXTTAB = 'VirtualColumnTXTTab'
WORDBOOKINDEXQUERYTAB = 'WordBookIndexQueryTab'
RESULTSTAB = 'ResultsTab'

INTRA_BOOK_SEARCH_TYPE = 'INTRA_BOOK_SEARCH'
INTER_BOOK_SEARCH_TYPE = 'INTER_BOOK_SEARCH'
REGULAR_EXPRESSIONS_TYPE = 'REGULAR_EXPRESSIONS'
FINAL_FILTERS_TYPE = 'FINAL_FILTERS'
SPECIAL_QUERIES_TYPE = 'SPECIAL_QUERIES'
SIMILARITY_QUERIES_TYPE = 'SIMILARITY_QUERIES'
SQL_QUERIES_TYPE = 'SQL_QUERIES'
TXT_QUERIES_TYPE = 'TXT_QUERIES'
WORD_QUERIES_TYPE = 'WORD_QUERIES'
POST_SEARCH_TYPE = 'POST_SEARCH'

cust_columns_integer_list = []
cust_columns_float_list = []
cust_columns_datetime_list = []
cust_columns_search_list = []

do_log = False

update_mcs_was_indexed_custom_column = False
custom_column_id = as_unicode(0)

class ActionMultiColumnSearch(InterfaceAction):

    name = 'MultiColumnSearch'
    action_spec = ('MCS', 'images/mcsicon.png', 'Search Across Multiple Columns Interactively, Collectively and Simultaneously, plus Full-Text and Word Searches.', None)
    action_type = 'global'
    accepts_drops = False
    auto_repeat = False
    priority = 9
    popup_type = 1

    documentation_path = "unknown"

    total_elapsed = 0

    global do_log
    do_log = False
    search_log = []
    log_title_search_path = ""

    global cust_columns_integer_list
    global cust_columns_float_list
    global cust_columns_datetime_list
    global cust_columns_search_list

    #-----------------------------------------------------------------------------------------
    def genesis(self):

        self.is_library_selected = True

        icon_resources = self.load_resources(PLUGIN_ICONS)
        set_plugin_icon_resources(self.name, icon_resources )

        self.menu = QMenu(self.gui)
        self.build_menus(self.gui)

        # main icon
        self.qaction.setMenu(self.menu)
        self.qaction.setIcon(get_icon(PLUGIN_ICONS[0]))
        self.qaction.triggered.connect(self.search_request_dialog)

        self.icon = get_icon(PLUGIN_ICONS[0])
        self.font = QFont()
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def initialization_complete(self):

        self.guidb = self.gui.library_view.model().db

        self.maingui = None

        prefs['BOOKS_VALID_FOR_POST_SEARCH_ACTIONS'] = unicode_type("[]")
        if 'RAW_SQL_QUERY_LAST_CURRENT' in prefs:
            del prefs['RAW_SQL_QUERY_LAST_CURRENT']  #deprecated
        prefs
        self.prefs = prefs

        self.build_custom_column_dicts()

        self.number_jobs_executing = 0

        #----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
        #------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
        # self.word_delimiter_characters_list is critical.  it is used in re.sub to change the characters to a single space in order to properly delimit a single word.
        # if the book-word index is sorted by word, bogus characters may appear at the end of the list in the 1st character of an indexed 'word'.
        # a few such words are unavoidable given the number of unicode symbols and number of unicode 'letters' for various alphabets.  undesireable, but unavoidable in the real world.
        # however, many such words will be too low in length to pass the minimum size threshhold and hence will not be indexed.
        # the LIKE _ wildcard in the first digit of the word in the word query is a workaround if necessary.  regardless, usually the correct 'real' word will have been indexed separately.
        #------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

        self.word_delimiter_characters_list = []    #  http://software.hixie.ch/utilities/cgi/unicode-decoder/character-identifier?characters=?
        s = unicode_type("")
        s = s + ',.;:!?¡¿-_«»()[]{}!="+“=*&^%$#@`|´‘©§¨¦¥¤£¢ °±²³µ¶¸¹ǂ‗–•’…€·~\\—/”' + "'" + "‘♦‹†‘˜" + "˚―"  + "˙ˌˈˇ˝¯" + "ˇ˝" + "ˆ÷" + "ªˇ˝¯" + "⧵⧶⧷⧸⧹⧺⧻⧼⧽⧾⧿" + "∗ߜ" + "⎞" + '⎬'
        s = s + "·¸¹º»¼½¾×" + "ǂ"  + " ʹ ʺ ʻ ʼ ʽ ʾ ʿ ˀ ˁ ˂ ˃ ˄ ˅ˆˇ ˈˉˊˋ ˌ ˍ ˎ ˏː ˑ ˒ ˓ ˔ ˕ ˖ ˗˘˙˚˛˜˝ ˞ ˟ˠˡˢˣ ˤ ˥ ˦ ˧ ˨ ˩ ˪ ˫ ˬ ˭ ˮ ˯ ˰ ˱ ˲ ˳ ˴ ˵ ˶ ˷ ˸ ˹ ˺ ˻ ˼ ˽ ˾ ˿ ̀ ́ ̂ ̃ ̄ ̅ ̆ ̇ ̈ ̉ ̊ ̋ ̌ ̍ ̎ ̏ ̐ ̑ ̒ ̓ ̔ ̕ ̖ ̗ ̘ ̙ ̚ ̛ ̜ ̝ ̞ ̟ ̠ ̡ ̢ ̣ ̤ ̥ ̧ ̨ ̩ ̪ ̫ ̬ ̭ ̮ ̯ ̰ ̱ ̲ ̳ ̴ ̵ ̶ ̷ ̸ ̹ ̺ ̻ ̼ ̽ ̾ ̿ ̀ ́ ͂ ̓ ̈́ ͅ ͆ ͇ ͈ ͉ ͊ ͋ ͌ ͍ ͎͏ ͐ ͑ ͒ ͓ ͔ ͕ ͖ ͗ ͘ ͙ ͚ ͛ ͜ ͝ ͞ ͟ ͠ ͡ ͢"
        s = s + "" + "" + '“,”' + '“’”' + '¸' + "	"  # character is a horizontal tab, not a space.  there are no spaces in this string at all.
        s = s + '— ' + ' ' + '■■■★☆■■⎪⎪′′■​' + '​         ' + '' + ' ' + '         '
        s = s + ' ' + '∫' + '∆' + '→←↑↓ ™®¯{}‹›«»¬ ‰' + 'ʬʭʮʯʰʱʲʳʴʵʶʷʸʹʺʻʼʽʾʿ' + 'ˠ	ˡ	ˢ	ˣ	ˤ	˥	˦	˧	˨	˩	˪	˫	ˬ	˭	ˮ	˯	˰	˱	˲	˳	˴	˵	˶	˷	˸	˹	˺	˻	˼	˽	˾	˿'
        s = s + '̀	́	̂	̃	̄	̅	̆	̇	̈	̉	̊	̋	̌	̍	̎	̏	̐	̑	̒	̓	̔	̕	̖	̗	̘	̙	̚	̛	̜	̝	̞	̟' + '̠	̡	̢	̣	̤	̥	̦	̧	̨	̩	̪	̫	̬	̭	̮	̯	̰	̱	̲	̳	̴	̵	̶	̷	̸	̹	̺	̻	̼	̽	̾	̿'
        s = s + '̀	́	͂	̓	̈́	ͅ	͆	͇	͈	͉	͊	͋	͌	͍	͎	͏	͐	͑	͒	͓	͔	͕	͖	͗	͘	͙	͚	͛	͜	͝	͞	͟' +'͠	͡	͢	ͣ	ͤ	ͥ	ͦ	ͧ	ͨ	ͩ	ͪ	ͫ	ͬ	ͭ	ͮ	ͯ'
        s = s + 'Ͱͱʹ͵͸͹ͺͻ	ͼ	ͽ	;' + '​​​thew​ss browning'   # non-displayable characters complexly comingled with the '​thewss browning'; real letters will be ignored later.
        s = s + '΢ys΢prm΢dn' + '⁄⁄⁄' + '' + '' + '≪≫'
        s = s + '“”‘’（）［］｛｝｟｠⦅⦆〚〛⦃⦄」〈〉《》【】〔〕⦗⦘』〖〗〘〙｢｣⟦⟧⟨⟩⟪⟫⟮⟯⟬⟭⌈⌉⌊⌋⦇⦈⦉⦊❛❜❝❞❨❩❪❫❴❵❬❭❮❯❰❱❲❳﴾﴿⎴⎵﹁﹂﹃﹄︹︺︻︼︗︘︿﹀︽︾﹇﹈︷︸〈〉⦑⦒⧼⧽﹙﹚﹛﹜﹝﹞⁽⁾₍₎⦋⦌⦍⦎⦏⦐⁅⁆⸢⸣⸤⸥⟅⟆⦓⦔⦕⦖⸦⸧⸨⸩⧘⧙⧚⧛⸜⸝⸌⸍⸂⸃⸄⸅⸉⸊᚛᚜༺༻༼༽⊆⊇.⎧⎨⎩√∛∜∞⧜⧝⧞×✕✖÷−∕∗∘∙⋅⋆⊞⊟⊠⊡❦❡❗❢❣❤❥🙶🙷🙸🙾🙿🙼🙽'
        s = s + 'ことについて' + 'ْ' + '‡' + '・' + '„' + '∀' + '„“'
        for c in s:    # there are no spaces in this string at all.
            if isinstance(c,unicode_type):
                if c != " " and len(c) == 1:
                    if not c in "abcdefghijklmnopqrstuvwxyz" and not c in "0123456789":   # the re.sub syntax used is different for numbers to avoid escaping issues with re
                        if not c in self.word_delimiter_characters_list:
                            self.word_delimiter_characters_list.append(c)
            else:
                try:
                    if DEBUG: print("character is not unicode: ", as_unicode(c))
                except:
                    pass
        #END FOR
        for n in range(0,32):     # control characters thru 31; http://www.ascii-code.com/
            c = unichr(n)
            self.word_delimiter_characters_list.append(c)
            c = chr(n)
            self.word_delimiter_characters_list.append(c)
        #END FOR
        for n in range(123,192):     # undesirable characters 123 thru 191; http://www.ascii-code.com/
            if n != 138 and n != 140 and n != 154  and n != 156 and n != 158 and n != 159 :
                c = unichr(n)
                self.word_delimiter_characters_list.append(c)
                c = chr(n)
                self.word_delimiter_characters_list.append(c)
        #END FOR
        self.word_delimiter_characters_list.append(unichr(247))
        self.word_delimiter_characters_list.append(unichr(226))
        self.word_delimiter_characters_list.append(unichr(215))
        self.word_delimiter_characters_list.append(unichr(149))

        self.word_delimiter_characters_list.append(chr(247))
        self.word_delimiter_characters_list.append(chr(226))
        self.word_delimiter_characters_list.append(chr(215))
        self.word_delimiter_characters_list.append(chr(149))

        for n in range(91,97):     # undesirable characters 91 thru 96; http://www.ascii-code.com/
            c = unichr(n)
            self.word_delimiter_characters_list.append(c)
            c = chr(n)
            self.word_delimiter_characters_list.append(c)
        #END FOR

        #~ Japanese
        #~ 0x4E00-0x9FBF    Kanji         int(0x4E00) = 19968   unichr(19968) = u'\u4e00'
        #~ 0x3040-0x309F    Hiragana
        #~ 0x30A0-0x30FF    Katakana
        #~ Chinese, Japanese, Old Korean, Old Vietnamese
        #~ 4E00-62FF, 6300-77FF, 7800-8CFF, 8D00-9FFF

        #~ Modern Korean:  Hangul Jamo Extended-A (U+A960–U+A97F) and Hangul Jamo Extended-B (U+D7B0–U+D7FF)

        if DEBUG: print("[0] un-optimized number of undesirable characters accumulated: ", as_unicode(len(self.word_delimiter_characters_list)))    #  ~759 or so

        tmp_list = []
        for row in self.word_delimiter_characters_list:
            if isinstance(row,unicode_type):
                tmp_list.append(row)
            else:
                try:
                    row = row.decode('utf-8')
                    if isinstance(row,unicode_type):
                        tmp_list.append(row)
                    else:
                        pass
                        #~ if DEBUG: print("character is not unicode, and ignored[0]: ", as_unicode(row))
                except:
                    pass
                    #~ if DEBUG: print("character is not unicode, and ignored[1]: ", as_unicode(row))
        #END FOR

        #~ #python2 may give this warning: "UnicodeWarning: Unicode equal comparison failed to convert both arguments to Unicode - interpreting them as being unequal"
        try:
            tmp_set = set(tmp_list)
            self.word_delimiter_characters_list = list(tmp_set)
        except:
            pass

        if DEBUG: print("[1] optimized number of undesirable characters accumulated: ", as_unicode(len(self.word_delimiter_characters_list)))     #  ~ 599 or so


        #~ The following list contains the decimal equivalent of each character in self.word_delimiter_characters_list
        #~ self.word_delimiter_ordinal_list = []
        #~ for c in self.word_delimiter_characters_list:
            #~ self.word_delimiter_ordinal_list.append(ord(c))
        #~ #END FOR
        #~ self.word_delimiter_ordinal_list.sort()
        #~ for n in self.word_delimiter_ordinal_list:
            #~ if DEBUG: print(as_unicode(n))
        #END FOR
        #~ del self.word_delimiter_ordinal_list

        del c
        del s
        del tmp_set
        del tmp_list

        self.previous_now = ""
    #----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    #----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
        if DEBUG: print("MCS has finished initialization...")
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def library_changed(self,guidb):

        self.guidb = guidb

        self.build_menus(self.gui)
        self.qaction.setMenu(self.menu)
        self.qaction.setIcon(get_icon(PLUGIN_ICONS[0]))

        try:
            self.mcs_dialog.close()
        except:
            pass

        try:
            self.mcs_progress_bar_dialog.close()
        except:
            pass

        prefs['BOOKS_VALID_FOR_POST_SEARCH_ACTIONS'] = unicode_type("[]")
        prefs

        #~ self.custom_column_label_dict,self.custom_column_datatype_dict,self.custom_column_normalized_dict = self.build_custom_column_dicts()
        self.build_custom_column_dicts()

        self.number_jobs_executing = 0

        self.call_refresh_mcs_search_accelerator_tables()
    #-----------------------------------------------------------------------------------------
    def search_request_dialog(self):

        self.guidb = self.gui.library_view.model().db

        try:
            self.mcs_dialog.close()
        except:
            pass

        try:
            self.mcs_progress_bar_dialog.close()
        except:
            pass

        if self.number_jobs_executing > 0:
            n = self.number_jobs_executing
            if n == 1:
                msg = as_unicode(n) + " MCS Indexing Job is currently running.  Please wait until it finishes."
            else:
                msg = as_unicode(n) + " MCS Indexing Jobs are currently running.  Please wait until they finish."
            self.gui.status_bar.showMessage(msg)
            msg = msg + "<br><br>If this message is in error, you can reset the MCS Job Count by either switching briefly to another library and then switching back, or restarting Calibre. "
            error_dialog(self.gui, _('MCS'),_((msg)), show=True)
            return

        my_db = self.guidb
        path = my_db.library_path
        if isbytestring(path):
            path = path.decode(filesystem_encoding)
        path = path.replace(os.sep, '/')
        path = os.path.join(path, 'metadata.db')
        path = path.replace(os.sep, '/')
        self.qaction.setIcon(get_icon(PLUGIN_ICONS[1]))
        self.mcs_dialog = MCSDialog(self.gui,self.qaction.icon(),self.font,self.guidb,self.prefs,self.search_request_control,
                                                        self.swap_criteria_refresh,path, self.custom_column_label_dict,
                                                        self.custom_column_datatype_dict,self.custom_column_normalized_dict,
                                                        self.special_query_control,self.raw_sql_query_control,
                                                        self.search_virtual_column_format_txt_control,
                                                        self.search_word_in_book_query_control,
                                                        self.search_similarity_control,
                                                        self.mcs_build_index_job_control,
                                                        self.mcs_trim_index_job_control,
                                                        self.load_criteria_settings_generic,
                                                        self.create_criteria_parm_list_generic,
                                                        self.save_settings_parm_history_to_prefs_generic,
                                                        self.unpack_selected_parm_generic,
                                                        self.manage_saved_criteria)

        self.mcs_dialog.setAttribute(Qt.WA_DeleteOnClose)
        self.mcs_dialog.show()
        self.qaction.setIcon(get_icon(PLUGIN_ICONS[0]))

        self.call_refresh_mcs_search_accelerator_tables()
    #-----------------------------------------------------------------------------------------
    def swap_criteria_refresh(self):
        self.mcs_dialog.close()
        self.search_request_dialog()
    #-----------------------------------------------------------------------------------------
    def search_request_control(self,guidb,param_dict,sel_type,force_repaint,force_abort,current_path,search_path):

        if param_dict['CROSS_LIBRARY_DUPLICATES_SEARCH'] == unicode_type("True"):
            self.is_cross_library_duplicates_search = True
        else:
            self.is_cross_library_duplicates_search = False

        if self.is_cross_library_duplicates_search:
            self.search_request_control_cl_duplicates(guidb,param_dict,sel_type,force_repaint,force_abort,current_path,search_path)
        else:
            self.search_request_control_routine(guidb,param_dict,sel_type,force_repaint,force_abort,current_path,search_path)
    #---------------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------------
    def search_request_control_routine(self,guidb,param_dict,sel_type,force_repaint,force_abort,current_path,search_path):
        global log_title_search_path
        global do_log

        log_title_search_path = search_path

        self.current_path = current_path
        self.search_path = search_path

        if isbytestring(self.current_path):
            self.current_path = self.current_path.decode(filesystem_encoding)
        self.current_path = self.current_path.replace(os.sep, '/')

        if isbytestring(self.search_path):
            self.search_path = self.search_path.decode(filesystem_encoding)
        self.search_path = self.search_path.replace(os.sep, '/')

        if force_abort:
            self.mcs_dialog.close()
            self.search_request_dialog()
            self.gui.status_bar.showMessage("MCS Criteria Error(s).  Search Canceled.")
            return

        if param_dict['INTER_BOOK_SEARCH'] == unicode_type("True"):
            self.is_interbooksearch = True
        else:
            self.is_interbooksearch = False

        if not (self.current_path == self.search_path):
            if self.is_interbooksearch:
                self.gui.status_bar.showMessage("MCS Criteria Error(s).  Search Canceled.")
                error_dialog(self.gui, _('MCS'),_(("Inter-Book Searches for 'Another' Library are Prohibited.<br><br>\
                                                                    Execution Canceled. ")), show=True)
                return
            else:
                # Cross-Library searches do not support Transform Functions other than 'as-is' due to complexity and performance issues.
                param_dict['TRANSFORM_FUNCTION1'] = unicode_type(COMPARE_AS_IS)
                param_dict['TRANSFORM_FUNCTION2'] = unicode_type(COMPARE_AS_IS)

        if force_repaint:
            self.mcs_dialog.close()

        total_start = time.time()

        self.gui.status_bar.showMessage("MCS is Searching...")

        self.comments_dialog_active = False

        param_dict = self.tweak_param_dict(param_dict)

        if self.is_interbooksearch:
            do_log = False

       #-----------------------------
        if self.current_path == self.search_path:
            if self.is_interbooksearch:
                sel_type = "all"
                selected_books_list = self.get_selected_books(guidb,sel_type)
            else:
                selected_books_list = self.get_selected_books(guidb,sel_type)
            n_books = len(selected_books_list)
            if n_books == 0:
                self.gui.status_bar.showMessage("MCS Found No Selected Books to Search...")
                if force_repaint:
                    self.mcs_dialog.close()
                    self.search_request_dialog()
                return
            if force_repaint:
                msg = ('MCS corrected your invalid criteria and is now searching %d book(s)' %(n_books))
            else:
                msg = ('MCS is Searching %d book(s)' %(n_books))
            self.gui.status_bar.showMessage(msg)
        else:
            if self.is_interbooksearch:
                self.gui.status_bar.showMessage("MCS Criteria Error(s).  Search Canceled.")
                return
            else:
                do_log = True
                self.need_tags_too = True  #otherwise, the GUI Log won't show the Tags...
                self.need_authors_too = True  #ditto
                selected_books_list = []
                if force_repaint:
                    self.mcs_dialog.close()
                    self.search_request_dialog()
                    msg = ('MCS is Searching Another Calibre Library')
                    self.gui.status_bar.showMessage(msg)
       #-----------------------------
        path = unicode_type(self.search_path)
        path = path.replace(os.sep, '/')
        try:
            my_db = apsw.Connection(path)
        except Exception as e:
            if DEBUG: print(as_unicode(e))
            path = path.encode("ascii", "ignore")
            msg = as_unicode("MCS cannot use the path that you selected:" + as_unicode(path) + " - " + as_unicode(e) )
            self.gui.status_bar.showMessage(msg)
            self.mcs_dialog.close()
            self.search_request_dialog()
            return
        my_cursor = my_db.cursor()
        mysql = "PRAGMA main.busy_timeout = 5000;"      #PRAGMA busy_timeout = milliseconds;
        my_cursor.execute(mysql)
        #--------------------------------------
        self.create_temp_views_and_tables(my_db,my_cursor)
        if self.current_path != self.search_path:
            self.refresh_mcs_search_accelerator_tables(my_db,my_cursor)
        #-----------------------------
        if self.current_path != self.search_path:
            selected_books_list = self.get_all_external_library_books(my_db,my_cursor)
            n_books = len(selected_books_list)
            if n_books == 0:
                self.gui.status_bar.showMessage("MCS Found No External Books to Search...")
                return

        msg = ('MCS is Searching %d book(s)' %(n_books))
        self.gui.status_bar.showMessage(msg)

        if n_books >= 10000:
            msg = ('MCS is Searching %d books, and it might take a very long time. Please be patient. ' %(n_books))
            self.gui.status_bar.showMessage(msg)
        else:
            if n_books >= 1000:
                msg = ('MCS is Searching %d books.  Please be patient. ' %(n_books))
                self.gui.status_bar.showMessage(msg)

        self.display_progress_bar(n_books)

        if param_dict['TRANSFORM_FUNCTION1'] == unicode_type(COMPARE_AS_IS):
            self.transform_1_type = 1
        elif param_dict['TRANSFORM_FUNCTION1'] == unicode_type(COMPARE_UPPER_CASE):
            self.transform_1_type = 2
        elif param_dict['TRANSFORM_FUNCTION1'] == unicode_type(COMPARE_NO_SPACES_NO_PUNCTUATION):
            self.transform_1_type = 3
        elif param_dict['TRANSFORM_FUNCTION1'] == unicode_type(COMPARE_UPPER_CASE_NO_SPACES_NO_PUNCTUATION):
            self.transform_1_type = 4
        elif param_dict['TRANSFORM_FUNCTION1'] == unicode_type(COMPARE_DECOMPOSED_NORMALIZED_ALPHABET):
            self.transform_1_type = 5
        else:
            self.transform_1_type = 1

        if param_dict['TRANSFORM_FUNCTION2'] == unicode_type(COMPARE_AS_IS):
            self.transform_2_type = 1
        elif param_dict['TRANSFORM_FUNCTION2'] == unicode_type(COMPARE_UPPER_CASE):
            self.transform_2_type = 2
        elif param_dict['TRANSFORM_FUNCTION2'] == unicode_type(COMPARE_NO_SPACES_NO_PUNCTUATION):
            self.transform_2_type = 3
        elif param_dict['TRANSFORM_FUNCTION2'] == unicode_type(COMPARE_UPPER_CASE_NO_SPACES_NO_PUNCTUATION):
            self.transform_2_type = 4
        elif param_dict['TRANSFORM_FUNCTION2'] == unicode_type(COMPARE_DECOMPOSED_NORMALIZED_ALPHABET):
            self.transform_2_type = 5
        else:
            self.transform_2_type = 1

        if self.is_interbooksearch:
            self.current_column_fuzzywuzzy = 0
            found_list = self.inter_book_search_control(my_db,my_cursor,selected_books_list,param_dict)
        else:
            found_list = self.intra_book_search_control(my_db,my_cursor,selected_books_list,param_dict)

        #-----------------------------------------
        self.skip_message = False
        if self.show_mcs_progress_bar:
            if self.mcs_progress_bar_dialog.wasCanceled():
                self.user_clicked_cancel = True
            if self.user_clicked_cancel:
                self.gui.search.clear()
                msg = "Search Canceled by User.  Disregard Any Results."
                self.gui.status_bar.showMessage(msg)
                self.skip_message = True
            self.mcs_progress_bar_dialog.close()
        #-----------------------------------------
        if not self.skip_message:
            total_end = time.time()
            total_elapsed = total_end - total_start
            msg = 'Searched %d book(s), found %d match(es) in %.3f seconds' %(len(selected_books_list), len(found_list), total_elapsed)
            self.gui.status_bar.showMessage(msg)
        #-----------------------------------------
        if force_repaint:
            self.search_request_dialog()
        #-----------------------------------------
        if self.comments_dialog_active:
            from calibre_plugins.multi_column_search.mcs_search_comments_dialog import MCSSearchCommentsDialog
            self.guidb = self.gui.library_view.model().db
            self.mcs_search_comments_dialog = MCSSearchCommentsDialog(self.gui,self.guidb,self.qaction.icon(),found_list,param_dict)
            self.mcs_search_comments_dialog.show()
            del MCSSearchCommentsDialog
        #-----------------------------------------
        del found_list
        del param_dict
        del self.param_dict
    #-----------------------------------------------------------------------------------------
    def intra_book_search_control(self,my_db,my_cursor,selected_books_list,param_dict):
        global do_log
        #-----------------------------
        std_metadata_list, cust_metadata_list = self.get_all_book_metadata(my_db,my_cursor,selected_books_list,param_dict)
        #-----------------------------
        my_db.close()
        #-----------------------------
        #-----------------------------
        if self.show_mcs_progress_bar:
            if self.mcs_progress_bar_dialog.wasCanceled():
                self.user_clicked_cancel = True
            if self.user_clicked_cancel:
                dummy_list = []
                return dummy_list
        #-----------------------------
        if self.current_path == self.search_path:
            do_log = False
            found_list = self.do_intra_book_searches_by_book(param_dict,selected_books_list,std_metadata_list, cust_metadata_list)
            #-----------------------------
            if self.show_mcs_progress_bar:
                if self.mcs_progress_bar_dialog.wasCanceled():
                    self.user_clicked_cancel = True
                if self.user_clicked_cancel:
                    dummy_list = []
                    return dummy_list
            #-----------------------------
            if self.show_all_authors_books:
                found_list = self.add_all_authors_books(found_list)
            if self.apply_final_filters:
                found_list = self.apply_final_filters_control(found_list)
            #-----------------------------
            self.mark_found_books(found_list)
            #-----------------------------
            if param_dict['NAME1'] == "comments" and param_dict['TYPE1'] == 'TEXT':
                self.comments_dialog_active = True
            else:
                self.comments_dialog_active = False
            #-----------------------------
        else:
            self.comments_dialog_active = False
            do_log = True
            found_list = self.do_intra_book_searches_by_book(param_dict,selected_books_list,std_metadata_list, cust_metadata_list)
            #-----------------------------
            if self.show_mcs_progress_bar:
                if self.mcs_progress_bar_dialog.wasCanceled():
                    self.user_clicked_cancel = True
                if self.user_clicked_cancel:
                    do_log = False
                    dummy_list = []
                    return dummy_list
            #-----------------------------
            self.display_search_log()
        #-----------------------------

        del param_dict

        return found_list
    #-----------------------------------------------------------------------------------------
    def inter_book_search_control(self,my_db,my_cursor,selected_books_list,param_dict):
        #-----------------------------
        std_metadata_list, cust_metadata_list = self.get_all_book_metadata(my_db,my_cursor,selected_books_list,param_dict)
        #-----------------------------
        if self.show_mcs_progress_bar:
            if self.mcs_progress_bar_dialog.wasCanceled():
                self.user_clicked_cancel = True
            if self.user_clicked_cancel:
                dummy_list = []
                return dummy_list
        #-----------------------------
        found_list = self.do_inter_book_searches_all_books(my_db,my_cursor,param_dict,selected_books_list,std_metadata_list, cust_metadata_list)
        #-----------------------------
        if self.show_mcs_progress_bar:
            if self.mcs_progress_bar_dialog.wasCanceled():
                self.user_clicked_cancel = True
            if self.user_clicked_cancel:
                dummy_list = []
                return dummy_list
        #-----------------------------
        if self.show_all_authors_books:
            found_list = self.add_all_authors_books(found_list)
        if self.apply_final_filters:
            found_list = self.apply_final_filters_control(found_list)
        #-----------------------------
        self.mark_found_books(found_list)

        del param_dict

        return found_list
    #-----------------------------------------------------------------------------------------
    def mark_found_books(self, found_list):

        found_dict = {}
        s_true = 'true'
        for row in found_list:
            key = int(row)
            found_dict[key] = s_true

        marked_ids = dict.fromkeys(found_dict, s_true)
        self.gui.current_db.set_marked_ids(marked_ids)
        self.gui.search.clear()
        self.gui.search.set_search_string('marked:true')

        prefs['BOOKS_VALID_FOR_POST_SEARCH_ACTIONS'] = unicode_type(found_list)
        prefs

        del found_list
        del found_dict
    #-----------------------------------------------------------------------------------------
    def get_selected_books(self,guidb,sel_type):

        selected_books_list = []
        del selected_books_list
        selected_books_list = []

        book_ids_list = []
        work_book_ids_frozenset = ""

        if sel_type == "selected":
            book_ids_list = list(map( partial(self.convert_id_to_book), self.gui.library_view.get_selected_ids() ) )
            n = len(book_ids_list)
            if  n == 0:
                return selected_books_list
            for item in book_ids_list:
                s = as_unicode(item['calibre_id'])
                selected_books_list.append(s)
        else:
            db = self.gui.current_db.new_api
            work_book_ids_frozenset = db.all_book_ids()
            for row in work_book_ids_frozenset:
                selected_books_list.append(row)

        del book_ids_list
        del work_book_ids_frozenset

        return selected_books_list
    #-----------------------------------------------------------------------------------------
    def get_all_external_library_books(self,my_db,my_cursor):

        sleep(0.02)

        selected_books_list = []

        mysql = "SELECT id FROM books"
        my_cursor.execute(mysql)
        tmp_rows = my_cursor.fetchall()
        if not tmp_rows:
            pass
        else:
            if len(tmp_rows) == 0:
                pass
            else:
                for row in tmp_rows:
                    for col in row:
                        if self.is_cross_library_duplicates_search:
                            selected_books_list.append(int(col))
                        else:
                            selected_books_list.append(as_unicode(col))

        return selected_books_list
    #-----------------------------------------------------------------------------------------
    def convert_id_to_book(self, idval):
        book = {}
        book['calibre_id'] = idval
        return book
    #-----------------------------------------------------------------------------------------
    def do_intra_book_searches_by_book(self,param_dict,selected_books_list,std_metadata_list,cust_metadata_list):

        global do_log
        if not do_log:
            do_log = True
        global search_log
        search_log = []

        found_list = []

        selected_books_list.sort()

        if self.show_mcs_progress_bar:
            n_progress_counter = 0
            self.mcs_progress_bar_dialog.setLabelText("Intra-Book Search: Analyzing Metadata")
            self.mcs_progress_bar_dialog.setValue(0)

        for row in selected_books_list:
            book = int(row)
            was_found = self.search_metadata_for_book(param_dict,book,std_metadata_list,cust_metadata_list)
            if was_found:
                found_list.append(book)
            if self.show_mcs_progress_bar:
                n_progress_counter = n_progress_counter + 1
                self.mcs_progress_bar_dialog.setValue(n_progress_counter)
                if self.mcs_progress_bar_dialog.wasCanceled():
                    self.user_clicked_cancel = True
                if self.user_clicked_cancel:
                    dummy_list = []
                    return  dummy_list
        #END FOR

        return found_list
    #-----------------------------------------------------------------------------------------
    def search_metadata_for_book(self,param_dict,current_book,std_metadata_list,cust_metadata_list):

        global cust_columns_integer_list
        global cust_columns_float_list
        global cust_columns_datetime_list

        global search_log
        global do_log

        try:

            #-------------------------------
            my_std_row1 = ""
            my_std_row2 = ""
            for row in std_metadata_list:
                book,title,authorsall,series,tagsall,publisher,comments,pubdate,path = row
                if not book == current_book:
                    continue
                else:
                    my_std_row1 = row
                    my_std_row2 = row
                    break
            #END FOR
            #-------------------------------
            value_dict = {}
            del value_dict
            value_dict = {}
            n = len(cust_metadata_list)

            for row in cust_metadata_list:
                book_found_in_cust_metadata_list = False
                new_book_data_dict = row
                book =  new_book_data_dict['book']
                if not book == current_book:
                    continue
                else:
                    book_found_in_cust_metadata_list = True
                    label = new_book_data_dict['label']
                    value = new_book_data_dict['value']
                    if not label:
                        label = ""
                    if not value:
                        value = ""
                    if value == "":
                        if label in cust_columns_integer_list and label != "":
                            value = 0
                        else:
                            if label in cust_columns_float_list and label != "":
                                value = 0.0
                            else:
                                if label in cust_columns_datetime_list and label != "":
                                    value = '0000-00-00 00:00:00+00:00'
                                    value = value[0:10]
                                    value = as_unicode(value)
                                else:
                                    pass
                    else:
                        if label in cust_columns_datetime_list and label != "":
                            value = value[0:10]
                            value = as_unicode(value)

                    value_dict[label] = value

            #END FOR

            #-------------------------------
            #  The Rules:
            #-------------------------------
            # Rule [A]: name1 is ALWAYS a column name (either custom or standard)
            # Rule [B]: if type1 = TEXT, then text1 is an arbitrary value of the column name1 to search for.   example:  the value of 'Isaac Asimov'
            # Rule [C]: if type1 = LABEL, then text1 is a column name itself, and its value must be compared to the value for column name1.  example:  authors (a standard column), that has a value such as 'Isaac Asimov'
            #-------------------------------

            name1 = param_dict['NAME1']        #e.g. #work_author or authors
            text1    = param_dict['TEXT1']          #e.g.  authors  OR  Isaac Asimov
            type1   = param_dict['TYPE1']          #e.g. LABEL   OR   TEXT
            operator1 = param_dict['OPERATOR1']      #  =,<,>, contains, notcontains, regex
            if operator1 == "regex":
                regex1 = param_dict['REGEX1']                   # regular expression selected
                c = param_dict['IGNORECASE1']
                if as_unicode(c) == as_unicode("1"):
                    ignorecase1 = True
                else:
                    ignorecase1 = False
            else:
                regex1 = ""
                ignorecase1 = False

            andornotignore = param_dict['ANDORNOT']     #AND or OR or NOT or IGNORE

            name2 = param_dict['NAME2']
            text2    = param_dict['TEXT2']
            type2   = param_dict['TYPE2']
            operator2 = param_dict['OPERATOR2']
            if operator2 == "regex":
                regex2 = param_dict['REGEX2']                   # regular expression selected
                c = param_dict['IGNORECASE2']
                if as_unicode(c) == as_unicode("1"):
                    ignorecase2 = True
                else:
                    ignorecase2 = False
            else:
                regex2 = ""
                ignorecase2 = False

            #-------------------------------
            if name1 in cust_columns_integer_list  and name1 != "" and (text1 == "" or text1 == "0"):
                text1 = 0
            else:
                if name1 in cust_columns_float_list and name1 != "" and (text1 == "" or text1 == "0" or text1 == "0." or text1 == "0.0"):
                    text1 = 0.0
                else:
                    if name1 in cust_columns_datetime_list and name1 != "" \
                                    and (text1 == "" or text1 == "0" or text1[0:4] == '0000' or text1[0:7] == '0000-00' or text1[0:10] == '0000-00-00' \
                                      or text1[0:6] == '000000' or text1[0:8] == '00000000'):
                        newvalue = '0000-00-00 00:00:00+00:00'
                        newvalue = newvalue[0:10]
                        newvalue = as_unicode(newvalue)
                        text1 = newvalue
            #-------------------------------
            if name2 in cust_columns_integer_list  and name2 != "" and (text2 == "" or text2 == "0"):
                text2 = 0
            else:
                if name2 in cust_columns_float_list and name2 != "" and (text2 == "" or text2 == "0" or text2 == "0." or text2 == "0.0"):
                    text2 = 0.0
                else:
                    if name2 in cust_columns_datetime_list and name2 != "" \
                                    and (text2 == "" or text2 == "0" or text2[0:4] == '0000' or text2[0:7] == '0000-00' or text2[0:10] == '0000-00-00' \
                                      or text2[0:6] == '000000' or text2[0:8] == '00000000'):
                        newvalue = '0000-00-00 00:00:00+00:00'
                        newvalue = newvalue[0:10]
                        newvalue = as_unicode(newvalue)
                        text2 = newvalue
            #-------------------------------
            if type1 == 'TEXT':
                if isinstance(text1,int) or isinstance(text1,float):
                    pass
                else:
                    self.current_column_fuzzywuzzy = 1
                    text1 = self.apply_transform_functions_single_value(1,text1)
            if type2 == 'TEXT':
                if isinstance(text2,int) or isinstance(text2,float):
                    pass
                else:
                    self.current_column_fuzzywuzzy = 2
                    text2 = self.apply_transform_functions_single_value(2,text2)
            #-------------------------------
            if name1 == 'pubdate':
                if type1 == 'TEXT':
                    text1 = as_unicode(text1)
                    if text1 == as_unicode(""):
                        text1 = as_unicode("0000-00")
            if name2 == 'pubdate':
                if type2 == 'TEXT':
                    text2 = as_unicode(text2)
                    if text2 == as_unicode(""):
                        text2 = as_unicode("0000-00")

            #-------------------------------
            found_name1 = False
            found_text1 = False
            found_title = False
            found_author = False
            found_series = False
            found_tag = False
            found_publisher = False
            found_comments = False
            found_pubdate = False
            found_path = False

            name1_found = ""
            text1_found = ""
            name1_value = ""
            value = ""

            text1_found_was_label = False
            current_status = "INCOMPLETE"
            column1_is_complete = False
            name1_is_custom_column = False

            self.current_column_fuzzywuzzy = 1

            #---------------------------------------------------------------------------------------------------------------------------------------
            book_dict = {}   # is what gets output in the GUI log for cross-library
            #---------------------------------------------------------------------------------------------------------------------------------------
            #---------------------------------------------------------------------------------------------------------------------------------------
            # First, Custom Columns for Column 1:
            #---------------------------------------------------------------------------------------------------------------------------------------
            #~ for k,v in value_dict.iteritems():  #custom column (only) metadata for book
            for k,v in iteritems(value_dict):  #custom column (only) metadata for book
                label = k    #e.g. #work_author
                value = v   #e.g.  Isaac Asimov   OR  #work_title

                if name1 == label:  # Rule [A] name1 is ALWAYS a column name (either custom or standard)
                    found_name1 = True  #e.g. #work_author
                    #name1_found for custom columns is the column value (contents), not the column label.  different than for standard columns.
                    if not label in cust_columns_integer_list:
                        if not label in cust_columns_float_list:
                            value = self.apply_transform_functions_single_value(1,value)
                    name1_found = value  #e.g. Joe Blow (Joe Blow was the #work_author for this book)
                    name1_is_custom_column = True  #so, must use name1_found as the value to be compared
                    book_dict[label] = value   # for standard columns, it is, for example:  book_dict['AUTHORS'] = authorsconcat

                if type1 == "TEXT": # Rule [B] if text1 = TEXT, then text1 is an arbitrary value of the column name1 to search for.
                    if label in cust_columns_integer_list  and label != "":
                        if text1 == "":
                            text1 = 0
                        text1_found = int(text1)
                        found_text1 = True
                        text1 = int(text1)
                        book_dict[label] = text1
                    else:
                        if label in cust_columns_float_list  and label != "":
                            if text1 == "":
                                text1 = 0.00
                            text1_found = float(text1)
                            found_text1 = True
                            text1 = float(text1)
                            book_dict[label] = text1
                        else:
                            value = self.apply_transform_functions_single_value(1,value)
                            text1_found = value
                            found_text1 = True
                            book_dict[label] = value

                if type1 == "LABEL": # Rule [C] if text1 = LABEL, then text1 is a column name itself, and its value must be compared to the value for column name1.
                    if text1 == label:
                        value = self.apply_transform_functions_single_value(1,value)
                        found_text1 = True
                        text1_found = value
                        text1_found_was_label = True
                        book_dict[label] = value

            #-----------------------------------------------
            # Second, Standard Columns for Column 1:
            #-----------------------------------------------
            if not found_name1:
                if name1 == "title":
                    found_name1 = True
                    name1_found = "title"   #name1_found for standard  columns is the column name, not its value (contents)
                if name1 == "authors":
                    found_name1 = True
                    name1_found = "authors"
                if name1 == "series":
                    found_name1 = True
                    name1_found = "series"
                if name1 == "tags":
                    found_name1 = True
                    name1_found = "tags"
                if name1 == "publisher":
                    found_name1 = True
                    name1_found = "publisher"
                if name1 == "comments":
                    found_name1 = True
                    name1_found = "comments"
                if name1 == "pubdate":
                    found_name1 = True
                    name1_found = "pubdate"
                if name1 == "path":
                    found_name1 = True
                    name1_found = "path"
            #-------------------------------
            if not found_name1: #no point in continuing...unless OR for column 2.
                if as_unicode(andornotignore) == as_unicode("OR"):
                    pass
                else:
                    was_found = False
                    search_log.append("Name1 Not Found; no point in continuing")

                    return was_found,search_log
            #-------------------------------
            # Rule [A] has been totally satisfied as of this point for Column 1
            #-------------------------------
            #-------------------------------
            column_source = 1
            my_std_row1 = self.apply_transform_functions_standard_columns(column_source,my_std_row1)
            book,title,authorsconcat,series,tagsconcat,publisher,comments,pubdate,path = my_std_row1
            #~ if DEBUG: print("[column source 1] >>> book,title,authorsall,series,tagsall", as_unicode(book), title, authorsall, series, tagsall)
            #-------------------------------
            if not title:
                title = ""
            if not authorsconcat:
                authorsconcat = ""
            if not series:
                series = ""
            if not tagsconcat:
                tagsconcat = ""
            if not publisher:
                publisher = ""
            if not comments:
                comments = ""
            if not pubdate:
                pubdate = "0101-01-01 00:00:00+00:00"   #default in table books
            if not path:
                path = ""

            pubdate = as_unicode(pubdate)
            pubdate = pubdate[0:7]               #e.g.  1916-01-02 00:00:00+00:00
            pubdate = pubdate.strip()            #e.g.  1916-01
            if pubdate == "0101-01":             #default in table books
                pubdate = "0000-00"
            pubdate = as_unicode(pubdate)


            #-------------------------------
            #-------------------------------
            #-------------------------------
            if do_log:
                tags = tagsconcat[0:75]
                tags = tags.strip()
                published = as_unicode(pubdate)
                published = published[0:7]
                published = published.strip()
                book_dict['AUTHORS'] = authorsconcat
                book_dict['TITLE'] = title
                book_dict['SERIES'] = series
                book_dict['TAGS'] = tags
                book_dict['PUBLISHER'] = publisher
                book_dict['PUBLISHED'] = as_unicode(published)
                book_dict['PATH'] = path
            #-------------------------------
            #-------------------------------
            #-------------------------------
            if not found_text1:
                if type1 == "TEXT":       # Rule [B]: if text1 = TEXT, then text1 is an arbitrary value of the column name1 to search for.   example:  'Isaac Asimov' in the name1 column of "author".
                    if operator1 == "=":
                        if text1 == title and name1 == "title":
                            found_text1 = True
                            text1_found = title
                        if text1 == authorsconcat and name1 == "authors":
                            found_text1 = True
                            text1_found = authorsconcat
                        if text1 == series and name1 == "series":
                            found_text1 = True
                            text1_found = series
                        if text1 == tagsconcat and name1 == "tags":
                            found_text1 = True
                            text1_found = tagsconcat
                        if text1 == publisher and name1 == "publisher":
                            found_text1 = True
                            text1_found = publisher
                        if text1 == comments and name1 == "comments":
                            found_text1 = True
                            text1_found = comments
                        if text1 == pubdate and name1 == "pubdate":
                            found_text1 = True
                            text1_found = pubdate
                        if text1 == path and name1 == "path":
                            found_text1 = True
                            text1_found = path
                    if not found_text1 and operator1 == ">":
                        if text1 > title and name1 == "title":
                            found_text1 = True
                            text1_found = title
                        if text1 > authorsconcat and name1 == "authors":
                            found_text1 = True
                            text1_found = authorsconcat
                        if text1 > series and name1 == "series":
                            found_text1 = True
                            text1_found = series
                        if text1 > tagsconcat and name1 == "tags":
                            found_text1 = True
                            text1_found = tagsconcat
                        if text1 > publisher and name1 == "publisher":
                            found_text1 = True
                            text1_found = publisher
                        if text1 > comments and name1 == "comments":
                            found_text1 = True
                            text1_found = comments
                        if text1 > pubdate and name1 == "pubdate":
                            found_text1 = True
                            text1_found = pubdate
                        if text1 > path and name1 == "path":
                            found_text1 = True
                            text1_found = path
                    if not found_text1 and operator1 == "<":
                        if text1 < title and name1 == "title":
                            found_text1 = True
                            text1_found = title
                        if text1 < authorsconcat and name1 == "authors":
                            found_text1 = True
                            text1_found = authorsconcat
                        if text1 < series and name1 == "series":
                            found_text1 = True
                            text1_found = series
                        if text1 < tagsconcat and name1 == "tags":
                            found_text1 = True
                            text1_found = tagsconcat
                        if text1 < publisher and name1 == "publisher":
                            found_text1 = True
                            text1_found = publisher
                        if text1 < comments and name1 == "comments":
                            found_text1 = True
                            text1_found = comments
                        if text1 < pubdate and name1 == "pubdate":
                            found_text1 = True
                            text1_found = pubdate
                        if text1 < path and name1 == "path":
                            found_text1 = True
                            text1_found = path
                    if not found_text1 and operator1 == ">=":
                        if text1 >= title and name1 == "title":
                            found_text1 = True
                            text1_found = title
                        if text1 >= authorsconcat and name1 == "authors":
                            found_text1 = True
                            text1_found = authorsconcat
                        if text1 >= series and name1 == "series":
                            found_text1 = True
                            text1_found = series
                        if text1 >= tagsconcat and name1 == "tags":
                            found_text1 = True
                            text1_found = tagsconcat
                        if text1 >= publisher and name1 == "publisher":
                            found_text1 = True
                            text1_found = publisher
                        if text1 >= comments and name1 == "comments":
                            found_text1 = True
                            text1_found = comments
                        if text1 >= pubdate and name1 == "pubdate":
                            found_text1 = True
                            text1_found = pubdate
                        if text1 >= path and name1 == "path":
                            found_text1 = True
                            text1_found = path
                    if not found_text1 and operator1 == "<=":
                        if text1 <= title and name1 == "title":
                            found_text1 = True
                            text1_found = title
                        if text1 <= authorsconcat and name1 == "authors":
                            found_text1 = True
                            text1_found = authorsconcat
                        if text1 <= series and name1 == "series":
                            found_text1 = True
                            text1_found = series
                        if text1 <= tagsconcat and name1 == "tags":
                            found_text1 = True
                            text1_found = tagsconcat
                        if text1 <= publisher and name1 == "publisher":
                            found_text1 = True
                            text1_found = publisher
                        if text1 <= comments and name1 == "comments":
                            found_text1 = True
                            text1_found = comments
                        if text1 <= pubdate and name1 == "pubdate":
                            found_text1 = True
                            text1_found = pubdate
                        if text1 <= path and name1 == "path":
                            found_text1 = True
                            text1_found = path
                    if not found_text1 and operator1 == "!=":
                        if text1 != title and name1 == "title":
                            found_text1 = True
                            text1_found = title
                        if text1 != authorsconcat and name1 == "authors":
                            found_text1 = True
                            text1_found = authorsconcat
                        if text1 != series and name1 == "series":
                            found_text1 = True
                            text1_found = series
                        if text1 != tagsconcat and name1 == "tags":
                            found_text1 = True
                            text1_found = tagsconcat
                        if text1 != publisher and name1 == "publisher":
                            found_text1 = True
                            text1_found = publisher
                        if text1 != comments and name1 == "comments":
                            found_text1 = True
                            text1_found = comments
                        if text1 != pubdate and name1 == "pubdate":
                            found_text1 = True
                            text1_found = pubdate
                        if text1 != path and name1 == "path":
                            found_text1 = True
                            text1_found = path
                    if not found_text1 and operator1 == "contains":
                        if title.count(text1) > 0 and name1 == "title":
                            found_text1 = True
                            text1_found = title
                        if authorsconcat.count(text1) > 0 and name1 == "authors":
                            found_text1 = True
                            text1_found = authorsconcat
                        if series.count(text1) > 0 and name1 == "series":
                            found_text1 = True
                            text1_found = series
                        if tagsconcat.count(text1) > 0 and name1 == "tags":
                            found_text1 = True
                            text1_found = tagsconcat
                        if publisher.count(text1) > 0 and name1 == "publisher":
                            found_text1 = True
                            text1_found = publisher
                        if comments.count(text1) > 0 and name1 == "comments":
                            found_text1 = True
                            text1_found = comments
                        if pubdate.count(text1) > 0 and name1 == "pubdate":
                            found_text1 = True
                            text1_found = pubdate
                        if path.count(text1) > 0 and name1 == "path":
                            found_text1 = True
                            text1_found = path
                    if not found_text1 and operator1 == "notcontains":
                        if title.count(text1) == 0 and name1 == "title":
                            found_text1 = True
                            text1_found = title
                        if authorsconcat.count(text1) == 0 and name1 == "authors":
                            found_text1 = True
                            text1_found = authorsconcat
                        if series.count(text1) == 0 and name1 == "series":
                            found_text1 = True
                            text1_found = series
                        if tagsconcat.count(text1) == 0 and name1 == "tags":
                            found_text1 = True
                            text1_found = tagsconcat
                        if publisher.count(text1) == 0 and name1 == "publisher":
                            found_text1 = True
                            text1_found = publisher
                        if comments.count(text1) == 0 and name1 == "comments":
                            found_text1 = True
                            text1_found = comments
                        if pubdate.count(text1) == 0 and name1 == "pubdate":
                            found_text1 = True
                            text1_found = pubdate
                        if path.count(text1) == 0 and name1 == "path":
                            found_text1 = True
                            text1_found = path
                    #-------------------------------------------------------------------------------
                    if not found_text1 and  operator1 == "regex":

                        if name1 == "title":
                            match_found = self.match_regex(name1,text1,regex1,ignorecase1)
                            if match_found:
                                found_text1 = True
                                text1_found = title
                        if name1 == "authors":
                            match_found = self.match_regex(name1,text1,regex1,ignorecase1)
                            if match_found:
                                found_text1 = True
                                text1_found = authorsconcat
                        if name1 == "series":
                            match_found = self.match_regex(name1,text1,regex1,ignorecase1)
                            if match_found:
                                found_text1 = True
                                text1_found = series
                        if name1 == "tags":
                            match_found = self.match_regex(name1,text1,regex1,ignorecase1)
                            if match_found:
                                found_text1 = True
                                text1_found = tagsconcat
                        if name1 == "publisher":
                            match_found = self.match_regex(name1,text1,regex1,ignorecase1)
                            if match_found:
                                found_text1 = True
                                text1_found = publisher
                        if name1 == "comments":
                            match_found = self.match_regex(name1,text1,regex1,ignorecase1)
                            if match_found:
                                found_text1 = True
                                text1_found = comments
                        if name1 == "pubdate":
                            match_found = self.match_regex(name1,text1,regex1,ignorecase1)
                            if match_found:
                                found_text1 = True
                                text1_found = pubdate
                        if name1 == "path":
                            match_found = self.match_regex(name1,text1,regex1,ignorecase1)
                            if match_found:
                                found_text1 = True
                                text1_found = path
                    #-------------------------------------------------------------------------------
                    # Fuzzy Wuzzy Equality - Column 1
                    #-------------------------------------------------------------------------------
                    if self.fuzzytype1 > 0:
                        if (not found_text1) and operator1 == "=":
                            first_value = text1
                            if name1 == "title":
                                second_value = title
                            elif name1 == "authors":
                                second_value = authorsconcat
                            elif name1 == "series":
                                second_value = series
                            elif name1 == "tags":
                                second_value = tagsconcat
                            elif name1 == "publisher":
                                second_value = publisher
                            elif name1 == "comments":
                                second_value = comments
                            elif name1 == "pubdate":
                                second_value = pubdate
                            elif name1 == "path":
                                second_value = path
                            fuzzy_equality = self.fuzzywuzzy_comparison_for_equality(1,first_value,second_value)
                            if fuzzy_equality:
                                found_text1 = True
                                text1_found = second_value
                #END OF Rule [B] logic.

            #-------------------------------
            if not found_text1:
                if type1 == "LABEL":    # Rule [C]: if text1 = LABEL, then text1 is a column name itself, and later its value must be compared to the value for column name1.
                    if text1 == "title":
                        found_text1 = True
                        text1_found = "title"
                    if text1 == "authors":
                        found_text1 = True
                        text1_found = "authors"
                    if text1 == "series":
                        found_text1 = True
                        text1_found = "series"
                    if text1 == "tags":
                        found_text1 = True
                        text1_found = "tags"
                    if text1 == "publisher":
                        found_text1 = True
                        text1_found = "publisher"
                    if text1 == "comments":
                        found_text1 = True
                        text1_found = "comments"
                    if text1 == "comments":
                        found_text1 = True
                        text1_found = "comments"
                    if text1 == "pubdate":
                        found_text1 = True
                        text1_found = "pubdate"
                    if text1 == "path":
                        found_text1 = True
                        text1_found = "path"
                    if found_text1:
                        text1_found_was_label = True
            else:
                pass

            if text1_found_was_label:   # Rule [C]: if text1 = LABEL, then text1 is a column name itself, and now its value must be compared to the value for column name1.
                current_status = "INCOMPLETE"
                #-------------------------------
                # custom columns:  value_dict[label] = value
                # standard columns: book,title,authorsconcat,series,tagsconcat,publisher,comments,pubdate,path = my_std_row1
                #-------------------------------
                if name1_is_custom_column:               #so, must use name1_found as the value to be compared
                    name1_value = name1_found         #name1_found was set during the k,v loop
                if name1 == "title":
                     name1_value = title
                if name1 == "authors":
                     name1_value = authorsconcat
                if name1 == 'series':
                    name1_value = series
                if name1 == 'tags':
                    name1_value = tagsconcat
                if name1 == 'publisher':
                    name1_value = publisher
                if name1 == 'comments':
                    name1_value = comments
                if name1 == 'pubdate':
                    name1_value = pubdate
                if name1 == 'path':
                    name1_value = path

                if '#' in text1:                                # text1 is a column name itself...
                    text1_value = text1_found        #text1_found was set during the k,v loop
                if text1 == "title":
                     text1_value = title
                if text1 == "authors":
                     text1_value = authorsconcat
                if text1 == 'series':
                    text1_value = series
                if text1 == 'tags':
                    text1_value = tagsconcat
                if text1 == 'publisher':
                    text1_value = publisher
                if text1 == 'comments':
                    text1_value = comments
                if text1 == 'pubdate':
                    text1_value = pubdate
                if text1 == 'path':
                    text1_value = path

                if operator1 == "=":
                    if name1_value == text1_value:
                        current_status = "COMPLETE"
                    else:
                        if self.fuzzytype1 > 0:
                            fuzzy_equality = self.fuzzywuzzy_comparison_for_equality(1,name1_value,text1_value)
                            if fuzzy_equality:
                                current_status = "COMPLETE"
                if operator1 == ">":
                    if name1_value > text1_value:
                        current_status = "COMPLETE"
                if operator1 == "<":
                    if name1_value < text1_value:
                        current_status = "COMPLETE"
                if operator1 == ">=":
                    if name1_value >= text1_value:
                        current_status = "COMPLETE"
                if operator1 == "<=":
                    if name1_value <= text1_value:
                        current_status = "COMPLETE"
                if operator1 == "!=":
                    if name1_value != text1_value:
                        current_status = "COMPLETE"
                if operator1 == "contains":
                    if name1_value.count(text1_value) > 0 and text1_value != "":
                        current_status = "COMPLETE"
                if operator1 == "notcontains":
                    if name1_value.count(text1_value) == 0 and text1_value != "":
                        current_status = "COMPLETE"
                if operator1 == "regex":
                    match_found = self.match_regex(name1_value,text1_value,regex1,ignorecase1)
                    if match_found:
                        current_status = "COMPLETE"

                #END OF Rule [C] logic when TEXT1 is a LABEL, not just TEXT

            if not text1_found_was_label:
                current_status = "INCOMPLETE"
                #-------------------------------
                # custom columns:  value_dict[label] = value where value was saved as:  name1_found
                # standard columns: book,title,authorsconcat,series,tagsconcat,publisher,comments,pubdate,path = my_std_row1
                #-------------------------------
                if name1_is_custom_column:               #so, must use name1_found as the value to be compared
                    name1_value = name1_found         #name1_found was set during the k,v loop
                if name1 == "title":
                     name1_value = title
                if name1 == "authors":
                     name1_value = authorsconcat
                if name1 == 'series':
                    name1_value = series
                if name1 == 'tags':
                    name1_value = tagsconcat
                if name1 == 'publisher':
                    name1_value = publisher
                if name1 == 'comments':
                    name1_value = comments
                if name1 == 'pubdate':
                    name1_value = pubdate
                if name1 == 'path':
                    name1_value = path

                text1_value = text1   #whatever was passed directly from the dialog with the user

                if operator1 == "=":
                    if name1_value == text1_value:
                        current_status = "COMPLETE"
                    else:
                        if self.fuzzytype1 > 0:
                            fuzzy_equality = self.fuzzywuzzy_comparison_for_equality(1,name1_value,text1_value)
                            if fuzzy_equality:
                                current_status = "COMPLETE"
                if operator1 == ">":
                    if name1_value > text1_value:
                        current_status = "COMPLETE"
                if operator1 == "<":
                    if name1_value < text1_value:
                        current_status = "COMPLETE"
                if operator1 == ">=":
                    if name1_value >= text1_value:
                        current_status = "COMPLETE"
                if operator1 == "<=":
                    if name1_value <= text1_value:
                        current_status = "COMPLETE"
                if operator1 == "!=":
                    if name1_value != text1_value:
                        current_status = "COMPLETE"
                if operator1 == "contains":
                    if name1_value.count(text1_value) > 0 and text1_value != "":
                        current_status = "COMPLETE"
                if operator1 == "notcontains":
                    if name1_value.count(text1_value) == 0 and text1_value != "":
                        current_status = "COMPLETE"
                if operator1 == "regex":
                    match_found = self.match_regex(name1_value,text1_value,regex1,ignorecase1)
                    if match_found:
                        current_status = "COMPLETE"

                #END OF Rule [C] logic when TEXT1 is just TEXT, not a LABEL
            #-------------------------------
            #-------------------------------
            #-------------------------------
            if not current_status == "COMPLETE":
                if as_unicode(andornotignore) != as_unicode("OR"):
                    was_found = False
                    return was_found
            #-------------------------------

            if current_status == "COMPLETE" and as_unicode(andornotignore) == as_unicode("IGNORE"):
                was_found = True
                search_log.append(book_dict)
                return was_found

            if current_status == "COMPLETE" :
                column1_is_complete = True

            #-------------------------------
            found_name2 = False
            found_text2 = False
            found_title = False
            found_author = False
            found_series = False
            found_tag = False
            found_publisher = False
            found_comments = False
            found_pubdate = False
            found_path = False

            name2_found = ""
            text2_found = ""
            name2_value = ""
            value = ""

            text2_found_was_label = False
            current_status = "INCOMPLETE"
            column2_is_complete = False
            name2_is_custom_column = False
            #-------------------------------

            self.current_column_fuzzywuzzy = 2

            #---------------------------------------------------------------------------------------------------------------------------------------
            # First, Custom Columns for Column 2:
            #---------------------------------------------------------------------------------------------------------------------------------------
            #~ for k,v in value_dict.iteritems():  #custom column (only) metadata for book
            for k,v in iteritems(value_dict):  #custom column (only) metadata for book
                label = k    #e.g. #work_author
                value = v   #e.g.  Isaac Asimov   OR  #work_title
                if name2 == label:  # Rule [A] name2 is ALWAYS a column name (either custom or standard)
                    found_name2 = True  #e.g. #work_author
                    #name2_found for custom columns is the column value (contents), not the column label.  different than for standard columns.
                    if not label in cust_columns_integer_list:
                        if not label in cust_columns_float_list:
                            value = self.apply_transform_functions_single_value(2,value)
                    name2_found = value  #e.g. Joe Blow (Joe Blow was the #work_author for this book)
                    name2_is_custom_column = True  #so, must use name2_found as the value to be compared
                    book_dict[label] = value
                if type2 == "TEXT": # Rule [B] if text2 = TEXT, then text2 is an arbitrary value of the column name2 to search for.
                    if label in cust_columns_integer_list  and label != "":
                        if text2 == "":
                            text2 = 0
                        text2_found = int(text2)
                        found_text2 = True
                        text2 = int(text2)
                        book_dict[label] = text2
                    else:
                        if label in cust_columns_float_list  and label != "":
                            if text2 == "":
                                text2 = 0.00
                            text2_found = float(text2)
                            found_text2 = True
                            text2 = float(text2)
                            book_dict[label] = text2
                        else:
                            if label in cust_columns_float_list  and label != "":
                                if text2 == "":
                                    text2 = 0.00
                                text2_found = float(text2)
                                found_text2 = True
                                text2 = float(text2)
                                book_dict[label] = text2
                            else:
                                value = self.apply_transform_functions_single_value(2,value)
                                text2_found = value
                                found_text2 = True
                                book_dict[label] = value

                if type2 == "LABEL": # Rule [C] if text2 = LABEL, then text2 is a column name itself, and its value must be compared to the value for column name2.
                    if text2 == label:
                        value = self.apply_transform_functions_single_value(2,value)
                        found_text2 = True
                        text2_found = value
                        text2_found_was_label = True
                        book_dict[label] = value

            #-----------------------------------------------
            # Second, Standard Columns for Column 2:
            #-----------------------------------------------
            if not found_name2:
                if name2 == "title":
                    found_name2 = True
                    name2_found = "title"
                if name2 == "authors":
                    found_name2 = True
                    name2_found = "authors"
                if name2 == "series":
                    found_name2 = True
                    name2_found = "series"
                if name2 == "tags":
                    found_name2 = True
                    name2_found = "tags"
                if name2 == "publisher":
                    found_name2 = True
                    name2_found = "publisher"
                if name2 == "comments":
                    found_name2 = True
                    name2_found = "comments"
                if name2 == "pubdate":
                    found_name2 = True
                    name2_found = "pubdate"
                if name2 == "path":
                    found_name2 = True
                    name2_found = "path"

            #-------------------------------
            if not found_name2:  #must continue due to final AND, OR, NOT logic at the end
                current_status = "INCOMPLETE"
                column2_is_complete = False


            #-------------------------------
            # Rule [A] has been totally satisfied as of this point.
            #-------------------------------
            #-------------------------------
            column_source = 2
            my_std_row2 = self.apply_transform_functions_standard_columns(column_source,my_std_row2)
            book,title,authorsconcat,series,tagsconcat,publisher,comments,pubdate,path = my_std_row2
            #~ if DEBUG: print("[column source 2] >>> book,title,authorsall,series,tagsall", as_unicode(book), title, authorsall, series, tagsall)
            #-------------------------------
            if not title:
                title = ""
            if not authorsconcat:
                authorsconcat = ""
            if not series:
                series = ""
            if not tagsconcat:
                tagsconcat = ""
            if not publisher:
                publisher = ""
            if not comments:
                comments = ""
            if not pubdate:
                pubdate = "0101-01-01 00:00:00+00:00"   #default in table books
            if not path:
                path = ""
            #-------------------------------
            if not found_text2:
                if type2 == "TEXT":       # Rule [B]: if text2 = TEXT, then text2 is an arbitrary value of the column name2 to search for.   example:  'Isaac Asimov' in the name2 column of "author".
                    if operator2 == "=":
                        if text2 == title and name2 == "title":
                            found_text2 = True
                            text2_found = title
                        if text2 == authorsconcat and name2 == "authors":
                            found_text2 = True
                            text2_found = authorsconcat
                        if text2 == series and name2 == "series":
                            found_text2 = True
                            text2_found = series
                        if text2 == tagsconcat and name2 == "tags":
                            found_text2 = True
                            text2_found = tagsconcat
                        if text2 == publisher and name2 == "publisher":
                            found_text2 = True
                            text2_found = publisher
                        if text2 == comments and name2 == "comments":
                            found_text2 = True
                            text2_found = comments
                        if text2 == pubdate and name2 == "pubdate":
                            found_text2 = True
                            text2_found = pubdate
                        if text2 == path and name2 == "path":
                            found_text2 = True
                            text2_found = path
                    if not found_text2 and operator2 == ">":
                        if text2 > title and name2 == "title":
                            found_text2 = True
                            text2_found = title
                        if text2 > authorsconcat and name2 == "authors":
                            found_text2 = True
                            text2_found = authorsconcat
                        if text2 > series and name2 == "series":
                            found_text2 = True
                            text2_found = series
                        if text2 > tagsconcat and name2 == "tags":
                            found_text2 = True
                            text2_found = tagsconcat
                        if text2 > publisher and name2 == "publisher":
                            found_text2 = True
                            text2_found = publisher
                        if text2 > comments and name2 == "comments":
                            found_text2 = True
                            text2_found = comments
                        if text2 > pubdate and name2 == "pubdate":
                            found_text2 = True
                            text2_found = pubdate
                        if text2 > path and name2 == "path":
                            found_text2 = True
                            text2_found = path
                    if not found_text2 and operator2 == "<":
                        if text2 < title and name2 == "title":
                            found_text2 = True
                            text2_found = title
                        if text2 < authorsconcat and name2 == "authors":
                            found_text2 = True
                            text2_found = authorsconcat
                        if text2 < series and name2 == "series":
                            found_text2 = True
                            text2_found = series
                        if text2 < tagsconcat and name2 == "tags":
                            found_text2 = True
                            text2_found = tagsconcat
                        if text2 < publisher and name2 == "publisher":
                            found_text2 = True
                            text2_found = publisher
                        if text2 < comments and name2 == "comments":
                            found_text2 = True
                            text2_found = comments
                        if text2 < pubdate and name2 == "pubdate":
                            found_text2 = True
                            text2_found = pubdate
                        if text2 < path and name2 == "path":
                            found_text2 = True
                            text2_found = path
                    if not found_text2 and operator2 == ">=":
                        if text2 >= title and name2 == "title":
                            found_text2 = True
                            text2_found = title
                        if text2 >= authorsconcat and name2 == "authors":
                            found_text2 = True
                            text2_found = authorsconcat
                        if text2 >= series and name2 == "series":
                            found_text2 = True
                            text2_found = series
                        if text2 >= tagsconcat and name2 == "tags":
                            found_text2 = True
                            text2_found = tagsconcat
                        if text2 >= publisher and name2 == "publisher":
                            found_text2 = True
                            text2_found = publisher
                        if text2 >= comments and name2 == "comments":
                            found_text2 = True
                            text2_found = comments
                        if text2 >= pubdate and name2 == "pubdate":
                            found_text2 = True
                            text2_found = pubdate
                        if text2 >= path and name2 == "path":
                            found_text2 = True
                            text2_found = path
                    if not found_text2 and operator2 == "<=":
                        if text2 <= title and name2 == "title":
                            found_text2 = True
                            text2_found = title
                        if text2 <= authorsconcat and name2 == "authors":
                            found_text2 = True
                            text2_found = authorsconcat
                        if text2 <= series and name2 == "series":
                            found_text2 = True
                            text2_found = series
                        if text2 <= tagsconcat and name2 == "tags":
                            found_text2 = True
                            text2_found = tagsconcat
                        if text2 <= publisher and name2 == "publisher":
                            found_text2 = True
                            text2_found = publisher
                        if text2 <= comments and name2 == "comments":
                            found_text2 = True
                            text2_found = comments
                        if text2 <= pubdate and name2 == "pubdate":
                            found_text2 = True
                            text2_found = pubdate
                        if text2 <= path and name2 == "path":
                            found_text2 = True
                            text2_found = path
                    if not found_text2 and operator2 == "!=":
                        if text2 != title and name2 == "title":
                            found_text2 = True
                            text2_found = title
                        if text2 != authorsconcat and name2 == "authors":
                            found_text2 = True
                            text2_found = authorsconcat
                        if text2 != series and name2 == "series":
                            found_text2 = True
                            text2_found = series
                        if text2 != tagsconcat and name2 == "tags":
                            found_text2 = True
                            text2_found = tagsconcat
                        if text2 != publisher and name2 == "publisher":
                            found_text2 = True
                            text2_found = publisher
                        if text2 != comments and name2 == "comments":
                            found_text2 = True
                            text2_found = comments
                        if text2 != pubdate and name2 == "pubdate":
                            found_text2 = True
                            text2_found = pubdate
                        if text2 != path and name2 == "path":
                            found_text2 = True
                            text2_found = path
                    if not found_text2 and operator2 == "contains":
                        if title.count(text2) > 0 and name2 == "title":
                            found_text2 = True
                            text2_found = title
                        if authorsconcat.count(text2) > 0 and name2 == "authors":
                            found_text2 = True
                            text2_found = authorsconcat
                        if series.count(text2) > 0 and name2 == "series":
                            found_text2 = True
                            text2_found = series
                        if tagsconcat.count(text2) > 0 and name2 == "tags":
                            found_text2 = True
                            text2_found = tagsconcat
                        if publisher.count(text2) > 0 and name2 == "publisher":
                            found_text2 = True
                            text2_found = publisher
                        if comments.count(text2) > 0 and name2 == "comments":
                            found_text2 = True
                            text2_found = comments
                        if pubdate.count(text2) > 0 and name2 == "pubdate":
                            found_text2 = True
                            text2_found = pubdate
                        if path.count(text2) > 0 and name2 == "path":
                            found_text2 = True
                            text2_found = path
                    if not found_text2 and operator2 == "notcontains":
                        if title.count(text2) == 0 and name2 == "title":
                            found_text2 = True
                            text2_found = title
                        if authorsconcat.count(text2) == 0 and name2 == "authors":
                            found_text2 = True
                            text2_found = authorsconcat
                        if series.count(text2) == 0 and name2 == "series":
                            found_text2 = True
                            text2_found = series
                        if tagsconcat.count(text2) == 0 and name2 == "tags":
                            found_text2 = True
                            text2_found = tagsconcat
                        if publisher.count(text2) == 0 and name2 == "publisher":
                            found_text2 = True
                            text2_found = publisher
                        if comments.count(text2) == 0 and name2 == "comments":
                            found_text2 = True
                            text2_found = comments
                        if pubdate.count(text2) == 0 and name2 == "pubdate":
                            found_text2 = True
                            text2_found = pubdate
                        if path.count(text2) == 0 and name2 == "path":
                            found_text2 = True
                            text2_found = path
                   #-------------------------------------------------------------------------------
                    if not found_text2 and  operator2 == "regex":
                        if name2 == "title":
                            match_found = self.match_regex(name2,text2,regex2,ignorecase2)
                            if match_found:
                                found_text2 = True
                                text2_found = title
                        if name2 == "authors":
                            match_found = self.match_regex(name2,text2,regex2,ignorecase2)
                            if match_found:
                                found_text2 = True
                                text2_found = authorsconcat
                        if name2 == "series":
                            match_found = self.match_regex(name2,text2,regex2,ignorecase2)
                            if match_found:
                                found_text2 = True
                                text2_found = series
                        if name2 == "tags":
                            match_found = self.match_regex(name2,text2,regex2,ignorecase2)
                            if match_found:
                                found_text2 = True
                                text2_found = tagsconcat
                        if name2 == "publisher":
                            match_found = self.match_regex(name2,text2,regex2,ignorecase2)
                            if match_found:
                                found_text2 = True
                                text2_found = publisher
                        if name2 == "comments":
                            match_found = self.match_regex(name2,text2,regex2,ignorecase2)
                            if match_found:
                                found_text2 = True
                                text2_found = comments
                        if name2 == "pubdate":
                            match_found = self.match_regex(name2,text2,regex2,ignorecase2)
                            if match_found:
                                found_text2 = True
                                text2_found = pubdate
                        if name2 == "path":
                            match_found = self.match_regex(name2,text2,regex2,ignorecase2)
                            if match_found:
                                found_text2 = True
                                text2_found = path
                    #-------------------------------------------------------------------------------
                    # Fuzzy Wuzzy Equality - Column 2
                    #-------------------------------------------------------------------------------
                    if self.fuzzytype2 > 0:
                        if not found_text2 and operator2 == "=":
                            first_value = text2
                            if name2 == "title":
                                second_value = title
                            elif name2 == "authors":
                                second_value = authorsconcat
                            elif name2 == "series":
                                second_value = series
                            elif name2 == "tags":
                                second_value = tagsconcat
                            elif name2 == "publisher":
                                second_value = publisher
                            elif name2 == "comments":
                                second_value = comments
                            elif name2 == "pubdate":
                                second_value = pubdate
                            elif name2 == "path":
                                second_value = path
                            fuzzy_equality = self.fuzzywuzzy_comparison_for_equality(2,first_value,second_value)
                            if fuzzy_equality:
                                found_text2 = True
                                text2_found = second_value
                        #-------------------------------------------------------------------------------
                #END OF Rule [B] logic.

            if not found_text2:
                if type2 == "LABEL":    # Rule [C]: if text2 = LABEL, then text2 is a column name itself, and later its value must be compared to the value for column name2.
                    if text2 == "title":
                        found_text2 = True
                        text2_found = "title"
                    if text2 == "authors":
                        found_text2 = True
                        text2_found = "authors"
                    if text2 == "series":
                        found_text2 = True
                        text2_found = "series"
                    if text2 == "tags":
                        found_text2 = True
                        text2_found = "tags"
                    if text2 == "publisher":
                        found_text2 = True
                        text2_found = "publisher"
                    if text2 == "comments":
                        found_text2 = True
                        text2_found = "comments"
                    if text2 == "pubdate":
                        found_text2 = True
                        text2_found = "pubdate"
                    if text2 == "path":
                        found_text2 = True
                        text2_found = "path"
                    if found_text2:
                        text2_found_was_label = True
            else:
                pass


            if text2_found_was_label:   # Rule [C]: if text2 = LABEL, then text2 is a column name itself, and now its value must be compared to the value for column name2.
                current_status = "INCOMPLETE"
                #-------------------------------
                # custom columns:  value_dict[label] = value
                # standard columns: book,title,authorsconcat,series,tagsconcat,publisher,comments,pubdate,path = my_std_row2
                #-------------------------------
                if name2_is_custom_column:               #so, must use name2_found as the value to be compared
                    name2_value = name2_found         #name2_found was set during the k,v loop
                if name2 == "title":
                     name2_value = title
                if name2 == "authors":
                     name2_value = authorsconcat
                if name2 == 'series':
                    name2_value = series
                if name2 == 'tags':
                    name2_value = tagsconcat
                if name2 == 'publisher':
                    name2_value = publisher
                if name2 == 'comments':
                    name2_value = comments
                if name2 == 'pubdate':
                    name2_value = pubdate
                if name2 == 'path':
                    name2_value = path

                if '#' in text2:                            # text2 is a column name itself...
                    text2_value = text2_found    #text2_found was set during the k,v loop
                if text2 == "title":
                     text2_value = title
                if text2 == "authors":
                     text2_value = authorsconcat
                if text2 == 'series':
                    text2_value = series
                if text2 == 'tags':
                    text2_value = tagsconcat
                if text2 == 'publisher':
                    text2_value = publisher
                if text2 == 'comments':
                    text2_value = comments
                if text2 == 'pubdate':
                    text2_value = pubdate
                if text2 == 'path':
                    text2_value = path

                if operator2 == "=":
                    if name2_value == text2_value:
                        current_status = "COMPLETE"
                    else:
                        if self.fuzzytype2 > 0:
                            fuzzy_equality = self.fuzzywuzzy_comparison_for_equality(2,name2_value,text2_value)
                            if fuzzy_equality:
                                current_status = "COMPLETE"
                if operator2 == ">":
                    if name2_value > text2_value:
                        current_status = "COMPLETE"
                if operator2 == "<":
                    if name2_value < text2_value:
                        current_status = "COMPLETE"
                if operator2 == ">=":
                    if name2_value >= text2_value:
                        current_status = "COMPLETE"
                if operator2 == "<=":
                    if name2_value <= text2_value:
                        current_status = "COMPLETE"
                if operator2 == "!=":
                    if name2_value != text2_value:
                        current_status = "COMPLETE"
                if operator2 == "contains":
                    if name2_value.count(text2_value) > 0 and text2_value != "":
                        current_status = "COMPLETE"
                if operator2 == "notcontains":
                    if name2_value.count(text2_value) == 0 and text2_value != "":
                        current_status = "COMPLETE"
                if operator2 == "regex":
                    match_found = self.match_regex(name2_value,text2_value,regex2,ignorecase2)
                    if match_found:
                        current_status = "COMPLETE"
                #END OF Rule [C] logic when TEXT2 is a LABEL, not just TEXT


            if not text2_found_was_label:
                current_status = "INCOMPLETE"
                #now compare the literal TEXT2 to NAME2's value
                #-------------------------------
                # custom columns:  value_dict[label] = value
                # standard columns: book,title,authorsconcat,series,tagsconcat,publisher,comments,pubdate,path = my_std_row2
                #-------------------------------
                if name2_is_custom_column:               #so, must use name2_found as the value to be compared
                    name2_value = name2_found         #name2_found was set during the k,v loop
                if name2 == "title":
                     name2_value = title
                if name2 == "authors":
                     name2_value = authorsconcat
                if name2 == 'series':
                    name2_value = series
                if name2 == 'tags':
                    name2_value = tagsconcat
                if name2 == 'publisher':
                    name2_value = publisher
                if name2 == 'comments':
                    name2_value = comments
                if name2 == 'pubdate':
                    name2_value = pubdate
                if name2 == 'path':
                    name2_value = path

                text2_value = text2   #whatever was passed directly from the dialog with the user

                if operator2 == "=":
                    if name2_value == text2_value:
                        current_status = "COMPLETE"
                    else:
                        if self.fuzzytype2 > 0:
                            fuzzy_equality = self.fuzzywuzzy_comparison_for_equality(2,name2_value,text2_value)
                            if fuzzy_equality:
                                current_status = "COMPLETE"
                if operator2 == ">":
                    if name2_value > text2_value:
                        current_status = "COMPLETE"
                if operator2 == "<":
                    if name2_value < text2_value:
                        current_status = "COMPLETE"
                if operator2 == ">=":
                    if name2_value >= text2_value:
                        current_status = "COMPLETE"
                if operator2 == "<=":
                    if name2_value <= text2_value:
                        current_status = "COMPLETE"
                if operator2 == "!=":
                    if name2_value != text2_value:
                        current_status = "COMPLETE"
                if operator2 == "contains":
                    if name2_value.count(text2_value) > 0 and text2_value != "":
                        current_status = "COMPLETE"
                if operator2 == "notcontains":
                    if name2_value.count(text2_value) == 0 and text2_value != "":
                        current_status = "COMPLETE"
                if operator2 == "regex":
                    match_found = self.match_regex(name2_value,text2_value,regex2,ignorecase2)
                    if match_found:
                        current_status = "COMPLETE"
                #END OF Rule [C] logic when TEXT2 is just TEXT, not a LABEL

            #-------------------------------
            #-------------------------------
            #-------------------------------
            if current_status == "COMPLETE" :
                column2_is_complete = True
            else:
                column2_is_complete = False

            #at end:  compare column 1 results to column 2 results using andornotignore, and return boolean answer

            if column1_is_complete and column2_is_complete and as_unicode(andornotignore) == as_unicode("AND"):
                was_found = True
                search_log.append(book_dict)

                return was_found

            if (column1_is_complete or column2_is_complete) and as_unicode(andornotignore) == as_unicode("OR"):
                was_found = True
                search_log.append(book_dict)

                return was_found

            if column1_is_complete:
                if as_unicode(andornotignore) == as_unicode("NOT"):
                    if column2_is_complete:
                        was_found = False
                        return was_found
                    else:
                        was_found = True
                        search_log.append(book_dict)

                        return was_found

            if column1_is_complete and as_unicode(andornotignore) == as_unicode("IGNORE"):
                was_found = True
                search_log.append(book_dict)

                return was_found
            #-------------------------------

            was_found = False
            return was_found

        except Exception as e:
            if DEBUG: print("Search Criteria Incompatibility (e.g. Expected Integer but got Boolean):", as_unicode(e))
            if DEBUG: print("was_found was set to False due to error; returning")
            was_found = False
            return was_found
    #-----------------------------------------------------------------------------------------
    def apply_transform_functions_standard_columns(self,column_source,my_std_row):
        #~ Type 1:  COMPARE_AS_IS = unicode_type("Compare: 'as-is'")
        #~ Type 2:  COMPARE_UPPER_CASE = unicode_type("Compare: Upper Case")
        #~ Type 3:  COMPARE_NO_SPACES_NO_PUNCTUATION = unicode_type("Compare: No Spaces; No Punctuation")
        #~ Type 4:  COMPARE_UPPER_CASE_NO_SPACES_NO_PUNCTUATION = unicode_type("Compare: Upper Case; No Spaces; No Punctuation")
        #~ Type 5:  COMPARE_DECOMPOSED_NORMALIZED_ALPHABET = unicode_type('Compare as: Decomposed & Normalized Alphabet')

        if not my_std_row:   # None
            if DEBUG: print("ERROR: my_std_row is None for column source: ", as_unicode(column_source))
            return my_std_row

        if column_source == 1:
            if self.transform_1_type == 1:
                return my_std_row
            elif self.transform_1_type == 2:
                my_std_row = self.transform_standard_type_2(my_std_row)
                return my_std_row
            elif self.transform_1_type == 3:
                my_std_row = self.transform_standard_type_3(my_std_row)
                return my_std_row
            elif self.transform_1_type == 4:
                my_std_row = self.transform_standard_type_4(my_std_row)
                return my_std_row
            elif self.transform_1_type == 5:
                my_std_row = self.transform_standard_type_5(my_std_row)
                return my_std_row
            else:
                return my_std_row
        else:
            if self.transform_2_type == 1:
                return my_std_row
            elif self.transform_2_type == 2:
                my_std_row = self.transform_standard_type_2(my_std_row)
                return my_std_row
            elif self.transform_2_type == 3:
                my_std_row = self.transform_standard_type_3(my_std_row)
                return my_std_row
            elif self.transform_2_type == 4:
                my_std_row = self.transform_standard_type_4(my_std_row)
                return my_std_row
            elif self.transform_2_type == 5:
                my_std_row = self.transform_standard_type_5(my_std_row)
                return my_std_row
            else:
                return my_std_row
    #-----------------------------------------------------------------------------------------
    def transform_standard_type_2(self,my_std_row):
        #~ convert to UPPERCASE
        #~ book,title,authorsconcat,series,tagsconcat,publisher,comments,pubdate,path = my_std_row
        my_std_row = list(my_std_row)
        new = []
        i = 0
        for column in my_std_row:
            if i != 0 and i != 7:  #book is an integer, not a string; pubdate is a date
                if column:  # not None
                    column = column.upper()
            new.append(column)
            #~ if i != 6:  #comments...
                #~ if DEBUG: print("standard type_2",as_unicode(column))
            i = i + 1
        #END FOR
        del my_std_row
        my_std_row = new[0],new[1],new[2],new[3],new[4],new[5],new[6],new[7],new[8]
        del new
        return my_std_row
    #-----------------------------------------------------------------------------------------
    def transform_standard_type_3(self,my_std_row):
        #~ convert to NO SPACES AND NO PUNCTUATION
        #~ book,title,authorsconcat,series,tagsconcat,publisher,comments,pubdate,path = my_std_row
        my_std_row = list(my_std_row)
        new = []
        i = 0
        for column in my_std_row:
            if i != 0 and i != 7:  #book is an integer, not a string; pubdate is a date
                if column:  # not None
                    column = column.replace(" ","")
                    column = self.remove_punctuation(column)
            new.append(column)
            #~ if i != 6:  #comments..
                #~ if DEBUG: print("standard type_3: ", as_unicode(column))
            i = i + 1
        #END FOR
        del my_std_row
        my_std_row = new[0],new[1],new[2],new[3],new[4],new[5],new[6],new[7],new[8]
        del new
        return my_std_row
    #-----------------------------------------------------------------------------------------
    def transform_standard_type_4(self,my_std_row):
        #~ Type 4:  COMPARE_UPPER_CASE_NO_SPACES_NO_PUNCTUATION = unicode_type("Compare: Upper Case; No Spaces; No Punctuation")
        my_std_row = self.transform_standard_type_2(my_std_row)
        my_std_row = self.transform_standard_type_3(my_std_row)
        return my_std_row
    #-----------------------------------------------------------------------------------------
    def transform_standard_type_5(self,my_std_row):
        #~ Type 5:  COMPARE_DECOMPOSED_NORMALIZED_ALPHABET = unicode_type('Compare as: Decomposed & Normalized Alphabet')
        my_std_row = self.transform_standard_convert_to_best_ascii_equivalents(my_std_row)
        if self.current_column_fuzzywuzzy == 0:   # inter-book searches only
            single_value = self.transform_single_value_type_4(single_value)
            return single_value
        if self.fuzzytype1 == 0 and self.current_column_fuzzywuzzy == 1:
            my_std_row = self.transform_standard_type_4(my_std_row)
            return my_std_row
        elif self.fuzzytype2 == 0 and self.current_column_fuzzywuzzy == 2:
            my_std_row = self.transform_standard_type_4(my_std_row)
            return my_std_row
        else:
            my_std_row = self.transform_standard_type_2(my_std_row)  # upper case only; spaces required for fuzzywuzzy.
            return my_std_row
    #-----------------------------------------------------------------------------------------
    def transform_standard_convert_to_best_ascii_equivalents(self,my_std_row):
        #~ https://docs.python.org/2/library/unicodedata.html
        #~ Even if two unicode strings are normalized and look the same to a human reader, if one has combining characters and the other doesn’t, they may not compare equal.
        #~ book,title,authorsconcat,series,tagsconcat,publisher,comments,pubdate,path = my_std_row
        my_std_row = list(my_std_row)
        new = []
        i = 0
        for column in my_std_row:
            if i != 0 and i != 7:  #book is an integer, not a string; pubdate is a date
                if column:  # not None
                    column = self.transform_single_value_convert_to_best_ascii_equivalents(column)
            new.append(column)
            i = i + 1
        #END FOR
        del my_std_row
        my_std_row = new[0],new[1],new[2],new[3],new[4],new[5],new[6],new[7],new[8]
        del new
        return my_std_row
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def apply_transform_functions_single_value(self,column_source,single_value):
        #~ Type 1:  COMPARE_AS_IS = unicode_type("Compare: 'as-is'")
        #~ Type 2:  COMPARE_UPPER_CASE = unicode_type("Compare: Upper Case")
        #~ Type 3:  COMPARE_NO_SPACES_NO_PUNCTUATION = unicode_type("Compare: No Spaces; No Punctuation")
        #~ Type 4:  COMPARE_UPPER_CASE_NO_SPACES_NO_PUNCTUATION = unicode_type("Compare: Upper Case; No Spaces; No Punctuation")
        #~ Type 5:  COMPARE_DECOMPOSED_NORMALIZED_ALPHABET = unicode_type('Compare as: Decomposed & Normalized Alphabet')

        if not single_value:   # None
            return single_value

        if column_source == 1:
            if self.transform_1_type == 1:
                return single_value
            elif self.transform_1_type == 2:
                single_value = self.transform_single_value_type_2(single_value)
                return single_value
            elif self.transform_1_type == 3:
                single_value = self.transform_single_value_type_3(single_value)
                return single_value
            elif self.transform_1_type == 4:
                single_value = self.transform_single_value_type_4(single_value)
                return single_value
            elif self.transform_1_type == 5:
                single_value = self.transform_single_value_type_5(single_value)
                return single_value
            else:
                return single_value
        else:
            if self.transform_2_type == 1:
                return single_value
            elif self.transform_2_type == 2:
                single_value = self.transform_single_value_type_2(single_value)
                return single_value
            elif self.transform_2_type == 3:
                single_value = self.transform_single_value_type_3(single_value)
                return single_value
            elif self.transform_2_type == 4:
                single_value = self.transform_single_value_type_4(single_value)
                return single_value
            elif self.transform_2_type == 5:
                single_value = self.transform_single_value_type_5(single_value)
                return single_value
            else:
                return single_value
    #-----------------------------------------------------------------------------------------
    def transform_single_value_type_2(self,single_value):
        #~ convert to UPPERCASE
        #~ if integer etc., nothing will be done.
        try:
            single_value = single_value.upper()
            #~ if DEBUG: print("single_value type_2: ", single_value)
        except Exception as e:
            #~ if DEBUG: print("single_value type_2 exception: ", as_unicode(e))
            pass
        return single_value
    #-----------------------------------------------------------------------------------------
    def transform_single_value_type_3(self,single_value):
        #~ convert to NO SPACES AND NO PUNCTUATION
        #~ if integer etc., nothing will be done.
        try:
            single_value = single_value.replace(" ","")
            single_value = self.remove_punctuation(single_value)
            #~ if DEBUG: print("single_value type_3: ", single_value)
        except Exception as e:
            #~ if DEBUG: print("single_value type_3 exception: ", as_unicode(e))
            pass

        return single_value
    #-----------------------------------------------------------------------------------------
    def transform_single_value_type_4(self,single_value):
        #~ Type 4:  COMPARE_UPPER_CASE_NO_SPACES_NO_PUNCTUATION = unicode_type("Compare: Upper Case; No Spaces; No Punctuation")
        single_value = self.transform_single_value_type_2(single_value)
        single_value = self.transform_single_value_type_3(single_value)
        return single_value
    #-----------------------------------------------------------------------------------------
    def transform_single_value_type_5(self,single_value):
        #~ Type 5:  COMPARE_DECOMPOSED_NORMALIZED_ALPHABET = unicode_type('Compare as: Decomposed & Normalized Alphabet')
        single_value = self.transform_single_value_convert_to_best_ascii_equivalents(single_value)
        if self.current_column_fuzzywuzzy == 0:   # inter-book searches only
            single_value = self.transform_single_value_type_4(single_value)
            return single_value
        elif self.fuzzytype1 == 0 and self.current_column_fuzzywuzzy == 1:
            single_value = self.transform_single_value_type_4(single_value)
            return single_value
        elif self.fuzzytype2 == 0 and self.current_column_fuzzywuzzy == 2:
            single_value = self.transform_single_value_type_4(single_value)
            return single_value
        else:
            single_value = self.transform_single_value_type_2(single_value)  # upper case only; spaces required for fuzzywuzzy.
            return single_value
    #-----------------------------------------------------------------------------------------
    def transform_single_value_convert_to_best_ascii_equivalents(self,single_value):

        if isinstance(single_value,str):
            try:
                single_value = single_value.decode('utf-8')
            except:
                return single_value

        try:
            single_value = unidecode(single_value)
        except:
            pass

        return single_value
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def fuzzywuzzy_comparison_for_equality(self,column,first_value,second_value):
        # Important:  fuzz will return a < 100 for sort_ratio and set_ratio if the spaces are removed from the values, because it cannot then split the string into a list of words to do its comparisons...
        # Important:  remember that Cross-Library searches do NOT use transform functions,
        #                   so doing two identical transform & fuzzy searches for the identical library, once while in it and once while not in it, could easily cause differences in the search results having non-ascii letters for that reason alone!
        first_value = unicode_type(first_value)
        second_value = unicode_type(second_value)
        n = 0
        #~ if DEBUG: print("=====================fuzzywuzzy_comparison_for_equality: ", as_unicode(column), first_value, second_value)
        if column == 1:
            if self.fuzzytype1 == 1:
                n = fuzz.token_sort_ratio(first_value,second_value)
                if n == 100:
                    #~ if DEBUG: print("fuzzywuzzy sort_ratio equality: ", first_value, "  ---  ", second_value, ">>>>", as_unicode(n))
                    return True
            elif self.fuzzytype1 == 2:
                n = fuzz.token_set_ratio(first_value,second_value)
                if n == 100:
                    #~ if DEBUG: print("fuzzywuzzy set_ratio equality: ", first_value, "  ---  ", second_value, ">>>>", as_unicode(n))
                    return True
            elif self.fuzzytype1 == 3:
                first_value = self.remove_punctuation(first_value)
                second_value = self.remove_punctuation(second_value)
                n = fuzz.ratio(first_value,second_value)
                if n == 100:
                    #~ if DEBUG: print("fuzzywuzzy simple ratio equality: ", first_value, "  ---  ", second_value, ">>>>", as_unicode(n))
                    return True
            elif self.fuzzytype1 == 4:
                first_value = self.remove_punctuation(first_value)
                second_value = self.remove_punctuation(second_value)
                first_value = self.fuzzywuzzy_convert_to_best_ascii_equivalents(first_value)
                second_value = self.fuzzywuzzy_convert_to_best_ascii_equivalents(second_value)
                n = fuzz.ratio(first_value,second_value)
                if n == 100:
                    #~ if DEBUG: print("fuzzywuzzy decomposed & normalized alphabet equality: ", first_value, "  ---  ", second_value, ">>>>", as_unicode(n))
                    return True
            elif self.fuzzytype1 == 5:
                #~ if DEBUG: print("calling doublemetaphone next...")
                first_result = MCSDoubleMetaphone(first_value)
                #~ if DEBUG: print("first_result: ", as_unicode(first_result))
                second_result = MCSDoubleMetaphone(second_value)
                #~ if DEBUG: print("second_result: ", as_unicode(second_result))
                n = fuzz.ratio(as_unicode(first_result),as_unicode(second_result))
                if n == 100:
                    #~ if DEBUG: print("Column 1: DoubleMetaphone Sounds-Like Match 100%: ", first_value, "  ---  ", second_value, ">>>>", as_unicode(n))
                    return True
        else:
            if column == 2:
                if self.fuzzytype2 == 1:
                    n = fuzz.token_sort_ratio(first_value,second_value)
                    if n == 100:
                        #~ if DEBUG: print("fuzzywuzzy sort_ratio equality: ", first_value, "  ---  ", second_value, ">>>>", as_unicode(n))
                        return True
                elif self.fuzzytype2 == 2:
                    n = fuzz.token_set_ratio(first_value,second_value)
                    if n == 100:
                        #~ if DEBUG: print("fuzzywuzzy set_ratio equality: ", first_value, "  ---  ", second_value, ">>>>", as_unicode(n))
                        return True
            elif self.fuzzytype2 == 3:
                first_value = remove_punctuation(first_value)
                second_value = remove_punctuation(second_value)
                n = fuzz.ratio(first_value,second_value)
                if n == 100:
                    #~ if DEBUG: print("fuzzywuzzy simple ratio equality: ", first_value, "  ---  ", second_value, ">>>>", as_unicode(n))
                    return True
            elif self.fuzzytype2 == 4:
                first_value = self.remove_punctuation(first_value)
                second_value = self.remove_punctuation(second_value)
                first_value = self.fuzzywuzzy_convert_to_best_ascii_equivalents(first_value)
                second_value = self.fuzzywuzzy_convert_to_best_ascii_equivalents(second_value)
                n = fuzz.ratio(first_value,second_value)
                if n == 100:
                    #~ if DEBUG: print("fuzzywuzzy decomposed & normalized alphabet equality: ", first_value, "  ---  ", second_value, ">>>>", as_unicode(n))
                    return True
            elif self.fuzzytype2 == 5:
                first_result = MCSDoubleMetaphone(first_value)
                second_result = MCSDoubleMetaphone(second_value)
                n = fuzz.ratio(first_result,second_result)
                if n == 100:
                    #~ if DEBUG: print("Column 2: DoubleMetaphone Sounds-Like Match 100%: ", first_value, "  ---  ", second_value, ">>>>", as_unicode(n))
                    return True
            else:
                pass
        #~ if DEBUG: print("fuzzywuzzy *inequality*: ", first_value, "  ---  ", second_value, ">>>>", as_unicode(n))
        return False
    #-----------------------------------------------------------------------------------------
    def remove_punctuation(self,single_value):
        #~ convert to NO PUNCTUATION
        #~ if integer etc., nothing will be done.
        try:
            single_value = single_value.replace("!","")
            single_value = single_value.replace("@","")
            single_value = single_value.replace("#","")
            single_value = single_value.replace("$","")
            single_value = single_value.replace("%","")
            single_value = single_value.replace("^","")
            single_value = single_value.replace("&","")
            single_value = single_value.replace("*","")
            single_value = single_value.replace("(","")
            single_value = single_value.replace(")","")
            single_value = single_value.replace("-","")
            single_value = single_value.replace("_","")
            single_value = single_value.replace("+","")
            single_value = single_value.replace("=","")
            single_value = single_value.replace("{","")
            single_value = single_value.replace("}","")
            single_value = single_value.replace("[","")
            single_value = single_value.replace("]","")
            single_value = single_value.replace("|","")
            single_value = single_value.replace("\\","")
            single_value = single_value.replace(":","")
            single_value = single_value.replace(";","")
            single_value = single_value.replace("'","")
            single_value = single_value.replace('"','')
            single_value = single_value.replace("<","")
            single_value = single_value.replace(">","")
            single_value = single_value.replace(",","")
            single_value = single_value.replace(".","")
            single_value = single_value.replace("?","")
            single_value = single_value.replace("/","")
            single_value = single_value.replace("~","")
            single_value = single_value.replace("`","")
        except Exception as e:
            pass

        return single_value
    #-----------------------------------------------------------------------------------------
    def fuzzywuzzy_convert_to_best_ascii_equivalents(self,single_value):
        # also converts to upper case after conversion to ascii

        if isinstance(single_value,str):
            try:
                single_value = single_value.decode('utf-8')
            except:
                return single_value

        try:
            single_value = unidecode(single_value)
        except:
            pass

        try:
            single_value = single_value.upper()
        except:
            pass

        return single_value
    #-----------------------------------------------------------------------------------------
    def match_regex(self,name_value,text_value,regex,ignorecase):
        #does an re.search for both name and text values separately

        try:
            if ignorecase:
                p = re.compile(regex, re.IGNORECASE)
            else:
                p = re.compile(regex)
            match_name_value = p.search(name_value)
            match_text_value = p.search(text_value)
            if match_name_value and match_text_value:
                return True
            else:
                return False
        except:
            if DEBUG: print("RE COMPILE ERROR IN match_regex")
            return False
    #-----------------------------------------------------------------------------------------
    def do_inter_book_searches_all_books(self,my_db,my_cursor,param_dict,selected_books_list,std_metadata_list,cust_metadata_list):

        # the validation of the search dialog forces that the left mcs dialog side specifies only one column, as does the right side.  the operator is always '='.  the upper label always equals the lower label.
        search_column_1 = param_dict['NAME1']      # may be any custom column or any of these 3 standard columns:  title; author; series; may not be identical to search_column_2
        search_column_2 = param_dict['NAME2']      # may be any custom column or any of these 3 standard columns:  title; author; series; may not be identical to search_column_1
        search_operator_1 = '=' # only currently supported value for inter_book
        search_operator_2 = '=' # only currently supported value for inter_book

        std_metadata_list.sort()
        s_no_value_found_yet = "...NO_VALUE_FOUND_YET_MCS..."

        my_cursor.execute("begin")
        mysql = "DELETE FROM _mcs_interbook_search_columns_by_book "
        my_cursor.execute(mysql)
        my_cursor.execute("commit")

        if self.show_mcs_progress_bar:
            n_progress_counter = 0
            self.mcs_progress_bar_dialog.setLabelText("Inter-Book Search: Analyzing Metadata")
            self.mcs_progress_bar_dialog.setValue(0)
        my_cursor.execute("begin")
        for row in std_metadata_list:
            value_1 = s_no_value_found_yet
            value_2 = s_no_value_found_yet

            book,title,authorsall,series,tagsall,publisher,comments,pubdate,path = row

            # Only 3 of 8 Standard Columns Supported for Inter-Book Searches:  Title, Authors, Series.  Other 5 make no sense to use.
            if search_column_1.count('#') == 0:   # i.e., is a standard column
                if search_column_1 == 'title':
                    value_1 = title
                elif search_column_1 == 'authors':
                    value_1 = authorsall
                elif search_column_1 == 'series':
                    value_1 = series
                else:
                    pass

            # Only 3 of 8 Standard Columns Supported for Inter-Book Searches:  Title, Authors, Series.  Other 5 make no sense to use.
            if search_column_2.count('#') == 0:   # i.e., is a standard column
                if search_column_2 == 'title':
                    value_2 = title
                elif search_column_2 == 'authors':
                    value_2 = authorsall
                elif search_column_2 == 'series':
                    value_2 = series
                else:
                    pass

            if value_1 == s_no_value_found_yet or value_2 == s_no_value_found_yet:   # need to look through the custom column data for one or both required values
                for row in cust_metadata_list:
                    new_book_data_dict = row
                    if not (as_unicode(new_book_data_dict['book']) == as_unicode(book)):
                        continue
                    label = new_book_data_dict['label']
                    value = new_book_data_dict['value']
                    if value_1 == s_no_value_found_yet:
                        if label == search_column_1:
                            value_1 = value
                    if value_2 == s_no_value_found_yet:
                        if label == search_column_2:
                            value_2 = value
                #END FOR

            value_1 = self.apply_transform_functions_single_value(1,value_1)
            value_2 = self.apply_transform_functions_single_value(2,value_2)

            if value_1 != s_no_value_found_yet and value_2 != s_no_value_found_yet:
                if value_1 > "" and value_2 > "":         # blanks/null values are to be ignored, so remind the users in the tooltips that they must want to search for non-blanks...
                    mysql = "INSERT OR REPLACE INTO _mcs_interbook_search_columns_by_book SELECT ?,?,?"
                    my_cursor.execute(mysql,(book,value_1,value_2))

            if self.show_mcs_progress_bar:
                n_progress_counter = n_progress_counter + 1
                self.mcs_progress_bar_dialog.setValue(n_progress_counter)
                if self.mcs_progress_bar_dialog.wasCanceled():
                    self.user_clicked_cancel = True
                if self.user_clicked_cancel:
                    my_db.close()
                    dummy_list = []
                    return dummy_list
            #-----------------------------

        #END FOR
        my_cursor.execute("commit")

        #-----------------------------
        if self.show_mcs_progress_bar:
            if self.mcs_progress_bar_dialog.wasCanceled():
                self.user_clicked_cancel = True
            if self.user_clicked_cancel:
                my_db.close()
                dummy_list = []
                return dummy_list
        #-----------------------------

        #-----------------------------------------
        #------------------------------------------------------------------------
        # Only Supported Operator Combination for inter-book searches
        #------------------------------------------------------------------------
        if search_operator_1 == '=' and search_operator_2 == '=':
            #now, delete books that have unique values in both value columns, since there is no 'inter book' in that scenario:
            my_cursor.execute("begin")
            mysql = "DELETE FROM _mcs_interbook_search_columns_by_book \
                                        WHERE book IN(SELECT book FROM _mcs_interbook_search_columns_by_book GROUP BY value_1 HAVING COUNT(value_1) = 1) \
                                               OR book IN(SELECT book FROM _mcs_interbook_search_columns_by_book GROUP BY value_2 HAVING COUNT(value_2) = 1)"
            my_cursor.execute(mysql)
            my_cursor.execute("commit")
        #-----------------------------
        if self.show_mcs_progress_bar:
            if self.mcs_progress_bar_dialog.wasCanceled():
                self.user_clicked_cancel = True
            if self.user_clicked_cancel:
                my_db.close()
                dummy_list = []
                return dummy_list
        #-----------------------------
        #-----------------------------------------
        #now, get all remaining book ids:
        #-----------------------------------------
        found_list = []
        mysql = "SELECT book FROM _mcs_interbook_search_columns_by_book"
        my_cursor.execute(mysql)
        tmp_list = my_cursor.fetchall()
        for row in tmp_list:
            for col in row:
                found_list.append(col)
            #END FOR
        #END FOR
        del tmp_list
        #-----------------------------------------
        my_db.close()
        #-----------------------------------------
        if self.show_mcs_progress_bar:
            self.mcs_progress_bar_dialog.setLabelText("Inter-Book Search: Completed")
            self.mcs_progress_bar_dialog.setValue(100)
        #-----------------------------------------
        return found_list
    #-----------------------------------------------------------------------------------------
    def get_all_book_metadata(self,my_db,my_cursor,selected_books_list,param_dict):

        global cust_columns_search_list
        del cust_columns_search_list[:]

        std_metadata_list = []
        del std_metadata_list
        std_metadata_list = []
        if self.show_mcs_progress_bar:
            n_progress_counter = 0
            self.mcs_progress_bar_dialog.setLabelText("Retrieving Standard Metadata")
            self.mcs_progress_bar_dialog.setValue(0)
            if self.mcs_progress_bar_dialog.wasCanceled():
                self.user_clicked_cancel = True
            if self.user_clicked_cancel:
                dummy_list = []
                return  dummy_list, dummy_list
        for row in selected_books_list:
            book = int(row)
            mysql = "SELECT book,title,authorsall,series,tagsall,publisher,comments,pubdate,path FROM __mcs_searchable_metadata WHERE book = ?"
            my_cursor.execute(mysql,([book]))
            tmp_rows = my_cursor.fetchall()
            if not tmp_rows:
                continue
            else:
                if len(tmp_rows) == 0:
                    continue
                else:
                    for item in tmp_rows:
                        std_metadata_list.append(item)
                        #~ book,title,authorsall,series,tagsall,publisher,comments,pubdate,path = item
                        #~ if DEBUG: print("[0] >>> book,title,authorsall,series,tagsall", as_unicode(book), title, authorsall, series, tagsall)
                    #END FOR
                    del tmp_rows
            if self.show_mcs_progress_bar:
                n_progress_counter = n_progress_counter + 1
                self.mcs_progress_bar_dialog.setValue(n_progress_counter)
                if self.mcs_progress_bar_dialog.wasCanceled():
                    self.user_clicked_cancel = True
                    break
        #END FOR

        if self.show_mcs_progress_bar:
            if self.mcs_progress_bar_dialog.wasCanceled():
                self.user_clicked_cancel = True
            if self.user_clicked_cancel:
                dummy_list = []
                return  dummy_list, dummy_list

        cust_column_dict,cust_columns_normalized_list = self.get_list_of_all_custom_columns(my_db,my_cursor,param_dict)

        if self.show_mcs_progress_bar:
            if self.mcs_progress_bar_dialog.wasCanceled():
                self.user_clicked_cancel = True
            if self.user_clicked_cancel:
                dummy_list = []
                return  dummy_list, dummy_list

        cust_metadata_list = []
        del cust_metadata_list
        cust_metadata_list = []

        n = len(cust_column_dict)
        if n > 0:
            if self.show_mcs_progress_bar:
                n_progress_counter = 0
                self.mcs_progress_bar_dialog.setLabelText("Retrieving Custom Column Metadata")
                self.mcs_progress_bar_dialog.setValue(0)
            for row in selected_books_list:
                book = int(row)
                #~ for k,v in cust_column_dict.iteritems():
                for k,v in iteritems(cust_column_dict):
                    id = k
                    label = v
                    if label not in cust_columns_search_list:
                        cust_columns_search_list.append(label)
                    i = as_unicode(id)
                    if i in cust_columns_normalized_list:
                        new_book_data_dict = self.get_custom_column_metadata_normalized(my_db,my_cursor,id,label,book)
                    else:
                        new_book_data_dict = self.get_custom_column_metadata_unnormalized(my_db,my_cursor,id,label,book)
                    cust_metadata_list.append(new_book_data_dict)
                    if self.show_mcs_progress_bar:
                        if self.mcs_progress_bar_dialog.wasCanceled():
                            self.user_clicked_cancel = True
                            break
                #END FOR
                if self.show_mcs_progress_bar:
                    n_progress_counter = n_progress_counter + 1
                    self.mcs_progress_bar_dialog.setValue(n_progress_counter)
                    if self.mcs_progress_bar_dialog.wasCanceled():
                        self.user_clicked_cancel = True
                        dummy_list = []
                        return  dummy_list, dummy_list
            #END FOR
            #now have standard metadata by book, and the same for custom metadata.
        else:
            #~ if DEBUG: print("no custom column data needed for this query")
            pass

        del cust_column_dict

        return  std_metadata_list, cust_metadata_list
    #-----------------------------------------------------------------------------------------
    def get_custom_column_metadata_normalized(self,my_db,my_cursor,id,label,current_book):
        new_book_data_dict = {}
        del new_book_data_dict
        new_book_data_dict = {}

        new_book_data_dict['book'] = current_book
        new_book_data_dict['label'] = label
        value = ""

        mysql = "SELECT id,value FROM custom_column_XX \
                        WHERE id = (SELECT value FROM books_custom_column_XX_link WHERE book = ?)"
        mysql = mysql.replace("XX",as_unicode(id),2)
        my_cursor.execute(mysql,([current_book]))
        tmp_rows = my_cursor.fetchall()
        if not tmp_rows:
            value = ""
        else:
            if len(tmp_rows) == 0:
                value = ""
            else:
                for item in tmp_rows:
                    id,value = item

        new_book_data_dict['value'] = value

        return new_book_data_dict
    #-----------------------------------------------------------------------------------------
    def get_custom_column_metadata_unnormalized(self,my_db,my_cursor,id,label,current_book):
        new_book_data_dict = {}
        del new_book_data_dict
        new_book_data_dict = {}

        new_book_data_dict['book'] = current_book
        new_book_data_dict['label'] = label
        value = "NONE"

        mysql = "SELECT id,book,value FROM custom_column_XX WHERE book = ?"
        mysql = mysql.replace("XX",as_unicode(id),1)
        my_cursor.execute(mysql,([current_book]))
        tmp_rows = my_cursor.fetchall()
        if not tmp_rows:
            value = ""
        else:
            if len(tmp_rows) == 0:
                value = ""
            else:
                for item in tmp_rows:
                    id,book,value = item

        new_book_data_dict['value'] = value

        return new_book_data_dict
    #-----------------------------------------------------------------------------------------
    def get_list_of_all_custom_columns(self,my_db,my_cursor,param_dict):

        name1x = as_unicode(param_dict['NAME1'])
        text1x   = as_unicode(param_dict['TEXT1'])

        andornotignore = unicode_type(param_dict['ANDORNOT'])

        name2x = as_unicode(param_dict['NAME2'])
        text2x = as_unicode(param_dict['TEXT2'])

        try:
            del cust_column_dict
        except:
            pass

        cust_column_dict = {}
        cust_column_dict.clear()

        cust_columns_normalized_list = []
        del cust_columns_normalized_list[:]

        global cust_columns_integer_list
        global cust_columns_float_list
        global cust_columns_datetime_list
        del cust_columns_integer_list[:]
        del cust_columns_float_list[:]
        del cust_columns_datetime_list[:]

        sleep(0.02)

        mysql = "SELECT id,label,datatype,normalized FROM custom_columns"
        my_cursor.execute(mysql)
        tmp_rows = my_cursor.fetchall()
        if not tmp_rows:
            pass
        else:
            if len(tmp_rows) == 0:
                pass
            else:
                for item in tmp_rows:
                    id,label,datatype,normalized = item
                    normalized = as_unicode(normalized)
                    label = ('#' + label)
                    label = as_unicode(label)
                    if (as_unicode(label) == as_unicode(name1x)) or (as_unicode(label) == as_unicode(text1x)):     #only keep those that are needed for the current query
                        cust_column_dict[id] = label
                        if as_unicode(normalized) == as_unicode("1"):
                            cust_columns_normalized_list.append(as_unicode(id))
                        if as_unicode(datatype) == as_unicode("int"):
                            cust_columns_integer_list.append(label)
                        if as_unicode(datatype) == as_unicode("float"):
                            cust_columns_float_list.append(label)
                        if as_unicode(datatype) == as_unicode("datetime"):
                            cust_columns_datetime_list.append(label)
                    if andornotignore != "IGNORE":
                        #~ if DEBUG: print("if andornotignore != IGNORE: ", andornotignore)
                        if as_unicode(label) == as_unicode(name2x) or as_unicode(label) == as_unicode(text2x):     #only keep those that are needed for the current query
                            cust_column_dict[id] = label
                        if as_unicode(normalized) == as_unicode("1"):
                            cust_columns_normalized_list.append(as_unicode(id))
                        if as_unicode(datatype) == as_unicode("int"):
                            cust_columns_integer_list.append(label)
                        if as_unicode(datatype) == as_unicode("float"):
                            cust_columns_float_list.append(label)
                        if as_unicode(datatype) == as_unicode("datetime"):
                            cust_columns_datetime_list.append(label)
                    else:
                        #~ if DEBUG: print("andornotignore == IGNORE: ", andornotignore)
                        pass
                #END FOR
                del tmp_rows

        return cust_column_dict,cust_columns_normalized_list
    #-----------------------------------------------------------------------------------------
    def tweak_param_dict(self,param_dict):

        #---------------
        if param_dict['NAME1'] == 'published':
            param_dict['NAME1'] = 'pubdate'
        if param_dict['TYPE1'] == 'LABEL':
            if param_dict['TEXT1'] == 'published':
                param_dict['TEXT1'] = 'pubdate'

        if param_dict['NAME2'] == 'published':
            param_dict['NAME2'] = 'pubdate'
        if param_dict['TYPE2'] == 'LABEL':
            if param_dict['TEXT2'] == 'published':
                param_dict['TEXT2'] = 'pubdate'

        #---------------

        if param_dict['NAME1'] == 'author':
            param_dict['NAME1'] = 'authors'
        if param_dict['TYPE1'] == 'LABEL':
            if param_dict['TEXT1'] == 'author':
                param_dict['TEXT1'] = 'authors'

        if param_dict['NAME2'] == 'author':
            param_dict['NAME2'] = 'authors'
        if param_dict['TYPE2'] == 'LABEL':
            if param_dict['TEXT2'] == 'author':
                param_dict['TEXT2'] = 'authors'

        #---------------

        if param_dict['TEXT1'] == '':
            param_dict['TYPE1'] == 'TEXT'
        if param_dict['TEXT2'] == '':
            param_dict['TYPE2'] == 'TEXT'

        #---------------

        if param_dict['OPERATOR1'] == "regex":
            if not param_dict['REGEX1']:
                param_dict['REGEX1'] = "^.+$"

        if param_dict['OPERATOR2'] == "regex":
            if not param_dict['REGEX2']:
                param_dict['REGEX2'] = "^.+$"

        #---------------

        if param_dict['INTER_BOOK_SEARCH'] == unicode_type("True"):
            self.is_interbooksearch = True
        else:
            self.is_interbooksearch = False

        if param_dict['ALL_AUTHORS_BOOKS'] == unicode_type("True"):
            self.show_all_authors_books = True
        else:
            self.show_all_authors_books = False

        if param_dict['USE_FINAL_FILTERS'] == unicode_type("False"):
            self.apply_final_filters = False
        else:
            self.apply_final_filters = True

        #---------------
        if param_dict['FUZZYWUZZY1'] == unicode_type(FUZZYWUZZY_NONE):
            self.fuzzytype1 = 0
        elif param_dict['FUZZYWUZZY1'] == unicode_type(FUZZYWUZZY_TOKEN_SORT_RATIO):
            self.fuzzytype1 = 1
        elif param_dict['FUZZYWUZZY1'] == unicode_type(FUZZYWUZZY_TOKEN_SET_RATIO):
            self.fuzzytype1 = 2
        elif param_dict['FUZZYWUZZY1'] == unicode_type(FUZZYWUZZY_SIMPLE_RATIO_UPPER_CASE_NO_PUNCTUATION):
            self.fuzzytype1 = 3
        elif param_dict['FUZZYWUZZY1'] == unicode_type(FUZZYWUZZY_SIMPLE_RATIO_DECOMPOSED_NORMALIZED_ALPHABET):
            self.fuzzytype1 = 4
        elif param_dict['FUZZYWUZZY1'] == unicode_type(FUZZY_DOUBLEMETAPHONE_SIMPLE):
            self.fuzzytype1 = 5

        if param_dict['FUZZYWUZZY2'] == unicode_type(FUZZYWUZZY_NONE):
            self.fuzzytype2 = 0
        elif param_dict['FUZZYWUZZY2'] == unicode_type(FUZZYWUZZY_TOKEN_SORT_RATIO):
            self.fuzzytype2 = 1
        elif param_dict['FUZZYWUZZY2'] == unicode_type(FUZZYWUZZY_TOKEN_SET_RATIO):
            self.fuzzytype2 = 2
        elif param_dict['FUZZYWUZZY2'] == unicode_type(FUZZYWUZZY_SIMPLE_RATIO_UPPER_CASE_NO_PUNCTUATION):
            self.fuzzytype2 = 3
        elif param_dict['FUZZYWUZZY2'] == unicode_type(FUZZYWUZZY_SIMPLE_RATIO_DECOMPOSED_NORMALIZED_ALPHABET):
            self.fuzzytype2 = 4
        elif param_dict['FUZZYWUZZY2'] == unicode_type(FUZZY_DOUBLEMETAPHONE_SIMPLE):
            self.fuzzytype2 = 5

        self.need_tags_too = False
        if param_dict['NAME1'] == 'tags':
            self.need_tags_too = True
        if param_dict['TYPE1'] == 'LABEL':
            if param_dict['TEXT1'] == 'tags':
                self.need_tags_too = True
        if param_dict['ANDORNOT'] != unicode_type("IGNORE"):         #AND or OR or NOT or IGNORE
            if param_dict['NAME2'] == 'tags':
                self.need_tags_too = True
            if param_dict['TYPE2'] == 'LABEL':
                if param_dict['TEXT2'] == 'tags':
                    self.need_tags_too = True

        self.need_comments_too = False
        if param_dict['NAME1'] == 'comments':
            self.need_comments_too = True
        if param_dict['TYPE1'] == 'LABEL':
            if param_dict['TEXT1'] == 'comments':
                self.need_comments_too = True
        if param_dict['ANDORNOT'] != unicode_type("IGNORE"):         #AND or OR or NOT or IGNORE
            if param_dict['NAME2'] == 'comments':
                self.need_comments_too = True
            if param_dict['TYPE2'] == 'LABEL':
                if param_dict['TEXT2'] == 'comments':
                    self.need_comments_too = True

        self.need_authors_too = False
        if param_dict['NAME1'] == 'authors':
            self.need_authors_too = True
        if param_dict['TYPE1'] == 'LABEL':
            if param_dict['TEXT1'] == 'authors':
                self.need_authors_too = True
        if param_dict['ANDORNOT'] != unicode_type("IGNORE"):         #AND or OR or NOT or IGNORE
            if param_dict['NAME2'] == 'authors':
                self.need_authors_too = True
            if param_dict['TYPE2'] == 'LABEL':
                if param_dict['TEXT2'] == 'authors':
                    self.need_authors_too = True

        if self.is_cross_library_duplicates_search:
            self.need_comments_too = False
            self.show_all_authors_books = False
            self.apply_final_filters = False
            # Cross-Library searches do not support Transform Functions other than 'as-is' due to complexity and performance issues.
            param_dict['TRANSFORM_FUNCTION1'] = unicode_type("Compare: 'as-is'")
            param_dict['TRANSFORM_FUNCTION2'] = unicode_type("Compare: 'as-is'")
            self.transform_1_type = 1
            self.transform_2_type = 1

        try:
            del self.param_dict
        except:
            pass

        self.param_dict = param_dict.copy()

        if DEBUG:
            #~ for k,v in self.param_dict.iteritems():
            for k,v in iteritems(self.param_dict):
                print("self.param_dict: ",as_unicode(k),as_unicode(v))

        return param_dict
    #-----------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------
    def display_progress_bar(self,n_books):

        try:
            self.mcs_progress_bar_dialog.close()
        except:
            pass

        if int(n_books) >= int(SHOW_MCS_PROGRESS_BAR_MINIMUM_BOOKS):
            self.show_mcs_progress_bar = True
        else:
            self.show_mcs_progress_bar = False
            return

        self.mcs_progress_bar_dialog = QProgressDialog()
        self.mcs_progress_bar_dialog.setWindowTitle('MCS Progress')
        icon = self.qaction.icon()
        self.mcs_progress_bar_dialog.setWindowIcon(icon)
        self.mcs_progress_bar_dialog.setLabelText("Books Searched")
        self.mcs_progress_bar_dialog.setWindowModality(Qt.WindowModal)
        self.mcs_progress_bar_dialog.setRange(0,n_books)
        self.mcs_progress_bar_dialog.setValue(0)
        self.mcs_progress_bar_dialog.setAutoClose(False)
        self.mcs_progress_bar_dialog.setAutoReset(False)
        self.user_clicked_cancel = False
        self.mcs_progress_bar_dialog.show()
    #---------------------------------------------------------------------------------------------------------------------------------------
    def call_refresh_mcs_search_accelerator_tables(self):
        if self.number_jobs_executing > 0:
            n = self.number_jobs_executing
            if n == 1:
                msg = as_unicode(n) + " MCS Indexing Job is currently running.  Please wait until it finishes."
            else:
                msg = as_unicode(n) + " MCS Indexing Jobs are currently running.  Please wait until they finish."
            self.gui.status_bar.showMessage(msg)
            msg = msg + "<br><br>If this message is in error, you can reset the MCS Job Count by either switching briefly to another library and then switching back, or restarting Calibre. "
            error_dialog(self.gui, _('MCS'),_((msg)), show=True)
            return
        self.refresh_mcs_search_accelerator_tables(None,None)
    #---------------------------------------------------------------------------------------------------------------------------------------
    def refresh_mcs_search_accelerator_tables(self,my_db,my_cursor):
        # reduces a search on a 40,000 book library using Authors or Tags as a search criterion from hours to a few minutes or less...

        if self.number_jobs_executing > 0:
            if DEBUG: print("Jobs running; no refreshing of search accelerator tables etc. allowed at this time.")
            return

        need_to_close_db = False

        if not my_db:
            need_to_close_db = True
           #-----------------------------
            self.guidb = self.gui.library_view.model().db
            my_db = self.guidb
            path = my_db.library_path
            if isbytestring(path):
                path = path.decode(filesystem_encoding)
            path = path.replace(os.sep, '/')
            path = os.path.join(path, 'metadata.db')
            path = path.replace(os.sep, '/')
            try:
                my_db = apsw.Connection(path)
            except Exception as e:
                if DEBUG: print(as_unicode(e))
                path = path.encode("ascii", "ignore")
                msg = as_unicode("MCS cannot use the path that you selected:" + as_unicode(path) + " - " + as_unicode(e) )
                self.gui.status_bar.showMessage(msg)
                return
            my_cursor = my_db.cursor()
            mysql = "PRAGMA main.busy_timeout = 15000;"      #PRAGMA busy_timeout = milliseconds;
            my_cursor.execute(mysql)
            #--------------------------------------
        else:
            pass
            if DEBUG: print("refresh_mcs_search_accelerator_tables: invoked due to Cross-Library Search.")
            #--------------------------------------

        self.mcs_create_txt_format_word_index_table(my_db,my_cursor)    #if not exists...

        #-----------------------------------------
        # AUTHORS
        #-----------------------------------------
        my_cursor.execute("begin")
        mysql = "CREATE TABLE IF NOT EXISTS _mcs_authors_by_book (book INTEGER NOT NULL , authorname TEXT NOT NULL , PRIMARY KEY (book, authorname))"
        my_cursor.execute(mysql)
        mysql = "CREATE TABLE IF NOT EXISTS _mcs_authors_concatenate (book INTEGER NOT NULL  UNIQUE , authorsconcat TEXT NOT NULL , PRIMARY KEY (book, authorsconcat))"
        my_cursor.execute(mysql)
        my_cursor.execute("commit")
        sleep(0.02)

        my_cursor.execute("begin")
        mysql = "DELETE FROM _mcs_authors_by_book"
        my_cursor.execute(mysql)
        mysql = "DELETE FROM _mcs_authors_concatenate"
        my_cursor.execute(mysql)
        my_cursor.execute("commit")
        sleep(0.02)

        #~ if DEBUG: print("MCS Accelerator Tables for Authors purged.")

        my_cursor.execute("begin")
        mysql = "INSERT OR REPLACE INTO _mcs_authors_by_book \
                      SELECT book, (SELECT name FROM authors WHERE id = books_authors_link.author) as authorname FROM books_authors_link"
        my_cursor.execute(mysql)
        my_cursor.execute("commit")
        sleep(0.02)

        my_cursor.execute("begin")
        mysql = "INSERT OR REPLACE INTO _mcs_authors_concatenate \
                            SELECT book, group_concat(myconcat,',') AS authorsconcat \
                            FROM \
                            (SELECT book, \
                            CASE WHEN authorname IS NULL THEN 'unknown' \
                            WHEN authorname = ',' THEN 'unknown' \
                            WHEN authorname = ' ' THEN 'unknown' \
                            WHEN authorname = ''  THEN 'unknown' \
                            ELSE authorname \
                            END AS myconcat \
                            FROM _mcs_authors_by_book ORDER BY authorname  ) GROUP BY book "
        my_cursor.execute(mysql)
        my_cursor.execute("commit")
        sleep(0.02)
        #~ if DEBUG: print("MCS Accelerator Tables for Authors refreshed.")

        #-----------------------------------------
        # TAGS
        #-----------------------------------------
        my_cursor.execute("begin")
        mysql = "CREATE TABLE IF NOT EXISTS _mcs_tags_by_book (book INTEGER NOT NULL , tagname TEXT NOT NULL , PRIMARY KEY (book, tagname))"
        my_cursor.execute(mysql)
        mysql = "CREATE TABLE IF NOT EXISTS _mcs_tags_concatenate (book INTEGER NOT NULL  UNIQUE , tagsconcat TEXT NOT NULL , PRIMARY KEY (book, tagsconcat))"
        my_cursor.execute(mysql)
        my_cursor.execute("commit")
        sleep(0.02)

        my_cursor.execute("begin")
        mysql = "DELETE FROM _mcs_tags_by_book"
        my_cursor.execute(mysql)
        mysql = "DELETE FROM _mcs_tags_concatenate"
        my_cursor.execute(mysql)
        my_cursor.execute("commit")
        sleep(0.02)

        #~ if DEBUG: print("MCS Accelerator Tables for Tags purged.")

        my_cursor.execute("begin")
        mysql = "INSERT OR REPLACE INTO _mcs_tags_by_book \
                      SELECT book, (SELECT name FROM tags WHERE id = books_tags_link.tag) as tagname FROM books_tags_link"
        my_cursor.execute(mysql)
        my_cursor.execute("commit")
        sleep(0.02)
        my_cursor.execute("begin")
        mysql = "INSERT OR IGNORE INTO _mcs_tags_by_book \
                      SELECT id, 'NONE' FROM books WHERE id NOT IN (SELECT book FROM _mcs_tags_by_book)"
        my_cursor.execute(mysql)      # add a placeholder string if otherwise no tag so that the final filters will work properly since nulls evaluate ambiguously
        my_cursor.execute("commit")
        sleep(0.02)

        my_cursor.execute("begin")
        mysql = "INSERT OR REPLACE INTO _mcs_tags_concatenate \
                            SELECT book, group_concat(myconcat,',') AS tagsconcat \
                            FROM \
                            (SELECT book, \
                            CASE WHEN tagname IS NULL THEN 'unknown' \
                            WHEN tagname = ',' THEN 'unknown' \
                            WHEN tagname = ' ' THEN 'unknown' \
                            WHEN tagname = ''  THEN 'unknown' \
                            ELSE tagname \
                            END AS myconcat \
                            FROM _mcs_tags_by_book ORDER BY tagname  ) GROUP BY book "
        my_cursor.execute(mysql)
        my_cursor.execute("commit")
        sleep(0.02)
        #~ if DEBUG: print("MCS Accelerator Tables for Tags refreshed.")
        #-----------------------------------------
        if need_to_close_db:
            my_db.close()
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def create_temp_views_and_tables(self,my_db,my_cursor):

        if self.need_tags_too:  # only retrieve tag data if required; otherwise, dummy it out.
            #~ if DEBUG: print("DO need tags for this search")
            my_cursor.execute("begin")
            mysql = "CREATE TABLE IF NOT EXISTS _mcs_tags_by_book (book INTEGER NOT NULL , tagname TEXT NOT NULL , PRIMARY KEY (book, tagname))"
            my_cursor.execute(mysql)
            mysql = "CREATE TABLE IF NOT EXISTS _mcs_tags_concatenate (book INTEGER NOT NULL  UNIQUE , tagsconcat TEXT NOT NULL , PRIMARY KEY (book, tagsconcat))"
            my_cursor.execute(mysql)
            my_cursor.execute("commit")
            sleep(0.02)
            my_cursor.execute("begin")
            mysql = "CREATE TEMP VIEW IF NOT EXISTS __mcs_tags_by_book AS SELECT * FROM _mcs_tags_by_book"
            my_cursor.execute(mysql)
            my_cursor.execute("commit")
            sleep(0.02)
            my_cursor.execute("begin")
            mysql = "CREATE TEMP VIEW IF NOT EXISTS __mcs_tags_concatenate AS  SELECT  * FROM  _mcs_tags_concatenate"
            my_cursor.execute(mysql)
            my_cursor.execute("commit")
            sleep(0.02)
        else:  # dummy it out.  avoid huge performance hit.
            #~ if DEBUG: print("do NOT need tags for this search")
            my_cursor.execute("begin")
            mysql = "CREATE TEMP VIEW IF NOT EXISTS __mcs_tags_by_book AS SELECT Null as book, Null as tagname "
            my_cursor.execute(mysql)
            my_cursor.execute("commit")
            sleep(0.02)
            my_cursor.execute("begin")
            mysql = "CREATE TEMP VIEW IF NOT EXISTS __mcs_tags_concatenate AS  SELECT book, tagname AS tagsconcat \
                            FROM __mcs_tags_by_book "
            my_cursor.execute(mysql)
            my_cursor.execute("commit")
            sleep(0.02)

        if self.need_authors_too:  # only if absolutely required...
            #~ if DEBUG: print("DO need authors for this search")
            my_cursor.execute("begin")
            mysql = "CREATE TABLE IF NOT EXISTS _mcs_authors_by_book (book INTEGER NOT NULL , authorname TEXT NOT NULL , PRIMARY KEY (book, authorname))"
            my_cursor.execute(mysql)
            mysql = "CREATE TABLE IF NOT EXISTS _mcs_authors_concatenate (book INTEGER NOT NULL  UNIQUE , authorsconcat TEXT NOT NULL , PRIMARY KEY (book, authorsconcat))"
            my_cursor.execute(mysql)
            my_cursor.execute("commit")
            sleep(0.02)
            my_cursor.execute("begin")
            mysql = "CREATE TEMP VIEW IF NOT EXISTS __mcs_authors_by_book AS SELECT * FROM _mcs_authors_by_book"
            my_cursor.execute(mysql)
            my_cursor.execute("commit")
            sleep(0.02)
            my_cursor.execute("begin")
            mysql = "CREATE TEMP VIEW IF NOT EXISTS __mcs_authors_concatenate AS  SELECT  * FROM  _mcs_authors_concatenate"
            my_cursor.execute(mysql)
            my_cursor.execute("commit")
            sleep(0.02)
        else:  # dummy it out; avoid performance hit.
            #~ if DEBUG: print("do NOT need authors for this search")
            my_cursor.execute("begin")
            mysql = "CREATE TEMP VIEW IF NOT EXISTS __mcs_authors_by_book AS SELECT Null as book, Null as authorname "
            my_cursor.execute(mysql)
            my_cursor.execute("commit")
            sleep(0.02)
            my_cursor.execute("begin")
            mysql = "CREATE TEMP VIEW IF NOT EXISTS __mcs_authors_concatenate AS  SELECT book, authorname AS authorsconcat \
                            FROM __mcs_authors_by_book "
            my_cursor.execute(mysql)
            my_cursor.execute("commit")
            sleep(0.02)


        if self.need_comments_too:  # only if absolutely required...
            #~ if DEBUG: print("DO need comments for this search")
            my_cursor.execute("begin")
            mysql = "CREATE TEMP VIEW IF NOT EXISTS __mcs_searchable_metadata AS \
                            SELECT id AS book, title, \
                            (SELECT authorsconcat FROM __mcs_authors_concatenate WHERE __mcs_authors_concatenate.book = books.id) authorsall, \
                            (SELECT name FROM series WHERE series.id IN (SELECT series FROM books_series_link WHERE book=books.id)) series, \
                            (SELECT tagsconcat FROM __mcs_tags_concatenate WHERE __mcs_tags_concatenate.book = books.id) tagsall, \
                            (SELECT name FROM publishers WHERE publishers.id IN (SELECT publisher from books_publishers_link WHERE book=books.id)) publisher, \
                            (SELECT text FROM comments WHERE book=books.id) comments, \
                            pubdate,\
                            path \
                            FROM books;"
            my_cursor.execute(mysql)
            my_cursor.execute("commit")
            sleep(0.02)
        else:  # dummy it out; avoid performance hit.
            #~ if DEBUG: print("do NOT need comments for this search")
            my_cursor.execute("begin")
            mysql = "CREATE TEMP VIEW IF NOT EXISTS __mcs_searchable_metadata AS \
                            SELECT id AS book, title, \
                            (SELECT authorsconcat FROM __mcs_authors_concatenate WHERE __mcs_authors_concatenate.book = books.id) authorsall, \
                            (SELECT name FROM series WHERE series.id IN (SELECT series FROM books_series_link WHERE book=books.id)) series, \
                            (SELECT tagsconcat FROM __mcs_tags_concatenate WHERE __mcs_tags_concatenate.book = books.id) tagsall, \
                            (SELECT name FROM publishers WHERE publishers.id IN (SELECT publisher from books_publishers_link WHERE book=books.id)) publisher, \
                            Null AS comments, \
                            pubdate,\
                            path \
                            FROM books;"
            my_cursor.execute(mysql)
            my_cursor.execute("commit")
            sleep(0.02)

        my_cursor.execute("begin")
        mysql = "CREATE TEMP TABLE IF NOT EXISTS _mcs_interbook_search_columns_by_book (book INTEGER NOT NULL  UNIQUE , value_1 TEXT NOT NULL , value_2 TEXT NOT NULL , PRIMARY KEY (book, value_1, value_2)) "
        my_cursor.execute(mysql)
        my_cursor.execute("commit")
        sleep(0.02)
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def uninstall_mcs_search_accelerator_tables(self):
        if self.number_jobs_executing > 0:
            n = self.number_jobs_executing
            if n == 1:
                msg = as_unicode(n) + " MCS Indexing Job is currently running.  Please wait until it finishes."
            else:
                msg = as_unicode(n) + " MCS Indexing Jobs are currently running.  Please wait until they finish."
            self.gui.status_bar.showMessage(msg)
            msg = msg + "<br><br>If this message is in error, you can reset the MCS Job Count by either switching briefly to another library and then switching back, or restarting Calibre. "
            error_dialog(self.gui, _('MCS'),_((msg)), show=True)
            return
       #-----------------------------
        self.guidb = self.gui.library_view.model().db
        my_db = self.guidb
        path = my_db.library_path
        if isbytestring(path):
            path = path.decode(filesystem_encoding)
        path = path.replace(os.sep, '/')
        path = os.path.join(path, 'metadata.db')
        path = path.replace(os.sep, '/')
        try:
            my_db = apsw.Connection(path)
        except Exception as e:
            if DEBUG: print(as_unicode(e))
            path = path.encode("ascii", "ignore")
            msg = as_unicode("MCS cannot use the path that you selected:" + as_unicode(path) + " - " + as_unicode(e) )
            self.gui.status_bar.showMessage(msg)
            return
        my_cursor = my_db.cursor()
        mysql = "PRAGMA main.busy_timeout = 10000;"      #PRAGMA busy_timeout = milliseconds;
        my_cursor.execute(mysql)
        #--------------------------------------
        my_cursor.execute("begin")
        mysql = "DROP TABLE IF EXISTS _mcs_tags_by_book "
        my_cursor.execute(mysql)
        mysql = "DROP TABLE IF EXISTS _mcs_tags_concatenate "
        my_cursor.execute(mysql)
        mysql = "DROP TABLE IF EXISTS _mcs_authors_by_book "
        my_cursor.execute(mysql)
        mysql = "DROP TABLE IF EXISTS _mcs_authors_concatenate "
        my_cursor.execute(mysql)
        my_cursor.execute("commit")
        sleep(0.02)
        if question_dialog(self.gui, _('Delete the Word-by-Book Index?'),_('Delete the Word-by-Book Index too?  Are you sure?')) :
            my_cursor.execute("begin")
            mysql = "DROP TABLE IF EXISTS _mcs_word_book_index "
            my_cursor.execute(mysql)
            try:
                mysql = "DROP INDEX __mcs_word_book_index_by_word "
                my_cursor.execute(mysql)
            except:
                pass
            my_cursor.execute("commit")

        my_db.close()
        if DEBUG: print("MCS Search Accelerators Tables Uninstalled")
        error_dialog(self.gui, _('MCS'),_(("MCS Search Accelerators have been uninstalled.<br><br>You must uninstall this MCS plug-in right now, or you will have to perform this same action again later.<br><br>To Reinstall the MCS Search Accelerators, simply click the MCS Main Icon as usual.")), show=True)
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def build_menus(self,gui):

        self.gui = gui

        m = self.menu
        m.clear()
        m.addSeparator()
        create_menu_action_unique(self, m, ' ', ' ',
                              triggered=None)
        m.addSeparator()
        unique_name = "Read MCS User Guide"
        create_menu_action_unique(self, m, 'Read MCS User Guide', 'images/readinstructionsicon.png',
                              triggered=partial(self.view_user_instructions),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        m.addSeparator()
        create_menu_action_unique(self, m, ' ', ' ',
                              triggered=None)
        m.addSeparator()
        unique_name = "MCS Multi-Column Search"
        create_menu_action_unique(self, m, 'Multi-Column Search', 'images/mcsicon.png',
                              triggered=partial(self.search_request_dialog),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        m.addSeparator()
        create_menu_action_unique(self, m, ' ', ' ',
                              triggered=None)
        m.addSeparator()
        unique_name = "Refresh MCS Search Accelerators"
        create_menu_action_unique(self, m, 'Refresh MCS Search Accelerators', 'images/wrench-hammer.png',
                              triggered=partial(self.call_refresh_mcs_search_accelerator_tables),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        m.addSeparator()
        unique_name = "Uninstall MCS Search Accelerators"
        create_menu_action_unique(self, m, 'Uninstall MCS Search Accelerators', 'images/wrench-hammer.png',
                              triggered=partial(self.uninstall_mcs_search_accelerator_tables),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        m.addSeparator()
        create_menu_action_unique(self, m, ' ', ' ',
                              triggered=None)
        m.addSeparator()

        m.setToolTip("<p style='white-space:wrap'>'Refresh MCS Search Accelerators' is done <b>automatically and routinely</b> for you. \
                                <br><br> The <b>ONLY</b> reason you <b>should</b> click this menu option is if <i>(a) you have just changed any Author Names or Author/Book Combinations, \
                                AND (b) you do NOT want to exit from the MCS Search Dialog and then immediately re-click the MCS Plug-in Icon.</i>\
                                <br><br><br>If you plan on uninstalling the entire MCS Plug-in, please first uninstall the MCS Search Accelerators.  Otherwise, there is never a need to do so.  \
                                If you change your mind and continue to use MCS, no worries, they will be automatically be recreated and refreshed the next time you click the MCS Menu Icon.")

        self.gui.keyboard.finalize()
    #-----------------------------------------------------------------------------------------
    def view_user_instructions(self):

        global documentation_path

        self.extract_documentation_from_zip()

        try:
            p_pid = subprocess.Popen(documentation_path, shell=True)

            #Subprocess "p_pid"  is totally independent of Calibre,  and will never be checked or killed by this plugin...

        except:
            return error_dialog(self.gui, _('Documentation .pdf Not Found. Try reinstalling this plugin, and restarting Calibre.'),
                                       _('It is supposed to be: ' + documentation_path ), show=True)
    #-----------------------------------------------------------------------------------------
    def extract_documentation_from_zip(self):
        #extract the .pdf file from the zip so it can be executed directly.

        global documentation_path

        documentation_path = "unknown"

        zipfile_path = self.plugin_path

        destination_path = self.plugin_path
        destination_path = destination_path.replace("\MultiColumnSearch.zip", "", 1)
        destination_path = destination_path.replace("/MultiColumnSearch.zip", "", 1)

        zfile = zipfile.ZipFile(zipfile_path)

        dir_name = "mcs_documentation"
        if not is_py3:
            dir_name = dir_name.encode("ascii", "strict")

        file_name = 'mcs_instructions.pdf'
        if not is_py3:
            file_name = file_name.encode("ascii", "strict")

        file_name = os.path.join(dir_name, file_name )

        for name in zfile.namelist():
            n = name.find(dir_name)
            if n >= 0:
                zfile.extract(name, destination_path)
                documentation_path = os.path.join(destination_path, file_name)
                #"C:\Users\DaltonST\AppData\Roaming\calibre\plugins\mcs_documentation\mcs_instructions.pdf"
    #-----------------------------------------------------------------------------------------
    def display_search_log(self):
        global search_log
        global log_title_search_path
        global cust_columns_search_list

        self.mcs_gui_log_dialog = GUILogDialog(None,search_log, log_title_search_path, cust_columns_search_list)  #parent = None
        self.mcs_gui_log_dialog.show()
    #-----------------------------------------------------------------------------------------
    def add_all_authors_books(self,found_list):
        #~ use accelerator table _mcs_authors_by_book to get all books by the same authors represented in found_list so they can be marked too...
       #-----------------------------
        self.guidb = self.gui.library_view.model().db
        my_db = self.guidb
        path = my_db.library_path
        if isbytestring(path):
            path = path.decode(filesystem_encoding)
        path = path.replace(os.sep, '/')
        path = os.path.join(path, 'metadata.db')
        path = path.replace(os.sep, '/')
        try:
            my_db = apsw.Connection(path)
        except Exception as e:
            if DEBUG: print(as_unicode(e))
            path = path.encode("ascii", "ignore")
            msg = as_unicode("MCS cannot use the path that you selected:" + as_unicode(path) + " - " + as_unicode(e) )
            self.gui.status_bar.showMessage(msg)
            return
        my_cursor = my_db.cursor()
        mysql = "PRAGMA main.busy_timeout = 10000;"      #PRAGMA busy_timeout = milliseconds;
        my_cursor.execute(mysql)
       #-----------------------------
        add_list = []
        for book in found_list:
            #~ if DEBUG: print("book in found_list: ", as_unicode(book))
            mysql = "SELECT book FROM _mcs_authors_by_book WHERE authorname = (SELECT authorname FROM _mcs_authors_by_book WHERE book = ?)"
            my_cursor.execute(mysql,([book]))
            tmp_rows = my_cursor.fetchall()
            for row in tmp_rows:
                for col in row:
                    add_list.append(col)
                    #~ if DEBUG: print("book appended to add_list: ", as_unicode(col))
                #END FOR
            #END FOR
            del tmp_rows
        #END FOR

        my_db.close()

        for book in add_list:
            found_list.append(book)
        #END FOR
        del add_list

        found_set = set(found_list)
        found_list = list(found_set)
        del found_set

        return found_list
    #-----------------------------------------------------------------------------------------
    def apply_final_filters_control(self,found_list):

        active_final_filters_list = []
        for i in range(0,10):
            key = 'FINAL_FILTER_?'
            key = key.replace("?",as_unicode(i))
            raw = self.param_dict[key]
            if is_py3:
                raw = as_unicode(raw)
            else:
                raw = as_bytes(raw)
            row_list = ast.literal_eval(raw)
            a = row_list[0]                # self.checkbox_filter_0
            if a == unicode_type("True"):
                active_final_filters_list.append(key)
        #END FOR

        if len(active_final_filters_list) == 0:
            del active_final_filters_list
            return found_list
        else:
            found_list = self.actually_apply_final_filters(active_final_filters_list,found_list)

        del active_final_filters_list

        return found_list
    #-----------------------------------------------------------------------------------------
    def actually_apply_final_filters(self,active_final_filters_list,found_list):
        mysql_list = []
        for filter in active_final_filters_list:
            mysql = self.build_mysql(filter)
            mysql_list.append(mysql)
        #END FOR
        found_list = self.execute_final_filter_sql(mysql_list,found_list)
        return found_list
    #-----------------------------------------------------------------------------------------
    def build_mysql(self,filter):
        raw = self.param_dict[filter]
        row_list = eval(raw)
        a = row_list[0]                          # self.checkbox_filter_0
        table = as_unicode(row_list[1])             # self.table_0_combobox
        isnot = as_unicode(row_list[2])              # self.isnot_0_combobox
        sqloperator = as_unicode(row_list[5])   # self.operators_0_combobox
        value = row_list[3]                    # self.qlineedit_value_0
        andornot = as_unicode(row_list[4])      # self.andornot_0_combobox

        if "#" in table:
            table = table.replace("cc: ", "")

        table = as_unicode(table)

        mysql = ""
        mysql_is_complete = False
        mysql,mysql_is_complete = self.add_table_specific_sql(table,isnot,value,sqloperator,mysql)
        if not mysql_is_complete:
            mysql = self.add_isnot_specific_sql(isnot,sqloperator,mysql)
            mysql = self.add_sqloperator_specific_sql(sqloperator,mysql)
            mysql = self.add_value_specific_sql(table,value,sqloperator,mysql)
            mysql = self.add_andornot_specific_sql(andornot,mysql)

        # now repair it for systemic syntax errors...
        mysql = mysql.replace(" IS LIKE ", " LIKE ")
        mysql = mysql.replace(" IS NOT ", " NOT ")

        return mysql
    #-----------------------------------------------------------------------------------------
    def build_standard_tables_sql_template_dict(self):
        try:
            del self.sql_template_dict
        except:
            pass
        self.sql_template_dict = {}
        self.sql_template_dict['authors: name'] = "SELECT book FROM _mcs_authors_by_book WHERE book = ? AND authorname "
        self.sql_template_dict['authors: sort'] = "SELECT sort FROM authors WHERE id IN (SELECT author FROM books_authors_link WHERE book = ?) AND sort "
        self.sql_template_dict['books: title'] = " SELECT id FROM books WHERE id = ? AND title "
        self.sql_template_dict['books: pubdate'] = " SELECT id FROM books WHERE id = ? AND pubdate "
        self.sql_template_dict['books: path'] = " SELECT id FROM books WHERE id = ? AND path "
        self.sql_template_dict['books: has_cover'] = " SELECT id FROM books WHERE id = ? AND has_cover "
        self.sql_template_dict['books: series_index'] = " SELECT id FROM books WHERE id = ? AND series_index "
        self.sql_template_dict['publishers: name'] = "SELECT name FROM publishers WHERE id IN (SELECT publisher FROM books_publishers_link WHERE book = ?) AND name "
        self.sql_template_dict['series: name'] = "SELECT name FROM series WHERE id IN (SELECT series FROM books_series_link WHERE book = ?) AND name "
        self.sql_template_dict['tags: name'] = "SELECT book FROM _mcs_tags_by_book WHERE book = ? AND tagname "
        self.sql_template_dict['comments: text'] = " SELECT book FROM comments WHERE book = ? AND text "
        self.sql_template_dict['identifiers: type'] = " SELECT book FROM identifiers WHERE book = ? AND type "
        self.sql_template_dict['data: format'] = "SELECT book FROM data WHERE book = ? AND format "

        self.sql_template_dict['authors: name EXISTENT'] = "SELECT book FROM _mcs_authors_by_book WHERE book = ? AND authorname NOT NULL"
        self.sql_template_dict['authors: sort EXISTENT'] = "SELECT sort FROM authors WHERE id IN (SELECT author FROM books_authors_link WHERE book = ?) AND sort NOT NULL "
        self.sql_template_dict['books: title EXISTENT'] = " SELECT id FROM books WHERE id = ? AND title NOT NULL"
        self.sql_template_dict['books: pubdate EXISTENT'] = " SELECT id FROM books WHERE id = ? AND pubdate NOT NULL "
        self.sql_template_dict['books: path EXISTENT'] = " SELECT id FROM books WHERE id = ? AND path NOT NULL "
        self.sql_template_dict['books: has_cover EXISTENT'] = " SELECT id FROM books WHERE id = ? AND has_cover = 1 AND has_cover NOT NULL"
        self.sql_template_dict['books: series_index EXISTENT'] = " SELECT id FROM books WHERE id = ? AND series_index > 0 AND series_index NOT NULL "
        self.sql_template_dict['publishers: name EXISTENT'] = "SELECT publisher FROM books_publishers_link WHERE book = ? "
        self.sql_template_dict['series: name EXISTENT'] = "SELECT series FROM books_series_link WHERE book = ? "
        self.sql_template_dict['tags: name EXISTENT'] = "SELECT book FROM _mcs_tags_by_book WHERE book = ?  AND tagname NOT NULL"
        self.sql_template_dict['comments: text EXISTENT'] = " SELECT book FROM comments WHERE book = ? AND text NOT NULL"
        self.sql_template_dict['identifiers: type EXISTENT'] = " SELECT book FROM identifiers WHERE book = ? AND type =  "        # additional functionality for identifiers...
        self.sql_template_dict['data: format EXISTENT'] = "SELECT book FROM data WHERE book = ? AND format NOT NULL"

        self.sql_template_dict['authors: name REGEXP'] = "SELECT book FROM _mcs_authors_by_book WHERE book = ? AND authorname NOT NULL AND [IS_ISNOT] authname REGEXP [RE]"
        self.sql_template_dict['authors: sort REGEXP'] = "SELECT sort FROM authors WHERE id IN (SELECT author FROM books_authors_link WHERE book = ?) AND sort NOT NULL AND [IS_ISNOT] sort REGEXP [RE] "
        self.sql_template_dict['books: title REGEXP'] = " SELECT id FROM books WHERE id = ? AND title NOT NULL AND [IS_ISNOT] title REGEXP [RE]"
        self.sql_template_dict['books: pubdate REGEXP'] = " SELECT id FROM books WHERE id = ? AND pubdate NOT NULL  AND [IS_ISNOT] pubdate REGEXP [RE] "
        self.sql_template_dict['books: path REGEXP'] = " SELECT id FROM books WHERE id = ? AND path NOT NULL  AND [IS_ISNOT] path REGEXP [RE]"
        self.sql_template_dict['publishers: name REGEXP'] = "SELECT publisher FROM books_publishers_link WHERE book = ?  AND [IS_ISNOT] publisher REGEXP [RE]"
        self.sql_template_dict['series: name REGEXP'] = "SELECT series FROM books_series_link WHERE book = ?  AND [IS_ISNOT] series REGEXP [RE]"
        self.sql_template_dict['tags: name REGEXP'] = "SELECT book FROM _mcs_tags_by_book WHERE book = ?  AND tagname NOT NULL  AND [IS_ISNOT] tagname REGEXP [RE]"
        self.sql_template_dict['comments: text REGEXP'] = " SELECT book FROM comments WHERE book = ? AND text NOT NULL AND [IS_ISNOT] text REGEXP [RE]"
        self.sql_template_dict['data: format REGEXP'] = "SELECT book FROM data WHERE book = ? AND format NOT NULL AND [IS_ISNOT] format REGEXP [RE]"

        return self.sql_template_dict
    #-----------------------------------------------------------------------------------------
    def add_table_specific_sql(self,table,isnot,value,sqloperator,mysql):

        self.sql_template_dict = self.build_standard_tables_sql_template_dict()
        #~ if DEBUG: print("table for key is: ", table)
        #~ if DEBUG: print("rows in self.sql_template_dict: ", as_unicode(len(self.sql_template_dict)) )
        if "#" in table:
            mysql, mysql_is_complete = self.add_custom_column_specific_sql(table,isnot,value,sqloperator,mysql)
        else:
            mysql = self.add_standard_column_specific_sql(table,sqloperator,mysql)
            if "REGEXP [RE]" in mysql:
                repattern = "'" + value + "'"
                mysql = mysql.replace("[RE]",repattern)
            mysql_is_complete = False
        return mysql,mysql_is_complete
    #-----------------------------------------------------------------------------------------
    def add_standard_column_specific_sql(self,table,sqloperator,mysql):
        if sqloperator == "EXISTENT":
            table = as_unicode(table) +  as_unicode(" EXISTENT")
        elif sqloperator == "REGEX":
            table = as_unicode(table) +  as_unicode(" REGEXP")
        mysql = self.sql_template_dict[table]
        return mysql
    #-----------------------------------------------------------------------------------------
    def add_isnot_specific_sql(self,isnot,sqloperator,mysql):
        if isnot == "IS NOT":
            mysql = "SPECIAL_COMBINATION||" + isnot + "||" + sqloperator + "||" + mysql
        if isnot == "IS NOT":
            isnot = "NOT"
        if sqloperator == "EXISTENT":
            isnot = ""
        if sqloperator == "REGEX":
            isnot = ""
        if not sqloperator.isalpha():    #  > , = , < , etc.
            pass
        else:
            mysql = mysql + " " + isnot + " "
        return mysql
    #-----------------------------------------------------------------------------------------
    def add_sqloperator_specific_sql(self,sqloperator,mysql):
        if sqloperator == "REGEX":
            pass
        elif sqloperator == "EXISTENT":
            pass
        else:
            mysql = mysql + sqloperator + " "
        return mysql
    #-----------------------------------------------------------------------------------------
    def add_value_specific_sql(self,table,value,sqloperator,mysql):

        fragment = ""

        if not value:
            value = ""

        value = value.strip()

        value = value.replace("'","")   # quotes will cause sql errors.
        value = value.replace('"','')
        value = value.replace('?','')     # and a ? will be treated as a parameter variable...


        if sqloperator == "LIKE":
            fragment = "'%" + value + "%'" + " "
        elif sqloperator == "TRUE":
            pass
        elif sqloperator == "FALSE":
            pass
        elif sqloperator == "EXISTENT":
            if as_unicode("identifiers: type") in as_unicode(table):
                fragment = "'" + value + "'"
            else:
                pass
        elif sqloperator == "NULL":
            pass
        elif sqloperator == "REGEX":
            pass
        else:
            if table == 'books: series_index':        # real
                fragment = value
            else:
                if table == 'books: has_cover':       # bool
                    fragment = ""
                else:
                    if "#" in table:
                        id = self.custom_column_label_dict[table]
                        datatype = self.custom_column_datatype_dict[id]
                        if datatype == "int" or datatype == "float":
                            fragment = value
                        else:
                            if datatype == "bool":
                                fragment = ""
                            else:
                                fragment = "'" + value + "'"
                    else:
                        fragment = "'" + value + "'"

        fragment = fragment.replace('"',"'")   #  replace " with '     since should never have double quotes in sql queries
        fragment = fragment.replace("''","'")  #  replace '' with '    since should be:  'george w. washington'  and not 2 single quotes together

        mysql = mysql + fragment

        if "#" in table:
            mysql = mysql.replace('AND value  IS FALSE','AND value  = 0')
            mysql = mysql.replace('AND value  IS TRUE','AND value  = 1')

        return mysql
    #-----------------------------------------------------------------------------------------
    def add_andornot_specific_sql(self,andornot,mysql):
        if andornot == "":
            pass
        else:
            mysql = mysql + "|" + andornot
        return mysql
    #-----------------------------------------------------------------------------------------
    def build_sql_template_dict(self):
        # executed after mcs dialog is first created, so only created once per calibre session...
        self.sql_template_dict = {}
    #-----------------------------------------------------------------------------------------
    def add_custom_column_specific_sql(self,label,isnot,value,sqloperator,mysql):
        #~ self.custom_column_label_dict = {}                 #  [label] = id
        #~ self.custom_column_datatype_dict = {}          #  [id] = datatype
        #~ self.custom_column_normalized_dict = {}      #  [id] = is_normalized

        id = self.custom_column_label_dict[label]
        datatype = self.custom_column_datatype_dict[id]
        is_normalized = self.custom_column_normalized_dict[id]

        mysql_is_complete = False

        if sqloperator == "EXISTENT":
            mysql_normalized = "SELECT book FROM books_custom_column_NN_link WHERE book = ? "
            mysql_not_normalized =  "SELECT book FROM custom_column_NN WHERE book = ?  "
        elif sqloperator == "REGEX":
            repattern = "'" + value + "'"
            mysql_normalized = "SELECT value FROM custom_column_NN WHERE id IN (SELECT value FROM books_custom_column_NN_link WHERE book = ?) AND [IS_ISNOT] value REGEXP [RE]  "
            mysql_normalized = mysql_normalized.replace("[RE]",repattern)
            mysql_not_normalized = "SELECT value FROM custom_column_NN WHERE book = ? AND [IS_ISNOT] value REGEXP [RE]  "
            mysql_not_normalized = mysql_normalized.replace("[RE]",repattern)
        else:
            mysql_normalized = "SELECT value FROM custom_column_NN WHERE id IN (SELECT value FROM books_custom_column_NN_link WHERE book = ?) AND value "
            mysql_not_normalized = "SELECT value FROM custom_column_NN WHERE book = ? AND value "

        if datatype == "series":
            mysql_normalized = self.add_datatype_series_sql(isnot,value,sqloperator,mysql)
            mysql_is_complete = True

        if is_normalized:
            mysql = mysql_normalized
        else:
            mysql = mysql_not_normalized

        mysql = mysql.replace("NN",as_unicode(id))

        return mysql, mysql_is_complete
    #-----------------------------------------------------------------------------------------
    def add_datatype_series_sql(self,isnot,value, sqloperator,mysql):
        # unique to this datatype is an additional column, "extra", that holds the series index in books_custom_column_NN_link

        if not as_unicode("[") in as_unicode(value):
            value = value + " [*]"

        p = re.compile("[[][0-9.*]*[]]")
        match_index = p.search(value)
        if not match_index:
            return "ERROR - NO INDEX FOR SERIES-LIKE CUSTOM COLUMN"
        full_index = match_index.group()
        series = value.replace(full_index,"")
        series = series.strip()
        full_index = full_index.replace("[","")
        full_index = full_index.replace("]","")
        series_index = full_index.strip()
        if series_index == "*":
            is_wildcard = True
        else:
            is_wildcard = False
            try:
                series_index = int(series_index)
                series_index = as_unicode(series_index)
            except:
                return "ERROR - BAD INDEX FOR SERIES-LIKE CUSTOM COLUMN"

        mysql_normalized = "ERROR - BAD OPERATOR FOR SERIES-LIKE CUSTOM COLUMN"

        if sqloperator == "EXISTENT":
            if is_wildcard:
                if isnot == "IS":
                    mysql_normalized = "SELECT book,\
                                                                  (SELECT value FROM custom_column_NN WHERE id = books_custom_column_NN_link.value) as seriesname, \
                                                                   extra FROM books_custom_column_NN_link WHERE book = ?"
                else:
                    mysql_normalized = "SELECT book,\
                                                                  (SELECT value FROM custom_column_NN WHERE id = books_custom_column_NN_link.value) as seriesname, \
                                                                   extra FROM books_custom_column_NN_link WHERE book = ?"
            else:
                if isnot == "IS":
                    mysql_normalized = "SELECT book,\
                                                                  (SELECT value FROM custom_column_NN WHERE id = books_custom_column_NN_link.value) as seriesname, \
                                                                   extra FROM books_custom_column_NN_link WHERE book = ? AND extra = " + series_index + "   "
                else:
                    mysql_normalized = "SELECT book,\
                                                              (SELECT value FROM custom_column_NN WHERE id = books_custom_column_NN_link.value) as seriesname, \
                                                               extra FROM books_custom_column_NN_link WHERE book = ? AND extra = " + series_index + "   "

        elif sqloperator == "LIKE":
            if is_wildcard:
                if isnot == "IS":
                    mysql_normalized = "SELECT book,\
                                                                  (SELECT value FROM custom_column_NN WHERE id = books_custom_column_NN_link.value AND value LIKE '%" + series + "%') as seriesname, \
                                                                   extra FROM books_custom_column_NN_link WHERE book = ?"
                else:
                    mysql_normalized = "SELECT book,\
                                                                  (SELECT value FROM custom_column_NN WHERE id = books_custom_column_NN_link.value AND value NOT LIKE '%" + series + "%') as seriesname, \
                                                                   extra FROM books_custom_column_NN_link WHERE book = ?"
            else:
                if isnot == "IS":
                    mysql_normalized = "SELECT book,\
                                                                  (SELECT value FROM custom_column_NN WHERE id = books_custom_column_NN_link.value AND value LIKE '%" + series + "%') as seriesname, \
                                                                   extra FROM books_custom_column_NN_link WHERE book = ? AND extra = " + series_index + "   "
                else:
                    mysql_normalized = "SELECT book,\
                                                                  (SELECT value FROM custom_column_NN WHERE id = books_custom_column_NN_link.value AND value NOT LIKE '%" + series + "%') as seriesname, \
                                                                   extra FROM books_custom_column_NN_link WHERE book = ? AND extra = " + series_index + "   "
        elif sqloperator == "=":
            if is_wildcard:
                if isnot == "IS":
                    mysql_normalized = "SELECT book,\
                                                                  (SELECT value FROM custom_column_NN WHERE id = books_custom_column_NN_link.value AND value = '" + series + "' ) as seriesname, \
                                                                   extra FROM books_custom_column_NN_link WHERE book = ?"
                else:
                    mysql_normalized = "SELECT book,\
                                                                  (SELECT value FROM custom_column_NN WHERE id = books_custom_column_NN_link.value AND value != '" + series + "' ) as seriesname, \
                                                                   extra FROM books_custom_column_NN_link WHERE book = ?"
            else:
                if isnot == "IS":
                    mysql_normalized = "SELECT book,\
                                                                  (SELECT value FROM custom_column_NN WHERE id = books_custom_column_NN_link.value AND value = '" + series + "' ) as seriesname, \
                                                                   extra FROM books_custom_column_NN_link WHERE book = ? AND extra = " + series_index + "   "
                else:
                    mysql_normalized = "SELECT book,\
                                                                  (SELECT value FROM custom_column_NN WHERE id = books_custom_column_NN_link.value AND value != '" + series + "' ) as seriesname, \
                                                                   extra FROM books_custom_column_NN_link WHERE book = ? AND extra != " + series_index + "   "
        elif sqloperator == "!=":
            if is_wildcard:
                if isnot == "IS":
                    mysql_normalized = "SELECT book,\
                                                                  (SELECT value FROM custom_column_NN WHERE id = books_custom_column_NN_link.value AND value != '" + series + "' ) as seriesname, \
                                                                   extra FROM books_custom_column_NN_link WHERE book = ?"
                else:
                    mysql_normalized = "SELECT book,\
                                                                  (SELECT value FROM custom_column_NN WHERE id = books_custom_column_NN_link.value AND value = '" + series + "' ) as seriesname, \
                                                                   extra FROM books_custom_column_NN_link WHERE book = ?"
            else:
                if isnot == "IS":
                    mysql_normalized = "SELECT book,\
                                                                  (SELECT value FROM custom_column_NN WHERE id = books_custom_column_NN_link.value AND value != '" + series + "' ) as seriesname, \
                                                                   extra FROM books_custom_column_NN_link WHERE book = ? AND extra != " + series_index + "   "
                else:
                    mysql_normalized = "SELECT book,\
                                                                  (SELECT value FROM custom_column_NN WHERE id = books_custom_column_NN_link.value AND value = '" + series + "' ) as seriesname, \
                                                                   extra FROM books_custom_column_NN_link WHERE book = ? AND extra != " + series_index + "   "

        if isnot == "IS NOT":
            mysql_normalized = "SPECIAL_COMBINATION||" + isnot + "||" + sqloperator + "||" + mysql_normalized

        return mysql_normalized
    #-----------------------------------------------------------------------------------------
    def build_custom_column_dicts(self):
        # executed after mcs is first initialized by calibre, so only created once per calibre session unless library changes.

        try:
            self.custom_column_label_dict.clear()
            self.custom_column_datatype_dict.clear()
            self.custom_column_normalized_dict.clear()
        except:
            self.custom_column_label_dict = {}                 #  [label] = id
            self.custom_column_datatype_dict = {}          #  [id] = datatype
            self.custom_column_normalized_dict = {}      #  [id] = is_normalized

        #-----------------------------
        my_db,my_cursor = self.apsw_connect_to_current_guidb()
        #-----------------------------

        mysql = "SELECT id,label,datatype,normalized FROM custom_columns"
        my_cursor.execute(mysql)
        tmp_rows = my_cursor.fetchall()
        for row in tmp_rows:
            id,label,datatype,normalized = row
            label = '#' + label
            self.custom_column_label_dict[label] = id
            self.custom_column_datatype_dict[id] = datatype
            self.custom_column_normalized_dict[id] = normalized
        #END FOR

        my_db.close()
        #~ return self.custom_column_label_dict,self.custom_column_datatype_dict,self.custom_column_normalized_dict
    #-----------------------------------------------------------------------------------------
    def apsw_connect_to_current_guidb(self):
        self.guidb = self.gui.library_view.model().db
       #-----------------------------
        my_db = self.guidb
        path = my_db.library_path
        if isbytestring(path):
            path = path.decode(filesystem_encoding)
        path = path.replace(os.sep, '/')
        self.lib_path = path
        path = os.path.join(path, 'metadata.db')
        path = path.replace(os.sep, '/')
        try:
            my_db = apsw.Connection(path)
        except Exception as e:
            if DEBUG: print(as_unicode(e))
            path = path.encode("ascii", "ignore")
            msg = as_unicode("MCS cannot use the path that you selected:" + as_unicode(path) + " - " + as_unicode(e) )
            self.gui.status_bar.showMessage(msg)
            return None,None
        my_cursor = my_db.cursor()
        mysql_pragma = "PRAGMA main.busy_timeout = 10000;"      #PRAGMA busy_timeout = milliseconds;
        my_cursor.execute(mysql_pragma)

        return my_db,my_cursor
    #-----------------------------------------------------------------------------------------
    def execute_final_filter_sql(self,mysql_list,found_list):
        #~ #-----------------------------
        #~                      top
        #~ is                    or
        #~ is not             or
        #~ is                 and
        #~ is not          and
        #~ is                 not
        #~ is                 not
        #~ is not          end of query
        #~ #-----------------------------

        n_number_of_filters = len(mysql_list)

        #~ if DEBUG: print("n_number_of_filters: ", as_unicode(n_number_of_filters))

        if n_number_of_filters == 0:
            return found_list

        #-----------------------------
        if self.show_mcs_progress_bar:
            if self.mcs_progress_bar_dialog.wasCanceled():
                self.user_clicked_cancel = True
            if self.user_clicked_cancel:
                dummy_list = []
                return dummy_list
        #-----------------------------

        my_db,my_cursor = self.apsw_connect_to_current_guidb()
        func = "REGEXP"
        self.apsw_create_user_functions(my_db,my_cursor,func)
       #-----------------------------

        s_variable = as_unicode("?")

        new_found_list = []

        try:
            for book in found_list:
                #-----------------------------
                if self.show_mcs_progress_bar:
                    if self.mcs_progress_bar_dialog.wasCanceled():
                        self.user_clicked_cancel = True
                    if self.user_clicked_cancel:
                        dummy_list = []
                        return dummy_list
                #-----------------------------
                book = int(book)
                book_is_a_keeper = True
                #~ if DEBUG: print("-----------------------------------filtering book in found_list: ", as_unicode(book))
                previous_andornot = "TOP"
                pass_or_die = False
                n_count = 0
                for mysql in mysql_list:
                    n_count = n_count + 1
                    #~ if DEBUG: print("n_count = ", as_unicode(n_count))
                    if mysql.startswith("SPECIAL_COMBINATION||"):      #~ "SPECIAL_COMBINATION||" + isnot + "||" sqloperator + "||" + mysql_normalized
                        s_split = mysql.split("||")
                        isnot = s_split[1]
                        sqloperator = s_split[2]
                        mysql = s_split[3]
                        is_special_combination = True
                    else:
                        isnot = ""
                        sqloperator = ""
                        is_special_combination = False
                    mysql,next_andornot = self.finalize_mysql(mysql,isnot)
                    n_parameters = mysql.count(s_variable)
                    if n_parameters == 1:
                        my_cursor.execute(mysql,([book]))
                    elif n_parameters == 2:
                        my_cursor.execute(mysql,(book,book))
                    else:
                        if DEBUG: print("n_parameters is invalid.")
                        break
                    tmp_rows = my_cursor.fetchall()
                    if not tmp_rows:
                        tmp_rows = []
                    #------------------------------------------------------------------------------------------------
                    if len(tmp_rows) == 0:
                        #---------------------
                        if pass_or_die:
                            if previous_andornot == "OR" and next_andornot == "OR":   # [got nothing]
                                # but, if should have gotten nothing and did get nothing, then next OR does not matter, so do not make pass or die true.  next OR does not matter.
                                if is_special_combination:
                                    if isnot == "IS NOT":   # IS NOT = FALSE [got nothing, which is correct]
                                        #~ if DEBUG: print("[4 got nothing, which is correct] ")
                                        pass_or_die = False
                                        previous_andornot = next_andornot
                                        continue
                                    else: # IS  = TRUE [got nothing, which is not correct]
                                        #~ if DEBUG: print("[4a got nothing; defer judgement until after next filter evaluated]  previous_andornot == OR and next_andornot == OR ; pass or die...")
                                        pass_or_die = True
                                        previous_andornot = next_andornot
                                        continue
                                else:
                                    #~ if DEBUG: print("[4b got nothing; defer judgement until after next filter evaluated]  previous_andornot == OR and next_andornot == OR ; pass or die...")
                                    pass_or_die = True
                                    previous_andornot = next_andornot
                                    if n_count == n_number_of_filters:
                                        break
                                    else:
                                        continue
                            else:
                                if previous_andornot == "OR" and next_andornot != "OR":  #~ scenario: final OR must be passed since the previous OR filters were deferred until the very last OR, which is here right now.
                                    if is_special_combination:
                                        if isnot == "IS NOT":   # IS NOT = FALSE [got nothing, which is correct]
                                            #~ if DEBUG: print("[5 got nothing, which is correct] ")
                                            pass_or_die = False
                                            previous_andornot = next_andornot
                                            if n_count == n_number_of_filters:
                                                break
                                            else:
                                                continue
                                        else:  # IS  = TRUE [got nothing, which is not correct]
                                            #~ if DEBUG: print("[5a got nothing; this is the very last OR filter, so pass_or_die ends here with a 'die'] ")
                                            book_is_a_keeper = False
                                            pass_or_die = False
                                            break
                                    else:
                                        #~ if DEBUG: print("[5b got nothing; this is the very last OR filter, so pass_or_die ends here with a 'die'] ")
                                        book_is_a_keeper = False
                                        pass_or_die = False
                                        break
                                else:
                                    book_is_a_keeper = False
                                    pass_or_die = False
                                    #~ if DEBUG: print("05c pass or die failed;  breaking")
                                    break
                        else:
                            #---------------------
                            if previous_andornot == "TOP":
                                #---------------------
                                if n_count == n_number_of_filters or next_andornot == "":
                                    if is_special_combination:
                                        if isnot == "IS NOT":
                                            if sqloperator != "FALSE":   # TRUE   +  IS NOT  = FALSE        # changed in v2.0.36
                                                book_is_a_keeper = False
                                                pass_or_die = False
                                                #~ if DEBUG: print("[6 got nothing, which is not correct] special combination; failed;  breaking")
                                                break
                                            else: # IS NOT = FALSE   + FALSE = TRUE [got nothing, which is correct]
                                                #~ if DEBUG: print("[6a got nothing, which is correct] previous_andornot == TOP; pass; special combination: ", isnot, sqloperator)
                                                pass_or_die = False
                                                break  # one and only one final filter...which was just passed...
                                        else:  # IS [but got nothing, which is not correct]
                                            book_is_a_keeper = False
                                            pass_or_die = False
                                            #~ if DEBUG: print("[6b got nothing, which is not correct] special combination; failed;  breaking")
                                            break
                                    else:
                                        book_is_a_keeper = False
                                        pass_or_die = False
                                        #~ if DEBUG: print("[6c got nothing, which is not correct] failed;  breaking")
                                        break
                                elif next_andornot == "OR":
                                    if is_special_combination:
                                        if isnot == "IS NOT":  # IS NOT = FALSE [got nothing, which is correct]
                                            #~ if DEBUG: print("[7 got nothing, which is correct] previous_andornot == TOP; pass; special combination: ", isnot, sqloperator)
                                            pass_or_die = False
                                            pass
                                        else:  # IS [but got nothing, which is not correct]
                                            #~ if DEBUG: print("[7a got nothing; defer judgement until after next filter evaluated] previous_andornot == TOP and next_andornot == OR ; pass or die...")
                                            pass_or_die = True
                                            pass
                                    else:  # [got nothing; defer judgement until after next filter evaluated]
                                        #~ if DEBUG: print("[7b got nothing; defer judgement until after next filter evaluated] previous_andornot == TOP and next_andornot == OR ; pass or die...")
                                        pass_or_die = True
                                        pass
                                else:
                                    book_is_a_keeper = False
                                    pass_or_die = False
                                    #~ if DEBUG: print("[7c got nothing, which is not correct] previous_andornot == TOP and next_andornot != OR; failed;  breaking")
                                    break
                                #---------------------
                            elif previous_andornot == "NOT":
                                if is_special_combination:
                                    if isnot == "IS NOT":       # NOT + IS NOT = TRUE, but got FALSE       [should have had something, but did not]
                                        if sqloperator != "FALSE":
                                            book_is_a_keeper = False
                                            pass_or_die = False
                                            #~ if DEBUG: print("[should have had something, but did not] 08 special combination; failed;  breaking")
                                            break
                                        else:  # NOT + IS NOT + FALSE  =  TRUE + FALSE = FALSE [got nothing, which is correct]
                                            #~ if DEBUG: print("[got nothing, which is correct] previous_andornot == NOT; pass; special combination: ", isnot, sqloperator)
                                            pass
                                    else: # NOT + IS  = NOT = FALSE [got nothing, which is correct]
                                        #~ if DEBUG: print("[got nothing, which is correct] previous_andornot == NOT; pass; special combination: ", isnot, sqloperator)
                                        pass
                                else:  # NOT = FALSE [got nothing, which is correct]
                                    #~ if DEBUG: print("[got nothing, which is correct] previous_andornot == NOT; pass")
                                    pass
                                #---------------------
                            elif previous_andornot == "OR":
                                if book_is_a_keeper:  # if a or b, and a is true, then b does not matter...
                                    #~ if DEBUG: print("previous_andornot == OR; pass")
                                    pass
                                else:
                                    if is_special_combination:
                                        if sqloperator != "FALSE":
                                            if isnot == "IS NOT":     # IS NOT = FALSE     [got no results, and should not have]
                                                #~ if DEBUG: print("[got no results, and should not have]  09 special combination: ", isnot, sqloperator)
                                                pass
                                            else:  # IS = TRUE  but got FALSE      [got no results, but should have]
                                                book_is_a_keeper = False
                                                pass_or_die = False
                                                #~ if DEBUG: print("[got no results, but should have]  09a special combination; failed;  breaking")
                                                break
                                        else:  # IS NOT FALSE = IS TRUE  but got FALSE  [got no results, but should have]
                                            book_is_a_keeper = False
                                            pass_or_die = False
                                            #~ if DEBUG: print("[got no results, but should have]  09b special combination; failed;  breaking")
                                            break
                                    else:  # [if a or b, and a is false, then if b is also false: bad]
                                        book_is_a_keeper = False
                                        pass_or_die = False
                                        #~ if DEBUG: print("[if a or b, and a is false, then if b is also false: bad]10 book_is_a_keeper = False; breaking")
                                        break
                            else:  # "AND"          [previous andornot had to have been AND]
                                if is_special_combination:       # changed in v2.0.36
                                    if isnot == "IS NOT":     # FALSE
                                        if sqloperator == "FALSE":     # changed in v2.0.36
                                            #~ if DEBUG: print("[20 got nothing, which is correct] ")
                                            pass_or_die = False
                                            pass
                                        elif sqloperator == "LIKE":  # got nothing that IS NOT LIKE, which by definition is a bad thing...
                                            #~ if DEBUG: print("Got Nothing - IS NOT LIKE - [if it got nothing, then it should not have, so reject this book] 20a reject as a valid filter rejection")
                                            book_is_a_keeper = False
                                            pass_or_die = False
                                            break
                                        else: #  # changed in v2.0.36
                                            book_is_a_keeper = False
                                            pass_or_die = False
                                            #~ if DEBUG: print("[20b got nothing, which is not correct] special combination; failed;  breaking")
                                            break
                                    else:  # IS = TRUE [but got nothing, which is not correct]
                                        book_is_a_keeper = False
                                        pass_or_die = False
                                        #~ if DEBUG: print("[20c got nothing, which is not correct] special combination; failed;  breaking")
                                        break
                                else:
                                    #~ if DEBUG: print("[20d previous andornot was AND; got nothing, but should have] book_is_a_keeper = False; breaking")
                                    book_is_a_keeper = False
                                    pass_or_die = False
                                    break
                            #---------------------------------------------------------------------------------------------
                    else:  # len(tmp_rows) > 0    [got something, but may or may not be a good thing]

                        pass_or_die = False  # [got something, so can reset pass or die for now]

                        if previous_andornot == "TOP":
                            if is_special_combination:
                                if isnot == "IS NOT":
                                    if sqloperator != "FALSE":       # IS NOT + NOT FALSE = NOT TRUE = FALSE     but got TRUE instead  [should not have gotten any results, but did]
                                        if next_andornot == "OR":   # [defer judgement until after next filter evaluated]
                                            #~ if DEBUG: print("[30 should not have gotten any results, but did.  however, deferring judgement until after next OR filter evaluated]")
                                            pass_or_die = True
                                            pass
                                        else:
                                            #~ if DEBUG: print("[should have, and did] previous_andornot == TOP;  30a1 special combination pass")
                                            pass_or_die = False
                                            pass
                                    else:  # == FALSE  + IS NOT  = TRUE [should have, and did]
                                        #~ if DEBUG: print("[should have, and did] previous_andornot == TOP;  30a2 special combination pass")
                                        pass
                                else:  #  IS = TRUE [should have, and did]
                                    #~ if DEBUG: print("[should have, and did] previous_andornot == TOP; 30b special combination pass")
                                    pass
                            else:
                                if sqloperator != "FALSE":  # TRUE [should have, and did]
                                    #~ if DEBUG: print("[should have, and did]  30c previous_andornot == TOP; pass")
                                    pass
                                else: # == FALSE  but got TRUE  [should not have gotten any results, but did]
                                    #~ if DEBUG: print("[should not have gotten any results, but did] 30d ; breaking")
                                    book_is_a_keeper = False
                                    pass_or_die = False
                                    break
                        elif previous_andornot == "AND":
                            if is_special_combination:       # changed in v2.0.36
                                if isnot == "IS NOT":     # FALSE
                                    if sqloperator == "FALSE":    # FALSE + FALSE = TRUE
                                        #~ if DEBUG: print("IS NOT FALSE - [should have, and did] 35 special combination previous_andornot == AND; pass")
                                        pass
                                    elif sqloperator == "LIKE":  # got something that IS NOT LIKE, which by definition is a good thing...
                                        #~ if DEBUG: print("IS NOT LIKE - [if it got something, then it should have, and did] 35a special combination previous_andornot == AND; pass")
                                        pass
                                    elif sqloperator == "REGEX":  # got something that IS NOT A MATCH, which by definition is a good thing...
                                        #~ if DEBUG: print("IS NOT REGEX- [if it got something, then it should have, and did] 35a special combination previous_andornot == AND; pass")
                                        pass
                                    else: # TRUE   + TRUE = FALSE   but got TRUE  [should not have gotten any results, but did]
                                        #~ if DEBUG: print("[should not have gotten any results, but did] 35b special combination; breaking")
                                        book_is_a_keeper = False
                                        pass_or_die = False
                                        break
                                else:  # IS = TRUE
                                    if sqloperator == "FALSE":    # FALSE + TRUE = FALSE
                                        #~ if DEBUG: print("[should not have gotten any results, but did] 35c special combination; book_is_a_keeper = False; breaking")
                                        book_is_a_keeper = False
                                        pass_or_die = False
                                        break
                                    else: # TRUE    + TRUE =  TRUE [should have, and did]
                                        #~ if DEBUG: print("[should have, and did] 35d special combination previous_andornot == AND; pass")
                                        pass
                            else:
                                if sqloperator != "FALSE":  # TRUE [should have, and did]
                                    #~ if DEBUG: print("[should have, and did]  35e previous_andornot == AND; pass")
                                    pass
                                else: # == FALSE  but got TRUE  [should not have gotten any results, but did]
                                    #~ if DEBUG: print("[should not have gotten any results, but did] 35f ; breaking")
                                    book_is_a_keeper = False
                                    pass_or_die = False
                                    break
                        elif previous_andornot == "OR":  # [if a or b, and a is true, then b does not matter, but if a is false, then b matters.]
                            if is_special_combination:
                                if sqloperator != "FALSE":   # so NOT FALSE = TRUE
                                    if isnot == "IS":   # == TRUE    + TRUE  =  TRUE  [should have, and did]
                                        #~ if DEBUG: print("[should have, and did] 36 special combination; previous_andornot == OR; ")
                                        pass
                                    else:  #so == IS NOT   +  NOT FALSE  =  NOT TRUE = FALSE but got TRUE [should not have gotten any results, but did.  BUT since an OR, it does not matter since NOT pass_or_die]
                                        #~ if DEBUG: print("[should not have gotten any results, but did.  BUT since an OR, it does not matter since NOT pass_or_die] 36b special combination; passing...")
                                        pass
                                else: # so == FALSE  +  OR  =  FALSE  but got TRUE [should not have gotten any results, but did]
                                    #~ if DEBUG: print("36c special combination; book_is_a_keeper = False; breaking")
                                    book_is_a_keeper = False
                                    pass_or_die = False
                                    break
                            else:
                                if sqloperator != "FALSE":  # TRUE [should have, and did]
                                    #~ if DEBUG: print("[if a or b, and a is true, then b does not matter, but if a is false, then b matters.] 37 previous_andornot == OR: book_is_a_keeper = True")
                                    pass
                                else: # == FALSE  but got TRUE  [should not have gotten any results, but did]
                                    #~ if DEBUG: print("[should not have gotten any results, but did] 37a ; breaking")
                                    book_is_a_keeper = False
                                    pass_or_die = False
                                    break
                        elif previous_andornot == "NOT":
                            if is_special_combination:
                                if sqloperator != "FALSE":   # NOT FALSE = TRUE
                                    if isnot == "IS":   # NOT + TRUE = FALSE   but got TRUE [should not have gotten any results, but did]
                                        #~ if DEBUG: print("40 special combination; book_is_a_keeper = False; breaking")
                                        book_is_a_keeper = False
                                        pass_or_die = False
                                        break
                                    else:  #so == IS NOT   + TRUE  + NOT  =  TRUE  [should have, and did]
                                        #~ if DEBUG: print("[should have, and did] 40a special combination; previous_andornot == NOT; ")
                                        pass
                                else:  # so == FALSE    + NOT = TRUE [should have, and did]
                                    #~ if DEBUG: print("[should have, and did] 40b special combination; previous_andornot == NOT; ")
                                    pass
                            else:
                                if sqloperator != "FALSE":  # TRUE   + NOT = FALSE but got TRUE [should not have gotten any results, but did]
                                    #~ if DEBUG: print("[should not have gotten any results, but did] 50a ; breaking")
                                    book_is_a_keeper = False
                                    pass_or_die = False
                                    break
                                else: # == FALSE    + NOT = TRUE  [should have, and did]
                                    #~ if DEBUG: print("[should have, and did] 50b ; previous_andornot == NOT: book_is_a_keeper = True")
                                    pass
                        else:
                            pass
                    #------------------------------------------------------------------------------------------------
                    previous_andornot = next_andornot
                    if n_count == n_number_of_filters:
                        break
                #END FOR
                if book_is_a_keeper:
                    new_found_list.append(book)
                    #~ if DEBUG: print("--------------------------------")
                    #~ if DEBUG: print("book id added to new_found_list: ", as_unicode(book))
                    #~ if DEBUG: print("--------------------------------")
            #END FOR
            my_db.close()
            del found_list
            del mysql_list
            return new_found_list
        except Exception as e:
            my_db.close()
            if DEBUG: print("Exception:=================")
            if DEBUG: print(mysql)
            if DEBUG: print(as_unicode(e))
            if DEBUG: print("Exception:=================")
            del mysql_list
            del new_found_list
            return found_list
    #-----------------------------------------------------------------------------------------
    def finalize_mysql(self,mysql,isnot):

        next_andornot = ""      #  END_OF_QUERY
        if mysql.endswith("|AND"):
            mysql = mysql[0:-4]
            next_andornot = "AND"
        if mysql.endswith("|OR"):
            mysql = mysql[0:-3]
            next_andornot = "OR"
        if mysql.endswith("|NOT"):
            mysql = mysql[0:-4]
            next_andornot = "NOT"

        mysql = mysql.replace("has_cover  NOT TRUE","has_cover = 0 ")
        mysql = mysql.replace("has_cover  IS TRUE","has_cover = 1 ")
        mysql = mysql.replace("has_cover  NOT FALSE","has_cover = 1 ")
        mysql = mysql.replace("has_cover  IS FALSE","has_cover = 0 ")

        if "REGEXP" in mysql:
            if isnot == "IS NOT":
                mysql = mysql.replace("[IS_ISNOT]","NOT")
            else:
                mysql = mysql.replace("[IS_ISNOT]","")

        #~ if DEBUG: print("final mysql: ", mysql, "   >>>>  ", next_andornot)

        return mysql,next_andornot
    #-----------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------
    def special_query_control(self,guidb,param_dict,current_path):
        # totally unrelated to the primary search dialog logic, above.  always all books and across all books.

        #~ if DEBUG: print("special_query_control")
        #~ for k,v in iteritems(param_dict):
            #~ if DEBUG: print(k,v)

        self.gui.status_bar.showMessage("MCS Special Query: Executing Search")
        total_start = time.time()

        self.sqtable1a = param_dict['TABLE_1_1']
        self.sqoperator1 = as_unicode(param_dict['OPERATOR_1'])
        self.sqtable1b = param_dict['TABLE_1_2']

        self.sqandorinactive1 = param_dict['SPECIAL_QUERY_AND_OR_INACTIVE_1']
        if param_dict['SPECIAL_QUERY_AND_OR_INACTIVE_1'] == unicode_type("INACTIVE"):
            self.sq1_is_inactive = True
            self.sq2_is_inactive = True
        else:
            self.sq1_is_inactive = False

        self.sqtable2a = param_dict['TABLE_2_1']
        self.sqoperator2 = as_unicode(param_dict['OPERATOR_2'])
        self.sqtable2b = param_dict['TABLE_2_2']

        self.sqandorinactive2 = param_dict['SPECIAL_QUERY_AND_OR_INACTIVE_2']

        self.sqtable3a = param_dict['TABLE_3_1']
        self.sqoperator3 = as_unicode(param_dict['OPERATOR_3'])
        self.sqtable3b = param_dict['TABLE_3_2']

        self.sqandorinactive2 = param_dict['SPECIAL_QUERY_AND_OR_INACTIVE_2']
        if param_dict['SPECIAL_QUERY_AND_OR_INACTIVE_2'] == unicode_type("INACTIVE"):
            self.sq2_is_inactive = True
        else:
            if not self.sq1_is_inactive:
                self.sq2_is_inactive = False


        if param_dict['ALL_AUTHORS_BOOKS'] == unicode_type("True"):
            self.sq_use_all_authors_books = True
        else:
            self.sq_use_all_authors_books = False

        if param_dict['USE_FINAL_FILTERS'] == unicode_type("True"):
            self.sq_use_final_filters = True
        else:
            self.sq_use_final_filters = False

        my_db,my_cursor = self.apsw_connect_to_current_guidb()

        need_temp_views = False

        if self.sqtable1a == 'series: name' or self.sqtable1b == 'series: name' :
            need_temp_views = True

        if not self.sq1_is_inactive:
            if self.sqtable2a == 'series: name' or self.sqtable2b == 'series: name' :
                need_temp_views = True

        if not self.sq2_is_inactive:
            if self.sqtable3a == 'series: name' or self.sqtable3b == 'series: name' :
                need_temp_views = True

        if need_temp_views:
            self.create_temp_views(my_db,my_cursor)

        self.display_progress_bar(0)  # never display it for Special Queries

        found_list = self.execute_special_query_sql_control(my_db,my_cursor)

        my_db.close()

        found_set = set(found_list)
        found_list = list(found_set)
        del found_set
        found_list.sort()
        #-----------------------------

        if self.sq_use_all_authors_books:
            self.gui.status_bar.showMessage("MCS Special Query: Adding All Authors' Books")
            found_list = self.add_all_authors_books(found_list)
            my_db.close()
        found_set = set(found_list)
        found_list = list(found_set)
        del found_set
        found_list.sort()

        if self.sq_use_final_filters:
            self.gui.status_bar.showMessage("MCS Special Query: Applying Final Filters")
            self.param_dict = {}
            #~ for k,v in prefs.iteritems():
            for k,v in iteritems(prefs):
                self.param_dict[k] = v
            found_list = self.apply_final_filters_control(found_list)
            my_db.close()
            del self.param_dict

        del param_dict

        found_set = set(found_list)
        found_list = list(found_set)
        del found_set

        self.mark_found_books(found_list)

        #-----------------------------------------
        total_end = time.time()
        total_elapsed = total_end - total_start
        msg = 'Special Query: searched all books, found %d match(es) in %.3f seconds' %(len(found_list), total_elapsed)
        self.gui.status_bar.showMessage(msg)
        #-----------------------------------------
        del found_list
    #-----------------------------------------------------------------------------------------
    def execute_special_query_sql_control(self,my_db,my_cursor):

        try:
            del books_found_list_1
        except:
            pass
        books_found_list_1 = []

        #---------------------------------------
        # First Criteria Row
        #---------------------------------------
        self.sqtable1 = self.sqtable1a
        self.sqoperator = self.sqoperator1
        self.sqtable2 = self.sqtable1b

        if self.sqtable2 == 'authors: name':
            books_found_list_1 = self.do_sqtable2_authors_name(my_db,my_cursor)
        elif self.sqtable2 == 'books: title':
            books_found_list_1 = self.do_sqtable2_books_title(my_db,my_cursor)
        elif self.sqtable2 == 'series: name':
            books_found_list_1 = self.do_sqtable2_series_name(my_db,my_cursor)
        elif self.sqtable2.count("#") > 0:
            books_found_list_1 = self.do_sqtable2_text_custom_column(my_db,my_cursor)
        else:
            pass

        if self.sq1_is_inactive:
            #~ if DEBUG: print("books_found_list_1: ", as_unicode(len(books_found_list_1)))
            #~ if DEBUG: print("=====================================")
            return books_found_list_1

        #---------------------------------------
        #---------------------------------------
        # Second Criteria Row
        #---------------------------------------
        self.sqtable1 = self.sqtable2a
        self.sqoperator = self.sqoperator2
        self.sqtable2 = self.sqtable2b

        books_founds_list_2 = []

        if self.sqtable2 == 'authors: name':
            books_found_list_2 = self.do_sqtable2_authors_name(my_db,my_cursor)
        elif self.sqtable2 == 'books: title':
            books_found_list_2 = self.do_sqtable2_books_title(my_db,my_cursor)
        elif self.sqtable2 == 'series: name':
            books_found_list_2 = self.do_sqtable2_series_name(my_db,my_cursor)
        elif self.sqtable2.count("#") > 0:
            books_found_list_2 = self.do_sqtable2_text_custom_column(my_db,my_cursor)
        else:
            pass

        if self.sq2_is_inactive:
            books_found_list_3 = None
            books_found_list_final = self.apply_and_or_criteria(books_found_list_1,books_found_list_2,books_found_list_3)
            del books_found_list_1
            del books_found_list_2
            del books_found_list_3
            return books_found_list_final
        else:
            pass  #defer judgement until after the Third Criteria Row

       #---------------------------------------
        #---------------------------------------
        # Third Criteria Row
        #---------------------------------------
        self.sqtable1 = self.sqtable3a
        self.sqoperator = self.sqoperator3
        self.sqtable2 = self.sqtable3b

        books_found_list_3 = []

        if self.sqtable2 == 'authors: name':
            books_found_list_3 = self.do_sqtable2_authors_name(my_db,my_cursor)
        elif self.sqtable2 == 'books: title':
            books_found_list_3 = self.do_sqtable2_books_title(my_db,my_cursor)
        elif self.sqtable2 == 'series: name':
            books_found_list_3 = self.do_sqtable2_series_name(my_db,my_cursor)
        elif self.sqtable2.count("#") > 0:
            books_found_list_3 = self.do_sqtable2_text_custom_column(my_db,my_cursor)
        else:
            pass
        #---------------------------------------
        books_found_list_final = self.apply_and_or_criteria(books_found_list_1,books_found_list_2,books_found_list_3)
        del books_found_list_1
        del books_found_list_2
        del books_found_list_3
        return books_found_list_final

    #-----------------------------------------------------------------------------------------
    def apply_and_or_criteria(self,books_found_list_1,books_found_list_2,books_found_list_3):
        try:
            del books_found_list_final
        except:
            pass
        books_found_list_final = []

        if self.sq2_is_inactive:  # just compare 1 and 2
            #~ if DEBUG: print("books_found_list_1: ", as_unicode(len(books_found_list_1)))
            #~ if DEBUG: print("books_found_list_2: ", as_unicode(len(books_found_list_2)))
            #~ if DEBUG: print("=====================================")
            if self.sqandorinactive1 == unicode_type("AND"):
                for book in books_found_list_1:
                    if book in books_found_list_2:
                        books_found_list_final.append(book)
                #END FOR
                return books_found_list_final
            elif self.sqandorinactive1 == unicode_type("OR"):
                for book in books_found_list_1:
                    books_found_list_final.append(book)
                for book in books_found_list_2:
                    books_found_list_final.append(book)
                #END FOR
                return books_found_list_final
            else:
                return books_found_list_final
        else:  # compare 1, 2, 3
            if not books_found_list_3:
                books_found_list_3 = []
            #~ if DEBUG: print("books_found_list_1: ", as_unicode(len(books_found_list_1)))
            #~ if DEBUG: print("books_found_list_2: ", as_unicode(len(books_found_list_2)))
            #~ if DEBUG: print("books_found_list_3: ", as_unicode(len(books_found_list_3)))
            #~ if DEBUG: print("=====================================")
            if self.sqandorinactive1 == unicode_type("AND"):
                for book in books_found_list_1:
                    if book in books_found_list_2:
                        if book in books_found_list_3:
                            books_found_list_final.append(book)
                #END FOR
                return books_found_list_final
            elif self.sqandorinactive1 == unicode_type("OR"):
                for book in books_found_list_1:
                    books_found_list_final.append(book)
                for book in books_found_list_2:
                    books_found_list_final.append(book)
                for book in books_found_list_3:
                    books_found_list_final.append(book)
                #END FOR
                return books_found_list_final
            else:
                return books_found_list_final
    #-----------------------------------------------------------------------------------------
    def do_sqtable2_authors_name(self,my_db,my_cursor):

        try:
            del books_found_list
        except:
            pass

        books_found_list = []

        mysql = "SELECT id,name FROM authors"
        my_cursor.execute(mysql)
        all_authors_list = my_cursor.fetchall()
        if not all_authors_list:
            return books_found_list

        if self.sqtable1.count("#") > 0:
            label = self.sqtable1.replace("cc: ","").strip()
            if label in self.custom_column_label_dict:
                id = self.custom_column_label_dict[label]
                id = as_unicode(id)
                custom_column_N1 = "custom_column_[N1]".replace("[N1]",id)
                books_custom_column_N1_link = "books_custom_column_[N1]_link".replace("[N1]",id)
                #~ if DEBUG: print("self.sqtable1: ", label, custom_column_N1,books_custom_column_N1_link)
            else:
                return books_found_list

        # query: is the author name contained in the book title?
        if self.sqtable1 == "books: title":
            if self.sqoperator == "CONTAINS":
                mysql = "SELECT id,title FROM books WHERE title LIKE  ? AND title NOT NULL"        #  ('%'+name+'%')
            elif self.sqoperator == "EQUALS":
                mysql = "SELECT id,title FROM books WHERE title = ? AND title NOT NULL"
            else:
                mysql = "SELECT id,title FROM books WHERE title LIKE ? AND title NOT NULL"    # default
            for row in all_authors_list:
                dummy,name = row
                if not name:
                    continue
                if "LIKE" in mysql:
                    if name.count("_") > 0 or name.count("%") > 0:
                        continue
                    my_cursor.execute(mysql,(['%'+name+'%']))
                else:
                    my_cursor.execute(mysql,([name]))
                match_list = my_cursor.fetchall()
                if not match_list:
                    match_list = []
                for item in match_list:
                    id,title = item
                    #~ if DEBUG: print("book,title,author: ", as_unicode(id), title," -- ", name)
                    books_found_list.append(id)
                #END FOR
                del match_list
            #END FOR
        # query: is the author name contained in the series name?
        elif self.sqtable1 == "series: name":
            if self.sqoperator == "CONTAINS":
                mysql = "SELECT book,book FROM __mcs_series_by_book WHERE seriesname LIKE  ? AND seriesname NOT NULL"
            elif self.sqoperator == "EQUALS":
                mysql = "SELECT book,book FROM __mcs_series_by_book WHERE seriesname = ? AND seriesname NOT NULL"
            else:
                mysql = "SELECT book,book FROM __mcs_series_by_book WHERE seriesname LIKE  ?  AND seriesname NOT NULL"    # default
            for row in all_authors_list:
                dummy,name = row
                if not name:
                    continue
                if "LIKE" in mysql:
                    if name.count("_") > 0 or name.count("%") > 0:
                        continue
                    my_cursor.execute(mysql,(['%'+name+'%']))
                else:
                    my_cursor.execute(mysql,([name]))
                match_list = my_cursor.fetchall()
                if not match_list:
                    match_list = []
                for item in match_list:
                    book,name = item
                    books_found_list.append(book)
                #END FOR
                del match_list
            #END FOR
        # query: is the author name contained in the custom column value?
        elif self.sqtable1.count("#") > 0:
            if self.sqoperator == "CONTAINS":
                mysql = "SELECT book,value FROM [CCLINK1] WHERE value IN (SELECT id FROM [CUSTOM_COLUMN_N1] WHERE [CUSTOM_COLUMN_N1].value LIKE  ? ) AND book NOT NULL"
            elif self.sqoperator == "EQUALS":
                mysql = "SELECT book,value FROM [CCLINK1] WHERE value IN (SELECT id FROM [CUSTOM_COLUMN_N1] WHERE [CUSTOM_COLUMN_N1].value = ? ) AND book NOT NULL "
            else:
                mysql = "SELECT book,value FROM [CCLINK1] WHERE value IN (SELECT id FROM [CUSTOM_COLUMN_N1] WHERE [CUSTOM_COLUMN_N1].value LIKE  ? ) AND book NOT NULL "         # default
            mysql = mysql.replace("[CCLINK1]",books_custom_column_N1_link)
            mysql = mysql.replace("[CUSTOM_COLUMN_N1]",custom_column_N1)
            for row in all_authors_list:
                dummy,name = row
                if not name:
                    continue
                if "LIKE" in mysql:
                    if name.count("_") > 0 or name.count("%") > 0:
                        continue
                    my_cursor.execute(mysql,(['%'+name+'%']))
                else:
                    my_cursor.execute(mysql,([name]))
                match_list = my_cursor.fetchall()
                if not match_list:
                    match_list = []
                for item in match_list:
                    id,dummy = item
                    books_found_list.append(id)
                #END FOR
                del match_list
            #END FOR
        else:
            pass

        del all_authors_list

        return books_found_list
    #-----------------------------------------------------------------------------------------
    def do_sqtable2_books_title(self,my_db,my_cursor):

        try:
            del books_found_list
        except:
            pass

        books_found_list = []

        mysql = "SELECT id,title FROM books"
        my_cursor.execute(mysql)
        all_titles_list = my_cursor.fetchall()
        if not all_titles_list:
            return books_found_list

        if self.sqtable1.count("#") > 0:
            label = self.sqtable1.replace("cc: ","").strip()
            if label in self.custom_column_label_dict:
                id = self.custom_column_label_dict[label]
                id = as_unicode(id)
                custom_column_N1 = "custom_column_[N1]".replace("[N1]",id)
                books_custom_column_N1_link = "books_custom_column_[N1]_link".replace("[N1]",id)
                #~ if DEBUG: print("self.sqtable1: ", label, custom_column_N1,books_custom_column_N1_link)
            else:
                return books_found_list

        # query: is the book title contained in the author name?
        if self.sqtable1 == "authors: name":
            if self.sqoperator == "CONTAINS":
                mysql = "SELECT book,book FROM _mcs_authors_by_book WHERE authorname LIKE  ?  AND authorname NOT NULL"
            elif self.sqoperator == "EQUALS":
                mysql = "SELECT book,book FROM _mcs_authors_by_book WHERE authorname = ?  AND authorname NOT NULL"
            else:
                mysql = "SELECT book,book FROM _mcs_authors_by_book WHERE authorname LIKE ?  AND authorname NOT NULL"    # default
            for row in all_titles_list:
                dummy,title = row
                if not title:
                    continue
                if "LIKE" in mysql:
                    my_cursor.execute(mysql,(['%'+title+'%']))
                else:
                    my_cursor.execute(mysql,([title]))
                match_list = my_cursor.fetchall()
                if not match_list:
                    match_list = []
                for item in match_list:
                    id,dummy = item
                    books_found_list.append(id)
                #END FOR
                del match_list
            #END FOR
        # query: is the book title contained in the series name?
        elif self.sqtable1 == "series: name":
            if self.sqoperator == "CONTAINS":
                mysql = "SELECT book,book FROM __mcs_series_by_book WHERE seriesname LIKE  ?  AND seriesname NOT NULL"
            elif self.sqoperator == "EQUALS":
                mysql = "SELECT book,book FROM __mcs_series_by_book WHERE seriesname = ?  AND seriesname NOT NULL"
            else:
                mysql = "SELECT book,book FROM __mcs_series_by_book WHERE seriesname LIKE ?  AND seriesname NOT NULL"    # default
            for row in all_titles_list:
                dummy,title = row
                if not title:
                    continue
                if "LIKE" in mysql:
                    my_cursor.execute(mysql,(['%'+title+'%']))
                else:
                    my_cursor.execute(mysql,([title]))
                match_list = my_cursor.fetchall()
                if not match_list:
                    match_list = []
                for item in match_list:
                    id,dummy = item
                    books_found_list.append(id)
                #END FOR
                del match_list
            #END FOR
        # query: is the author name contained in the custom column value?
        elif self.sqtable1.count("#") > 0:
            if self.sqoperator == "CONTAINS":
                mysql = "SELECT book,value FROM [CCLINK1] WHERE value IN (SELECT id FROM [CUSTOM_COLUMN_N1] WHERE [CUSTOM_COLUMN_N1].value LIKE  ? ) AND book NOT NULL"
            elif self.sqoperator == "EQUALS":
                mysql = "SELECT book,value FROM [CCLINK1] WHERE value IN (SELECT id FROM [CUSTOM_COLUMN_N1] WHERE [CUSTOM_COLUMN_N1].value = ? ) AND book NOT NULL "
            else:
                mysql = "SELECT book,value FROM [CCLINK1] WHERE value IN (SELECT id FROM [CUSTOM_COLUMN_N1] WHERE [CUSTOM_COLUMN_N1].value LIKE  ? ) AND book NOT NULL "         # default
            mysql = mysql.replace("[CCLINK1]",books_custom_column_N1_link)
            mysql = mysql.replace("[CUSTOM_COLUMN_N1]",custom_column_N1)
            for row in all_titles_list:
                dummy,title = row
                if not title:
                    continue
                if "LIKE" in mysql:
                    my_cursor.execute(mysql,(['%'+title+'%']))
                else:
                    my_cursor.execute(mysql,([title]))
                match_list = my_cursor.fetchall()
                if not match_list:
                    match_list = []
                for item in match_list:
                    id,dummy = item
                    books_found_list.append(id)
                #END FOR
                del match_list
            #END FOR
        else:
            pass

        del all_titles_list

        return books_found_list
    #-----------------------------------------------------------------------------------------
    def do_sqtable2_series_name(self,my_db,my_cursor):
        try:
            del books_found_list
        except:
            pass

        books_found_list = []

        mysql = "SELECT id,name FROM series"
        my_cursor.execute(mysql)
        all_series_list = my_cursor.fetchall()
        if not all_series_list:
            all_series_list = []

        if self.sqtable1.count("#") > 0:
            label = self.sqtable1.replace("cc: ","").strip()
            if label in self.custom_column_label_dict:
                id = self.custom_column_label_dict[label]
                id = as_unicode(id)
                custom_column_N1 = "custom_column_[N1]".replace("[N1]",id)
                books_custom_column_N1_link = "books_custom_column_[N1]_link".replace("[N1]",id)
                #~ if DEBUG: print("self.sqtable1: ", label, custom_column_N1,books_custom_column_N1_link)
            else:
                return books_found_list

        # query: is the series name contained in the book title?
        if self.sqtable1 == "books: title":
            if self.sqoperator == "CONTAINS":
                mysql = "SELECT id,title FROM books WHERE title LIKE  ?  AND title NOT NULL"
            elif self.sqoperator == "EQUALS":
                mysql = "SELECT id,title FROM books WHERE title = ? AND title NOT NULL"
            else:
                mysql = "SELECT id,title FROM books WHERE title LIKE ? AND title NOT NULL"    # default
            for row in all_series_list:
                dummy,name = row
                if not name:
                    continue
                if "LIKE" in mysql:
                    if name.count("_") > 0 or name.count("%") > 0:
                        continue
                    my_cursor.execute(mysql,(['%'+name+'%']))
                else:
                    my_cursor.execute(mysql,([name]))
                match_list = my_cursor.fetchall()
                if not match_list:
                    match_list = []
                for item in match_list:
                    id,dummy = item
                    books_found_list.append(id)
                #END FOR
                del match_list
            #END FOR
        # query: is the series name contained in the author name?
        elif self.sqtable1 == "authors: name":
            if self.sqoperator == "CONTAINS":
                mysql = "SELECT book,book FROM _mcs_authors_by_book WHERE authorname LIKE  ? AND authorname NOT NULL"
            elif self.sqoperator == "EQUALS":
                mysql = "SELECT book,book FROM _mcs_authors_by_book WHERE authorname = ? AND authorname NOT NULL"
            else:
                mysql = "SELECT book,book FROM _mcs_authors_by_book WHERE authorname LIKE  ? AND authorname NOT NULL"    # default
            for row in all_series_list:
                dummy,name = row
                if not name:
                    continue
                if "LIKE" in mysql:
                    if name.count("_") > 0 or name.count("%") > 0:
                        continue
                    my_cursor.execute(mysql,(['%'+name+'%']))
                else:
                    my_cursor.execute(mysql,([name]))
                match_list = my_cursor.fetchall()
                if not match_list:
                    match_list = []
                for item in match_list:
                    book,name = item
                    books_found_list.append(book)
                #END FOR
                del match_list
            #END FOR
        # query: is the author name contained in the custom column value?
        elif self.sqtable1.count("#") > 0:
            if self.sqoperator == "CONTAINS":
                mysql = "SELECT book,value FROM [CCLINK1] WHERE value IN (SELECT id FROM [CUSTOM_COLUMN_N1] WHERE [CUSTOM_COLUMN_N1].value LIKE  ? ) AND book NOT NULL"
            elif self.sqoperator == "EQUALS":
                mysql = "SELECT book,value FROM [CCLINK1] WHERE value IN (SELECT id FROM [CUSTOM_COLUMN_N1] WHERE [CUSTOM_COLUMN_N1].value = ? ) AND book NOT NULL"
            else:
                mysql = "SELECT book,value FROM [CCLINK1] WHERE value IN (SELECT id FROM [CUSTOM_COLUMN_N1] WHERE [CUSTOM_COLUMN_N1].value LIKE  ? ) AND book NOT NULL"        # default
            mysql = mysql.replace("[CCLINK1]",books_custom_column_N1_link)
            mysql = mysql.replace("[CUSTOM_COLUMN_N1]",custom_column_N1)
            for row in all_series_list:
                dummy,name = row
                if not name:
                    continue
                if "LIKE" in mysql:
                    if name.count("_") > 0 or name.count("%") > 0:
                        continue
                    my_cursor.execute(mysql,(['%'+name+'%']))
                else:
                    my_cursor.execute(mysql,([name]))
                match_list = my_cursor.fetchall()
                if not match_list:
                    match_list = []
                for item in match_list:
                    book,name = item
                    books_found_list.append(book)
                #END FOR
                del match_list
            #END FOR
        else:
            pass

        del all_series_list

        return books_found_list
    #-----------------------------------------------------------------------------------------
    def do_sqtable2_text_custom_column(self,my_db,my_cursor):

        try:
            del books_found_list
        except:
            pass

        books_found_list = []

        #~ self.custom_column_label_dict[#label] = id       label includes the #
        label2 = self.sqtable2.replace("cc: ","").strip()
        if label2 in self.custom_column_label_dict:
            id = self.custom_column_label_dict[label2]
            id = as_unicode(id)
        else:
            return books_found_list

        custom_column_N2 = "custom_column_[N2]".replace("[N2]",id)
        books_custom_column_N2_link = "books_custom_column_[N2]_link".replace("[N2]",id)

        if self.sqtable1.count("#") > 0:
            label1 = self.sqtable1.replace("cc: ","").strip()
            if label1 in self.custom_column_label_dict:
                id = self.custom_column_label_dict[label1]
                id = as_unicode(id)
                custom_column_N1 = "custom_column_[N1]".replace("[N1]",id)
                books_custom_column_N1_link = "books_custom_column_[N1]_link".replace("[N1]",id)
                #~ if DEBUG: print("self.sqtable1: ", label1, custom_column_N1,books_custom_column_N1_link)
            else:
                return books_found_list

        #~ if DEBUG: print("self.sqtable2: ", label2, custom_column_N2,books_custom_column_N2_link)

        mysql = "SELECT id,value FROM [custom_column_N2]"
        mysql = mysql.replace("[custom_column_N2]",custom_column_N2)
        my_cursor.execute(mysql)
        all_custom_column_n2_list = my_cursor.fetchall()
        if not all_custom_column_n2_list:
            return books_found_list

        # query: is the custom column value contained in the book title?
        if self.sqtable1 == "books: title":
            if self.sqoperator == "CONTAINS":
                mysql = "SELECT id,title FROM books WHERE title LIKE  ? AND title NOT NULL"
            elif self.sqoperator == "EQUALS":
                mysql = "SELECT id,title FROM books WHERE title = ? AND title NOT NULL"
            else:
                mysql = "SELECT id,title FROM books WHERE title LIKE ? AND title NOT NULL"    # default
            for row in all_custom_column_n2_list:
                dummy,value = row
                if not value:
                    continue
                if "LIKE" in mysql:
                    my_cursor.execute(mysql,(['%'+value+'%']))
                else:
                    my_cursor.execute(mysql,([value]))
                match_list = my_cursor.fetchall()
                if not match_list:
                    match_list = []
                for item in match_list:
                    id,dummy = item
                    books_found_list.append(id)
                #END FOR
                del match_list
            #END FOR
        # query: is the custom column value contained in the author name?
        elif self.sqtable1 == "authors: name":
            if self.sqoperator == "CONTAINS":
                mysql = "SELECT book,book FROM _mcs_authors_by_book WHERE authorname LIKE  ? AND authorname NOT NULL"
            elif self.sqoperator == "EQUALS":
                mysql = "SELECT book,book FROM _mcs_authors_by_book WHERE authorname = ? AND authorname NOT NULL"
            else:
                mysql = "SELECT book,book FROM _mcs_authors_by_book WHERE authorname LIKE  ? AND authorname NOT NULL"     # default
            for row in all_custom_column_n2_list:
                dummy,value = row
                if not value:
                    continue
                if "LIKE" in mysql:
                    my_cursor.execute(mysql,(['%'+value+'%']))
                else:
                    my_cursor.execute(mysql,([value]))
                match_list = my_cursor.fetchall()
                if not match_list:
                    match_list = []
                for item in match_list:
                    id,dummy = item
                    books_found_list.append(id)
                #END FOR
                del match_list
            #END FOR
        # query: is the custom column value contained in the series name?
        elif self.sqtable1 == "series: name":
            if self.sqoperator == "CONTAINS":
                mysql = "SELECT book,book FROM __mcs_series_by_book WHERE seriesname LIKE  ? AND seriesname NOT NULL"
            elif self.sqoperator == "EQUALS":
                mysql = "SELECT book,book FROM __mcs_series_by_book WHERE seriesname = ? AND seriesname NOT NULL"
            else:
                mysql = "SELECT book,book FROM __mcs_series_by_book WHERE seriesname LIKE ? AND seriesname NOT NULL"    # default
            for row in all_custom_column_n2_list:
                dummy,value = row
                if not value:
                    continue
                if "LIKE" in mysql:
                    my_cursor.execute(mysql,(['%'+value+'%']))
                else:
                    my_cursor.execute(mysql,([value]))
                match_list = my_cursor.fetchall()
                if not match_list:
                    match_list = []
                for item in match_list:
                    id,dummy = item
                    books_found_list.append(id)
                #END FOR
                del match_list
            #END FOR
        elif self.sqtable1.count("#") > 0:
            if self.sqoperator == "CONTAINS":
                mysql = "SELECT book,value FROM [CCLINK1] WHERE value IN (SELECT id FROM [CUSTOM_COLUMN_N1] WHERE [CUSTOM_COLUMN_N1].value LIKE  ? ) AND book NOT NULL"
            elif self.sqoperator == "EQUALS":
                mysql = "SELECT book,value FROM [CCLINK1] WHERE value IN (SELECT id FROM [CUSTOM_COLUMN_N1] WHERE [CUSTOM_COLUMN_N1].value = ? ) AND book NOT NULL"
            else:
                mysql = "SELECT book,value FROM [CCLINK1] WHERE value IN (SELECT id FROM [CUSTOM_COLUMN_N1] WHERE [CUSTOM_COLUMN_N1].value LIKE  ? ) AND book NOT NULL"         # default
            mysql = mysql.replace("[CCLINK1]",books_custom_column_N1_link)
            mysql = mysql.replace("[CUSTOM_COLUMN_N1]",custom_column_N1)
            for row in all_custom_column_n2_list:
                dummy,value = row
                if not value:
                    continue
                if "LIKE" in mysql:
                    my_cursor.execute(mysql,(['%'+value+'%']))
                else:
                    my_cursor.execute(mysql,([value]))
                match_list = my_cursor.fetchall()
                if not match_list:
                    match_list = []
                for item in match_list:
                    id,dummy = item
                    books_found_list.append(id)
                #END FOR
                del match_list
            #END FOR
        else:
            pass

        del all_custom_column_n2_list

        return books_found_list
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def create_temp_views(self,my_db,my_cursor):
        # series does not have an accelerator table because there is only one series per book, maximum.
        my_cursor.execute("begin")
        mysql = "CREATE TEMP VIEW IF NOT EXISTS __mcs_series_by_book AS \
                        SELECT book, \
                        (SELECT name FROM series WHERE id = books_series_link.series AND name NOT NULL) AS seriesname \
                        FROM books_series_link WHERE book NOT NULL"
        my_cursor.execute(mysql)
        my_cursor.execute("commit")
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def raw_sql_query_control(self,guidb,param_dict,current_path):

        self.gui.status_bar.showMessage("MCS SQL Query: Executing Search")
        total_start = time.time()

        if param_dict['ALL_AUTHORS_BOOKS'] == unicode_type("True"):
            self.sq_use_all_authors_books = True
        else:
            self.sq_use_all_authors_books = False

        if param_dict['USE_FINAL_FILTERS'] == unicode_type("True"):
            self.sq_use_final_filters = True
        else:
            self.sq_use_final_filters = False

        self.display_progress_bar(0)  # never display it for Raw SQL Queries

        my_db,my_cursor = self.apsw_connect_to_current_guidb()

        mysql = param_dict['RAW_SQL_QUERY']

        if mysql.count("REGEXP") > 0:
            func = "REGEXP"
            is_valid = self.apsw_create_user_functions(my_db,my_cursor,func)
        else:
            is_valid = True

        if mysql.count("SIMILARTO") > 0:
            func = "SIMILARTO"
            is_valid = self.apsw_create_user_functions(my_db,my_cursor,func)
        else:
            is_valid = True

        if mysql.count("PARSEJSON") > 0:
            func = "PARSEJSON"
            is_valid = self.apsw_create_user_functions(my_db,my_cursor,func)
        else:
            is_valid = True

        if is_valid:
            found_list = self.execute_raw_sql_query(my_db,my_cursor,mysql)
        else:
            found_list = []

        found_set = set(found_list)
        found_list = list(found_set)
        del found_set
        found_list.sort()
        #-----------------------------

        if self.sq_use_all_authors_books:
            self.gui.status_bar.showMessage("MCS Raw SQL Query: Adding All Authors' Books")
            found_list = self.add_all_authors_books(found_list)
            my_db.close()
        found_set = set(found_list)
        found_list = list(found_set)
        del found_set
        found_list.sort()

        if self.sq_use_final_filters:
            self.gui.status_bar.showMessage("MCS Raw SQL Query: Applying Final Filters")
            self.param_dict = {}
            #~ for k,v in prefs.iteritems():
            for k,v in iteritems(prefs):
                self.param_dict[k] = v
            found_list = self.apply_final_filters_control(found_list)
            my_db.close()
            del self.param_dict

        del param_dict

        found_set = set(found_list)
        found_list = list(found_set)
        del found_set

        self.mark_found_books(found_list)

        #-----------------------------------------
        total_end = time.time()
        total_elapsed = total_end - total_start
        msg = 'Raw SQL Query: searched all books, found %d match(es) in %.3f seconds' %(len(found_list), total_elapsed)
        self.gui.status_bar.showMessage(msg)
        #-----------------------------------------
        del found_list
    #-----------------------------------------------------------------------------------------
    def execute_raw_sql_query(self,my_db,my_cursor,mysql):

        found_list = []

        try:
            my_cursor.execute(mysql)
            tmp_rows = my_cursor.fetchall()
            if not tmp_rows:
                tmp_rows = []
            for row in tmp_rows:
                for col in row:
                    id = int(col)
                    found_list.append(id)
                    break
                #END FOR
            #END FOR
            my_db.close()
            return found_list
        except Exception as e:
            my_db.close()
            msg = "Raw SQL Query Fatal Error:   " + as_unicode(e)
            error_dialog(self.gui, _('MCS'),_((msg)), show=True)
            dummy_list = []
            return dummy_list
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def apsw_create_user_functions(self,my_db,my_cursor,func):
        try:
            if func == "REGEXP":
                my_db.createscalarfunction("regexp", self.apsw_user_function_regexp)
                #~ if DEBUG: print("Create_SQLite_User_Function REGEXP was successful...")
                return True
            if func == "SIMILARTO":
                my_db.createscalarfunction("similarto", self.apsw_user_function_similarto)
                #~ if DEBUG: print("Create_SQLite_User_Function SIMILARTO was successful...")
                return True
            if func == "PARSEJSON":
                self.blank_string_prefix = "                                                                                                                                                                 "
                self.blank_string_prefix = self.blank_string_prefix + self.blank_string_prefix
                my_db.createscalarfunction("parsejson", self.apsw_user_function_parsejson)
                #~ if DEBUG: print("Create_SQLite_User_Function PARSEJSON was successful...")
                return True

        except Exception as e:
            if DEBUG: print("Create_SQLite_User_Function [3] failed...cannot proceed...")
            if DEBUG: print(as_unicode(e))
            return False
    #-----------------------------------------------------------------------------------------
    def apsw_user_function_regexp(self,regexpr,avalue):
        #http://www.sqlite.org/lang_expr.html:  The "X REGEXP Y" operator will be implemented as a call to "regexp(Y,X)"
        #------------------------------------------------------------------------------------------------------------
        #mysql = 'SELECT id FROM custom_column_8 WHERE value REGEXP '^.+$'
        #------------------------------------------------------------------------------------------------------------
        if regexpr:
            if avalue:
                try:
                    s_string = unicode_type(avalue)
                    re_string = unicode_type(regexpr)
                    re.escape("\\")
                    p = re.compile(re_string, re.IGNORECASE|re.DOTALL|re.MULTILINE)
                    match = p.search(s_string)
                    if match:
                        return True
                    else:
                        return False
                except Exception as e:
                    if DEBUG: print(as_unicode(e))
                    return False
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def apsw_user_function_similarto(self,a,b):
        #----------------------------------------------------------------
        #returns the probability the string a is the same as string b
        #~ SELECT book FROM _mcs_authors_by_book,_mcs_book_awards WHERE award LIKE '%hugo%' AND subas_unicode(authorname,1,3) = subas_unicode(author,1,3)  AND SIMILARTO(authorname,author) > 0.80 AND _mcs_authors_by_book.book IN ( SELECT id FROM books WHERE books.id = _mcs_authors_by_book.book AND SIMILARTO(books.title,_mcs_book_awards.title) > 0.80)
        #----------------------------------------------------------------
        try:
            p = SequenceMatcher(None, a, b).ratio()
            if p:
                return p
            else:
                return 0.0000
        except Exception as e:
            if DEBUG: print(as_unicode(e))
            return  0.0000
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def apsw_user_function_parsejson(self,raw=None,key=None,op=None,value=None):
        #------------------------------------------------------------------------------------------------------------
        #~ Mode:  Do not compare anything; only return the value loaded from JSON...
        #------------------------------------------------------------------------------------------------------------
                #~ SELECT book FROM books_plugin_data
                #~ WHERE name = 'CompositeColumnValues'
                   #~ AND val IS NOT NULL
                   #~ AND PARSEJSON(val,'#isbn') LIKE '%55%'

                #~ SELECT book FROM books_plugin_data
                #~ WHERE name = 'CompositeColumnValues'
                   #~ AND val IS NOT NULL
                   #~ AND (PARSEJSON(val,'#mydate1) > (SELECT timestamp FROM books WHERE id = books_plugin_data.book) )
        #------------------------------------------------------------------------------------------------------------
        #------------------------------------------------------------------------------------------------------------
        #~ Mode: Compare all values as text and return True or False...
        #------------------------------------------------------------------------------------------------------------
                #~ SELECT book FROM books_plugin_data
                #~ WHERE name = 'CompositeColumnValues'
                   #~ AND val IS NOT NULL
                   #~ AND PARSEJSON(val,'#mysize','>=','188282') IS TRUE
        #------------------------------------------------------------------------------------------------------------
        #------------------------------------------------------------------------------------------------------------
        if not raw:
            return None

        if not key:
            return None

        if not isinstance(raw, unicode_type):
            raw = as_unicode(raw)

        if not isinstance(key, unicode_type):
            key = as_unicode(key)

        if op is None or value is None or op == "" or value == "":
            #~ Mode:  Do not compare anything; only return the value loaded from JSON...
            raw_dict = json_loads(raw)
            dictval = None
            if key in raw_dict:
                dictval = raw_dict[key]
            return dictval

        #~ Mode: Compare all values as text and return True or False...
        if not isinstance(op, unicode_type):
            op = as_unicode(op)
        op = op.strip()

        if not isinstance(value, unicode_type):
            value = as_unicode(value)
        value = value.strip()

        try:
            found = False
            raw_dict = json_loads(raw)
            if key in raw_dict:
                dictval = raw_dict[key]
                if not isinstance(dictval, unicode_type):
                    dictval = as_unicode(dictval)
                dictval = dictval.strip()
                if dictval is not None:
                    ndv = len(dictval)      # "12345"              = 5
                    nval = len(value)       # "12345678"        = 8
                    if ndv > nval:
                        n = ndv - nval
                        fill = self.blank_string_prefix[0:n]
                        value = fill + value
                    else:
                        n = nval - ndv  # 8 - 5
                        fill = self.blank_string_prefix[0:n]  # 3
                        dictval = fill + dictval

                    if DEBUG:
                        print("value:", value, "   dictval:",dictval)
                        if len(value) != len(dictval):
                            print("lengths of strings are different:  Program Error...")

                    if len(value) != len(dictval):
                        return False

                    if op == "=":
                        if dictval == value:
                            found = True
                    elif op == "!=" or op == "<>":
                        if dictval != value:
                            found = True
                    elif op == ">":
                        if dictval > value:
                            found = True
                    elif op == "<":
                        if dictval < value:
                            found = True
                    elif op == ">=":
                        if dictval >= value:
                            found = True
                    elif op == "<=":
                        if dictval <= value:
                            found = True

            if DEBUG: print("found?  ", found)
            return found
        except Exception as e:
            if DEBUG: print(as_unicode(e))
            return False

    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def search_virtual_column_format_txt_control(self,guidb,param_dict,sel_type):

        #----------------------------------------
        if not self.maingui:
            from calibre.gui2.ui import get_gui
            self.maingui = get_gui()
         #----------------------------------------

        self.gui.status_bar.showMessage("MCS TXT Search: Executing Search")
        total_start = time.time()

        if param_dict['ALL_AUTHORS_BOOKS'] == unicode_type("True"):
            self.sq_use_all_authors_books = True
        else:
            self.sq_use_all_authors_books = False

        if param_dict['USE_FINAL_FILTERS'] == unicode_type("True"):
            self.sq_use_final_filters = True
        else:
            self.sq_use_final_filters = False


        if param_dict['OPTIONS_TEXT_ACTIVATE'] == unicode_type("True"):
            self.options_text_activate = True
        else:
            self.options_text_activate = False


        self.display_progress_bar(0)  # never display it for this

        txt_search_terms = param_dict['TXT_SEARCH_TERMS']

        selected_books_list = self.get_selected_books(guidb,sel_type)

        found_list = []

        self.found_books_update_dict = {}

        my_db,my_cursor = self.apsw_connect_to_current_guidb()

        for current_book in selected_books_list:
            full_book_path = self.build_book_path(my_db,my_cursor,current_book)
            if not full_book_path:
                continue
            was_found,re_text_found = self.search_single_format_txt_control(full_book_path,txt_search_terms)
            if was_found:
                if self.options_text_activate:
                    was_found = False
                    was_found = self.options_text_control(my_db,my_cursor,param_dict,current_book,re_text_found)
                    if was_found:
                        found_list.append(current_book)
                else:
                    found_list.append(current_book)
        #END FOR
        my_db.close()

        found_set = set(found_list)
        found_list = list(found_set)
        del found_set
        found_list.sort()

        if self.options_text_activate:
            if param_dict['OPTIONS_UPDATE_FOUND_TEXT'] == unicode_type("True"):
                if "#" in param_dict['OPTIONS_UPDATE_FOUND_TEXT_CC']:
                    if len(self.found_books_update_dict) > 0:
                        self.options_text_update_cc_with_results(param_dict)

        del self.found_books_update_dict

        #-----------------------------

        if self.sq_use_all_authors_books:
            self.gui.status_bar.showMessage("MCS TXT Search: Adding All Authors' Books")
            found_list = self.add_all_authors_books(found_list)
            my_db.close()
        found_set = set(found_list)
        found_list = list(found_set)
        del found_set
        found_list.sort()

        if self.sq_use_final_filters:
            self.gui.status_bar.showMessage("MCS TXT Search: Applying Final Filters")
            self.param_dict = {}
            #~ for k,v in prefs.iteritems():
            for k,v in iteritems(prefs):
                self.param_dict[k] = v
            found_list = self.apply_final_filters_control(found_list)
            my_db.close()
            del self.param_dict

        del param_dict

        found_set = set(found_list)
        found_list = list(found_set)
        del found_set

        self.mark_found_books(found_list)

        try:
            self.mcs_dialog.show()
        except:
            pass

        #-----------------------------------------
        total_end = time.time()
        total_elapsed = total_end - total_start
        msg = 'MCS TXT Search: searched selected books, found %d match(es) in %.3f seconds' %(len(found_list), total_elapsed)
        self.gui.status_bar.showMessage(msg)
        #-----------------------------------------
        del found_list
    #-----------------------------------------------------------------------------------------
    def search_single_format_txt_control(self,path,txt_search_terms):

        file_data = self.load_text_file(path)
        if not file_data:
            return False,None

        if txt_search_terms:
            if file_data:
                try:
                    s = unicode_type("")
                    s_string = s.join(file_data)   # to not corrupt unicode text such as:  Mein schönes Fräulein
                    re_string = txt_search_terms
                    re.escape("\\")
                    p = re.compile(re_string, re.IGNORECASE|re.DOTALL|re.MULTILINE)
                    match = p.search(s_string)
                    if match:
                        re_text_found = match.group()
                        del match
                        return True,re_text_found
                    else:
                        del match
                        return False,None
                except Exception as e:
                    if DEBUG: print(as_unicode(e))
                    return False,None

        return False,None
    #-----------------------------------------------------------------------------------------
    def load_text_file(self,path):

        #~ if DEBUG: print("Loading text file: ", path)

        file_data = None

        try:
            with open(path, 'r') as f:
                file_data = f.readlines()
            f.close()
        except Exception as e:
            if DEBUG: print(as_unicode(e))

        try:
            del path
            del f
        except:
            pass

        return file_data
    #-----------------------------------------------------------------------------------------
    def build_book_path(self,my_db,my_cursor,current_book):

        full_book_path = None
        path_to_use = ""
        format_to_use = ""
        name = ""

        mysql = 'SELECT path FROM books WHERE id = ?'
        my_cursor.execute(mysql,([current_book]))
        tmp = my_cursor.fetchall()
        if not tmp:
            return full_book_path
        else:
            if len(tmp) == 0:
                return full_book_path
            else:
                for row in tmp:
                    for col in row:
                        path_to_use = col

        mysql = "SELECT format,name FROM data WHERE book = ? AND format = 'TXT' "
        my_cursor.execute(mysql,([current_book]))
        tmp = my_cursor.fetchall()
        if not tmp:
            pass
        else:
            if len(tmp) == 0:
                pass
            else:
                for row in tmp:
                    format_to_use,name = row
                    break
                del tmp

        if not format_to_use == "TXT":
            return full_book_path

        s_lower = format_to_use.lower()

        name = name + "." + s_lower

        path_to_use = os.path.join(path_to_use,name)

        full_book_path = os.path.join(self.lib_path,path_to_use)

        full_book_path = full_book_path.replace(os.sep,"/")

        del path_to_use
        del name
        del s_lower

        #~ if DEBUG: print("full_book_path: ", full_book_path)

        return full_book_path
    #-----------------------------------------------------------------------------------------
    def options_text_control(self,my_db,my_cursor,param_dict,current_book,re_text_found):

        was_found = True

        if DEBUG: print("current book: ", as_unicode(current_book), "re_text_found: ", re_text_found)

        if param_dict['OPTIONS_COMPARE_FOUND_TEXT'] == unicode_type("True"):
            compare_cc = param_dict['OPTIONS_COMPARE_FOUND_TEXT_CC']
            if DEBUG: print("compare cc: ", compare_cc)
            book = int(current_book)
            try:
                book_mi_object = self.guidb.get_metadata(book, index_is_id=True, get_cover=False, get_user_categories=True, cover_as_data=False)    # a :class:`Metadata` object.
            except:
                book_mi_object = self.guidb.get_metadata(book, index_is_id=False, get_cover=False, get_user_categories=True, cover_as_data=False)    # a :class:`Metadata` object.
            compare_to_value = book_mi_object.get(compare_cc)
            del book_mi_object
            if DEBUG: print("compare_to_value: ", compare_to_value)

            if param_dict['OPTIONS_COMPARE_FOUND_TEXT_REGEX'] > "":
                use_regex = True
                regex = param_dict['OPTIONS_COMPARE_FOUND_TEXT_REGEX']   # number to use, not the regex itself...
                if DEBUG: print("Compare CC is using regex: ", regex)
                regex_full = self.get_regex_full(regex)
            else:
                use_regex = False
                regex = None
                regex_full = None

            if not use_regex:
                if not compare_to_value:
                    was_found = False
                elif re_text_found == compare_to_value:
                    was_found = True
                else:
                    was_found = False
            else:
                if DEBUG: print("searching: ", re_text_found, " using regex: ", regex_full)
                re.escape("\\")
                p = re.compile(regex_full, re.IGNORECASE|re.DOTALL|re.MULTILINE)
                match = p.search(re_text_found)
                if match:
                    re_text_found_final = match.group()
                    if re_text_found_final == compare_to_value:
                        if DEBUG: print("Match found and final text to compare is: ", re_text_found_final, " with compare_to_value of: ", compare_to_value)
                        was_found = True
                    else:
                        if is_py3:
                            compare_to_value = as_unicode(compare_to_value)
                        else:
                            compare_to_value = as_bytes(compare_to_value)
                        try:
                            tag_like_list = ast.literal_eval(compare_to_value)          #[u'2131277']  meaning a Tag-like Custom Column...
                        except:
                            tag_like_list = None
                        if isinstance(tag_like_list,list):
                            was_found = False
                            for tag in tag_like_list:
                                if tag == re_text_found_final:
                                    was_found = True
                                    if DEBUG: print("Match found and final text to compare is: ", as_unicode(re_text_found_final), " with Tag-like compare_to_value of: ", as_unicode(tag))
                                    break
                            #END FOR
                        else:
                            if DEBUG: print("Match found but re_text_found_final != compare_to_value: ",re_text_found_final,"   ",compare_to_value)
                            was_found = False
                else:
                    if DEBUG: print("Match NOT found")
                    was_found = False

            if DEBUG: print("was_found: ", was_found)

        else:
            pass

        if was_found:
            if param_dict['OPTIONS_UPDATE_FOUND_TEXT'] == unicode_type("True"):
                self.found_books_update_dict[current_book] = re_text_found

        return was_found
    #-----------------------------------------------------------------------------------------
    def options_text_update_cc_with_results(self,param_dict):

        if not param_dict['OPTIONS_UPDATE_FOUND_TEXT'] == unicode_type("True"):
            return

        was_success = self.update_custom_metadata_using_standard_calibre(param_dict)
        if not was_success:
            if DEBUG: print("ERROR: update_custom_metadata_using_standard_calibre from MCS full-text search.")
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def update_custom_metadata_using_standard_calibre(self,param_dict):

        update_cc = param_dict['OPTIONS_UPDATE_FOUND_TEXT_CC']
        update_cc = unicode_type(update_cc)
        update_cc_str = as_unicode(update_cc)

        #~ self.custom_column_label_dict = {}                 #  [label] = id
        #~ self.custom_column_datatype_dict = {}          #  [id] = datatype
        #~ self.custom_column_normalized_dict = {}      #  [id] = is_normalized

        self.update_cc_is_taglike = False
        k_okay = False
        #~ for k,v in self.custom_column_label_dict.iteritems():    # label, id
        for k,v in iteritems(self.custom_column_label_dict):    # label, id
            if k == update_cc or k == update_cc_str:
                if v in self.custom_column_datatype_dict:             # id,datatype
                    datatype = self.custom_column_datatype_dict[v]
                    if datatype == "text" or datatype == "comments":
                        k_okay = True
                        if v in self.custom_column_normalized_dict:          # id,is_normalized
                            is_normalized = self.custom_column_normalized_dict[v]
                            if is_normalized == 1:
                                self.update_cc_is_taglike = True
                break
        #END FOR

        if not k_okay:
            if DEBUG: print("[1] Invalid update cc: ", update_cc_str)
            return False

        if DEBUG: print("Update CC: ", update_cc_str)

        if param_dict['OPTIONS_UPDATE_FOUND_TEXT_REGEX'] > "   ":
            use_regex = True
            regex = param_dict['OPTIONS_UPDATE_FOUND_TEXT_REGEX']   # number to use, not the regex itself...
            if DEBUG: print("Update CC is using regex: ", regex)
            regex_full = self.get_regex_full(regex)
        else:
            use_regex = False
            regex = None

        payload = []

        custom_columns_metadata_dict = self.gui.current_db.field_metadata.custom_field_metadata()

        id_map = {}

        #~ for k,v in self.found_books_update_dict.iteritems():
        for k,v in iteritems(self.found_books_update_dict):
            book = int(k)
            if DEBUG: print("--------------book being processed for metadata update: ", as_unicode(book))
            re_text_found = v.strip()
            re_text_found_final = v.strip()
            mi = Metadata(_('Unknown'))
            mi_field = update_cc_str    #label
            custcol = custom_columns_metadata_dict[mi_field]
            #~ if DEBUG: print("custcol: ", as_unicode(custcol))
            if use_regex:
                if DEBUG: print("searching: ", re_text_found, " using regex: ", regex_full)
                re.escape("\\")
                p = re.compile(regex_full, re.IGNORECASE|re.DOTALL|re.MULTILINE)
                match = p.search(re_text_found)
                if match:
                    re_text_found_final = match.group()
                    if DEBUG: print("Match found and final text to update is: ", re_text_found_final)
                else:
                    if DEBUG: print("Match NOT found")
                    continue
            if self.update_cc_is_taglike:
                try:
                    book_mi_object = self.guidb.get_metadata(book, index_is_id=True, get_cover=False, get_user_categories=True, cover_as_data=False)    # a :class:`Metadata` object.
                except:
                    book_mi_object = self.guidb.get_metadata(book, index_is_id=False, get_cover=False, get_user_categories=True, cover_as_data=False)    # a :class:`Metadata` object.
                cc_existing_tags_list = book_mi_object.get(update_cc_str)
                #~ if isinstance(cc_existing_tags_list,list):
                    #~ pass
                #~ else:
                    #~ cc_existing_tags_list = ast.literal_eval(as_unicode(cc_existing_tags_list))
                re_text_found_final.strip()
                cc_existing_tags_list.append(re_text_found_final)
                cc_existing_tags_list = list(set(cc_existing_tags_list))   #no duplicate tags
                cc_existing_tags_list.sort()
                custcol['#value#'] = cc_existing_tags_list    # only for Tag-like custom columns...
                if DEBUG: print("custcol['#value#']: for tag-like", as_unicode(cc_existing_tags_list))
            else:
                custcol['#value#'] = re_text_found_final.strip()     # only for non-Tag-like custom columns...
                if DEBUG: print("custcol['#value#']: for non-tag-like",re_text_found_final)
            mi.set_user_metadata(mi_field, custcol)
            id_map[book] = mi
            payload.append(book)
        #END FOR

        #----------------------------------------
        from calibre.gui2.ui import get_gui
        self.maingui = get_gui()
         #----------------------------------------

        edit_metadata_action = self.maingui.iactions['Edit Metadata']
        edit_metadata_action.apply_metadata_changes(id_map, title=None, msg='', callback=None, merge_tags=False, merge_comments=False, icon=None)

        del id_map
        del mi
        del custom_columns_metadata_dict

        return True
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def get_regex_full(self,regex):

        if regex == "":
            regex = "^.+$"
        elif regex == "1":
            regex = prefs['REGEXVALUE_0']
        elif regex == "2":
            regex = prefs['REGEXVALUE_1']
        elif regex == "3":
            regex = prefs['REGEXVALUE_2']
        elif regex == "4":
            regex = prefs['REGEXVALUE_3']
        elif regex == "5":
            regex = prefs['REGEXVALUE_4']
        elif regex == "6":
            regex = prefs['REGEXVALUE_5']
        elif regex == "7":
            regex = prefs['REGEXVALUE_6']
        elif regex == "8":
            regex = prefs['REGEXVALUE_7']
        elif regex == "9":
            regex = prefs['REGEXVALUE_8']
        elif regex == "10":
            regex = prefs['REGEXVALUE_9']
        else:
            if DEBUG: print("Error in converting: ", regex, " to a full saved regular expression.")
            regex = "^.+$"

        if DEBUG: print("full regex is: ", regex)

        return regex
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def mcs_update_txt_format_word_index_table(self,my_db,my_cursor,current_book, current_book_full_path):
        # this function is used in main.py as a job.  changes here need to be made there and vice versa.

        try:
            current_txt_format_file = open(current_book_full_path, "r")
            current_txt_format_data = current_txt_format_file.readlines()
            current_txt_format_file.close()
            del current_txt_format_file

            if not current_txt_format_data:
                return False

            s = unicode_type(" ")
            long_string = s.join(current_txt_format_data)

            del current_txt_format_data

            long_string = long_string.replace(r'\\a',' ') # https://docs.python.org/2.0/ref/strings.html
            long_string = long_string.replace(r'\\b',' ')
            long_string = long_string.replace(r'\\f',' ')
            long_string = long_string.replace(r'\\n',' ')
            long_string = long_string.replace(r'\\r',' ')
            long_string = long_string.replace(r'\\t',' ')
            long_string = long_string.replace(r'\\v',' ')
            long_string = long_string.replace(r'\\',' ')

            re.escape("\\")

            for c in self.word_delimiter_characters_list:
                sleep(0)
                try:
                    s = "[\\" + c + "]"
                    long_string = re.sub(s," ",long_string,0,re.DOTALL|re.MULTILINE|re.IGNORECASE)        # re.sub(pattern, repl, string, count=0, flags=0)
                except Exception as e:
                    try:
                        long_string = long_string.replace(c," ")
                    except:
                        pass
                    continue
            #END FOR

            s_numbers = '0123456789'
            for s in s_numbers:
                long_string = re.sub(s," ",long_string,0,re.DOTALL|re.MULTILINE|re.IGNORECASE)
            #END FOR

            long_string = re.sub("   "," ",long_string,0,re.DOTALL|re.MULTILINE|re.IGNORECASE)
            long_string = re.sub("   "," ",long_string,0,re.DOTALL|re.MULTILINE|re.IGNORECASE)
            long_string = re.sub("  "," ",long_string,0,re.DOTALL|re.MULTILINE|re.IGNORECASE)
            long_string = re.sub("  "," ",long_string,0,re.DOTALL|re.MULTILINE|re.IGNORECASE)

            try:
                words_list = long_string.split(" ")
                del long_string
            except Exception as e:
                if DEBUG: print("[1] words_list = long_string.split(" ") >>>>", as_unicode(e))
                return False

            if len(words_list) == 0:
                #~ if DEBUG: print("words_list is empty; returning")
                return False

            re_skip_regex = prefs['WORD_INDEX_SKIP_REGEX']
            re_skip_regex = re_skip_regex.strip()

            try:
                re_skip = re.compile(re_skip_regex, re.IGNORECASE)
            except Exception as e:
                if DEBUG: print("re_skip re.compile exception: ", as_unicode(e))
                re_skip_regex = ""

            word_occurrences_list = []

            for word in words_list:
                try:
                    if not isinstance(word,unicode_type):
                        try:
                            word = unicode_type(word)
                        except Exception as e:
                            continue
                    word = word.replace(" ","")
                    word.strip()
                    if re_skip_regex > " " and re_skip_regex != "":
                        try:
                            match1 = re_skip.search(word)
                            if match1:
                                word = None
                        except:
                            pass
                    if word:
                        word.strip()
                        if len(word) >= self.word_index_minimum_number_of_letters:
                            try:
                                word = word.lower()
                            except:
                                pass
                            word_occurrences_list.append(word)
                except Exception as e:
                    if DEBUG: print("[2]", as_unicode(e))
                    continue
            #END FOR

            sleep(0)

            word_occurrences_set = set(word_occurrences_list)
            word_occurrences_list = list(word_occurrences_set)
            del word_occurrences_set

            my_cursor.execute("begin")
            mysql = "DELETE FROM _mcs_word_book_index WHERE book = ?"
            my_cursor.execute(mysql,([current_book]))
            my_cursor.execute("commit")

            my_cursor.execute("begin")
            mysql = "INSERT OR IGNORE INTO _mcs_word_book_index (book,word) VALUES(?,?) "
            for word in word_occurrences_list:
                my_cursor.execute(mysql,(current_book,word))
            #END FOR
            my_cursor.execute("commit")
            del word_occurrences_list
            return True
        except Exception as e:
            try:
                del word_occurrences_list
            except:
                pass
            try:
                my_cursor.execute("commit")
            except:
                pass
            if DEBUG: print("Exception in 'def mcs_update_txt_format_word_index_table': ", as_unicode(e))
            return False
    #-----------------------------------------------------------------------------------------
    def mcs_create_txt_format_word_index_table(self,my_db,my_cursor):
        my_cursor.execute("begin")
        mysql = "CREATE TABLE IF NOT EXISTS _mcs_word_book_index (book INTEGER NOT NULL, word TEXT NOT NULL , PRIMARY KEY (book, word))"
        my_cursor.execute(mysql)
        my_cursor.execute("commit")
        my_cursor.execute("begin")
        mysql =  "CREATE INDEX IF NOT EXISTS __mcs_word_book_index_by_word ON _mcs_word_book_index ( word )  "
        my_cursor.execute(mysql)
        my_cursor.execute("commit")
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def search_word_in_book_query_control(self,guidb,param_dict,sel_type):
        #~ if DEBUG: print("search_word_in_book_query_control: ",sel_type)
        #~ for k,v in iteritems(param_dict):
            #~ if DEBUG: print(k,v)

        self.gui.status_bar.showMessage("MCS Word Search: Executing Search")

        total_start = time.time()

        if param_dict['ALL_AUTHORS_BOOKS'] == unicode_type("True"):
            self.sq_use_all_authors_books = True
        else:
            self.sq_use_all_authors_books = False

        if param_dict['USE_FINAL_FILTERS'] == unicode_type("True"):
            self.sq_use_final_filters = True
        else:
            self.sq_use_final_filters = False

        self.display_progress_bar(0)  # never display it for this

        word_search_terms = param_dict['WORD_SEARCH_TERMS']
        bar_means_or = param_dict['BAR_MEANS_OR']
        if bar_means_or == "True":
            bar_means_or = True
        else:
            bar_means_or = False
        auto_index = param_dict['AUTOINDEX_SELECTED_BOOKS_IF_REQUIRED']
        if auto_index == "True":
            auto_index = True
        else:
            auto_index = False

        selected_books_list = self.get_selected_books(guidb,sel_type)

        found_list = []

        my_db,my_cursor = self.apsw_connect_to_current_guidb()

        if auto_index:
            if len(selected_books_list) < 11:
                self.auto_index_unindexed_selected_books(my_db,my_cursor,selected_books_list)
                total_start = time.time()

        for current_book in selected_books_list:
            was_found = self.search_word_index_table_control(my_db,my_cursor,bar_means_or,current_book,word_search_terms)
            if was_found:
                found_list.append(current_book)
        #END FOR

        self.count_records_in_index(my_db,my_cursor)

        my_db.close()

        found_set = set(found_list)
        found_list = list(found_set)
        del found_set
        found_list.sort()
        #-----------------------------

        if self.sq_use_all_authors_books:
            self.gui.status_bar.showMessage("MCS Word Search: Adding All Authors' Books")
            found_list = self.add_all_authors_books(found_list)
            my_db.close()
        found_set = set(found_list)
        found_list = list(found_set)
        del found_set
        found_list.sort()

        if self.sq_use_final_filters:
            self.gui.status_bar.showMessage("MCS Word Search: Applying Final Filters")
            self.param_dict = {}
            #~ for k,v in prefs.iteritems():
            for k,v in iteritems(prefs):
                self.param_dict[k] = v
            found_list = self.apply_final_filters_control(found_list)
            my_db.close()
            del self.param_dict

        del param_dict

        found_set = set(found_list)
        found_list = list(found_set)
        del found_set

        self.mark_found_books(found_list)

        try:
            self.mcs_dialog.show()
        except:
            pass

        #-----------------------------------------
        total_end = time.time()
        total_elapsed = total_end - total_start
        msg = 'MCS Word Search: searched selected books, found %d match(es) in %.3f seconds' %(len(found_list), total_elapsed)
        self.gui.status_bar.showMessage(msg)
        #-----------------------------------------
        del found_list
    #-----------------------------------------------------------------------------------------
    def search_word_index_table_control(self,my_db,my_cursor,bar_means_or,current_book,word_search_terms):
        word_list = []
        if "|" in word_search_terms:
            tmp_list = word_search_terms.split("|")
            for word in tmp_list:
                if word:
                    word = word.strip()
                    if word > " ":
                        word_list.append(word)
            #END FOR
            del tmp_list
        else:
            word_search_terms = word_search_terms.strip()
            word_list.append(word_search_terms)
        if bar_means_or:
            is_keepable = False
        else:
            is_keepable = True
        mysql = "SELECT word FROM _mcs_word_book_index WHERE book = ? AND word LIKE ?  "
        for word in word_list:
            if not bar_means_or:
                #~ if DEBUG: print("word being searched: ", word)
                if word.count("!") > 0:
                    word_is_a_not = True
                    word = word.replace("!","")
                    word = word.strip()
                else:
                    word_is_a_not = False
            my_cursor.execute(mysql,(current_book,word))
            tmp_rows = my_cursor.fetchall()
            if tmp_rows:
                if len(tmp_rows) > 0:
                    if bar_means_or:
                        #~ if DEBUG: print("OR:  word found was: ", word)
                        is_keepable = True
                        break
                    else:
                        #~ if DEBUG: print("AND:  word found was: ", word)
                        if word_is_a_not:
                            #~ if DEBUG: print("found it but should not have; reject it")
                            is_keepable = False
                            break
                        else:
                            pass #found it and should have; keep it
                else:
                    if not bar_means_or:
                        #~ if DEBUG: print("AND:  word NOT found was: ", word)
                        if word_is_a_not:
                            pass #should not have found it and did not; continue
                        else:
                            is_keepable = False
                            break
                    else:
                        pass
            else:
                if not bar_means_or:
                    #~ if DEBUG: print("AND:  word NOT found was: ", word)
                    if word_is_a_not:
                        pass #should not have found it and did not; continue
                    else:
                        is_keepable = False
                        break
                else:
                    pass
        #END FOR
        del word_list
        #~ if DEBUG: print("word is keepable: ", is_keepable)
        if is_keepable:
            return True
        else:
            return False
    #-----------------------------------------------------------------------------------------
    def auto_index_unindexed_selected_books(self,my_db,my_cursor,selected_books_list):

        self.gui.status_bar.showMessage("MCS Word Search: Indexing Unindexed Selected Books by Request")

        global update_mcs_was_indexed_custom_column
        if prefs['WORD_INDEX_UPDATE_MCS_INDEXED_CUSTOM_COLUMN'] == unicode_type("True"):
            update_mcs_was_indexed_custom_column = True
        else:
            update_mcs_was_indexed_custom_column = False
        if update_mcs_was_indexed_custom_column:
            self.get_custom_column_id(my_db,my_cursor)

        unindexed_selected_book_list = []

        mysql = "SELECT count(*) FROM _mcs_word_book_index WHERE book = ?"
        for current_book in selected_books_list:
            my_cursor.execute(mysql,([current_book]))
            tmp_rows = my_cursor.fetchall()
            if not tmp_rows:
                unindexed_selected_book_list.append(current_book)
            else:
                for row in tmp_rows:
                    for col in row:
                        if col == 0:
                            unindexed_selected_book_list.append(current_book)
                        break
                    #END FOR
                    break
                #END FOR
        #END FOR

        if len(unindexed_selected_book_list) == 0:
            #~ if DEBUG: print("no unindexed selected books found")
            del unindexed_selected_book_list
            return

        self.word_index_minimum_number_of_letters = int(prefs['WORD_INDEX_MINIMUM_NUMBER_OF_LETTERS'])

        try:
            del self.selected_books_list
        except:
            pass
        self.selected_books_list = []
        for current_book in unindexed_selected_book_list:
            full_book_path = self.build_book_path(my_db,my_cursor,current_book)
            if not full_book_path:
                continue
            self.mcs_update_txt_format_word_index_table(my_db,my_cursor,current_book, full_book_path)
            if update_mcs_was_indexed_custom_column:
                self.set_mcs_was_indexed_custom_column(my_db,my_cursor,current_book)
            self.selected_books_list.append(current_book)
        #END FOR
        self.force_refresh_of_cache()
        del unindexed_selected_book_list

        self.gui.status_bar.showMessage("MCS Word Search: Executing Search")
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def mcs_build_index_job_control(self,gui,guidb,sel_type,mwicc):

        try:
            del self.selected_books_list
        except:
            pass

        if sel_type == "selected":
            self.selected_books_list = self.get_selected_books(guidb,sel_type)
        elif sel_type == "all":
            self.selected_books_list = self.get_selected_books(guidb,sel_type)
        elif sel_type == "all_unindexed":
            sel_type == "all"
            self.selected_books_list = self.get_selected_books(guidb,sel_type)
            sel_type = "all_unindexed"

        n_books = len(self.selected_books_list)

        msg = ('MCS Build Index Job was submitted and is processing %d book(s)' %(n_books))
        self.gui.status_bar.showMessage(msg)

        wdcl = self.word_delimiter_characters_list

        start_mcs_build_index(self,gui,guidb,sel_type,self.selected_books_list,mwicc,wdcl,Dispatcher(self.mcs_finish_with_refresh))

        self.mcs_dialog.close()

        self.number_jobs_executing =  self.number_jobs_executing + 1
    #-----------------------------------------------------------------------------------------
    def mcs_finish_with_refresh(self,job=None):
        self.force_refresh_of_cache_index()
        del job
        self.number_jobs_executing =  self.number_jobs_executing - 1
        if  self.number_jobs_executing < 0:
            self.number_jobs_executing = 0
    #-----------------------------------------------------------------------------------------
    def force_refresh_of_cache_index(self):
        db = self.maingui.current_db.new_api
        db.reload_from_db(clear_caches=False)
        frozen = db.all_book_ids()
        books = list(frozen)
        self.maingui.library_view.model().refresh_ids(books)
        self.maingui.tags_view.recount()
        QApplication.instance().processEvents()
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def count_records_in_index(self,my_db,my_cursor):
        n_records = 0
        mysql = "SELECT count(*) FROM _mcs_word_book_index "
        my_cursor.execute(mysql)
        tmp_rows = my_cursor.fetchall()
        if not tmp_rows:
            pass
        else:
            for row in tmp_rows:
                for col in row:
                    n_records = col
                #END FOR
            #END FOR
        prefs['WORD_INDEX_LATEST_RECORD_COUNT'] = unicode_type(n_records)
        prefs
    #-----------------------------------------------------------------------------------------
    def mcs_trim_index_job_control(self,gui,guidb,trim_type,selected_books_list):
        msg = ('MCS Trim Job was submitted')
        self.gui.status_bar.showMessage(msg)
        start_mcs_trim_index(self,gui,guidb,trim_type,selected_books_list,Dispatcher(self.mcs_finish))
        self.mcs_dialog.close()
        self.number_jobs_executing = self.number_jobs_executing + 1
    #-----------------------------------------------------------------------------------------
    def mcs_finish(self,job):
        del job
        self.number_jobs_executing =  self.number_jobs_executing - 1
        if  self.number_jobs_executing < 0:
            self.number_jobs_executing = 0
        return
    #-----------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------------
    def set_mcs_was_indexed_custom_column(self,my_db,my_cursor,current_book):

        global custom_column_id
        global mysql_set_mcs_was_indexed

        my_cursor.execute("begin")
        my_cursor.execute(mysql_set_mcs_was_indexed,([current_book]))
        my_cursor.execute("commit")
    #---------------------------------------------------------------------------------------------------------------------------------------------
    def get_custom_column_id(self,my_db,my_cursor):

        global update_mcs_was_indexed_custom_column
        global custom_column_id
        global mysql_set_mcs_was_indexed

        custom_column_id = as_unicode(0)

        mysql = "SELECT id,editable FROM custom_columns WHERE label = 'mcs_was_indexed' AND datatype = 'bool' AND normalized = 0   "
        my_cursor.execute(mysql)
        tmp_rows = my_cursor.fetchall()
        if not tmp_rows:
            update_mcs_was_indexed_custom_column = False
            return
        for row in tmp_rows:
            custom_column_id,editable = row
        #END FOR

        if editable == 1:
            my_cursor.execute("begin")
            mysql = "UPDATE custom_columns SET editable = 0 WHERE id = ?"
            my_cursor.execute(mysql,([custom_column_id]))
            my_cursor.execute("commit")

        custom_column_id = as_unicode(custom_column_id)

        mysql_set_mcs_was_indexed = as_unicode("INSERT OR REPLACE INTO custom_column_[N] (id,book,value) VALUES(NULL,?,1) ")
        mysql_set_mcs_was_indexed = mysql_set_mcs_was_indexed.replace("[N]",as_unicode(custom_column_id))
    #---------------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------------
    def search_request_control_cl_duplicates(self,guidb,param_dict,sel_type,force_repaint,force_abort,current_path,search_path):
        #~ Purpose: to find books in another library that have the identical values in the 2 specified MCS search criteria columns.
        #~ Custom Columns are of course supported, unlike the 'Find Duplicates' plug-in.  That is why this exists.
        #~ If the user wants to match duplicates using only 1 specified MCS search criteria column, easy: put the same column name in both MCS search columns...

        global log_title_search_path
        global do_log

        if as_unicode(current_path) == as_unicode(search_path):
            return

        self.current_path = current_path
        self.search_path = search_path

        if isbytestring(self.current_path):
            self.current_path = self.current_path.decode(filesystem_encoding)
        self.current_path = self.current_path.replace(os.sep, '/')

        if isbytestring(self.search_path):
            self.search_path = self.search_path.decode(filesystem_encoding)
        self.search_path = self.search_path.replace(os.sep, '/')

        if DEBUG: print(self.current_path, " <----> ", self.search_path)

        if force_abort:
            self.mcs_dialog.close()
            self.search_request_dialog()
            self.gui.status_bar.showMessage("MCS Criteria Error(s).  Search Canceled.")
            return

        param_dict = self.tweak_param_dict(param_dict)

        if force_repaint:
            self.mcs_dialog.close()

        total_start = time.time()

        self.gui.status_bar.showMessage("MCS is Searching for CL Duplicates...")

       #-----------------------------
        do_log = True
        self.need_tags_too = True  #otherwise, the GUI Log won't show the Tags...
        self.need_authors_too = True  #ditto
        selected_books_list = []
        if force_repaint:
            self.mcs_dialog.close()
            self.search_request_dialog()
            msg = ('MCS is Searching Another Calibre Library for Duplicates')
            self.gui.status_bar.showMessage(msg)
       #-----------------------------
        my_db,my_cursor,is_valid = self.apsw_connect_to_a_library_path(self.search_path)     # the external library...
        if not is_valid:
            return
        #----------------------------
        self.create_temp_views_and_tables(my_db,my_cursor)               # the external library...
        self.refresh_mcs_search_accelerator_tables(my_db,my_cursor)   # the external library...
        #-----------------------------

        sel_type = "all"
        internal_selected_books_list = self.get_selected_books(guidb,sel_type)
        n_books = len(internal_selected_books_list)
        if n_books == 0:
            my_db.close()
            self.gui.status_bar.showMessage("MCS Found No Current Library Books...")
            return

        external_selected_books_list = self.get_all_external_library_books(my_db,my_cursor)           # book.id is returned as a string...
        my_db.close()

        n_books = len(external_selected_books_list)
        if n_books == 0:
            self.gui.status_bar.showMessage("MCS Found No External Books to Search...")
            return

        n_comparisons =len(internal_selected_books_list) * len(external_selected_books_list) * 2

        msg = ('MCS is Making %d Comparisons' %(n_comparisons))
        self.gui.status_bar.showMessage(msg)

        if n_comparisons >= 10000:
            msg = ('MCS is Making %d Comparisons, and it might take a very long time. Please be patient. ' %(n_comparisons))
            self.gui.status_bar.showMessage(msg)
        else:
            if n_comparisons >= 1000:
                msg = ('MCS is Making %d Comparisons.  Please be patient. ' %(n_comparisons))
                self.gui.status_bar.showMessage(msg)

        self.display_progress_bar(n_books)

        #-----------------------------------------
        self.cl_duplicates_search_control(internal_selected_books_list,external_selected_books_list,param_dict)
        #-----------------------------------------

        self.skip_message = False
        if self.show_mcs_progress_bar:
            if self.mcs_progress_bar_dialog.wasCanceled():
                self.user_clicked_cancel = True
            if self.user_clicked_cancel:
                self.gui.search.clear()
                msg = "Search Canceled by User.  Disregard Any Results."
                self.gui.status_bar.showMessage(msg)
                self.skip_message = True
            self.mcs_progress_bar_dialog.close()
        #-----------------------------------------
        if not self.skip_message:
            total_end = time.time()
            total_elapsed = total_end - total_start
            msg = 'Cross-Library Duplicates Search completed in %.3f seconds' %(total_elapsed)
            self.gui.status_bar.showMessage(msg)
        #-----------------------------------------
        #-----------------------------------------
        #-----------------------------------------

        del param_dict
        del self.param_dict
        del internal_selected_books_list
        del external_selected_books_list
    #---------------------------------------------------------------------------------------------------------------------------------------
    def cl_duplicates_search_control(self,internal_selected_books_list,external_selected_books_list,param_dict):
        #~ ------------------------------------------------------------------------------------
        #~ one row of std_metadata_list = book,title,authorsall,series,tagsall,publisher,comments,pubdate,path
        #~ ------------------------------------------------------------------------------------
        #~ one row of cc_metadata_list =  book_data_dict               [but only those #labels needed by the query are included; not all of them.]
        #~ book_data_dict['book'] = bookid
        #~ book_data_dict['label'] = label of the custom column        [starts with a #]
        #~ book_data_dict['value'] = value in that custom column for the bookid
        #~ ------------------------------------------------------------------------------------
        global cust_columns_search_list

        del cust_columns_search_list[:]

        if DEBUG: print(self.current_path,"      ", self.search_path)

        internal_std_metadata_list, internal_cc_metadata_list = self.cl_duplicates_get_standard_and_custom_metadata(internal_selected_books_list,param_dict,self.current_path)
        external_std_metadata_list, external_cc_metadata_list = self.cl_duplicates_get_standard_and_custom_metadata(external_selected_books_list,param_dict,self.search_path)

        if DEBUG: print("column 1: ", param_dict['NAME1']," column 2: ", param_dict['NAME2'])

        column_to_compare_1 = param_dict['NAME1']
        column_to_compare_2 = param_dict['NAME2']

        column_to_compare_1 = as_unicode(column_to_compare_1).strip()
        column_to_compare_2 = as_unicode(column_to_compare_2).strip()

        if "#" in column_to_compare_1:
            col_to_compare_1_is_cc = True
            cust_columns_search_list.append(column_to_compare_1)
        else:
            col_to_compare_1_is_cc = False

        if "#" in column_to_compare_2:
            col_to_compare_2_is_cc = True
            cust_columns_search_list.append(column_to_compare_2)
        else:
            col_to_compare_2_is_cc = False

        if col_to_compare_1_is_cc or col_to_compare_2_is_cc:
            must_compare_cc = True
        else:
            must_compare_cc = False

        if col_to_compare_1_is_cc and col_to_compare_2_is_cc:
            must_compare_only_cc = True
            must_compare_cc = True
            must_compare_std = False
            must_compare_both = False
        else:
            must_compare_only_cc = False
            must_compare_std = True

        if must_compare_std and must_compare_cc:
            must_compare_both = True
            must_compare_only_cc = False
            must_compare_only_std = False
        else:
            must_compare_both = False

        if (not col_to_compare_1_is_cc) and (not col_to_compare_2_is_cc):
            must_compare_only_std = True
            must_compare_both = False
            must_compare_only_cc = False
        else:
            must_compare_only_std = False

        if DEBUG:
            print("total internal books: ", as_unicode(len(internal_selected_books_list)))
            print("total external books: ", as_unicode(len(external_selected_books_list)))
            print("len of internal_std_metadata_list: ", as_unicode(len(internal_std_metadata_list)))
            print("len of external_std_metadata_list: ", as_unicode(len(external_std_metadata_list)))
            print("column_to_compare_1: ", column_to_compare_1)
            print("column_to_compare_2: ", column_to_compare_2)
            print("col_to_compare_1_is_cc: ",col_to_compare_1_is_cc)
            print("col_to_compare_2_is_cc: ",col_to_compare_2_is_cc)
            print("must_compare_only_cc: ",must_compare_only_cc)
            print("must_compare_only_std: ",must_compare_only_std)
            print("must_compare_both: ",must_compare_both)

        del internal_selected_books_list
        del external_selected_books_list

        if must_compare_both:
            final_int_list,final_ext_list = self.do_final_comparisons_both(column_to_compare_1,column_to_compare_2,internal_std_metadata_list,external_std_metadata_list,\
                                    internal_cc_metadata_list,external_cc_metadata_list,col_to_compare_1_is_cc,col_to_compare_2_is_cc)

        elif must_compare_only_cc:                                # both search columns 1 and 2 are custom columns
            final_int_list,final_ext_list = self.do_final_comparisons_only_cc(column_to_compare_1,column_to_compare_2,internal_cc_metadata_list,external_cc_metadata_list,\
                                    col_to_compare_1_is_cc,col_to_compare_2_is_cc)

        elif must_compare_only_std:                                # both search columns 1 and 2 are standard columns
            final_int_list,final_ext_list = self.do_final_comparisons_only_std(column_to_compare_1,column_to_compare_2,internal_std_metadata_list,external_std_metadata_list)

        else:
            if DEBUG: print("SERIOUS ERROR in cl_duplicates_search_control...")
            final_int_list = []
            final_ext_list = []

        #~ ------------------------------------------------------------------------------------
        if DEBUG:
            print("==============final tally===============")
            print("final_int_list total: ", as_unicode(len(final_int_list)))
            print("final_ext_list total: ", as_unicode(len(final_ext_list)))
            print("================end================")
        #~ ------------------------------------------------------------------------------------

        self.mark_found_books(final_int_list)

        self.cl_duplicates_build_gui_log(final_ext_list,external_std_metadata_list, external_cc_metadata_list)

        try:
            del final_int_list
            del final_ext_list
            del external_std_metadata_list
            del external_cc_metadata_list
            del internal_std_metadata_list
            del internal_cc_metadata_list
        except:
            pass
    #---------------------------------------------------------------------------------------------------------------------------------------
    def cl_duplicates_build_gui_log(self,final_ext_list,external_std_metadata_list, external_cc_metadata_list):
        #~ ------------------------------------------------------------------------------------
        #~ one row of std_metadata_list = book,title,authorsall,series,tagsall,publisher,comments,pubdate,path
        #~ ------------------------------------------------------------------------------------
        #~ one row of cc_metadata_list =  book_data_dict               [but only those #labels needed by the query are included; not all of them.]
        #~ book_data_dict['book'] = bookid
        #~ book_data_dict['label'] = label of the custom column        [starts with a #]
        #~ book_data_dict['value'] = value in that custom column for the bookid
        #~ ------------------------------------------------------------------------------------
        global search_log
        global log_title_search_path
        global cust_columns_search_list

        log_title_search_path = self.search_path

        search_log = []
        del search_log[:]

        for bookid in final_ext_list:
            bookid = int(bookid)
            book_dict = {}
            for line in external_std_metadata_list:
                book,title,authorsall,series,tagsall,publisher,comments,pubdate,path = line
                book = int(book)
                if book == bookid:
                    tags = tagsall[0:75]
                    tags = tags.strip()
                    published = as_unicode(pubdate)
                    published = published[0:7]
                    published = published.strip()
                    book_dict['AUTHORS'] = authorsall
                    book_dict['TITLE'] = title
                    book_dict['SERIES'] = series
                    book_dict['TAGS'] = tags
                    book_dict['PUBLISHER'] = publisher
                    book_dict['PUBLISHED'] = as_unicode(published)
                    book_dict['PATH'] = path
                    for book_data_dict in external_cc_metadata_list:
                        bid = book_data_dict['book']
                        bid = int(bid)
                        if bid == bookid:
                            label = book_data_dict['label']
                            value = book_data_dict['value']
                            book_dict[label] = value
                    #END FOR
            #END FOR
            if len(book_dict) > 0:
                search_log.append(book_dict)
            del book_dict
        #END FOR

        self.display_search_log()

        try:
            del search_log
            del external_cc_metadata_list
            del final_ext_list
        except:
            pass
    #---------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------
    def cl_duplicates_get_standard_and_custom_metadata(self,books_list,param_dict,path):

        try:
            del std_metadata_list
        except:
            pass
        try:
            del cust_metadata_list
        except:
            pass

        my_db,my_cursor,is_valid = self.apsw_connect_to_a_library_path(path)
        if not is_valid:
            return None,None

        #----------------------------
        self.need_authors_too = True
        #----------------------------
        self.create_temp_views_and_tables(my_db,my_cursor)
        #----------------------------
        std_metadata_list, cust_metadata_list = self.get_all_book_metadata(my_db,my_cursor,books_list,param_dict)
        #-----------------------------
        my_db.close()

        return std_metadata_list, cust_metadata_list
    #---------------------------------------------------------------------------------------------------------------------------------------
    def do_final_comparisons_only_std(self,column_to_compare_1,column_to_compare_2,internal_std_metadata_list,external_std_metadata_list):
        #~ one row of std_metadata_list = book,title,authorsall,series,tagsall,publisher,comments,pubdate,path

        internal_books_to_keep_list = []
        external_books_to_keep_list = []

        internal_combinations_list = []
        external_combinations_list = []

        if column_to_compare_1 == "title":
            colnum_1 = 1
        elif column_to_compare_1 == "authors":
            colnum_1 = 2
        elif column_to_compare_1 == "series":
            colnum_1 = 3
        elif column_to_compare_1 == "tags":
            colnum_1 = 4
        elif column_to_compare_1 == "publisher":
            colnum_1 = 5
        elif column_to_compare_1 == "comments":
            colnum_1 = 6
        elif column_to_compare_1 == "pubdate":
            colnum_1 = 7
        else:
            if DEBUG: print("ERROR[1]: do_final_comparisons_both_std")
            return external_books_to_keep_list

        if column_to_compare_2 == "title":
            colnum_2 = 1
        elif column_to_compare_2 == "authors":
            colnum_2 = 2
        elif column_to_compare_2 == "series":
            colnum_2 = 3
        elif column_to_compare_2 == "tags":
            colnum_2 = 4
        elif column_to_compare_2 == "publisher":
            colnum_2 = 5
        elif column_to_compare_2 == "comments":
            colnum_2 = 6
        elif column_to_compare_2 == "pubdate":
            colnum_2 = 7
        else:
            if DEBUG: print("ERROR[2]: do_final_comparisons_both_std")
            return external_books_to_keep_list

        #~ get all internal book value combinations
        for row in internal_std_metadata_list:
            book,title,authorsall,series,tagsall,publisher,comments,pubdate,path = row
            book = int(book)
            comp_1i = row[colnum_1]
            comp_2i = row[colnum_2]
            if not comp_1i:
                comp_1i = ""
            if not comp_2i:
                comp_2i = ""
            s = comp_1i + "||" + comp_2i
            internal_combinations_list.append(s)
        #END FOR

        #~ get all external book value combinations
        for line in external_std_metadata_list:
            book,title,authorsall,series,tagsall,publisher,comments,pubdate,path = line
            book = int(book)
            comp_1e = line[colnum_1]
            comp_2e = line[colnum_2]
            if not comp_1e:
                comp_1e = ""
            if not comp_2e:
                comp_2e = ""
            s = comp_1e + "||" + comp_2e
            external_combinations_list.append(s)
        #END FOR

        internal_combinations_set = set(internal_combinations_list)
        external_combinations_set = set(external_combinations_list)

        del external_combinations_list
        del internal_combinations_list

        final_combinations_set = internal_combinations_set.intersection(external_combinations_set)

        if DEBUG:
            final_combinations_list = list(final_combinations_set)
            final_combinations_list.sort()
            for combo in final_combinations_list:
                print("final combination: ", combo)
            del final_combinations_list

        for row in internal_std_metadata_list:
            book,title,authorsall,series,tagsall,publisher,comments,pubdate,path = row
            book = int(book)
            if not row[colnum_1]:
                tmpvalue1 = ""
            else:
                tmpvalue1 = row[colnum_1]
            if not row[colnum_2]:
                tmpvalue2 = ""
            else:
                tmpvalue2 = row[colnum_2]
            try:
                combination = row[colnum_1] + "||" + row[colnum_2]
            except:
                combination = tmpvalue1 + "||" + tmpvalue2
                #~ if DEBUG: print("null found for internal std metadata value: ", combination)
            if combination in final_combinations_set:
                internal_books_to_keep_list.append(book)
                #~ if DEBUG: print("internal book kept: ", as_unicode(book))
        #END FOR
        del internal_std_metadata_list

        for row in external_std_metadata_list:
            book,title,authorsall,series,tagsall,publisher,comments,pubdate,path = row
            book = int(book)
            if not row[colnum_1]:
                tmpvalue1 = ""
            else:
                tmpvalue1 = row[colnum_1]
            if not row[colnum_2]:
                tmpvalue2 = ""
            else:
                tmpvalue2 = row[colnum_2]
            try:
                combination = row[colnum_1] + "||" + row[colnum_2]
            except:
                combination = tmpvalue1 + "||" + tmpvalue2
                #~ if DEBUG: print("null found for external std metadata value: ", combination)
            if combination in final_combinations_set:
                external_books_to_keep_list.append(book)
                #~ if DEBUG: print("external book kept: ", as_unicode(book))
        #END FOR
        del external_std_metadata_list
        del final_combinations_set

        if DEBUG:
            if len(internal_books_to_keep_list) == 0:
                print("No Internal Books to Keep Were Found in 'do_final_comparisons_both_std'")
            if len(external_books_to_keep_list) == 0:
                print("No External Books to Keep Were Found in 'do_final_comparisons_both_std'")

        return internal_books_to_keep_list, external_books_to_keep_list
    #---------------------------------------------------------------------------------------------------------------------------------------
    def do_final_comparisons_only_cc(self,column_to_compare_1,column_to_compare_2,internal_cc_metadata_list,external_cc_metadata_list,col_to_compare_1_is_cc,col_to_compare_2_is_cc):
        #~ ------------------------------------------------------------------------------------
        #~ one row of cc_metadata_list =  book_data_dict               [but only those #labels needed by the query are included; not all of them.]
        #~ book_data_dict['book'] = bookid
        #~ book_data_dict['label'] = label of the custom column        [starts with a #]
        #~ book_data_dict['value'] = value in that custom column for the bookid
        #~ ------------------------------------------------------------------------------------

        internal_books_to_keep_list = []
        external_books_to_keep_list = []

        internal_book_completion_dict = {}
        external_book_completion_dict = {}

        if not col_to_compare_1_is_cc:
            empty_list = []
            return empty_list,empty_list

        if not col_to_compare_2_is_cc:
            empty_list = []
            return empty_list,empty_list

        if (not internal_cc_metadata_list) or (not external_cc_metadata_list):
            empty_list = []
            return empty_list,empty_list

        if len(internal_cc_metadata_list) == 0 or len(external_cc_metadata_list) == 0:
            empty_list = []
            return empty_list,empty_list

        # the metadata_list has a dict for each combination of book and custom column label.  up to 2 dicts for one book in cl duplicates, because can have only 2 search columns.
        for int_book_data_dict in internal_cc_metadata_list:
            intbid = int_book_data_dict['book']
            intbid = int(intbid)
            intlabel = int_book_data_dict['label']
            intlabel = as_unicode(intlabel)
            intvalue= int_book_data_dict['value']
            if not intvalue:
                continue
            try:
                if as_unicode(intvalue).strip() <= " ":
                    continue
            except:
                continue
            if intbid in internal_book_completion_dict:    # max of 2 custom columns to compare...so must have already found the first one...
                s = internal_book_completion_dict[intbid]
                s = s + "||" + intlabel + intvalue
                internal_book_completion_dict[intbid] = s
                #~ if DEBUG: print(s)
            else:
                internal_book_completion_dict[intbid] = intlabel + intvalue
        #END FOR

        for ext_book_data_dict in external_cc_metadata_list:
            extbid = ext_book_data_dict['book']
            extbid = int(extbid)
            extlabel = ext_book_data_dict['label']
            extlabel = as_unicode(extlabel)
            extvalue= ext_book_data_dict['value']
            if not extvalue:
                continue
            if as_unicode(extvalue).strip() <= " ":
                continue
            if extbid in external_book_completion_dict:    # max of 2 custom columns to compare...so must have already found the first one...
                s = external_book_completion_dict[extbid]
                s = s + "||" + extlabel + extvalue
                external_book_completion_dict[extbid] = s
                #~ if DEBUG: print(s)
            else:
                external_book_completion_dict[extbid] = extlabel + extvalue
        #END FOR

        internal_combinations_list = []
        external_combinations_list = []

        #~ for book,combo in internal_book_completion_dict.iteritems():
        for book,combo in iteritems(internal_book_completion_dict):
            if "||" in combo:
                internal_combinations_list.append(combo)
                #~ if DEBUG: print("internal combo: ", combo)
        #END FOR

        #~ for book,combo in external_book_completion_dict.iteritems():
        for book,combo in iteritems(external_book_completion_dict):
            if "||" in combo:
                external_combinations_list.append(combo)
                #~ if DEBUG: print("external combo: ", combo)
        #END FOR

        internal_combinations_set = set(internal_combinations_list)
        external_combinations_set = set(external_combinations_list)

        del external_combinations_list
        del internal_combinations_list

        reversed_combo_list = []

        for combo in internal_combinations_set:
            s = combo.split("||")
            value1 = s[0]
            value2 = s[1]
            s = value2 + "||" + value1
            reversed_combo_list.append(s)
        #END FOR
        for combo in reversed_combo_list:
            internal_combinations_set.add(combo)
        #END FOR
        del reversed_combo_list

        reversed_combo_list = []

        for combo in external_combinations_set:
            s = combo.split("||")
            value1 = s[0]
            value2 = s[1]
            s = value2 + "||" + value1
            reversed_combo_list.append(s)
        #END FOR
        for combo in reversed_combo_list:
            external_combinations_set.add(combo)
        #END FOR
        del reversed_combo_list

        final_combinations_set = internal_combinations_set.intersection(external_combinations_set)

        if DEBUG:
            final_combinations_list = list(final_combinations_set)
            final_combinations_list.sort()
            for combo in final_combinations_list:
                print("final combination: ", combo)
            del final_combinations_list

        #~ for book,combo in internal_book_completion_dict.iteritems():
        for book,combo in iteritems(internal_book_completion_dict):
            if combo in final_combinations_set:
                internal_books_to_keep_list.append(book)
                #~ if DEBUG: print("internal book kept: ", as_unicode(book))
        #END FOR

        #~ for book,combo in external_book_completion_dict.iteritems():
        for book,combo in iteritems(external_book_completion_dict):
            if combo in final_combinations_set:
                external_books_to_keep_list.append(book)
                #~ if DEBUG: print("external book kept: ", as_unicode(book))
        #END FOR

        if DEBUG:
            if len(internal_books_to_keep_list) == 0:
                print("No Internal Books to Keep Were Found in 'do_final_comparisons_both_std'")
            if len(external_books_to_keep_list) == 0:
                print("No External Books to Keep Were Found in 'do_final_comparisons_both_std'")

        internal_books_to_keep_list = list(set(internal_books_to_keep_list))
        external_books_to_keep_list = list(set(external_books_to_keep_list))

        return internal_books_to_keep_list, external_books_to_keep_list
    #---------------------------------------------------------------------------------------------------------------------------------------
    def do_final_comparisons_both(self,column_to_compare_1,column_to_compare_2,internal_std_metadata_list,external_std_metadata_list,internal_cc_metadata_list,external_cc_metadata_list,col_to_compare_1_is_cc,col_to_compare_2_is_cc):
        #~ ------------------------------------------------------------------------------------
        #~ one row of cc_metadata_list =  book_data_dict               [but only those #labels needed by the query are included; not all of them.]
        #~ book_data_dict['book'] = bookid
        #~ book_data_dict['label'] = label of the custom column        [starts with a #]
        #~ book_data_dict['value'] = value in that custom column for the bookid
        #~ ------------------------------------------------------------------------------------
        #~ one row of std_metadata_list = book,title,authorsall,series,tagsall,publisher,comments,pubdate,path
        #~ ------------------------------------------------------------------------------------
        if (not internal_cc_metadata_list) or (not external_cc_metadata_list):
            empty_list = []
            return empty_list,empty_list

        if len(internal_cc_metadata_list) == 0 or len(external_cc_metadata_list) == 0:
            empty_list = []
            return empty_list,empty_list

        internal_books_to_keep_list = []
        external_books_to_keep_list = []

        internal_book_completion_dict = {}
        external_book_completion_dict = {}

        if (not col_to_compare_1_is_cc) and (not col_to_compare_2_is_cc):
            empty_list = []
            return empty_list,empty_list

        if (not internal_cc_metadata_list) or (not external_cc_metadata_list):
            empty_list = []
            return empty_list,empty_list

        if len(internal_cc_metadata_list) == 0 or len(external_cc_metadata_list) == 0:
            empty_list = []
            return empty_list,empty_list

        if (not internal_std_metadata_list) or (not external_std_metadata_list):
            empty_list = []
            return empty_list,empty_list

        if len(internal_std_metadata_list) == 0 or len(external_std_metadata_list) == 0:
            empty_list = []
            return empty_list,empty_list

        #~ -------------------------------
        #~ First, match custom columns
        #~ -------------------------------
        # the metadata_list has a dict for each combination of book and custom column label.  here, can only be one (1) cc label.
        for int_book_data_dict in internal_cc_metadata_list:
            intbid = int_book_data_dict['book']
            intbid = int(intbid)
            intlabel = int_book_data_dict['label']
            intlabel = as_unicode(intlabel)
            intvalue= int_book_data_dict['value']
            if not intvalue:
                continue
            if as_unicode(intvalue).strip() <= " ":
                continue
            internal_book_completion_dict[intbid] = intlabel + intvalue
        #END FOR

        for ext_book_data_dict in external_cc_metadata_list:
            extbid = ext_book_data_dict['book']
            extbid = int(extbid)
            extlabel = ext_book_data_dict['label']
            extlabel = as_unicode(extlabel)
            extvalue= ext_book_data_dict['value']
            if not extvalue:
                continue
            if as_unicode(extvalue).strip() <= " ":
                continue
            external_book_completion_dict[extbid] = extlabel + extvalue
        #END FOR

        #~ -------------------------------
        #~ -------------------------------
        #~ Second, match standard columns
        #~ -------------------------------
        if col_to_compare_1_is_cc:
            column_to_compare_x = column_to_compare_2
        else:
            column_to_compare_x = column_to_compare_1

        if column_to_compare_x == "title":
            colnum_x = 1
        elif column_to_compare_x == "authors":
            colnum_x = 2
        elif column_to_compare_x == "series":
            colnum_x = 3
        elif column_to_compare_x == "tags":
            colnum_x = 4
        elif column_to_compare_x == "publisher":
            colnum_x = 5
        elif column_to_compare_x == "comments":
            colnum_x = 6
        elif column_to_compare_x == "pubdate":
            colnum_x = 7

        #~ get all internal book value combinations
        for row in internal_std_metadata_list:
            book,title,authorsall,series,tagsall,publisher,comments,pubdate,path = row
            book = int(book)
            comp_xi = row[colnum_x]
            if book in internal_book_completion_dict:
                s = internal_book_completion_dict[book]
                s = s + "||" + comp_xi
                internal_book_completion_dict[book] = s
            else:
                internal_book_completion_dict[book] = comp_xi

        #~ get all external book value combinations
        for row in external_std_metadata_list:
            book,title,authorsall,series,tagsall,publisher,comments,pubdate,path = row
            book = int(book)
            comp_xe = row[colnum_x]
            if book in external_book_completion_dict:
                s = external_book_completion_dict[book]
                s = s + "||" + comp_xe
                external_book_completion_dict[book] = s
            else:
                external_book_completion_dict[book] = comp_xe


        #~ -------------------------------
        #~ -------------------------------

        internal_combinations_list = []
        external_combinations_list = []

        #~ for book,combo in internal_book_completion_dict.iteritems():
        for book,combo in iteritems(internal_book_completion_dict):
            if "||" in combo:
                internal_combinations_list.append(combo)
                #~ if DEBUG: print("internal combo: ", combo)
        #END FOR

        #~ for book,combo in external_book_completion_dict.iteritems():
        for book,combo in iteritems(external_book_completion_dict):
            if "||" in combo:
                external_combinations_list.append(combo)
                #~ if DEBUG: print("external combo: ", combo)
        #END FOR

        internal_combinations_set = set(internal_combinations_list)
        external_combinations_set = set(external_combinations_list)

        del external_combinations_list
        del internal_combinations_list

        reversed_combo_list = []

        for combo in internal_combinations_set:
            s = combo.split("||")
            value1 = s[0]
            value2 = s[1]
            s = value2 + "||" + value1
            reversed_combo_list.append(s)
        #END FOR
        for combo in reversed_combo_list:
            internal_combinations_set.add(combo)
        #END FOR
        del reversed_combo_list

        reversed_combo_list = []

        for combo in external_combinations_set:
            s = combo.split("||")
            value1 = s[0]
            value2 = s[1]
            s = value2 + "||" + value1
            reversed_combo_list.append(s)
        #END FOR
        for combo in reversed_combo_list:
            external_combinations_set.add(combo)
        #END FOR
        del reversed_combo_list

        final_combinations_set = internal_combinations_set.intersection(external_combinations_set)

        if DEBUG:
            final_combinations_list = list(final_combinations_set)
            final_combinations_list.sort()
            for combo in final_combinations_list:
                print("final combination: ", combo)
            del final_combinations_list

        #~ for book,combo in internal_book_completion_dict.iteritems():
        for book,combo in iteritems(internal_book_completion_dict):
            if combo in final_combinations_set:
                internal_books_to_keep_list.append(book)
                #~ if DEBUG: print("internal book kept: ", as_unicode(book))
        #END FOR

        #~ for book,combo in external_book_completion_dict.iteritems():
        for book,combo in iteritems(external_book_completion_dict):
            if combo in final_combinations_set:
                external_books_to_keep_list.append(book)
                #~ if DEBUG: print("external book kept: ", as_unicode(book))
        #END FOR

        if DEBUG:
            if len(internal_books_to_keep_list) == 0:
                print("No Internal Books to Keep Were Found in 'do_final_comparisons_both_std'")
            if len(external_books_to_keep_list) == 0:
                print("No External Books to Keep Were Found in 'do_final_comparisons_both_std'")

        internal_books_to_keep_list = list(set(internal_books_to_keep_list))
        external_books_to_keep_list = list(set(external_books_to_keep_list))

        return internal_books_to_keep_list, external_books_to_keep_list
    #---------------------------------------------------------------------------------------------------------------------------------------
    def apsw_connect_to_a_library_path(self,path):
        if isbytestring(path):
            path = path.decode(filesystem_encoding)
        path = path.replace(os.sep, '/')
        if DEBUG: print("apsw_connect_to_a_library_path",path)
        try:
            my_db = apsw.Connection(path)
        except Exception as e:
            if DEBUG: print(as_unicode(e))
            #~ path = path.encode("ascii", "ignore")
            msg = as_unicode("MCS CL Duplicates cannot use the path that you selected:" + path + " - " + as_unicode(e) )
            self.gui.status_bar.showMessage(msg)
            self.mcs_dialog.close()
            self.search_request_dialog()
            return None,None,False
        my_cursor = my_db.cursor()
        mysql = "PRAGMA main.busy_timeout = 15000;"      #PRAGMA busy_timeout = milliseconds;
        my_cursor.execute(mysql)

        return my_db,my_cursor,True
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def search_similarity_control(self,guidb,param_dict,sel_type,comparison_custom_column_dict,similarity_group_custom_column_dict):

        from difflib import SequenceMatcher
        self.sequencematcher = SequenceMatcher
        #----------------------------------------
        if not self.maingui:
            from calibre.gui2.ui import get_gui
            self.maingui = get_gui()
         #----------------------------------------

        self.gui.status_bar.showMessage("MCS Similarity Search: Executing Search")

        QApplication.instance().processEvents()

        total_start = time.time()

        self.display_progress_bar(0)  # never display it for this

        selected_books_list = []
        selected_books_list_str = self.get_selected_books(guidb,sel_type)
        for book in selected_books_list_str:
            book = int(book)
            selected_books_list.append(book)
        #END FOR
        del selected_books_list_str

        if len(selected_books_list) < 2:
            msg = "Insufficient Books Selected for Similarity Query"
            error_dialog(self.gui, _('MCS'),_((msg)), show=True)
            return

        found_list = []

        found_books_update_dict = {}

        self.is_taglike = False

        #-----------------------------
        column = param_dict['OPTIONS_SIMILARITY_QUERIES_COMPARISON_COLUMN']
        column = column.lower()
        group_prefix = column + "_"
        if group_prefix.startswith("#"):
            group_prefix = group_prefix[1: ]
            self.is_comparison_column_custom = True
            v_dict = comparison_custom_column_dict[column]
            datatype = v_dict['datatype']
            column_table1 = v_dict['table']
            if datatype == "text" or datatype == "enumeration" or datatype == "series":
                is_normalized = True
                column_table2 = "books_" + column_table1 + "_link"
                ismultiple = v_dict['is_multiple']
                if as_unicode(ismultiple) != as_unicode("{}"):   #e.g.  {u'ui_to_list': u',', u'list_to_ui': u', ', u'cache_to_list': u'|'}
                    self.is_taglike = True
            elif datatype == "comments":
                is_normalized = False
                column_table2 = None
            else:
                msg = "Similarity Group Custom Column Datatype Invalid: " + datatype + "for: " + column
                return error_dialog(self.gui, _('MCS'),_((msg)), show=True)
            if is_normalized:
                mysql_comparison = "SELECT value,'dummy' FROM " + column_table1 + \
                " WHERE id IN (SELECT value FROM " + column_table2 +  "  WHERE  book = ? AND value IS NOT NULL) AND value IS NOT NULL   "
            else:
                mysql_comparison = "SELECT value,'dummy' FROM " + column_table1 + \
                " WHERE book = ? AND value IS NOT NULL  "
        else:
            self.is_comparison_column_custom = False
            column_table1 = None
            column_table2 = None
            datatype = "text"
            if column == "title":
                is_normalized = False
                mysql_comparison = "SELECT title,'dummy' FROM books WHERE id = ? AND title IS NOT NULL  "
            elif column == "authors":
                is_normalized = True
                self.is_taglike = True
                mysql_comparison = "SELECT name,'dummy' FROM authors " \
                " WHERE authors.id IN (SELECT author FROM books_authors_link WHERE  book = ? AND author IS NOT NULL) AND name IS NOT NULL   "
            elif column == "series":
                is_normalized = True
                mysql_comparison = "SELECT name,'dummy' FROM series " \
                " WHERE series.id IN (SELECT series FROM books_series_link WHERE  book = ? AND series IS NOT NULL) AND name IS NOT NULL   "
            elif column == "publisher":
                is_normalized = True
                mysql_comparison = "SELECT name,'dummy' FROM publishers " \
                " WHERE publishers.id IN (SELECT publisher FROM books_publishers_link WHERE  book = ? AND publisher IS NOT NULL) AND name IS NOT NULL   "
            elif column == "tags":
                is_normalized = True
                self.is_taglike = True
                mysql_comparison = "SELECT name,'dummy' FROM tags " \
                " WHERE tags.id IN (SELECT tag FROM books_tags_link WHERE  book = ? AND tag IS NOT NULL) AND name IS NOT NULL   "
            elif column == "comments":
                is_normalized = False
                mysql_comparison = "SELECT text,'dummy' FROM comments WHERE book = ? AND text IS NOT NULL  "
            else:
                msg = "Comparison Standard Column is Not Supported: " + column
                return error_dialog(self.gui, _('MCS'),_((msg)), show=True)

        if DEBUG: print("comparison column is_taglike: ", self.is_taglike)
        #-----------------------------
        percent_val = param_dict['OPTIONS_SIMILARITY_QUERIES_PERCENTAGE']
        percent_val = (percent_val / 100)  # sequencematcher 100% == 1.00
        #-----------------------------
        cc = param_dict['OPTIONS_SIMILARITY_QUERIES_GROUP_UPDATE_TAGLIKE_CC']
        v_dict = similarity_group_custom_column_dict[cc]
        datatype = v_dict['datatype']
        cc_table1 = v_dict['table']
        if datatype == "text":
            cc_table2 = "books_" + cc_table1 + "_link"
        else:
            msg = "Similarity Group Custom Column Datatype Invalid; Must be Tag-Like: " + datatype + cc
            error_dialog(self.gui, _('MCS'),_((msg)), show=True)
            return
        #-----------------------------

        if column_table1 == cc_table1:
            msg = "Specified columns cannot be identical"
            error_dialog(self.gui, _('MCS'),_((msg)), show=True)
            return

        my_db,my_cursor = self.apsw_connect_to_current_guidb()

        self.remove_old_similarity_groups(my_db,my_cursor,selected_books_list,cc_table1,cc_table2,group_prefix)

        new_group_tags_set = set()

        n_books = len(selected_books_list)

        self.similarity_comparison_value_dict = {}
        if sel_type == "all" or n_books > 1000:
            self.load_similarity_comparison_value_dict(my_db,my_cursor,column,datatype,is_normalized,column_table1,column_table2)  #speed up comparison for lengthy textual values
            msg = "MCS Similarity Search: Executing Search of " + as_unicode(n_books) + " Books; wait..."
            self.gui.status_bar.showMessage(msg)
            QApplication.instance().processEvents()

        for current_book in selected_books_list:
            if not current_book in found_books_update_dict:
                similar_books_list = self.search_single_similarity_control(my_db,my_cursor,current_book,selected_books_list,is_normalized,\
                                                                        column_table1,column_table2,percent_val,mysql_comparison,found_books_update_dict)
                if len(similar_books_list) > 0:
                    group = group_prefix + as_unicode(current_book)
                    new_group_tags_set.add(group)
                    found_books_update_dict[current_book] = group
                    found_list.append(current_book)
                    for book in similar_books_list:
                        found_books_update_dict[book] = group
                        found_list.append(book)
                    #END FOR
                del similar_books_list
        #END FOR

        found_list = list(set(found_list))
        found_list.sort()

        #-----------------------------
        if sel_type == "selected":  # not "all"
            self.remove_orphaned_similarity_groups(my_db,my_cursor,new_group_tags_set,cc_table1,cc_table2)
        #-----------------------------
        self.add_new_similarity_groups(my_db,my_cursor,found_books_update_dict,new_group_tags_set,cc_table1,cc_table2)
        #-----------------------------
        my_db.close()
        #-----------------------------
        self.force_refresh_of_cache(found_list)
        #-----------------------------
        self.mark_found_books(found_list)
        #-----------------------------------------
        total_end = time.time()
        total_elapsed = total_end - total_start
        msg = 'MCS Similarity Search: searched selected books, updated groups in %d match(es) in %.3f seconds' %(len(found_list), total_elapsed)
        self.gui.status_bar.showMessage(msg)
        #-----------------------------------------
        del param_dict
        del found_list
        del found_books_update_dict
        del self.sequencematcher
        del SequenceMatcher
        del self.similarity_comparison_value_dict
    #-----------------------------------------------------------------------------------------
    def search_single_similarity_control(self,my_db,my_cursor,current_book,selected_books_list,\
                                                                is_normalized,column_table1,column_table2,percent_val,mysql_comparison,found_books_update_dict):
        #  a = value in current_book to be compared to all other selected books
        #  b = value in another selected book to be compared to 'a'

        similar_books_list = []

        a = self.get_comparison_value(my_db,my_cursor,current_book,mysql_comparison)
        if a is None:
            return similar_books_list

        for book in selected_books_list:
            if book != current_book:
                if not book in found_books_update_dict:  #for p = 1.00, this will be true much more often than p = .5 (so p = 1..00 is much slower to execute than p = .5)
                    b = self.get_comparison_value(my_db,my_cursor,book,mysql_comparison)
                    if b is not None:
                        if percent_val == 1.00 :
                            if a == b:  #much faster than letting the sequencematcher figure it out
                                similar_books_list.append(book)
                            continue
                        elif a == b:  #much faster than letting the sequencematcher figure it out
                            similar_books_list.append(book)
                            continue
                        else:
                            p = self.calculate_similarity(a,b)
                            if DEBUG: print("% similarity: ", as_unicode(p), " for: ", a, " & ", b)
                            if p >= percent_val:
                                similar_books_list.append(book)
                            continue
        #END FOR

        return similar_books_list
    #-----------------------------------------------------------------------------------------
    def get_comparison_value(self,my_db,my_cursor,book,mysql_comparison):
        if self.is_taglike:
            ab = self.get_comparison_value_taglike(my_db,my_cursor,book,mysql_comparison)
        else:
            ab = self.get_comparison_value_single(my_db,my_cursor,book,mysql_comparison)
        return ab
    #-----------------------------------------------------------------------------------------
    def get_comparison_value_single(self,my_db,my_cursor,book,mysql_comparison):
        ab = None
        if book in self.similarity_comparison_value_dict:
            ab = self.similarity_comparison_value_dict[book]
            if DEBUG: print("[1a] comparison value: ", ab, "       for book: ", as_unicode(book))
            return ab
        else:
            my_cursor.execute(mysql_comparison,([book]))
            tmp_rows = my_cursor.fetchall()
            if not tmp_rows:
                return None
            for row in tmp_rows:
                ab,dummy = row
                break
            #END FOR
            del tmp_rows
            if DEBUG: print("[1b] comparison value: ", ab, "       for book: ", as_unicode(book))
            return ab
    #-----------------------------------------------------------------------------------------
    def get_comparison_value_taglike(self,my_db,my_cursor,book,mysql_comparison):
        ab = None
        my_cursor.execute(mysql_comparison,([book]))
        tmp_rows = my_cursor.fetchall()
        if not tmp_rows:
            return None
        s = None
        tmp_rows.sort()        #for tag-like
        for row in tmp_rows:
            value,dummy = row
            if s is None:
                s = value
            else:
                s = s + value     #tag-like concatenation
        #END FOR
        del tmp_rows
        if s is not None:
            ab = s
        if DEBUG: print("[2] comparison value: ", ab, "       for book: ", as_unicode(book))
        return ab
    #-----------------------------------------------------------------------------------------
    def calculate_similarity(self,a,b):
        #returns the probability that string a is the same as string b
        try:
            p = self.sequencematcher(None, a, b).ratio()
            if p:
                return p
            else:
                return 0.0000
        except:
            return 0.0000
    #-----------------------------------------------------------------------------------------
    def remove_orphaned_similarity_groups(self,my_db,my_cursor,new_group_tags_set,cc_table1,cc_table2):
        #if user did not "select" previously identified books as being similar to a new similarity group tag, it would not have been removed as being "old" just previously
        #if user is "selecting all books", then this case cannot occur, since self.remove_old_similarity_groups() would have removed all similarity groups for the current group_prefix.
        mysql = "DELETE FROM " + cc_table1 + " WHERE value = ? "
        if DEBUG: print(mysql)
        my_cursor.execute("begin")
        for tag in new_group_tags_set:
            my_cursor.execute(mysql,([tag]))
        #END FOR
        my_cursor.execute("commit")

        #remove any book links that used the tags just deleted above.
        mysql = "DELETE FROM " + cc_table2 + " WHERE value NOT IN (SELECT id FROM " + cc_table1 + ")"
        if DEBUG: print(mysql)
        my_cursor.execute("begin")
        my_cursor.execute(mysql)
        my_cursor.execute("commit")
    #-----------------------------------------------------------------------------------------
    def remove_old_similarity_groups(self,my_db,my_cursor,selected_books_list,cc_table1,cc_table2,group_prefix):
        # selected books only; for current group_prefix only.
        mysql = "DELETE FROM " + cc_table2 + " WHERE book = ? AND value IN (SELECT id FROM " + cc_table1 + " WHERE value LIKE '" + group_prefix + "%')"
        if DEBUG: print(mysql)
        my_cursor.execute("begin")
        for book in selected_books_list:
            my_cursor.execute(mysql,([book]))
        #END FOR
        my_cursor.execute("commit")
    #-----------------------------------------------------------------------------------------
    def add_new_similarity_groups(self,my_db,my_cursor,found_books_update_dict,new_group_tags_set,cc_table1,cc_table2):
        #-----------------------------------------
        mysql = "INSERT OR IGNORE INTO " + cc_table1 + " (id,value) VALUES (null,?) "
        my_cursor.execute("begin")
        for tag in new_group_tags_set:
            my_cursor.execute(mysql,([tag]))
        #END FOR
        my_cursor.execute("commit")
        #-----------------------------------------
        mysql = "INSERT OR IGNORE INTO " + cc_table2 + " (id,book,value) VALUES (null,?,(SELECT id FROM " + cc_table1 + " WHERE value = ? AND id IS NOT NULL )  )"
        my_cursor.execute("begin")
        for book,tag in iteritems(found_books_update_dict):
            my_cursor.execute(mysql,(book,tag))
        #END FOR
        my_cursor.execute("commit")
    #-----------------------------------------------------------------------------------------
    def load_similarity_comparison_value_dict(self,my_db,my_cursor,column,datatype,is_normalized,column_table1,column_table2):
        if self.is_comparison_column_custom:
            if not is_normalized:
                mysql = "SELECT book,value FROM " + column_table1
            elif is_normalized:
                mysql = "SELECT book,(SELECT value FROM " + column_table1 + "  WHERE id = " + column_table2 +  ".value) FROM " + column_table2
            else:
                return
        else:
            if column == "title":
                mysql = "SELECT id,title FROM books"
            elif column == "authors":
                mysql = "SELECT book,(SELECT name FROM authors WHERE id = books_authors_link.author) FROM books_authors_link"
            elif column == "series":
                mysql = "SELECT book,(SELECT name FROM series WHERE id = books_series_link.series) FROM books_series_link"
            elif column == "publisher":
                mysql = "SELECT book,(SELECT name FROM publishers WHERE id = books_publishers_link.publisher) FROM books_publishers_link"
            elif column == "tags":
                mysql = "SELECT book,(SELECT name FROM tags WHERE id = books_tags_link.tag ) FROM books_tags_link "
            elif column == "comments":
                mysql = "SELECT book,text FROM comments"
            else:
                return

        if not self.is_taglike:
            my_cursor.execute(mysql)
            tmp_rows = my_cursor.fetchall()
            if not tmp_rows:
                return
            for row in tmp_rows:
                book,value = row
                self.similarity_comparison_value_dict[book] = value
            #END FOR
            del tmp_rows
        else:
            my_cursor.execute(mysql)
            tmp_rows = my_cursor.fetchall()
            if not tmp_rows:
                return
            for row in tmp_rows:
                book,value = row
                if not book in self.similarity_comparison_value_dict:
                    self.similarity_comparison_value_dict[book] = value
                else:
                    s = self.similarity_comparison_value_dict[book]
                    s = s + value  #concatenate tag-like & authors
                    self.similarity_comparison_value_dict[book] = s
            #END FOR
            del tmp_rows
    #-----------------------------------------------------------------------------------------
    def force_refresh_of_cache(self,found_list):
        if len(found_list) == 0:
            return
        db = self.maingui.current_db.new_api
        db.reload_from_db(clear_caches=False)
        frozen = db.all_book_ids()
        books = list(frozen)
        self.maingui.library_view.model().refresh_ids(books)
        self.maingui.tags_view.recount()
        QApplication.instance().processEvents()
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #~                                Generic functions called by each Tab.
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def load_criteria_settings_generic(self,current_tab=None):
        #~ prefs here are the "official" prefs imported as usual for a plugin's ui.py

        if current_tab is None:
            return

        parm_list,parm_dict,saveddate_for_name_dict = self.create_criteria_parm_list_generic(current_tab)

        settings_history_names_dict = prefs['SETTINGS_HISTORY_NAMES_DICT']
        if is_py3:
            settings_history_names_dict = as_unicode(settings_history_names_dict)
        else:
            settings_history_names_dict = as_bytes(settings_history_names_dict)
        settings_history_names_dict = ast.literal_eval(settings_history_names_dict)
        if not isinstance(settings_history_names_dict,dict):
            prefs['SETTINGS_HISTORY_NAMES_DICT'] = as_unicode({})
            settings_history_names_dict = {}
            if DEBUG: print("ERROR:  settings_history_names_dict is NOT a valid dict! ")

        items = []
        for parm in parm_list:
            saveddate = saveddate_for_name_dict[parm]
            if saveddate in settings_history_names_dict:
                name = settings_history_names_dict[saveddate]
                item = "[" + name + "]:" + parm
            else:
                item = "[ ]:" + parm
            items.append(item)
        #END FOR
        del parm_list

        if len(items) == 0:
            if DEBUG: print("nothing to show.......returning........")
            return

        selected_parm,ok = QInputDialog.getItem(None,"Saved Criteria","Select Criteria Previously Saved",items,0,False)
        del items
        if (not ok) or (selected_parm is None) or (not "❟" in selected_parm):   # "[1]❟[2]❟[3]❟[4]❟[5]❟[6]❟[7]❟[8]❟[9]❟[10]❟[11]❟[12]❟[13]❟[14]❟[15]❟[16]"
            return

        is_valid = self.unpack_selected_parm_generic(current_tab,selected_parm,parm_dict)

        if not is_valid:
            msg = "Loaded Criteria are invalid, and cannot be used to restore the Criteria..."
        else:
            msg = "Criteria have been loaded successfully..."
        self.gui.status_bar.showMessage(msg)
    #-----------------------------------------------------------------------------------------------
    def create_criteria_parm_list_generic(self,current_tab):
        #~ prefs here are the "official" prefs imported as usual for a plugin's ui.py

        PARM_MAX_DISPLAY_LENGTH = 100
        parm_list = []
        parm_dict = {}
        saveddate_for_name_dict = {}
        ITEM = "[1]❟[2]❟[3]❟[4]❟[5]❟[6]❟[7]❟[8]❟[9]❟[10]❟[11]❟[12]❟[13]❟[14]❟[15]❟[16]"
        n = 0

        settings_history_dict = prefs['SETTINGS_HISTORY_DICT']
        if is_py3:
            settings_history_dict = as_unicode(settings_history_dict)
        else:
            settings_history_dict = as_bytes(settings_history_dict)
        settings_history_dict = ast.literal_eval(settings_history_dict)
        if not isinstance(settings_history_dict,dict):
            prefs['SETTINGS_HISTORY_DICT'] = as_unicode({})
            settings_history_dict = {}
            if DEBUG: print("ERROR:  settings_history_dict is NOT a valid dict!")

        if len(settings_history_dict) == 0:
            return parm_list,parm_dict,saveddate_for_name_dict

        if current_tab == SEARCHTAB:
            for saveddate,dict_list in iteritems(settings_history_dict):
                finished = False
                for item_dict in dict_list:
                    if isinstance(item_dict,dict):
                        if INTRA_BOOK_SEARCH_TYPE in item_dict:
                            type = INTRA_BOOK_SEARCH_TYPE
                            typ = 'INTRA'
                        elif INTER_BOOK_SEARCH_TYPE in item_dict:
                            type = INTER_BOOK_SEARCH_TYPE
                            typ = 'INTER'
                        else:
                            continue
                        for ktype,vdict in iteritems(item_dict):
                            if ktype == type:
                                if isinstance(vdict,dict):
                                    parm = ITEM #   "[1]❟[2]❟[3]❟[4]❟[5]❟[6]❟[7]❟[8]❟[9]❟[10]❟[11]❟[12]❟[13]❟[14]❟[15]❟[16]"
                                    parm =  parm.replace("[1]",saveddate)
                                    parm =  parm.replace("[2]",typ)
                                    parm =  parm.replace("[3]",vdict['NAME1'])
                                    parm =  parm.replace("[4]",vdict['TEXT1'])
                                    parm =  parm.replace("[5]",vdict['TYPE1'])
                                    parm =  parm.replace("[6]",vdict['OPERATOR1'])
                                    parm =  parm.replace("[7]",vdict['ANDORNOT'])
                                    parm =  parm.replace("[8]",vdict['NAME2'])
                                    parm =  parm.replace("[9]",vdict['TEXT2'])
                                    parm =  parm.replace("[10]",vdict['TYPE2'])
                                    parm =  parm.replace("[11]",vdict['OPERATOR2'])
                                    s = "UFF:" + vdict['USE_FINAL_FILTERS']
                                    parm =  parm.replace("[12]",s)
                                    s = "AAB:" + vdict['ALL_AUTHORS_BOOKS']
                                    parm =  parm.replace("[13]",s)
                                    s = "CLD:" + vdict['CROSS_LIBRARY_DUPLICATES_SEARCH']
                                    parm = parm.replace("[14]",s)
                                    parm = parm.replace('❟[15]❟[16]','') #not used for this tab
                                    parm = parm[0:PARM_MAX_DISPLAY_LENGTH]  #do not allow QInputDialog popup to overflow the screen...
                                    parm = parm.strip() + " #" + as_unicode(n)
                                    parm_list.append(parm)
                                    parm_dict[n] = saveddate,type,vdict,parm  #for actual restoration of loaded criteria, not for the user to "pick one"...
                                    saveddate_for_name_dict[parm] = saveddate
                                    finished = True
                                    n = n + 1
                                    if DEBUG: print("SearchTab parm: ", parm)
                        #END FOR
                    else:
                        pass
                    if finished:
                        break
                #END FOR
            #END FOR
        #----------------------------------------
        elif current_tab == REGEXTAB:
            for saveddate,dict_list in iteritems(settings_history_dict):
                finished = False
                for item_dict in dict_list:
                    if isinstance(item_dict,dict):
                        if REGULAR_EXPRESSIONS_TYPE in item_dict:
                            type = REGULAR_EXPRESSIONS_TYPE
                            typ = 'REGEX'
                        else:
                            continue
                        for ktype,vdict in iteritems(item_dict):
                            if ktype == type:
                                if DEBUG: print("== ...", ktype,type)
                                if isinstance(vdict,dict):
                                    parm = ITEM #  "[1]❟[2]❟[3]❟[4]❟[5]❟[6]❟[7]❟[8]❟[9]❟[10]❟[11]❟[12]❟[13]❟[14]❟[15]❟[16]"
                                    parm =  parm.replace("[1]",saveddate)
                                    parm =  parm.replace("[2]",typ)
                                    s = self.shorten_regex_values_for_parm(vdict['REGEX1'])
                                    parm =  parm.replace("[3]",s)
                                    s = vdict['REGEX1#']
                                    n = int(s) + 1  #user sees choice #0 as #1...
                                    parm =  parm.replace("[4]",s)
                                    s = self.shorten_regex_values_for_parm(vdict['REGEX2'])
                                    parm =  parm.replace("[5]",s)
                                    s = vdict['REGEX2#']
                                    n = int(s) + 1  #user sees choice #0 as #1...
                                    parm =  parm.replace("[6]",s)
                                    s = self.shorten_regex_values_for_parm(vdict['REGEXVALUE_0'])
                                    parm =  parm.replace("[7]",s)
                                    s = self.shorten_regex_values_for_parm(vdict['REGEXVALUE_1'])
                                    parm =  parm.replace("[8]",s)
                                    s = self.shorten_regex_values_for_parm(vdict['REGEXVALUE_2'])
                                    parm =  parm.replace("[9]",s)
                                    s = self.shorten_regex_values_for_parm(vdict['REGEXVALUE_3'])
                                    parm =  parm.replace("[10]",s)
                                    s = self.shorten_regex_values_for_parm(vdict['REGEXVALUE_4'])
                                    parm =  parm.replace("[11]",s)
                                    s = self.shorten_regex_values_for_parm(vdict['REGEXVALUE_5'])
                                    parm =  parm.replace("[12]",s)
                                    s = self.shorten_regex_values_for_parm(vdict['REGEXVALUE_6'])
                                    parm =  parm.replace("[13]",s)
                                    s = self.shorten_regex_values_for_parm(vdict['REGEXVALUE_7'])
                                    parm =  parm.replace("[14]",s)
                                    s = self.shorten_regex_values_for_parm(vdict['REGEXVALUE_8'])
                                    parm =  parm.replace("[15]",s)
                                    s = self.shorten_regex_values_for_parm(vdict['REGEXVALUE_9'])
                                    parm =  parm.replace("[16]",s)
                                    parm = parm[0:PARM_MAX_DISPLAY_LENGTH]  #do not allow QInputDialog popup to overflow the screen...
                                    parm = parm + " #" + as_unicode(n)
                                    parm_list.append(parm)
                                    parm_dict[n] = saveddate,type,vdict,parm  #for actual restoration of loaded criteria, not for the user to "pick one"...
                                    saveddate_for_name_dict[parm] = saveddate
                                    finished = True
                                    n = n + 1
                                    if DEBUG: print("RegexTab parm: ", parm)
                                else:
                                    pass
                            else:
                                pass
                        #END FOR
                    else:
                        pass
                    if finished:
                        break
                #END FOR
            #END FOR
        #----------------------------------------
        elif current_tab == FINALFILTERTAB:
          for saveddate,dict_list in iteritems(settings_history_dict):
                finished = False
                for item_dict in dict_list:  #
                    if isinstance(item_dict,dict):
                        if FINAL_FILTERS_TYPE in item_dict:
                            type = FINAL_FILTERS_TYPE
                        else:
                            continue
                        for ktype,vdict in iteritems(item_dict):  #   ktype = 'FINAL_FILTERS'
                            if ktype == type:
                                if isinstance(vdict,dict):
                                    parm = ITEM #   "[1]❟[2]❟[3]❟[4]❟[5]❟[6]❟[7]❟[8]❟[9]❟[10]❟[11]❟[12]❟[13]❟[14]❟[15]❟[16]"
                                    parm =  parm.replace("[1]",saveddate)
                                    parm =  parm.replace(",[2]",type)    # type
                                    s = vdict['FINAL_FILTER_0']
                                    parm =  parm.replace("[3]",s)
                                    s = vdict['FINAL_FILTER_1']
                                    parm =  parm.replace("[4]",s)
                                    s = vdict['FINAL_FILTER_2']
                                    parm =  parm.replace("[5]",s)
                                    s = vdict['FINAL_FILTER_3']
                                    parm =  parm.replace("[6]",s)
                                    s = vdict['FINAL_FILTER_4']
                                    parm =  parm.replace("[7]",s)
                                    s = vdict['FINAL_FILTER_5']
                                    parm =  parm.replace("[8]",s)
                                    s = vdict['FINAL_FILTER_6']
                                    parm =  parm.replace("[9]",s)
                                    s = vdict['FINAL_FILTER_7']
                                    parm =  parm.replace("[10]",s)
                                    s = vdict['FINAL_FILTER_8']
                                    parm =  parm.replace("[11]",s)
                                    s = vdict['FINAL_FILTER_9']
                                    parm =  parm.replace("[12]",s)
                                    parm = parm.replace('❟[13]❟[14]❟[15]❟[16]', '') #too long for QInputDialog popup to show...
                                    parm = parm[0:PARM_MAX_DISPLAY_LENGTH]  #do not allow QInputDialog popup to overflow the screen...
                                    parm = parm + "...#" + as_unicode(n)
                                    parm_list.append(parm)
                                    parm_dict[n] = saveddate,type,vdict,parm  #for actual restoration of loaded criteria, not for the user to "pick one"...
                                    saveddate_for_name_dict[parm] = saveddate
                                    finished = True
                                    n = n + 1
                                    if DEBUG: print("Final Filters parm: ", parm)
                        #END FOR
                    else:
                        if DEBUG: print("ERROR in settings_history_dict:  item in list of dicts is NOT a valid dict: ", as_unicode(v))
                    if finished:
                        break
                #END FOR
            #END FOR
        #----------------------------------------
        elif current_tab == SPECIALQUERIESTAB:
            for saveddate,dict_list in iteritems(settings_history_dict):
                finished = False
                for item_dict in dict_list:  #
                    if isinstance(item_dict,dict):
                        if SPECIAL_QUERIES_TYPE in item_dict:
                            type = SPECIAL_QUERIES_TYPE
                        else:
                            continue
                        for ktype,vdict in iteritems(item_dict):
                            if ktype == type:
                                if isinstance(vdict,dict):
                                    parm = ITEM #   "[1]❟[2]❟[3]❟[4]❟[5]❟[6]❟[7]❟[8]❟[9]❟[10]❟[11]❟[12]❟[13]❟[14]❟[15]❟[16]"
                                    parm =  parm.replace("[1]",saveddate)
                                    parm =  parm.replace("[2]",type)
                                    val1 = vdict['TABLE_1_1']
                                    parm =  parm.replace("[3]",val1)
                                    val2 = vdict['OPERATOR_1']
                                    parm =  parm.replace("[4]",val2)
                                    val3 = vdict['TABLE_1_2']
                                    parm =  parm.replace("[5]",val3)
                                    status = vdict['SPECIAL_QUERY_AND_OR_INACTIVE_1']
                                    parm =  parm.replace("[6]",status)
                                    parm =  parm.replace("❟[7]❟[8]❟[9]❟[10]❟[11]❟[12]❟[13]❟[14]❟[15]❟[16]","")#too long for QInputDialog popup to show...
                                    parm = parm[0:PARM_MAX_DISPLAY_LENGTH]  #do not allow QInputDialog popup to overflow the screen...
                                    parm = parm + "...#" + as_unicode(n)
                                    parm_list.append(parm)
                                    parm_dict[n] = saveddate,type,vdict,parm  #for actual restoration of loaded criteria, not for the user to "pick one"...
                                    saveddate_for_name_dict[parm] = saveddate
                                    finished = True
                                    n = n + 1
                                    if DEBUG: print("SPECIALQUERIESTAB parm: ", parm)
        #----------------------------------------
        elif current_tab == SIMILARITYTAB:
            for saveddate,dict_list in iteritems(settings_history_dict):
                finished = False
                for item_dict in dict_list:  #
                    if isinstance(item_dict,dict):
                        if SIMILARITY_QUERIES_TYPE in item_dict:
                            type = SIMILARITY_QUERIES_TYPE
                        else:
                            continue
                        for ktype,vdict in iteritems(item_dict):
                            if ktype == type:
                                if isinstance(vdict,dict):
                                    parm = ITEM #   "[1]❟[2]❟[3]❟[4]❟[5]❟[6]❟[7]❟[8]❟[9]❟[10]❟[11]❟[12]❟[13]❟[14]❟[15]❟[16]"
                                    parm =  parm.replace("[1]",saveddate)
                                    parm =  parm.replace("[2]",type)
                                    column = vdict['OPTIONS_SIMILARITY_QUERIES_COMPARISON_COLUMN']
                                    parm =  parm.replace("[3]",column)
                                    percent_val = vdict['OPTIONS_SIMILARITY_QUERIES_PERCENTAGE']
                                    parm =  parm.replace("[4]",as_unicode(percent_val))
                                    cc = vdict['OPTIONS_SIMILARITY_QUERIES_GROUP_UPDATE_TAGLIKE_CC']
                                    parm =  parm.replace("[5]",cc)
                                    parm =  parm.replace("❟[6]❟[7]❟[8]❟[9]❟[10]❟[11]❟[12]❟[13]❟[14]❟[15]❟[16]","")#too long for QInputDialog popup to show...
                                    parm = parm[0:PARM_MAX_DISPLAY_LENGTH]  #do not allow QInputDialog popup to overflow the screen...
                                    parm = parm + "...#" + as_unicode(n)
                                    parm_list.append(parm)
                                    parm_dict[n] = saveddate,type,vdict,parm  #for actual restoration of loaded criteria, not for the user to "pick one"...
                                    saveddate_for_name_dict[parm] = saveddate
                                    finished = True
                                    n = n + 1
                                    if DEBUG: print("SIMILARITY_QUERIES parm: ", parm)
        #----------------------------------------
        elif current_tab == RAWSQLTAB:
            for saveddate,dict_list in iteritems(settings_history_dict):
                finished = False
                for item_dict in dict_list:  #
                    if isinstance(item_dict,dict):
                        if SQL_QUERIES_TYPE in item_dict:
                            type = SQL_QUERIES_TYPE
                        else:
                            continue
                        for ktype,vdict in iteritems(item_dict):
                            if ktype == type:
                                if isinstance(vdict,dict):
                                    parm = ITEM #   "[1]❟[2]❟[3]❟[4]❟[5]❟[6]❟[7]❟[8]❟[9]❟[10]❟[11]❟[12]❟[13]❟[14]❟[15]❟[16]"
                                    parm =  parm.replace("[1]",saveddate)
                                    parm =  parm.replace("[2]",type)
                                    sql = vdict['RAW_SQL_QUERY_LAST_SAVED']
                                    parm =  parm.replace("[3]",sql)
                                    parm = parm.replace("❟[4]❟[5]❟[6]❟[7]❟[8]❟[9]❟[10]❟[11]❟[12]❟[13]❟[14]❟[15]❟[16]","")#too long for QInputDialog popup to show...
                                    parm = parm[0:PARM_MAX_DISPLAY_LENGTH]  #do not allow QInputDialog popup to overflow the screen...
                                    parm = parm + "...#" + as_unicode(n)
                                    parm_list.append(parm)
                                    parm_dict[n] = saveddate,type,vdict,parm  #for actual restoration of loaded criteria, not for the user to "pick one"...
                                    saveddate_for_name_dict[parm] = saveddate
                                    finished = True
                                    n = n + 1
                                    if DEBUG: print("create_criteria_parm_list_generic         RAW_SQL_QUERIES parm: ", parm)
        #----------------------------------------
        elif current_tab == VIRTUALCOLUMNTXTTAB:
            for saveddate,dict_list in iteritems(settings_history_dict):
                finished = False
                for item_dict in dict_list:  #
                    if isinstance(item_dict,dict):
                        if TXT_QUERIES_TYPE in item_dict:
                            type = TXT_QUERIES_TYPE
                        else:
                            continue
                        for ktype,vdict in iteritems(item_dict):
                            if ktype == type:
                                if isinstance(vdict,dict):
                                    parm = ITEM #   "[1]❟[2]❟[3]❟[4]❟[5]❟[6]❟[7]❟[8]❟[9]❟[10]❟[11]❟[12]❟[13]❟[14]❟[15]❟[16]"
                                    parm =  parm.replace("[1]",saveddate)
                                    parm =  parm.replace("[2]",type)
                                    parm =  parm.replace("[3]",vdict['TXT_SEARCH_TERMS'].strip() )
                                    parm =  parm.replace("❟[4]❟[5]❟[6]❟[7]❟[8]❟[9]❟[10]❟[11]❟[12]❟[13]❟[14]❟[15]❟[16]","")#too long for QInputDialog popup to show...
                                    parm = parm[0:PARM_MAX_DISPLAY_LENGTH]  #do not allow QInputDialog popup to overflow the screen...
                                    parm = parm + "...#" + as_unicode(n)
                                    parm_list.append(parm)
                                    parm_dict[n] = saveddate,type,vdict,parm  #for actual restoration of loaded criteria, not for the user to "pick one"...
                                    saveddate_for_name_dict[parm] = saveddate
                                    finished = True
                                    n = n + 1
                                    if DEBUG: print("TXT_QUERIES parm: ", parm)
        #----------------------------------------
        elif current_tab == WORDBOOKINDEXQUERYTAB:
            for saveddate,dict_list in iteritems(settings_history_dict):
                finished = False
                for item_dict in dict_list:  #
                    if isinstance(item_dict,dict):
                        if WORD_QUERIES_TYPE in item_dict:
                            type = WORD_QUERIES_TYPE
                        else:
                            continue
                        for ktype,vdict in iteritems(item_dict):
                            if ktype == type:
                                if isinstance(vdict,dict):
                                    parm = ITEM #   "[1]❟[2]❟[3]❟[4]❟[5]❟[6]❟[7]❟[8]❟[9]❟[10]❟[11]❟[12]❟[13]❟[14]❟[15]❟[16]"
                                    parm =  parm.replace("[1]",saveddate)
                                    parm =  parm.replace("[2]",type)
                                    parm =  parm.replace("[3]",vdict['WORD_SEARCH_TERMS'].strip() )
                                    parm =  parm.replace("❟[4]❟[5]❟[6]❟[7]❟[8]❟[9]❟[10]❟[11]❟[12]❟[13]❟[14]❟[15]❟[16]","")#too long for QInputDialog popup to show...
                                    parm = parm[0:PARM_MAX_DISPLAY_LENGTH]  #do not allow QInputDialog popup to overflow the screen...
                                    parm = parm + "...#" + as_unicode(n)
                                    parm_list.append(parm)
                                    parm_dict[n] = saveddate,type,vdict,parm  #for actual restoration of loaded criteria, not for the user to "pick one"...
                                    saveddate_for_name_dict[parm] = saveddate
                                    finished = True
                                    n = n + 1
                                    if DEBUG: print("WORD_QUERIES parm: ", parm)
        #----------------------------------------
        elif current_tab == RESULTSTAB:
            for saveddate,dict_list in iteritems(settings_history_dict):
                finished = False
                for item_dict in dict_list:  #
                    if isinstance(item_dict,dict):
                        if POST_SEARCH_TYPE in item_dict:
                            type = POST_SEARCH_TYPE
                        else:
                            continue
                        for ktype,vdict in iteritems(item_dict):
                            if ktype == type:
                                if isinstance(vdict,dict):
                                    parm = ITEM #   "[1]❟[2]❟[3]❟[4]❟[5]❟[6]❟[7]❟[8]❟[9]❟[10]❟[11]❟[12]❟[13]❟[14]❟[15]❟[16]"
                                    parm =  parm.replace("[1]",saveddate)
                                    parm =  parm.replace("[2]",type)
                                    s = vdict['TARGET_CUSTOM_COLUMN']
                                    parm =  parm.replace("[3]",as_unicode(s))
                                    s = vdict['NEW_VALUE']
                                    parm =  parm.replace("[4]",as_unicode(s))
                                    s = vdict['NEW_VALUE_OBJECT']
                                    parm =  parm.replace("[5]",as_unicode(s))
                                    s = vdict['SOURCE_COLUMN']
                                    parm =  parm.replace("[6]",as_unicode(s))
                                    parm = parm.replace("❟[7]❟[8]❟[9]❟[10]❟[11]❟[12]❟[13]❟[14]❟[15]❟[16]","")#too long for QInputDialog popup to show...
                                    parm = parm[0:PARM_MAX_DISPLAY_LENGTH]  #do not allow QInputDialog popup to overflow the screen...
                                    parm = parm + "...#" + as_unicode(n)
                                    parm_list.append(parm)
                                    parm_dict[n] = saveddate,type,vdict,parm  #for actual restoration of loaded criteria, not for the user to "pick one"...
                                    saveddate_for_name_dict[parm] = saveddate
                                    finished = True
                                    n = n + 1
                                    if DEBUG: print("POST_SEARCH parm: ", parm)
        #----------------------------------------
        else:
            pass
        #----------------------------------------
        parm_list.sort(reverse=True)

        return parm_list,parm_dict,saveddate_for_name_dict
    #-----------------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------------
    def shorten_regex_values_for_parm(self,s):
        s = s.strip()
        if '(?#' in s:
            s_split = s.split('(?#')
            s = s_split[0].strip()
        if len(s) > 50:
            s = s[0:50]
        return s
    #-----------------------------------------------------------------------------------------------
    def save_settings_parm_history_to_prefs_generic(self,mcsprefs,current_tab=None,param_dict=None):
        #~ Important: the ui.py prefs are the real "master" instantiated prefs, so the mcsdialog prefs must always explicitly use the ui.py prefs passed originally to mcsdialog...
        #~ Since ui.py 'prefs' is imported from Calibre and MCS, use the parameter name 'mcsprefs' in the ..._generic functions instead of 'prefs' to avoid conflict or confusion.
        #----------------------------------------
        if current_tab is None or param_dict is None:
            return mcsprefs
        if not isinstance(param_dict,dict):
            return mcsprefs
        #~ ---------------------------------------------
        if not self.maingui:
            from calibre.gui2.ui import get_gui
            self.maingui = get_gui()
         #----------------------------------------

        if DEBUG: print("###########[1] Saving: ", current_tab, " ########################")

        settings_history_dict = mcsprefs['SETTINGS_HISTORY_DICT']
        if is_py3:
            settings_history_dict = as_unicode(settings_history_dict)
        else:
            settings_history_dict = as_bytes(settings_history_dict)
        settings_history_dict = ast.literal_eval(settings_history_dict)
        if not isinstance(settings_history_dict,dict):
            if DEBUG: print("settings_history_dict is NOT a valid dict! Aborting!")
            return
        if len(settings_history_dict) == 0:
            if DEBUG: print("settings_history_dict is empty...never previously saved...")
        #~ ---------------------------------------------

        settings_history_names_dict = mcsprefs['SETTINGS_HISTORY_NAMES_DICT']
        if is_py3:
            settings_history_names_dict = as_unicode(settings_history_names_dict)
        else:
            settings_history_names_dict = as_bytes(settings_history_names_dict)
        settings_history_names_dict = ast.literal_eval(settings_history_names_dict)
        if not isinstance(settings_history_names_dict,dict):
            if DEBUG: print("settings_history_names_dict is NOT a valid dict!")
            settings_history_names_dict = {}
        #~ ---------------------------------------------

        if current_tab == SEARCHTAB:
            if INTER_BOOK_SEARCH_TYPE in param_dict:
                ans = param_dict[INTER_BOOK_SEARCH_TYPE]
                if ans == "False":
                    type = INTRA_BOOK_SEARCH_TYPE
                else:
                    type = INTER_BOOK_SEARCH_TYPE
                mcsprefs = self.prefs_history_save_util(current_tab,type,settings_history_dict,settings_history_names_dict,param_dict,mcsprefs)
            return mcsprefs
        elif current_tab == REGEXTAB:
            type = REGULAR_EXPRESSIONS_TYPE
            mcsprefs = self.prefs_history_save_util(current_tab,type,settings_history_dict,settings_history_names_dict,param_dict,mcsprefs)
            return mcsprefs
        elif current_tab == FINALFILTERTAB:
            type = FINAL_FILTERS_TYPE
            mcsprefs = self.prefs_history_save_util(current_tab,type,settings_history_dict,settings_history_names_dict,param_dict,mcsprefs)
            return mcsprefs
        elif current_tab == SPECIALQUERIESTAB:
            type = SPECIAL_QUERIES_TYPE
            mcsprefs = self.prefs_history_save_util(current_tab,type,settings_history_dict,settings_history_names_dict,param_dict,mcsprefs)
            return mcsprefs
        elif current_tab == SIMILARITYTAB:
            type = SIMILARITY_QUERIES_TYPE
            mcsprefs = self.prefs_history_save_util(current_tab,type,settings_history_dict,settings_history_names_dict,param_dict,mcsprefs)
            return mcsprefs
        elif current_tab == RAWSQLTAB:
            type = SQL_QUERIES_TYPE
            mcsprefs = self.prefs_history_save_util(current_tab,type,settings_history_dict,settings_history_names_dict,param_dict,mcsprefs)
            return mcsprefs
        elif current_tab == VIRTUALCOLUMNTXTTAB:
            type = TXT_QUERIES_TYPE
            mcsprefs = self.prefs_history_save_util(current_tab,type,settings_history_dict,settings_history_names_dict,param_dict,mcsprefs)
            return mcsprefs
        elif current_tab == WORDBOOKINDEXQUERYTAB:
            type = WORD_QUERIES_TYPE
            mcsprefs = self.prefs_history_save_util(current_tab,type,settings_history_dict,settings_history_names_dict,param_dict,mcsprefs)
            return mcsprefs
        elif current_tab == RESULTSTAB:
            type = POST_SEARCH_TYPE
            mcsprefs = self.prefs_history_save_util(current_tab,type,settings_history_dict,settings_history_names_dict,param_dict,mcsprefs)
            return mcsprefs
    #-----------------------------------------------------------------------------------------
    def prefs_history_save_util(self,current_tab,type,settings_history_dict,settings_history_names_dict,param_dict,mcsprefs):
        #~ Important: the ui.py prefs are the real "master" instantiated prefs, so the mcsdialog prefs must always explicitly use the ui.py prefs passed originally to mcsdialog...
        #~ Since ui.py 'prefs' is imported from Calibre and MCS, use the parameter name 'mcsprefs' instead of 'prefs' to avoid conflict or confusion.
        item_dict = {}
        item_dict[type] = param_dict
        item_list = []
        item_list.append(item_dict)
        now = as_unicode(time.strftime("%Y-%b-%d-%H:%M:%S", time.localtime(time.time())))  # IMPORTANT:  datetime is a dict key, so :%S is required...uniqueness...
        keep = self.prefs_history_avoid_duplicates(current_tab,type,settings_history_dict,now,item_dict)
        if keep:
            settings_history_dict[now] = item_list
            mcsprefs['SETTINGS_HISTORY_DICT'] = as_unicode(settings_history_dict)
            qid = QInputDialog()
            qid.setOption(QInputDialog.UsePlainTextEditForTextInput)
            name,ok = qid.getText(self.maingui,"Name New (Unique) Criteria for Future Ease of Use","Enter its Name")
            if (not ok) or (not name) or (not name > ""):
                name = "Unnamed"
            settings_history_names_dict[now] = name
            mcsprefs['SETTINGS_HISTORY_NAMES_DICT'] = as_unicode(settings_history_names_dict)
            if DEBUG: print(type + " -- Criteria have NOT been previously saved: ", type, now, name)
        else:
            if DEBUG: print("Duplicate Criteria; saving is not necessary... ", now)
        return mcsprefs
    #-----------------------------------------------------------------------------------------
    def prefs_history_avoid_duplicates(self,current_tab,type,settings_history_dict,now,item_dict):
        #---------------------------------------------
        #~ SETTINGS_HISTORY_DICT INTERNAL STRUCTURE:
        #---------------------------------------------
        #~ item_dict[type] = param_dict or regex_dict or other tab-specific_dict
        #~ item_list.append(item_dict)
        #~ settings_history_dict[now] = item_list
        #---------------------------------------------
        keep = True

        if current_tab == SEARCHTAB:
            for saveddate,item_list in iteritems(settings_history_dict):
                for saved_dict in item_list:
                    if INTRA_BOOK_SEARCH_TYPE in saved_dict or INTER_BOOK_SEARCH_TYPE in saved_dict:
                        if saved_dict == item_dict:
                            keep = False
                            if DEBUG: print(current_tab, " -- Criteria have previously been saved; Do NOT Keep this duplicate. ")
                #END FOR
            #END FOR
        elif current_tab == REGEXTAB:
            for saveddate,item_list in iteritems(settings_history_dict):
                for saved_dict in item_list:
                    if REGULAR_EXPRESSIONS_TYPE in saved_dict:
                        if saved_dict == item_dict:
                            keep = False
                            if DEBUG: print(current_tab, " -- Criteria have previously been saved; Do NOT Keep this duplicate. ")
                #END FOR
            #END FOR
        elif current_tab == FINALFILTERTAB:
            for saveddate,item_list in iteritems(settings_history_dict):
                for saved_dict in item_list:
                    if FINAL_FILTERS_TYPE in saved_dict:
                        if saved_dict == item_dict:
                            keep = False
                            if DEBUG: print(current_tab, " -- Criteria have previously been saved; Do NOT Keep this duplicate. ")
                #END FOR
            #END FOR
        elif current_tab == SPECIALQUERIESTAB:
            for saveddate,item_list in iteritems(settings_history_dict):
                for saved_dict in item_list:
                    if SPECIAL_QUERIES_TYPE in saved_dict:
                        if saved_dict == item_dict:
                            keep = False
                            if DEBUG: print(current_tab, " -- Criteria have previously been saved; Do NOT Keep this duplicate. ")
                #END FOR
            #END FOR
        elif current_tab == SIMILARITYTAB:
            for saveddate,item_list in iteritems(settings_history_dict):
                for saved_dict in item_list:
                    if SIMILARITY_QUERIES_TYPE in saved_dict:
                        if saved_dict == item_dict:
                            keep = False
                            if DEBUG: print(current_tab, " -- Criteria have previously been saved; Do NOT Keep this duplicate. ")
                #END FOR
            #END FOR
        elif current_tab == RAWSQLTAB:
            for saveddate,item_list in iteritems(settings_history_dict):
                for saved_dict in item_list:
                    if SQL_QUERIES_TYPE in saved_dict:
                        if saved_dict == item_dict:
                            keep = False
                            if DEBUG: print(current_tab, " -- Criteria have previously been saved; Do NOT Keep this duplicate. ")
                #END FOR
            #END FOR
        elif current_tab == VIRTUALCOLUMNTXTTAB:
            for saveddate,item_list in iteritems(settings_history_dict):
                for saved_dict in item_list:
                    if TXT_QUERIES_TYPE in saved_dict:
                        if saved_dict == item_dict:
                            keep = False
                            if DEBUG: print(current_tab, " -- Criteria have previously been saved; Do NOT Keep this duplicate. ")
                #END FOR
            #END FOR
        elif current_tab == WORDBOOKINDEXQUERYTAB:
            for saveddate,item_list in iteritems(settings_history_dict):
                for saved_dict in item_list:
                    if WORD_QUERIES_TYPE in saved_dict:
                        if saved_dict == item_dict:
                            keep = False
                            if DEBUG: print(current_tab, " -- Criteria have previously been saved; Do NOT Keep this duplicate. ")
                #END FOR
            #END FOR
        elif current_tab == RESULTSTAB:
            for saveddate,item_list in iteritems(settings_history_dict):
                for saved_dict in item_list:
                    if POST_SEARCH_TYPE in saved_dict:
                        if saved_dict == item_dict:
                            keep = False
                            if DEBUG: print(current_tab, " -- Criteria have previously been saved; Do NOT Keep this duplicate. ")
                #END FOR
            #END FOR
        return keep
    #-----------------------------------------------------------------------------------------
    def unpack_selected_parm_generic(self,current_tab,selected_parm,parm_dict):

        is_valid = True

        if current_tab == SEARCHTAB:
            is_valid = self.mcs_dialog.MCSSearchTab.unpack_selected_parm_searchtab(selected_parm,parm_dict)
        elif current_tab == REGEXTAB:
            is_valid = self.mcs_dialog.MCSRegexTab.unpack_selected_parm_regextab(selected_parm,parm_dict)
        elif current_tab == FINALFILTERTAB:
            is_valid = self.mcs_dialog.MCSFinalFilterTab.unpack_selected_parm_finalfiltertab(selected_parm,parm_dict)
        elif current_tab == SPECIALQUERIESTAB:
            is_valid = self.mcs_dialog.MCSSpecialQueriesTab.unpack_selected_parm_specialqueriestab(selected_parm,parm_dict)
        elif current_tab == SIMILARITYTAB:
            is_valid = self.mcs_dialog.MCSSimilarityTab.unpack_selected_parm_similaritytab(selected_parm,parm_dict)
        elif current_tab == RAWSQLTAB:
            is_valid = self.mcs_dialog.MCSRawSQLTab.unpack_selected_parm_rawsqltab(selected_parm,parm_dict)
        elif current_tab == VIRTUALCOLUMNTXTTAB:
            is_valid = self.mcs_dialog.MCSVirtualColumnTXTTab.unpack_selected_parm_virtualcolumntxttab(selected_parm,parm_dict)
        elif current_tab == WORDBOOKINDEXQUERYTAB:
            is_valid = self.mcs_dialog.MCSWordBookIndexQueryTab.unpack_selected_parm_wordbookindexquerytab(selected_parm,parm_dict)
        elif current_tab == RESULTSTAB:
            is_valid = self.mcs_dialog.MCSResultsTab.unpack_selected_parm_resultstab(selected_parm,parm_dict)

        return is_valid
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #~                                END of Generic functions called by each Tab.
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def manage_saved_criteria(self):

        self.mcs_dialog.close()

        try:
            self.managesavedcriteria_dialog.close()
        except:
            pass

        from calibre_plugins.multi_column_search.mcs_manage_criteria_dialog import ManageSavedCriteriaDialog

        self.managesavedcriteria_dialog = ManageSavedCriteriaDialog(self.gui,self.icon,self.font,prefs,
                                                                                                                self.load_criteria_settings_generic,
                                                                                                                self.create_criteria_parm_list_generic,
                                                                                                                self.save_settings_parm_history_to_prefs_generic,
                                                                                                                self.restart_mcs)
        self.managesavedcriteria_dialog.setAttribute(Qt.WA_DeleteOnClose)
        self.managesavedcriteria_dialog.show()
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def restart_mcs(self,proxyprefs=None):
        if proxyprefs is not None:
            for k,v in iteritems(proxyprefs):
                prefs[k] = v
                if DEBUG: print(">>>>>>>> update prefs after 'restart_mcs': ", k, v)
            #END FOR
            prefs
        try:
            self.managesavedcriteria_dialog.close()
            if DEBUG: print("self.mcs_dialog.managesavedcriteria_dialog.close()")
        except:
            pass
        self.search_request_dialog()
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
class SizePersistedDialog(QDialog):

    initial_extra_size = QSize(10, 10)

    def __init__(self, parent, unique_pref_name):
        QDialog.__init__(self, parent)
        self.unique_pref_name = unique_pref_name
        self.geom = gprefs.get(unique_pref_name, None)
        self.finished.connect(self.dialog_closing)

    def resize_dialog(self):

        if self.geom is None:
            self.resize(self.sizeHint()+self.initial_extra_size)
        else:
            self.restoreGeometry(self.geom)

    def dialog_closing(self, result):
        geom = bytearray(self.saveGeometry())
        gprefs[self.unique_pref_name] = geom
#-----------------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------

from calibre.utils.config import JSONConfig
from calibre_plugins.multi_column_search.config import prefs
from calibre_plugins.multi_column_search.config import ConfigWidget

class GUILogDialog(SizePersistedDialog):

    def __init__(self, parent, search_log, log_title_search_path,cust_columns_search_list):
        unique_pref_name = 'MultiColumnSearch:gui_log_dialog'
        SizePersistedDialog.__init__(self, parent, unique_pref_name)
        mytitle = 'MCS Search Log for Library:     ' +  log_title_search_path
        self.setWindowTitle(_(mytitle))

        layout = QVBoxLayout(self)
        self.setLayout(layout)

        do_abort = False
        for row in search_log:
            if "no point in continuing" in row:
                do_abort = True
                if DEBUG: print("GUI Log: no worthwhile data, so no point in continuing")
                break
        #END FOR
        if do_abort:
            search_log = []

        self.n_log_rows = len(search_log)

        #--------------------------------------------------
        column_label_list = []
        column_label_list.append("Author")
        column_label_list.append("Title")
        column_label_list.append("Series")
        column_label_list.append("Tags")
        column_label_list.append("Publisher")
        column_label_list.append("Published")
        column_label_list.append("Path")

        tmp_set = set(cust_columns_search_list)
        cust_columns_search_list = list(tmp_set)
        cust_columns_search_list.sort()
        n_cust_columns = len(cust_columns_search_list)
        cust_column_value_dict = {}
        cust_column_label_dict = {}
        i = 7     # 0-6 are standard columns...
        for row in cust_columns_search_list:  #e.g.  ...#work_author
            row = as_unicode(row).strip()
            if not row.startswith("#"):
                continue
            column_label_list.append(row)
            cust_column_label_dict[i] = row     #so: column i has a label of ...#work_author
            cust_column_value_dict[row] = i    #so:  ...#work_author goes in column i
            i = i + 1

        self.n_total_cols = 7 + n_cust_columns

        self.logtable = QTableWidget(self.n_log_rows,self.n_total_cols)

        self.logtable.setSortingEnabled(False)

        self.logtable.setHorizontalHeaderLabels(column_label_list)

        self.logtable.setColumnWidth(0, 100)
        self.logtable.setColumnWidth(1, 100)
        self.logtable.setColumnWidth(2, 100)
        self.logtable.setColumnWidth(3, 100)
        self.logtable.setColumnWidth(4, 100)
        self.logtable.setColumnWidth(5, 100)
        self.logtable.setColumnWidth(6, 100)

        #~ for k,v in cust_column_label_dict.iteritems():
        for k,v in iteritems(cust_column_label_dict):
            self.logtable.setColumnWidth(k, 100)

        self.logtable.clearContents()
        #--------------------------------------------------

        r = 0
        for row in search_log:
            try:
                #~ if DEBUG: print(as_unicode(row))
                book_dict = row
                #---------------------------
                # Standard Columns 0 - 6
                #---------------------------
                authors = book_dict['AUTHORS']
                title = book_dict['TITLE']
                series = book_dict['SERIES']
                tags = book_dict['TAGS']
                if as_unicode(tags) == as_unicode("NONE"):
                    tags = ""
                publisher = book_dict['PUBLISHER']
                published = book_dict['PUBLISHED']
                path = book_dict['PATH']
                #---------------------------
                # Standard Columns 0 - 6
                #---------------------------
                authors_ = QTableWidgetItem(authors)
                title_ = QTableWidgetItem(title)
                series_ = QTableWidgetItem(series)
                tags_ = QTableWidgetItem(tags)
                publisher_ = QTableWidgetItem(publisher)
                published_ = QTableWidgetItem(published)
                path_ = QTableWidgetItem(path)
                #---------------------------
                # Standard Columns 0 - 6
                #---------------------------
                self.logtable.setItem(r,0,authors_)
                self.logtable.setItem(r,1,title_)
                self.logtable.setItem(r,2,series_)
                self.logtable.setItem(r,3,tags_)
                self.logtable.setItem(r,4,publisher_)
                self.logtable.setItem(r,5,published_)
                self.logtable.setItem(r,6,path_)

                #--------------------------------------
                # Custom Columns 7,8,9,10 (even if no data found for them at all)
                #--------------------------------------
                #can only have a maximum of 4 custom columns in a single search per search; for CL Duplicates, can only have 2.
                cust_field_7 = "cc 7"
                cust_field_8 = "cc 8"
                cust_field_9 = "cc 9"
                cust_field_10 = "cc 10"

                for cc in cust_columns_search_list:
                    cc = as_unicode(cc).strip()
                    if not cc in book_dict:
                        book_dict[cc] = "error...missing value..."    # guarantees no key errors...

                if self.n_total_cols == 7:
                    pass
                elif self.n_total_cols == 8:
                    #~ cust_column_label_dict[i] = row     #so: key = i has a value of #work_author
                    cust_field_7 = book_dict[cust_column_label_dict[7]]
                elif self.n_total_cols == 9:
                    cust_field_7 = book_dict[cust_column_label_dict[7]]
                    cust_field_8 = book_dict[cust_column_label_dict[8]]
                elif self.n_total_cols == 10:
                    cust_field_7 = book_dict[cust_column_label_dict[7]]
                    cust_field_8 = book_dict[cust_column_label_dict[8]]
                    cust_field_9 = book_dict[cust_column_label_dict[9]]
                elif self.n_total_cols == 11:
                    cust_field_7 = book_dict[cust_column_label_dict[7]]
                    cust_field_8 = book_dict[cust_column_label_dict[8]]
                    cust_field_9 = book_dict[cust_column_label_dict[9]]
                    cust_field_10 = book_dict[cust_column_label_dict[10]]

                #--------------------------------------
                # Custom Columns 7,8,9,10 (even if empty)
                #--------------------------------------
                cust_field_7_ = QTableWidgetItem(cust_field_7)
                cust_field_8_ = QTableWidgetItem(cust_field_8)
                cust_field_9_ = QTableWidgetItem(cust_field_9)
                cust_field_10_ =  QTableWidgetItem(cust_field_10)

                #--------------------------------------
                # Custom Columns 7,8,9,10 (even if empty)
                #--------------------------------------
                self.logtable.setItem(r,7,cust_field_7_)
                self.logtable.setItem(r,8,cust_field_8_)
                self.logtable.setItem(r,9,cust_field_9_)
                self.logtable.setItem(r,10,cust_field_10_)
                #--------------------------------------
                #--------------------------------------
                r = r + 1
                #--------------------------------------
            except Exception as e:
                if DEBUG: print("class GUILogDialog(SizePersistedDialog):", as_unicode(e))
                return
        #END FOR

        self.n_total_rows = r

        layout.addWidget(self.logtable)

        self.logtable.setSortingEnabled(True)

        self.resize_all_columns()

        #-----------------------------------------------------

        self.bottom_buttonbox = QDialogButtonBox(QDialogButtonBox.Cancel)
        self.bottom_buttonbox.rejected.connect(self.reject)
        layout.addWidget(self.bottom_buttonbox)

        self.push_button_optimize_column_widths = QPushButton(" ", self)
        self.push_button_optimize_column_widths.setText("Optimize")
        self.push_button_optimize_column_widths.clicked.connect(self.optimize_column_widths)
        self.bottom_buttonbox.addButton(self.push_button_optimize_column_widths,0)

        self.push_button_deoptimize_column_widths = QPushButton(" ", self)
        self.push_button_deoptimize_column_widths.setText("Deoptimize")
        self.push_button_deoptimize_column_widths.clicked.connect(self.deoptimize_column_widths)
        self.bottom_buttonbox.addButton(self.push_button_deoptimize_column_widths,0)

        self.push_button_save_column_widths = QPushButton(" ", self)
        self.push_button_save_column_widths.setText("Save")
        self.push_button_save_column_widths.clicked.connect(self.save_column_widths)
        self.bottom_buttonbox.addButton(self.push_button_save_column_widths,0)

        self.push_button_copy_to_clipboard = QPushButton(" ", self)
        self.push_button_copy_to_clipboard.setText("Copy to Clipboard")
        self.push_button_copy_to_clipboard.clicked.connect(self.copy_log_to_clipboard)
        self.bottom_buttonbox.addButton(self.push_button_copy_to_clipboard,0)

        self.bottom_buttonbox.setCenterButtons(True)

        #-----------------------------------------------------
        self.resize_dialog()

        self.clip = QApplication.clipboard()

    #-----------------------------------------------------
    def resize_all_columns(self):

        n = prefs['COLUMN__0_WIDTH']
        n = int(n)
        if n == 1:       #config.py default value
            self.logtable.resizeColumnsToContents()
        else:               #previously saved by the user
            column__0_width = int(prefs['COLUMN__0_WIDTH'])
            column__1_width = int(prefs['COLUMN__1_WIDTH'])
            column__2_width = int(prefs['COLUMN__2_WIDTH'])
            column__3_width = int(prefs['COLUMN__3_WIDTH'])
            column__4_width = int(prefs['COLUMN__4_WIDTH'])
            column__5_width = int(prefs['COLUMN__5_WIDTH'])
            column__6_width = int(prefs['COLUMN__6_WIDTH'])
            column__7_width = int(prefs['COLUMN__7_WIDTH'])
            column__8_width = int(prefs['COLUMN__8_WIDTH'])
            column__9_width = int(prefs['COLUMN__9_WIDTH'])
            column__10_width = int(prefs['COLUMN__10_WIDTH'])

            if column__0_width < 10: column__0_width = 25
            if column__1_width < 10: column__1_width = 25
            if column__2_width < 10: column__2_width = 25
            if column__3_width < 10: column__3_width = 25
            if column__4_width < 10: column__4_width = 25
            if column__5_width < 10: column__5_width = 25
            if column__6_width < 10: column__6_width = 25
            if column__7_width < 10: column__7_width = 25
            if column__8_width < 10: column__8_width = 25
            if column__9_width < 10: column__9_width = 25
            if column__10_width < 10: column__10_width = 25

            self.logtable.setColumnWidth(0, column__0_width)
            self.logtable.setColumnWidth(1, column__1_width)
            self.logtable.setColumnWidth(2, column__2_width)
            self.logtable.setColumnWidth(3, column__3_width)
            self.logtable.setColumnWidth(4, column__4_width)
            self.logtable.setColumnWidth(5, column__5_width)
            self.logtable.setColumnWidth(6, column__6_width)

            if self.n_total_cols == 7:
                pass
            elif self.n_total_cols == 8:
                self.logtable.setColumnWidth(7, column__7_width)
            elif self.n_total_cols == 9:
                self.logtable.setColumnWidth(7, column__7_width)
                self.logtable.setColumnWidth(8, column__8_width)
            elif self.n_total_cols == 10:
                self.logtable.setColumnWidth(7, column__7_width)
                self.logtable.setColumnWidth(8, column__8_width)
                self.logtable.setColumnWidth(9, column__9_width)
            elif self.n_total_cols == 11:
                self.logtable.setColumnWidth(7, column__7_width)
                self.logtable.setColumnWidth(8, column__8_width)
                self.logtable.setColumnWidth(9, column__9_width)
                self.logtable.setColumnWidth(10, column__10_width)

    #-----------------------------------------------------
    def save_column_widths(self):

        column__0_width = self.logtable.columnWidth(0)
        column__1_width = self.logtable.columnWidth(1)
        column__2_width = self.logtable.columnWidth(2)
        column__3_width = self.logtable.columnWidth(3)
        column__4_width = self.logtable.columnWidth(4)
        column__5_width = self.logtable.columnWidth(5)
        column__6_width = self.logtable.columnWidth(6)
        column__7_width = 100
        column__8_width = 100
        column__9_width = 100
        column__10_width = 100

        try:  #custom column columns will exist only from left to right, never skipping a column
            column__7_width = self.logtable.columnWidth(7)
            column__8_width = self.logtable.columnWidth(8)
            column__9_width = self.logtable.columnWidth(9)
            column__10_width = self.logtable.columnWidth(10)
        except:
            pass

        prefs['COLUMN__0_WIDTH'] = unicode_type(column__0_width)
        prefs['COLUMN__1_WIDTH'] = unicode_type(column__1_width)
        prefs['COLUMN__2_WIDTH'] = unicode_type(column__2_width)
        prefs['COLUMN__3_WIDTH'] = unicode_type(column__3_width)
        prefs['COLUMN__4_WIDTH'] = unicode_type(column__4_width)
        prefs['COLUMN__5_WIDTH'] = unicode_type(column__5_width)
        prefs['COLUMN__6_WIDTH'] = unicode_type(column__6_width)
        prefs['COLUMN__7_WIDTH'] = unicode_type(column__7_width)
        prefs['COLUMN__8_WIDTH'] = unicode_type(column__8_width)
        prefs['COLUMN__9_WIDTH'] = unicode_type(column__9_width)
        prefs['COLUMN__10_WIDTH'] = unicode_type(column__10_width)

        prefs


    #-----------------------------------------------------
    def optimize_column_widths(self):
        self.logtable.resizeColumnsToContents()

    #-----------------------------------------------------
    def deoptimize_column_widths(self):

        for r in range(0,self.n_total_rows):
            self.logtable.setColumnWidth(r, 100)
    #-----------------------------------------------------
    def copy_log_to_clipboard(self):
        #tab delimited, ready to "paste special" into Calc or just paste into text document

        self.logtable.selectAll()
        #~ selected = self.logtable.selectedIndexes()

        s = ''
        s = s + "\t".join([as_unicode(self.logtable.horizontalHeaderItem(i).text()) for i in range(0, self.n_total_cols)])
        s = s +  '\n'
        for r in range(0, self.n_total_rows):
            for c in range(0, self.n_total_cols):
                try:
                    s = s + as_unicode(self.logtable.item(r,c).text()) + "\t"
                except AttributeError:
                    s = s + "\t"
            s = s[:-1] + "\n"        #eliminate last '\t'
        self.clip.setText(s)
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------

#END of ui.py