# -*- coding: utf-8 -*-
from __future__ import absolute_import, division, print_function, unicode_literals
__license__   = 'GPL v3'
__copyright__ = '2016,2017,2018,2019,2020 DaltonST <DaltonShiTzu@outlook.com>'
__my_version__ = "1.0.189"  #Miscellaneous Technical

import os,apsw
from functools import partial

from PyQt5.Qt import (Qt, QDialog, QLabel,  QFont, QWidget, QApplication,
                                       QIcon, QGridLayout, QGroupBox, QMargins, QScrollArea, QComboBox,
                                       QTableWidget, QTableWidgetItem, QDialogButtonBox, QLineEdit,
                                       QSize, QPushButton, QVBoxLayout, QHBoxLayout, QFrame)

from calibre import isbytestring
from calibre.constants import filesystem_encoding, DEBUG
from calibre.gui2 import gprefs, error_dialog

from polyglot.builtins import as_unicode, iteritems, map, range

#-----------------------------------------------------------------------------------------
class SizePersistedDialog(QDialog):
    initial_extra_size = QSize(10, 10)
    def __init__(self, parent, unique_pref_name):
        QDialog.__init__(self, parent, Qt.WindowSystemMenuHint|Qt.WindowMinimizeButtonHint )
        self.unique_pref_name = unique_pref_name
        self.geom = gprefs.get(unique_pref_name, None)
        self.finished.connect(self.dialog_closing)
    def resize_dialog(self):
        if self.geom is None:
            self.resize(self.sizeHint()+self.initial_extra_size)
        else:
            self.restoreGeometry(self.geom)
    def dialog_closing(self, result):
        geom = bytearray(self.saveGeometry())
        gprefs[self.unique_pref_name] = geom
#-----------------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
class IdentifiersEditDialog(SizePersistedDialog):

    def __init__(self,maingui,parent):
        unique_pref_name = 'Job_Spy:identifiers_edit_dialog'
        SizePersistedDialog.__init__(self, parent, unique_pref_name)

        self.maingui = maingui
        self.guidb = self.maingui.library_view.model().db

        self.font = QFont()
        self.font.setBold(False)
        self.font.setPointSize(12)

        mytitle = 'JS+ GUI Tool:  Edit Identifiers'
        self.setWindowTitle(_(mytitle))

        self.setToolTip("<p style='white-space:wrap'>Add, Change, or Delete Identifiers.")

        self.layout_top = QVBoxLayout(self)
        self.layout_top.setAlignment(Qt.AlignCenter)
        self.setLayout(self.layout_top)

        #--------------------------------------------------
        column_label_list = []
        column_label_list.append("ID")
        column_label_list.append("Authors")
        column_label_list.append("Title")
        column_label_list.append("Identifiers")

        self.n_total_cols = 4

        self.n_total_rows = 1

        self.matrix = QTableWidget(self.n_total_rows,self.n_total_cols)
        self.matrix.setFont(self.font)
        self.matrix.setToolTip("<p style='white-space:wrap'>Current Book's Identifying Metadata")
        self.matrix.setSortingEnabled(False)
        self.matrix.setHorizontalHeaderLabels(column_label_list)
        self.matrix.verticalHeader().setVisible(False)

        self.matrix.setColumnWidth(0,   50)
        self.matrix.setColumnWidth(1, 200)
        self.matrix.setColumnWidth(2, 200)
        self.matrix.setColumnWidth(3, 600)

        self.matrix.clearContents()

        self.matrix.setMaximumHeight(100)
        #--------------------------------------------------
        self.last_bookid = 0
        self.layout_top.addWidget(self.matrix)
        self.resize_all_columns()
        #-----------------------------------------------------
        self.frame_1 = QFrame()
        self.frame_1.setFrameShape(QFrame.HLine)
        self.frame_1.setFrameShadow(QFrame.Sunken)
        self.layout_top.addWidget(self.frame_1)
        #-----------------------------------------------------
        self.layout_identifier = QHBoxLayout()
        self.layout_identifier.setAlignment(Qt.AlignCenter)
        self.layout_top.addLayout(self.layout_identifier)
        #-----------------------------------------------------
        self.font.setPointSize(20)
        self.font.setBold(True)

        self.identifier_type_combobox = QComboBox()
        self.identifier_type_combobox.setFont(self.font)
        self.identifier_type_combobox.setEditable(True)
        self.identifier_type_combobox.setFrame(True)
        self.identifier_type_combobox.setDuplicatesEnabled(False)
        self.identifier_type_combobox.setMaxVisibleItems(25)
        self.identifier_type_combobox.setMinimumWidth(350)
        self.identifier_type_combobox.setMaximumWidth(350)
        self.identifier_type_combobox.setInsertPolicy(QComboBox.InsertAtTop)
        self.identifier_type_combobox.setSizeAdjustPolicy(QComboBox.AdjustToContents)
        self.identifier_type_combobox.setToolTip("<p style='white-space:wrap'>Identifier Types used in the current Library.")
        self.layout_identifier.addWidget(self.identifier_type_combobox)

        self.get_all_identifier_types()
        for type in self.identifier_types_list:
            self.identifier_type_combobox.addItem(type)
        #END FOR
        self.identifier_type_combobox.setCurrentIndex(-1)
        self.identifier_type_combobox.setCurrentText("?")

        self.identifier_type_combobox.currentIndexChanged.connect(self.event_type_changed)

        self.colon_label = QLabel("：")   # Fullwidth Colon
        self.colon_label.setFont(self.font)
        self.colon_label.setAlignment(Qt.AlignCenter)
        self.colon_label.setMinimumWidth(30)
        self.colon_label.setMaximumWidth(30)
        self.layout_identifier.addWidget(self.colon_label)

        self.identifier_value_lineedit = QLineEdit(self)
        self.identifier_value_lineedit.setFont(self.font)
        self.identifier_value_lineedit.setToolTip("<p style='white-space:wrap'>Value of Identifier Type.  This Value is <i><u><b>not</b></u></i> validated in any way to ensure it is formatted properly for its specified Type.")
        self.identifier_value_lineedit.setMinimumWidth(500)
        self.identifier_value_lineedit.setMaximumWidth(500)
        self.identifier_value_lineedit.setText("?")
        self.layout_identifier.addWidget(self.identifier_value_lineedit)

        self.font.setBold(False)
        self.font.setPointSize(12)
        #--------------------------------------------------
        #--------------------------------------------------
        #--------------------------------------------------
        self.layout_add_identifier = QHBoxLayout()
        self.layout_add_identifier.setAlignment(Qt.AlignCenter)
        self.layout_top.addLayout(self.layout_add_identifier)
        #--------------------------------------------------
        #--------------------------------------------------
        self.push_button_add_identifier = QPushButton(" ", self)
        self.push_button_add_identifier.setText("Add Identifier")
        self.push_button_add_identifier.setFont(self.font)
        self.push_button_add_identifier.setToolTip("<p style='white-space:wrap'>Add a new Identifier.  If the Identifier Type already exists, the value will not be changed by 'Adding'.")
        self.push_button_add_identifier.setMaximumWidth(450)
        self.push_button_add_identifier.clicked.connect(self.add_identifier_button)
        self.layout_add_identifier.addWidget(self.push_button_add_identifier)
        #--------------------------------------------------
        #--------------------------------------------------
        self.push_button_add_change_identifier = QPushButton(" ", self)
        self.push_button_add_change_identifier.setText("Add or Change Identifier")
        self.push_button_add_change_identifier.setFont(self.font)
        self.push_button_add_change_identifier.setToolTip("<p style='white-space:wrap'>Add a new Identifier if it does not already exist, or Change the existing Identifier to the new Value.")
        self.push_button_add_change_identifier.setMaximumWidth(450)
        self.push_button_add_change_identifier.clicked.connect(self.add_change_identifier_button)
        self.layout_add_identifier.addWidget(self.push_button_add_change_identifier)
        #--------------------------------------------------
        #--------------------------------------------------
        self.push_button_change_identifier = QPushButton(" ", self)
        self.push_button_change_identifier.setText("Change Identifier")
        self.push_button_change_identifier.setFont(self.font)
        self.push_button_change_identifier.setToolTip("<p style='white-space:wrap'>Change an existing Identifier. If the Identifier Type does not already exist, the value will not be added by 'Changing'.")
        self.push_button_change_identifier.setMaximumWidth(450)
        self.push_button_change_identifier.clicked.connect(self.change_identifier_button)
        self.layout_add_identifier.addWidget(self.push_button_change_identifier)
        #--------------------------------------------------
        #--------------------------------------------------
        self.push_button_delete_identifier = QPushButton(" ", self)
        self.push_button_delete_identifier.setText("Delete Identifier")
        self.push_button_delete_identifier.setFont(self.font)
        self.push_button_delete_identifier.setToolTip("<p style='white-space:wrap'>Delete an existing Identifier Type for the current Book.")
        self.push_button_delete_identifier.setMaximumWidth(450)
        self.push_button_delete_identifier.clicked.connect(self.delete_identifier_button)
        self.layout_add_identifier.addWidget(self.push_button_delete_identifier)
        #--------------------------------------------------
        #--------------------------------------------------
        self.frame_2 = QFrame()
        self.frame_2.setFrameShape(QFrame.HLine)
        self.frame_2.setFrameShadow(QFrame.Sunken)
        self.layout_top.addWidget(self.frame_2)
        #--------------------------------------------------
        #--------------------------------------------------
        #-----------------------------------------------------
        self.bottom_buttonbox = QDialogButtonBox()
        self.bottom_buttonbox.rejected.connect(self.reject)
        self.layout_top.addWidget(self.bottom_buttonbox)

        self.push_button_optimize_column_widths = QPushButton(" ", self)
        self.push_button_optimize_column_widths.setText("Optimize")
        self.push_button_optimize_column_widths.setFont(self.font)
        self.push_button_optimize_column_widths.setToolTip("<p style='white-space:wrap'>The metadata columns will be resized based on their longest row contents for each column.")
        self.push_button_optimize_column_widths.clicked.connect(self.optimize_column_widths)
        self.bottom_buttonbox.addButton(self.push_button_optimize_column_widths,0)

        self.push_button_deoptimize_column_widths = QPushButton(" ", self)
        self.push_button_deoptimize_column_widths.setText("Deoptimize")
        self.push_button_deoptimize_column_widths.setFont(self.font)
        self.push_button_deoptimize_column_widths.setToolTip("<p style='white-space:wrap'>The metadata columns will be resized to a fixed width regardless of their contents.")
        self.push_button_deoptimize_column_widths.clicked.connect(self.deoptimize_column_widths)
        self.bottom_buttonbox.addButton(self.push_button_deoptimize_column_widths,0)

        self.push_button_refresh_metadata = QPushButton(" ", self)
        self.push_button_refresh_metadata.setText("Refresh Current Book")
        self.push_button_refresh_metadata.setDefault(True)
        self.push_button_refresh_metadata.setFont(self.font)
        self.push_button_refresh_metadata.setToolTip("<p style='white-space:wrap'>The metadata displayed will be refreshed for the current book.")
        self.push_button_refresh_metadata.clicked.connect(self.refresh_metadata_pushbutton)
        self.bottom_buttonbox.addButton(self.push_button_refresh_metadata,0)

        self.push_button_save_and_exit = QPushButton(" ", self)
        self.push_button_save_and_exit.setText("Exit")
        self.push_button_save_and_exit.setFont(self.font)
        self.push_button_save_and_exit.setToolTip("<p style='white-space:wrap'>Save the 'Edit Identifiers' window size and then exit.")
        self.push_button_save_and_exit.clicked.connect(self.save_and_exit)
        self.bottom_buttonbox.addButton(self.push_button_save_and_exit,0)

        self.bottom_buttonbox.setCenterButtons(True)
        #-----------------------------------------------------
        self.source_list = ["new","addchange","delete","same"]
        self.last_bookid = 0
        self.refresh_metadata("new")
        #-----------------------------------------------------
        self.resize_dialog()

        self.optimize_column_widths()

        self.update()

        self.maingui.library_view.selectionModel().currentChanged.connect(self.refresh_metadata)
    #-----------------------------------------------------
    #-----------------------------------------------------
    def event_type_changed(self,event):
        type = self.identifier_type_combobox.currentText()
        type = type.lower()
        if type in self.identifiers_dict:
            value = self.identifiers_dict[type]
        else:
            value = "?"
        self.identifier_value_lineedit.setText(value)
    #-----------------------------------------------------
    #-----------------------------------------------------
    def resize_all_columns(self):
        self.matrix.resizeColumnsToContents()
        self.save_custom_columns_listing_dialog_geometry()
    #-----------------------------------------------------
    #-----------------------------------------------------
    def optimize_column_widths(self):
        self.matrix.update()
        self.matrix.resizeColumnsToContents()
        self.save_custom_columns_listing_dialog_geometry()
    #-----------------------------------------------------
    #-----------------------------------------------------
    def deoptimize_column_widths(self):
        self.matrix.update()
        self.matrix.setColumnWidth(0, 50)
        self.matrix.setColumnWidth(1, 100)
        self.matrix.setColumnWidth(2, 100)
        self.matrix.setColumnWidth(3, 600)
    #-----------------------------------------------------
    #-----------------------------------------------------
    def save_custom_columns_listing_dialog_geometry(self):
        self.dialog_closing(None)
    #-----------------------------------------------------
    #-----------------------------------------------------
    def save_and_exit(self):
        self.dialog_closing(None)
        try:
            self.maingui.library_view.selectionModel().currentChanged.disconnect(self.refresh_metadata)
        except:
            pass
        self.close()
    #-----------------------------------------------------
    #-----------------------------------------------------
    def refresh_metadata_pushbutton(self):
        self.refresh_cache()
        self.refresh_metadata("same")
        self.identifier_value_lineedit.setText("?")
    #-----------------------------------------------------
    #-----------------------------------------------------
    def convert_id_to_book(self, idval):
        book = {}
        book['calibre_id'] = idval
        return book
    #-----------------------------------------------------
    #-----------------------------------------------------
    def refresh_metadata(self,source="new"):

        source = as_unicode(source)
        if source not in self.source_list:
            source = "new"

        try:
            bookid = self.maingui.library_view.current_book
            if not isinstance(bookid,int):
                bookid = self.maingui.library_view.current_id
            if not isinstance(bookid,int):
                if DEBUG: print("bookid = self.maingui.library_view.current_book: ", as_unicode(bookid))
                cix = self.maingui.library_view.currentIndex()
                if cix.isValid():
                    bookid = self.maingui.library_view.model().id(cix)
                    bookid = int(bookid)
                    if DEBUG: print("cix bookid =", as_unicode(bookid))
                else:
                    books = list(map(partial(self.convert_id_to_book), self.maingui.library_view.get_selected_ids()))
                    if DEBUG: print("get_selected_ids() book(s): ", as_unicode(books))
                    n = len(books)
                    if n == 0:
                        return
                    else:
                        bookid = books[0]
                        bookid = int(bookid)
            else:
                 if DEBUG: print("self.maingui.library_view.current_book: ", as_unicode(bookid))
            if not isinstance(bookid,int):
                if DEBUG: print("target bookid is not an integer: ", as_unicode(bookid))
                return
            if DEBUG: print("current in-process bookid =", as_unicode(bookid))
            if bookid != self.last_bookid:
                source = "new"
                self.identifier_value_lineedit.setText("?")
            elif source == "same":
                pass
            elif source == "addchange":
                pass
            elif source == "delete":
                pass
            elif source == "new":
                self.identifier_value_lineedit.setText("?")
            else:
                if DEBUG: print("current bookid =", as_unicode(bookid), "but:  error: 'source' is invalid: ", source)
                return
            self.last_bookid = bookid
            mi = self.maingui.library_view.model().db.get_metadata(bookid, index_is_id=True, get_user_categories=False)
            authorsx = mi.authors
            if isinstance(authorsx,list):
                authors = authorsx[0]
            elif isinstance(authorsx,tuple):
                authors = list(authorsx)
                authors = authors[0]
            title = mi.title
            bookid = as_unicode(bookid)
            identifiers = mi.identifiers
            self.identifiers_dict = identifiers
            if identifiers is None:
                identifiers = ""
            else:
                if len(identifiers) > 0:
                    identifiers_list = []
                    s = ""
                    for k,v in iteritems(identifiers):
                        row = k,v
                        identifiers_list.append(row)
                    #END FOR
                    identifiers_list.sort()
                    for row in identifiers_list:
                        k,v = row
                        s = s + as_unicode(k) + ":" + as_unicode(v) + ", "
                    #END FOR
                    identifiers = s[0:-2].strip()
                else:
                    identifiers = ""
            del mi
            del authorsx
            identifiers_list = None
            del identifiers_list
            #---------------------------
            #---------------------------
            bookid_ = QTableWidgetItem(bookid)
            authorsort_ = QTableWidgetItem(authors)
            title_ = QTableWidgetItem(title)
            identifiers_ = QTableWidgetItem(identifiers)
            #---------------------------
            #---------------------------
            myflags = Qt.ItemFlags(Qt.ItemIsSelectable | Qt.ItemIsEnabled)
            bookid_.setFlags(myflags)      # not enabled; read-only
            authorsort_.setFlags(myflags)
            title_.setFlags(myflags)
            identifiers_.setFlags(myflags)
            #---------------------------
            #---------------------------
            r = 0
            self.matrix.setItem(r,0,bookid_)
            self.matrix.setItem(r,1,authorsort_)
            self.matrix.setItem(r,2,title_)
            self.matrix.setItem(r,3,identifiers_)

            self.matrix.update()
            #--------------------------------------
            #--------------------------------------
        except Exception as e:
            if DEBUG: print("Exception in: class identifiersDialog, def refresh_metadata :", as_unicode(e))
            return

        self.event_type_changed(None)

        if source == "new":
            self.matrix.setCurrentCell(0,0)
        elif source == "same":
            self.matrix.setCurrentCell(0,3)
        elif source == "addchange":
            self.matrix.setCurrentCell(0,3)
        elif source == "delete":
            self.matrix.setCurrentCell(0,0)
        else:
            if DEBUG: print("source not specified...defaulting currentCell: ", source)
            self.matrix.setCurrentCell(0,0)

        self.optimize_column_widths()
    #-----------------------------------------------------
    #-----------------------------------------------------
    def add_identifier_button(self):
        type = self.identifier_type_combobox.currentText()
        type = type.lower()
        if type in self.identifiers_dict:
            return
        self.add_change_identifier(action="add")
    #-----------------------------------------------------
    #-----------------------------------------------------
    def add_change_identifier_button(self):
        self.add_change_identifier(action="addchange")
    #-----------------------------------------------------
    #-----------------------------------------------------
    def change_identifier_button(self):
        type = self.identifier_type_combobox.currentText()
        type = type.lower()
        if not type in self.identifiers_dict:
            return
        self.add_change_identifier(action="change")
    #-----------------------------------------------------
    #-----------------------------------------------------
    def add_change_identifier(self,action="add"):
        type = self.identifier_type_combobox.currentText()
        type = type.lower()
        if type == "?":
            return
        if type > " ":
            d = [self.identifier_type_combobox.itemText(i) for i in range(self.identifier_type_combobox.count())]
            if not type in d:
                self.identifier_type_combobox.insertItem(0,type)
                self.identifier_type_combobox.setCurrentIndex(0)
                self.identifier_type_combobox.update()
        else:
            return
        #-------------------------------------
        value = self.identifier_value_lineedit.text()
        value = value.strip()
        if not value > " ":
            return
        if value == "?":
            return
        #-------------------------------------
        my_db,my_cursor,is_valid = self.apsw_connect_to_library()
        if not is_valid:
             return error_dialog(self.maingui, _('JS+ GUI Tool'),_('Database Connection Error.  Cannot Connect to the Current Library.'), show=True)
        #-------------------------------------
        my_cursor.execute("begin")
        if action == "add":
            mysql = "INSERT OR IGNORE INTO identifiers (id,book,type,val) VALUES (null,?,?,?)"
            my_cursor.execute(mysql,(self.last_bookid,type,value))
            msg = "Identifier Added"
        elif action == "addchange":
            mysql = "INSERT OR REPLACE INTO identifiers (id,book,type,val) VALUES (null,?,?,?)"
            my_cursor.execute(mysql,(self.last_bookid,type,value))
            msg = "Identifier Added/Changed"
        else:
            mysql = "UPDATE identifiers SET val = ? WHERE book = ? and type = ?"
            my_cursor.execute(mysql,(value,self.last_bookid,type))
            msg = "Identifier Changed"
        my_cursor.execute("commit")
        my_db.close()

        msg = msg + " >> " + type + ":" + value

        if DEBUG: print(msg)

        self.refresh_cache()

        self.refresh_metadata("new")

        self.event_type_changed(None)

        msg = msg + " ⇾ " + type + ":" + value
        self.maingui.status_bar.show_message(_(msg), 10000)
    #-----------------------------------------------------
    #-----------------------------------------------------
    def delete_identifier_button(self):
        type = self.identifier_type_combobox.currentText()
        type = type.lower()
        if type == "?":
            return
        if not type > " ":
            return
        if not type in self.identifiers_dict:
            return
        my_db,my_cursor,is_valid = self.apsw_connect_to_library()
        if not is_valid:
             return error_dialog(self.maingui, _('JS+ GUI Tool'),_('Database Connection Error.  Cannot Connect to the Current Library.'), show=True)
        #-------------------------------------
        mysql = "DELETE FROM identifiers WHERE book = ? AND type = ?"
        my_cursor.execute("begin")
        my_cursor.execute(mysql,(self.last_bookid,type))
        my_cursor.execute("commit")
        my_db.close()
        self.refresh_cache()
        self.refresh_metadata("same")
        self.identifier_type_combobox.setCurrentIndex(-1)
        self.identifier_type_combobox.setCurrentText("?")
        self.identifier_value_lineedit.setText("?")
        msg = "Identifier Type Deleted"
        msg = msg + " ⇾ " + type
        self.maingui.status_bar.show_message(_(msg), 10000)
    #-----------------------------------------------------
    #-----------------------------------------------------
    def refresh_cache(self):
        db = self.maingui.current_db.new_api
        db.reload_from_db(clear_caches=False)
        tmp_list = []
        tmp_list.append(self.last_bookid)
        self.maingui.library_view.model().refresh_ids(tmp_list)
        self.maingui.tags_view.recount()
        QApplication.instance().processEvents()
        identifiers = set(tmp_list)
        self.maingui.library_view.select_rows(identifiers,using_ids=True,change_current=True,scroll=True)  #refreshing the cache also clears the currently selected book...
        if DEBUG: print("refreshed cache for all books")
        del tmp_list
    #-----------------------------------------------------
    #-----------------------------------------------------
    def get_all_identifier_types(self):
        self.identifier_types_list = []
        #-------------------------------------
        my_db,my_cursor,is_valid = self.apsw_connect_to_library()
        if not is_valid:
             return error_dialog(self.maingui, _('JS+ GUI Tool'),_('Database Connection Error.  Cannot Connect to the Current Library.'), show=True)
        #-------------------------------------
        mysql = "SELECT DISTINCT type FROM identifiers"
        my_cursor.execute(mysql)
        tmp_rows = my_cursor.fetchall()
        my_db.close()
        if not tmp_rows:
            tmp_rows = []
        if len(tmp_rows) > 0:
            for row in tmp_rows:
                l = list(row)
                type = l[0]
                self.identifier_types_list.append(type)
                del l
            #END FOR
        del tmp_rows
        self.identifier_types_list.sort()
    #-----------------------------------------------------
    #-----------------------------------------------------
    def apsw_connect_to_library(self):

        path = self.guidb.library_path
        if isbytestring(path):
            path = path.decode(filesystem_encoding)
        path = path.replace(os.sep, '/')
        path = os.path.join(path, 'metadata.db')
        path = path.replace(os.sep, '/')

        if isbytestring(path):
            path = path.decode(filesystem_encoding)

        if path.endswith("/"):
            path = path[0:-1]

        try:
            my_db =apsw.Connection(path)
            is_valid = True
        except Exception as e:
            if DEBUG: print("path to metadata.db is: ", path)
            if DEBUG: print("error: ", as_unicode(e))
            is_valid = False
            return None,None,is_valid

        my_cursor = my_db.cursor()

        mysql = "PRAGMA main.busy_timeout = 15000;"      #PRAGMA busy_timeout = milliseconds;
        my_cursor.execute(mysql)

        return my_db,my_cursor,is_valid
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------------
#END OF identifiers_edit_dialog.py