# -*- coding: utf-8 -*-
from __future__ import absolute_import, division, print_function, unicode_literals
__license__   = 'GPL v3'
__copyright__ = '2020 DaltonST <DaltonShiTzu@outlook.com>'
__my_version__ = "1.0.3"  #Miscellany

from PyQt5.Qt import (Qt, QWidget, QDialog, QLabel, QPushButton, QVBoxLayout,
                                        QScrollArea, QLayout, QSize, QHBoxLayout, QGroupBox,
                                        QCheckBox, QLineEdit, QToolTip, QFont, QTableWidget, QTableWidgetItem)

from calibre.constants import DEBUG
from calibre.gui2 import gprefs, error_dialog
from calibre.utils.config import JSONConfig

from polyglot.builtins import as_unicode, iteritems, range

from calibre_plugins.extract_ris_citations.ris_tag_mapping_dicts import ris_tag_desc_mapping_dict, ris_tag_column_mapping_dict

prefs = JSONConfig('plugins/Extract RIS Citations')

prefs.defaults['ERC_CUSTOMIZED'] =  0
prefs.defaults['ERC_COPY_UNKNOWN_COLUMNS_TO_COMMENTS'] =  '0'   #  'unknown' means 'a specified #column that does not exist in the current-used Library'.
prefs.defaults['ERC_ALSO_CREATE_TAGS_FOR_ITEMIZED_RIS_TAGS'] = "TY,JO"
prefs.defaults['ERC_OPEN_DOI_URL_FOR_CITATION'] = '0'  #User can then easily download the .pdf

prefs.defaults['A1'] = 'authors'
prefs.defaults['A2'] = 'authors'
prefs.defaults['A3'] = 'authors'
prefs.defaults['A4'] = 'authors'
prefs.defaults['AB'] = 'comments'
prefs.defaults['AD'] = 'comments'
prefs.defaults['AN'] = 'comments'
prefs.defaults['AU'] = 'authors'
prefs.defaults['AV'] = 'comments'
prefs.defaults['BT'] = 'comments'
prefs.defaults['C1'] = 'comments'
prefs.defaults['C2'] = 'comments'
prefs.defaults['C3'] = 'comments'
prefs.defaults['C4'] = 'comments'
prefs.defaults['C5'] = 'comments'
prefs.defaults['C6'] = 'comments'
prefs.defaults['C7'] = 'comments'
prefs.defaults['C8'] = 'comments'
prefs.defaults['CA'] = 'comments'
prefs.defaults['CN'] = 'comments'
prefs.defaults['CP'] = 'comments'
prefs.defaults['CT'] = 'comments'
prefs.defaults['CY'] = 'comments'
prefs.defaults['DA'] = 'pubdate'
prefs.defaults['DB'] = 'comments'
prefs.defaults['DO'] = 'identifiers'
prefs.defaults['DP'] = 'comments'
prefs.defaults['ED'] = 'comments'
prefs.defaults['EP'] = 'comments'
prefs.defaults['ER'] = 'reserved'
prefs.defaults['ET'] = 'comments'
prefs.defaults['ID'] = 'comments'
prefs.defaults['IS'] = 'comments'
prefs.defaults['J1'] = 'comments'
prefs.defaults['J2'] = 'comments'
prefs.defaults['JA'] = 'comments'
prefs.defaults['JF'] = 'comments'
prefs.defaults['JO'] = 'comments'
prefs.defaults['KW'] = 'tags'
prefs.defaults['L1'] = 'comments'
prefs.defaults['L2'] = 'comments'
prefs.defaults['L3'] = 'comments'
prefs.defaults['L4'] = 'comments'
prefs.defaults['LA'] = 'languages'
prefs.defaults['LB'] = 'comments'
prefs.defaults['LK'] = 'comments'
prefs.defaults['M1'] = 'comments'
prefs.defaults['M2'] = 'comments'
prefs.defaults['M3'] = 'comments'
prefs.defaults['N1'] = 'comments'
prefs.defaults['N2'] = 'comments'
prefs.defaults['NV'] = 'comments'
prefs.defaults['OP'] = 'comments'
prefs.defaults['PB'] = 'publisher'
prefs.defaults['PP'] = 'comments'
prefs.defaults['PY'] = 'comments'
prefs.defaults['RI'] = 'comments'
prefs.defaults['RN'] = 'comments'
prefs.defaults['RP'] = 'comments'
prefs.defaults['SE'] = 'comments'
prefs.defaults['SN'] = 'identifiers'
prefs.defaults['SP'] = 'comments'
prefs.defaults['ST'] = 'comments'
prefs.defaults['T1'] = 'title'
prefs.defaults['T2'] = 'comments'
prefs.defaults['T3'] = 'comments'
prefs.defaults['TA'] = 'comments'
prefs.defaults['TI'] = 'comments'
prefs.defaults['TT'] = 'comments'
prefs.defaults['TY'] = 'comments'
prefs.defaults['U1'] = 'comments'
prefs.defaults['U2'] = 'comments'
prefs.defaults['U3'] = 'comments'
prefs.defaults['U4'] = 'comments'
prefs.defaults['U5'] = 'comments'
prefs.defaults['UK'] = 'comments'
prefs.defaults['UR'] = 'comments'
prefs.defaults['VL'] = 'comments'
prefs.defaults['VO'] = 'comments'
prefs.defaults['Y1'] = 'comments'
prefs.defaults['Y2'] = 'comments'

#~ PREFS_NAMESPACE = 'ExtractRISCitations'
#~ PREFS_KEY_SETTINGS = 'settings'
#-------------------------------------------------------

#-----------------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
class SizePersistedDialog(QDialog):
    initial_extra_size = QSize(50, 275)  # w,h

    def __init__(self, parent, unique_pref_name):
        QDialog.__init__(self, parent)
        self.unique_pref_name = unique_pref_name
        self.geom = gprefs.get(unique_pref_name, None)

    def resize_dialog(self):
        if self.geom is None:
            self.resize(self.sizeHint()+self.initial_extra_size)
        else:
            self.restoreGeometry(self.geom)

    def save_dialog_geometry(self):
        geom = bytearray(self.saveGeometry())
        gprefs[self.unique_pref_name] = geom
#-----------------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
#-----------------------------------------------------------------------------------------
class ConfigWidget(SizePersistedDialog):
    #-----------------------------------------------------------------------------------------
    def __init__(self):
        parent = None
        unique_pref_name = 'extract_ris_citations:customize_erc_dialog'
        SizePersistedDialog.__init__(self, parent, unique_pref_name)
        #-----------------------------------------------------
        from calibre.gui2.ui import get_gui
        self.maingui = get_gui()
        self.gui = self.maingui
        self.guidb = self.maingui.current_db
        #-----------------------------------------------------
        self.custom_columns_dict = self.guidb.field_metadata.custom_field_metadata()
        #-----------------------------------------------------
        for k,v in iteritems(prefs.defaults):
            if not k in prefs:
                prefs[k] = v
        prefs
        #-----------------------------------------------------
        self.ris_tag_desc_mapping_dict = ris_tag_desc_mapping_dict  #all legal tags possible
        self.mytagprefs = {}
        for tag,name in iteritems(self.ris_tag_desc_mapping_dict):
            self.mytagprefs[tag] = "#" + name   #temporary placeholder
        #-----------------------------------------------------
        self.ris_tag_column_mapping_dict = ris_tag_column_mapping_dict  #all default tag:column mappings
        self.supported_standard_columns_set = set()
        for tag,col in iteritems(self.ris_tag_column_mapping_dict):
            self.mytagprefs[tag] = col
            if col > "":  #ER
                self.supported_standard_columns_set.add(col)
        #-----------------------------------------------------
        for k,v in iteritems(prefs): #all prefs, including any customized tag:column mappings
            if k in ris_tag_column_mapping_dict:  #all default tag:column mappings for standard columns only
                col = ris_tag_column_mapping_dict[k]
                self.mytagprefs[k] = col  #to maintain integrity of the RIS mappings for a few standard columns
            elif k in ris_tag_desc_mapping_dict:  #valid tag, but not with a standard column as a default column mapping
                self.mytagprefs[k] = v
        #-----------------------------------------------------

        #-----------------------------------------------------
        font = QFont()
        font.setBold(False)
        font.setPointSize(10)
        #-----------------------------------------------------
        self.setWindowTitle("Customize 'Extract RIS Citations'")
        #-----------------------------------------------------
        t = "<p style='white-space:wrap'>"
        t = t + "This matrix is used to customize how the RIS files that you add to Calibre will have their bibliographic citation details (a.k.a. metadata) updated in Calibre #custom and standard columns."
        t = t + "<br>&bull;'Tag' refers the a standard RIS 'Tag' that is contained within a textual RIS file to uniquely identify the type of citation detail text that follows."
        t = t + "<br>&bull;'Meaning' refers to the generic name, or description, of the RIS 'Tag'."
        t = t + "<br>&bull;'Calibre Column' refers to the #custom or Standard Calibre column that will be updated by ERC for the citation metadata. "
        t = t + "<br>"
        t = t + "<br><u>Tag&nbsp;-&nbsp;Example Tag Value&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;</u><br>\
        TY  - JOUR<br>\
        T1  - Ethics and Resource Scarcity: ASCO Recommendations for the Oncology Community During the COVID-19 Pandemic<br>\
        AU  - Marron, Jonathan M.<br>\
        AU  - Joffe, Steven<br>\
        AU  - Jagsi, Reshma<br>\
        AU  - Spence, Rebecca A.<br>\
        AU  - Hlubocky, Fay J.<br>\
        DA  - 2020/04/27<br>\
        PY  - 2020<br>\
        N1  - doi: 10.1200/JCO.20.00960<br>\
        DO  - 10.1200/JCO.20.00960<br>\
        T2  - Journal of Clinical Oncology<br>\
        JF  - Journal of Clinical Oncology<br>\
        JO  - JCO<br>\
        SP  - JCO.20.00960<br>\
        PB  - American Society of Clinical Oncology<br>\
        SN  - 0732-183X<br>\
        M3  - doi: 10.1200/JCO.20.00960<br>\
        UR  - https://doi.org/10.1200/JCO.20.00960<br>\
        Y2  - 2020/04/29<br>\
        LA  - eng<br>\
        ER  -"
        t = t + "<br><br>&bull;'Calibre Column' refers to the #custom and standard columns to which each RIS Tag should be mapped.\
        <br>&bull;All of the basic standard columns are valid: 'authors'; 'title'; 'series'; 'publisher'; 'published date'; 'tags'; 'identifiers'; 'comments'; and 'languages'.\
        <br>&bull;All custom column types are valid to use except for 'composite' (built from others) and 'ratings'.  RIS data is all textual, so ERC converts data as needed."
        t = t + "<br>&bull;Description of a bibliographic 'Citation':  <i>https://en.m.wikipedia.org/wiki/Citation</i>"
        t = t + "<br>&bull;Description of 'RIS':  <i>https://en.m.wikipedia.org/wiki/RIS_(file_format)</i>"
        #-----------------------------------------------------
        self.setToolTip(t)
        #-----------------------------------------------------
        font = QFont()
        font.setBold(False)
        font.setPointSize(10)
        #-----------------------------------------------------
        self.layout_top = QVBoxLayout()
        self.layout_top.setSpacing(0)
        self.layout_top.setAlignment(Qt.AlignCenter)
        self.setLayout(self.layout_top)
        #-----------------------------------------------------
        self.scroll_area_frame = QScrollArea()
        self.scroll_area_frame.setAlignment(Qt.AlignCenter)
        self.scroll_area_frame.setWidgetResizable(True)
        self.scroll_area_frame.ensureVisible(300,300)

        self.layout_top.addWidget(self.scroll_area_frame)       # the scroll area is now the child of the parent of self.layout_top
        #-----------------------------------------------------
        self.scroll_widget = QWidget()
        self.layout_top.addWidget(self.scroll_widget)           # causes automatic reparenting of QWidget to the parent of self.layout_top, which is:  self .
        #-----------------------------------------------------
        self.layout_frame = QVBoxLayout()
        self.layout_frame.setAlignment(Qt.AlignCenter)

        self.scroll_widget.setLayout(self.layout_frame)        # causes automatic reparenting of any widget later added to self.layout_frame to the parent of self.layout_frame, which is:  QWidget .
        #-----------------------------------------------------
        self.erc_groupbox = QGroupBox('Extract RIS Citations')
        self.erc_groupbox.setToolTip(t)
        self.layout_frame.addWidget(self.erc_groupbox)

        self.erc_layout = QVBoxLayout()
        self.erc_layout.setAlignment(Qt.AlignCenter)
        self.erc_groupbox.setLayout(self.erc_layout)
        #--------------------------------------------------
        #-----------------------------------------------------
        ris_assignments_list,ris_assignments_dict = self.get_ris_assignments()

        self.n_ris_rows = len(ris_assignments_list)
        #-----------------------------------------------------
        column_label_list = []
        column_label_list.append("RIS Tag")
        column_label_list.append("Meaning")  # ~name
        column_label_list.append("Calibre Column")

        self.n_total_cols = 3
        #-----------------------------------------------------

        self.ristable = QTableWidget(self.n_ris_rows,self.n_total_cols)

        self.ristable.setSortingEnabled(False)

        self.ristable.setHorizontalHeaderLabels(column_label_list)

        self.ristable.setColumnWidth(0, 100)
        self.ristable.setColumnWidth(1, 300)
        self.ristable.setColumnWidth(1, 300)

        self.ristable.clearContents()
        #--------------------------------------------------

        myflags = Qt.ItemFlags()
        myflags != Qt.ItemIsEnabled  # not enabled; read-only

        r = int(0)
        for row in ris_assignments_list:  # the list here must have 100% of legal tags...
            try:
                #---------------------------
                #---------------------------
                tag,column = row
                item = ris_assignments_dict[tag]
                ris_tag,name,calibre_target = item
                ris_tag = ris_tag.upper()
                #---------------------------
                #---------------------------
                ris_tag_ = QTableWidgetItem(ris_tag)
                ris_tag_.setFlags(myflags)                      # not enabled; read-only

                name_ = QTableWidgetItem(name)
                name_.setFlags(myflags)                       # not enabled; read-only

                if ris_tag in self.tags_to_disallow_edits_list:  # e.g. 'ER'
                    calibre_target = "reserved"

                if calibre_target == "language":
                    calibre_target == "languages"
                elif calibre_target == "published":
                    calibre_target == "pubdate"
                elif calibre_target == "author":
                    calibre_target == "authors"
                elif calibre_target == "tag":
                    calibre_target == "tags"
                elif calibre_target == "identifier":
                    calibre_target == "identifiers"

                calibre_target_ = QTableWidgetItem(calibre_target)

                if ris_tag in self.tags_to_disallow_edits_list:
                    calibre_target_.setFlags(myflags)    # not enabled; read-only
                #---------------------------
                #---------------------------
                self.ristable.setItem(r,0,ris_tag_)
                self.ristable.setItem(r,1,name_)
                self.ristable.setItem(r,2,calibre_target_)
                #--------------------------------------
                r = r + 1
                #--------------------------------------
            except Exception as e:
                if DEBUG: print("RIS Tag Matrix Creation:  Exception>>>>", as_unicode(e))
        #END FOR

        self.n_ristable_rows = r

        self.ristable.resizeColumnsToContents()
        self.erc_layout.addWidget(self.ristable)
        #-----------------------------------------------------------------------------------------
        #-----------------------------------------------------------------------------------------
        self.erc_copy_unknown_columns_to_comments_checkbox = QCheckBox("Copy unknown #columns to 'comments'?")
        self.erc_copy_unknown_columns_to_comments_checkbox.setToolTip("<p style='white-space:wrap'>Do you want to have RIS Tag values mapped to a #column that does not exist in the current Library be added to the standard Calibre 'comments' column instead?")
        self.erc_layout.addWidget(self.erc_copy_unknown_columns_to_comments_checkbox)
        a = prefs['ERC_COPY_UNKNOWN_COLUMNS_TO_COMMENTS']
        a = int(a)
        self.erc_copy_unknown_columns_to_comments_checkbox.setChecked(a)
        #-----------------------------------------------------------------------------------------
        #-----------------------------------------------------------------------------------------
        self.layout_erc_also_create_tags_for_ris_tags = QHBoxLayout()
        self.layout_erc_also_create_tags_for_ris_tags.setAlignment(Qt.AlignLeft)
        self.erc_layout.addLayout(self.layout_erc_also_create_tags_for_ris_tags)

        self.erc_also_create_tags_for_ris_tags_qlabel = QLabel("Additionally Create Tags for:")
        self.erc_also_create_tags_for_ris_tags_qlabel.setToolTip("<p style='white-space:wrap'>Do you want to have certain RIS Tag values to <b>also</b> be added as Calibre Tags?")
        self.erc_also_create_tags_for_ris_tags_qlabel.setMaximumWidth(300)
        self.layout_erc_also_create_tags_for_ris_tags.addWidget(self.erc_also_create_tags_for_ris_tags_qlabel)

        self.erc_also_create_tags_for_ris_tags_lineedit = QLineEdit()
        self.erc_also_create_tags_for_ris_tags_lineedit.setText(prefs['ERC_ALSO_CREATE_TAGS_FOR_ITEMIZED_RIS_TAGS'])
        self.erc_also_create_tags_for_ris_tags_lineedit.setCursorPosition(0)
        self.erc_also_create_tags_for_ris_tags_lineedit.setToolTip("<p style='white-space:wrap'>Do you want to have certain RIS Tag values to <b>also</b> be added as Calibre Tags?")
        self.erc_also_create_tags_for_ris_tags_lineedit.setMaximumWidth(300)
        self.layout_erc_also_create_tags_for_ris_tags.addWidget(self.erc_also_create_tags_for_ris_tags_lineedit)
        #-----------------------------------------------------------------------------------------
        #-----------------------------------------------------------------------------------------
        self.erc_open_doi_url_for_citation_checkbox = QCheckBox("Open DOI's URL?")
        self.erc_open_doi_url_for_citation_checkbox.setToolTip("<p style='white-space:wrap'>Do you want to have each newly added .ris 'book' to have opened its DOI's URL in a <u>new tab in your default web browser</u> so you can download its .pdf or other document file to add as an additional 'format' file to the .ris?")
        self.erc_layout.addWidget(self.erc_open_doi_url_for_citation_checkbox)
        a = prefs['ERC_OPEN_DOI_URL_FOR_CITATION']
        a = int(a)
        self.erc_open_doi_url_for_citation_checkbox.setChecked(a)
        #-----------------------------------------------------------------------------------------
        #-----------------------------------------------------------------------------------------
        self.push_button_save_options = QPushButton("Save Settings")
        self.push_button_save_options.clicked.connect(self.save_preferences)
        self.push_button_save_options.setDefault(True)
        self.push_button_save_options.setFont(font)
        self.push_button_save_options.setToolTip("<p style='white-space:wrap'>Save customizations and window size.")
        self.erc_layout.addWidget(self.push_button_save_options)
        #-----------------------------------------------------
        self.scroll_widget.resize(self.sizeHint())
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.scroll_area_frame.setWidget(self.scroll_widget)    # now that all widgets have been created and assigned to a layout...
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.scroll_area_frame.resize(self.sizeHint())
        #-----------------------------------------------------
        #-----------------------------------------------------
        self.resize(self.sizeHint())
        self.resize_dialog()
    #-----------------------------------------------------------------------------------------
    def get_ris_assignments(self):
        ris_assignments_list = self.build_ris_assignments_list()
        ris_assignments_dict = self.build_ris_assignments_dict(ris_assignments_list)
        return ris_assignments_list,ris_assignments_dict
    #-----------------------------------------------------------------------------------------
    def build_ris_assignments_list(self):
        ris_assignments_list = []
        self.tags_to_disallow_edits_list = []
        self.tags_count_dict = {}

        for tag,val in iteritems(self.mytagprefs):
            if tag.isupper() and len(tag) == 2:
                if tag in ris_tag_desc_mapping_dict:
                    name = ris_tag_desc_mapping_dict[tag]
                    r = tag,name
                    ris_assignments_list.append(r)
        #END FOR

        ris_assignments_list.sort()

        self.tags_to_disallow_edits_list.append("ER")

        return ris_assignments_list
    #-----------------------------------------------------------------------------------------
    def build_ris_assignments_dict(self,ris_assignments_list):

        ris_assignments_dict = {}

        for row in ris_assignments_list:
            tag,name = row
            calibre_target = self.get_calibre_target(tag)
            ris_assignments_dict[tag] = tag,name,calibre_target
        #END FOR

        del ris_assignments_list

        return ris_assignments_dict
    #-----------------------------------------------------------------------------------------
    def get_calibre_target(self,tag):
        if tag in self.mytagprefs:
            calibre_target = self.mytagprefs[tag]
        elif tag in self.ris_tag_column_mapping_dict:
            calibre_target = self.ris_tag_column_mapping_dict[tag]
        else:
            calibre_target = "comments"
        return calibre_target
    #-----------------------------------------------------------------------------------------
    def save_preferences(self):
        for r in range(0,self.n_ristable_rows):
            item = self.ristable.item(r,0)
            tag = item.text()
            item = self.ristable.item(r,2)
            calibre_target  = item.text()
            if tag is None:
                continue
            if calibre_target is None:
                calibre_target = ""
            if calibre_target == "language":
                calibre_target == "languages"
            elif calibre_target == "published":
                calibre_target == "pubdate"
            elif calibre_target == "author":
                calibre_target == "authors"
            elif calibre_target == "tag":
                calibre_target == "tags"
            elif calibre_target == "identifier":
                calibre_target == "identifiers"
            if (calibre_target != "") and (not calibre_target.startswith('#')):
                if not calibre_target in self.supported_standard_columns_set:
                    if not calibre_target == "reserved":  # ER - end of reference - reserved
                        msg = "An unsupported Standard Column has been specified: " + calibre_target
                        msg = msg + "<br><br>Supported Standard Columns:<br>"
                        for col in self.supported_standard_columns_set:
                            msg = msg + col + ";"
                        msg = msg[0:-1]
                        msg = msg + "."
                        return error_dialog(self.maingui, _('Extract RIS Citations: Customizing'),_(msg), show=True)
            self.mytagprefs[tag] = calibre_target
        #END FOR

        for k,v in iteritems(self.mytagprefs):
            if k is None or v is None:
                continue
            prefs[k] = v
        #END FOR

        prefs['ERC_CUSTOMIZED'] = '1'

        if self.erc_copy_unknown_columns_to_comments_checkbox.isChecked():
            prefs['ERC_COPY_UNKNOWN_COLUMNS_TO_COMMENTS'] = '1'
        else:
            prefs['ERC_COPY_UNKNOWN_COLUMNS_TO_COMMENTS'] = '0'

        prefs['ERC_ALSO_CREATE_TAGS_FOR_ITEMIZED_RIS_TAGS'] = self.erc_also_create_tags_for_ris_tags_lineedit.text()

        if self.erc_open_doi_url_for_citation_checkbox.isChecked():
            prefs['ERC_OPEN_DOI_URL_FOR_CITATION'] = '1'
        else:
            prefs['ERC_OPEN_DOI_URL_FOR_CITATION'] = '0'

        prefs

        self.save_dialog_geometry()
    #-----------------------------------------------------------------------------------------
    def validate(self):
        return True
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------