# -*- coding: utf-8 -*-
from __future__ import absolute_import, division, print_function, unicode_literals
__license__   = 'GPL v3'
__copyright__ = '2014,2015,2016,2017,2018,2019,2020,2021 DaltonST <DaltonShiTzu@outlook.com>'
__my_version__ = "2.2.1"   # Translations

from PyQt5.Qt import QMenu, QDialog, QIcon, QAction

import os, sys
from functools import partial
import subprocess
import zipfile

from calibre.constants import DEBUG
from calibre.gui2 import info_dialog, question_dialog, error_dialog, Dispatcher
from calibre.gui2.actions import InterfaceAction

from polyglot.builtins import as_unicode
from polyglot.queue import Queue

try:
    load_translations()
except Exception as e:
    if DEBUG: print("author_book_count:  ui.py - load_translations() - Exception: ", as_unicode(e))

from calibre_plugins.author_book_count.common_utils import set_plugin_icon_resources, get_icon, create_menu_action_unique
from calibre_plugins.author_book_count.jobs import start_abc_threaded
from calibre_plugins.author_book_count.main import get_all_author_book_counts

PLUGIN_ICONS = ['images/abcicon.png', 'images/abcsmallicon.png', 'images/readinstructionsicon.png']

class AuthorBookCountUpdate(InterfaceAction):

    name = 'author_book_count'
    action_spec = ('ABC', 'images/abcicon.png', _('Count the Books of All Authors'), None)
    action_type = 'global'
    accepts_drops = False
    auto_repeat = False
    priority = 9
    popup_type = 1

    #-----------------------------------------------------------------------------------------
    def genesis(self):
        self.maingui = self.gui
        self.menu = QMenu(self.gui)
        icon_resources = self.load_resources(PLUGIN_ICONS)
        set_plugin_icon_resources(self.name, icon_resources )
        self.rebuild_menus()
        self.qaction.setMenu(self.menu)
        self.qaction.setIcon(get_icon(PLUGIN_ICONS[0]))
        self.qaction.triggered.connect(self._question_user_to_continue)
 #-----------------------------------------------------------------------------------------
    def initialization_complete(self):
        self.maingui = self.gui
        self.guidb = self.maingui.library_view.model().db
 #-----------------------------------------------------------------------------------------
    def library_changed(self, guidb):
        self.guidb = self.maingui.library_view.model().db
        if DEBUG: print(_("ABC: library has changed to: "), self.guidb.library_path)
    #-----------------------------------------------------------------------------------------
    def _question_user_to_continue(self):
        self.guidb = self.gui.library_view.model().db
        if question_dialog(self.gui, _("Author Book Count"), _("Execute Author Book Count: Continue?")):
            start_abc_threaded(self, self.gui, self.guidb, Dispatcher(self.abc_job_complete) )
            self.gui.status_bar.show_message(_('ABC Job Has Been Submitted'), 5000)
    #-----------------------------------------------------------------------------------------
    def abc_job_complete(self, job):
        if job.failed:
            return error_dialog(self.gui, _('ABC: Job Has Failed'), _('You Must Restart Calibre Before Running ABC again in this Library.') , ' ' ).show()
        self.force_refresh_of_cache()
        self.gui.status_bar.show_message(_('ABC Job Has Completed'), 10000)
    #-----------------------------------------------------------------------------------------
    def rebuild_menus(self):
        m = self.menu
        m.clear()
        m.addSeparator()
        unique_name = _("Author Book Count: Run ABC Job")
        create_menu_action_unique(self, m, _('&Author Book Count'), 'images/abcsmallicon.png',
                              triggered=partial(self._question_user_to_continue),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        m.addSeparator()
        unique_name = _("Author Book Count: Read Instructions")
        create_menu_action_unique(self, m, _('&Read Instructions'), 'images/readinstructionsicon.png',
                              triggered=partial(self.view_user_instructions),unique_name=unique_name, favourites_menu_unique_name=unique_name)
        m.addSeparator()
        self.gui.keyboard.finalize()
    #-----------------------------------------------------------------------------------------
    def view_user_instructions(self):
        self.extract_documentation_from_zip() #every time to ensure latest copy is available
        try:
            p_pid = subprocess.Popen(self.documentation_path, shell=True)
        except:
            return error_dialog(self.gui, _('Documentation .pdf Not Found. Try reinstalling this plugin, and restarting Calibre.'),
                                       _('It is supposed to be: ' + self.documentation_path ), show=True)
    #-----------------------------------------------------------------------------------------
    def extract_documentation_from_zip(self):
        #extract the .pdf file from the zip so it can be executed directly.

        self.documentation_path = None

        zipfile_path = as_unicode(self.plugin_path)

        destination_path = as_unicode(self.plugin_path)
        destination_path = destination_path.replace("\Author Book Count.zip", "")
        destination_path = destination_path.replace("/Author Book Count.zip", "")

        zfile = zipfile.ZipFile(zipfile_path)

        dir_name = "abc_documentation"

        file_name = 'abc_instructions.pdf'

        file_name = os.path.join(dir_name, file_name)

        for name in zfile.namelist(): #all files in zip with full internal paths
            n = name.find(dir_name)
            if n >= 0:
                zfile.extract(name, destination_path)
                self.documentation_path = os.path.join(destination_path, file_name)     #"C:\Users\DaltonST\AppData\Roaming\calibre\plugins\abc_documentation\abc_instructions.pdf"
        #END FOR
        self.documentation_path = as_unicode(self.documentation_path)
        if DEBUG: print("documentation path: ", self.documentation_path)
  #-------------------------------------------------------------------------------------------------------------------------------------
    def force_refresh_of_cache(self):
        db = self.maingui.current_db.new_api
        db.reload_from_db(clear_caches=False)
        frozen = db.all_book_ids()
        books = list(frozen)
        self.maingui.library_view.model().refresh_ids(books)
        self.maingui.tags_view.recount()
    #-----------------------------------------------------------------------------------------
#END of ui.py