# -*- coding: utf-8 -*-
from __future__ import absolute_import, division, print_function, unicode_literals
__license__   = 'GPL v3'
__copyright__ = '2017,2018,2019,2020 DaltonST <DaltonShiTzu@outlook.com>'
__my_version__ = "1.0.16"  # Miscellany

from PyQt5.Qt import QMenu, QDialog, QIcon, QAction

import os, sys
import apsw
from functools import partial

from calibre import isbytestring
from calibre.constants import filesystem_encoding, DEBUG
from calibre.gui2 import error_dialog, question_dialog, info_dialog
from calibre.gui2.actions import InterfaceAction
from calibre.utils.config import JSONConfig

from polyglot.builtins import as_unicode, iteritems, map, unicode_type

from calibre_plugins.audit_log.audit_log_dialog import AuditLogDialog
from calibre_plugins.audit_log.audit_undo_dialog import AuditUndoDialog

from calibre_plugins.audit_log.config import prefs  # PREFS_NAMESPACE,PREFS_KEY_SETTINGS
from calibre_plugins.audit_log.config import ConfigWidget
from calibre_plugins.audit_log.config import __my_db_version__
from calibre_plugins.audit_log.common_utils import set_plugin_icon_resources, get_icon, get_pixmap, get_local_images_dir, create_menu_action_unique

PLUGIN_ICONS = ['images/audit.png','images/undo.png','images/empty.png','images/wrench-hammer.png']

#-------------------------------------------------------------------------------------------
class ActionAuditLog(InterfaceAction):

    name = 'Audit Log'
    action_spec = ('AL','images/audit.png', "Undo/Reverse most metadata deletes and updates weeks later for selected books and log entries.  Creates a log entry for every addition, change or delete for most Calibre standard metadata columns, and for all custom metadata columns.", None)
    action_type = 'global'
    accepts_drops = False
    auto_repeat = False
    priority = 9
    popup_type = 1

    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def genesis(self):
        icon_resources = self.load_resources(PLUGIN_ICONS)
        set_plugin_icon_resources(self.name, icon_resources )
        self.menu = QMenu(self.gui)
        self.build_menus(self.gui)
        self.qaction.setMenu(self.menu)
        self.qaction.setIcon(get_icon(PLUGIN_ICONS[0]))
        self.qaction.triggered.connect(self.audit_log_control_all)
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def initialization_complete(self):

        self.guidb = self.gui.library_view.model().db

        self.check_al_status_for_current_library()
        self.check_al_status_for_current_user()

        #----------------------------------------
        for k,v in iteritems(prefs.defaults):
            if k in prefs:
                continue
            else:
                prefs[k] = v
                prefs
        #END FOR

        self.selected_books_list = []
        self.selected_books_set = set(self.selected_books_list)

        #----------------------------------------
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def library_changed(self,guidb):
        self.guidb = self.gui.library_view.model().db

        try:
            self.auditlogdialog.close()
        except:
            pass

        try:
            self.auditundodialog.close()
        except:
            pass

        self.check_al_status_for_current_library()

    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def shutting_down(self):
        try:
            self.auditlogdialog.close()
        except:
            pass

        try:
            self.auditundodialog.close()
        except:
            pass

        return True
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def apply_settings(self):
        prefs
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def show_configuration(self):
        self.interface_action_base_plugin.do_user_config(self.gui)
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def build_menus(self,gui):

        self.gui = gui

        self.audit_log_menu = self.menu
        self.audit_log_menu.clear()

        self.audit_log_menu.setTearOffEnabled(True)
        self.audit_log_menu.setWindowTitle('Audit Log Menu')

        self.audit_log_menu.addSeparator()

        create_menu_action_unique(self, self.audit_log_menu, ' ', ' ',
                              triggered=None)

        self.audit_log_menu.addSeparator()

        unique_name = "View/Export Audit Log to Clipboard [All Books] [Current Library Only]"
        create_menu_action_unique(self, self.audit_log_menu, 'View/Export Audit Log to Clipboard [All Books] [Current Library Only]', 'images/audit.png',
                              triggered=partial(self.audit_log_control_all),unique_name=unique_name, favourites_menu_unique_name=unique_name)

        self.audit_log_menu.addSeparator()

        unique_name = "View/Export Audit Log to Clipboard [Selected Books Only][Current Library Only]"
        create_menu_action_unique(self, self.audit_log_menu, 'View/Export Audit Log to Clipboard [Selected Books Only][Current Library Only]', 'images/audit.png',
                              triggered=partial(self.audit_log_control_selected),unique_name=unique_name, favourites_menu_unique_name=unique_name)

        self.audit_log_menu.addSeparator()

        create_menu_action_unique(self, self.audit_log_menu, ' ', ' ',
                              triggered=None)

        self.audit_log_menu.addSeparator()

        unique_name = "Undo Logged Changes [Selected Books]"
        create_menu_action_unique(self, self.audit_log_menu, 'Undo Logged Changes [Currently Selected Books]', 'images/undo.png',
                              triggered=partial(self.undo_logged_changes),unique_name=unique_name, favourites_menu_unique_name=unique_name)

        self.audit_log_menu.addSeparator()

        create_menu_action_unique(self, self.audit_log_menu, ' ', ' ',
                              triggered=None)

        self.audit_log_menu.addSeparator()

        unique_name = "Clear the Audit Log [Current Library Only]"
        create_menu_action_unique(self, self.audit_log_menu, 'Clear the Audit Log [Current Library Only]', 'images/empty.png',
                              triggered=partial(self.purge_audit_log),unique_name=unique_name, favourites_menu_unique_name=unique_name)

        self.audit_log_menu.addSeparator()

        create_menu_action_unique(self, self.audit_log_menu, ' ', ' ',
                              triggered=None)

        self.audit_log_menu.addSeparator()

        unique_name = "Activate/Deactivate the Audit Log [Current Library Only]"
        create_menu_action_unique(self, self.audit_log_menu, 'Activate/Deactivate the Audit Log [Current Library Only]', 'images/wrench-hammer.png',
                              triggered=partial(self.show_configuration),unique_name=unique_name, favourites_menu_unique_name=unique_name)

        self.audit_log_menu.addSeparator()

        create_menu_action_unique(self, self.audit_log_menu, ' ', ' ',
                              triggered=None)

        self.audit_log_menu.addSeparator()

        unique_name = "Show Audit Log Status for Calibre Libraries"
        create_menu_action_unique(self, self.audit_log_menu, 'Show Audit Log Status for Calibre Libraries', 'images/wrench-hammer.png',
                              triggered=partial(self.show_all_library_statuses),unique_name=unique_name, favourites_menu_unique_name=unique_name)

        self.audit_log_menu.addSeparator()

        create_menu_action_unique(self, self.audit_log_menu, ' ', ' ',
                              triggered=None)

        tip = "<p style='white-space:wrap'>The Audit Log functionality is activated and deactivated only for the Current Library.  That means that if you wish to Uninstall Audit Log entirely, you must first deactivate it in each and every Library for which you previously activated it.  Otherwise, it will continue to 'work behind the scenes' adding new rows to the Audit Log, but you won't be able to see it or use it. "
        self.audit_log_menu.setToolTip(tip)
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def check_al_status_for_current_library(self):
        # Executed after initialization and also after library is changed...
        my_db = self.gui.library_view.model().db
        path = my_db.library_path
        path = os.path.join(path,'metadata.db')
        path = path.replace(os.sep, '/')
        if isbytestring(path):
            path = path.decode(filesystem_encoding)
        self.library_full_path = path
        k = self.library_full_path.upper()
        #~ if DEBUG: print(as_unicode(k))
        self.unknown_library = False
        self.upgrade_required = False
        if not k in prefs:
            self.unknown_library = True
            #~ if DEBUG: print("Audit Log: self.unknown_library = True; not in prefs.")
        else:
            if prefs[k] == None:
                self.unknown_library = True
                #~ if DEBUG: print("Audit Log: self.unknown_library = True; prefs[k] equals None")
            else:
                self.unknown_library = False
                #~ if DEBUG: print("Audit Log: self.unknown_library = False; prefs[k] = version of AL")
                if prefs[k] != __my_db_version__ :
                    self.upgrade_current_library_audit_log_db_version()

        if self.unknown_library:
            my_db,my_cursor,is_valid = self.apsw_connect_to_current_library_path()
            if not is_valid:
                return
            mysql = "SELECT name FROM sqlite_master WHERE type='table' AND name='_audit_log';"
            my_cursor.execute(mysql)
            tmp_rows = my_cursor.fetchall()
            if not tmp_rows:
                tmp_rows = []
            if len(tmp_rows) == 1:
                self.unknown_library = False
                #~ if DEBUG: print("Audit Log: self.unknown_library = False per SQL")
            my_db.close()

    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def check_al_status_for_current_user(self):

        self.new_user = True

        if not self.unknown_library:
            self.new_user = False

        for k,v in iteritems(prefs):
            if k:
                if "METADATA.DB" in k:
                    self.new_user = False
                    break
        #END FOR

        if self.new_user:
            if 'COLUMN__0_WIDTH' in prefs:
                self.new_user = False

        if 'ORIGINAL_SQLITE_OBJECTS_VERSION' in prefs:  # obsolete key will be deleted after next activation/deactivation
            if prefs['ORIGINAL_SQLITE_OBJECTS_VERSION'] == unicode_type("None"):  # obsolete key will be deleted after next activation/deactivation
                self.upgrade_required = True
                #~ if DEBUG: print("Audit Log: self.upgrade_required = True")
            else:
                if not self.unknown_library:  # legacy user who already upgraded to version 1.0.3
                    self.upgrade_required = False
                    k = self.library_full_path.upper()
                    prefs[k] = prefs['ORIGINAL_SQLITE_OBJECTS_VERSION']
                    prefs

        if self.new_user:
            self.upgrade_required = True
            #~ if DEBUG: print("Audit Log: self.new_user = True")
            #~ if DEBUG: print("Audit Log: self.upgrade_required = True")
            prefs['AUDIT_LOG_USER_TYPE'] = unicode_type("NEW")
            prefs
        else:
            prefs['AUDIT_LOG_USER_TYPE'] = unicode_type("OLD")
            prefs
            #~ if DEBUG: print("Audit Log: self.new_user = False")

    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def audit_log_control_all(self):
        self.selection_type = "ALL"
        self.audit_log_control()
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def audit_log_control_selected(self):
        self.selection_type = "SELECTED"
        self.audit_log_control()
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def audit_log_control(self):
        self.guidb = self.gui.library_view.model().db
        try:
            self.auditlogdialog.close()
        except:
            pass

        if self.new_user or self.unknown_library:
            return error_dialog(self.gui, _('Current Library Requires Activation'),_('<b>You must manually Activate AL, and then Restart Calibre. '), show=True)
        elif self.upgrade_required:
            return error_dialog(self.gui, _('AL: Current Library Installed Version Is Either Obsolete or Requires Reactivation'),_('<b>You must manually Deactive AL, Restart Calibre, Activate AL, and then Restart Calibre individually for ALL Libraries for which you have ever used Audit Log, if any. '), show=True)

        self.selected_books_list[:] = []
        self.selected_books_set = set(self.selected_books_list)

        self.get_selected_books()

        if len(self.selected_books_set) > 0 or self.selection_type == "ALL":
            audit_log_detail_list,custom_column_name_dict = self.obtain_audit_log()
            self.qaction.setIcon(get_icon(PLUGIN_ICONS[0]))
            self.auditlogdialog = AuditLogDialog(None,audit_log_detail_list,custom_column_name_dict)
            self.auditlogdialog.show()
            del audit_log_detail_list
            del custom_column_name_dict

        self.selected_books_list[:] = []
        self.selected_books_set = set(self.selected_books_list)

        k = self.library_full_path.upper()
        if not k in prefs:  #legacy library that was activated in an early version of AL
            prefs[k] = "1.0.3"
            prefs
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def get_selected_books(self):

        self.selected_books_list[:] = []
        self.selected_books_set = set(self.selected_books_list)

        if self.selection_type == "ALL":
            return

        book_ids_list = []

        book_ids_list = list(map(partial(self.convert_id_to_book), self.gui.library_view.get_selected_ids()))  #https://stackoverflow.com/questions/50671360/map-in-python-3-vs-python-2
        n = len(book_ids_list)
        if n == 0:
            del book_ids_list
            return error_dialog(self.gui, _('AL: '),_('No Books Were Selected; Canceled.'), show=True)

        for item in book_ids_list:
            book = item['calibre_id']
            book = as_unicode(book)
            self.selected_books_list.append(book)
        #END FOR

        self.selected_books_set = set(self.selected_books_list)

        del book_ids_list
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def obtain_audit_log(self):
        audit_log_detail_list,custom_column_name_dict = self.get_all_audit_log_rows()
        audit_log_detail_list.sort()
        return audit_log_detail_list,custom_column_name_dict
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def get_all_audit_log_rows(self):
        try:
            del audit_log_detail_list
            del custom_column_name_dict
        except:
            pass

        audit_log_detail_list = []
        custom_column_name_dict = {}

        my_db,my_cursor,is_valid = self.apsw_connect_to_current_library_path()
        if not is_valid:
            return audit_log_detail_list

        mysql = "SELECT name FROM sqlite_master WHERE type='table' AND name='_audit_log';"
        my_cursor.execute(mysql)
        tmp_rows = my_cursor.fetchall()
        if not tmp_rows:
            tmp_rows = []
        if len(tmp_rows) != 1:
            my_db.close()
            id = "0"
            book = "Error: Audit Log Is Not Active For Current Library"
            time_stamp = ""
            table_name = ""
            table_column = ""
            table_action = ""
            value_old = ""
            value_new = ""
            other = ""
            s = id,book,time_stamp,table_name,table_column,table_action,value_old,value_new,other
            audit_log_detail_list.append(s)
            k = self.library_full_path.upper()
            prefs[k] = None
            return audit_log_detail_list,custom_column_name_dict


        mysql = "SELECT id,book,time_stamp,table_name,table_column,table_action,value_old,value_new,other FROM _audit_log ORDER BY id DESC"
        my_cursor.execute(mysql)
        tmp_rows = my_cursor.fetchall()
        if not tmp_rows:
            tmp_rows = []
        for row in tmp_rows:
            if self.selection_type == "ALL":
                audit_log_detail_list.append(row)
            else:
                id,book,time_stamp,table_name,table_column,table_action,value_old,value_new,other = row
                book = as_unicode(book)
                if book in self.selected_books_set:
                    audit_log_detail_list.append(row)
        #END FOR
        del tmp_rows

        mysql = "SELECT id,label,name FROM custom_columns"
        my_cursor.execute(mysql)
        tmp_rows = my_cursor.fetchall()
        if not tmp_rows:
            tmp_rows = []
        for row in tmp_rows:
            id,label,name = row
            id = as_unicode(id)
            custom_column_name_dict[id] = row
        #END FOR
        del tmp_rows

        my_db.close()

        k = self.library_full_path.upper()
        if not k in prefs:
            prefs[k] = __my_version__
            prefs
        else:
            if prefs[k] == None:
                prefs[k] = __my_version__
                prefs

        return audit_log_detail_list,custom_column_name_dict
    #-----------------------------------------------------------------------------------------
    #-----------------------------------------------------------------------------------------
    def purge_audit_log(self):

        if self.new_user:
            return error_dialog(self.gui, _('New User: AL is Deactivated'),_('<b>You must manually Activate AL, and then Restart Calibre. '), show=True)
        elif self.upgrade_required:
             return error_dialog(self.gui, _('AL: Current Installed Version Is Obsolete'),_('<b>You must manually Deactive AL, Restart Calibre, Activate AL, and then Restart Calibre individually for ALL Libraries for which you have ever used Audit Log, if any. '), show=True)

        msg = "Are you sure you want to entirely clear the Audit Log Detail?"
        if question_dialog(self.gui, "AL Clear/Purge Log", msg):
            pass
        else:
            return

        my_db,my_cursor,is_valid = self.apsw_connect_to_current_library_path()
        if not is_valid:
            return

        my_cursor.execute("begin")
        mysql = "DELETE FROM _audit_log"
        try:
            my_cursor.execute(mysql)
        except:
            pass
        my_cursor.execute("commit")

        my_db.close()

        msg = "The Audit Log has been cleared/purged/emptied."
        self.gui.status_bar.show_message(_(msg), 10000)
    #---------------------------------------------------------------------------------------------------------------------------------------
    def apsw_connect_to_current_library_path(self):

        my_db = self.gui.library_view.model().db
        path = my_db.library_path
        path = os.path.join(path,'metadata.db')
        if isbytestring(path):
            path = path.decode(filesystem_encoding)
        path = path.replace(os.sep, '/')
        try:
            my_db = apsw.Connection(path)
        except Exception as e:
            if DEBUG: print(as_unicode(e))
            msg = as_unicode("Audit Log cannot connect to the path that you selected:" + path + " - " + as_unicode(e) )
            self.gui.status_bar.showMessage(msg)
            return None,None,False
        my_cursor = my_db.cursor()
        mysql = "PRAGMA main.busy_timeout = 15000;"      #PRAGMA busy_timeout = milliseconds;
        my_cursor.execute(mysql)

        self.library_full_path = path

        return my_db,my_cursor,True
    #---------------------------------------------------------------------------------------------------------------------------------------
    def undo_logged_changes(self):

        if self.new_user:
            return error_dialog(self.gui, _('New User: AL is Deactivated'),_('<b>You must manually Activate AL, and then Restart Calibre. '), show=True)
        elif self.upgrade_required:
            return error_dialog(self.gui, _('AL: Current Installed Version Is Obsolete'),_('<b>You must manually Deactive AL, Restart Calibre, Activate AL, and then Restart Calibre individually for ALL Libraries for which you have ever used Audit Log, if any. '), show=True)

        self.guidb = self.gui.library_view.model().db

        try:
            self.auditundodialog.close()
        except:
            pass

        self.selection_type = "SELECTED"
        self.get_selected_books()
        if len(self.selected_books_list) == 0:
            return

        filtered_audit_log_detail_list,custom_column_name_dict = self.obtain_audit_log()

        if not len(filtered_audit_log_detail_list) > 0:
            del custom_column_name_dict
            del filtered_audit_log_detail_list
            return error_dialog(self.gui, _('AL: Undo '),_('No Selected Books Have Existing Audit Log Detail; Undo Canceled.'), show=True)

        filtered_books_list = []

        for row in filtered_audit_log_detail_list:
            id,book,time_stamp,table_name,table_column,table_action,value_old,value_new,other = row
            book = as_unicode(book)
            filtered_books_list.append(book)

        custom_column_list = []

        for id,v in iteritems(custom_column_name_dict):
            id,label,name = v
            id = as_unicode(id)
            label = as_unicode(label)
            name = as_unicode(name)
            r = id,label,name
            custom_column_list.append(r)
        #END FOR

        custom_column_list.sort()

        filtered_books_list = list(set(filtered_books_list))
        filtered_books_list.sort()

        if len(filtered_books_list) > 0:
            self.auditundodialog = AuditUndoDialog(None,self.guidb,filtered_books_list,filtered_audit_log_detail_list,custom_column_name_dict,custom_column_list)
            self.auditundodialog.show()

        del filtered_books_list
        del custom_column_name_dict
        del custom_column_list
        del filtered_audit_log_detail_list

        k = self.library_full_path.upper()
        if not k in prefs:  #legacy library that was activated in an early version of AL
            prefs[k] = "1.0.3"
            prefs

    #---------------------------------------------------------------------------------------------------------------------------------------
    def convert_id_to_book(self, idval):
        book = {}
        book['calibre_id'] = idval
        return book
    #---------------------------------------------------------------------------------------------------------------------------------------
    def show_all_library_statuses(self):

        msg = "These are the current Audit Log statuses of each Library for which AL activation has taken place. \
                    Missing Libraries, if any, will be added here the next time that you use them:\
                    <br>"

        #~ for k,v in prefs.iteritems():
        for k,v in iteritems(prefs):
            if k:
                if "METADATA.DB" in k:
                    if not v:
                        continue
                    else:
                        msg = msg + "<br>" + k + "   status: "
                        v = "...ACTIVATED..." + "<br>"
                        msg = msg + v
                        msg = msg.replace("/METADATA.DB","")
        #END FOR

        #~ if DEBUG: print(msg)

        info_dialog(self.gui,"Audit Log",msg,show=True)
    #---------------------------------------------------------------------------------------------------------------------------------------
    def upgrade_current_library_audit_log_db_version(self):
        # Executed after initialization and also after library is changed...but only if needed...
        #~ -----------------------------------------------------------------
        # Current Database Version is:  1.0.7    as of:  22 May 2017
        # Current Database Version is:  1.0.8    as of:  12 Sept 2017
        #~ -----------------------------------------------------------------

        my_db,my_cursor,is_valid = self.apsw_connect_to_current_library_path()
        if not is_valid:
            return
        my_cursor.execute("begin")
        mysql1 = "\
                        CREATE TRIGGER IF NOT EXISTS '_audit_comments_text_update' \
                        AFTER UPDATE ON 'comments' \
                        FOR EACH ROW \
                        WHEN NEW.text IS NOT NULL \
                        BEGIN \
                        INSERT OR REPLACE INTO _audit_log (id,book,time_stamp,table_name,table_column,table_action,value_old,value_new,other) \
                        VALUES(NULL,OLD.book,datetime(),'comments','text','update',OLD.text,NEW.text,''); \
                        END"

        mysql2 = "\
                        CREATE TRIGGER IF NOT EXISTS '_audit_data_insert' \
                        AFTER INSERT ON 'data' \
                        FOR EACH ROW \
                        WHEN NEW.format IS NOT NULL \
                        BEGIN \
                        INSERT OR REPLACE INTO _audit_log (id,book,time_stamp,table_name,table_column,table_action,value_old,value_new,other) \
                        VALUES(NULL,NEW.book,datetime(),'data','format','insert','',NEW.format,''); \
                        END ;    "

        mysql3 = "\
                        CREATE TRIGGER  IF NOT EXISTS '_audit_data_delete' \
                        AFTER DELETE ON 'data' \
                        FOR EACH ROW \
                        WHEN OLD.format IS NOT NULL \
                        BEGIN \
                        INSERT OR REPLACE INTO _audit_log (id,book,time_stamp,table_name,table_column,table_action,value_old,value_new,other)   \
                        VALUES(NULL,OLD.book,datetime(),'data','format','delete',OLD.format,'',''); \
                        END; "

        my_cursor.execute(mysql1)      # Current Database Version is:  1.0.7    as of:  22 May 2017
        my_cursor.execute(mysql2)      # Current Database Version is:  1.0.8    as of:  12 Sept 2017
        my_cursor.execute(mysql3)      # Current Database Version is:  1.0.8    as of:  12 Sept 2017
        my_cursor.execute("commit")
        my_db.close()

        k = self.library_full_path.upper()
        prefs[k] = __my_db_version__      # example: "S:/CALIBRE/CALIBREAUDIT/METADATA.DB": "1.0.8"
        prefs

        if DEBUG: print("Audit Log:  Current Library ",k, " was upgraded to latest version of AL database: ", __my_db_version__)

    #---------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------
    #---------------------------------------------------------------------------------------------------------------------------------------
#END of ui.py