# -*- coding: utf-8 -*-
from __future__ import absolute_import, division, print_function, unicode_literals
__license__   = 'GPL v3'
__copyright__ = '2017,2018,2019,2020 DaltonST <DaltonShiTzu@outlook.com>'
__my_version__ = "1.0.16"  # Miscellany
#-------------------------------------------------------
__my_db_version__ = "1.0.8"  #only changes when any triggers change...imported by ui.py for automatic upgrade function...
#-------------------------------------------------------

from PyQt5.Qt import  Qt, QVBoxLayout, QHBoxLayout, QWidget, QFont, QPushButton, QLabel

import os, sys
import apsw

from calibre import isbytestring
from calibre.constants import filesystem_encoding, DEBUG, iswindows
from calibre.utils.config import JSONConfig

from polyglot.builtins import as_unicode, iteritems, unicode_type

prefs = JSONConfig('plugins/Audit Log')

prefs.defaults['COLUMN__0_WIDTH'] = unicode_type(1)     # special use
prefs.defaults['COLUMN__1_WIDTH'] = unicode_type(50)
prefs.defaults['COLUMN__2_WIDTH'] = unicode_type(200)
prefs.defaults['COLUMN__3_WIDTH'] = unicode_type(100)
prefs.defaults['COLUMN__4_WIDTH'] = unicode_type(125)
prefs.defaults['COLUMN__5_WIDTH'] = unicode_type(130)
prefs.defaults['COLUMN__6_WIDTH'] = unicode_type(125)
prefs.defaults['COLUMN__7_WIDTH'] = unicode_type(100)
prefs.defaults['COLUMN__8_WIDTH'] = unicode_type(50)   # Undo dialog only


PREFS_NAMESPACE = 'AuditLogPlugin'
PREFS_KEY_SETTINGS = 'settings'
#-------------------------------------------------------


class ConfigWidget(QWidget):

    #-----------------------------------------------------
    def __init__(self):

        QWidget.__init__(self)

        try:
            from calibre.gui2.ui import get_gui
            self.gui = get_gui()
        except Exception as e:
            if DEBUG: print(as_unicode(e))
            self.gui = None
        #----------------------------------------

        tip = "<p style='white-space:wrap'>You must Restart Calibre for any changes to take effect. You should then vacuum/compress this Library using menu path Library > Library Maintenance > Check Library. If you wish to Uninstall the entire AL plug-in, you must first Deactivate AL in every Library for which you have previously Activated it.  Otherwise, it will continue to operate as normal."

        self.layout_frame = QVBoxLayout()
        self.setLayout(self.layout_frame)
        self.layout_frame.setAlignment(Qt.AlignCenter)
        self.setToolTip(tip)

        font = QFont()

        font.setBold(False)
        font.setPointSize(10)

        #----------------------------------------

        for k,v in iteritems(prefs.defaults):
            if k in prefs:
                continue
            else:
                prefs[k] = v
                prefs
        #END FOR

        #----------------------------------------

        self.layout_pushbutton = QVBoxLayout()
        self.layout_pushbutton.setAlignment(Qt.AlignCenter)
        self.layout_frame.addLayout(self.layout_pushbutton)

        self.layout_pushbutton.addStretch(4)

        self.push_button_activate_standard_columns_current_library = QPushButton(" ", self)
        self.push_button_activate_standard_columns_current_library.setText("Activate 'Audit Log' for Standard Metadata for the Current Calibre Library")
        self.push_button_activate_standard_columns_current_library.clicked.connect(self.activate_audit_log_for_current_library_standard)
        self.push_button_activate_standard_columns_current_library.setFont(font)
        self.push_button_activate_standard_columns_current_library.setToolTip("<p style='white-space:wrap'>Activate 'Audit Log' for the Current Library for Standard Columns only.  You must Restart Calibre for any changes to take effect.  You should then vacuum/compress this Library using menu path Library > Library Maintenance > Check Library. If you wish to Uninstall the entire AL plug-in, you must first Deactivate AL in every Library for which you have previously Activated it.  Otherwise, it will continue to operate as normal.")
        self.layout_pushbutton.addWidget(self.push_button_activate_standard_columns_current_library)

        self.layout_pushbutton.addStretch(4)

        self.push_button_activate_custom_columns_current_library = QPushButton(" ", self)
        self.push_button_activate_custom_columns_current_library.setText("Activate 'Audit Log' for Custom Metadata for the Current Calibre Library")
        self.push_button_activate_custom_columns_current_library.clicked.connect(self.activate_audit_log_for_current_library_custom)
        self.push_button_activate_custom_columns_current_library.setFont(font)
        self.push_button_activate_custom_columns_current_library.setToolTip("<p style='white-space:wrap'>Activate 'Audit Log' for the Current Library for Custom Columns only.  You must Restart Calibre for any changes to take effect.  You should then vacuum/compress this Library using menu path Library > Library Maintenance > Check Library. If you wish to Uninstall the entire AL plug-in, you must first Deactivate AL in every Library for which you have previously Activated it.  Otherwise, it will continue to operate as normal.")
        self.layout_pushbutton.addWidget(self.push_button_activate_custom_columns_current_library)

        self.layout_pushbutton.addStretch(4)

        self.push_button_deactivate_current_library = QPushButton(" ", self)
        self.push_button_deactivate_current_library.setText("Deactivate 'Audit Log' Completely for the Current Calibre Library")
        self.push_button_deactivate_current_library.clicked.connect(self.deactivate_audit_log_for_current_library)
        self.push_button_deactivate_current_library.setFont(font)
        self.push_button_deactivate_current_library.setToolTip("<p style='white-space:wrap'>Deactivate 'Audit Log' for the Current Library for BOTH Standard and Custom Columns.  You must Restart Calibre for any changes to take effect.  You should then vacuum/compress this Library using menu path Library > Library Maintenance > Check Library. If you wish to Uninstall the entire AL plug-in, you must first Deactivate AL in every Library for which you have previously Activated it.  Otherwise, it will continue to operate as normal.")
        self.layout_pushbutton.addWidget(self.push_button_deactivate_current_library)

        self.layout_pushbutton.addStretch(4)

        #......................................................

        self.resize(self.sizeHint())

        self.library_full_path = None

        if prefs['AUDIT_LOG_USER_TYPE'] == unicode_type("NEW"):
            self.deactivate_audit_log_for_current_library  #failsafe

    #-----------------------------------------------------
    def activate_audit_log_for_current_library_standard(self):
        self.create_sqlite_objects_standard()

        k = self.library_full_path.upper()
        prefs[k] =__my_db_version__
        prefs

        if 'ORIGINAL_SQLITE_OBJECTS_VERSION' in prefs:
            del prefs['ORIGINAL_SQLITE_OBJECTS_VERSION']
        if 'CURRENT_SQLITE_OBJECTS_VERSION' in prefs:
            del prefs['CURRENT_SQLITE_OBJECTS_VERSION']

    #-----------------------------------------------------
    def activate_audit_log_for_current_library_custom(self):
        self.create_sqlite_objects_custom()

        k = self.library_full_path.upper()
        prefs[k] =__my_db_version__            # example: "S:/CALIBRE/CALIBREAUDIT/METADATA.DB": "1.0.7"
        prefs

        if 'ORIGINAL_SQLITE_OBJECTS_VERSION' in prefs:
            del prefs['ORIGINAL_SQLITE_OBJECTS_VERSION']
            prefs
        if 'CURRENT_SQLITE_OBJECTS_VERSION' in prefs:
            del prefs['CURRENT_SQLITE_OBJECTS_VERSION']
            prefs
    #-----------------------------------------------------
    def deactivate_audit_log_for_current_library(self):
        self.drop_sqlite_objects_custom()
        self.drop_sqlite_objects_standard()
        msg = "Audit Log recording has been completely deactivated for the current Library.  Restart Calibre Now."
        self.gui.status_bar.show_message(_(msg), 10000)

        prefs['AUDIT_LOG_USER_TYPE'] = unicode_type("OLD")
        k = self.library_full_path.upper()
        prefs[k] = None
        prefs

        if 'ORIGINAL_SQLITE_OBJECTS_VERSION' in prefs:
            del prefs['ORIGINAL_SQLITE_OBJECTS_VERSION']
            prefs
        if 'CURRENT_SQLITE_OBJECTS_VERSION' in prefs:
            del prefs['CURRENT_SQLITE_OBJECTS_VERSION']
            prefs
    #-----------------------------------------------------
    def create_sqlite_objects_standard(self):

        my_db,my_cursor,is_valid = self.apsw_connect_to_current_library_path()
        if not is_valid:
            return

        my_cursor.execute("begin")
        mysql = 'CREATE TABLE IF NOT EXISTS "_audit_log" ("id" INTEGER PRIMARY KEY  AUTOINCREMENT  NOT NULL , "book" INTEGER, "time_stamp" TIMESTAMP DEFAULT CURRENT_TIMESTAMP, "table_name" TEXT NOT NULL , "table_column" TEXT NOT NULL , "table_action" TEXT NOT NULL , "value_old" TEXT, "value_new" TEXT, "other" TEXT);'
        my_cursor.execute(mysql)
        my_cursor.execute("commit")

        #~ 22 Triggers as of Version 1.0.0
        #~ 23 Standard Triggers as of Version 1.0.7
        #~ 25 Standard Triggers as of Version 1.0.8

        #~ _audit_books_author_sort_insert
        #~ _audit_books_author_sort_update
        #~ _audit_books_id_delete
        #~ _audit_books_languages_link_delete    # changed in Version 1.0.3
        #~ _audit_books_languages_link_insert     # changed in Version 1.0.3
        #~ _audit_books_pubdate_insert
        #~ _audit_books_pubdate_update
        #~ _audit_books_publishers_link_delete
        #~ _audit_books_publishers_link_insert
        #~ _audit_books_ratings_link_delete
        #~ _audit_books_ratings_link_insert
        #~ _audit_books_series_index_update
        #~ _audit_books_series_link_delete
        #~ _audit_books_series_link_insert
        #~ _audit_books_tags_link_delete
        #~ _audit_books_tags_link_insert
        #~ _audit_books_title_insert
        #~ _audit_books_title_update
        #~ _audit_comments_text_delete
        #~ _audit_comments_text_insert
        #~ _audit_comments_text_update           # new in Version 1.0.7; invoked by plugins...see ui.py for automatic upgrade function...
        #~ _audit_data_delete                             # new in Version 1.0.8
        #~ _audit_data_insert                              # new in Version 1.0.8
        #~ _audit_identifiers_delete
        #~ _audit_identifiers_insert

        my_cursor.execute("begin")

        mysql19 = "\
            CREATE TRIGGER IF NOT EXISTS '_audit_books_author_sort_insert' \
            AFTER INSERT ON 'books' \
            FOR EACH ROW \
            WHEN NEW.author_sort IS NOT NULL \
              BEGIN \
                        INSERT OR REPLACE INTO _audit_log (id,book,time_stamp,table_name,table_column,table_action,value_old,value_new,other) \
                        VALUES(NULL,NEW.id,datetime(),'books','author_sort','insert','',NEW.author_sort,'');      \
              END; "

        mysql1 = "\
            CREATE TRIGGER IF NOT EXISTS '_audit_books_author_sort_update' \
            AFTER UPDATE ON 'books' \
            FOR EACH ROW \
            WHEN OLD.author_sort != NEW.author_sort \
              BEGIN \
                        INSERT OR REPLACE INTO _audit_log (id,book,time_stamp,table_name,table_column,table_action,value_old,value_new,other) \
                        VALUES(NULL,OLD.id,datetime(),'books','author_sort','update',OLD.author_sort,NEW.author_sort,''); \
              END; "

        mysql22 = "\
            CREATE TRIGGER IF NOT EXISTS '_audit_books_id_delete' \
            AFTER DELETE ON 'books' \
            FOR EACH ROW \
            WHEN  OLD.id IS NOT NULL \
              BEGIN \
                        INSERT OR REPLACE INTO _audit_log (id,book,time_stamp,table_name,table_column,table_action,value_old,value_new,other) \
                        VALUES(NULL,OLD.id,datetime(),'books','id','delete book',OLD.id,'',OLD.title||' ---- '||OLD.author_sort); \
              END; "

        mysql2 = "\
            CREATE TRIGGER IF NOT EXISTS '_audit_books_languages_link_delete' \
            AFTER DELETE ON 'books_languages_link' \
            FOR EACH ROW \
            WHEN  OLD.lang_code IS NOT NULL \
              BEGIN \
                        INSERT OR REPLACE INTO _audit_log (id,book,time_stamp,table_name,table_column,table_action,value_old,value_new,other) \
                        VALUES(NULL,OLD.book,datetime(),'books_languages_link','lang_code','delete',(SELECT lang_code FROM languages WHERE languages.id = OLD.lang_code),'',''); \
              END; "

        mysql3 = "\
            CREATE TRIGGER IF NOT EXISTS '_audit_books_languages_link_insert' \
            AFTER INSERT ON 'books_languages_link' \
            FOR EACH ROW \
            WHEN  NEW.lang_code IS NOT NULL \
              BEGIN \
                   INSERT OR REPLACE INTO _audit_log (id,book,time_stamp,table_name,table_column,table_action,value_old,value_new,other) \
                   VALUES(NULL,NEW.book,datetime(),'books_languages_link','lang_code','insert','',(SELECT lang_code FROM languages WHERE languages.id = NEW.lang_code),''); \
              END; "

        mysql20 = "\
            CREATE TRIGGER IF NOT EXISTS '_audit_books_pubdate_insert' \
            AFTER INSERT ON 'books' \
            FOR EACH ROW \
            WHEN NEW.pubdate IS NOT NULL \
              BEGIN \
                        INSERT OR REPLACE INTO _audit_log (id,book,time_stamp,table_name,table_column,table_action,value_old,value_new,other) \
                        VALUES(NULL,NEW.id,datetime(),'books','pubdate','insert','',NEW.pubdate,'');      \
              END; "

        mysql4 = "\
            CREATE TRIGGER IF NOT EXISTS '_audit_books_pubdate_update' \
            AFTER UPDATE ON 'books' \
            FOR EACH ROW \
            WHEN OLD.pubdate != NEW.pubdate \
              BEGIN \
                        INSERT OR REPLACE INTO _audit_log (id,book,time_stamp,table_name,table_column,table_action,value_old,value_new,other) \
                        VALUES(NULL,OLD.id,datetime(),'books','pubdate','update',OLD.pubdate,NEW.pubdate,''); \
              END; "

        mysql5 = "\
            CREATE TRIGGER IF NOT EXISTS '_audit_books_publishers_link_delete' \
            AFTER DELETE ON 'books_publishers_link' \
            FOR EACH ROW \
            WHEN  OLD.publisher IS NOT NULL \
              BEGIN \
                        INSERT OR REPLACE INTO _audit_log (id,book,time_stamp,table_name,table_column,table_action,value_old,value_new,other) \
                        VALUES(NULL,OLD.book,datetime(),'books_publishers_link','publisher','delete',(SELECT name FROM publishers WHERE publishers.id = OLD.publisher),'',''); \
              END; "

        mysql6 = "\
            CREATE TRIGGER IF NOT EXISTS '_audit_books_publishers_link_insert' \
            AFTER INSERT ON 'books_publishers_link' \
            FOR EACH ROW \
            WHEN  NEW.publisher IS NOT NULL \
              BEGIN \
                   INSERT OR REPLACE INTO _audit_log (id,book,time_stamp,table_name,table_column,table_action,value_old,value_new,other) \
                   VALUES(NULL,NEW.book,datetime(),'books_publishers_link','publisher','insert','',(SELECT name FROM publishers WHERE publishers.id = NEW.publisher),''); \
              END; "

        mysql7 = "\
            CREATE TRIGGER IF NOT EXISTS '_audit_books_ratings_link_delete' \
            AFTER DELETE ON 'books_ratings_link' \
            FOR EACH ROW \
            WHEN  OLD.rating IS NOT NULL \
              BEGIN \
                   INSERT OR REPLACE INTO _audit_log (id,book,time_stamp,table_name,table_column,table_action,value_old,value_new,other) \
                   VALUES(NULL,OLD.book,datetime(),'books_ratings_link','rating','delete',(SELECT rating FROM ratings WHERE ratings.id = OLD.rating),'',''); \
              END; "

        mysql8 = "\
            CREATE TRIGGER IF NOT EXISTS '_audit_books_ratings_link_insert' \
            AFTER INSERT ON 'books_ratings_link' \
            FOR EACH ROW \
            WHEN  NEW.rating IS NOT NULL \
              BEGIN \
                   INSERT OR REPLACE INTO _audit_log (id,book,time_stamp,table_name,table_column,table_action,value_old,value_new,other) \
                   VALUES(NULL,NEW.book,datetime(),'books_ratings_link','rating','insert','',(SELECT rating FROM ratings WHERE ratings.id = NEW.rating),''); \
              END; "

        mysql9 = "\
            CREATE TRIGGER IF NOT EXISTS '_audit_books_series_index_update' \
            AFTER UPDATE ON 'books' \
            FOR EACH ROW \
            WHEN OLD.series_index != NEW.series_index \
              BEGIN \
                        INSERT OR REPLACE INTO _audit_log (id,book,time_stamp,table_name,table_column,table_action,value_old,value_new,other) \
                        VALUES(NULL,OLD.id,datetime(),'books','series_index','update',OLD.series_index,NEW.series_index,''); \
              END; "

        mysql10 = "\
            CREATE TRIGGER IF NOT EXISTS '_audit_books_series_link_delete' \
            AFTER DELETE ON 'books_series_link' \
            FOR EACH ROW \
            WHEN  OLD.series IS NOT NULL \
              BEGIN \
                        INSERT OR REPLACE INTO _audit_log (id,book,time_stamp,table_name,table_column,table_action,value_old,value_new,other) \
                        VALUES(NULL,OLD.book,datetime(),'books_series_link','series','delete',(SELECT name FROM series WHERE series.id = OLD.series),'',''); \
              END; "

        mysql11 = "\
            CREATE TRIGGER IF NOT EXISTS '_audit_books_series_link_insert' \
            AFTER INSERT ON 'books_series_link' \
            FOR EACH ROW \
            WHEN  NEW.series IS NOT NULL \
              BEGIN \
                   INSERT OR REPLACE INTO _audit_log (id,book,time_stamp,table_name,table_column,table_action,value_old,value_new,other) \
                   VALUES(NULL,NEW.book,datetime(),'books_series_link','series','insert','',(SELECT name FROM series WHERE series.id = NEW.series),''); \
              END; "

        mysql12 = "\
            CREATE TRIGGER IF NOT EXISTS '_audit_books_tags_link_delete' \
            AFTER DELETE ON 'books_tags_link' \
            FOR EACH ROW \
            WHEN  OLD.tag IS NOT NULL \
              BEGIN \
                        INSERT OR REPLACE INTO _audit_log (id,book,time_stamp,table_name,table_column,table_action,value_old,value_new,other) \
                        VALUES(NULL,OLD.book,datetime(),'books_tags_link','tag ','delete',(SELECT name FROM tags WHERE tags.id = OLD.tag),'',''); \
              END; "

        mysql13 = "\
            CREATE TRIGGER IF NOT EXISTS '_audit_books_tags_link_insert' \
            AFTER INSERT ON 'books_tags_link' \
            FOR EACH ROW \
            WHEN  NEW.tag IS NOT NULL \
              BEGIN \
                   INSERT OR REPLACE INTO _audit_log (id,book,time_stamp,table_name,table_column,table_action,value_old,value_new,other) \
                   VALUES(NULL,NEW.book,datetime(),'books_tags_link','tag ','insert','',(SELECT name FROM tags WHERE tags.id = NEW.tag),''); \
              END; "


        mysql21 = "\
            CREATE TRIGGER IF NOT EXISTS '_audit_books_title_insert' \
            AFTER INSERT ON 'books' \
            FOR EACH ROW \
            WHEN NEW.title IS NOT NULL \
              BEGIN \
                        INSERT OR REPLACE INTO _audit_log (id,book,time_stamp,table_name,table_column,table_action,value_old,value_new,other) \
                        VALUES(NULL,NEW.id,datetime(),'books','title','insert new book','',NEW.title,'');      \
              END; "


        mysql14 = "\
            CREATE TRIGGER IF NOT EXISTS '_audit_books_title_update' \
            AFTER UPDATE ON 'books' \
            FOR EACH ROW \
            WHEN OLD.title != NEW.title \
              BEGIN \
                        INSERT OR REPLACE INTO _audit_log (id,book,time_stamp,table_name,table_column,table_action,value_old,value_new,other) \
                        VALUES(NULL,OLD.id,datetime(),'books','title','update',OLD.title,NEW.title,''); \
              END; "

        mysql15 = "\
            CREATE TRIGGER IF NOT EXISTS '_audit_comments_text_delete' \
            AFTER DELETE ON 'comments' \
            FOR EACH ROW \
            WHEN OLD.text IS NOT NULL \
              BEGIN \
                        INSERT OR REPLACE INTO _audit_log (id,book,time_stamp,table_name,table_column,table_action,value_old,value_new,other) \
                        VALUES(NULL,OLD.book,datetime(),'comments','text','delete',OLD.text,'',''); \
              END; "

        mysql16 = "\
            CREATE TRIGGER IF NOT EXISTS '_audit_comments_text_insert' \
            AFTER INSERT ON 'comments' \
            FOR EACH ROW \
            WHEN NEW.text IS NOT NULL \
              BEGIN \
                        INSERT OR REPLACE INTO _audit_log (id,book,time_stamp,table_name,table_column,table_action,value_old,value_new,other) \
                        VALUES(NULL,NEW.book,datetime(),'comments','text','insert or replace','',NEW.text,''); \
              END; "

        mysql23 = "\
                CREATE TRIGGER IF NOT EXISTS '_audit_comments_text_update' \
                AFTER UPDATE ON 'comments' \
                FOR EACH ROW \
                WHEN NEW.text IS NOT NULL \
                BEGIN \
                INSERT OR REPLACE INTO _audit_log (id,book,time_stamp,table_name,table_column,table_action,value_old,value_new,other) \
                VALUES(NULL,OLD.book,datetime(),'comments','text','update',OLD.text,NEW.text,''); \
                END"

        mysql17 = "\
            CREATE TRIGGER IF NOT EXISTS '_audit_identifiers_delete' \
            AFTER DELETE ON 'identifiers' \
            FOR EACH ROW \
            WHEN  OLD.type IS NOT NULL \
              BEGIN \
                        INSERT OR REPLACE INTO _audit_log (id,book,time_stamp,table_name,table_column,table_action,value_old,value_new,other) \
                        VALUES(NULL,OLD.book,datetime(),'identifiers','type','delete',OLD.type,'',OLD.val); \
              END; "

        mysql18 = "\
            CREATE TRIGGER IF NOT EXISTS '_audit_identifiers_insert' \
            AFTER INSERT ON 'identifiers' \
            FOR EACH ROW \
            WHEN  NEW.type IS NOT NULL \
              BEGIN \
                        INSERT OR REPLACE INTO _audit_log (id,book,time_stamp,table_name,table_column,table_action,value_old,value_new,other) \
                        VALUES(NULL,NEW.book,datetime(),'identifiers','type','insert','',NEW.type,NEW.val); \
              END; "

        mysql24 = "\
            CREATE TRIGGER IF NOT EXISTS '_audit_data_insert' \
            AFTER INSERT ON 'data' \
            FOR EACH ROW \
            WHEN NEW.format IS NOT NULL \
            BEGIN \
            INSERT OR REPLACE INTO _audit_log (id,book,time_stamp,table_name,table_column,table_action,value_old,value_new,other) \
            VALUES(NULL,NEW.book,datetime(),'data','format','insert','',NEW.format,''); \
            END ;  "

        mysql25 = "\
            CREATE TRIGGER IF NOT EXISTS '_audit_data_delete' \
            AFTER DELETE ON 'data' \
            FOR EACH ROW \
            WHEN OLD.format IS NOT NULL \
            BEGIN \
            INSERT OR REPLACE INTO _audit_log (id,book,time_stamp,table_name,table_column,table_action,value_old,value_new,other)   \
            VALUES(NULL,OLD.book,datetime(),'data','format','delete',OLD.format,'',''); \
            END; "

        my_cursor.execute(mysql1)
        my_cursor.execute(mysql2)
        my_cursor.execute(mysql3)
        my_cursor.execute(mysql4)
        my_cursor.execute(mysql5)
        my_cursor.execute(mysql6)
        my_cursor.execute(mysql7)
        my_cursor.execute(mysql8)
        my_cursor.execute(mysql9)
        my_cursor.execute(mysql10)
        my_cursor.execute(mysql11)
        my_cursor.execute(mysql12)
        my_cursor.execute(mysql13)
        my_cursor.execute(mysql14)
        my_cursor.execute(mysql15)
        my_cursor.execute(mysql16)
        my_cursor.execute(mysql17)
        my_cursor.execute(mysql18)
        my_cursor.execute(mysql19)
        my_cursor.execute(mysql20)
        my_cursor.execute(mysql21)
        my_cursor.execute(mysql22)
        my_cursor.execute(mysql23)  # New in Version 1.0.7; used by plugins.
        my_cursor.execute(mysql24)  # New in Version 1.0.8
        my_cursor.execute(mysql25)  # New in Version 1.0.8

        my_cursor.execute("commit")

        my_db.close()

        k = self.library_full_path.upper()
        prefs[k] = __my_db_version__          # example: "S:/CALIBRE/CALIBREAUDIT/METADATA.DB": "1.0.8"
        prefs

        msg = "Audit Log recording has been activated for the current Library.  Restart Calibre Now."
        self.gui.status_bar.show_message(_(msg), 10000)
    #-----------------------------------------------------
    def create_sqlite_objects_custom(self):

        #~ -----------------------------
        #~ -----------------------------
        mysql_unnormalized_insert = "\
            CREATE TRIGGER IF NOT EXISTS '_audit_custom_column_[NN]_insert' \
            AFTER INSERT ON 'custom_column_[NN]' \
            FOR EACH ROW \
              BEGIN \
                        INSERT INTO _audit_log (id,book,time_stamp,table_name,table_column,table_action,value_old,value_new,other) \
                        VALUES(NULL,NEW.book,datetime(),'custom_column_[NN]','value','insert or replace','',NEW.value,''); \
              END; "
        #~ -----------------------------
        mysql_unnormalized_update = "\
            CREATE TRIGGER IF NOT EXISTS '_audit_custom_column_[NN]_update' \
            AFTER UPDATE ON 'custom_column_[NN]' \
            FOR EACH ROW \
              BEGIN \
                        INSERT INTO _audit_log (id,book,time_stamp,table_name,table_column,table_action,value_old,value_new,other) \
                        VALUES(NULL,OLD.book,datetime(),'custom_column_[NN]','value','update',OLD.value,NEW.value,''); \
              END; "
        #~ -----------------------------
        mysql_unnormalized_delete = "\
            CREATE TRIGGER IF NOT EXISTS '_audit_custom_column_[NN]_delete' \
            AFTER DELETE ON 'custom_column_[NN]' \
            FOR EACH ROW \
              BEGIN \
                        INSERT INTO _audit_log (id,book,time_stamp,table_name,table_column,table_action,value_old,value_new,other) \
                        VALUES(NULL,OLD.book,datetime(),'custom_column_[NN]','value','delete',OLD.value,'',''); \
              END; "

        #~ -----------------------------
        #~ -----------------------------
        mysql_normalized_link_insert = "\
            CREATE TRIGGER IF NOT EXISTS '_audit_books_custom_column_[NN]_link_insert' \
            AFTER INSERT ON 'books_custom_column_[NN]_link' \
            FOR EACH ROW \
              BEGIN \
                        INSERT INTO _audit_log (id,book,time_stamp,table_name,table_column,table_action,value_old,value_new,other) \
                        VALUES(NULL,NEW.book,datetime(),'books_custom_column_[NN]_link','value','insert','',(SELECT value FROM custom_column_[NN] WHERE custom_column_[NN].id = NEW.value),''); \
              END; "
        #~ -----------------------------
        mysql_normalized_link_update = "\
            CREATE TRIGGER IF NOT EXISTS '_audit_books_custom_column_[NN]_link_update' \
            AFTER UPDATE ON 'books_custom_column_[NN]_link' \
            FOR EACH ROW \
              BEGIN \
                        INSERT INTO _audit_log (id,book,time_stamp,table_name,table_column,table_action,value_old,value_new,other) \
                        VALUES(NULL,OLD.book,datetime(),'books_custom_column_[NN]_link','value','update',(SELECT value FROM custom_column_[NN] WHERE custom_column_[NN].id = OLD.value),(SELECT value FROM custom_column_[NN] WHERE custom_column_[NN].id = NEW.value),''); \
              END; "
        #~ -----------------------------
        mysql_normalized_link_delete = "\
            CREATE TRIGGER IF NOT EXISTS '_audit_books_custom_column_[NN]_link_delete' \
            AFTER DELETE ON 'books_custom_column_[NN]_link' \
            FOR EACH ROW \
              BEGIN \
                        INSERT INTO _audit_log (id,book,time_stamp,table_name,table_column,table_action,value_old,value_new,other) \
                        VALUES(NULL,OLD.book,datetime(),'books_custom_column_[NN]_link','value','delete',(SELECT value FROM custom_column_[NN] WHERE custom_column_[NN].id = OLD.value),'',''); \
              END; "
        #~ -----------------------------
        #~ -----------------------------
        #~ -----------------------------
        #~ -----------------------------

        my_db,my_cursor,is_valid = self.apsw_connect_to_current_library_path()
        if not is_valid:
            return

        mysql = "SELECT id,name,datatype,normalized FROM custom_columns WHERE datatype != 'composite' "
        my_cursor.execute(mysql)
        tmp_rows = my_cursor.fetchall()
        if not tmp_rows:
            my_db.close()
            return

        #~ User can log only custom columns, and not standard columns, so the log table must exist.
        my_cursor.execute("begin")
        mysql = 'CREATE TABLE IF NOT EXISTS "_audit_log" ("id" INTEGER PRIMARY KEY  AUTOINCREMENT  NOT NULL , "book" INTEGER, "time_stamp" TIMESTAMP DEFAULT CURRENT_TIMESTAMP, "table_name" TEXT NOT NULL , "table_column" TEXT NOT NULL , "table_action" TEXT NOT NULL , "value_old" TEXT, "value_new" TEXT, "other" TEXT);'
        my_cursor.execute(mysql)
        my_cursor.execute("commit")

        my_cursor.execute("begin")

        for row in tmp_rows:
            id,name,datatype,normalized = row
            id = as_unicode(id)
            normalized = as_unicode(normalized)
            if normalized == as_unicode("0"):
                s = mysql_unnormalized_insert.replace("[NN]",id)
                my_cursor.execute(s)
                if DEBUG: print("trigger created for custom column: ", id, name, normalized)
                s = mysql_unnormalized_update.replace("[NN]",id)
                my_cursor.execute(s)
                if DEBUG: print("trigger created for custom column: ", id, name, normalized)
                s = mysql_unnormalized_delete.replace("[NN]",id)
                my_cursor.execute(s)
                if DEBUG: print("trigger created for custom column: ", id, name, normalized)
            else:
                #~ -------base table----------
                #~ not supported
                #~ -------link table----------
                s = mysql_normalized_link_insert.replace("[NN]",id)
                my_cursor.execute(s)
                if DEBUG: print("trigger created for custom column link: ", id, name, normalized)
                s = mysql_normalized_link_update.replace("[NN]",id)
                my_cursor.execute(s)
                if DEBUG: print("trigger created for custom column link: ", id, name, normalized)
                s = mysql_normalized_link_delete.replace("[NN]",id)
                my_cursor.execute(s)
                if DEBUG: print("trigger created for custom column link: ", id, name, normalized)
        #END FOR

        my_cursor.execute("commit")

        my_db.close()

        k = self.library_full_path.upper()
        prefs[k] = __my_db_version__
        prefs

        del tmp_rows
    #-----------------------------------------------------
    def drop_sqlite_objects_standard(self):

        my_db,my_cursor,is_valid = self.apsw_connect_to_current_library_path()
        if not is_valid:
            return

        #~ 22 Standard Triggers as of Version 1.0.0
        #~ 25 Standard Triggers as of Version 1.0.8

        my_cursor.execute("begin")

        mysql = "DROP TRIGGER IF EXISTS _audit_books_author_sort_insert"
        my_cursor.execute(mysql)
        mysql = "DROP TRIGGER IF EXISTS _audit_books_author_sort_update"
        my_cursor.execute(mysql)
        mysql = "DROP TRIGGER IF EXISTS _audit_books_id_delete"
        my_cursor.execute(mysql)
        mysql = "DROP TRIGGER IF EXISTS _audit_books_languages_link_delete"
        my_cursor.execute(mysql)
        mysql = "DROP TRIGGER IF EXISTS _audit_books_languages_link_insert"
        my_cursor.execute(mysql)
        mysql = "DROP TRIGGER IF EXISTS _audit_books_pubdate_insert"
        my_cursor.execute(mysql)
        mysql = "DROP TRIGGER IF EXISTS _audit_books_pubdate_update"
        my_cursor.execute(mysql)
        mysql = "DROP TRIGGER IF EXISTS _audit_books_publishers_link_delete"
        my_cursor.execute(mysql)
        mysql = "DROP TRIGGER IF EXISTS _audit_books_publishers_link_insert"
        my_cursor.execute(mysql)
        mysql = "DROP TRIGGER IF EXISTS _audit_books_ratings_link_delete"
        my_cursor.execute(mysql)
        mysql = "DROP TRIGGER IF EXISTS _audit_books_ratings_link_insert"
        my_cursor.execute(mysql)
        mysql = "DROP TRIGGER IF EXISTS _audit_books_series_index_update"
        my_cursor.execute(mysql)
        mysql = "DROP TRIGGER IF EXISTS _audit_books_series_link_delete"
        my_cursor.execute(mysql)
        mysql = "DROP TRIGGER IF EXISTS _audit_books_series_link_insert"
        my_cursor.execute(mysql)
        mysql = "DROP TRIGGER IF EXISTS _audit_books_tags_link_delete"
        my_cursor.execute(mysql)
        mysql = "DROP TRIGGER IF EXISTS _audit_books_tags_link_insert"
        my_cursor.execute(mysql)
        mysql = "DROP TRIGGER IF EXISTS _audit_books_title_insert"
        my_cursor.execute(mysql)
        mysql = "DROP TRIGGER IF EXISTS _audit_books_title_update"
        my_cursor.execute(mysql)
        mysql = "DROP TRIGGER IF EXISTS _audit_comments_text_delete"
        my_cursor.execute(mysql)
        mysql = "DROP TRIGGER IF EXISTS _audit_comments_text_insert"
        my_cursor.execute(mysql)
        mysql = "DROP TRIGGER IF EXISTS _audit_comments_text_update"    # new in Version 1.0.7
        my_cursor.execute(mysql)
        mysql = "DROP TRIGGER IF EXISTS _audit_data_insert"    # new in Version 1.0.8
        my_cursor.execute(mysql)
        mysql = "DROP TRIGGER IF EXISTS _audit_data_delete"    # new in Version 1.0.8
        my_cursor.execute(mysql)
        mysql = "DROP TRIGGER IF EXISTS _audit_identifiers_delete"
        my_cursor.execute(mysql)
        mysql = "DROP TRIGGER IF EXISTS _audit_identifiers_insert"
        my_cursor.execute(mysql)

        my_cursor.execute("commit")

        my_cursor.execute("begin")
        mysql = "DROP TABLE IF EXISTS _audit_log"   #drop the table last...
        my_cursor.execute(mysql)
        my_cursor.execute("commit")

        my_db.close()
    #-----------------------------------------------------
    def drop_sqlite_objects_custom(self):

        my_db,my_cursor,is_valid = self.apsw_connect_to_current_library_path()
        if not is_valid:
            return

        mysql = "SELECT type,name FROM sqlite_master WHERE type = 'trigger' AND name LIKE '_audit%custom%column%'  "
        my_cursor.execute(mysql)
        tmp_rows = my_cursor.fetchall()
        if not tmp_rows:
            my_db.close()
            return

        my_cursor.execute("begin")

        mysql = as_unicode("DROP TRIGGER IF EXISTS [XXXXXXXXXXX]")

        for type,name in tmp_rows:
            name = as_unicode(name)
            s = mysql.replace("[XXXXXXXXXXX]",name)
            #~ if DEBUG: print("trigger dropped: ", name)
            my_cursor.execute(s)
        #END FOR

        my_cursor.execute("commit")


        my_db.close()
    #-----------------------------------------------------
    def save_settings(self):
        prefs
    #-----------------------------------------------------
    def validate(self):
        return True
    #---------------------------------------------------------------------------------------------------------------------------------------
    def apsw_connect_to_current_library_path(self):

        my_db = self.gui.library_view.model().db
        path = my_db.library_path
        path = os.path.join(path,'metadata.db')
        if isbytestring(path):
            path = path.decode(filesystem_encoding)
        path = path.replace(os.sep, '/')
        try:
            my_db = apsw.Connection(path)
        except Exception as e:
            if DEBUG: print(as_unicode(e))
            msg = as_unicode("Audit Log cannot connect to the path that you selected:" + path + " - " + as_unicode(e) )
            self.gui.status_bar.showMessage(msg)
            return None,None,False
        my_cursor = my_db.cursor()
        mysql = "PRAGMA main.busy_timeout = 15000;"      #PRAGMA busy_timeout = milliseconds;
        my_cursor.execute(mysql)

        self.library_full_path = path

        return my_db,my_cursor,True
    #-----------------------------------------------------
#END of config.py