import java.io.BufferedReader;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.PrintWriter;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Date;

/**
 * This class providers a formatter that scans a directory of JPG and PNG files and then creates
 * and iLiad friendly directory of HTML/picture/manifest.xml files to form a "Picutre Book".
 * Usage: java PictureBook pictures picturebook
 * 
 * @author Scott Turner (scotty1024@mac.com)
 * @version $Revision: 1.0 $
 * @Date October 19, 2006
 */
public class PictureBook {

    /**
     * Method is invoked by the command line.
     * Argument 0 is the name of the directory to scan for pictures.
     * Argument 1 is the name of a directory to create and place the HTML files, pictures and manifest.xml in.
     * Application returns a positive non-zero result value if there is an error.
     *
     * @param args[] a <code>String</code> see above.
     * @exception Exception if an error occurs
     */
    public static void main(String args[])
	throws Exception
    {
	// Get picture directory from argument 0
	String pictureName = args[0];
	File pictureDir = new File(pictureName);
	if (!pictureDir.exists()) {
	    System.err.println("Picture Directory " + pictureName + " not found.");
	    System.exit(1);
	}
	if (!pictureDir.isDirectory()) {
	    System.err.println("Picture Directory " + pictureName + " not a directory.");
	    System.exit(1);
	}

	// Get destination directory from argument 1
	File destDir = new File(args[1]);
	if (destDir.exists()) {
	    // Zap its contents if it exists
	    if (!deleteFile(destDir)) {
		System.err.println("Couldn't clear " + args[1]);
		System.exit(2);
	    }
	}
	destDir.mkdirs();

	// Create image directory
	File imgDir = new File(destDir, "img");
	imgDir.mkdirs();

	File[] pictures = pictureDir.listFiles(new FileFilter() {
		public boolean accept(File aFile) {
		    return aFile.isFile() && (aFile.getName().toLowerCase().endsWith(".jpg") ||
					      aFile.getName().toLowerCase().endsWith(".png"));
		}
	    });

	System.out.println("Picture Directory contains " + pictures.length + " pictures.");

	// Setup to create manifest file
	StringBuffer manifest = new StringBuffer(2048 + (pictures.length * 64));
	Date now = new Date();
	SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss");

	manifest.append("<?xml version=\"1.0\" encoding=\"utf-8\"?>\n");
	manifest.append("<package>\n");
	manifest.append("  <metadata>\n");
	manifest.append("    <dc-metadata>\n");
	manifest.append("      <Title>").append(args[0]).append("</Title>\n");
	manifest.append("      <Description>").append("Picutre Book").append("</Description>\n");
	manifest.append("      <Date>").append(format.format(now)).append("</Date>\n");
	manifest.append("      <Format/>\n");
	manifest.append("      <Identifier/>\n");
	manifest.append("      <Language>en-us</Language>\n");
	manifest.append("      <Type>overview</Type>\n");
	manifest.append("    </dc-metadata>\n");
	manifest.append("    <y-metadata>\n");
	manifest.append("      <startpage>page0001.html</startpage>\n");
	manifest.append("      <version>000</version>\n");
	manifest.append("    </y-metadata>\n");
	manifest.append("  </metadata>\n");
	manifest.append("  <index>\n");
	manifest.append("    <pagelist type=\"overview\" count=\"").append(pictures.length).append("\">\n");

	// Iterator through pages and create an HTML file for each page
	StringBuffer index = new StringBuffer(8192);
	for (int page = 1; page <= pictures.length; page++) {
	    File picture = pictures[page-1];
	    index.setLength(0);

	    // Add line to manifest for this HTML page
	    manifest.append("      <page number=\"").append(makePageNumber(page)).append("\" url=\"").append(makePageName(page)).append("\"/>\n");

	    // HTML file pre-amble
	    index.append("<html>\n");
	    index.append("<head>\n");

	    // iLiad links and metadata
	    if (page != 1) {
		index.append("<link rel=\"prev\" href=\"").append(makePageName(page - 1)).append("\"/>\n");
	    }
	    if (page != pictures.length) {
		index.append("<link rel=\"next\" href=\"").append(makePageName(page + 1)).append("\"/>\n");
	    }
	    index.append("<meta name=\"keywords\" type=\"overview\">\n");

	    // HTML header info
	    index.append("<title>Page ").append(String.valueOf(page)).append("</title>\n");
	    index.append("</head>\n");
	    index.append("<body>\n");
	    index.append("<center>\n");
	    index.append("<img src=\"img/").append(picture.getName()).append("\"/>\n");
	    index.append("</center>\n");
	    index.append("</body>\n");
	    index.append("</html>\n");

	    // Write HTML file for page
	    File pageFile = new File(destDir, makePageName(page));
	    PrintWriter out = new PrintWriter( new FileOutputStream( pageFile, false));
	    out.print(index);
	    out.close();

	    // Copy picture file
	    byte[] buffer = readWholeFile(picture);
	    File copyFile = new File(imgDir, picture.getName());
	    FileOutputStream copyOut = new FileOutputStream( copyFile, false);
	    copyOut.write(buffer);
	    copyOut.close();
	    buffer = null;
	}

	// Finish manifest
	manifest.append("    </pagelist>\n");
	manifest.append("  </index>\n");
	manifest.append("</package>\n");

	// Write manifest file
	File manifestFile = new File(destDir, "manifest.xml");
	PrintWriter out = new PrintWriter( new FileOutputStream( manifestFile, false));
	out.print(manifest);
	out.close();
    }

    /**
     * Given aPageNumber return a zero padded String value.
     *
     * @param aPageNumber an <code>int</code> value contianing page number, first page is numbered as 1.
     * @return a <code>String</code> zero padded value
     */
    public static String makePageNumber(int aPageNumber) {
	String pageName = null;
	if (aPageNumber < 10) {
	    pageName = "000";
	} else 	if (aPageNumber < 100) {
	    pageName = "00";
	} else 	if (aPageNumber < 1000) {
	    pageName = "0";
	} else 	if (aPageNumber < 10000) {
	    pageName = "";
	} else {
	    System.err.println("RFC is more than 10000 pages and is too large.");
	    System.exit(3);
	}

	return pageName + aPageNumber;
    }

    /**
     * Given a page of aPageNumber returns a string of the name to store that page's HTML in.
     *
     * @param aPageNumber an <code>int</code> value contianing page number, first page is numbered as 1.
     * @return a <code>String</code> value containing name for page's HTML to be stored as.
     */
    public static String makePageName(int aPageNumber) {
	return "page" + makePageNumber( aPageNumber) + ".html";
    }

    /**
     * This method will delete the given File. If the file is a directory then all files in
     * the directory will be deleted and then the directory itself will be deleted.
     * If if File is a simple file, it is deleted.
     *
     * @param aFile a <code>File</code> containing the path to the file/directory to be deleted
     * @return a <code>boolean</code> of true if file was deleted, false if it couldn't be removed.
     */
    public static boolean deleteFile(File aFile) {
        if (aFile.isDirectory()) {
            String[] children = aFile.list();
            for (int i=0; i<children.length; i++) {
                boolean success = deleteFile(new File(aFile, children[i]));
                if (!success) {
                    return false;
                }
            }
        }
    
        // The directory is now empty (or a simple file) so delete it
        return aFile.delete();
    }

    /**
     * Methods reads an entire file into a byte[].
     *
     * @param file a <code>File</code> pointing to file to read.
     * @return a <code>byte[]</code> containing contents of the file or null if file could not be read.
     */
    private static byte[] readWholeFile(File file) {
	int fileSize = (int)file.length();
	byte[] buf = new byte[fileSize];

	try {
	    FileInputStream in = new FileInputStream(file);

	    if (in.read(buf) != buf.length) {
		System.err.println("Error reading file: " + file);
		System.exit(1);
	    }
	    in.close();
	    in = null;

	    return buf;
	} catch (Exception e) {
	    System.err.println("Error loading picture file: " + file + " message: " + e.getMessage());
	    e.printStackTrace();
	}
	return null;
    }
}
