#!/usr/bin/env python
# -*- coding: utf-8 -*-
# vim:ts=4:sw=4:softtabstop=4:smarttab:expandtab
import sys, os, re, shutil, platform, tempfile, imghdr, time
from datetime import timedelta

# find optipng binary
def find_optipng_exe(bk):
    ''' returns the optipng exe file path '''
    optipng_exe_path = None
    if sys.platform.startswith('win'):
        optipng_exe_path = os.path.join(bk._w.plugin_dir, bk._w.plugin_name, 'optipng.exe')
    elif sys.platform.startswith('darwin'):
        optipng_exe_path = os.path.join(bk._w.plugin_dir, bk._w.plugin_name, 'optipng')
        os.chmod(optipng_exe_path, 0o744)
    else:
        optipng_which_path = shutil.which('optipng')
        if optipng_which_path: optipng_exe_path = optipng_which_path
    return optipng_exe_path

# find pngquant binary
def find_pngquant_exe(bk):
    ''' returns the pngquant exe file path '''
    pngquant_exe_path = None
    if sys.platform.startswith('win'):
        pngquant_exe_path = os.path.join(bk._w.plugin_dir, bk._w.plugin_name, 'pngquant.exe')
    elif sys.platform.startswith('darwin'):
        pngquant_exe_path = os.path.join(bk._w.plugin_dir, bk._w.plugin_name, 'pngquant')
        os.chmod(pngquant_exe_path, 0o744)
    else:
        pngquant_which_path = shutil.which('pngquant')
        if pngquant_which_path: pngquant_exe_path = pngquant_which_path
    return pngquant_exe_path

# wrapper for optipng
def png_wrapper(*args):
    ''' a wrapper for optipng '''
    import subprocess
    startupinfo = None

    # stop the windows console popping up every time the prog is run
    if sys.platform.startswith('win'):
        startupinfo = subprocess.STARTUPINFO()
        startupinfo.dwFlags |= subprocess.STARTF_USESHOWWINDOW
        startupinfo.wShowWindow = subprocess.SW_HIDE

    process = subprocess.Popen(list(args), stdout=subprocess.PIPE, stderr=subprocess.PIPE, startupinfo=startupinfo)
    stdout, stderr = process.communicate()

    returncode = process.returncode
    return stdout.decode('utf-8', 'ignore'), stderr.decode('utf-8', 'ignore'), returncode

# main routine
def run(bk):
    ''' the main routine '''

    # get ebook root folder
    ebook_root = bk._w.ebook_root

    # get os temp folder
    temp_dir = tempfile.gettempdir()
    
    #----------------------------
    # get preferences
    #----------------------------
    prefs = prefs = bk.getPrefs()

    # save default preferences
    if prefs == {}:
        prefs['lossless'] = True
        prefs['quality'] = '0-100'
        optipng_exe_path = find_optipng_exe(bk)
        if optipng_exe_path:
            prefs['optipng_exe_path'] = optipng_exe_path
            bk.savePrefs(prefs)
        pngquant_exe_path = find_pngquant_exe(bk)
        if pngquant_exe_path:
            prefs['pngquant_exe_path'] = pngquant_exe_path
            bk.savePrefs(prefs)

    # get pref values
    optipng_exe_path = prefs.get('optipng_exe_path', None)
    pngquant_exe_path = prefs.get('pngquant_exe_path', None)
    debug = prefs.get('debug', False) # show debug messages
    lossless = prefs.get('lossless', True)
    quality = prefs.get('quality', '0-100')

    # get png_files
    png_manifest_items = []
    for manifest_id, href, mime in bk.manifest_iter():
        if mime == "image/png":
            png_manifest_items.append((manifest_id, href, mime))

    if png_manifest_items == []:
        print('No png files found.\nClick OK to close the Plugin Runner window.')
        return 0

    # make sure that the optipng binary was found
    if not optipng_exe_path or not os.path.isfile(optipng_exe_path):
        optipng_exe_path = find_optipng_exe(bk)
        if optipng_exe_path:
            prefs['optipng_exe_path'] = optipng_exe_path
            bk.savePrefs(prefs)
        else:
            print('optipng binary not found!\nPlease reinstall the plugin.\nClick OK to close the Plugin Runner window.')
            return -1

    # make sure that the optipng binary was found
    if (not pngquant_exe_path or not os.path.isfile(pngquant_exe_path)) and not lossless:
        pngquant_exe_path = find_pngquant_exe(bk) 
        if pngquant_exe_path:
            prefs['pngquant_exe_path'] = pngquant_exe_path
            bk.savePrefs(prefs)
        else:
            print('pngquant binary not found!\nPlease reinstall the plugin.\nClick OK to close the Plugin Runner window.')
            return -1

    # get mode for status messages
    if not lossless:
        mode = 'Lossy'
    else:
        mode = 'Lossless'


    # process all manifested png files in the Images folder
    bytes_saved = 0
    start_time = time.time()
    for manifest_id, href, mime in png_manifest_items:
        png_file_name = os.path.basename(href)
        print('\n{} processing: {} ...'.format(mode, png_file_name))
        processed_file_path = os.path.join(temp_dir, png_file_name)
        if os.path.isfile(processed_file_path):
            os.remove(processed_file_path)

        # Sigil 1.x and higher supports custom image paths
        if  bk.launcher_version() >= 20190927:
            original_png_path = os.path.abspath(os.path.join(ebook_root, bk.id_to_bookpath(manifest_id)))
        else:
            original_png_path = os.path.abspath(os.path.join(ebook_root, 'OEBPS', href))

        # double-check the image header to make sure it's really a png file
        image_header = imghdr.what(original_png_path)
        if image_header != 'png':
            print(png_file_name, 'is a {} file!\n'.format(image_header))
            continue

        # get original png size
        original_png_size = os.path.getsize(original_png_path)
        print('Original file size:', '{0:.2f}KB'.format(original_png_size/1024))

        # assemble args
        if lossless:
            args = [optipng_exe_path, '-fix', '-strip', 'all', original_png_path, '-dir', temp_dir]
        else:
            args = [pngquant_exe_path, '--quality=' + quality, '--strip', '--force', '--output', processed_file_path, '--', original_png_path]

        # run optipng/pngquant
        if debug: print(args)
        stdout, stderr, returncode = png_wrapper(*args)

        # display error messages, if optipng failed
        if returncode != 0:
            print('Optimization failed!\n{}\nClick OK to close the Plugin Runner window'.format(stdout + stderr, returncode))
            prefs['quality'] = '0-100'
            bk.savePrefs(prefs)
            return -1

        # check whether optipng/pngquant created an optimized file
        if os.path.isfile(processed_file_path):
            processed_file_size = os.path.getsize(processed_file_path)
            percentage = '({0:.2f}%)'.format(processed_file_size/original_png_size * 100)
            print('Optimized file size:', '{0:.2f}KB'.format(processed_file_size/1024), percentage)

            # only replace the image if the processed image is smaller
            if processed_file_size < original_png_size:
                with open(processed_file_path, 'rb') as file:
                    optimized_image_data = file.read()
                bk.writefile(manifest_id, optimized_image_data)
                print('Image replaced with optimized image.')
                bytes_saved += (original_png_size - processed_file_size)
            else:
                print('Skipped')

            # delete temp file
            os.remove(processed_file_path)
        else:
            # if there's no output file, optimized and original files sizes are identical
            print('Optimized file size:', '{0:.2f}KB'.format(original_png_size/1024))
            print('Skipped')

    end_time = time.time()
    print('\nProcessing time:', str(timedelta(seconds=(end_time - start_time)))[:-4])

    # display the number of bytes that were saved
    if bytes_saved > 0: print('\nTotal reduction: {0:.2f}KB'.format(bytes_saved/1024))
    print('\nDone.\nClick OK to close the Plugin Runner window.')

    return 0

def main():
    print('I reached main when I should not have\n')
    return -1

if __name__ == "__main__":
    sys.exit(main())
