#!/usr/bin/env python
# -*- coding: utf-8 -*-
# vim:ts=4:sw=4:softtabstop=4:smarttab:expandtab
import sys, os, re, shutil, platform, tempfile, imghdr

# find jpegoptim binary
def find_jpegoptim_exe(bk):
    ''' returns the md exe file path '''
    jpegoptim_exe_path = None
    if sys.platform.startswith('win'):
        jpegoptim_exe_path = os.path.join(bk._w.plugin_dir, bk._w.plugin_name, 'jpegoptim.exe')
    elif sys.platform.startswith('darwin'):
        jpegoptim_exe_path = os.path.join(bk._w.plugin_dir, bk._w.plugin_name, 'jpegoptim')
        os.chmod(jpegoptim_exe_path, 0o744)
    else:
        jpegoptim_exe_path = shutil.which('jpegoptim')
    return jpegoptim_exe_path

# wrapper for jpegoptim
def jo_wrapper(*args):
    ''' a wrapper for jpegoptim '''
    import subprocess
    startupinfo = None

    # stop the windows console popping up every time the prog is run
    if sys.platform.startswith('win'):
        startupinfo = subprocess.STARTUPINFO()
        startupinfo.dwFlags |= subprocess.STARTF_USESHOWWINDOW
        startupinfo.wShowWindow = subprocess.SW_HIDE

    process = subprocess.Popen(list(args), stdout=subprocess.PIPE, stderr=subprocess.PIPE, startupinfo=startupinfo)
    stdout, stderr = process.communicate()

    returncode = process.returncode
    return stdout.decode('utf-8', 'ignore'), stderr.decode('utf-8', 'ignore'), returncode

# main routine
def run(bk):
    ''' the main routine '''

    # get ebook root folder
    ebook_root = bk._w.ebook_root

    # get os temp folder
    temp_dir = tempfile.gettempdir()

    # get jpeg_files
    jpeg_manifest_items = []
    for manifest_id, href, mime in bk.manifest_iter():
        if mime == "image/jpeg":
            jpeg_manifest_items.append((manifest_id, href, mime))
    
    if jpeg_manifest_items == []:
        print('No jpeg files found.\nClick OK to close the Plugin Runner window.')
        return 0

    #----------------------------
    # get preferences
    #----------------------------
    prefs = prefs = bk.getPrefs()

    # save default preferences
    if prefs == {}:
        prefs['quality'] = 100
        jpegoptim_exe_path = find_jpegoptim_exe(bk)
        if jpegoptim_exe_path:
            prefs['jpegoptim_exe_path'] = jpegoptim_exe_path
            bk.savePrefs(prefs)

    # get pref values
    jpegoptim_exe_path = prefs.get('jpegoptim_exe_path', None)
    debug = prefs.get('debug', False) # show debug messages
    quality = prefs.get('quality', 100)
    
    # get mode for status messages
    if int(quality) < 100:
        mode = 'Lossy ({}%)'.format(str(quality))
    else:
        mode = 'Lossless'

    # make sure that the jpegoptim binary was found
    if not jpegoptim_exe_path or not os.path.isfile(jpegoptim_exe_path): 
        print('jpegoptim binary not found!\nPlease reinstall the plugin.\nClick OK to close the Plugin Runner window.')
        return -1

    # process all manifested jpeg files in the Images folder
    bytes_saved = 0
    for manifest_id, href, mime in jpeg_manifest_items:
        jpeg_file_name = os.path.basename(href)
        print('\n{} optimization: {} ...'.format(mode, jpeg_file_name))
        processed_file_path = os.path.join(temp_dir, jpeg_file_name)

        # Sigil 1.x and higher supports custom image paths
        if  bk.launcher_version() >= 20190927:
            original_jpeg_path = os.path.abspath(os.path.join(ebook_root, bk.id_to_bookpath(manifest_id)))
        else:
            original_jpeg_path = os.path.abspath(os.path.join(ebook_root, 'OEBPS', href))

        # double-check the file header to make sure it's really a jpeg file
        image_header = imghdr.what(original_jpeg_path)
        if image_header != 'jpeg':
            print(jpeg_file_name, 'is a {} file!\n'.format(image_header))
            continue

        # get original jpeg size
        original_jpeg_size = os.path.getsize(original_jpeg_path)
        print('Original file size:', '{0:.2f}KB'.format(original_jpeg_size/1024))

        # assemble args
        if quality < 100:
            # lossy optmization (-m = max quality)
            args = [jpegoptim_exe_path,'-m', str(quality), '-s', '-o', original_jpeg_path, '-d', temp_dir]
        else:
            # lossless optimization
            args = [jpegoptim_exe_path, '-s', '-o', original_jpeg_path, '-d', temp_dir]

        # run jpegoptim
        if debug: print(args)
        stdout, stderr, returncode = jo_wrapper(*args)

        # display error messages, if jpegoptim failed
        if returncode != 0:
            print('jpegoptim failed!\n{}\nClick OK to close the Plugin Runner window'.format(stdout + stderr, returncode))
            return -1

        # check whether jpegoptim created an optimized file
        if os.path.isfile(processed_file_path):
            processed_file_size = os.path.getsize(processed_file_path)
            percentage = '({0:.2f}%)'.format(processed_file_size/original_jpeg_size * 100)
            print('Optimized file size:', '{0:.2f}KB'.format(processed_file_size/1024), percentage)

            # only replace the image if the processed image is smaller
            if processed_file_size < original_jpeg_size:
                with open(processed_file_path, 'rb') as file:
                    optimized_image_data = file.read()
                bk.writefile(manifest_id, optimized_image_data)
                print('Image replaced with optimized image.')
                bytes_saved += (original_jpeg_size - processed_file_size)

            # delete temp file
            os.remove(processed_file_path)
        else:
            # if there's no output file, optimized and original files sizes are identical
            print(stderr)
            print('Optimized file size:', '{0:.2f}KB'.format(original_jpeg_size/1024))
            print('Skipped')

    # display the number of bytes that were saved
    if bytes_saved > 0: print('\nTotal reduction: {0:.2f}KB'.format(bytes_saved/1024))
    print('\nDone.\nClick OK to close the Plugin Runner window.')

    return 0

def main():
    print('I reached main when I should not have\n')
    return -1

if __name__ == "__main__":
    sys.exit(main())
