#!/usr/bin/env python3

from qt.core import Qt, QProgressDialog, QDialog, QVBoxLayout, QPushButton, QMessageBox, QLabel, QTimer
import textwrap

from calibre_plugins.tematika_pricecheck.config import prefs
from calibre_plugins.tematika_pricecheck.pricecheck import pcheck
from calibre_plugins.tematika_pricecheck.unzip import install_libs
from calibre.utils.date import format_date

class PricecheckProgressDialog(QProgressDialog):

    def __init__(self, gui, book_ids, db, pricecount, bookstotal, dtable, col, status_msg_type='books', action_type=_('Pricing')):
        self.total_count = len(book_ids)
        QProgressDialog.__init__(self, '', _(
            'Cancel'), 0, self.total_count, gui)
        self.setMinimumWidth(400)
        self.setMinimumHeight(150)
        
        self.book_ids, self.db, self.pricecount, self.bookstotal, self.dtable, self.col = book_ids, db, pricecount, bookstotal, dtable, col
        self.action_type, self.status_msg_type = action_type, status_msg_type
        self.gui = gui
        self.setWindowTitle('%s %d %s...' % (
            self.action_type, self.total_count, self.status_msg_type))
        # ... ...
        self.i, self.t, = 0, 0
        # ... ...
        QTimer.singleShot(0, self.do_timer_start)
        self.exec_()

    def do_timer_start(self):
        dtitlestr = ''
        title_id = self.book_ids[self.t]
        dtitle = textwrap.wrap(self.db.title(title_id, index_is_id=True), width=60)
        for d in dtitle:
            dtitlestr += str(d)
            dtitlestr += "\n"
        self.setWindowTitle(_('%s %d %s  (%d prices found)...') % (
            self.action_type, self.total_count, self.status_msg_type, self.pricecount))
        self.setLabelText('%s: %s' % (self.action_type, dtitlestr))
        
        self.setValue(self.i)
        QTimer.singleShot(0, self.do_book_action)

    def do_book_action(self):
        from datetime import datetime
        if self.wasCanceled():
            return self.do_close()
        if self.i >= self.total_count:
            return self.do_close()

        # code for processing a single book ...
        dtitlestr = ''
        book_id = self.book_ids[self.i]
        if self.t < self.total_count - 1:
            self.t += 1
        title_id = self.book_ids[self.t]
        self.i += 1
        dtitle = textwrap.wrap(self.db.title(title_id, index_is_id=True), width=60)
        for d in dtitle:
            dtitlestr += str(d)
            dtitlestr += "\n"
        self.setWindowTitle(_('%s %d %s  (%d prices found)...') % (
            self.action_type, self.total_count, self.status_msg_type, self.pricecount))
        self.setLabelText('%s: %s' % (self.action_type, dtitlestr))
        date_col = prefs['date_column']
        
        mi = self.db.get_metadata(book_id, index_is_id=True)
        if mi:
            mi, self.pricecount, self.dtable, adddate = pcheck(mi, self.pricecount, self.dtable, self.col)
            if adddate:
                pricedate = format_date(datetime.now(), 'iso', assume_utc=False, as_utc=False)
                mi.set(date_col,pricedate)
            self.db.set_metadata(book_id, mi)
            self.bookstotal += 1
        self.setValue(self.i)
        QTimer.singleShot(0, self.do_book_action)

    def do_close(self):
        self.hide()
        self.gui = None



class Dialog(QDialog):

    def __init__(self, gui, icon, do_user_config):
        QDialog.__init__(self, gui)
        self.gui = gui
        self.do_user_config = do_user_config

        # The current database shown in the GUI
        # db is an instance of the class LibraryDatabase from db/legacy.py
        # This class has many, many methods that allow you to do a lot of
        # things. For most purposes you should use db.new_api, which has
        # a much nicer interface from db/cache.py
        self.db = gui.current_db

        self.l = QVBoxLayout()
        self.setLayout(self.l)

        self.label = QLabel('Tematika Pricecheck')
        self.l.addWidget(self.label)

        self.setWindowTitle('Tematika Pricecheck')
        self.setWindowIcon(icon)

        self.about_button = QPushButton('About', self)
        self.about_button.clicked.connect(self.about)
        self.l.addWidget(self.about_button)

        self.update_metadata_button = QPushButton(
            'Add Price to Custom Column', self)
        self.update_metadata_button.clicked.connect(self.update_metadata)
        self.l.addWidget(self.update_metadata_button)

        self.conf_button = QPushButton(
            'Configure this plugin', self)
        self.conf_button.clicked.connect(self.config)
        self.l.addWidget(self.conf_button)

        self.resize(self.sizeHint())

    def about(self):
        # Get the about text from a file inside the plugin zip file
        # The get_resources function is a builtin function defined for all your
        # plugin code. It loads files from the plugin zip file. It returns
        # the bytes from the specified file.
        #
        # Note that if you are loading more than one file, for performance, you
        # should pass a list of names to get_resources. In this case,
        # get_resources will return a dictionary mapping names to bytes. Names that
        # are not found in the zip file will not be in the returned dictionary.
        text = get_resources('about.txt')
        QMessageBox.about(self, 'About pricecheck',
                          text.decode('utf-8'))

    def update_metadata(self):
        '''
        Set the metadata in the files in the selected book's record to
        match the current metadata in the database.
        '''
        install_libs(None)
        from tabulate import tabulate

        from calibre.ebooks.metadata.meta import set_metadata
        from calibre.gui2 import error_dialog, info_dialog

        # Get currently selected books
        rows = self.gui.library_view.selectionModel().selectedRows()
        if not rows or len(rows) == 0:
            return error_dialog(self.gui, 'Cannot update metadata',
                                'No books selected', show=True)
        # Map the rows to book ids
        ids = list(map(self.gui.library_view.model().id, rows))
        db = self.gui.current_db
        #Get custom column from preferences
        col = prefs['custom_column']
        if col == '':
            info_dialog(self, 'No Column Set',
                        'You must first create a custom column to store your prices and add the column to this plugin using the configuration button', show=True)
        bookstotal = 0
        pricecount = 0
        dtable = []

        dlg = PricecheckProgressDialog(self.gui, ids, db, pricecount, bookstotal, dtable, col)
        nf = dlg.bookstotal - dlg.pricecount
        if dlg.wasCanceled():
            # do whatever should be done if user cancelled
            print(tabulate(dlg.table, headers=["Book", "ISBN", "Price"]))

            info_dialog(self, 'Canceled',
                        'Process was canceled after updating %d book(s) \n\nAdded a total of %d Prices. \n\n %d prices were not found.' % (
                            dlg.bookstotal, dlg.pricecount, nf),
                        show=True)
        else:
            print(tabulate(dlg.dtable, headers=["Book", "ISBN", "Price"]))
            info_dialog(self, 'Updated files',
                        'Updated the metadata of %d book(s) \n\nAdded a total of %d Prices. \n\n %d prices were not found.' % (
                            dlg.bookstotal, dlg.pricecount, nf),
                        show=True)
            self.close()        

    def config(self):
        self.do_user_config(parent=self)
