import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileReader;
import java.io.InputStreamReader;
import java.io.IOException;
import java.io.Reader;
import java.util.HashSet;
import java.util.Iterator;
import java.util.zip.ZipInputStream;

import org.pdfbox.pdmodel.PDDocument;
import org.pdfbox.pdmodel.PDPage;
import org.pdfbox.pdmodel.common.PDRectangle;
import org.pdfbox.pdmodel.edit.PDPageContentStream;
import org.pdfbox.pdmodel.font.PDSimpleFont;
import org.pdfbox.pdmodel.font.PDType1Font;

/**
 * This class provides a formatter that parses an RTF file and converts it to an iLiad sized
 * PDF file. At present it processes only embed and par tags in the RTF.
 *
 * This class requires the PDFBox jar: http://www.pdfbox.org.
 *
 * NOTE: A zip'd RTF file (ala http://www.Baen.com) can be processed directly.
 *
 * Usage: java RTFToIliad your.rtf iLiad.pdf
 * or
 * Usage: java RTFToIliad baen-rtf.zip iLiad.pdf
 * 
 * @author Scott Turner (scotty1024@mac.com)
 * @version $Revision: 1.1 $
 * @Date October 18, 2006
 */
public class RTFToIliad {
    private static int currentPageNumber = 1;

    /**
     * My Best Guess at the Media Box of an iLiad (4.88in x 5.9in
     */
    public static final PDRectangle PAGE_SIZE_ILIAD = new PDRectangle( 351, 425 );

    /* Paragraph properties: reset by pard */
    private static float ri = 0.0f;	// Right indent
    private static float li = 0.0f;	// Left indent
    private static float fi = 0.0f;	// First Sentence indent
    private static float sa = 0.0f;	// Space After

    private static int marginH = 8;
    private static int marginV = 20;
    private static int fontSize = 10;
    private static boolean bold = false;
    private static boolean italic = false;
    private static PDSimpleFont curFont = null;
    private static PDSimpleFont fontRoman = null;
    private static PDSimpleFont fontBold = null;
    private static PDSimpleFont fontItalic = null;
    private static PDSimpleFont fontBoldItalic = null;
    private static float height = 0.0f;
    private static PDDocument doc = null;
    private static PDPage page = null;
    private static PDPageContentStream contentStream = null;
    private static float y = -1;
    private static float x = -1;
    private static float curWidth = 0.0f;
    private static RTFPullParser rpp = null;
    private static StringBuffer nextLineToDraw = new StringBuffer(16384);

    /**
     * A really big static main method that opens, parses and converts an RTF file into an iLiad
     * sized PDF document.
     *
     * @param args Command line arguments of which there are two: name of RTF file (which can be zipped) and the name of the PDF to be written.
     * 
     * @exception IOException If there is an error reading the RTF or writing the PDF.
     */
    public static void main(String[] args)
	throws IOException
    {
	if (args.length != 2) {
	    usage();
	}

	fontRoman = PDType1Font.HELVETICA;
	fontBold = PDType1Font.HELVETICA_BOLD;
	fontItalic= PDType1Font.HELVETICA_OBLIQUE;
	fontBoldItalic= PDType1Font.HELVETICA_BOLD_OBLIQUE;

	height = fontRoman.getFontDescriptor().getFontBoundingBox().getHeight()/1000;
            
	//calculate font height and increase by 5 percent.
	height = height*fontSize*1.05f;
	doc = new PDDocument();
	page = new PDPage();
	page.setMediaBox( PAGE_SIZE_ILIAD );
	doc.addPage(page);

	float maxStringLength = PAGE_SIZE_ILIAD.getWidth() - 2*marginH;

	rpp = new RTFPullParser();

	if (args[0].toLowerCase().endsWith(".zip")) {
	    ZipInputStream zipFile = new ZipInputStream(new FileInputStream(args[0]));
	    zipFile.getNextEntry();
	    rpp.setInput ( new InputStreamReader(zipFile));
	} else {
	    rpp.setInput ( new FileReader(new File(args[0])));
	}

	try {
	    for (int eventType = rpp.getEventType(); eventType != RTFPullParser.END_DOCUMENT; eventType = rpp.next()) {
		if (eventType == RTFPullParser.COMMAND) {
		    //Paragraph-formatting attributes include justification (like \qj for full justification), \liN, \riN, and \fiN for (respectively) paragraph indenting on the left, paragraph indenting on the right, and left indenting for just the first line.
		    if ("pard".equals(rpp.getName())) {
			ri = 0.0f;
			li = 0.0f;
			fi = 0.0f;
			sa = 0.0f;
		    } else if ("par".equals(rpp.getName())) {
			// Flush any pending text to page
			flushCurrentText();

			// Move down the page
			//moveTextPosition(marginH - x, -(height / 2));
			//moveTextPosition(0, -(height / 2));
			System.out.println("sa: " + sa + " height / 2 " + (height/2));
			moveTextPosition(0, -sa);
			baseLineSet = false;
			curWidth = 0.0f;
		    } else if ("b".equals(rpp.getName())) {
			// Flush any pending text to page
			flushCurrentText();
			if (rpp.hasArgument()) {
			    bold = rpp.getArgument() != 0;
			} else {
			    bold = true;
			}
			//System.out.println("Page: " + currentPageNumber + " Set bold: " + bold);
		    } else if ("i".equals(rpp.getName())) {
			// Flush any pending text to page
			flushCurrentText();
			if (rpp.hasArgument()) {
			    italic = rpp.getArgument() != 0;
			} else {
			    italic = true;
			}
			//System.out.println("Page: " + currentPageNumber + " Set italic: " + italic);
		    } else if ("pagebb".equals(rpp.getName())) {
			// Flush any pending text to page
			flushCurrentText();
			newPage();
		    } else if ("sa".equals(rpp.getName())) {
			if (rpp.hasArgument()) {
			    sa = ((rpp.getArgument() / 1440.0f) * 1000.0f) / height;
			} else {
			    sa = 0.0f;
			}
		    } else if ("emdash".equals(rpp.getName())) {
			nextLineToDraw.append((char)0x2014);
		    }
		} else if (eventType == RTFPullParser.TEXT) {
		    String[] lineWords = rpp.getTextCharacters().trim().split( " " );
		    int lineIndex = 0;
		    selectCurrentFont();
		    while (lineIndex < lineWords.length) {   
			float lengthIfUsingNextWord = 0;

			nextLineToDraw.append( lineWords[lineIndex] );
			nextLineToDraw.append( " " );
			lineIndex++;
			if (lineIndex < lineWords.length) {
			    String lineWithNextWord = nextLineToDraw.toString() + lineWords[lineIndex];
			    lengthIfUsingNextWord = (curFont.getStringWidth( lineWithNextWord )/1000) * fontSize;

			    if ((curWidth + lengthIfUsingNextWord) >= maxStringLength) {
				//System.out.println("lengthIfUsingNextWord: " + lengthIfUsingNextWord + " '" + lineWithNextWord + "'");
				if (y < marginV) {
				    newPage();
				}
				//System.out.println( "Drawing string at " + x + "," + y );
                    
				// Draw an entire line of text
				if (!baseLineSet) {
				    moveTextPosition(marginH - x, -height);
				}
				//System.out.println("drawText: x " + x + " y " + y + " text '" + nextLineToDraw.toString() + "'");
				contentStream.drawString( nextLineToDraw.toString() );
				nextLineToDraw.setLength(0);
				baseLineSet = false; // need to start new line for next text
				curWidth = 0.0f;
			    }
			}
		    }
		} else if (eventType == RTFPullParser.GROUP_BEGIN) {
		    parseGroup(rpp);
		} else if (eventType == RTFPullParser.GROUP_END) {
		    // Should be early end of document
		    break;
		}
	    }

	    // Flush any pending text to page
	    flushCurrentText();

	    // Finish current page
	    if (contentStream != null) {
		contentStream.endText();
		contentStream.close();
	    }

	    // Save document
	    doc.save( args[1]);
	} catch (Exception e) {
	    e.printStackTrace();
	} finally {
	    if (doc != null) {
		doc.close();
	    }
	}
    }

    private static void newPage()
	throws IOException
    {
	page = new PDPage();
	currentPageNumber++;
	page.setMediaBox( PAGE_SIZE_ILIAD );
	doc.addPage( page );
	if (contentStream != null) {
	    contentStream.endText();
	    contentStream.close();
	    contentStream = null;
	}
	selectCurrentFont();
    }

    private static void moveTextPosition(float deltaX, float deltaY)
	throws IOException
    {
	x += deltaX;
	y += deltaY;
	//System.out.println("moveTextPosition(" + deltaX + ", " + deltaY + " newX " + x + " newY " + y);
	contentStream.moveTextPositionByAmount( deltaX, deltaY);
    }

    /**
     * This method is invoked to parse RTF groups.
     * For now it skips the group (and any nested groups.)
     *
     * @param rpp The current <code>RTFPullParser</code> to pull events from.
     * @exception IOException if an error reading the RTF occurs.
     */
    private static void parseGroup(RTFPullParser rpp)
	throws IOException
    {
	int eventType = rpp.next();
	if ("rtf".equals(rpp.getName())) {
	    return;
	}

	int level = 1;
	for (eventType = rpp.next(); level != 0; eventType = rpp.next()) {
	    if (eventType == RTFPullParser.GROUP_BEGIN) {
		eventType = rpp.next();
		level++;
	    } else if (eventType == RTFPullParser.GROUP_END) {
		level--;
		if (level == 0) {
		    return;
		}
	    }
	}
    }

     private static boolean selectCurrentFont()
	throws IOException
    {
	PDSimpleFont newFont = null;
	if (bold) {
	    if (italic) {
		newFont = fontBoldItalic;
	    } else {
		newFont = fontBold;
	    }
	} else {
	    if (italic) {
		newFont = fontItalic;
	    } else {
		newFont = fontRoman;
	    }
	}

	if (contentStream == null) {
	    contentStream = new PDPageContentStream(doc, page);
	    contentStream.setFont( newFont, fontSize );
	    contentStream.beginText();
	    y = 0;
	    x = 0;
	    curWidth = 0.0f;
	    moveTextPosition( marginH, PAGE_SIZE_ILIAD.getHeight() - marginV + height);
	    //	    contentStream.moveTextPositionByAmount( marginH, y );

	    if (newFont != curFont) {
		curFont = newFont;
		//System.out.println("Changed current font");
		return true;
	    } else {
		return false;
	    }
	} else {
	    if (newFont != curFont) {
		curFont = newFont;
		contentStream.setFont( newFont, fontSize );
		//System.out.println("Changed current font");
		return true;
	    } else {
		return false;
	    }
	}

    }

    private static boolean baseLineSet = false;

     private static void flushCurrentText()
	throws IOException
    {
	if (nextLineToDraw.length() != 0) {
	    selectCurrentFont();
	    if (!baseLineSet) {
		moveTextPosition( 0, -height);
		baseLineSet = true;
		curWidth = 0.0f;
	    }
	    String curText = nextLineToDraw.toString();
	    nextLineToDraw.setLength(0);

	    contentStream.drawString( curText);
	    //System.out.println("flushCurrentText: x " + x + " y " + y + " text '" + curText + "'");
	    curWidth += (curFont.getStringWidth( curText )/1000) * fontSize;

	    if (currentPageNumber > 1000) {
		try {
		    doc.save( "temp.pdf");
		} catch (Exception e) {
		}
		System.exit(1);
	    }
	}
    }

  /**
     * Print usage information for this application.
     */
    private static void usage() {
        System.err.println( "usage: RTFToIliad file.rtf output.pdf" );
	System.exit(1);
    }
}