#!/usr/bin/env python3
# -*- coding: utf-8 -*-
# vim:ts=4:sw=4:softtabstop=4:smarttab:expandtab

# Copyright 2021 Kevin B. Hendricks, Stratford Ontario

# This plugin's source code is available under the GNU LGPL Version 2.1 or GNU LGPL Version 3 License.
# See https://www.gnu.org/licenses/old-licenses/lgpl-2.1.en.html or
# https://www.gnu.org/licenses/lgpl.html for the complete text of the license.
# If a different license is required, please contact the author directly for written permission

import sys
import os

import zlib
import zipfile
from zipfile import ZipFile
from uuid import uuid4

from contextlib import contextmanager

from hrefutils import parseRelativeHREF, buildBookPath, startingDir

PY3 = sys.version_info[0] == 3

# convert string to utf-8
def utf8_str(p, enc='utf-8'):
    if p is None:
        return None
    if isinstance(p, str):
        return p.encode('utf-8')
    if enc != 'utf-8':
        return p.decode(enc, errors='replace').encode('utf-8')
    return p

# convert string to be unicode encoded
def unicode_str(p, enc='utf-8'):
    if p is None:
        return None
    if isinstance(p, str):
        return p
    return p.decode(enc, errors='replace')

fsencoding = sys.getfilesystemencoding()

# handle paths that might be filesystem encoded
def pathof(s, enc=fsencoding):
    if s is None:
        return None
    if isinstance(s, str):
        return s
    if isinstance(s, bytes):
        try:
            return s.decode(enc)
        except:
            pass
    return s

# properly handle relative paths as well
def relpath(path, start=None):
    return os.path.relpath(pathof(path) , pathof(start))

# generate a list of files in a folder
def walk_folder(top):
    top = pathof(top)
    rv = []
    for base, dnames, names  in os.walk(top):
        base = pathof(base)
        for name in names:
            name = pathof(name)
            rv.append(relpath(os.path.join(base, name), top))
    return rv

@contextmanager
def make_temp_directory():
    import tempfile
    import shutil
    temp_dir = tempfile.mkdtemp()
    yield temp_dir
    shutil.rmtree(temp_dir)

_SKIP_LIST = [
    'encryption.xml',
    'rights.xml',
    '.gitignore',
    '.gitattributes'
]

# check for valid source folder
def valid_source(foldpath):
    files = walk_folder(foldpath)
    for file in files:
        segs = file.split(os.sep)
        if "META-INF" in segs or "meta-inf" in segs:
            if "encryption.xml" in segs or "ENCRYPTION.XML" in segs:
                return False
    return True

# return True if file should be copied to destination folder
def valid_file_to_copy(rpath):
    segs = rpath.split(os.sep)
    if ".git" in segs:
        return False
    filename = os.path.basename(rpath)
    keep = filename not in _SKIP_LIST
    return keep


def build_epub_from_folder_contents(foldpath, epub_filepath):
    outzip = zipfile.ZipFile(pathof(epub_filepath), mode='w')
    files = walk_folder(foldpath)
    if 'mimetype' in files:
        outzip.write(pathof(os.path.join(foldpath, 'mimetype')), pathof('mimetype'), zipfile.ZIP_STORED)
        print("  loading: ", "mimetype")
    else:
        raise Exception('mimetype file is missing')
    files.remove('mimetype')
    for file in files:
        print("  loading: ", file)
        if valid_file_to_copy(file):
            filepath = os.path.join(foldpath, file)
            outzip.write(pathof(filepath),pathof(file),zipfile.ZIP_DEFLATED)
    outzip.close()

    
# find path to opf from META-INF/container.xml data
def find_opf(bk, cdata):
    opfpath = None
    bk.qp.setContent(cdata)
    for txt, tp, tname, ttype, tattr in bk.qp.parse_iter():
        if txt is not None:
            txt = txt
        else:
            if tname == 'rootfile' and ttype == 'single':
                if tattr:
                    mt = tattr.get('media-type', None)
                    if mt and mt == 'application/oebps-package+xml':
                        apath = tattr.get('full-path', None)
                        if apath:
                            opfpath, fill  = parseRelativeHREF(apath)
                            return opfpath
    return opfpath


# update the opf with new bookid
def update_opf(bk, opfdata):
    uniqueid = None
    ncxpath = None
    version = None
    replace_contents = False
    newbookid = None
    res = []
    bk.qp.setContent(opfdata)
    for txt, tp, tname, ttype, tattr in bk.qp.parse_iter():
        if txt is not None:
            if replace_contents:
                res.append(newbookid)
                replace_contents = False
            else:
                res.append(txt)
        else:
            if tname == 'package' and ttype == 'begin':
                if tattr:
                    version = tattr.get('version', None)
                    uniqueid = tattr.get('unique-identifier', None)
            if tname == 'dc:identifier' and ttype == 'begin':
                if tattr:
                    aid = tattr.get('id', None)
                    if aid and uniqueid and aid == uniqueid:
                        newbookid = 'urn:uuid:' + str(uuid4())
                        replace_contents = True
            if tname == 'item' and ttype == 'single':
                if tattr:
                    mt = tattr.get('media-type', None)
                    if mt and mt == 'application/x-dtbncx+xml':
                        ncxpath = tattr.get('href', None)
            res.append(bk.qp.tag_info_to_xml(tname, ttype, tattr))
    newopfdata = "".join(res)
    return newopfdata, newbookid, version, ncxpath


def update_ncx(bk, ncxdata, newbookid):
    bk.qp.setContent(ncxdata)
    res = []
    for txt, tp, tname, ttype, tattr in bk.qp.parse_iter():
        if txt is not None:
            res.append(txt)
        else:
            if tname == "meta" and ttype == "single":
                if tattr.get("name","") == "dtb:uid":
                    tattr["content"] = newbookid
            res.append(bk.qp.tag_info_to_xml(tname, ttype, tattr))
    newncxdata = "".join(res)
    return newncxdata


def readutf8file(filepath):
    data = None
    if not os.path.exists(filepath):
        return data
    try:
        with open(filepath, 'rb') as fp:
            data = fp.read()
            data = unicode_str(data)
    except Exception as e:
        print("Failed trying to read: " + filepath)
        print(str(e))
        pass
    return data


def writeutf8file(data, filepath):
    success = False
    try:
        with open(filepath, 'wb') as fw:
            fw.write(utf8_str(data))
            success = True
    except Exception as e:
        print("Failed trying to write: " + filepath)
        print(str(e))
        success = False
        pass
    return success


# the plugin entry point
def run(bk):

    if not PY3:
        print("This plugin requires Python 3.4 or later")
        return -1

    usrsupdir = bk._w.usrsupdir
    foldpath = os.path.join(usrsupdir, 'template3')
    
    if not os.path.isdir(foldpath):
        print("Template folder is not a directory or does not exist")
        return -1
        
    if not valid_source(foldpath):
        print("Template epub is invalid due to existing encryption.xml file")
        return -1

    # find opf bookpath for template epub
    container = os.path.join(foldpath, 'META-INF', 'container.xml')
    cdata = readutf8file(container)
    if not cdata:
        print("Missing or unreadable META-INF/container.xml file")
        return -1

    opfbookpath = find_opf(bk, cdata)

    if not opfbookpath:
        print("Template epub is invalid due to missing OPF file")
        return -1
        
    opfpath = os.path.join(foldpath,opfbookpath)
    opfdata = readutf8file(opfpath)
    if not opfdata:
        print("OPF file can not be found or read")
        return -1

    newopfdata, newbookid, version, ncxpath = update_opf(bk, opfdata)

    if not newbookid:
        print("Template epub is invalid because OPF uniqueid can not be found or updated")
        return -1

    if not version or not version.startswith('3'):
        print("Template epub is invalid because it is not an epub3")
        return -1

    success = writeutf8file(newopfdata, opfpath)
    if not success:
        print("Could not update OPF file")
        return -1

    # update any existing ncx uid info
    if ncxpath:
        ncxpath, filler = parseRelativeHREF(ncxpath)
        sd = startingDir(opfbookpath)
        ncxbookpath = buildBookPath(ncxpath, sd)
        ncxpath = os.path.join(foldpath, ncxbookpath)
        ncxdata = readutf8file(ncxpath)
        if not ncxdata:
            print("Template epub is invalid because NCX in OPF manifest can not be found or read")
            return -1
        newncxdata = update_ncx(bk, ncxdata, newbookid)
        success = writeutf8file(newncxdata, ncxpath)
        if not success:
            print("Failed trying to update the NCX")
            return -1
    
    # use folder name as name for epub
    epubname = os.path.basename(foldpath) + ".epub"
                
    rv = -1
    data = b''        
    with make_temp_directory() as scratchdir:
        epubpath = os.path.join(scratchdir,epubname)
        try:
            build_epub_from_folder_contents(foldpath, epubpath)
            with open(epubpath,'rb') as fp:
                data = fp.read()
            rv = 0
        except Exception as e:
            print("Import of Template3 failed")
            print(str(e))
            rv = -1

    if rv == -1:
        return -1

    # add this epub to Sigil as a new ebook
    bk.addotherfile(epubname, data)

    return 0

def main():
    print("I reached main when I should not have\n")
    return -1
    
if __name__ == "__main__":
    sys.exit(main())
