__license__ = 'GPL v3'
__copyright__ = '2020, Ephemerality <ephemeral.vilification@gmail.com>'
__docformat__ = 'markdown en'

import os
from PyQt5.Qt import QPushButton

from calibre.devices.kindle.driver import KINDLE2
from calibre.ebooks.mobi.reader.mobi6 import MobiReader
from calibre.ebooks.pdb.header import PdbHeaderReader
from calibre.ebooks.mobi.reader.headers import MetadataHeader
from calibre.utils.logging import default_log
from calibre.devices.usbms.deviceconfig import DeviceConfig
from calibre.devices.interface import DevicePlugin
from calibre.devices.usbms.cli import CLI
from calibre.devices.usbms.device import Device
from calibre.gui2 import choose_dir, get_current_db

class DeviceConfigXRay(DeviceConfig):
    @classmethod
    def config_widget(cls):
        cw = super(DeviceConfigXRay, cls).config_widget()
        browse = QPushButton('Browse', cw)
        browse.clicked.connect(cls.browse_dir)
        cw.extra_layout.addWidget(browse)
        cls.setDirText = cw.opt_extra_customization[len(cw.opt_extra_customization) - 1].setText
        return cw
    
    @classmethod
    def browse_dir(cls):
        cls.setDirText(choose_dir(None, 'choose output dir', 'Select your X-Ray Builder output directory'))

class DeviceXRay(DeviceConfigXRay, Device):
    name = 'Device class for KindleXRay'

class USBMSXRAY(CLI, DeviceXRay):
    name = 'USBMS class for KindleXRay'
    
class KINDLEXRAY(USBMSXRAY, KINDLE2):

    description = _('Extends John Schember\'s plugin for communicating with the Kindle 2/3/4/Touch/PaperWhite/Voyage e-book reader. Copies extra metadata files built by X-Ray Builder to the device.')
    name        = 'KindleXRay'
    author      = 'Ephemerality'
    version     = (0, 0, 6)
    FORMATS     = ['azw', 'mobi', 'azw3', 'prc', 'azw1', 'tpz', 'azw4', 'kfx', 'pobi', 'pdf', 'txt']
    
    EXTRA_CUSTOMIZATION_MESSAGE = KINDLE2.EXTRA_CUSTOMIZATION_MESSAGE[:]
    EXTRA_CUSTOMIZATION_DEFAULT = KINDLE2.EXTRA_CUSTOMIZATION_DEFAULT[:]
    EXTRA_CUSTOMIZATION_MESSAGE.append(_('Overwrite existing asc files on device') + ':::' + _(
        'Uncheck this option to allow any existing .asc files on the device to remain in place.'))
    EXTRA_CUSTOMIZATION_MESSAGE.append(_('X-Ray Builder\'s output directory:') + ':::' + _(
        'X-Ray Builder output directory. Usually \out unless you\'ve changed it.'))
    EXTRA_CUSTOMIZATION_DEFAULT.append(True)
    EXTRA_CUSTOMIZATION_DEFAULT.append("")
    OPT_ASC_OVERWRITE = len(EXTRA_CUSTOMIZATION_MESSAGE) - 2
    OPT_XRAY_DIR = len(EXTRA_CUSTOMIZATION_MESSAGE) - 1

    # From the QualityCheck plugin
    def get_asin_or_uuid(self, bookId):
        db = get_current_db()
        identifiers = db.get_identifiers(bookId, index_is_id=True)
        if identifiers is not None:
            for key, val in identifiers.items():
                if key.lower() in ['asin', 'mobi-asin', 'amazon']:
                    return val
                if key.lower().startswith('amazon_'):
                    return val
        # No amazon id present, so use value off the book
        uuid = db.uuid(bookId, index_is_id=True)
        return uuid
    
    def upload_cover(self, path, filename, metadata, filepath):
        super().upload_cover(path, filename, metadata, filepath)
        self.upload_extras(path, filename, metadata, filepath)
        
    def upload_extras(self, path, filename, metadata, filepath):
        opts = self.settings()
        filetype = os.path.splitext(filepath.lower())[1]
        if filetype not in ('.azw', '.mobi', '.prc', '.azw3', '.kfx'):
            return
        
        # Check if device even supports apnx/xrays and create sidecar
        sdr_path = ""
        if (self.sidecar_apnx):
            sdr_path = os.path.join(os.path.dirname(filepath), filename + '.sdr')
            if not os.path.exists(sdr_path):
                os.makedirs(sdr_path)
        else:
            return
            
        xraypath = opts.extra_customization[self.OPT_XRAY_DIR]
        if xraypath == "" or not os.path.exists(xraypath):
            raise Exception(_('X-Ray directory not found. Set the directory within the plugin preferences.'))
        
        ASIN = None
        if filetype == '.kfx':
            # Find an ID on the Calibre metadata directly
            #book_id = getattr(metadata, "_data", None)
            ASIN = self.get_asin_or_uuid(metadata.id)
            print(ASIN)
        else:
            # From calibre.devices.kindle.apnx
            with open(filepath, 'rb') as mf:
                ident = PdbHeaderReader(mf).identity()
                if ident != 'BOOKMOBI':
                    # Check that this is really a MOBI file.
                    raise Exception(_('Not a valid MOBI file. Reports identity of %s') % ident)
                mh = MetadataHeader(mf, default_log)
                if mh.exth is None or not mh.exth.uuid:
                    raise Exception(_('No ASIN found, please ensure your book has the asin listed ' + \
                        'in its metadata and use the Quality Check plugin to update the book.'))
                else:
                    ASIN = str(mh.exth.uuid)
        
        asc_search = ['XRAY.entities.' + ASIN + '.asc',
            'EndActions.data.' + ASIN + '.asc',
            'StartActions.data.' + ASIN + '.asc',
            'AuthorProfile.profile.' + ASIN + '.asc',
            'XRAY.' + ASIN + '.previewData']
        asc_files = []
        for root, dirs, files in os.walk(xraypath):
            Done = False
            for file in files:
                if len(asc_search) == 0:
                    Done = True
                    break;
                if (file in asc_search):
                    asc_files += (os.path.join(root, file), )
                    asc_search.remove(file)
            if Done:
                break;
        
        if len(asc_files) == 0:
            print('No .asc files found for this book.')
            return
        
        # Copy found asc files to Kindle
        import shutil
        for asc_file in asc_files:
            asc_filename = os.path.basename(asc_file)
            print('Copying file: ' + asc_filename)
            if not opts.extra_customization[self.OPT_ASC_OVERWRITE] and os.path.exists(asc_file):
                print(asc_file + ' already exists, skipping.')
            else:
                shutil.copyfile(asc_file, os.path.join(sdr_path, asc_filename))
        print('Done!')