import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileReader;
import java.io.InputStreamReader;
import java.io.IOException;
import java.io.Reader;
import java.util.HashSet;
import java.util.Iterator;
import java.util.zip.ZipInputStream;

import org.pdfbox.pdmodel.PDDocument;
import org.pdfbox.pdmodel.PDPage;
import org.pdfbox.pdmodel.common.PDRectangle;
import org.pdfbox.pdmodel.edit.PDPageContentStream;
import org.pdfbox.pdmodel.font.PDSimpleFont;
import org.pdfbox.pdmodel.font.PDType1Font;

/**
 * This class provides a formatter that parses an RTF file and converts it to an iLiad sized
 * PDF file. At present it processes only embed and par tags in the RTF.
 *
 * This class requires the PDFBox jar: http://www.pdfbox.org.
 *
 * NOTE: A zip'd RTF file (ala http://www.Baen.com) can be processed directly.
 *
 * Usage: java RTFToIliad your.rtf iLiad.pdf
 * or
 * Usage: java RTFToIliad baen-rtf.zip iLiad.pdf
 * 
 * @author Scott Turner (scotty1024@mac.com)
 * @version $Revision: 1.0 $
 * @Date October 17, 2006
 */
public class RTFToIliad {
    /**
     * My Best Guess at the Media Box of an iLiad (4.88in x 5.9in
     */
    public static final PDRectangle PAGE_SIZE_ILIAD = new PDRectangle( 351, 425 );

    /**
     * A really big static main method that opens, parses and converts an RTF file into an iLiad
     * sized PDF document.
     *
     * @param args Command line arguments of which there are two: name of RTF file (which can be zipped) and the name of the PDF to be written.
     * 
     * @exception IOException If there is an error reading the RTF or writing the PDF.
     */
    public static void main(String[] args)
	throws IOException
    {
	if (args.length != 2) {
	    usage();
	}
	int marginH = 8;
	int marginV = 20;
	int fontSize = 10;
	PDSimpleFont font = PDType1Font.HELVETICA;
	float height = font.getFontDescriptor().getFontBoundingBox().getHeight()/1000;
            
	//calculate font height and increase by 5 percent.
	height = height*fontSize*1.05f;
	PDDocument doc = new PDDocument();
	PDPage page = new PDPage();
	page.setMediaBox( PAGE_SIZE_ILIAD );

	PDPageContentStream contentStream = null;
	float y = -1;

	float maxStringLength = page.getMediaBox().getWidth() - 2*marginH;

	RTFPullParser rpp = new RTFPullParser();

	if (args[0].toLowerCase().endsWith(".zip")) {
	    ZipInputStream zipFile = new ZipInputStream(new FileInputStream(args[0]));
	    zipFile.getNextEntry();
	    rpp.setInput ( new InputStreamReader(zipFile));
	} else {
	    rpp.setInput ( new FileReader(new File(args[0])));
	}

	StringBuffer nextLineToDraw = new StringBuffer(16384);

	try {
	    for (int eventType = rpp.getEventType(); eventType != RTFPullParser.END_DOCUMENT; eventType = rpp.next()) {
		if (eventType == RTFPullParser.COMMAND) {
		    if ("par".equals(rpp.getName())) {
			if (nextLineToDraw.length() != 0) {
			    if (contentStream == null) {
				contentStream = new PDPageContentStream(doc, page);
				contentStream.setFont( font, fontSize );
				contentStream.beginText();
				y = page.getMediaBox().getHeight() - marginV + height;
				contentStream.moveTextPositionByAmount( marginH, y );
			    }

			    y -= height;
			    contentStream.moveTextPositionByAmount( 0, -height);
			    contentStream.drawString( nextLineToDraw.toString() );
			    nextLineToDraw.setLength(0);
			}
			y -= height / 2;
			contentStream.moveTextPositionByAmount( 0, -(height / 2));
		    } else if ("emdash".equals(rpp.getName())) {
			nextLineToDraw.append((char)0x2014);
		    }
		} else if (eventType == RTFPullParser.TEXT) {
		    String[] lineWords = rpp.getTextCharacters().trim().split( " " );
		    int lineIndex = 0;
		    while (lineIndex < lineWords.length) {   
			float lengthIfUsingNextWord = 0;

			nextLineToDraw.append( lineWords[lineIndex] );
			nextLineToDraw.append( " " );
			lineIndex++;
			if (lineIndex < lineWords.length) {
			    String lineWithNextWord = nextLineToDraw.toString() + lineWords[lineIndex];
			    lengthIfUsingNextWord = (font.getStringWidth( lineWithNextWord )/1000) * fontSize;

			    if (lengthIfUsingNextWord >= maxStringLength) {
				if (y < marginV) {
				    page = new PDPage();
				    page.setMediaBox( PAGE_SIZE_ILIAD );
				    doc.addPage( page );
				    if (contentStream != null) {
					contentStream.endText();
					contentStream.close();
				    }
				    contentStream = new PDPageContentStream(doc, page);
				    contentStream.setFont( font, fontSize );
				    contentStream.beginText();
				    y = page.getMediaBox().getHeight() - marginV + height;
				    contentStream.moveTextPositionByAmount( marginH, y );
				}
				//System.out.println( "Drawing string at " + x + "," + y );
                    
                    
				contentStream.moveTextPositionByAmount( 0, -height);
				y -= height;
				contentStream.drawString( nextLineToDraw.toString() );
				nextLineToDraw.setLength(0);
			    }
			}
		    }
		} else if (eventType == RTFPullParser.GROUP_BEGIN) {
		    parseGroup(rpp);
		} else if (eventType == RTFPullParser.GROUP_END) {
		    // Should be early end of document
		    break;
		}
	    }

	    // Finish current text chunk
	    if (nextLineToDraw.length() != 0) {
		contentStream.moveTextPositionByAmount( 0, -height);
		contentStream.drawString( nextLineToDraw.toString() );
	    }

	    // Finish current page
	    if (contentStream != null) {
		contentStream.endText();
		contentStream.close();
	    }

	    // Save document
	    doc.save( args[1]);
	} catch (Exception e) {
	    e.printStackTrace();
	} finally {
	    if (doc != null) {
		doc.close();
	    }
	}
    }

    /**
     * This method is invoked to parse RTF groups.
     * For now it skips the group (and any nested groups.)
     *
     * @param rpp The current <code>RTFPullParser</code> to pull events from.
     * @exception IOException if an error reading the RTF occurs.
     */
    private static void parseGroup(RTFPullParser rpp)
	throws IOException
    {
	int eventType = rpp.next();
	if ("rtf".equals(rpp.getName())) {
	    return;
	}

	int level = 1;
	for (eventType = rpp.next(); level != 0; eventType = rpp.next()) {
	    if (eventType == RTFPullParser.GROUP_BEGIN) {
		eventType = rpp.next();
		level++;
	    } else if (eventType == RTFPullParser.GROUP_END) {
		level--;
		if (level == 0) {
		    return;
		}
	    }
	}
    }

    /**
     * Print usage information for this application.
     */
    private static void usage() {
        System.err.println( "usage: RTFToIliad file.rtf output.pdf" );
	System.exit(1);
    }
}