#!/usr/bin/env python
# -*- coding: utf-8 -*-

from __future__ import absolute_import, division, print_function, unicode_literals

__license__   = 'GPL v3'
__copyright__ = u'2019, Jürgen Habelt <juergen@habelt-jena.de>'
__docformat__ = 'restructuredtext en'

import os
from xml.dom.minidom import Document

try:
    from calibre_plugins.toc_view_generator.logging_factory import LoggingFactory
    from calibre_plugins.toc_view_generator.ruby_port.pdf_extractor import PdfExtractor
    from calibre_plugins.toc_view_generator.ruby_port.epub_extractor import EpubExtractor

except ImportError:
    from logging_factory import LoggingFactory
    from ruby_port.pdf_extractor import PdfExtractor
    from ruby_port.epub_extractor import EpubExtractor
    

class HtmlGenerator(object):
    '''
    Main class of the Ruby port: generates the Html document
    '''
        
    def __init__(self, invoker):
        '''
        Initialization, injects the Invoker with settings
        '''
        self.extractors = { '.pdf' : PdfExtractor, '.epub' : EpubExtractor }         # generator classes dictionary
        self.invoker = invoker
        
        
    @property
    def settings(self):
        '''
        Settings with property semantic
        '''
        return self.invoker.settings


    def generate_list(self, bookmark, parent = None):
        '''
        Generates part of a Html document (one level list)
        @param bookmark: the book mark from which to generate
        '''
        if bookmark.level == 0:
            self.document = Document()
            div_elem = self.document.createElement('div')
            self.document.appendChild(div_elem)
            for child in bookmark.children:
                h3_elem = self.document.createElement(self.settings['header_markup'])
                text_node = self.document.createTextNode(child.title)
                h3_elem.appendChild(text_node)
                div_elem.appendChild(h3_elem)
                self.generate_list(child, div_elem)
        
        else:
            if len(bookmark.children) > 0 and self.settings['max_level'] > bookmark.level:
                ol_elem = self.document.createElement('ol')
                parent.appendChild(ol_elem)
                if self.settings['continuous'] and bookmark.level == 1:
                    ol_elem.setAttribute('start', bookmark.number)
                if self.settings['omit']:
                    ol_elem.setAttribute('style', 'list-style-type:none;')
                for child in bookmark.children:
                    li_elem = self.document.createElement('li')
                    ol_elem.appendChild(li_elem)
                    text_node = self.document.createTextNode(child.title)
                    li_elem.appendChild(text_node)
                    
                    self.generate_list(child, li_elem)
                    
        return self.document                    


    def generate_html(self, bookmark):
        '''
        Generates the Html document with book marks
        @param bookmark: the root book mark
        '''
        base = os.path.basename(self.book_path)
        LoggingFactory().getLogger(self).info("Generating from %s" % base)
        
        xml_data = self.generate_list(bookmark)
        with open(self.result_path, 'w') as f:
            xml_data.writexml(f)
        
        LoggingFactory().getLogger(self).info("File from %s generated" % base)
        
        
    def generate(self, book_path, result_path):
        '''
        The main generate method
        @param book_path: the path of the e-book
        @param result_path: the path of the result
        '''
        try:
            self.book_path = book_path                              # store for later use
            self.result_path = result_path
            
            ext = os.path.splitext(book_path)[1]
            extractor_class = self.extractors[ext]
            extractor = extractor_class(self.invoker)
            to = extractor.extract(book_path)
            root = extractor.parse(to)
            root.to_log()
            self.generate_html(root)
            
            return True
            
        except:                                                     # logging and handling in the calling code
            raise
        