#!/usr/bin/env python
# -*- coding: utf-8 -*-

from __future__ import absolute_import, division, print_function, unicode_literals

__license__   = 'GPL v3'
__copyright__ = u'2019, Jürgen Habelt <juergen@habelt-jena.de>'
__docformat__ = 'restructuredtext en'

import os, re

try:
    from calibre_plugins.toc_view_generator.logging_factory import LoggingFactory
    from calibre_plugins.toc_view_generator.ruby_port.bookmark import Bookmark

except ImportError:
    from logging_factory import LoggingFactory
    from ruby_port.bookmark import Bookmark


class Extractor(object):
    '''
    The base class of all derived extractor classes
    '''
        
    def parse_node(self, bookmark, toc):
        '''
        Parses a xml element node recursively and adds children to the parent book mark
        '''
        for toc_child in toc['children']:
            title = toc_child['title']
            title = self.strip_from_title(title)                                                    # strip from the beginning
            LoggingFactory().getLogger(self).debug("title text: %s" % title)        
            child = Bookmark(title, bookmark.level + 1, bookmark)
            
            if bookmark.level == 1:
                self.level2_number += 1
                child.number = self.level2_number
            
            bookmark.children.append(child)
      
            self.parse_node(child, toc_child)
                
        
    def parse(self, frm):
        '''
        Parses a file into a book mark tree and returns the root node
        @param frm: the xml file from which to evaluate the book marks
        '''        
        root = Bookmark( "Root" )
        self.level2_number = 0
        
        self.parse_node(root, frm)

        return root

    
    @property
    def settings(self):
        '''
        Settings with attribute semantic
        '''
        return self.invoker.settings
    
    
    @property
    def strip(self):
        '''
        Strip regex  with attribute semantic
        '''
        if not self.settings['use_strip']:
            return r'^'
        else:
            return self.settings['strip']
    
    
    def strip_from_title(self, title):
        '''
        Use the strip option to strip from the beginning
        @param title: the title from which to strip
        '''
        if title == None:
            return ''
                
        title = title.strip()
        title = re.sub(r'(?m)\s+', ' ', title)

        strip = self.strip                                                      # stems from derived class
        regex = strip + r'(.*?)$'                                               # concatenate 2 regexes
        return re.search(regex, title).group(1)                                 # strip from the beginning    
        
    
    def change_extension(self, path, ext):
        '''
        Can be used to change the extension of a file path
        @param path: the path to be changed
        @param ext: the new extension
        '''
        pre = os.path.splitext(path)[0]
        return pre + ext