# -*- coding: utf-8 -*-

from __future__ import absolute_import
from __future__ import print_function
import six
__license__ = 'GPL 3'
__copyright__ = '2014, Saulius P. <saulius@kofmarai.net>'
__docformat__ = 'restructuredtext en'

import os
import re

try:
    load_translations()
except NameError:
    pass # load_translations() added in calibre 1.9

try:
    from PyQt5.Qt import QWidget, QPushButton, QGridLayout, QHBoxLayout, \
        QVBoxLayout, QLabel, QLineEdit, QListWidget, QUrl, Qt, QCheckBox, QGroupBox
except ImportError:
    from PyQt4.Qt import QWidget, QPushButton, QGridLayout, QHBoxLayout, \
        QVBoxLayout, QLabel, QLineEdit, QListWidget, QUrl, Qt, QCheckBox, QGroupBox

from calibre.utils.config import JSONConfig
from calibre.gui2 import choose_files, error_dialog, open_url
from calibre.utils.zipfile import ZipFile
from lxml import etree
from calibre.ebooks.oeb.parse_utils import RECOVER_PARSER
from calibre.constants import config_dir
from calibre.utils.localization import canonicalize_lang, get_language
from calibre.ptempfile import PersistentTemporaryFile

OOR_NS = 'http://openoffice.org/2001/registry'
DICT_ZIP = config_dir+os.sep+'plugins'+os.sep+'hyphthisdicts.zip'

# This is where all preferences for this plugin will be stored
# Remember that this name (i.e. plugins/interface_demo) is also
# in a global namespace, so make it as unique as possible.
# You should always prefix your config file name with plugins/,
# so as to ensure you dont accidentally clobber a calibre config file

prefs = JSONConfig('plugins'+os.sep+'hyphenatethis')

# Set defaults
prefs.defaults['length'] = 5
prefs.defaults['lroverride'] = {}
prefs.defaults['tags_ignore'] = 'h1, h2, h3'
prefs.defaults['tags_consider'] = ''
prefs.defaults['custom_col'] = ['', 'Hyphenated', 'Hyphens removed']


class HyphenateThisConfigWidget(QWidget):

    def __init__(self):
        from calibre_plugins.hyphenatethis.hyphenator.hutils import HYP_REGEX_LEFT_MIN, HYP_REGEX_RIGHT_MIN
        self.regl = re.compile(HYP_REGEX_LEFT_MIN)
        self.regr = re.compile(HYP_REGEX_RIGHT_MIN)
        
        QWidget.__init__(self)
        self.l = QGridLayout()
        self.setLayout(self.l)
        
        self.titlel = QHBoxLayout()
        #pxmap = QPixmap()
        #pxmap.load(get_icons('icons/download-dictionary.png'))
        self.img = QLabel()
        self.img.setPixmap(get_icons('icons/download-dictionary.png').pixmap(36, 48))
        self.img.setMaximumSize(36, 48)
        self.img.setScaledContents(True)
        self.titlel.addWidget(self.img)
        
        self.download_label = QLabel()
        self.download_label.setText(u'<strong>'+_('Download ')+u'<a href="http://extensions.libreoffice.org/extension-center?getCategories=Dictionary&getCompatibility=any">'+
                                    _(u'Libre Office dictionaries')+u'</a>'+_(u' from their site')+u'</strong>')
        self.download_label.setTextInteractionFlags(Qt.LinksAccessibleByMouse | Qt.LinksAccessibleByKeyboard)
        self.download_label.linkActivated.connect(self._clickLink)
        self.titlel.addWidget(self.download_label)
        #self.l.addWidget(self.download_label, 0, 0, 1, 2)
        self.l.addLayout(self.titlel, 0, 0, 1, 2)
        
        #self.dicts_label = QLabel(_('Currently added dictionaries:'))
        #self.l.addWidget(self.dicts_label, 1, 0, 1, 1)
        
        self.dicts_group = QGroupBox(_('Currently added dictionaries:'))
        self.dgl = QVBoxLayout()
        self.dicts_group.setLayout(self.dgl)
        
        self.dicts = QListWidget()
        self.dicts.itemClicked.connect(self._dictActivated)
        self._updateDictList()
        self.dgl.addWidget(self.dicts)
        self.l.addWidget(self.dicts_group, 1, 0, 2, 1)
        
        self.vl = QVBoxLayout()
        
        self.dact_group = QGroupBox(_('Dictionary actions:'))
        self.dal = QVBoxLayout()
        self.dact_group.setLayout(self.dal)
        self.addbtn = QPushButton(_('Add dictionary (oxt or dic)'), self)
        self.addbtn.clicked.connect(self._addDict)
        self.dal.addWidget(self.addbtn)
        
        self.delbtn = QPushButton(_('Remove dictionary'), self)
        self.delbtn.clicked.connect(self._removeDict)
        self.dal.addWidget(self.delbtn)
        
        self.vl.addWidget(self.dact_group)
        
        self.short_group = QGroupBox(_('Skip words shorter than:'))
        #self.shorter_label = QLabel(_('Don\'t hyphenate words shorter than:'))
        #self.vl.addWidget(self.shorter_label)
        
        self.hl = QHBoxLayout()
        
        self.length = QLineEdit()
        self.length.setText(str(prefs['length']))
        self.hl.addWidget(self.length)
        
        self.lenlabel = QLabel(_('characters'))
        self.hl.addWidget(self.lenlabel)
        self.short_group.setLayout(self.hl)
        self.vl.addWidget(self.short_group)
        
        
        self.lro_group = QGroupBox(_('Right and left length limits:'))
        self.lro_gl = QVBoxLayout()
        self.lro_group.setLayout(self.lro_gl)
        self.hl_ov1 = QHBoxLayout()
        self.ov_label = QLabel(_('Override:'))
        self.hl_ov1.addWidget(self.ov_label)
        
        self.ov_activate = QCheckBox(self)
        self.ov_activate.clicked.connect(self._ovClicked)
        self.ov_activate.setEnabled(False)
        self.hl_ov1.addWidget(self.ov_activate)
        
        self.lro_gl.addLayout(self.hl_ov1)
        
        self.gl_ov = QGridLayout()
        self.ov_left_label = QLabel('LEFTHYPHENMIN')
        self.gl_ov.addWidget(self.ov_left_label, 0, 0, 1, 1)
        
        self.ov_left = QLineEdit()
        self.ov_left.setEnabled(False)
        self.ov_left.textChanged.connect(self._textChangedLeft)
        self.gl_ov.addWidget(self.ov_left, 0, 1, 1, 1)
        
        self.ov_right_label = QLabel('RIGHTHYPHENMIN')
        self.gl_ov.addWidget(self.ov_right_label, 1, 0, 1, 1)
        
        self.ov_right = QLineEdit()
        self.ov_right.setEnabled(False)
        self.ov_right.textChanged.connect(self._textChangedRight)
        self.gl_ov.addWidget(self.ov_right, 1, 1, 1, 1)
        
        self.lro_gl.addLayout(self.gl_ov)
        
        self.vl.addWidget(self.lro_group)
        self.l.addLayout(self.vl, 1, 1, 1, 1)
        
        self.tag_group = QGroupBox(_('Comma separated list of tags to:'))
        self.tgl = QGridLayout()
        self.tag_group.setLayout(self.tgl)
        self.ti_label = QLabel(_('ignore'))
        self.tgl.addWidget(self.ti_label, 0, 0, 1, 1)
        self.tc_label = QLabel(_('parse'))
        self.tgl.addWidget(self.tc_label, 0, 1, 1, 1)
        
        self.ti = QLineEdit()
        self.ti.setText(prefs['tags_ignore'])
        self.tgl.addWidget(self.ti, 1, 0, 1, 1)
        
        self.tc = QLineEdit()
        self.tc.setText(prefs['tags_consider'])
        self.tgl.addWidget(self.tc, 1, 1, 1, 1)
        
        self.l.addWidget(self.tag_group, 3, 0, 1, 2)
        
        self._dictMap = prefs['lroverride']
        
        self.custg = QGroupBox(_('Custom column to save hyphenation status:'))
        self.custl = QGridLayout()
        self.custg.setLayout(self.custl)
        
        self.collabel = QLabel(_('Column name:'))
        self.custl.addWidget(self.collabel, 0, 0, 1, 1)
        
        self.col = QLineEdit()
        self.col.setText(prefs['custom_col'][0])
        self.custl.addWidget(self.col, 1, 0, 1, 1)
        
        self.hylabel = QLabel(_('"Hyhpenated" value:'))
        self.custl.addWidget(self.hylabel, 0, 1, 1, 1)
        
        self.hyvalue = QLineEdit()
        self.hyvalue.setText(six.text_type(prefs['custom_col'][1]))
        self.custl.addWidget(self.hyvalue, 0, 2, 1, 1)
        
        
        self.remlabel = QLabel(_('"Hyphens removed" value:'))
        self.custl.addWidget(self.remlabel, 1, 1, 1, 1)
        
        self.remvalue = QLineEdit()
        self.remvalue.setText(six.text_type(prefs['custom_col'][2]))
        self.custl.addWidget(self.remvalue, 1, 2, 1, 1)
        
        self.l.addWidget(self.custg, 4, 0, 1, 2)
    
    def _addDict(self):
        dfiles = choose_files(self, 'add hyphenation dictionaries dialog',
                _('Select dictionaries'), filters=[('OpenOffice dictionaries', ['oxt', 'dic'])])
        
        if dfiles is None: return
        
        for df in dfiles:
            if df[len(df)-3:].lower() == 'oxt':
                self._addOxt(df)
            elif df[len(df)-3:].lower() == 'dic':
                self._addDic(df)
    
    def _addOxt(self, path):
        with ZipFile(path) as z:
            dict_desc = None
            
            for desc_file in ['dictionaries.xcu', 'Dictionaries.xcu']:
                try:
                    dict_desc = z.read(desc_file)
                except:
                    print((_("Could not open %s") % desc_file))
            
            if dict_desc is None:
                raise NameError(_("Could not load dictionary description"))
            
            root = etree.fromstring(dict_desc, RECOVER_PARSER)
            values = etree.XPath('//node/node/node/prop/value/text()')(root)
            for v in values:
                if v == 'DICT_HYPH':
                    lang, dic = None, None
                    propsNode = v.getparent().getparent().getparent()
                    for propval in etree.XPath('prop[@oor:name="Locations"]/value', namespaces={'oor':OOR_NS})(propsNode):
                        dic = propval.text[9:]
                        break
                    for propval in  etree.XPath('prop[@oor:name="Locales"]/value', namespaces={'oor':OOR_NS})(propsNode):
                        lang = propval.text.split(' ')[0]
                    
                    if dic is not None and lang is not None:
                        f = PersistentTemporaryFile()
                        f.write(z.read(dic))
                        self._addDictionary(f.name, canonicalize_lang(lang))
                    
                    break
        
    def _addDic(self, path):
        fn = os.path.basename(path)
        s = fn.split('_')
        if len(s) < 2:
            return self._errorFileName()
        l = canonicalize_lang(s[1].split('.')[0])
        if l is None:
            return self._errorFileName()
        self._addDictionary(path, l)
        
    
    def _addDictionary(self, f, lang):
        with ZipFile(DICT_ZIP, 'a', allowZip64=True) as z:
            z.replace(f, lang+'.dic')
        self._updateDictList()
    
    def _removeDict(self):
        for di in self.dicts.selectedItems():
            with ZipFile(DICT_ZIP, 'a', allowZip64=True) as z:
                z.delete(di.text().split(' ')[0])
        self._updateDictList()
    
    def _updateDictList(self):
        self.dicts.clear()
        if not os.path.exists(DICT_ZIP):
            with ZipFile(DICT_ZIP, "w"):
                pass
        
        with ZipFile(DICT_ZIP) as z:
            for d in z.namelist():
                self.dicts.addItem(d+' - '+get_language(d.split('.')[0]))
    
    def _dictActivated(self, itm):
        self.ov_activate.setEnabled(True)
        n = str(itm.text().split(' ')[0])
        if n in self._dictMap:
            self.ov_activate.setChecked(True)
            self.ov_left.setText(str(self._dictMap[n]['LEFTHYPHENMIN']))
            self.ov_right.setText(str(self._dictMap[n]['RIGHTHYPHENMIN']))
            self.ov_left.setEnabled(True)
            self.ov_right.setEnabled(True)
        else:
            self.ov_activate.setChecked(False)
            self.ov_left.setEnabled(False)
            self.ov_right.setEnabled(False)
            with ZipFile(DICT_ZIP, 'r') as z:
                dct = z.read(n)[0:64]
                try:
                    t = self.regl.search(dct).group(1)
                except:
                    t = 'N/A'
                self.ov_left.setText(t)
                try:
                    t = self.regr.search(dct).group(1)
                except:
                    t = 'N/A'
                self.ov_right.setText(t)
    
    def _ovClicked(self):
        n = str(self.dicts.selectedItems()[0].text().split(' ')[0])
        if self.ov_activate.isChecked():
            self._dictMap[n] = {}
            if self.ov_left.text() == 'N/A': self.ov_left.setText('2')
            if self.ov_right.text() == 'N/A': self.ov_right.setText('2')
            self._dictMap[n]['LEFTHYPHENMIN'] = int(self.ov_left.text())
            self._dictMap[n]['RIGHTHYPHENMIN'] = int(self.ov_right.text())
            self.ov_left.setEnabled(True)
            self.ov_right.setEnabled(True)
            self.ov_left.inputMask = '9'
            self.ov_right.inputMask = '9'
        else:
            try:
                del self._dictMap[n]
            except:
                pass
            self.ov_left.setEnabled(False)
            self.ov_right.setEnabled(False)
            self.ov_left.inputMask = ''
            self.ov_right.inputMask = ''
            self._dictActivated(self.dicts.selectedItems()[0])
    
    def _textChangedLeft(self, qs):
        if self.ov_activate.isChecked() is False:
            return
        n = str(self.dicts.selectedItems()[0].text().split(' ')[0])
        self._dictMap[n]['LEFTHYPHENMIN'] = int(str(qs))
    
    def _textChangedRight(self, qs):
        if self.ov_activate.isChecked() is False:
            return
        n = str(self.dicts.selectedItems()[0].text().split(' ')[0])
        self._dictMap[n]['RIGHTHYPHENMIN'] = int(str(qs))
            
    def _clickLink(self):
        open_url(QUrl("http://extensions.libreoffice.org/extension-center?getCategories=Dictionary&getCompatibility=any"))
    
    def _errorFileName(self):
        return error_dialog(self, _('Bad naming'),
                    _('Raw *.dic file should be named "hyph_<language code>.dic". Language code '
                     'is used to determine actual language. It is better to add *.oxt file though.'), show=True)

    def save_settings(self):
        prefs['length'] = int(self.length.text())
        prefs['lroverride'] = self._dictMap
        prefs['tags_ignore'] = str(self.ti.text())
        prefs['tags_consider'] = str(self.tc.text())
        prefs['custom_col'] = [str(self.col.text()), six.text_type(str(self.hyvalue.text())), six.text_type(str(self.remvalue.text()))]
        
